# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2021 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.
"""
Created on Mon May 24 21:19:41 2021
Authors: thimios.sokos@upatras.gr, laurentiu.danciu@sed.ethz.ch

Module exports :class:`BooreEtAl2020`
"""
import numpy as np

from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA

CONSTS = {
    "Mref": 4.5,
    "Rref": 1.0,
    "Vref": 760.0}


def _get_inter_event_tau(C, mag):
    """
    Returns the inter-event standard deviation (tau), which is dependent
    on magnitude
    """
    magncof = (mag - C["Mtau1"]) / (C["Mtau2"] - C["Mtau1"])
    tau = C["tau1"] + (C["tau2"] - C["tau1"]) * magncof
    tau[mag <= C["Mtau1"]] = C["tau1"]
    tau[mag >= C["Mtau2"]] = C["tau2"]
    return tau


def _get_linear_site_term(C, vs30):
    """
    Returns the linear site scaling term (equation 6)
    """
    flin = np.ones_like(vs30)
    flin = flin*C["clin"] * np.log(C["Vc"]/CONSTS["Vref"])
    flin[vs30 <= C["V1"]] = C["clin"] * np.log(C["V1"] / CONSTS["Vref"])
    ok = (vs30 > C["V1"]) & (vs30 <= C["Vc"])
    flin[ok] = C["clin"] * np.log(vs30[ok] / C["Vref"])
    return flin


def _get_magnitude_scaling_term(C, rup):
    """
    Returns the magnitude scling term defined in equation (2)
    it is the same as Boore 2014
    """
    dmag = rup.mag - C["Mh"]
    mag_term = np.where(rup.mag <= C["Mh"],
                        C["e4"] * dmag + C["e5"] * dmag**2,
                        C["e6"] * dmag)
    return _get_style_of_faulting_term(C, rup) + mag_term


def _get_nonlinear_site_term(C, vs30, pga_rock):
    """
    Returns the nonlinear site scaling term (equation 7)
    """
    atmp = C["f5"] * (np.minimum(vs30, CONSTS["Vref"]) - 360.)
    f2 = C["f4"] * (np.exp(atmp) - np.exp(C["f5"] * (CONSTS["Vref"] - 360.)))
    fnl = C["f1"] + f2 * np.log(1.0 + pga_rock/C["f3"])
    return fnl


def _get_path_scaling(C, dists, mag):
    """
    Returns the path scaling term given by equation (3)
    """
    rval = np.sqrt(dists.rjb**2 + C["h"]**2)
    scaling = (C["c1"] + C["c2"] * (mag - CONSTS["Mref"])) * np.log(
        rval / CONSTS["Rref"])
    return scaling + (C["c3"] * (rval - CONSTS["Rref"]))


def _get_pga_on_rock(C, rup, dists):
    """
    Returns the median PGA on rock, which is a sum of the
    magnitude and distance scaling
    """
    return np.exp(_get_magnitude_scaling_term(C, rup) +
                  _get_path_scaling(C, dists, rup.mag))


def _get_site_scaling(C, pga_rock, sites):
    """
    Returns the site-scaling term (equation 5), broken down into a
    linear scaling, a nonlinear scaling and a basin scaling term
    """
    flin = _get_linear_site_term(C, sites.vs30)
    fnl = _get_nonlinear_site_term(C, sites.vs30, pga_rock)
    return flin + fnl


def _get_style_of_faulting_term(C, rup):
    """
    Get fault type dummy variables
    Fault type (Strike-slip, Normal, Thrust/reverse) is
    derived from rake angle.
    Rakes angles within 30 of horizontal are strike-slip,
    angles from 30 to 150 are reverse, and angles from
    -30 to -150 are normal.
    Note that the 'Unspecified' case is not considered here as
    rake is always given.
    what about bias (B) ??? it should be taken care here
    """
    res = np.full_like(rup.rake, C["e2"])
    res[(np.abs(rup.rake) <= 30) | ((180 - np.abs(rup.rake)) <= 30)] = C["e1"]
    res[(rup.rake > 30.0) & (rup.rake < 150)] = C["e3"]
    return res


class BooreEtAl2020(GMPE):
    """
    Implements GMPE developed by David M. Boore, Jonathan P. Stewart, Andreas
    A. Skarlatoudis,Emel Seyhan, Basil Margaris, Nikos Theodoulidis,Emmanuel
    Scordilis, Ioannis Kalogeras,Nikolaos Klimis, and Nikolaos S. Melis, and
    published as "Ground-Motion Prediction Model for Shallow Crustal
    Earthquakes in Greece (2020, BSSA, ).
    implemented by thimios.sokos@upatras.gr & laurentiu.danciu@sed.ethz.ch
    """
    #: Supported tectonic region type is active shallow crust
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types are spectral acceleration,
    #: peak ground velocity and peak ground acceleration
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is orientation-independent
    #: measure :attr:`~openquake.hazardlib.const.IMC.RotD50`
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.RotD50

    #: Supported standard deviation types are inter-event, intra-event
    #: and total, see equation 2, pag 106.
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL, const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT}

    #: Required site parameters is Vs30
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: Required rupture parameters are magnitude, and rake.
    REQUIRES_RUPTURE_PARAMETERS = {'mag', 'rake'}

    #: Required distance measure is Rjb
    REQUIRES_DISTANCES = {'rjb'}

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        C_PGA = self.COEFFS[PGA()]
        pga_rock = _get_pga_on_rock(C_PGA, ctx, ctx)
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            mean[m] = (_get_magnitude_scaling_term(C, ctx) +
                       _get_path_scaling(C, ctx, ctx.mag) +
                       _get_site_scaling(C, pga_rock, ctx))
            if imt.string != "PGV":
                mean[m] = np.log((np.exp(mean[m]) / 981.))
            tau[m] = _get_inter_event_tau(C, ctx.mag)
            phi[m] = C['phi']
        sig[:] = np.sqrt(tau**2 + phi**2)

    COEFFS = CoeffsTable(sa_damping=5, table="""\
IMT	           B           e0	      e1	      e2	     e3	         e4	        e5	           e6	         Mh	          c1	         c2	         c3	            Mref	    Rref	     h	        clin	         V1	          Vc	     Vref	     f1	       f3	         f4	              f5	         phi	   tau1	        Mtau1	     Mtau2	  tau2	sigma_M_ge_6_0
pgv     -4.31371E-01	4.99652E+00	5.07987E+00	4.84797E+00	5.03142E+00	8.52485E-01	-1.53600E-01	2.25200E-01	    6.20000E+00	-1.24300E+00	1.48900E-01	-4.71666E-03	    4.50000E+00	1.00000E+00	5.30000E+00	-8.40000E-01	2.00000E+02	1.30000E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.00000E-01	-8.44000E-03	5.96444E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91553E-01
pga	    -1.49920E-01	7.28273E+00	7.35054E+00	7.13618E+00	7.37659E+00	8.33636E-01	5.05300E-02	    -1.66200E-01	6.20000E+00	-1.13400E+00	1.91700E-01	-1.15138E-02	4.50000E+00	1.00000E+00	4.50000E+00	-6.00000E-01	2.00000E+02	1.50000E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.50000E-01	-7.01000E-03	5.97441E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.92413E-01
0.010	-1.49920E-01	7.28690E+00	7.35245E+00	7.14247E+00	7.38444E+00	8.31000E-01	4.93200E-02	    -1.65900E-01	6.20000E+00	-1.13400E+00	1.91600E-01	-1.15146E-02	4.50000E+00	1.00000E+00	4.50000E+00	-6.03720E-01	2.00000E+02	1.50020E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.48330E-01	-7.01000E-03	5.96770E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91834E-01
0.020	-1.27090E-01	7.32122E+00	7.38472E+00	7.18162E+00	7.41499E+00	8.14760E-01	5.33880E-02	    -1.65610E-01	6.20000E+00	-1.13940E+00	1.89620E-01	-1.15032E-02	4.50000E+00	1.00000E+00	4.50000E+00	-5.73880E-01	2.00000E+02	1.50036E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.47100E-01	-7.28000E-03	5.96560E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91653E-01
0.022	-1.37040E-01	7.33474E+00	7.39732E+00	7.19747E+00	7.42638E+00	8.09850E-01	5.48880E-02	    -1.65200E-01	6.20000E+00	-1.14050E+00	1.89240E-01	-1.15162E-02	4.50000E+00	1.00000E+00	4.50000E+00	-5.66750E-01	2.00000E+02	1.50068E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.48010E-01	-7.32000E-03	5.96040E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91205E-01
0.025	-1.51600E-01	7.36078E+00	7.42215E+00	7.22755E+00	7.44718E+00	8.04410E-01	5.75290E-02	    -1.64990E-01	6.20000E+00	-1.14190E+00	1.88750E-01	-1.15586E-02	4.50000E+00	1.00000E+00	4.50000E+00	-5.55200E-01	2.00000E+02	1.50104E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.50150E-01	-7.36000E-03	5.96160E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91308E-01
0.029	-1.74290E-01	7.40073E+00	7.46008E+00	7.27476E+00	7.47733E+00	7.91590E-01	6.07320E-02	    -1.66320E-01	6.20000E+00	-1.14230E+00	1.88440E-01	-1.16500E-02	4.50000E+00	1.00000E+00	4.50000E+00	-5.38500E-01	2.00000E+02	1.50126E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.53870E-01	-7.37000E-03	5.96640E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.91722E-01
0.030	-1.80000E-01	7.41130E+00	7.47005E+00	7.28739E+00	7.48515E+00	7.88240E-01	6.14440E-02	    -1.66900E-01	6.20000E+00	-1.14210E+00	1.88420E-01	-1.16560E-02	4.50000E+00	1.00000E+00	4.49000E+00	-5.34140E-01	2.00000E+02	1.50295E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.54850E-01	-7.35000E-03	5.97290E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.92283E-01
0.032	-1.90370E-01	7.43186E+00	7.48960E+00	7.31170E+00	7.50052E+00	7.82500E-01	6.28060E-02	    -1.68130E-01	6.20000E+00	-1.14120E+00	1.88400E-01	-1.17250E-02	4.50000E+00	1.00000E+00	4.45000E+00	-5.25290E-01	2.00000E+02	1.50312E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.56850E-01	-7.31000E-03	5.97640E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.92585E-01
0.035	-2.02010E-01	7.46198E+00	7.51827E+00	7.34681E+00	7.52411E+00	7.75210E-01	6.45590E-02	    -1.70150E-01	6.20000E+00	-1.13880E+00	1.88390E-01	-1.18820E-02	4.50000E+00	1.00000E+00	4.40000E+00	-5.11920E-01	2.00000E+02	1.50324E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.60160E-01	-7.21000E-03	5.98040E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.92930E-01
0.036	-2.03900E-01	7.47160E+00	7.52747E+00	7.35782E+00	7.53199E+00	7.72920E-01	6.50280E-02	    -1.70830E-01	6.20000E+00	-1.13780E+00	1.88370E-01	-1.19050E-02	4.50000E+00	1.00000E+00	4.38000E+00	-5.07520E-01	2.00000E+02	1.50332E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.61420E-01	-7.17000E-03	5.98480E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.93310E-01
0.040	-2.11280E-01	7.50984E+00	7.56450E+00	7.40006E+00	7.56514E+00	7.66440E-01	6.61830E-02	    -1.73570E-01	6.20000E+00	-1.13240E+00	1.88160E-01	-1.21800E-02	4.50000E+00	1.00000E+00	4.32000E+00	-4.90650E-01	2.00000E+02	1.50335E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.67770E-01	-6.98000E-03	5.98950E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.93715E-01
0.042	-2.17220E-01	7.52798E+00	7.58229E+00	7.41932E+00	7.58183E+00	7.64760E-01	6.64380E-02	    -1.74850E-01	6.20000E+00	-1.12920E+00	1.87970E-01	-1.23050E-02	4.50000E+00	1.00000E+00	4.29000E+00	-4.82900E-01	2.00000E+02	1.50334E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.71930E-01	-6.87000E-03	5.99440E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94139E-01
0.044	-2.23390E-01	7.54591E+00	7.60007E+00	7.43781E+00	7.59906E+00	7.63150E-01	6.66630E-02	    -1.76190E-01	6.20000E+00	-1.12590E+00	1.87750E-01	-1.24300E-02	4.50000E+00	1.00000E+00	4.27000E+00	-4.75720E-01	2.00000E+02	1.50313E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.76640E-01	-6.77000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
0.045	-2.26680E-01	7.55524E+00	7.60934E+00	7.44727E+00	7.60823E+00	7.62370E-01	6.67740E-02	    -1.76930E-01	6.20000E+00	-1.12420E+00	1.87640E-01	-1.24810E-02	4.50000E+00	1.00000E+00	4.25000E+00	-4.72360E-01	2.00000E+02	1.50284E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.79140E-01	-6.72000E-03	6.01020E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.95503E-01
0.046	-2.29890E-01	7.56469E+00	7.61877E+00	7.45677E+00	7.61764E+00	7.61610E-01	6.68910E-02	    -1.77690E-01	6.20000E+00	-1.12240E+00	1.87520E-01	-1.25410E-02	4.50000E+00	1.00000E+00	4.24000E+00	-4.69150E-01	2.00000E+02	1.50247E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.81700E-01	-6.67000E-03	6.01820E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.96195E-01
0.048	-2.33060E-01	7.58224E+00	7.63638E+00	7.47410E+00	7.63549E+00	7.60080E-01	6.71270E-02	    -1.79200E-01	6.20000E+00	-1.11920E+00	1.87300E-01	-1.27160E-02	4.50000E+00	1.00000E+00	4.22000E+00	-4.63210E-01	2.00000E+02	1.50201E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.86880E-01	-6.56000E-03	6.02700E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.96956E-01
0.050	-2.35540E-01	7.60045E+00	7.65478E+00	7.49178E+00	7.65427E+00	7.58570E-01	6.73570E-02	    -1.80820E-01	6.20000E+00	-1.11590E+00	1.87090E-01	-1.28790E-02	4.50000E+00	1.00000E+00	4.20000E+00	-4.57950E-01	2.00000E+02	1.50142E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.92000E-01	-6.47000E-03	6.03610E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.97743E-01
0.055	-2.40660E-01	7.64417E+00	7.69951E+00	7.53273E+00	7.70087E+00	7.55060E-01	6.77970E-02	    -1.84800E-01	6.20000E+00	-1.10820E+00	1.86550E-01	-1.32200E-02	4.50000E+00	1.00000E+00	4.15000E+00	-4.47870E-01	2.00000E+02	1.50071E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.03690E-01	-6.25000E-03	6.04370E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.98400E-01
0.060	-2.42520E-01	7.68700E+00	7.74445E+00	7.57044E+00	7.74797E+00	7.54000E-01	6.85910E-02	    -1.88580E-01	6.20000E+00	-1.10090E+00	1.85820E-01	-1.34900E-02	4.50000E+00	1.00000E+00	4.11000E+00	-4.41860E-01	2.00000E+02	1.49983E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.13740E-01	-6.07000E-03	6.05030E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.98972E-01
0.065	-2.42660E-01	7.72731E+00	7.78733E+00	7.60404E+00	7.79467E+00	7.54000E-01	7.01270E-02	    -1.91760E-01	6.20000E+00	-1.09420E+00	1.84850E-01	-1.37300E-02	4.50000E+00	1.00000E+00	4.08000E+00	-4.39510E-01	2.00000E+02	1.49874E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.22250E-01	-5.93000E-03	6.05690E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.99543E-01
0.067	-2.42200E-01	7.74242E+00	7.80349E+00	7.61626E+00	7.81274E+00	7.54000E-01	7.08950E-02	    -1.92910E-01	6.20000E+00	-1.09180E+00	1.84420E-01	-1.38800E-02	4.50000E+00	1.00000E+00	4.07000E+00	-4.39500E-01	2.00000E+02	1.49742E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.25240E-01	-5.88000E-03	6.06280E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00054E-01
0.070	-2.41530E-01	7.76453E+00	7.82725E+00	7.63375E+00	7.83970E+00	7.54000E-01	7.20750E-02	    -1.94510E-01	6.20000E+00	-1.08840E+00	1.83690E-01	-1.40700E-02	4.50000E+00	1.00000E+00	4.06000E+00	-4.40400E-01	2.00000E+02	1.49585E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.29310E-01	-5.82000E-03	6.06760E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00470E-01
0.075	-2.41440E-01	7.79940E+00	7.86486E+00	7.66058E+00	7.88358E+00	7.54000E-01	7.35490E-02	    -1.96650E-01	6.20000E+00	-1.08310E+00	1.82250E-01	-1.42600E-02	4.50000E+00	1.00000E+00	4.04000E+00	-4.44110E-01	2.00000E+02	1.49400E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.35000E-01	-5.73000E-03	6.07310E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00946E-01
0.080	-2.41920E-01	7.83195E+00	7.90020E+00	7.68490E+00	7.92532E+00	7.53860E-01	7.37350E-02	    -1.98160E-01	6.20000E+00	-1.07850E+00	1.80520E-01	-1.44300E-02	4.50000E+00	1.00000E+00	4.02000E+00	-4.50200E-01	2.00000E+02	1.49182E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.39440E-01	-5.67000E-03	6.07510E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.01119E-01
0.085	-2.42790E-01	7.86240E+00	7.93352E+00	7.70704E+00	7.96491E+00	7.51990E-01	7.19400E-02	    -1.99020E-01	6.20000E+00	-1.07450E+00	1.78560E-01	-1.45800E-02	4.50000E+00	1.00000E+00	4.03000E+00	-4.58130E-01	2.00000E+02	1.48929E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.42850E-01	-5.63000E-03	6.07750E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.01327E-01
0.090	-2.41810E-01	7.89192E+00	7.96602E+00	7.72827E+00	8.00309E+00	7.50240E-01	6.80970E-02	    -1.99290E-01	6.20000E+00	-1.07090E+00	1.76430E-01	-1.47300E-02	4.50000E+00	1.00000E+00	4.07000E+00	-4.67320E-01	2.00000E+02	1.48636E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.45440E-01	-5.61000E-03	6.07800E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.01371E-01
0.095	-2.39180E-01	7.92046E+00	7.99736E+00	7.74912E+00	8.03948E+00	7.48570E-01	6.23270E-02	    -1.99000E-01	6.20000E+00	-1.06780E+00	1.74200E-01	-1.48800E-02	4.50000E+00	1.00000E+00	4.10000E+00	-4.77210E-01	2.00000E+02	1.48298E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.47470E-01	-5.60000E-03	6.07680E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.01267E-01
0.100	-2.35340E-01	7.94685E+00	8.02619E+00	7.76919E+00	8.07188E+00	7.47000E-01	5.52310E-02	    -1.98380E-01	6.20000E+00	-1.06520E+00	1.72030E-01	-1.49900E-02	4.50000E+00	1.00000E+00	4.13000E+00	-4.87240E-01	2.00000E+02	1.47912E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.49160E-01	-5.60000E-03	6.07340E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00972E-01
0.110	-2.23590E-01	7.99540E+00	8.07798E+00	7.80911E+00	8.12885E+00	7.47000E-01	3.73890E-02	    -1.96010E-01	6.20000E+00	-1.06070E+00	1.67700E-01	-1.49740E-02	4.50000E+00	1.00000E+00	4.19000E+00	-5.06320E-01	2.00000E+02	1.47474E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.52130E-01	-5.62000E-03	6.07060E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00730E-01
0.120	-2.09160E-01	8.03863E+00	8.12290E+00	7.84825E+00	8.17547E+00	7.47000E-01	1.63730E-02	    -1.92650E-01	6.20000E+00	-1.05720E+00	1.63520E-01	-1.49420E-02	4.50000E+00	1.00000E+00	4.24000E+00	-5.24380E-01	2.00000E+02	1.46975E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.54550E-01	-5.67000E-03	6.07040E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00712E-01
0.130	-1.94250E-01	8.07552E+00	8.16007E+00	7.88501E+00	8.21164E+00	7.47000E-01	-5.15800E-03	-1.88980E-01	6.20000E+00	-1.05490E+00	1.59820E-01	-1.48860E-02	4.50000E+00	1.00000E+00	4.29000E+00	-5.42140E-01	2.00000E+02	1.46409E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.56280E-01	-5.72000E-03	6.07070E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00738E-01
0.133	-1.90000E-01	8.08538E+00	8.16986E+00	7.89532E+00	8.22069E+00	7.47000E-01	-1.13540E-02	-1.87920E-01	6.20000E+00	-1.05450E+00	1.58820E-01	-1.49720E-02	4.50000E+00	1.00000E+00	4.30000E+00	-5.47520E-01	2.00000E+02	1.45776E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.56650E-01	-5.74000E-03	6.07550E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.01154E-01
0.140	-1.81500E-01	8.10538E+00	8.18910E+00	7.91762E+00	8.23795E+00	7.47000E-01	-2.47110E-02	-1.85660E-01	6.20000E+00	-1.05370E+00	1.56720E-01	-1.49280E-02	4.50000E+00	1.00000E+00	4.34000E+00	-5.60320E-01	2.00000E+02	1.45071E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.57190E-01	-5.78000E-03	6.08680E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.02133E-01
0.150	-1.73830E-01	8.12908E+00	8.21118E+00	7.94615E+00	8.25620E+00	7.47000E-01	-4.20650E-02	-1.82340E-01	6.20000E+00	-1.05320E+00	1.54010E-01	-1.47670E-02	4.50000E+00	1.00000E+00	4.39000E+00	-5.79620E-01	2.00000E+02	1.44285E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.57130E-01	-5.85000E-03	6.09900E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.03191E-01
0.160	-1.66580E-01	8.14563E+00	8.22550E+00	7.96869E+00	8.26681E+00	7.47000E-01	-5.75930E-02	-1.78530E-01	6.20000E+00	-1.05330E+00	1.51580E-01	-1.45750E-02	4.50000E+00	1.00000E+00	4.44000E+00	-6.00520E-01	2.00000E+02	1.43422E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.56040E-01	-5.91000E-03	6.11310E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.04415E-01
0.170	-1.60040E-01	8.15546E+00	8.23275E+00	7.98485E+00	8.27116E+00	7.47580E-01	-7.18610E-02	-1.74210E-01	6.20000E+00	-1.05410E+00	1.49480E-01	-1.43420E-02	4.50000E+00	1.00000E+00	4.49000E+00	-6.22520E-01	2.00000E+02	1.42485E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.54140E-01	-5.97000E-03	6.12660E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.05586E-01
0.180	-1.56350E-01	8.15996E+00	8.23456E+00	7.99553E+00	8.27105E+00	7.48460E-01	-8.56400E-02	-1.69390E-01	6.20000E+00	-1.05560E+00	1.47680E-01	-1.41190E-02	4.50000E+00	1.00000E+00	4.53000E+00	-6.44860E-01	2.00000E+02	1.41477E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.51730E-01	-6.02000E-03	6.13980E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.06733E-01
0.190	-1.52330E-01	8.16004E+00	8.23203E+00	8.00124E+00	8.26757E+00	7.49290E-01	-9.88840E-02	-1.64040E-01	6.20000E+00	-1.05790E+00	1.46160E-01	-1.38770E-02	4.50000E+00	1.00000E+00	4.57000E+00	-6.66810E-01	2.00000E+02	1.40399E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.49110E-01	-6.08000E-03	6.15120E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.07724E-01
0.200	-1.46160E-01	8.15654E+00	8.22616E+00	8.00231E+00	8.26213E+00	7.50080E-01	-1.10960E-01	-1.58520E-01	6.20000E+00	-1.06070E+00	1.44890E-01	-1.36470E-02	4.50000E+00	1.00000E+00	4.61000E+00	-6.87620E-01	2.00000E+02	1.39261E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.46580E-01	-6.14000E-03	6.16300E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.08749E-01
0.220	-1.27960E-01	8.14208E+00	8.20791E+00	7.99337E+00	8.24787E+00	7.52130E-01	-1.33000E-01	-1.47040E-01	6.20000E+00	-1.06700E+00	1.42630E-01	-1.31640E-02	4.50000E+00	1.00000E+00	4.68000E+00	-7.24310E-01	2.00000E+02	1.38072E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.42350E-01	-6.26000E-03	6.17860E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.10106E-01
0.240	-1.12800E-01	8.12131E+00	8.18474E+00	7.97435E+00	8.23217E+00	7.54230E-01	-1.52990E-01	-1.34450E-01	6.20000E+00	-1.07370E+00	1.40350E-01	-1.26870E-02	4.50000E+00	1.00000E+00	4.75000E+00	-7.56460E-01	2.00000E+02	1.36851E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.38230E-01	-6.38000E-03	6.19500E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.11534E-01
0.250	-1.11480E-01	8.10948E+00	8.17202E+00	7.96255E+00	8.22380E+00	7.55210E-01	-1.62130E-01	-1.27840E-01	6.20000E+00	-1.07730E+00	1.39250E-01	-1.24670E-02	4.50000E+00	1.00000E+00	4.78000E+00	-7.71770E-01	2.00000E+02	1.35621E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.35740E-01	-6.44000E-03	6.21090E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.12919E-01
0.260	-1.10200E-01	8.09775E+00	8.15951E+00	7.95069E+00	8.21544E+00	7.56150E-01	-1.70410E-01	-1.21150E-01	6.20000E+00	-1.08080E+00	1.38180E-01	-1.22430E-02	4.50000E+00	1.00000E+00	4.82000E+00	-7.86970E-01	2.00000E+02	1.34389E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.32800E-01	-6.50000E-03	6.22240E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.13921E-01
0.280	-1.07680E-01	8.07500E+00	8.13552E+00	7.92729E+00	8.19906E+00	7.58500E-01	-1.84630E-01	-1.07140E-01	6.20000E+00	-1.08790E+00	1.36040E-01	-1.17960E-02	4.50000E+00	1.00000E+00	4.88000E+00	-8.16130E-01	2.00000E+02	1.33167E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.26010E-01	-6.60000E-03	6.22810E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.14417E-01
0.290	-1.06460E-01	8.06410E+00	8.12425E+00	7.91599E+00	8.19084E+00	7.60450E-01	-1.90570E-01	-1.00110E-01	6.20000E+00	-1.09130E+00	1.34990E-01	-1.15860E-02	4.50000E+00	1.00000E+00	4.90000E+00	-8.29500E-01	2.00000E+02	1.31983E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.22500E-01	-6.65000E-03	6.23440E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.14967E-01
0.300	-1.07200E-01	8.05315E+00	8.11317E+00	7.90429E+00	8.18223E+00	7.62320E-01	-1.95900E-01	-9.28550E-02	6.20000E+00	-1.09480E+00	1.33880E-01	-1.13650E-02	4.50000E+00	1.00000E+00	4.93000E+00	-8.41650E-01	2.00000E+02	1.30847E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.19120E-01	-6.70000E-03	6.24750E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.16109E-01
0.320	-1.11700E-01	8.03149E+00	8.09174E+00	7.88079E+00	8.16416E+00	7.65890E-01	-2.04540E-01	-7.89230E-02	6.20000E+00	-1.10130E+00	1.31790E-01	-1.09520E-02	4.50000E+00	1.00000E+00	4.98000E+00	-8.61750E-01	2.00000E+02	1.29765E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.13180E-01	-6.80000E-03	6.26560E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.17689E-01
0.340	-1.20200E-01	8.00941E+00	8.07046E+00	7.85683E+00	8.14359E+00	7.70270E-01	-2.11340E-01	-6.51340E-02	6.20000E+00	-1.10740E+00	1.29840E-01	-1.05580E-02	4.50000E+00	1.00000E+00	5.03000E+00	-8.77260E-01	2.00000E+02	1.28750E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.08160E-01	-6.89000E-03	6.29070E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.19881E-01
0.350	-1.25140E-01	7.99754E+00	8.05899E+00	7.84415E+00	8.13209E+00	7.73190E-01	-2.14460E-01	-5.79210E-02	6.20000E+00	-1.11050E+00	1.28900E-01	-1.03130E-02	4.50000E+00	1.00000E+00	5.06000E+00	-8.83750E-01	2.00000E+02	1.27806E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.05920E-01	-6.93000E-03	6.32050E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.22487E-01
0.360	-1.27500E-01	7.98591E+00	8.04782E+00	7.83182E+00	8.12040E+00	7.76030E-01	-2.17160E-01	-5.10400E-02	6.20000E+00	-1.11330E+00	1.28060E-01	-1.00670E-02	4.50000E+00	1.00000E+00	5.08000E+00	-8.89650E-01	2.00000E+02	1.26919E+03	7.60000E+02	0.00000E+00	9.81000E+02	-2.03790E-01	-6.97000E-03	6.35170E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.25218E-01
0.380	-1.31890E-01	7.96066E+00	8.02343E+00	7.80555E+00	8.09426E+00	7.81470E-01	-2.22140E-01	-3.67550E-02	6.20000E+00	-1.11900E+00	1.26470E-01	-9.69600E-03	4.50000E+00	1.00000E+00	5.12000E+00	-9.00380E-01	2.00000E+02	1.26074E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.99780E-01	-7.05000E-03	6.37940E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.27645E-01
0.400	-1.35350E-01	7.93511E+00	7.99881E+00	7.77930E+00	8.06679E+00	7.90490E-01	-2.26080E-01	-2.31890E-02	6.20000E+00	-1.12430E+00	1.25120E-01	-9.35300E-03	4.50000E+00	1.00000E+00	5.16000E+00	-9.10920E-01	2.00000E+02	1.25266E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.95820E-01	-7.13000E-03	6.40450E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.29847E-01
0.420	-1.37900E-01	7.90891E+00	7.97345E+00	7.75277E+00	8.03817E+00	8.01980E-01	-2.29240E-01	-1.04170E-02	6.20000E+00	-1.12910E+00	1.23890E-01	-9.04300E-03	4.50000E+00	1.00000E+00	5.20000E+00	-9.22410E-01	2.00000E+02	1.24480E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.91710E-01	-7.19000E-03	6.42860E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.31962E-01
0.440	-1.39960E-01	7.88249E+00	7.94806E+00	7.72577E+00	8.00915E+00	8.16700E-01	-2.31660E-01	1.16780E-03	    6.20000E+00	-1.13370E+00	1.22780E-01	-8.74300E-03	4.50000E+00	1.00000E+00	5.24000E+00	-9.34590E-01	2.00000E+02	1.23703E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.87470E-01	-7.26000E-03	6.46150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.34854E-01
0.450	-1.40690E-01	7.86869E+00	7.93476E+00	7.71169E+00	7.99404E+00	8.23810E-01	-2.32630E-01	6.58870E-03	    6.20000E+00	-1.13590E+00	1.22270E-01	-8.53520E-03	4.50000E+00	1.00000E+00	5.25000E+00	-9.40750E-01	2.00000E+02	1.22923E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.85340E-01	-7.29000E-03	6.50590E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.38761E-01
0.460	-1.41400E-01	7.85473E+00	7.92136E+00	7.69742E+00	7.97865E+00	8.31380E-01	-2.33500E-01	1.18710E-02	    6.20000E+00	-1.13810E+00	1.21770E-01	-8.39110E-03	4.50000E+00	1.00000E+00	5.27000E+00	-9.46860E-01	2.00000E+02	1.22116E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.83210E-01	-7.32000E-03	6.55030E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.42674E-01
0.480	-1.42790E-01	7.82915E+00	7.89634E+00	7.67104E+00	7.94997E+00	8.51510E-01	-2.34640E-01	2.07670E-02	    6.20000E+00	-1.14200E+00	1.20930E-01	-8.08590E-03	4.50000E+00	1.00000E+00	5.30000E+00	-9.58630E-01	2.00000E+02	1.21274E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.79020E-01	-7.38000E-03	6.58880E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.46072E-01
0.500	-1.43420E-01	7.80239E+00	7.87057E+00	7.64404E+00	7.91992E+00	8.72390E-01	-2.35220E-01	2.91190E-02	    6.20000E+00	-1.14590E+00	1.20150E-01	-7.83120E-03	4.50000E+00	1.00000E+00	5.34000E+00	-9.69300E-01	2.00000E+02	1.20391E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.75000E-01	-7.44000E-03	6.62500E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.49270E-01
0.550	-1.44640E-01	7.73884E+00	7.80867E+00	7.58118E+00	7.84834E+00	9.22300E-01	-2.34490E-01	4.69320E-02	    6.20000E+00	-1.15430E+00	1.18470E-01	-7.20820E-03	4.50000E+00	1.00000E+00	5.41000E+00	-9.89250E-01	2.00000E+02	1.19459E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.66010E-01	-7.58000E-03	6.66140E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.52491E-01
0.600	-1.44640E-01	7.67738E+00	7.74830E+00	7.52207E+00	7.77783E+00	9.77590E-01	-2.31280E-01	6.26670E-02	    6.20000E+00	-1.16150E+00	1.16710E-01	-6.64340E-03	4.50000E+00	1.00000E+00	5.48000E+00	-1.00120E+00	2.00000E+02	1.18493E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.58300E-01	-7.73000E-03	6.69910E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.55830E-01
0.650	-1.46830E-01	7.61903E+00	7.69063E+00	7.46644E+00	7.71057E+00	1.03510E+00	-2.26660E-01	7.79970E-02	    6.20000E+00	-1.16760E+00	1.14650E-01	-6.17160E-03	4.50000E+00	1.00000E+00	5.53000E+00	-1.00780E+00	2.00000E+02	1.17519E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.51440E-01	-7.87000E-03	6.74090E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.59538E-01
0.667	-1.47560E-01	7.60032E+00	7.67204E+00	7.44861E+00	7.68920E+00	1.05750E+00	-2.24970E-01	8.30580E-02	    6.20000E+00	-1.16940E+00	1.13940E-01	-6.01440E-03	4.50000E+00	1.00000E+00	5.54000E+00	-1.00930E+00	2.00000E+02	1.16569E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.49230E-01	-7.92000E-03	6.78650E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.63587E-01
0.700	-1.50130E-01	7.56375E+00	7.63566E+00	7.41365E+00	7.64794E+00	1.09450E+00	-2.21430E-01	9.31850E-02	    6.20000E+00	-1.17280E+00	1.12530E-01	-5.72740E-03	4.50000E+00	1.00000E+00	5.56000E+00	-1.01170E+00	2.00000E+02	1.15646E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.45030E-01	-8.00000E-03	6.83000E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.67456E-01
0.750	-1.55070E-01	7.50901E+00	7.58149E+00	7.36080E+00	7.58703E+00	1.14510E+00	-2.15910E-01	1.08290E-01	    6.20000E+00	-1.17770E+00	1.10540E-01	-5.32110E-03	4.50000E+00	1.00000E+00	5.60000E+00	-1.01540E+00	2.00000E+02	1.14759E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.38660E-01	-8.12000E-03	6.87770E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.71704E-01
0.800	-1.61300E-01	7.45437E+00	7.52746E+00	7.30795E+00	7.52703E+00	1.19510E+00	-2.10470E-01	1.22560E-01	    6.20000E+00	-1.18190E+00	1.08730E-01	-4.87370E-03	4.50000E+00	1.00000E+00	5.63000E+00	-1.02100E+00	2.00000E+02	1.13921E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.32010E-01	-8.22000E-03	6.92830E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.76217E-01
0.850	-1.68190E-01	7.40023E+00	7.47378E+00	7.25496E+00	7.46814E+00	1.24100E+00	-2.05280E-01	1.36080E-01	    6.20000E+00	-1.18540E+00	1.07090E-01	-4.48790E-03	4.50000E+00	1.00000E+00	5.66000E+00	-1.02820E+00	2.00000E+02	1.13134E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.25170E-01	-8.30000E-03	6.96850E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.79808E-01
0.900	-1.72500E-01	7.34543E+00	7.41938E+00	7.20116E+00	7.40920E+00	1.28680E+00	-2.00110E-01	1.49830E-01	    6.20000E+00	-1.18840E+00	1.05480E-01	-4.09380E-03	4.50000E+00	1.00000E+00	5.69000E+00	-1.03600E+00	2.00000E+02	1.12391E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.18310E-01	-8.36000E-03	7.00420E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.82999E-01
0.950	-1.76160E-01	7.29092E+00	7.36459E+00	7.14726E+00	7.35087E+00	1.33130E+00	-1.94900E-01	1.64320E-01	    6.20000E+00	-1.19090E+00	1.03890E-01	-3.82480E-03	4.50000E+00	1.00000E+00	5.72000E+00	-1.04360E+00	2.00000E+02	1.11683E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.11600E-01	-8.41000E-03	7.04580E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.86723E-01
1.000	-1.78740E-01	7.23826E+00	7.31208E+00	7.09583E+00	7.29452E+00	1.37240E+00	-1.89830E-01	1.78950E-01	    6.20000E+00	-1.19300E+00	1.02480E-01	-3.56320E-03	4.50000E+00	1.00000E+00	5.74000E+00	-1.05000E+00	2.00000E+02	1.10995E+03	7.60000E+02	0.00000E+00	9.81000E+02	-1.05210E-01	-8.44000E-03	7.08270E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.90029E-01
1.100	-1.81470E-01	7.13200E+00	7.20613E+00	6.99207E+00	7.18092E+00	1.44200E+00	-1.80010E-01	2.10420E-01	    6.20000E+00	-1.19660E+00	1.00160E-01	-3.13370E-03	4.50000E+00	1.00000E+00	5.82000E+00	-1.05730E+00	2.00000E+02	1.10307E+03	7.60000E+02	0.00000E+00	9.81000E+02	-9.38370E-02	-8.47000E-03	7.10940E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.92424E-01
1.200	-1.83610E-01	7.02303E+00	7.09729E+00	6.88535E+00	7.06355E+00	1.50120E+00	-1.70900E-01	2.44100E-01	    6.20000E+00	-1.19960E+00	9.84820E-02	-2.68300E-03	4.50000E+00	1.00000E+00	5.92000E+00	-1.05840E+00	2.00000E+02	1.09604E+03	7.60000E+02	0.00000E+00	9.81000E+02	-8.41440E-02	-8.42000E-03	7.12440E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.93770E-01
1.300	-1.90620E-01	6.91371E+00	6.98789E+00	6.77929E+00	6.94561E+00	1.55240E+00	-1.62330E-01	2.77990E-01	    6.20000E+00	-1.20180E+00	9.73750E-02	-2.28500E-03	4.50000E+00	1.00000E+00	6.01000E+00	-1.05540E+00	2.00000E+02	1.08867E+03	7.60000E+02	0.00000E+00	9.81000E+02	-7.58190E-02	-8.29000E-03	7.13220E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.94470E-01
1.400	-1.95360E-01	6.80873E+00	6.88310E+00	6.67848E+00	6.83186E+00	1.59590E+00	-1.54130E-01	3.09560E-01	    6.20000E+00	-1.20390E+00	9.67430E-02	-1.91000E-03	4.50000E+00	1.00000E+00	6.10000E+00	-1.05040E+00	2.00000E+02	1.08077E+03	7.60000E+02	0.00000E+00	9.81000E+02	-6.85430E-02	-8.06000E-03	7.13860E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.95045E-01
1.500	-2.02080E-01	6.70778E+00	6.78197E+00	6.58145E+00	6.72114E+00	1.63320E+00	-1.46700E-01	3.38960E-01	    6.20000E+00	-1.20630E+00	9.64450E-02	-1.54500E-03	4.50000E+00	1.00000E+00	6.18000E+00	-1.04540E+00	2.00000E+02	1.07239E+03	7.60000E+02	0.00000E+00	9.81000E+02	-6.20000E-02	-7.71000E-03	7.14840E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.95925E-01
1.600	-2.07620E-01	6.61151E+00	6.68554E+00	6.48962E+00	6.61487E+00	1.66550E+00	-1.39970E-01	3.66160E-01	    6.20000E+00	-1.20860E+00	9.63380E-02	-1.22000E-03	4.50000E+00	1.00000E+00	6.26000E+00	-1.04210E+00	2.00000E+02	1.06177E+03	7.60000E+02	0.00000E+00	9.81000E+02	-5.59270E-02	-7.23000E-03	7.14970E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.96042E-01
1.700	-2.14550E-01	6.52186E+00	6.59527E+00	6.40459E+00	6.51458E+00	1.69090E+00	-1.33610E-01	3.90650E-01	    6.20000E+00	-1.21060E+00	9.62540E-02	-8.95000E-04	4.50000E+00	1.00000E+00	6.33000E+00	-1.04040E+00	2.00000E+02	1.04929E+03	7.60000E+02	0.00000E+00	9.81000E+02	-5.02860E-02	-6.66000E-03	7.13360E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.94596E-01
1.800	-2.21870E-01	6.43631E+00	6.50942E+00	6.32465E+00	6.41488E+00	1.71440E+00	-1.26860E-01	4.12440E-01	    6.20000E+00	-1.21230E+00	9.62070E-02	-5.80000E-04	4.50000E+00	1.00000E+00	6.40000E+00	-1.03970E+00	2.00000E+02	1.03642E+03	7.60000E+02	0.00000E+00	9.81000E+02	-4.50960E-02	-6.03000E-03	7.11470E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.92899E-01
1.900	-2.34370E-01	6.35626E+00	6.42894E+00	6.25048E+00	6.31690E+00	1.73480E+00	-1.19590E-01	4.31510E-01	    6.20000E+00	-1.21410E+00	9.62550E-02	-4.27590E-04	4.50000E+00	1.00000E+00	6.48000E+00	-1.03950E+00	2.00000E+02	1.02314E+03	7.60000E+02	0.00000E+00	9.81000E+02	-4.03730E-02	-5.40000E-03	7.09480E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.91114E-01
2.000	-2.58400E-01	6.28552E+00	6.35742E+00	6.18569E+00	6.22935E+00	1.75220E+00	-1.12370E-01	4.47880E-01	    6.20000E+00	-1.21590E+00	9.63610E-02	-2.95680E-04	4.50000E+00	1.00000E+00	6.54000E+00	-1.03920E+00	2.00000E+02	1.00949E+03	7.60000E+02	0.00000E+00	9.81000E+02	-3.61360E-02	-4.79000E-03	7.05870E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.87878E-01
2.200	-2.83550E-01	6.15782E+00	6.22746E+00	6.07170E+00	6.07118E+00	1.79110E+00	-9.80170E-02	4.80240E-01	    6.20000E+00	-1.21900E+00	9.64970E-02	-1.59820E-04	4.50000E+00	1.00000E+00	6.66000E+00	-1.03680E+00	2.00000E+02	9.95520E+02	7.60000E+02	0.00000E+00	9.81000E+02	-2.91050E-02	-3.78000E-03	7.01900E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.84324E-01
2.400	-3.45200E-01	6.04054E+00	6.10432E+00	5.97004E+00	5.92368E+00	1.82700E+00	-8.37650E-02	5.18730E-01	    6.20000E+00	-1.22020E+00	9.61980E-02	-5.16170E-05	4.50000E+00	1.00000E+00	6.73000E+00	-1.03230E+00	2.00000E+02	9.81330E+02	7.60000E+02	0.00000E+00	9.81000E+02	-2.37100E-02	-3.02000E-03	6.98070E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.80898E-01
2.500	-3.72450E-01	5.98295E+00	6.04425E+00	5.92079E+00	5.85306E+00	1.83660E+00	-7.63080E-02	5.38830E-01	    6.20000E+00	-1.22010E+00	9.61060E-02	-1.11460E-05	4.50000E+00	1.00000E+00	6.77000E+00	-1.02940E+00	2.00000E+02	9.66940E+02	7.60000E+02	0.00000E+00	9.81000E+02	-2.15090E-02	-2.72000E-03	6.94170E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.77414E-01
2.600	-3.91550E-01	5.92732E+00	5.98678E+00	5.87219E+00	5.78560E+00	1.84580E+00	-6.89250E-02	5.58100E-01	    6.20000E+00	-1.21980E+00	9.61360E-02	2.53950E-05     4.50000E+00	1.00000E+00	6.81000E+00	-1.02620E+00	2.00000E+02	9.52340E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.95760E-02	-2.46000E-03	6.90250E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.73915E-01
2.800	-4.23710E-01	5.81948E+00	5.87708E+00	5.77417E+00	5.65695E+00	1.86940E+00	-5.52290E-02	5.93940E-01	    6.20000E+00	-1.21890E+00	9.66670E-02	9.44420E-05     4.50000E+00	1.00000E+00	6.87000E+00	-1.01900E+00	2.00000E+02	9.37520E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.63240E-02	-2.08000E-03	6.85890E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.70029E-01
3.000	-4.52200E-01	5.71560E+00	5.77277E+00	5.67730E+00	5.53445E+00	1.88830E+00	-4.33200E-02	6.26940E-01	    6.20000E+00	-1.21790E+00	9.76380E-02	1.56160E-04     4.50000E+00	1.00000E+00	6.93000E+00	-1.01120E+00	2.00000E+02	9.22430E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.35770E-02	-1.83000E-03	6.80800E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.65499E-01
3.200	-4.75460E-01	5.61723E+00	5.67289E+00	5.58559E+00	5.41975E+00	1.90460E+00	-3.44400E-02	6.58110E-01	    6.20000E+00	-1.21690E+00	9.86490E-02	1.70370E-04     4.50000E+00	1.00000E+00	6.99000E+00	-1.00320E+00	2.00000E+02	9.08790E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.10300E-02	-1.67000E-03	6.75320E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.60629E-01
3.400	-4.97440E-01	5.52518E+00	5.58009E+00	5.49966E+00	5.31291E+00	1.91070E+00	-2.78890E-02	6.87550E-01	    6.20000E+00	-1.21600E+00	9.95530E-02	1.88330E-04     4.50000E+00	1.00000E+00	7.08000E+00	-9.95120E-01	2.00000E+02	8.96150E+02	7.60000E+02	0.00000E+00	9.81000E+02	-8.66520E-03	-1.58000E-03	6.69240E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.55237E-01
3.500	-5.07530E-01	5.47926E+00	5.53392E+00	5.45646E+00	5.25979E+00	1.91370E+00	-2.49970E-02	7.02160E-01	    6.20000E+00	-1.21560E+00	9.99890E-02	2.00040E-04     4.50000E+00	1.00000E+00	7.12000E+00	-9.91000E-01	2.00000E+02	8.83160E+02	7.60000E+02	0.00000E+00	9.81000E+02	-7.56820E-03	-1.55000E-03	6.62240E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.49041E-01
3.600	-5.15470E-01	5.43724E+00	5.49186E+00	5.41662E+00	5.21112E+00	1.91620E+00	-2.25750E-02	7.15230E-01	    6.20000E+00	-1.21560E+00	1.00430E-01	2.12290E-04     4.50000E+00	1.00000E+00	7.16000E+00	-9.86820E-01	2.00000E+02	8.70050E+02	7.60000E+02	0.00000E+00	9.81000E+02	-6.53740E-03	-1.54000E-03	6.54710E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.42392E-01
3.800	-5.31640E-01	5.35520E+00	5.40869E+00	5.33915E+00	5.11569E+00	1.91850E+00	-1.83620E-02	7.40280E-01	    6.20000E+00	-1.21580E+00	1.01420E-01	2.55860E-04     4.50000E+00	1.00000E+00	7.24000E+00	-9.78260E-01	2.00000E+02	8.57070E+02	7.60000E+02	0.00000E+00	9.81000E+02	-4.70160E-03	-1.52000E-03	6.47430E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.35979E-01
4.000	-5.46640E-01	5.28049E+00	5.33428E+00	5.26568E+00	5.02996E+00	1.92070E+00	-1.46420E-02	7.63030E-01	    6.20000E+00	-1.21620E+00	1.02180E-01	3.17780E-04     4.50000E+00	1.00000E+00	7.32000E+00	-9.69380E-01	2.00000E+02	8.44480E+02	7.60000E+02	0.00000E+00	9.81000E+02	-3.21230E-03	-1.52000E-03	6.39710E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.29197E-01
4.200	-5.61050E-01	5.20664E+00	5.25979E+00	5.19486E+00	4.95007E+00	1.92280E+00	-1.22480E-02	7.85520E-01	    6.20000E+00	-1.21650E+00	1.02690E-01	3.80730E-04     4.50000E+00	1.00000E+00	7.39000E+00	-9.60120E-01	2.00000E+02	8.32450E+02	7.60000E+02	0.00000E+00	9.81000E+02	-2.10330E-03	-1.52000E-03	6.29940E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.20642E-01
4.400	-5.75250E-01	5.13672E+00	5.18835E+00	5.12904E+00	4.87558E+00	1.92300E+00	-1.14590E-02	8.07920E-01	    6.20000E+00	-1.21690E+00	1.03040E-01	4.43990E-04     4.50000E+00	1.00000E+00	7.46000E+00	-9.50490E-01	2.00000E+02	8.21180E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.32440E-03	-1.50000E-03	6.25370E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.16650E-01
4.600	-5.93030E-01	5.06822E+00	5.11800E+00	5.06508E+00	4.80321E+00	1.92300E+00	-1.17600E-02	8.31260E-01	    6.20000E+00	-1.21750E+00	1.03240E-01	5.17830E-04     4.50000E+00	1.00000E+00	7.52000E+00	-9.40500E-01	2.00000E+02	8.10790E+02	7.60000E+02	0.00000E+00	9.81000E+02	-8.04220E-04	-1.48000E-03	6.21360E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.13154E-01
4.800	-6.10580E-01	5.00766E+00	5.05491E+00	5.00980E+00	4.73963E+00	1.92290E+00	-1.28790E-02	8.52400E-01	    6.20000E+00	-1.21820E+00	1.03370E-01	5.90030E-04     4.50000E+00	1.00000E+00	7.64000E+00	-9.30180E-01	2.00000E+02	8.01410E+02	7.60000E+02	0.00000E+00	9.81000E+02	-4.71450E-04	-1.46000E-03	6.17520E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.09811E-01
5.000	-6.27420E-01	4.94686E+00	4.99136E+00	4.95435E+00	4.67734E+00	1.91960E+00	-1.48550E-02	8.73140E-01	    6.20000E+00	-1.21890E+00	1.03530E-01	6.60810E-04     4.50000E+00	1.00000E+00	7.78000E+00	-9.19540E-01	2.00000E+02	7.93130E+02	7.60000E+02	0.00000E+00	9.81000E+02	-2.54800E-04	-1.44000E-03	6.15410E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.07976E-01
5.500	-6.61620E-01	4.80167E+00	4.83953E+00	4.81785E+00	4.53724E+00	1.90920E+00	-1.95020E-02	9.14660E-01	    6.20000E+00	-1.22040E+00	1.04600E-01	7.93930E-04     4.50000E+00	1.00000E+00	8.07000E+00	-8.91760E-01	2.00000E+02	7.85730E+02	7.60000E+02	0.00000E+00	9.81000E+02	 7.23830E-05	-1.40000E-03	6.11450E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.04536E-01
6.000	-6.86690E-01	4.65410E+00	4.69087E+00	4.66593E+00	4.40480E+00	1.88260E+00	-2.63830E-02	9.48700E-01	    6.20000E+00	-1.22320E+00	1.07500E-01	9.38220E-04     4.50000E+00	1.00000E+00	8.48000E+00	-8.62860E-01	2.00000E+02	7.79910E+02	7.60000E+02	0.00000E+00	9.81000E+02	 1.87700E-04	-1.38000E-03	6.06980E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	7.00660E-01
6.500	-7.11390E-01	4.51810E+00	4.55716E+00	4.51761E+00	4.28263E+00	1.85400E+00	-3.95050E-02	9.76430E-01	    6.20000E+00	-1.22990E+00	1.12310E-01	1.12580E-03     4.50000E+00	1.00000E+00	8.90000E+00	-8.33550E-01	2.00000E+02	7.75600E+02	7.60000E+02	0.00000E+00	9.81000E+02	 1.59450E-04	-1.37000E-03	6.03560E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.97700E-01
7.000	-7.39110E-01	4.39368E+00	4.43730E+00	4.38090E+00	4.17138E+00	1.80670E+00	-5.91400E-02	9.97570E-01	    6.20000E+00	-1.24080E+00	1.18530E-01	1.34960E-03     4.50000E+00	1.00000E+00	9.20000E+00	-8.04570E-01	2.00000E+02	7.72680E+02	7.60000E+02	0.00000E+00	9.81000E+02	 5.59200E-05	-1.37000E-03	6.01990E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.96342E-01
7.500	-7.68310E-01	4.28398E+00	4.32899E+00	4.26677E+00	4.06910E+00	1.75020E+00	-8.16060E-02	1.01210E+00	    6.20000E+00	-1.25430E+00	1.25070E-01	1.60400E-03     4.50000E+00	1.00000E+00	9.48000E+00	-7.76650E-01	2.00000E+02	7.71010E+02	7.60000E+02	0.00000E+00	9.81000E+02	-5.46000E-05	-1.37000E-03	6.00530E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.95080E-01
8.000	-7.95240E-01	4.17886E+00	4.22545E+00	4.15709E+00	3.97070E+00	1.68980E+00	-1.03820E-01	1.02320E+00	    6.20000E+00	-1.26880E+00	1.31460E-01	1.84110E-03     4.50000E+00	1.00000E+00	9.57000E+00	-7.50330E-01	2.00000E+02	7.60810E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.17240E-04	-1.37000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
8.500	-8.12930E-01	4.07734E+00	4.12728E+00	4.04838E+00	3.87399E+00	1.63390E+00	-1.21140E-01	1.03350E+00	    6.20000E+00	-1.28390E+00	1.37420E-01	2.00850E-03     4.50000E+00	1.00000E+00	9.62000E+00	-7.25440E-01	2.00000E+02	7.64500E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.30840E-04	-1.37000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
9.000	-8.28840E-01	3.97709E+00	4.03152E+00	3.94017E+00	3.77478E+00	1.58320E+00	-1.34070E-01	1.04530E+00	    6.20000E+00	-1.29890E+00	1.42940E-01	2.30770E-03     4.50000E+00	1.00000E+00	9.66000E+00	-7.01610E-01	2.00000E+02	7.68070E+02	7.60000E+02	0.00000E+00	9.81000E+02	-1.07660E-04	-1.37000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
9.500	-8.44170E-01	3.87672E+00	3.93468E+00	3.83390E+00	3.67390E+00	1.53890E+00	-1.43640E-01	1.05670E+00	    6.20000E+00	-1.31300E+00	1.47810E-01	2.57340E-03     4.50000E+00	1.00000E+00	9.66000E+00	-6.78500E-01	2.00000E+02	7.71550E+02	7.60000E+02	0.00000E+00	9.81000E+02	-5.99620E-05	-1.36000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
10.000	-8.59240E-01	3.78195E+00	3.84235E+00	3.73569E+00	3.57656E+00	1.50110E+00	-1.50960E-01	1.06510E+00	    6.20000E+00	-1.32530E+00	1.51830E-01	2.69860E-03     4.50000E+00	1.00000E+00	9.66000E+00	-6.55750E-01	2.00000E+02	7.75000E+02	7.60000E+02	0.00000E+00	9.81000E+02	 0.00000E+00	-1.36000E-03	6.00150E-01	5.00000E-01	5.50000E+00	6.00000E+00	3.50000E-01	6.94752E-01
""")
