# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2018-2023 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake.  If not, see <http://www.gnu.org/licenses/>.
"""
Module exports :class:`DrouetAlpes2015Rjb`
               :class:`DrouetAlpes2015Rrup`
               :class:`DrouetAlpes2015Repi`
               :class:`DrouetAlpes2015Rhyp`
               :class:`DrouetAlpes2015RjbHR`
               :class:`DrouetAlpes2015RrupHR`
               :class:`DrouetAlpes2015RepiHR`
               :class:`DrouetAlpes2015RhypHR`
               :class:`DrouetAlpes2015Rjb_50bars`
               :class:`DrouetAlpes2015Rrup_50bars`
               :class:`DrouetAlpes2015Repi_50bars`
               :class:`DrouetAlpes2015Rhypo_50bars`
               :class:`DrouetAlpes2015RjbHR_50bars`
               :class:`DrouetAlpes2015RrupHR_50bars`
"""
# 8 GMPEs for large magnitude stress parameter 100 bars (recommended
# by the authors)
# including 4 for standard rock conditions vs30=800 m/s, kappa=0.03 s
# and 4 GMPEs for hard rock conditions vs30=2000 m/s, kappa=0.01 s
# The coefficients are published in the original paper and erratum
# Additional 6 GMPEs for large magnitude stress parameter 50 bars
# The coefficients are not published
import numpy as np

from openquake.hazardlib.gsim.base import CoeffsTable, GMPE
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA
from scipy.constants import g


def _compute_mean(C, mag, r):
    """
    Compute mean value according to equation 30, page 1021.
    """
    return C['c1'] + _compute_term1(C, mag) + _compute_term2(C, mag, r)


def _compute_term1(C, mag):
    """
    This computes the term f1 equation 8 Drouet & Cotton (2015)
    """
    return C['c2'] * (mag - 8.0) + C['c3'] * (mag - 8.0) ** 2


def _compute_term2(C, mag, r):
    """
    This computes the term f2 equation 8 Drouet & Cotton (2015)
    """
    return (C['c4'] + C['c5'] * mag) * \
        np.log(np.sqrt(r**2 + C['c6']**2)) + C['c7'] * r


class DrouetAlpes2015Rjb(GMPE):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 100 bars
    (recommended by the authors)
    Valid for vs30=800 m/s
    """
    non_verified = True

    #: Supported tectonic region type is stable continental crust given that
    #: the equations have been derived for Eastern North America.
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types are spectral acceleration,
    #: and peak ground acceleration, see table 6, page 1022 (PGA is assumed
    #: to be equal to SA at 0.01 s)
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is the geometric mean of
    #: two : horizontal components
    #: :attr:`~openquake.hazardlib.const.IMC.GEOMETRIC_MEAN`,
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.GEOMETRIC_MEAN

    #: Supported standard deviation type is only total, see equation 35, page
    #: 1021
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL, const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT}

    DEFINED_FOR_REFERENCE_VELOCITY = 800

    #: No site parameters are needed
    REQUIRES_SITES_PARAMETERS = set()

    #: Required rupture parameter is only magnitude, see equation 30 page
    #: 1021.
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rjb'}

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.compute>`
        for spec of input and result values.
        """
        dist_type = _get_dist_type(gmpe=self)
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            mean[m] = _compute_mean(C, ctx.mag, getattr(ctx, dist_type))
            if imt.string.startswith(("PGA", "SA")):
                # Convert from m/s**2 to g
                mean[m] -= np.log(g)
            elif imt.string == 'PGV':  # Convert from m/s to cm/s
                mean[m] += np.log(100.0)
            sig[m] = np.sqrt(C['sigma'] ** 2 + C['tau'] ** 2)
            phi[m] = C['sigma']
            tau[m] = C['tau']

    #: Coefficient tables are constructed from the electronic suplements of
    #: the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.071377 -0.104440 -0.256368 -2.082968 0.175980 6.092497 -0.003354 0.596648 0.435828
	pga 3.301210 -0.737892 -0.306729 -2.161341 0.160068 6.456162 -0.005434 0.681749 0.534690
	0.010000 3.316019 -0.739661 -0.306739 -2.166328 0.160287 6.466172 -0.005424 0.684600 0.534966
	0.020000 3.397746 -0.767832 -0.307173 -2.223926 0.165005 6.430227 -0.005301 0.718285 0.538337
	0.030000 3.713900 -0.802179 -0.307387 -2.335400 0.170016 6.776541 -0.005022 0.765576 0.541424
	0.040000 3.856821 -0.837170 -0.303832 -2.424559 0.178855 6.744506 -0.005145 0.794719 0.549611
	0.050000 4.202453 -0.826429 -0.304496 -2.421348 0.171621 7.223255 -0.005539 0.806723 0.549383
	0.075000 4.272509 -0.761342 -0.302493 -2.221265 0.153865 7.096516 -0.007126 0.770664 0.548284
	0.100000 4.196424 -0.734942 -0.300902 -2.095881 0.145491 6.730797 -0.007958 0.739038 0.550258
	0.150000 4.092752 -0.607407 -0.304960 -1.726689 0.111430 6.044787 -0.008780 0.684663 0.541002
	0.200000 3.902236 -0.541433 -0.304370 -1.631331 0.107681 5.565939 -0.007979 0.649140 0.530486
	0.250000 3.735820 -0.564165 -0.314904 -1.613639 0.108247 5.345891 -0.007363 0.626928 0.523275
	0.300000 3.813539 -0.567288 -0.330693 -1.562863 0.096358 5.894730 -0.006721 0.610974 0.511050
	0.400000 3.738093 -0.616395 -0.353019 -1.596618 0.097112 6.244053 -0.005635 0.596194 0.495412
	0.500000 3.500535 -0.584768 -0.365189 -1.520070 0.093403 5.744979 -0.005183 0.584400 0.483265
	0.750000 3.234558 -0.548882 -0.381135 -1.607358 0.102542 6.049448 -0.003642 0.576508 0.453998
	1.000000 3.175584 -0.443241 -0.381813 -1.687741 0.109819 6.648249 -0.002547 0.573153 0.427804
	1.250000 3.063413 -0.347957 -0.383817 -1.663268 0.106795 6.801522 -0.002023 0.571455 0.418229
	1.500000 2.881863 -0.260619 -0.381584 -1.658755 0.110979 6.353724 -0.001851 0.570031 0.411255
	1.750000 2.881210 -0.232600 -0.381592 -1.723555 0.115676 6.785918 -0.001748 0.566234 0.405577
	2.000000 2.671433 -0.163676 -0.377968 -1.669152 0.112208 6.492545 -0.002183 0.563816 0.402956
	3.000000 2.275590 0.247272 -0.326835 -1.646368 0.119810 6.194922 -0.001987 0.566383 0.384096
    """)


class DrouetAlpes2015Rrup(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    Valid for vs30=800 m/s
    """
    #: Required distance measure is rupture distance, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rrup'}

    #: Coefficient tables are constructed from the electronic suplements of
    #: the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.143358 -0.193706 -0.251300 -2.325608 0.204495 3.525612 -0.002746 0.578654 0.405972
	pga 3.456556 -0.823110 -0.301051 -2.426061 0.188952 4.235479 -0.004683 0.665923 0.505942
	0.010000 3.470062 -0.824788 -0.301053 -2.430558 0.189164 4.241373 -0.004677 0.668775 0.506151
	0.020000 3.556579 -0.856473 -0.301366 -2.497644 0.194896 4.212097 -0.004524 0.702415 0.508170
	0.030000 3.842632 -0.885079 -0.301432 -2.589669 0.198714 4.512255 -0.004368 0.750291 0.510792
	0.040000 3.994895 -0.924480 -0.297689 -2.691940 0.208866 4.501888 -0.004445 0.779164 0.517264
	0.050000 4.300470 -0.900427 -0.298388 -2.650152 0.198450 4.914150 -0.005026 0.792074 0.518794
	0.075000 4.384285 -0.832046 -0.296502 -2.447305 0.179942 4.762052 -0.006597 0.756406 0.519043
	0.100000 4.364097 -0.810410 -0.294965 -2.347628 0.172709 4.516529 -0.007253 0.724937 0.521722
	0.150000 4.369398 -0.683950 -0.299443 -2.010100 0.138573 4.138441 -0.007769 0.672457 0.515656
	0.200000 4.238967 -0.623990 -0.299157 -1.941909 0.135740 3.881449 -0.006748 0.637704 0.506078
	0.250000 4.087930 -0.651728 -0.309801 -1.938372 0.137341 3.711227 -0.006044 0.614808 0.498429
	0.300000 4.102449 -0.638958 -0.325736 -1.835260 0.121434 4.127071 -0.005683 0.599381 0.489431
	0.400000 3.967323 -0.683753 -0.348158 -1.842141 0.120983 4.329381 -0.004776 0.583723 0.474835
	0.500000 3.816244 -0.656659 -0.360516 -1.797755 0.117973 4.143848 -0.004060 0.573808 0.463401
	0.750000 3.481698 -0.619997 -0.376581 -1.861697 0.126573 4.272753 -0.002702 0.565181 0.433160
	1.000000 3.328505 -0.506447 -0.377396 -1.896933 0.131603 4.629935 -0.001899 0.561431 0.406889
	1.250000 3.180536 -0.407322 -0.379542 -1.852935 0.127436 4.680294 -0.001493 0.560114 0.397052
	1.500000 3.043493 -0.327887 -0.377343 -1.876445 0.133360 4.347970 -0.001156 0.558675 0.387362
	1.750000 3.001759 -0.293676 -0.377311 -1.916867 0.136569 4.688507 -0.001199 0.554995 0.381578
	2.000000 2.821004 -0.228618 -0.373699 -1.879046 0.134076 4.448737 -0.001530 0.552212 0.377552
	3.000000 2.389195 0.176965 -0.322698 -1.855682 0.142701 3.939782 -0.001381 0.553748 0.356058
    """)


class DrouetAlpes2015Repi(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    Valid for vs30=800 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'repi'}

    #: Coefficient tables are constructed from the electronic suplements of
    #: the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.594765 -0.063587 -0.241851 -2.325495 0.203065 7.952062 -0.003150 0.614933 0.433117
	pga 3.939373 -0.663477 -0.284718 -2.455689 0.194830 8.544821 -0.005278 0.709068 0.532155
	0.010000 3.956668 -0.665095 -0.284675 -2.461594 0.195118 8.559021 -0.005266 0.711908 0.532415
	0.020000 4.051797 -0.691309 -0.284889 -2.521430 0.199995 8.496238 -0.005158 0.745423 0.535636
	0.030000 4.418744 -0.726229 -0.284176 -2.656403 0.206848 8.968781 -0.004814 0.792345 0.538649
	0.040000 4.556488 -0.760281 -0.280282 -2.746336 0.216533 8.870885 -0.005004 0.822001 0.546600
	0.050000 4.960665 -0.752657 -0.279482 -2.780642 0.212497 9.622355 -0.005228 0.833787 0.546570
	0.075000 5.002367 -0.686014 -0.276425 -2.586009 0.196575 9.588909 -0.006764 0.799049 0.545728
	0.100000 4.880135 -0.653913 -0.274687 -2.441860 0.187445 9.111588 -0.007664 0.769100 0.547823
	0.150000 4.719874 -0.507192 -0.278470 -2.025931 0.149192 8.291828 -0.008599 0.716357 0.538996
	0.200000 4.470344 -0.437443 -0.279814 -1.884369 0.140601 7.521449 -0.007953 0.680277 0.528444
	0.250000 4.301780 -0.459725 -0.291406 -1.852313 0.138830 7.221646 -0.007354 0.658043 0.521374
	0.300000 4.446649 -0.466491 -0.306911 -1.827283 0.127332 8.153314 -0.006502 0.640724 0.509280
	0.400000 4.444646 -0.523518 -0.330123 -1.883934 0.127490 8.783781 -0.005195 0.624387 0.493685
	0.500000 4.143932 -0.484406 -0.343342 -1.763046 0.119972 7.960030 -0.004949 0.612006 0.481334
	0.750000 3.886633 -0.458802 -0.361217 -1.847001 0.127171 8.287259 -0.003374 0.601659 0.451755
	1.000000 3.849151 -0.367331 -0.363330 -1.941130 0.134332 9.067178 -0.002168 0.595104 0.425399
	1.250000 3.712912 -0.272819 -0.365957 -1.902359 0.129951 9.181794 -0.001718 0.591770 0.415602
	1.500000 3.477403 -0.182797 -0.364564 -1.868757 0.132141 8.449238 -0.001708 0.590463 0.408287
	1.750000 3.502813 -0.161564 -0.364548 -1.953379 0.138306 9.042165 -0.001503 0.586166 0.402378
	2.000000 3.304209 -0.090851 -0.360719 -1.902557 0.134925 8.784215 -0.001890 0.583947 0.399611
	3.000000 2.774119 0.312958 -0.311810 -1.830330 0.140061 8.090585 -0.002018 0.583488 0.381611
    """)


class DrouetAlpes2015Rhyp(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA
    doi: 10.1785/0120140240.
    Valid for vs30=800 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rhypo'}

    #: Coefficient tables are constructed from the electronic suplements of
    #: the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6        c7       sigma    tau                                                                               
	pgv 1.627812 -0.078852 -0.245831 -2.328666 0.202304 2.970501 -0.003140 0.599738 0.409327
	pga 3.946311 -0.674327 -0.288999 -2.440716 0.192893 3.813743 -0.005375 0.694400 0.509309
	0.010000 3.961337 -0.675763 -0.288962 -2.445604 0.193140 3.826981 -0.005370 0.697265 0.509577
	0.020000 4.061293 -0.703340 -0.289298 -2.508332 0.198191 3.764095 -0.005247 0.730635 0.511717
	0.030000 4.386725 -0.733578 -0.288568 -2.623167 0.204066 4.404588 -0.005025 0.778628 0.514682
	0.040000 4.531133 -0.769058 -0.284848 -2.716221 0.213905 4.276392 -0.005199 0.807632 0.521164
	0.050000 4.862128 -0.752202 -0.283808 -2.715204 0.208165 5.113457 -0.005634 0.821043 0.522865
	0.075000 4.893776 -0.682655 -0.280621 -2.513142 0.191775 4.825758 -0.007214 0.786443 0.523129
	0.100000 4.834986 -0.656494 -0.278906 -2.397315 0.183873 4.299656 -0.007938 0.756094 0.525594
	0.150000 4.759075 -0.518802 -0.282546 -2.022762 0.147660 3.225794 -0.008622 0.703759 0.519562
	0.200000 4.617387 -0.459717 -0.283868 -1.932108 0.141251 2.584466 -0.007627 0.667576 0.509367
	0.250000 4.468951 -0.485743 -0.295503 -1.912932 0.140266 2.108569 -0.006949 0.644260 0.501866
	0.300000 4.501402 -0.480669 -0.310677 -1.836331 0.126628 3.278147 -0.006447 0.628888 0.492826
	0.400000 4.410002 -0.531232 -0.333745 -1.857297 0.125619 3.907703 -0.005388 0.612697 0.477934
	0.500000 4.248572 -0.502332 -0.346966 -1.794385 0.120095 3.482998 -0.004732 0.601221 0.466580
	0.750000 3.950039 -0.474332 -0.364750 -1.863211 0.126892 3.919657 -0.003265 0.590911 0.436324
	1.000000 3.827738 -0.375612 -0.366644 -1.922067 0.132764 4.762955 -0.002295 0.585094 0.409878
	1.250000 3.673308 -0.279262 -0.369180 -1.875471 0.128055 4.844969 -0.001894 0.582131 0.400603
	1.500000 3.522818 -0.196561 -0.367928 -1.877897 0.131574 4.211803 -0.001640 0.580446 0.392074
	1.750000 3.507817 -0.171535 -0.367774 -1.945474 0.137121 4.959363 -0.001550 0.576902 0.386743
	2.000000 3.311391 -0.101864 -0.364013 -1.896737 0.133947 4.514625 -0.001933 0.574094 0.383316
	3.000000 2.766739 0.301056 -0.315235 -1.821128 0.139261 3.010114 -0.002109 0.571483 0.364027
    """)


class DrouetAlpes2015RjbHR(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 100 bars
    (recommended by the authors)
    Valid for vs30=2000 m/s
    """

    #: Coefficient tables are constructed from the electronic suplements of
    #: the erratum to the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 0.942202 -0.096360 -0.249846 -2.159260 0.184686 6.157444 -0.003121 0.580515 0.435237
    pga 3.502871 -0.802486 -0.302312 -2.370875 0.175248 6.729398 -0.005136 0.638637 0.543828
    0.010000 3.565304 -0.821883 -0.303466 -2.407764 0.177715 6.690289 -0.005018 0.652676 0.545900
    0.020000 4.050992 -0.929762 -0.307473 -2.657718 0.193282 6.900732 -0.004243 0.705755 0.555220
    0.030000 4.691048 -0.935742 -0.307451 -2.765942 0.190873 7.739100 -0.004172 0.710803 0.554633
    0.040000 4.715972 -0.923929 -0.303375 -2.719416 0.189440 7.681047 -0.005159 0.702242 0.559697
    0.050000 4.808292 -0.859750 -0.301940 -2.566466 0.173602 7.810588 -0.006302 0.689282 0.556118
    0.075000 4.542852 -0.769721 -0.300791 -2.255961 0.151510 7.345139 -0.008009 0.663297 0.551312
    0.100000 4.234265 -0.722066 -0.299397 -2.062668 0.140698 6.696867 -0.008837 0.649414 0.551174
    0.150000 3.963638 -0.600023 -0.303078 -1.725046 0.111277 5.978048 -0.008991 0.625353 0.540408
    0.200000 3.713795 -0.535134 -0.303177 -1.633263 0.107606 5.520753 -0.008014 0.609284 0.529124
    0.250000 3.493820 -0.555508 -0.313442 -1.617051 0.108770 5.291459 -0.007346 0.597736 0.521516
    0.300000 3.537878 -0.561992 -0.329149 -1.575924 0.097991 5.826652 -0.006631 0.588055 0.509204
    0.400000 3.457062 -0.607230 -0.351710 -1.604230 0.098077 6.188413 -0.005544 0.581860 0.492997
    0.500000 3.243511 -0.575336 -0.364464 -1.522954 0.093691 5.669695 -0.005098 0.574114 0.480631
    0.750000 3.036025 -0.535625 -0.380749 -1.604625 0.101926 5.996742 -0.003561 0.570709 0.450864
    1.000000 3.016962 -0.428035 -0.381852 -1.682332 0.108400 6.620164 -0.002443 0.568420 0.424469
    1.250000 2.919868 -0.330748 -0.384517 -1.647261 0.104126 6.748230 -0.001949 0.567403 0.415070
    1.500000 2.749652 -0.245843 -0.382722 -1.644734 0.108446 6.307728 -0.001765 0.566314 0.408666
    1.750000 2.760364 -0.215730 -0.382466 -1.709401 0.113070 6.751567 -0.001667 0.563170 0.402998
    2.000000 2.553832 -0.147150 -0.378930 -1.656279 0.109779 6.450533 -0.002089 0.560557 0.400305
    3.000000 2.148241 0.267200 -0.328069 -1.623369 0.116390 6.117724 -0.001925 0.562644 0.382704
    """)


class DrouetAlpes2015RrupHR(DrouetAlpes2015Rrup):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 100 bars
    (recommended by the authors)
    Valid for vs30=2000 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rrup'}
    DEFINED_FOR_REFERENCE_VELOCITY = 2000

    #: Coefficient tables are constructed from the electronic suplements of
    #: the erratum to the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.011632 -0.186467 -0.244724 -2.402830 0.213372 3.604144 -0.002514 0.561838 0.404482
    pga 3.656222 -0.887864 -0.296336 -2.637227 0.204462 4.534131 -0.004392 0.619947 0.512710
    0.010000 3.729653 -0.909898 -0.297406 -2.683317 0.207673 4.522980 -0.004230 0.633797 0.513741
    0.020000 4.229542 -1.020987 -0.301039 -2.946183 0.224422 4.809072 -0.003412 0.686340 0.519751
    0.030000 4.800663 -1.007592 -0.300977 -2.995890 0.217461 5.579850 -0.003639 0.692279 0.520869
    0.040000 4.797203 -0.995255 -0.296833 -2.941575 0.216154 5.394204 -0.004702 0.682523 0.525089
    0.050000 4.901700 -0.923699 -0.295554 -2.776188 0.198405 5.550909 -0.005858 0.671369 0.524292
    0.075000 4.677298 -0.835453 -0.294660 -2.479707 0.176586 5.096348 -0.007449 0.646832 0.521721
    0.100000 4.442685 -0.796743 -0.293407 -2.325256 0.167820 4.589685 -0.008025 0.633601 0.522194
    0.150000 4.270862 -0.677412 -0.297536 -2.019335 0.138622 4.165836 -0.007885 0.612083 0.514625
    0.200000 4.076904 -0.618200 -0.297937 -1.952840 0.135779 3.926013 -0.006700 0.597253 0.504507
    0.250000 3.869013 -0.644133 -0.308331 -1.950835 0.138083 3.735393 -0.005947 0.585145 0.496466
    0.300000 3.846701 -0.635600 -0.324197 -1.858026 0.123492 4.126070 -0.005521 0.576001 0.487258
    0.400000 3.701521 -0.676042 -0.346850 -1.857232 0.122281 4.326213 -0.004628 0.569060 0.472191
    0.500000 3.581793 -0.648656 -0.359793 -1.810204 0.118586 4.145801 -0.003899 0.563491 0.460619
    0.750000 3.299801 -0.607702 -0.376203 -1.865855 0.126182 4.279235 -0.002565 0.559378 0.429962
    1.000000 3.182887 -0.491713 -0.377441 -1.896308 0.130298 4.655163 -0.001755 0.556805 0.403581
    1.250000 3.053846 -0.390596 -0.380238 -1.843008 0.124912 4.693096 -0.001370 0.556205 0.394117
    1.500000 2.930072 -0.313580 -0.378480 -1.869066 0.130964 4.378112 -0.001013 0.555172 0.385029
    1.750000 2.895911 -0.277205 -0.378207 -1.907801 0.134048 4.716636 -0.001074 0.552129 0.379229
    2.000000 2.720867 -0.212507 -0.374663 -1.872189 0.131758 4.479180 -0.001385 0.549159 0.375092
    3.000000 2.281596 0.195816 -0.323947 -1.840794 0.139554 3.944804 -0.001255 0.550117 0.354813
    """)


class DrouetAlpes2015RepiHR(DrouetAlpes2015Repi):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 100 bars
    (recommended by the authors)
    Valid for vs30=2000 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'repi'}
    DEFINED_FOR_REFERENCE_VELOCITY = 2000

    #: Coefficient tables are constructed from the electronic suplements of
    #: the erratum to the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.453087 -0.057150 -0.235736 -2.395436 0.211467 7.957970 -0.002998 0.598554 0.432545
	pga 4.198896 -0.725712 -0.279239 -2.685851 0.211701 8.868349 -0.004976 0.668901 0.541172
	0.010000 4.280953 -0.742833 -0.280160 -2.726267 0.214180 8.825554 -0.004845 0.683085 0.543146
	0.020000 4.870686 -0.844591 -0.282767 -3.005948 0.231235 9.104488 -0.003997 0.737359 0.552261
	0.030000 5.606980 -0.855373 -0.280735 -3.168131 0.233287 10.277129 -0.003738 0.743021 0.551848
	0.040000 5.607137 -0.847563 -0.276241 -3.129571 0.233903 10.256513 -0.004722 0.735433 0.556826
	0.050000 5.685271 -0.784347 -0.273823 -2.987484 0.220116 10.530324 -0.005816 0.722746 0.553388
	0.075000 5.315349 -0.690620 -0.272702 -2.647489 0.197397 9.988792 -0.007599 0.697352 0.548807
	0.100000 4.921405 -0.635287 -0.271922 -2.413453 0.183956 9.110885 -0.008568 0.683849 0.548856
	0.150000 4.586203 -0.496201 -0.276293 -2.018875 0.148836 8.188120 -0.008866 0.659367 0.538381
	0.200000 4.285404 -0.428313 -0.278456 -1.882710 0.140156 7.462946 -0.008024 0.641610 0.527206
	0.250000 4.060798 -0.449211 -0.289962 -1.851356 0.138878 7.146858 -0.007374 0.629415 0.519579
	0.300000 4.169405 -0.459324 -0.305454 -1.834284 0.128357 8.042983 -0.006461 0.618114 0.507458
	0.400000 4.163554 -0.512785 -0.328897 -1.886542 0.127893 8.702098 -0.005143 0.609924 0.491408
	0.500000 3.887845 -0.472417 -0.342631 -1.760040 0.119604 7.852130 -0.004909 0.601691 0.478750
	0.750000 3.692737 -0.443196 -0.360804 -1.840290 0.125998 8.226868 -0.003322 0.595642 0.448708
	1.000000 3.694925 -0.349219 -0.363280 -1.931108 0.132320 9.033621 -0.002099 0.590175 0.422143
	1.250000 3.576295 -0.251890 -0.366488 -1.881893 0.126648 9.129843 -0.001675 0.587606 0.412574
	1.500000 3.353014 -0.164563 -0.365540 -1.850916 0.129019 8.407176 -0.001650 0.586684 0.405750
	1.750000 3.389227 -0.141213 -0.365241 -1.935365 0.135130 9.014006 -0.001451 0.582954 0.399830
	2.000000 3.193535 -0.070763 -0.361524 -1.885294 0.131871 8.742877 -0.001828 0.580564 0.397018
	3.000000 2.655991 0.336601 -0.312845 -1.803880 0.136056 8.023762 -0.001980 0.579737 0.380198
    """)


class DrouetAlpes2015RhypHR(DrouetAlpes2015Rhyp):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 100 bars
    (recommended by the authors)
    Valid for vs30=2000 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rhypo'}
    DEFINED_FOR_REFERENCE_VELOCITY = 2000

    #: Coefficient tables are constructed from the electronic suplements of
    #: the erratum to the original paper.
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pgv 1.503464 -0.076362 -0.239730 -2.410988 0.211580 3.221740 -0.002925 0.583454 0.408088
	pga 4.201352 -0.737321 -0.283657 -2.670056 0.209823 4.456104 -0.005084 0.653222 0.516623
	0.010000 4.288248 -0.755638 -0.284695 -2.712990 0.212421 4.428086 -0.004939 0.667212 0.517849
	0.020000 4.874345 -0.858440 -0.287527 -2.991853 0.229451 4.964363 -0.004094 0.721552 0.524782
	0.030000 5.514994 -0.858034 -0.285161 -3.110043 0.229479 6.354667 -0.004093 0.729341 0.526239
	0.040000 5.448641 -0.843753 -0.280722 -3.040707 0.228799 5.965006 -0.005282 0.720304 0.530300
	0.050000 5.532571 -0.778734 -0.278055 -2.899015 0.214844 6.286043 -0.006357 0.709285 0.529376
	0.075000 5.214094 -0.687309 -0.276736 -2.578440 0.192806 5.504908 -0.008023 0.684365 0.526187
	0.100000 4.910658 -0.641348 -0.276106 -2.385394 0.181200 4.487291 -0.008743 0.670372 0.526536
	0.150000 4.662478 -0.512097 -0.280364 -2.034438 0.148271 3.354935 -0.008775 0.646539 0.518644
	0.200000 4.462378 -0.454157 -0.282484 -1.946059 0.141637 2.801928 -0.007603 0.629145 0.507928
	0.250000 4.256096 -0.478835 -0.294039 -1.927149 0.141151 2.306312 -0.006878 0.615946 0.499967
	0.300000 4.258439 -0.477306 -0.309222 -1.860333 0.128499 3.405061 -0.006301 0.606533 0.490911
	0.400000 4.157978 -0.523822 -0.332503 -1.874710 0.126801 4.029080 -0.005248 0.598586 0.475528
	0.500000 4.025056 -0.493907 -0.346254 -1.807517 0.120533 3.593387 -0.004592 0.591428 0.464013
	0.750000 3.781564 -0.461675 -0.364323 -1.869546 0.126409 4.046692 -0.003135 0.585469 0.433454
	1.000000 3.698977 -0.360560 -0.366579 -1.925215 0.131459 4.923044 -0.002149 0.580757 0.406990
	1.250000 3.565335 -0.261563 -0.369684 -1.869495 0.125508 4.997219 -0.001764 0.578584 0.397992
	1.500000 3.425292 -0.181352 -0.368877 -1.873717 0.129166 4.382157 -0.001500 0.577384 0.390016
	1.750000 3.421376 -0.154060 -0.368402 -1.941200 0.134671 5.133325 -0.001417 0.574363 0.384719
	2.000000 3.227255 -0.084883 -0.364789 -1.893149 0.131623 4.680092 -0.001790 0.571400 0.381164
	3.000000 2.668543 0.321383 -0.316237 -1.807166 0.136069 3.132971 -0.002003 0.568302 0.363028
    """)


class DrouetAlpes2015Rjb_50bars(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=800 m/s
    """
    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pga      2.752127 -0.699014 -0.275105 -2.170136 0.162949 6.497180 -0.005339 0.677044 0.501451
    0.010000 2.766590 -0.700791 -0.275114 -2.175056 0.163168 6.506825 -0.005329 0.679861 0.501683
    0.020000 2.848130 -0.726641 -0.275121 -2.231988 0.167849 6.468200 -0.005208 0.712987 0.504809
    0.030000 3.160519 -0.759847 -0.274998 -2.343359 0.172980 6.810926 -0.004932 0.760100 0.507999
    0.040000 3.314141 -0.785356 -0.270066 -2.429648 0.181245 6.784957 -0.005045 0.789006 0.515484
    0.050000 3.651497 -0.778920 -0.271058 -2.430324 0.174592 7.257187 -0.005425 0.801239 0.515312
    0.075000 3.745921 -0.710276 -0.268536 -2.236047 0.156726 7.164403 -0.006962 0.765603 0.514543
    0.100000 3.661721 -0.685353 -0.266853 -2.113955 0.149009 6.799003 -0.007800 0.734514 0.515790
    0.150000 3.562904 -0.570134 -0.273945 -1.740334 0.113778 6.108076 -0.008669 0.681870 0.506299
    0.200000 3.397324 -0.491677 -0.272950 -1.630098 0.108070 5.591477 -0.007955 0.647012 0.496087
    0.250000 3.201916 -0.534893 -0.285950 -1.621432 0.110437 5.396506 -0.007344 0.625250 0.488472
    0.300000 3.259770 -0.556095 -0.304848 -1.568067 0.098309 5.912981 -0.006733 0.609604 0.477295
    0.400000 3.155072 -0.629101 -0.330095 -1.611391 0.100957 6.265931 -0.005637 0.595130 0.463805
    0.500000 2.929822 -0.612755 -0.346052 -1.527258 0.095508 5.761636 -0.005184 0.583430 0.452269
    0.750000 2.655826 -0.605067 -0.367129 -1.613829 0.104765 6.068111 -0.003646 0.574997 0.427957
    1.000000 2.593362 -0.512852 -0.370957 -1.686547 0.111812 6.629473 -0.002625 0.571282 0.403537
    1.250000 2.479708 -0.441778 -0.378428 -1.652953 0.107060 6.750127 -0.002106 0.569345 0.393826
    1.500000 2.306743 -0.374301 -0.380520 -1.647259 0.110576 6.319041 -0.001910 0.567896 0.383743
    1.750000 2.303693 -0.358668 -0.383152 -1.711423 0.115018 6.750836 -0.001776 0.565315 0.375491
    2.000000 2.093210 -0.302924 -0.381812 -1.662896 0.112197 6.483294 -0.002182 0.562819 0.370506
    3.000000 1.732627  0.086233 -0.338529 -1.620115 0.116304 6.202268 -0.002002 0.564420 0.353377
    pgv      0.602344 -0.098125 -0.234545 -2.097286 0.179245 6.176835 -0.003171 0.591797 0.406816
    """)


class DrouetAlpes2015Rrup_50bars(DrouetAlpes2015Rrup):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=800 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = set(('rrup', ))
    DEFINED_FOR_REFERENCE_VELOCITY = 800

    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pga      2.903141 -0.783207 -0.269497 -2.430576 0.191408 4.277526 -0.004602 0.661494 0.477399
    0.010000 2.916878 -0.784832 -0.269484 -2.435178 0.191619 4.285015 -0.004595 0.664306 0.477598
    0.020000 3.002832 -0.814115 -0.269365 -2.501357 0.197300 4.250838 -0.004445 0.697354 0.479422
    0.030000 3.285561 -0.841784 -0.269106 -2.593722 0.201281 4.548326 -0.004291 0.745094 0.482031
    0.040000 3.450637 -0.871443 -0.263986 -2.693275 0.210799 4.552568 -0.004352 0.773717 0.488020
    0.050000 3.744755 -0.852040 -0.265012 -2.655210 0.201065 4.944143 -0.004928 0.786770 0.489205
    0.075000 3.851689 -0.779725 -0.262579 -2.457284 0.182409 4.834691 -0.006457 0.751756 0.489908
    0.100000 3.816983 -0.759910 -0.260982 -2.359475 0.175874 4.565334 -0.007132 0.720605 0.491584
    0.150000 3.836917 -0.645773 -0.268458 -2.020600 0.140592 4.217317 -0.007668 0.669884 0.485124
    0.200000 3.734601 -0.573592 -0.267742 -1.939459 0.135944 3.926821 -0.006725 0.635690 0.475767
    0.250000 3.543960 -0.621825 -0.280864 -1.941560 0.139299 3.749602 -0.006055 0.613299 0.467808
    0.300000 3.544874 -0.627804 -0.299914 -1.839092 0.123330 4.148140 -0.005703 0.598253 0.459550
    0.400000 3.374535 -0.696534 -0.325252 -1.853744 0.124746 4.324664 -0.004801 0.582787 0.447091
    0.500000 3.239052 -0.684969 -0.341402 -1.803369 0.120090 4.151127 -0.004075 0.572945 0.436359
    0.750000 2.898362 -0.676199 -0.362601 -1.866448 0.128720 4.288797 -0.002717 0.563768 0.411442
    1.000000 2.745864 -0.576417 -0.366570 -1.895824 0.133615 4.620264 -0.001979 0.560091 0.387038
    1.250000 2.608719 -0.501276 -0.374177 -1.845843 0.127652 4.673279 -0.001545 0.558346 0.377097
    1.500000 2.484336 -0.441596 -0.376282 -1.869369 0.132935 4.392990 -0.001175 0.557372 0.365190
    1.750000 2.438202 -0.420104 -0.378915 -1.908967 0.135925 4.718448 -0.001190 0.554524 0.356212
    2.000000 2.249651 -0.367952 -0.377543 -1.874949 0.134079 4.480993 -0.001509 0.551520 0.349784
    3.000000 1.866311  0.017205 -0.334454 -1.832104 0.138784 4.052714 -0.001347 0.552346 0.328332
    pgv      0.673613 -0.185375 -0.229561 -2.334507 0.207077 3.649829 -0.002576 0.574791 0.381312
    """)


class DrouetAlpes2015Repi_50bars(DrouetAlpes2015Repi):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=800 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = set(('repi', ))
    DEFINED_FOR_REFERENCE_VELOCITY = 800

    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pga      3.366117 -0.627709 -0.253649 -2.457158 0.197314 8.542721 -0.005224 0.703389 0.499871
    0.010000 3.383272 -0.629367 -0.253608 -2.463100 0.197605 8.557279 -0.005211 0.706165 0.500141
    0.020000 3.477968 -0.653306 -0.253404 -2.522183 0.202441 8.493200 -0.005105 0.739190 0.503196
    0.030000 3.837126 -0.687002 -0.252367 -2.655659 0.209383 8.954055 -0.004775 0.786049 0.506192
    0.040000 3.990490 -0.710779 -0.246930 -2.743969 0.218511 8.876900 -0.004947 0.815380 0.513709
    0.050000 4.384941 -0.708337 -0.246616 -2.782109 0.215058 9.617175 -0.005157 0.827529 0.513523
    0.075000 4.447793 -0.637729 -0.242970 -2.592003 0.199061 9.608044 -0.006663 0.793428 0.512919
    0.100000 4.321265 -0.608368 -0.241217 -2.454010 0.190747 9.143345 -0.007547 0.763925 0.514347
    0.150000 4.176626 -0.472938 -0.247838 -2.036986 0.151452 8.337874 -0.008505 0.712925 0.505122
    0.200000 3.953424 -0.388528 -0.248564 -1.878996 0.140813 7.525830 -0.007962 0.677409 0.495270
    0.250000 3.746667 -0.433557 -0.262850 -1.855600 0.140969 7.234321 -0.007368 0.655494 0.487622
    0.300000 3.873121 -0.457841 -0.281379 -1.828513 0.129292 8.130473 -0.006542 0.638840 0.476562
    0.400000 3.828818 -0.540188 -0.307725 -1.891607 0.131300 8.727195 -0.005245 0.622718 0.463097
    0.500000 3.553068 -0.515192 -0.324527 -1.766456 0.122148 7.937002 -0.004984 0.610766 0.451643
    0.750000 3.285511 -0.517747 -0.347576 -1.848396 0.129363 8.261894 -0.003419 0.599662 0.426973
    1.000000 3.223445 -0.438840 -0.352820 -1.928000 0.136213 8.934380 -0.002337 0.593191 0.402037
    1.250000 3.098537 -0.366551 -0.360716 -1.881794 0.129983 9.042480 -0.001878 0.589684 0.392203
    1.500000 2.877284 -0.295774 -0.363596 -1.847641 0.131436 8.339888 -0.001842 0.588263 0.381953
    1.750000 2.904152 -0.287301 -0.366173 -1.933737 0.137487 8.956035 -0.001598 0.585392 0.373447
    2.000000 2.707461 -0.230688 -0.364686 -1.890594 0.134832 8.736355 -0.001941 0.582981 0.368317
    3.000000 2.238657  0.153449 -0.323342 -1.804521 0.136387 8.137432 -0.002025 0.581130 0.351297
    pgv      1.099445 -0.060566 -0.220644 -2.331083 0.205828 7.987609 -0.003024 0.609195 0.405069
    """)


class DrouetAlpes2015Rhyp_50bars(DrouetAlpes2015Rhyp):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA
    doi: 10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=800 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rhypo'}
    DEFINED_FOR_REFERENCE_VELOCITY = 800

    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
	pga      3.379583 -0.639148 -0.257885 -2.445455 0.195545 3.866535 -0.005302 0.689093 0.481665
    0.010000 3.394577 -0.640643 -0.257853 -2.450410 0.195794 3.880931 -0.005296 0.691907 0.481865
    0.020000 3.494076 -0.665918 -0.257761 -2.512409 0.200809 3.815051 -0.005175 0.724769 0.483967
    0.030000 3.815626 -0.695417 -0.256712 -2.627759 0.206883 4.451266 -0.004955 0.772724 0.486928
    0.040000 3.969005 -0.720471 -0.251574 -2.715956 0.215986 4.317920 -0.005130 0.801312 0.493021
    0.050000 4.293884 -0.708133 -0.250894 -2.719572 0.210822 5.143919 -0.005543 0.814988 0.494312
    0.075000 4.349481 -0.635023 -0.247112 -2.523728 0.194457 4.922155 -0.007083 0.781136 0.494715
    0.100000 4.273603 -0.610128 -0.245389 -2.407590 0.187044 4.338845 -0.007831 0.750993 0.496414
    0.150000 4.221145 -0.484743 -0.251868 -2.036120 0.150003 3.379651 -0.008510 0.700636 0.489877
    0.200000 4.099907 -0.410858 -0.252591 -1.926977 0.141516 2.598494 -0.007636 0.664723 0.480159
    0.250000 3.912521 -0.459656 -0.266906 -1.916339 0.142475 2.153968 -0.006966 0.641923 0.472131
    0.300000 3.937054 -0.472433 -0.285121 -1.841100 0.128683 3.307383 -0.006460 0.627157 0.463468
    0.400000 3.810611 -0.548548 -0.311318 -1.870970 0.129575 3.918128 -0.005394 0.611240 0.450292
    0.500000 3.662262 -0.533407 -0.328137 -1.799736 0.122338 3.476611 -0.004753 0.600038 0.439658
    0.750000 3.350491 -0.533404 -0.351095 -1.865326 0.129120 3.887053 -0.003306 0.589002 0.413996
    1.000000 3.222810 -0.448437 -0.356133 -1.917100 0.134901 4.668758 -0.002407 0.583324 0.388861
    1.250000 3.086201 -0.375011 -0.363939 -1.866256 0.128499 4.794876 -0.001977 0.580268 0.379254
    1.500000 2.938296 -0.310767 -0.366947 -1.863488 0.131133 4.134577 -0.001730 0.578492 0.367992
    1.750000 2.914515 -0.297295 -0.369401 -1.927275 0.136280 4.835119 -0.001631 0.575962 0.359228
    2.000000 2.717649 -0.241515 -0.367974 -1.885267 0.133796 4.436253 -0.001976 0.572960 0.353369
    3.000000 2.224748  0.142939 -0.326737 -1.791008 0.135283 3.016459 -0.002137 0.569133 0.334070
    pgv      1.133549 -0.075262 -0.224574 -2.333758 0.204952 3.063629 -0.003011 0.594324 0.385201
    """)


class DrouetAlpes2015RjbHR_50bars(DrouetAlpes2015Rjb):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=2000 m/s
    """
    DEFINED_FOR_REFERENCE_VELOCITy = 2000

    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
    pga      2.957204 -0.753535 -0.268904 -2.378341 0.177964 6.760295 -0.005020 0.634364 0.511190
    0.010000 3.017590 -0.773158 -0.270084 -2.414899 0.180441 6.718367 -0.004904 0.648232 0.512839
    0.020000 3.494738 -0.879190 -0.273425 -2.665334 0.196327 6.922717 -0.004127 0.700789 0.520895
    0.030000 4.134055 -0.886103 -0.273526 -2.774576 0.193965 7.751117 -0.004042 0.706450 0.520824
    0.040000 4.175674 -0.866741 -0.268462 -2.727675 0.191962 7.710073 -0.004993 0.697889 0.525445
    0.050000 4.274927 -0.805369 -0.267229 -2.580613 0.176542 7.858335 -0.006113 0.685034 0.521764
    0.075000 4.024693 -0.714097 -0.265843 -2.274898 0.154599 7.421722 -0.007806 0.658893 0.517943
    0.100000 3.705110 -0.671375 -0.265065 -2.083399 0.144287 6.775635 -0.008657 0.645657 0.517225
    0.150000 3.437205 -0.559865 -0.271595 -1.738694 0.113581 6.043763 -0.008878 0.622629 0.506801
    0.200000 3.209256 -0.484063 -0.271504 -1.632156 0.108021 5.544272 -0.007988 0.606539 0.496081
    0.250000 2.960323 -0.525556 -0.284367 -1.624969 0.110974 5.340434 -0.007322 0.596154 0.488055
    0.300000 2.984573 -0.550113 -0.303178 -1.581202 0.099947 5.845156 -0.006638 0.586583 0.477349
    0.400000 2.873920 -0.620199 -0.328818 -1.619211 0.101945 6.210559 -0.005540 0.580702 0.463450
    0.500000 2.673115 -0.603869 -0.345393 -1.530631 0.095843 5.688372 -0.005096 0.573355 0.451768
    0.750000 2.457467 -0.592404 -0.366847 -1.611170 0.104165 6.013879 -0.003565 0.569698 0.427111
    1.000000 2.432434 -0.499506 -0.371286 -1.681111 0.110446 6.595468 -0.002524 0.566755 0.402917
    1.250000 2.334280 -0.426811 -0.379446 -1.637559 0.104501 6.690288 -0.002034 0.565803 0.393351
    1.500000 2.172449 -0.361989 -0.381954 -1.634311 0.108215 6.268165 -0.001825 0.564233 0.384448
    1.750000 2.179689 -0.344174 -0.384354 -1.697495 0.112514 6.707356 -0.001700 0.562643 0.376459
    2.000000 1.973044 -0.288793 -0.383096 -1.650520 0.109888 6.433541 -0.002091 0.559922 0.371666
    3.000000 1.603747  0.102968 -0.340215 -1.598025 0.113000 6.123170 -0.001942 0.561043 0.356077
    pgv      0.471757 -0.087648 -0.227818 -2.169988 0.187617 6.229653 -0.002943 0.577001 0.408947
    """)


class DrouetAlpes2015RrupHR_50bars(DrouetAlpes2015Rrup):
    """
    Implements GMPE developed by Douet & Cotton (2015) BSSA doi:
    10.1785/0120140240.
    This version is for a large magnitude stress parameters of 50 bars
    Valid for vs30=2000 m/s
    """
    #: Required distance measure is closest distance to rupture, see equation
    #: 30 page 1021.
    REQUIRES_DISTANCES = {'rrup'}
    DEFINED_FOR_REFERENCE_VELOCITY = 2000

    #: Coefficient table is not published
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT      c1       c2        c3        c4        c5       c6       c7        sigma    tau                                                                               
    pga      3.110039 -0.838037 -0.262987 -2.641943 0.206804 4.577744 -0.004279 0.616028 0.484784
    0.010000 3.180440 -0.860400 -0.264096 -2.687472 0.210030 4.560479 -0.004122 0.629697 0.485480
    0.020000 3.672729 -0.969962 -0.267043 -2.951976 0.227209 4.844608 -0.003296 0.681766 0.490182
    0.030000 4.245449 -0.957799 -0.267130 -3.003625 0.220335 5.607478 -0.003506 0.688314 0.491605
    0.040000 4.260464 -0.937700 -0.261979 -2.949413 0.218450 5.447975 -0.004526 0.678509 0.495421
    0.050000 4.363386 -0.868744 -0.260914 -2.786969 0.201067 5.600834 -0.005682 0.667368 0.494282
    0.075000 4.153961 -0.778866 -0.259761 -2.494545 0.179335 5.185162 -0.007267 0.642977 0.493026
    0.100000 3.899125 -0.745011 -0.259129 -2.338964 0.171034 4.644928 -0.007889 0.630083 0.492589
    0.150000 3.740868 -0.636413 -0.266090 -2.029600 0.140604 4.245243 -0.007785 0.609614 0.485330
    0.200000 3.572070 -0.566682 -0.266301 -1.950365 0.136019 3.965423 -0.006677 0.594612 0.475673
    0.250000 3.325148 -0.613639 -0.279279 -1.954165 0.140068 3.771988 -0.005954 0.583755 0.467286
    0.300000 3.290290 -0.623680 -0.298236 -1.862072 0.125385 4.151109 -0.005535 0.574797 0.459376
    0.400000 3.107590 -0.689134 -0.323980 -1.868784 0.126071 4.318807 -0.004651 0.568000 0.446592
    0.500000 3.004129 -0.677557 -0.340736 -1.816262 0.120774 4.153515 -0.003913 0.562821 0.435940
    0.750000 2.716678 -0.664525 -0.362324 -1.870759 0.128352 4.294486 -0.002580 0.558474 0.410713
    1.000000 2.598512 -0.563615 -0.366895 -1.895573 0.132391 4.641276 -0.001836 0.555669 0.386641
    1.250000 2.481017 -0.487009 -0.375202 -1.837089 0.125272 4.682919 -0.001419 0.554947 0.376968
    1.500000 2.369133 -0.429856 -0.377728 -1.863197 0.130716 4.418120 -0.001033 0.553906 0.366257
    1.750000 2.330086 -0.406069 -0.380109 -1.900813 0.133560 4.738958 -0.001067 0.552050 0.357515
    2.000000 2.147485 -0.354488 -0.378847 -1.869130 0.131926 4.504462 -0.001365 0.548840 0.351498
    3.000000 1.757597  0.032984 -0.336145 -1.818072 0.135737 4.056567 -0.001223 0.549067 0.331647
    pgv      0.541232 -0.175928 -0.222774 -2.408882 0.215679 3.713840 -0.002344 0.559330 0.382729
    """)
    
    
def _get_dist_type(gmpe):
    """
    Get distance type required for the corresponding class of 
    Drouet and Cotton (2015)
    """
    if 'Repi' in gmpe.__class__.__name__:
        dist_type = 'repi'
    elif 'Rjb' in gmpe.__class__.__name__:
        dist_type = 'rjb'
    elif 'Rrup' in gmpe.__class__.__name__:
        dist_type = 'rrup'
    else:
        dist_type = 'rhypo' # Otherwise must be rhypo

    return dist_type
