# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2023 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`HassaniAtkinson2018`
"""

import re
import pathlib
import numpy as np
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA
from openquake.hazardlib.gsim.base import CoeffsTable, GMPE
from openquake.hazardlib.gsim.boore_2014 import BooreEtAl2014
from openquake.hazardlib.gsim.yenier_atkinson_2015 import (
    get_fs_SeyhanStewart2014)

TFP = pathlib.Path(__file__).parent.absolute()


def _get_pseudo_depth(mag):
    """ Pseudo depth as defined in eq. 17 """
    return 10**(-0.405+0.235*mag)


def _get_geometrical_attenuation(rrup, r):
    """
    Compute the geometrical attenuation of Fourier amplitudes. See equation
    15 at page 5 of ha18.
    """
    rt = 50.0
    b1 = -1.3
    b2 = -0.5
    z = np.zeros_like(rrup)
    # Direct wave spreading
    idx = r <= rt
    z[idx] = r[idx]**b1
    # Surface-wave spreading
    idx = r > rt
    z[idx] = rt**b1 * (r[idx]/rt)**b2
    return z


def _geometrical_spreading(C, rrup, mag):
    """ Geometrical spreading term as per eq. 15 """
    h = _get_pseudo_depth(mag)
    r = (rrup**2 + h**2)**0.5
    z = _get_geometrical_attenuation(rrup, r)
    rref = (1 + h**2)**0.5
    return np.log10(z) + (C['b3'] + C['b4'] * mag) * np.log10(r/rref)


def _source_scaling(C, mag, d_sigma):
    """ Source scaling as per eq. 2 """
    return _magnitude_scaling(C, mag) + _stress_scaling(C, d_sigma, mag)


def _magnitude_scaling(C, mag):
    """ Magnitude scaling as defined in eq. 3 """
    fm = np.zeros_like(mag)
    dff = (mag - C['Mh'])

    idx = mag <= C['Mh']
    fm[idx] = C['e0'] + C['e1'] * dff[idx] + C['e2'] * dff[idx]**2

    idx = mag > C['Mh']
    fm[idx] = C['e0'] + C['e3'] * dff[idx]

    return fm


def _stress_scaling(C, d_sigma, mag):
    """ Stress scaling as defined in eq. 3 """

    # e_{\Delta,\sigma_1}
    keys = [f'g1_{i}' for i in range(0, 5)]
    e_d_sigma_1 = sum(C[k]*mag**(i) for i, k in enumerate(keys))

    # e_{\Delta,\sigma_2}
    keys = [f'g2_{i}' for i in range(0, 5)]
    e_d_sigma_2 = sum(C[k]*mag**(i) for i, k in enumerate(keys))

    # e_{\Delta,\sigma_0}
    e_d_sigma_0 = -2 * e_d_sigma_1 - 4 * e_d_sigma_2

    # scaling factor
    l_d_sigma = np.log10(d_sigma)
    return e_d_sigma_0 + e_d_sigma_1 * l_d_sigma + e_d_sigma_2 * l_d_sigma**2


def _kappa_scaling(C, kappa0, d_sigma, mag):
    """ Stress scaling as defined in eq. 3 """

    ld = np.log10(d_sigma)
    lm = np.log10(mag)
    lk = np.log10(kappa0)

    p_1_0 = C['d1_0_0'] + C['d1_0_1'] * ld + C['d1_0_2'] * ld**2
    p_1_1 = C['d1_1_0'] + C['d1_1_1'] * ld + C['d1_1_2'] * ld**2
    p_1_2 = C['d1_2_0'] + C['d1_2_1'] * ld + C['d1_2_2'] * ld**2
    p_1_3 = C['d1_3_0'] + C['d1_3_1'] * ld + C['d1_3_2'] * ld**2
    e_k_1 = (p_1_0 + p_1_1 * lm + p_1_2 * lm**2 + p_1_3 * lm**3)

    p_2_0 = C['d2_0_0'] + C['d2_0_1'] * ld + C['d2_0_2'] * ld**2
    p_2_1 = C['d2_1_0'] + C['d2_1_1'] * ld + C['d2_1_2'] * ld**2
    p_2_2 = C['d2_2_0'] + C['d2_2_1'] * ld + C['d2_2_2'] * ld**2
    p_2_3 = C['d2_3_0'] + C['d2_3_1'] * ld + C['d2_3_2'] * ld**2
    e_k_2 = (p_2_0 + p_2_1 * lm + p_2_2 * lm**2 + p_2_3 * lm**3)

    p_3_0 = C['d3_0_0'] + C['d3_0_1'] * ld + C['d3_0_2'] * ld**2
    p_3_1 = C['d3_1_0'] + C['d3_1_1'] * ld + C['d3_1_2'] * ld**2
    p_3_2 = C['d3_2_0'] + C['d3_2_1'] * ld + C['d3_2_2'] * ld**2
    p_3_3 = C['d3_3_0'] + C['d3_3_1'] * ld + C['d3_3_2'] * ld**2
    e_k_3 = (p_3_0 + p_3_1 * lm + p_3_2 * lm**2 + p_3_3 * lm**3)

    p_4_0 = C['d4_0_0'] + C['d4_0_1'] * ld + C['d4_0_2'] * ld**2
    p_4_1 = C['d4_1_0'] + C['d4_1_1'] * ld + C['d4_1_2'] * ld**2
    p_4_2 = C['d4_2_0'] + C['d4_2_1'] * ld + C['d4_2_2'] * ld**2
    p_4_3 = C['d4_3_0'] + C['d4_3_1'] * ld + C['d4_3_2'] * ld**2
    e_k_4 = (p_4_0 + p_4_1 * lm + p_4_2 * lm**2 + p_4_3 * lm**3)

    e_k_0 = 3 * e_k_1 - 9 * e_k_2 + 27 * e_k_3 - 81 * e_k_4
    out = e_k_0 + e_k_1 * lk + e_k_2 * lk**2 + e_k_3 * lk**3 + e_k_4 * lk**4

    return out


def _anelastic_attenuation(gamma, rrup):
    """ Anelastic attenuation term """
    return gamma * rrup


def get_gm_rock(C, ctx, kappa0, d_sigma, CAE):
    """ Compute ground-motion on bedrock """
    gm = (_source_scaling(C, ctx.mag, d_sigma) +
          _kappa_scaling(C, kappa0, d_sigma, ctx.mag) +
          _geometrical_spreading(C, ctx.rrup, ctx.mag)
          )
    if CAE is not None:
        gm += _anelastic_attenuation(CAE, ctx.rrup)
    return gm


class HassaniAtkinson2018(GMPE):
    """
    Implements the model of Hassani and Atkinson (2018) as described in the
    BSSA paper titled "Adjustable Generic Ground-Motion Prediction Equation
    Based on Equivalent Point-Source Simulations: Accounting for Kappa
    Effects" (doi: 10.1785/0120170333).

    :param d_sigma:
        Stress frop [bar]
    :param kappa0:
        Kappa_0 [s]
    :param gamma_fle:
        The name of the file containing the anelastic attenuation term.
    """

    #: Supported tectonic region type is active shallow crust, see title!
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: Supported intensity measure types are spectral acceleration, peak
    #: ground velocity and peak ground acceleration, see tables 4
    #: pages 1036
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is orientation-independent
    #: average horizontal :attr:`~openquake.hazardlib.const.IMC.RotD50`,
    #: see page 1025.
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.GEOMETRIC_MEAN

    #: Supported standard deviation types are inter-event, intra-event
    #: and total, see paragraph "Equations for standard deviations", page
    #: 1046.
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {const.StdDev.TOTAL}

    #: Required site parameter is Vs30
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: Required rupture parameters are magnitude and hypocenter depth
    REQUIRES_RUPTURE_PARAMETERS = {'mag', 'hypo_depth'}

    #: Required distance measures is Rrup
    REQUIRES_DISTANCES = {'rrup'}

    def __init__(self, d_sigma, kappa0, gamma_fle=None, **kwargs):
        super().__init__(d_sigma=d_sigma, kappa0=kappa0, gamma_fle=gamma_fle,
                         **kwargs)
        self.d_sigma = d_sigma
        self.kappa0 = kappa0

        # Read the file and create the coefficient table for the anelastic
        # attenuation term
        if gamma_fle is not None:
            with open(gamma_fle, 'rb') as fle:
                data = fle.read().decode('utf-8')
                self.CAE = CoeffsTable(sa_damping=5, table=data)
        else:
            self.CAE = None

        # Boore et al. coefficients, used in the site model.
        self.BEA14 = BooreEtAl2014.COEFFS

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        """ Compute mean """

        # Compute PGA on rock. This is used for computing the soil term.
        imt = PGA()
        C = self.COEFFS[imt]
        gamma = None
        if self.CAE is not None:
            gamma = self.CAE[imt]['gamma']
        pga_rock = np.exp(get_gm_rock(C, ctx, self.kappa0, self.d_sigma, gamma))

        # Compute ground-motion on soil.
        for m, imt in enumerate(imts):

            # Get the mean
            C = self.COEFFS[imt]
            if self.CAE is not None:
                gamma = self.CAE[imt]['gamma']
            mean[m] = get_gm_rock(C, ctx, self.kappa0, self.d_sigma, gamma)

            # Site term
            TMP = self.BEA14[imt]
            mean[m] += get_fs_SeyhanStewart2014(TMP, imt, pga_rock, ctx.vs30)

            # To natural logarithm and [g]
            mean[m] /= np.log10(np.exp(1))
            if re.search('SA|PGA', str(imt)):
                mean[m] = np.log(np.exp(mean[m]) / 980.665)

    COEFFS = CoeffsTable(sa_damping=5, table="""\
          IMT    Mh            e0            e1            e2            e3            b3            b4          g1_0          g1_1          g1_2          g1_3          g1_4          g2_0          g2_1          g2_2          g2_3          g2_4         d1_0_0          d1_0_1         d1_0_2          d1_1_0          d1_1_1          d1_1_2          d1_2_0          d1_2_1          d1_2_2          d1_3_0          d1_3_1         d1_3_2         d2_0_0         d2_0_1         d2_0_2          d2_1_0          d2_1_1          d2_1_2          d2_2_0          d2_2_1          d2_2_2          d2_3_0          d2_3_1         d2_3_2         d3_0_0         d3_0_1        d3_0_2         d3_1_0          d3_1_1         d3_1_2         d3_2_0          d3_2_1         d3_2_2         d3_3_0         d3_3_1         d3_3_2        d4_0_0        d4_0_1        d4_0_2        d4_1_0         d4_1_1        d4_1_2         d4_2_0         d4_2_1        d4_2_2        d4_3_0        d4_3_1        d4_3_2
          PGV   8.8  3.3020832702  0.4389691965 -0.0167235139  0.4146350561 -0.4414468351  0.0394123571  0.4442506135  0.0569722379  0.0036437698 -0.0040458203  0.0003530051 -0.0942034933  0.0533434608 -0.0141633591  0.0019185922 -0.0001002408 -40.8261642212   40.1130392571  -5.1021285604  155.2523126945 -182.3686804039   27.0333494027 -196.6717542861  258.2107952699  -42.3653301122   82.3574790678 -116.4544082139  20.5376093143 -31.5589270744  36.8747740244  -5.7395492329  125.2777710877 -161.3065178224   27.8740449631 -163.4368685527  223.2664465831  -41.3918330682   69.9031574762  -99.2526602356  19.3502214642  -10.376280011  13.0514009303  -2.256508542  41.8677381569  -56.0550276459  10.5303317116 -55.1925939231   76.6960142436 -15.2384616501  23.7760701622 -33.8327130594   6.9969744833 -1.2294334314  1.6040511763  -0.296017448  4.9934816807   -6.816710646  1.3502018879  -6.6090820328   9.2624147354 -1.9246297215  2.8547014114 -4.0664231392    0.87437342
          PGA   7.4  4.3737580679   0.330007551 -0.0093950747  0.3015333317  -0.456482013  0.0382499085  0.7096633248 -0.0383918613  0.0130808215 -0.0011379644   0.000023783 -0.0323847716  0.0190432118 -0.0045166161  0.0003745764  -0.000009338 -51.0987175177   32.5787964105  -0.6027113079  193.3769389098 -154.5366560363    8.9701410121 -246.3770135374  224.3093400193  -19.0560387445  103.8674068929 -102.8628006083  10.7674035794  -44.562063396  38.7082792804  -3.9223740867  176.7251880414 -170.7934709018   21.0155290004 -230.9326523974  237.8661600889  -33.0837191768   99.1364594808 -106.2748859661  16.0786467012 -16.4224378175  15.8479312655   -2.16864161  65.9977373822  -67.9730386386  10.3855144404 -87.0701819028    93.033434856 -15.3297368832  37.6386624177   -41.09486034   7.1483955895 -2.1340416665  2.1508934311 -0.3370120058  8.6187576308  -9.0895962533   1.542770572 -11.4166836197  12.3244901435 -2.2132358804  4.9500459039 -5.4101159109  1.0122830622
         0.01  7.15  4.6451420678  0.3440412764 -0.0081551718  0.3056467387  -0.432123999  0.0352795984  1.4724155428  -0.541245476  0.1337792132 -0.0136064606  0.0004916184 -0.2222773523  0.1462761965 -0.0353526956  0.0035796845 -0.0001301312 -41.0107982823   20.9848341026   4.3329321558  143.7259148271 -112.5739642323   -8.7987572623 -185.5027172857  174.2625858466    2.1456980236   78.7931994598  -83.1287030049   2.3805087274  -35.565129626  27.7977763589   0.5050803946  130.2847004071 -131.1409418798     4.976748558 -173.6461342678  190.4355184412  -13.8639022128   75.4765505105  -87.5347798957   8.4534629675  -13.145746842  11.6016707688 -0.5525871582  48.0682184483  -52.4511024164   4.4788312726 -64.7659436448   74.3870319334  -8.2074496141  28.3889668495 -33.7045319011   4.3105214888 -1.7158836168  1.5762228316 -0.1320599394  6.2060720835  -6.9773140963  0.7864628056  -8.3878736064   9.7767580547 -1.2952752002   3.688341554 -4.3974049937  0.6448738171
 0.0125892541   7.4  4.6977535858  0.3384826092 -0.0082478225  0.3049000591 -0.4334631181  0.0359538028  1.5542238909 -0.5616075335  0.1313999067 -0.0127660208  0.0004433715 -0.2647504161  0.1652033142 -0.0379324038  0.0036832174  -0.000129329 -36.8285222558   19.5692518096   6.1259093932  133.7996869212 -108.9284407922  -14.5794277424 -175.1080288423  171.4375896683    8.4533260504   74.8445969118  -82.5435219825   0.0547346719 -31.1722799717   25.351333991   2.5316521888  118.7523252911  -123.338463955   -1.8054418993 -161.1733562416  182.1538053597   -6.1941164726   70.6651512206  -84.6018148804   5.5292344598 -11.2811780794  10.2045187952  0.3562641019  42.7543021769  -47.6413957823   1.3509704199 -58.8762358324   68.8542430537  -4.5789001237  26.0881979323 -31.5741788193   2.8951491463 -1.4497966017  1.3380674799  0.0036611068  5.4031719146  -6.1290931231  0.3107172861  -7.4856343948   8.7688733438 -0.7343493645  3.3334717308 -3.9973330647  0.4229556198
 0.0158489319   7.5  4.6991251896  0.3360164017 -0.0089372225   0.306043905  -0.428188954  0.0351373029  1.7146520875 -0.6426995059  0.1451525465 -0.0136492035  0.0004586999 -0.3499803018  0.2131200216 -0.0476059211  0.0045158963 -0.0001551993 -33.8430866217    21.852771734   6.4156265557  132.5335445804 -119.6575221006  -14.2894251369 -178.4664241493  186.8869254686    6.8355344807   77.8849827279  -89.6021754288   1.1066840508 -28.5771731971  27.5478380522   3.0405208004  118.5562094533 -134.3070025864   -2.2410582856 -166.1464288973  198.5478126675   -7.0358607721   74.4547367079  -92.2926779649   6.3136158245 -10.4091288278  11.0983245447    0.62375221  43.1805091637  -52.2978627131   0.9744495819 -61.6114017509   75.9878984805  -4.7119117615  27.9468069672 -34.9786896007   3.1445023355 -1.3504612482  1.4651275983  0.0464012634  5.5224345322  -6.8079421643  0.2406126264  -7.9489981997   9.8237757891 -0.7349478952  3.6283937608 -4.5057461333  0.4520391597
 0.0199526231   7.8  4.7556082962  0.3269384891  -0.010314344  0.3033860929 -0.4209048867  0.0343552884  1.9121433951  -0.751222247  0.1674140149 -0.0156735083  0.0005277542 -0.4478882645  0.2694635191 -0.0596360655  0.0056440975 -0.0001944388 -42.8277325797   41.1075882423   1.7670056326  175.9994852055 -195.1960146988    4.9952066498  -233.844386437  282.7603313271  -18.6047750739  100.4408355511 -129.2075056017  11.9279871025 -36.3622887447  45.1839849821  -0.9145728042  157.3334082243 -204.0219744559   14.5248123577 -216.2043782902  287.5455973403  -29.4581795161   95.0157015348 -129.2263383639  15.9404511618 -13.0410584594  17.5269872765 -0.7135972297  56.7766408002  -77.8755312445   6.7787823282 -79.3772814829  108.7930409071 -12.5851432285  35.2896380599 -48.6391063011    6.556052291 -1.6597774076  2.2800634076 -0.1116247994    7.17850873 -10.0680160362  0.9427128873 -10.1378354947  14.0210536456 -1.7002327916  4.5380100974 -6.2582653091  0.8738869272
 0.0251188643  8.95   5.065654626  0.2963765967 -0.0119281989  0.2969373901 -0.4098087961  0.0331186139  2.1539138139 -0.8927591847  0.1982305382 -0.0186268734  0.0006330414 -0.5719792694  0.3427423146 -0.0756752153   0.007182719  -0.000249038 -48.4797219107   55.7785000258  -4.0133993949  203.1744408844 -245.4657543936   25.9385381009 -266.7082108364  340.0759448663  -43.5920265039  113.9342895071 -151.0624029213  21.7937101574 -42.8382826139  60.8412193483  -6.4582542464  188.7203161972 -259.2502197726   35.0071150867 -255.7087778747  352.3306699851  -54.3115830467  111.6496865619 -154.5931007171  25.8987670888 -15.8365807353  23.9817850436 -2.8287849497  70.4391524705 -101.0599279693  14.7120663427 -96.9893067286  136.4572750195 -22.3325442459  42.8120570944 -59.6381792273  10.5030952707 -2.0648079922  3.1842048796  -0.391808802  9.1648695756 -13.3553052991  2.0057140981 -12.7365278013  17.9875948249 -3.0185636914  5.6573769687 -7.8508252878   1.411862961
 0.0316227766  8.25  4.8042229815  0.3066648064  -0.013746637  0.3078399835 -0.4074900854  0.0326778324  2.2779443538 -0.9647994178  0.2136928573    -0.0200645  0.0006814478 -0.6822120101   0.407086072 -0.0895727855     0.0084928  -0.000294393  -36.454347243   50.7470670963  -7.8549338548  150.7810895953 -212.7678867723   36.8724293356 -186.1677313642  284.2653730155  -53.1909335756   75.3824228617 -122.5808344289  24.2424134809 -31.5145157956  58.5002431151 -10.4300311739   140.047746542 -237.7623187054   46.9727499486 -180.7645677206  311.4806214509   -65.709208285   75.6530222016 -132.5800920099  29.2516732748 -11.5212403813  23.8157827678 -4.4385318785  52.1262556172  -95.7306291658  19.7452677421 -68.7772422049   124.527904845 -27.3598748843  29.2289366846   -52.75741697  12.0889092726 -1.4922527907  3.2332969498 -0.6141896891  6.7573608264 -12.9297889021    2.71801631  -9.0279043013  16.7643172496 -3.7506979128  3.8689317834  -7.087304533  1.6518175591
 0.0398107171   8.1  4.7102513005  0.3074828309 -0.0158357046  0.3073252909 -0.3942597934  0.0311008798  2.4257618805 -1.0486282081  0.2324936506 -0.0219985219  0.0007563393 -0.8116170327  0.4813015793 -0.1056517274  0.0100416538 -0.0003499646 -10.3296808074    9.7218351195  -2.4289715727   43.8835373652  -42.5548489301   10.0813149178  -39.4358854685   56.8097930842  -13.5285479233   10.6624405151  -23.9026730425   5.8525266906 -11.5238638904  27.2776685811  -7.0845191601   59.6125010155 -107.0341307722   29.1340653287  -69.9204347271  135.7820298694   -38.279501508   26.6462718907  -56.0638208278  16.2451257932  -5.0659154137  13.8325098524 -3.6007584002  26.7017141063  -53.5712843183  14.8041092212 -33.6834840085   67.5735160397 -19.4227798758  13.7041599117 -27.8736875237   8.2332014518 -0.7428266299  2.0860295086 -0.5425549506  3.8691868222  -8.0463132325  2.2316178293  -5.0391659319  10.1365352213 -2.9280321585  2.1048431281 -4.1834659625  1.2414397554
 0.0501187234   7.4   4.429340924  0.3166116365 -0.0210928142  0.3165357815 -0.3872042221  0.0302957982  2.2156974197 -0.8862265115  0.1911293198 -0.0176427578   0.000592351 -0.8563075456   0.494428259 -0.1063612452   0.009952747 -0.0003425328  27.0850641033  -63.7190986405  13.8342593134 -111.8532480444  249.1446647122  -59.2885275673  168.9499140029  -319.851083992   80.3566297996  -78.7585919722   134.893707937 -35.0579289393  23.6379766877 -38.4638846363   6.9208004148  -85.7477877924  155.8694392906  -31.1593470023  125.0726821873 -205.7601512083   43.9237705731  -57.3133616236   88.7120740467 -19.7948971478   8.1941856291 -10.0556218022  1.3573845746 -27.7941444041   42.4356361932  -6.6860583354  39.5670703652  -57.7111164583  10.0380863182 -17.9220676877    25.44934151  -4.7433074448  0.9970660559 -0.9622329303  0.0784628668 -3.2497631088   4.2505490264 -0.4738884899   4.5443928476   -5.964418711  0.7964843163 -2.0415302867  2.6903239242 -0.4049161701
 0.0630957344  6.85  4.1853555673  0.3267895857 -0.0276700262  0.3263815938 -0.3816353632  0.0301815426  1.5013679858 -0.3856248897  0.0679066885 -0.0048087958  0.0001099302 -0.7802019862  0.4244420911  -0.086695294  0.0077403729 -0.0002551261  51.1666406905 -130.6664837133  34.9670234774 -210.7195191836  498.9518851785 -142.5647839443  294.3730887277 -626.3972935527  186.5756079083 -129.8069489939  258.8850624229 -79.2621144666  46.6918844026 -99.6681916738  25.4180021761 -179.9661292356  386.4359650967 -104.6972771283  245.4105148838 -490.9231980829  138.3702405627 -106.5622354207  204.8012161561 -59.3178174586  16.9946509656  -32.642702226  8.0005981931 -63.5145803574  127.9964173966 -33.2386462651  85.3153644814 -163.9914945403  44.2792255459  -36.683226732   68.862481446 -19.1177115737  2.1631923026 -3.8800070864  0.9209758723   -7.95435307  15.3425907797 -3.8532069895  10.5772316838 -19.7797542237  5.1657914227 -4.5177376833  8.3450314582 -2.2428443416
 0.0794328235   6.1  3.8684739719  0.3355781662 -0.0485851618  0.3355382597 -0.3693744528  0.0283527647  0.2348472015  0.4704790756 -0.1376204277  0.0162373354 -0.0006719543 -0.5504798596  0.2521788408 -0.0427922227  0.0030693244  -0.000076988  -7.7636417404  -66.1496056368  26.9523163091    8.9758103214  219.6493165751 -103.3790755539   15.4057989378 -240.4576359003  127.1045681035  -12.5726340634   86.7353328772 -50.7178797448   0.0646037578 -52.7795833585  20.6306748825   -6.9286445457  179.8937465953  -79.8186980652   26.5565609653 -201.9567966837   99.0410551131   -14.748670151   74.6922317722 -39.8903524421   1.3713776752 -17.9386835505  6.7856191162   -5.622000186   62.2748938123  -26.435580008  12.3083993586  -71.1102818725  33.0358303153  -6.0963601452  26.7309338938 -13.4000889005  0.2789931324 -2.1980663728  0.8105845056 -0.9736908818   7.7318935747 -3.1749876251   1.7928548842  -8.9342753876  3.9893496846 -0.8410236507  3.3959670652 -1.6268421598
          0.1   5.9  3.7306312263  0.3412300018  -0.065869952  0.3408640601  -0.352445431  0.0265250085 -1.5748728214   1.682296287 -0.4263895761  0.0455874333 -0.0017536344 -0.1518560063 -0.0327600309  0.0277509596 -0.0042782164    0.00019837 -68.4584808555   28.8296885322   4.6055618374   237.828030328 -164.2666966224   -8.8364439771 -274.7648414346  260.0220513999   -0.9339373641  107.1387612987 -125.3748477634     5.31707408 -49.2973802529  21.6311909944   3.5186380577  177.9833174727 -122.6159568827   -7.0615069309 -206.8330558247  194.0177681139    0.0625499188   81.2412856002  -93.6598224191   3.5999615813 -15.6001241536   6.9828876851  1.1492038421  57.7905120359  -39.4623503224  -2.3852809988 -67.5101499858   62.4586866877   0.2110023906  26.6791445304 -30.1911588424   1.0652384365  -1.812876912  0.8165345421   0.137289781  6.8351878151  -4.6123470507 -0.2944384024  -8.0193717571   7.3082780654  0.0482165074  3.1845925033  -3.538277842  0.1137946823
 0.1258925412   5.7  3.5958686563  0.3457374118  -0.091198369  0.3455371221 -0.3417594906  0.0251987589 -3.7031057056  3.0667936054 -0.7491847826   0.077858571 -0.0029284383   0.381196376 -0.3993042179  0.1161756449 -0.0133213793  0.0005328625 -57.6614085851   52.2101531799  -6.5340085527  177.2434650293 -234.3683113559   33.4799257247 -183.2380501499  322.7072974311  -52.1896270303    62.323522194 -140.8212428576  25.3293098147 -42.1101841379  41.0447474284  -5.3887075493  133.9449721271 -182.9370310775   27.0780940499 -138.7571681444  251.1195172969  -41.7306512182   47.4259054209 -109.4540239153    20.11269377  -13.483785505  13.7583833791 -1.8817902918  43.8707410668  -61.0080173149   9.3041967093 -45.5797866859   83.5696822617 -14.2052999719  15.6622992163 -36.3995502437   6.8079540709 -1.5814899248  1.6611432948 -0.2338657936   5.224072633  -7.3409049847  1.1434152592  -5.4418772075  10.0425517849 -1.7344760497  1.8784351562 -4.3727261512  0.8280584136
 0.1584893192   5.5  3.4497618625  0.3526134347 -0.1253933749  0.3525096552 -0.3386420546   0.025211426 -5.4613288942  4.1543784819  -0.991765532   0.101167973 -0.0037470472  0.8908286605 -0.7375908727  0.1952694428 -0.0211856259  0.0008163879 -24.1771270565   31.5057486724  -6.3012017282    42.856646953 -131.4421294128   28.5901529536   -12.536822638  166.6237148171  -40.3867859764   -9.7257720547  -66.4033644826  18.0602818108 -17.0001396313  26.1599915548  -5.5456908075   31.6325320857 -107.9725844095   24.7207658495   -8.1701026677  136.3163669533  -34.5180107556   -7.9641766484  -54.2945891342  15.3188782163   -5.341652215   9.0995365531 -2.0180104894  10.3477684261  -37.2837964102   8.8750473128  -2.6069590715    46.936269202 -12.2841785369     -2.6437437 -18.6851574103   5.4203922716 -0.6200895376  1.1269902465 -0.2579150548  1.2363426535  -4.5953440584  1.1239568098  -0.3115197598   5.7737832158 -1.5464266064 -0.3147988124 -2.2976160209  0.6796794351
 0.1995262315  5.55  3.3923605786  0.3602997081 -0.1413442501  0.3579168731 -0.3261850027  0.0243650446 -6.7018386017  4.8428481185 -1.1284975226  0.1128208698 -0.0041093787  1.3412651352 -1.0221971406  0.2585603683 -0.0271855295  0.0010233916 -23.6603443359   26.4640310772  -5.4261084482   55.7912305175 -112.0606207342   24.6369636426   -47.626855193  146.5180596581   -35.041155878   11.2808351516  -60.8885768615  15.8379939299 -16.8282538817  21.4472589447  -4.6397399984   41.8100744099  -89.7280151303   20.7047092125  -35.8764912344  116.5908957867  -29.1067229314    8.6816155671  -48.2989493011  13.0518586753  -5.2889941154   7.3196956585 -1.6662095271  13.5826800785  -30.3517310858   7.3244072939  -11.670176524   39.2453744507 -10.1944177869   2.8533020582 -16.2126365603   4.5405735666 -0.6112686343  0.8930670707 -0.2113578915  1.6020629097  -3.6801303218  0.9191670944   -1.373910847   4.7413557459 -1.2702003891  0.3364472613 -1.9544432332   0.562998143
 0.2511886432  5.75  3.4024056321   0.359494885 -0.1448290497   0.357607348 -0.3077823707  0.0218772196 -7.1619369709  4.9757001877 -1.1257647377  0.1096855811 -0.0039080822  1.6473224703 -1.1975091894  0.2930108986 -0.0300069617  0.0011055908 -14.4453887135   18.4966671285  -3.7315585177   24.5481443774  -76.1411436292    16.603879761  -11.9759908226   97.2147911164  -23.1691014164   -2.3246173229   -39.552575663  10.3024062033  -9.9175475447  15.1682812459  -3.2585943132   17.6095356665  -61.5296181821   14.2215334331   -8.0052118495   77.8930126017  -19.5732509606   -2.1027677322  -31.5223535469   8.6162950145  -2.9965192097   5.1467871569 -1.1831732013   5.3666893153  -20.6387846366   5.0711554387  -2.1181500332   25.9241065212  -6.8924269626  -0.8863768191 -10.4287403705    3.006621484 -0.3324803706  0.6201402179 -0.1505008441  0.5867764247  -2.4653485097  0.6365992009  -0.1842143852   3.0765532115 -0.8572399895 -0.1334983894 -1.2308281967  0.3714025684
  0.316227766   5.9  3.3904815902  0.3610054068 -0.1495338004  0.3602790411 -0.3084956335  0.0223410857 -6.5416559674  4.3470296551 -0.9338506538  0.0866261002 -0.0029491416   1.717463815 -1.1996014938  0.2829507911 -0.0280148005  0.0010009573 -12.4571982754   13.4526035589  -2.2960758343   25.0419189827   -56.590694865   10.6651319374  -20.4292614556   74.0834356615  -15.3320329292      4.09878749  -30.9444019764   6.9647059986  -8.4969944075  11.1931863186  -2.1333866238   17.8927647072  -46.1070033694    9.5539854759  -14.4665321126   59.5898435801  -13.3991913591    2.8386539977  -24.6775846344   5.9810792692  -2.5750118956   3.8453857964  -0.820947882   5.5315012964  -15.5784380327   3.5586977588  -4.3456218497   19.9064819934  -4.8821626714   0.7827403697  -8.1733923818   2.1453809724 -0.2874936598  0.4673661463 -0.1087635098  0.6185985469  -1.8701060441  0.4610946905  -0.4657664999   2.3679693257 -0.6228085062    0.07209907 -0.9651067784  0.2705936125
 0.3981071706   6.2  3.4344726937  0.3547114962 -0.1421671394  0.3592644452 -0.2965977772  0.0212260759 -4.8154389796  2.9693291223  -0.563840001   0.045648218 -0.0013386935  1.5112911693 -1.0068678279  0.2248303239 -0.0210298682   0.000709735 -11.1714288977   13.4247596667  -2.3126759793   23.8285452447  -55.2129660534   10.5124921483  -20.5166764105   71.2664927434  -14.8567559791    4.6299206154  -29.4690829205    6.663017642  -8.7846400405  12.5321159192  -2.4312176459   21.4319522802  -50.4738736974   10.6093392359  -20.2027416617    64.497179632  -14.6056807113      5.59516093  -26.5599737371   6.4364150198  -2.8895971868   4.4882250715  -0.964384637   7.4352330104  -17.8049793195   4.0855910137  -7.1484813461   22.5344496703  -5.5134944385   2.0601591962   -9.219451867   2.3957023825 -0.3391773342  0.5545423712 -0.1284870987  0.8904253427  -2.1749807491  0.5336439357  -0.8534809196   2.7301904213 -0.7101812583  0.2452310997 -1.1098274242  0.3054405857
 0.5011872336  6.45  3.4498984617  0.3607438959 -0.1359072465  0.3632280923   -0.28125334  0.0201580254 -2.4781119819  1.2039956193 -0.1078942399 -0.0033519898  0.0005407045  1.1218713087 -0.6893550069  0.1369772714 -0.0110645377  0.0003112484 -11.8124076479   13.8103810756  -2.0938028474   31.4447779368  -57.8948328027    9.8779714878  -33.6999627222   76.3360054811  -14.2734263286   11.4352442815  -32.2815618215   6.4996395477  -9.5131576307  13.2732304476  -2.4110773938   27.7565065487  -54.1946119297    10.699908681  -30.8473938722   70.4088315186  -14.8903857108   11.0012452217  -29.5252697455   6.6148632772  -3.1217379547    4.754601397 -0.9822657512   9.3882954163  -19.0888311674   4.2033633662 -10.4608173485   24.5222316014   -5.710878828   3.7487095236 -10.2008345394   2.4947894181 -0.3635523498  0.5847582425  -0.132423398  1.1023401978  -2.3190977532  0.5536695187  -1.2191005868   2.9524926058 -0.7402472436  0.4333155303 -1.2195346643  0.3196128652
 0.6309573445   6.8  3.5255680829  0.3615410624 -0.1252002146  0.3600245607 -0.2650472407  0.0184867278  0.5264801463 -0.9708815969  0.4357565334 -0.0602721467  0.0026777319  0.5113490397  -0.224277741  0.0145783011  0.0023185749 -0.0002092516 -13.5408506606   17.0557757525  -2.8046653887    41.258829488   -70.294927866   12.7102571661  -48.2267899793    92.592344052  -18.0454882853     18.39272956  -39.4585086227   8.1708818317 -10.5620347137  16.1156782464   -3.036012741   33.6621322206  -64.8967488077   13.2011291924   -39.691195189   84.2066324024  -18.2228284086   15.2989539818  -35.5208597085   8.0875036988  -3.3484907348   5.6515455026   -1.17536018  10.7663422295  -22.4301345835   4.9787461015 -12.6091578308   28.8013705206  -6.7483441792   4.8304251565 -12.0537009231   2.9552802759 -0.3765717146  0.6806265514 -0.1523510867  1.2044592112   -2.672388024  0.6339885731  -1.3931899619   3.4032404524 -0.8484458214  0.5270275489 -1.4148979421   0.368027541
 0.7943282347   7.1   3.571361991    0.35652637 -0.1193241697  0.3570237495 -0.2649773794   0.018826848  3.4251849396 -2.9913246599  0.9214368763 -0.1092201354  0.0044499058 -0.1495126457  0.2603579955 -0.1085146009  0.0153528195 -0.0007017741  -9.7912128176   13.7699914453  -2.0580225679    28.968177401  -57.0642691601    9.6642954622  -33.5094137541   75.2020238203  -13.9470982658   12.6116166251   -31.974637592   6.3600239497   -8.121295776  14.8107350352  -2.7666379008    25.532425839  -59.4131216238   12.0808421183  -29.8141599743   76.6964155363  -16.6573095449   11.3386815693  -32.1614152907    7.365445564  -2.7354249088   5.5458057007 -1.1667486745    8.765383814  -21.8911747043   4.9315657038 -10.1863721671    27.942209804  -6.6523805282   3.8565136457  -11.620004502   2.8960020993  -0.324788036  0.6980589082  -0.158691045  1.0447286742  -2.7267077902  0.6579918521  -1.2034485193   3.4533488086 -0.8757474779  0.4515129887 -1.4273765368  0.3775010833
          1.0   7.4  3.6164291057  0.3597726264 -0.1119255063  0.3594945547 -0.2661810877  0.0201321375  5.9490466324 -4.6713273052  1.3046107116 -0.1456700116  0.0056901455 -0.8075476906  0.7262022959 -0.2227593676  0.0270341654 -0.0011281454  -7.3256048785   13.6140019159  -2.3866257947   20.4853894959  -55.6484349283   10.8451697256   -22.683851275   72.7248157795  -15.3582540654    8.1028782765  -30.7620665016   6.9246232071  -6.6000198371  15.8897679935  -3.3298834917   20.2206068601   -62.917428455   14.2487598296  -22.9483502552   80.5235089622  -19.3913914982    8.4426735374  -33.5701422174   8.5010971772  -2.3998508944   6.2056652848 -1.4262398215   7.6268407002   -24.238636318   5.9470998329  -8.7064714214   30.7266312873  -7.9503662549   3.2295781221  -12.720858172   3.4405936692 -0.3041661274  0.8052389962 -0.1963047274  0.9832228555   -3.119945459  0.8063911938  -1.1233704201   3.9321336409 -1.0667220577  0.4176807388 -1.6208055989  0.4580693737
 1.2589254118   7.7  3.6545484068  0.3575998427 -0.1057179109   0.356531999 -0.2782206558  0.0225276051  7.3681310616 -5.4999740698  1.4592526629 -0.1565216235  0.0059096678 -1.2840034335  1.0456808207 -0.2960404231  0.0339570093 -0.0013588406  -8.9399578514   14.6746850813   -2.505880466   28.4876600854  -60.5337280733   11.4471773346   -33.749540108   79.8395624305  -16.2724487262   13.1105579967  -34.0939513081   7.3635321984  -6.8232986367  16.3491826384  -3.3782515204   21.9455855601  -64.7884294325   14.4731677049  -25.4370197716   83.0143230851  -19.7001865816    9.6229790027  -34.6664921399   8.6368222002  -2.2300571005   6.2279924266 -1.4188136909   7.1397072806  -24.2394246245   5.9108681413  -8.0891976452   30.6324215459  -7.8881995559   2.9805703518 -12.6500681349   3.4074527883  -0.262856779  0.7946727887 -0.1928172797  0.8335907638  -3.0586362914  0.7900493548  -0.9245930055   3.8313169616 -1.0417456072  0.3322744273 -1.5706124363  0.4459012978
 1.5848931925   7.0  3.2857734297  0.6097338179 -0.0885782972  0.3841044767 -0.2895406921  0.0245989884   7.648530693 -5.4787108915  1.3931838181 -0.1435055576   0.005210428 -1.5563352471  1.2072403932 -0.3268127032  0.0361301549 -0.0014013408 -15.6267755898   19.5080593601  -3.5605314144   56.9852336746  -81.2724131551   15.9496386613  -71.9465712177  108.4182979312  -22.4363383422   29.7887407551  -46.8304019237  10.0923559631 -11.7684133905  20.4850750292  -4.3731437166   43.0184773363  -82.1740586811     18.66832572  -53.7804207458  106.5984888816  -25.3827671998   22.0293625855   -45.049167858  11.1300957791  -3.8204178224   7.6642307761 -1.7785147234  13.9181600796  -30.2217997137   7.4206289208 -17.2285461072   38.6888757883  -9.9248726382   6.9880366475 -16.1763205766   4.2978929131 -0.4494593724  0.9707618833 -0.2378174021  1.6292986976  -3.7888067179  0.9785393507  -1.9994408429   4.8110475767 -1.2955410523   0.804228697  -1.998174143  0.5566794825
  1.995262315  7.05  3.2049965568  0.6867309199 -0.0730270901  0.4021565328 -0.3001730944  0.0272869036  6.5009005488 -4.4139718949  1.0596086003 -0.1017082555  0.0034036819 -1.5296436434  1.1441104045 -0.2982329547  0.0317315593 -0.0011842892 -20.1960639891   21.6055845816  -3.9671594926   76.9558822304  -91.9371330215   18.0091022755  -99.1784901924  124.8886042533  -25.6256909132   41.8549374142  -54.7932637962  11.6418776548 -14.2376462653  21.4583001699   -4.630732686   54.2049540462  -87.4470244094   19.9803977582  -69.3873921459  115.0404659596  -27.4128595716   29.0659150266  -49.2249083259  12.1135286806  -4.4509644197   7.8794384564 -1.8541160076  16.8697479568  -31.4681943381   7.8067794295 -21.4317771363   40.7554636244 -10.5217536456   8.9120032274 -17.2206911675    4.586453303 -0.5129490306  0.9903848307 -0.2463738852  1.9342451795  -3.9084646085  1.0221728619  -2.4407618904   5.0146045608 -1.3628308909  1.0086004307 -2.1026437352  0.5891390921
 2.5118864315   7.2  3.1576062661  0.7182870656 -0.0633135535   0.422258717 -0.3243310103   0.032269761  4.4712377568 -2.7316187837  0.5755552666 -0.0446086239  0.0010424904 -1.2961313655  0.9303042089 -0.2312090051  0.0232306818 -0.0008112952 -21.0256536528   20.9146813933  -3.6575896188   80.2541587986  -89.9299623313   16.8386925588 -102.8672950437   122.982399485  -24.1638644623   43.2151911251  -54.2040602215  11.0392722356 -14.0588235626  20.3381797663  -4.3151386648   53.5788566869  -83.2808085004   18.7297517823  -68.2708070338  109.8605415716  -25.7765170036   28.4776069422  -47.0814527582  11.4088868273  -4.2203266299   7.3844256493  -1.728543227  16.0043471205  -29.5510451971   7.3034369707 -20.2460042913   38.2936647035  -9.8561163949   8.3823550516 -16.1759435595     4.29706609 -0.4710115161  0.9203065635 -0.2291256019  1.7760099786   -3.633508927  0.9529128758  -2.2313731002   4.6581700587 -1.2710936968  0.9176474424 -1.9503343838  0.5491990355
 3.1622776602  7.25  3.0744436887  0.7916821056 -0.0484992643   0.439448184 -0.3362655551  0.0347566543  1.7004096902 -0.5510132725 -0.0240901205  0.0235591069  -0.001691644 -0.8569787701  0.5704365482 -0.1279204877  0.0109751519 -0.0003001327 -22.7410654957   22.4113232837  -4.0962932971   86.6321200609  -97.0204107064   18.7941351914 -110.2342524931   133.365047689  -26.9408607716   46.0341130284  -59.0219141007  12.3078427671 -15.2639721536  21.4255214206  -4.7172319252   58.1094326082   -88.201374178   20.4437169856  -73.5766819602  116.8469476674  -28.1245927122   30.5295892104  -50.2520003072   12.450839956   -4.618710154    7.768048443 -1.8814029229  17.5088835851  -31.2150165481   7.9400357389 -22.0246646611   40.5869460018 -10.7113954384    9.075047828 -17.1937827054   4.6704233942 -0.5200722519  0.9703469698 -0.2493654837  1.9614720054  -3.8444317663  1.0361509301  -2.4517686191    4.942790342 -1.3817281231  1.0037815747 -2.0746013587  0.5970482799
 3.9810717055  7.35  3.0068769412  0.8374853908 -0.0381578519  0.4568235091 -0.3529564847  0.0379067937  -1.212860407  1.6599514315  -0.609562804  0.0876371821 -0.0041672285 -0.3309580379  0.1597031539  -0.015464143 -0.0017928777  0.0002112414 -26.2642683833     23.77125245  -4.0600543127  100.4723014588 -103.6463081839   18.8142823254  -127.685929156   143.161093468   -27.132821859   53.2785856302  -63.5727503929  12.4442193837 -17.7517744306  21.9652599806  -4.6270962417   68.1162764947  -91.1788182551   20.1837774851  -86.4673320533   121.521772285  -27.8789898943   35.9754166484  -52.5004112965  12.3745119814  -5.4010814145   7.8590481743 -1.8420068856  20.7060622979  -31.8305838151   7.8107058294 -26.2018972813   41.6328884279 -10.5674404814  10.8600524685  -17.718161219   4.6160526673 -0.6112283468  0.9757068778 -0.2441523499  2.3380440661  -3.8946665757  1.0183477869  -2.9485647122   5.0361976235 -1.3610742408  1.2177327242 -2.1234587236  0.5889343431
 5.0118723363   7.4  2.8863979901  0.8761701567 -0.0300809599   0.494472907 -0.3688931822  0.0409760006 -3.8896619277  3.6472431247  -1.123763816  0.1425881149 -0.0062388468  0.2070803357 -0.2513939403  0.0947734672 -0.0140724906  0.0006943685 -25.7057753316   24.0764870545  -3.8703593636   96.6299130503 -105.0203223551   18.0902179351 -120.7036700695  145.0266967878  -26.2282367214   49.6101090555  -64.3603050169  12.0710777046 -16.8400900423  21.1590531205   -4.272283011    63.439586357  -87.9918917847   18.7589825275   -79.104292167  117.3482960629  -26.0111674172   32.3906723316  -50.6930080159  11.5722780423  -4.9749838544   7.3593979319 -1.6779374224  18.6999214568  -29.8259447656   7.1458041401 -23.2146846024   38.9941945583  -9.6884668771   9.4549304529 -16.5777883287   4.2359706951   -0.54859969  0.8962112134 -0.2205512192  2.0540699083  -3.5743237702  0.9223614767  -2.5370284339   4.6137685769 -1.2337658517  1.0275780044 -1.9409142964  0.5337369216
 6.3095734448   7.4  2.7145837156  0.9136920622  -0.022484062  0.5424418719 -0.3897339855  0.0446430908 -6.0708435731  5.2140490076 -1.5133826253  0.1822767327 -0.0076542992  0.6991286473  -0.617005313  0.1898484823 -0.0243145634  0.0010833074 -24.7741102608   26.2245261356  -4.2004794939   90.8563796624 -112.9422844125   19.2973590395 -110.8814030446   154.637439971  -27.7030506371   44.6437877503  -68.1972251506   12.669506268 -16.3990628492  22.4992962093  -4.4541046335    60.312114837  -92.8570002862   19.3916006589  -73.5383591604  123.2074399332  -26.7578257653   29.5222708533  -53.0294842103  11.8686413016  -4.8826471818   7.7389607193 -1.7237387703  17.9239199003  -31.1866356906   7.2970280803 -21.7691562778   40.6252764836  -9.8605280952   8.6968396787 -17.2289904654   4.3028104713 -0.5416584244  0.9369746218 -0.2250093354  1.9811404057  -3.7192344466  0.9363850802  -2.3946926693   4.7871924624 -1.2491964921  0.9516993639 -2.0103910363  0.5396320596
 7.9432823472  7.45  2.5917169328  0.9472329304 -0.0155049898  0.5747947627 -0.4050606111  0.0472347676 -7.9678553541    6.53620996  -1.828752718  0.2125584795 -0.0086490871  1.1492239488 -0.9444905341  0.2727710514 -0.0329476371  0.0013975134 -23.2553995113   27.2691744178  -4.3551363064   83.8825933531 -116.9256606172   19.8961328484 -100.7237168778  159.5123686554  -28.4508015899   39.9595423262  -70.1283407588  12.9725152406 -15.5848632749  23.2275156197  -4.5486195116   56.4495087295  -95.5980759291   19.7391247686  -67.7697736114  126.5101496205  -27.1647461194   26.8124084301  -54.3146504366   12.021431467  -4.6784720275   7.9584046824 -1.7487960719  16.9233555175  -32.0027327941   7.3840728214 -20.2395489097   41.5941883071  -9.9546342549   7.9661975381 -17.5994958414   4.3344232811 -0.5215595225  0.9610430113 -0.2274099546   1.879886887  -3.8076857365   0.944166231  -2.2369694281   4.8906655061 -1.2567074541  0.8753696506 -2.0492510733  0.5416908604
         10.0  7.45  2.4150068661  0.9769528489 -0.0094547258  0.6130675421 -0.4202596844  0.0498150631 -8.9354219345  7.1448354269 -1.9524028243   0.221535124 -0.0088127276  1.4619812362 -1.1632889723  0.3255599324 -0.0381284772  0.0015731751 -21.1165651659   26.9505956594  -4.2954775018   74.8408208092 -115.4589237674    19.652996511  -88.3974310928  157.3435337055  -28.1316687888   34.5467928364  -69.0887596449  12.8339016048 -14.3093444264   23.080364195  -4.5151781851   51.0659752649  -94.9124015394   19.6050797556   -60.439984641  125.4898599103   -26.993140092   23.5975167866  -53.8220519431  11.9487252124  -4.3337070093   7.9320602539 -1.7408147062  15.4695949849  -31.8751183648   7.3526146861 -18.2615562803   41.3996295877  -9.9155280555   7.0991477759 -17.5033275869   4.3183388622 -0.4861689116  0.9596498027 -0.2267020786   1.730713456  -3.8001217853  0.9414137525  -2.0340294536   4.8783480293 -1.2533876179  0.7864120292 -2.0428058158  0.5403679552
""")
