# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2015-2017 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:'ZalachorisRathje2019'
"""
import numpy as np
from openquake.hazardlib.gsim.base import CoeffsTable, GMPE
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA
from openquake.hazardlib.gsim.boore_2014 import (
    _get_magnitude_scaling_term, _get_path_scaling, _get_site_scaling,
    _get_pga_on_rock)


def _get_C2_term(Cadj, rjb):
    """
    distance term, C2, from HA15
    """
    F2 = Cadj["C2"] * rjb
    return F2


def _get_C3_term(Cadj, rjb):
    """
    anelastic attenuation, C3, from HA15
    """
    F3 = Cadj["C3"] * np.maximum(0, np.log10(np.minimum(rjb, 150)/50))
    return F3


def _get_ZR19_distance_term(C_ZR19, rhypo):
    """
    Returns the distance scaling term
    """
    Rb_ = C_ZR19["Rb"]
    FR = np.zeros_like(rhypo)
    FR[(rhypo < Rb_) & (rhypo >= 4)] = (
        C_ZR19["a"] * np.log(rhypo[(rhypo < Rb_) & (rhypo >= 4)]/Rb_))
    FR[rhypo < 4] = C_ZR19["a"] * np.log(4/Rb_)
    return FR


def _get_ZR19_magnitude_term(C_ZR19, mag):
    """
    Returns the magnitude scaling term
    """
    Mb_ = C_ZR19["Mb"]
    FM = np.full_like(
        mag, C_ZR19["b0"] + C_ZR19["b1"] * (5.8 - Mb_) + C_ZR19["Cadj"])
    FM[mag < Mb_] = C_ZR19["b0"] + C_ZR19["Cadj"]
    between = (mag >= Mb_) & (mag < 5.8)
    FM[between] = (C_ZR19["b0"] + C_ZR19["b1"] * (mag[between] - Mb_) +
                   C_ZR19["Cadj"])
    return FM


def _get_ZR19_site_term(C_ZR19, ctx):
    """
    Returns the amplification factor, based on vs30
    """
    Vc_ = C_ZR19["Vc"]
    FS = np.zeros_like(ctx.vs30)
    FS[ctx.vs30 < Vc_] = C_ZR19["c"] * np.log(ctx.vs30[ctx.vs30 < Vc_]/Vc_)
    return FS


def _get_stddevs(C_ZR19):
    """
    Return standard deviations, in ln units [g]
    """
    # the units for PGV are not clearly stated in the paper,
    # assumed here cm/s
    return [C_ZR19["sigma"], C_ZR19["tau"], C_ZR19["phi"]]


def get_FENA(Cadj, ctx, imt):
    """
    See :meth:`superclass method
    <.base.GroundShakingIntensityModel.compute>`
    for spec of input and result values.
    """

    imean = (Cadj["C1"] + _get_C2_term(Cadj, ctx.rjb) +
             _get_C3_term(Cadj, ctx.rjb))

    # Convert from log10 to ln
    if imt.string.startswith(("SA", "PGA")):
        FENA = np.log(10.0 ** imean)
    else:
        FENA = np.log(10.0 ** imean)
    return FENA


class ZalachorisRathje2019(GMPE):
    """
    Implements the Induced Seismicity GMPE of Zalachoris & Rathje (2019)
    for Texas, Oklahoma and Kansas.
    Ground Motion Model for Small-to-Moderate Earthquakes
    in Texas, Oklahoma, and Kansas. Earthquake Spectra.
    Data: 4528 observations from 376 EQs
    of Mw 3.0-5.8 in Texas, Oklahoma and Kansas,
    distances between 4 and 500km. PGA, PGV, SA(0-10s)
    version: Apr 28, 2020. Verified with margin 1%
    """
    #: The GMPE is derived from induced earthquakes
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.INDUCED

    #: Supported intensity measure types are spectral acceleration,
    #: peak ground velocity and peak ground acceleration
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is orientation-independent
    #: measure :attr:`~openquake.hazardlib.const.IMC.RotD50`
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.RotD50

    #: Supported standard deviation types are inter-event, intra-event
    #: and total, see equation 2, pag 106.
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL, const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT}

    #: Required site parameters is Vs30
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: Required rupture parameters are magnitude, and rake.
    REQUIRES_RUPTURE_PARAMETERS = {'mag', 'rake'}

    #: Required distance measure is Rjb
    REQUIRES_DISTANCES = {'rjb', 'rhypo'}

    #: GMPE not tested against independent implementation so raise
    #: not verified warning
    non_verified = False

    region = "nobasin"
    kind = "base"
    sof = True

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.compute>`
        for spec of input and result values.
        """
        C_PGA = self.COEFFS_BSSA14[PGA()]
        pga_rock = _get_pga_on_rock(self.kind, self.region, self.sof,
                                    C_PGA, ctx)
        for m, imt in enumerate(imts):
            C = self.COEFFS_BSSA14[imt]
            Cadj = self.COEFFS_HA15[imt]
            C_ZR19 = self.COEFFS_ZR19[imt]
            imt_per = 0 if imt.string == 'PGV' else imt.period

            mean_BSSA14 = (
                _get_magnitude_scaling_term(self.sof, C, ctx) +
                _get_path_scaling(self.kind, self.region, C, ctx) +
                _get_site_scaling(self.kind, self.region, C, pga_rock, ctx,
                                  imt_per, ctx.rjb))

            FZR19 = (_get_ZR19_magnitude_term(C_ZR19, ctx.mag) +
                     _get_ZR19_distance_term(C_ZR19, ctx.rhypo) +
                     _get_ZR19_site_term(C_ZR19, ctx))

            # add HA15, all in ln units
            mean[m] = mean_BSSA14 + get_FENA(Cadj, ctx, imt)

            # add ZR19, all in ln units
            mean[m] += FZR19

            sig[m], tau[m], phi[m] = _get_stddevs(C_ZR19)

    #: terms for HA15
    #: terms for ZR19
    #: coeffs for BSSA14
    #: fewer decimals used for BSSA14 by Zalachoris compared to the original
    #: values
    COEFFS_BSSA14 = CoeffsTable(sa_damping=5, table="""\
    IMT		e0          e1          e2          e3 			e4          e5          e6          Mh          c1          c2          c3          h           Dc3         c           Vc              f4          f5          f6          f7          R1          R2          DfR         DfV         f1          f2          t1          t2
	pgv		5.0400000	5.0800000	4.8500000	5.0300000	1.0700000	-0.1540000	0.2250000	6.2000000	-1.2400000	0.1490000	-0.0034400	5.3000000	0.0000000	-0.8400000	1300.0000000	-0.1000000	-0.0084400	-9.9000000	-9.9000000	105.0000000	272.0000000	0.0820000	0.0800000	0.6440000	0.5520000	0.4010000	0.3460000
	pga		0.4470000	0.4860000	0.2460000	0.4540000	1.4300000	0.0505000	-0.1660000	5.5000000	-1.1300000	0.1920000	-0.0080900	4.5000000	0.0000000	-0.6000000	1500.0000000	-0.1500000	-0.0070100	-9.9000000	-9.9000000	110.0000000	270.0000000	0.1000000	0.0700000	0.6950000	0.4950000	0.3980000	0.3480000
	0.010	0.4530000	0.4920000	0.2520000	0.4600000	1.4200000	0.0493000	-0.1660000	5.5000000	-1.1300000	0.1920000	-0.0080900	4.5000000	0.0000000	-0.6040000	1500.2000000	-0.1480000	-0.0070100	-9.9000000	-9.9000000	111.6700000	270.0000000	0.0960000	0.0700000	0.6980000	0.4990000	0.4020000	0.3450000
	0.020	0.4860000	0.5240000	0.2970000	0.4890000	1.4300000	0.0534000	-0.1660000	5.5000000	-1.1400000	0.1900000	-0.0080700	4.5000000	0.0000000	-0.5740000	1500.3600000	-0.1470000	-0.0072800	-9.9000000	-9.9000000	113.1000000	270.0000000	0.0920000	0.0300000	0.7020000	0.5020000	0.4090000	0.3460000
	0.022	0.4990000	0.5360000	0.3130000	0.5000000	1.4300000	0.0549000	-0.1650000	5.5000000	-1.1400000	0.1890000	-0.0081000	4.5000000	0.0000000	-0.5670000	1500.6800000	-0.1480000	-0.0073200	-9.9000000	-9.9000000	113.3700000	270.0000000	0.0880000	0.0270000	0.7070000	0.5050000	0.4180000	0.3490000
	0.025	0.5230000	0.5610000	0.3440000	0.5200000	1.4300000	0.0575000	-0.1650000	5.5000000	-1.1400000	0.1890000	-0.0081500	4.5000000	0.0000000	-0.5550000	1501.0400000	-0.1500000	-0.0073600	-9.9000000	-9.9000000	113.0700000	270.0000000	0.0860000	0.0260000	0.7110000	0.5080000	0.4270000	0.3540000
	0.029	0.5590000	0.5990000	0.3910000	0.5500000	1.4300000	0.0607000	-0.1660000	5.5000000	-1.1400000	0.1880000	-0.0082900	4.5000000	0.0000000	-0.5390000	1501.2600000	-0.1540000	-0.0073700	-9.9000000	-9.9000000	112.3600000	270.0000000	0.0840000	0.0280000	0.7160000	0.5100000	0.4360000	0.3590000
	0.030	0.5690000	0.6090000	0.4040000	0.5580000	1.4300000	0.0614000	-0.1670000	5.5000000	-1.1400000	0.1880000	-0.0083400	4.4900000	0.0000000	-0.5340000	1502.9500000	-0.1550000	-0.0073500	-9.9000000	-9.9000000	112.1300000	270.0000000	0.0810000	0.0290000	0.7210000	0.5140000	0.4450000	0.3640000
	0.032	0.5880000	0.6290000	0.4280000	0.5730000	1.4200000	0.0628000	-0.1680000	5.5000000	-1.1400000	0.1880000	-0.0084500	4.4500000	0.0000000	-0.5250000	1503.1200000	-0.1570000	-0.0073100	-9.9000000	-9.9000000	111.6500000	270.0000000	0.0780000	0.0300000	0.7260000	0.5160000	0.4540000	0.3690000
	0.035	0.6160000	0.6580000	0.4630000	0.5970000	1.4200000	0.0646000	-0.1700000	5.5000000	-1.1400000	0.1880000	-0.0086400	4.4000000	0.0000000	-0.5120000	1503.2400000	-0.1600000	-0.0072100	-9.9000000	-9.9000000	110.6400000	270.0000000	0.0770000	0.0310000	0.7300000	0.5180000	0.4620000	0.3740000
	0.036	0.6260000	0.6680000	0.4730000	0.6050000	1.4200000	0.0650000	-0.1710000	5.5000000	-1.1400000	0.1880000	-0.0087200	4.3800000	0.0000000	-0.5080000	1503.3200000	-0.1610000	-0.0071700	-9.9000000	-9.9000000	109.5300000	270.0000000	0.0750000	0.0310000	0.7340000	0.5200000	0.4700000	0.3790000
	0.040	0.6630000	0.7060000	0.5150000	0.6380000	1.4100000	0.0662000	-0.1740000	5.5000000	-1.1300000	0.1880000	-0.0090300	4.3200000	0.0000000	-0.4910000	1503.3500000	-0.1680000	-0.0069800	-9.9000000	-9.9000000	108.2800000	270.0000000	0.0730000	0.0320000	0.7380000	0.5210000	0.4780000	0.3840000
	0.042	0.6810000	0.7240000	0.5340000	0.6550000	1.4100000	0.0664000	-0.1750000	5.5000000	-1.1300000	0.1880000	-0.0092000	4.2900000	0.0000000	-0.4830000	1503.3400000	-0.1720000	-0.0068700	-9.9000000	-9.9000000	106.9900000	270.0000000	0.0720000	0.0320000	0.7420000	0.5230000	0.4840000	0.3900000
	0.044	0.6990000	0.7430000	0.5530000	0.6720000	1.4000000	0.0667000	-0.1760000	5.5000000	-1.1300000	0.1880000	-0.0093600	4.2700000	0.0000000	-0.4760000	1503.1300000	-0.1770000	-0.0067700	-9.9000000	-9.9000000	105.4100000	270.0000000	0.0700000	0.0310000	0.7450000	0.5250000	0.4900000	0.3970000
	0.045	0.7080000	0.7520000	0.5620000	0.6810000	1.4000000	0.0668000	-0.1770000	5.5000000	-1.1200000	0.1880000	-0.0094400	4.2500000	0.0000000	-0.4720000	1502.8400000	-0.1790000	-0.0067200	-9.9000000	-9.9000000	103.6100000	270.0000000	0.0690000	0.0310000	0.7480000	0.5270000	0.4960000	0.4050000
	0.046	0.7180000	0.7620000	0.5720000	0.6910000	1.4000000	0.0669000	-0.1780000	5.5000000	-1.1200000	0.1880000	-0.0095200	4.2400000	0.0000000	-0.4690000	1502.4700000	-0.1820000	-0.0066700	-9.9000000	-9.9000000	101.7000000	270.0000000	0.0670000	0.0310000	0.7500000	0.5290000	0.4990000	0.4120000
	0.048	0.7360000	0.7800000	0.5890000	0.7090000	1.4000000	0.0671000	-0.1790000	5.5000000	-1.1200000	0.1870000	-0.0096800	4.2200000	0.0000000	-0.4630000	1502.0100000	-0.1870000	-0.0065600	-9.9000000	-9.9000000	99.7600000	270.0000000	0.0650000	0.0310000	0.7520000	0.5300000	0.5020000	0.4190000
	0.050	0.7540000	0.7990000	0.6070000	0.7270000	1.4000000	0.0674000	-0.1810000	5.5000000	-1.1200000	0.1870000	-0.0098200	4.2000000	0.0000000	-0.4580000	1501.4200000	-0.1920000	-0.0064700	-9.9000000	-9.9000000	97.9300000	270.0000000	0.0630000	0.0300000	0.7530000	0.5320000	0.5030000	0.4260000
	0.055	0.8000000	0.8450000	0.6480000	0.7740000	1.3900000	0.0678000	-0.1850000	5.5000000	-1.1100000	0.1870000	-0.0101000	4.1500000	0.0000000	-0.4480000	1500.7100000	-0.2040000	-0.0062500	-9.9000000	-9.9000000	96.0300000	270.0000000	0.0620000	0.0290000	0.7530000	0.5340000	0.5020000	0.4340000	
	0.060	0.8440000	0.8890000	0.6860000	0.8210000	1.4000000	0.0686000	-0.1890000	5.5000000	-1.1000000	0.1860000	-0.0103000	4.1100000	0.0000000	-0.4420000	1499.8300000	-0.2140000	-0.0060700	-9.9000000	-9.9000000	94.1000000	270.0100000	0.0610000	0.0270000	0.7530000	0.5360000	0.4990000	0.4410000
	0.065	0.8870000	0.9310000	0.7190000	0.8670000	1.4000000	0.0701000	-0.1920000	5.5000000	-1.0900000	0.1850000	-0.0105000	4.0800000	0.0000000	-0.4400000	1498.7400000	-0.2220000	-0.0059300	-9.9000000	-9.9000000	92.0800000	270.0200000	0.0610000	0.0250000	0.7520000	0.5380000	0.4950000	0.4480000
	0.067	0.9030000	0.9470000	0.7320000	0.8850000	1.4000000	0.0709000	-0.1930000	5.5000000	-1.0900000	0.1840000	-0.0105000	4.0700000	0.0000000	-0.4400000	1497.4200000	-0.2250000	-0.0058800	-9.9000000	-9.9000000	90.0100000	270.0200000	0.0610000	0.0250000	0.7500000	0.5400000	0.4890000	0.4550000
	0.070	0.9270000	0.9710000	0.7490000	0.9120000	1.4100000	0.0721000	-0.1950000	5.5000000	-1.0900000	0.1840000	-0.0106000	4.0600000	0.0000000	-0.4400000	1495.8500000	-0.2290000	-0.0058200	-9.9000000	-9.9000000	87.9700000	270.0300000	0.0620000	0.0240000	0.7480000	0.5410000	0.4830000	0.4610000
	0.075	0.9640000	1.0100000	0.7770000	0.9560000	1.4200000	0.0735000	-0.1970000	5.5000000	-1.0800000	0.1820000	-0.0106000	4.0400000	0.0000000	-0.4440000	1494.0000000	-0.2350000	-0.0057300	-9.9000000	-9.9000000	85.9900000	270.0400000	0.0640000	0.0220000	0.7450000	0.5420000	0.4740000	0.4660000
	0.080	1.0000000	1.0400000	0.8020000	0.9980000	1.4300000	0.0737000	-0.1980000	5.5000000	-1.0800000	0.1810000	-0.0106000	4.0200000	0.0000000	-0.4500000	1491.8200000	-0.2390000	-0.0056700	-9.9000000	-9.9000000	84.2300000	270.0500000	0.0670000	0.0200000	0.7410000	0.5430000	0.4640000	0.4680000
	0.085	1.0300000	1.0800000	0.8240000	1.0400000	1.4300000	0.0719000	-0.1990000	5.5100000	-1.0700000	0.1790000	-0.0105000	4.0300000	0.0000000	-0.4580000	1489.2900000	-0.2430000	-0.0056300	-9.9000000	-9.9000000	82.7400000	270.0600000	0.0720000	0.0190000	0.7370000	0.5430000	0.4520000	0.4680000
	0.090	1.0700000	1.1100000	0.8460000	1.0800000	1.4400000	0.0681000	-0.1990000	5.5200000	-1.0700000	0.1760000	-0.0104000	4.0700000	0.0000000	-0.4670000	1486.3600000	-0.2450000	-0.0056100	-9.9000000	-9.9000000	81.5400000	270.0700000	0.0760000	0.0170000	0.7340000	0.5420000	0.4400000	0.4660000
	0.095	1.1000000	1.1400000	0.8670000	1.1100000	1.4300000	0.0623000	-0.1990000	5.5300000	-1.0700000	0.1740000	-0.0103000	4.1000000	0.0000000	-0.4770000	1482.9800000	-0.2470000	-0.0056000	-9.9000000	-9.9000000	80.4600000	270.0800000	0.0820000	0.0160000	0.7310000	0.5420000	0.4280000	0.4640000
	0.100	1.1300000	1.1700000	0.8870000	1.1500000	1.4300000	0.0552000	-0.1980000	5.5400000	-1.0700000	0.1720000	-0.0102000	4.1300000	0.0000000	-0.4870000	1479.1200000	-0.2490000	-0.0056000	-9.9000000	-9.9000000	79.5900000	270.0900000	0.0870000	0.0140000	0.7280000	0.5410000	0.4150000	0.4580000
	0.110	1.1800000	1.2200000	0.9270000	1.2000000	1.4100000	0.0374000	-0.1960000	5.5700000	-1.0600000	0.1680000	-0.0099600	4.1900000	0.0000000	-0.5060000	1474.7400000	-0.2520000	-0.0056200	-9.9000000	-9.9000000	79.0500000	270.1100000	0.0930000	0.0120000	0.7260000	0.5400000	0.4030000	0.4510000
	0.120	1.2200000	1.2600000	0.9660000	1.2500000	1.3800000	0.0164000	-0.1930000	5.6200000	-1.0600000	0.1640000	-0.0097200	4.2400000	0.0000000	-0.5240000	1469.7500000	-0.2550000	-0.0056700	-9.9000000	-9.9000000	78.8500000	270.1300000	0.0990000	0.0110000	0.7240000	0.5390000	0.3920000	0.4410000
	0.130	1.2600000	1.3000000	1.0000000	1.2900000	1.3500000	-0.0051600	-0.1890000	5.6600000	-1.0500000	0.1600000	-0.0094800	4.2900000	0.0000000	-0.5420000	1464.0900000	-0.2560000	-0.0057200	-9.9000000	-9.9000000	78.9900000	270.1500000	0.1040000	0.0110000	0.7230000	0.5380000	0.3810000	0.4300000
	0.133	1.2700000	1.3100000	1.0100000	1.3000000	1.3400000	-0.0114000	-0.1880000	5.6700000	-1.0500000	0.1590000	-0.0094000	4.3000000	0.0000000	-0.5480000	1457.7600000	-0.2570000	-0.0057400	-9.9000000	-9.9000000	79.4700000	270.1500000	0.1100000	0.0110000	0.7220000	0.5380000	0.3710000	0.4170000
	0.140	1.2900000	1.3300000	1.0400000	1.3100000	1.3200000	-0.0247000	-0.1860000	5.7000000	-1.0500000	0.1570000	-0.0092300	4.3400000	0.0000000	-0.5600000	1450.7100000	-0.2570000	-0.0057800	-9.9000000	-9.9000000	80.2600000	270.1600000	0.1150000	0.0120000	0.7210000	0.5370000	0.3620000	0.4030000
	0.150	1.3100000	1.3500000	1.0600000	1.3300000	1.2800000	-0.0421000	-0.1820000	5.7400000	-1.0500000	0.1540000	-0.0089800	4.3900000	0.0000000	-0.5800000	1442.8500000	-0.2570000	-0.0058500	-9.9000000	-9.9000000	81.3300000	270.1600000	0.1200000	0.0150000	0.7200000	0.5370000	0.3540000	0.3880000
	0.160	1.3200000	1.3600000	1.0900000	1.3400000	1.2500000	-0.0576000	-0.1790000	5.7800000	-1.0500000	0.1520000	-0.0087300	4.4400000	0.0000000	-0.6010000	1434.2200000	-0.2560000	-0.0059100	-9.9000000	-9.9000000	82.8600000	270.1600000	0.1250000	0.0200000	0.7200000	0.5360000	0.3490000	0.3720000
	0.170	1.3300000	1.3700000	1.1000000	1.3500000	1.2200000	-0.0719000	-0.1740000	5.8200000	-1.0500000	0.1490000	-0.0084700	4.4900000	0.0000000	-0.6230000	1424.8500000	-0.2540000	-0.0059700	-9.9000000	-9.9000000	84.7200000	270.1400000	0.1280000	0.0260000	0.7180000	0.5360000	0.3460000	0.3570000
	0.180	1.3300000	1.3700000	1.1100000	1.3500000	1.1900000	-0.0856000	-0.1690000	5.8500000	-1.0600000	0.1480000	-0.0082200	4.5300000	0.0000000	-0.6450000	1414.7700000	-0.2520000	-0.0060200	-9.9000000	-9.9000000	86.6700000	270.1100000	0.1310000	0.0330000	0.7170000	0.5360000	0.3440000	0.3410000
	0.190	1.3300000	1.3700000	1.1200000	1.3500000	1.1600000	-0.0989000	-0.1640000	5.8900000	-1.0600000	0.1460000	-0.0079700	4.5700000	0.0000000	-0.6670000	1403.9900000	-0.2490000	-0.0060800	-9.9000000	-9.9000000	88.7300000	270.0600000	0.1340000	0.0390000	0.7140000	0.5370000	0.3430000	0.3240000
	0.200	1.3300000	1.3600000	1.1200000	1.3400000	1.1300000	-0.1110000	-0.1590000	5.9200000	-1.0600000	0.1450000	-0.0077200	4.6100000	0.0000000	-0.6880000	1392.6100000	-0.2470000	-0.0061400	-9.9000000	-9.9000000	90.9100000	270.0000000	0.1360000	0.0450000	0.7110000	0.5390000	0.3440000	0.3090000
	0.220	1.3100000	1.3400000	1.1100000	1.3300000	1.0800000	-0.1330000	-0.1470000	5.9700000	-1.0700000	0.1430000	-0.0072200	4.6800000	0.0000000	-0.7240000	1380.7200000	-0.2420000	-0.0062600	-9.9000000	-9.9000000	93.0400000	269.8300000	0.1380000	0.0520000	0.7080000	0.5410000	0.3450000	0.2940000
	0.240	1.2900000	1.3200000	1.0900000	1.3100000	1.0400000	-0.1530000	-0.1340000	6.0300000	-1.0700000	0.1400000	-0.0067500	4.7500000	0.0000000	-0.7560000	1368.5100000	-0.2380000	-0.0063800	-9.9000000	-9.9000000	95.0800000	269.5900000	0.1400000	0.0550000	0.7030000	0.5440000	0.3470000	0.2800000
	0.250	1.2800000	1.3000000	1.0800000	1.3100000	1.0200000	-0.1620000	-0.1280000	6.0500000	-1.0800000	0.1390000	-0.0065200	4.7800000	0.0000000	-0.7720000	1356.2100000	-0.2360000	-0.0064400	-9.9000000	-9.9000000	97.0400000	269.4500000	0.1410000	0.0550000	0.6980000	0.5470000	0.3500000	0.2660000
	0.260	1.2700000	1.2900000	1.0700000	1.3000000	0.9990000	-0.1700000	-0.1210000	6.0700000	-1.0800000	0.1380000	-0.0062900	4.8200000	0.0000000	-0.7870000	1343.8900000	-0.2330000	-0.0065000	-9.9000000	-9.9000000	98.8700000	269.3000000	0.1410000	0.0550000	0.6930000	0.5500000	0.3530000	0.2550000
	0.280	1.2400000	1.2600000	1.0500000	1.2800000	0.9730000	-0.1850000	-0.1070000	6.1100000	-1.0900000	0.1360000	-0.0058700	4.8800000	0.0000000	-0.8160000	1331.6700000	-0.2260000	-0.0066000	-9.9000000	-9.9000000	100.5300000	268.9600000	0.1400000	0.0530000	0.6870000	0.5540000	0.3570000	0.2440000
	0.290	1.2300000	1.2500000	1.0400000	1.2700000	0.9630000	-0.1910000	-0.1000000	6.1200000	-1.0900000	0.1350000	-0.0056700	4.9000000	0.0000000	-0.8300000	1319.8300000	-0.2230000	-0.0066500	-9.9000000	-9.9000000	102.0100000	268.7800000	0.1390000	0.0510000	0.6810000	0.5570000	0.3600000	0.2360000
	0.300	1.2200000	1.2400000	1.0200000	1.2700000	0.9570000	-0.1960000	-0.0929000	6.1400000	-1.0900000	0.1340000	-0.0054800	4.9300000	0.0000000	-0.8420000	1308.4700000	-0.2190000	-0.0067000	-9.9000000	-9.9000000	103.1500000	268.5900000	0.1380000	0.0500000	0.6750000	0.5610000	0.3630000	0.2290000
	0.320	1.2000000	1.2200000	1.0000000	1.2500000	0.9500000	-0.2050000	-0.0789000	6.1600000	-1.1000000	0.1320000	-0.0051200	4.9800000	0.0000000	-0.8620000	1297.6500000	-0.2130000	-0.0068000	-9.9000000	-9.9000000	104.0000000	268.2000000	0.1350000	0.0480000	0.6700000	0.5660000	0.3660000	0.2230000
	0.340	1.1800000	1.2000000	0.9770000	1.2300000	0.9500000	-0.2110000	-0.0651000	6.1800000	-1.1100000	0.1300000	-0.0048100	5.0300000	0.0000000	-0.8770000	1287.5000000	-0.2080000	-0.0068900	-9.9000000	-9.9000000	104.7000000	267.7900000	0.1330000	0.0470000	0.6640000	0.5700000	0.3690000	0.2180000
	0.350	1.1700000	1.1800000	0.9640000	1.2200000	0.9510000	-0.2140000	-0.0579000	6.1800000	-1.1100000	0.1290000	-0.0046600	5.0600000	0.0000000	-0.8840000	1278.0600000	-0.2060000	-0.0069300	-9.9000000	-9.9000000	105.2600000	267.5800000	0.1300000	0.0470000	0.6580000	0.5730000	0.3720000	0.2150000
	0.360	1.1600000	1.1700000	0.9510000	1.2100000	0.9530000	-0.2170000	-0.0510000	6.1900000	-1.1100000	0.1280000	-0.0045300	5.0800000	0.0000000	-0.8900000	1269.1900000	-0.2040000	-0.0069700	-9.9000000	-9.9000000	105.6100000	267.3700000	0.1280000	0.0470000	0.6530000	0.5760000	0.3750000	0.2120000
	0.380	1.1300000	1.1500000	0.9240000	1.1800000	0.9590000	-0.2220000	-0.0368000	6.1900000	-1.1200000	0.1260000	-0.0042800	5.1200000	0.0000000	-0.9000000	1260.7400000	-0.2000000	-0.0070500	-9.9000000	-9.9000000	105.8700000	266.9500000	0.1250000	0.0480000	0.6480000	0.5780000	0.3780000	0.2100000
	0.400	1.1000000	1.1200000	0.8980000	1.1600000	0.9680000	-0.2260000	-0.0232000	6.2000000	-1.1200000	0.1250000	-0.0040500	5.1600000	0.0000000	-0.9110000	1252.6600000	-0.1960000	-0.0071300	-9.9000000	-9.9000000	106.0200000	266.5400000	0.1220000	0.0490000	0.6430000	0.5800000	0.3810000	0.2100000
	0.420	1.0800000	1.1000000	0.8710000	1.1300000	0.9790000	-0.2290000	-0.0104000	6.2000000	-1.1300000	0.1240000	-0.0038500	5.2000000	0.0000000	-0.9220000	1244.8000000	-0.1920000	-0.0071900	-9.9000000	-9.9000000	106.0300000	266.1600000	0.1200000	0.0510000	0.6380000	0.5830000	0.3840000	0.2100000
	0.440	1.0500000	1.0700000	0.8440000	1.1000000	0.9910000	-0.2320000	0.0011700	6.2000000	-1.1300000	0.1230000	-0.0036700	5.2400000	0.0000000	-0.9350000	1237.0300000	-0.1870000	-0.0072600	-9.9000000	-9.9000000	105.9200000	265.8000000	0.1170000	0.0530000	0.6340000	0.5850000	0.3880000	0.2110000
	0.450	1.0400000	1.0600000	0.8290000	1.0800000	0.9990000	-0.2330000	0.0065900	6.2000000	-1.1400000	0.1220000	-0.0035900	5.2500000	0.0000000	-0.9410000	1229.2300000	-0.1850000	-0.0072900	-9.9000000	-9.9000000	105.7900000	265.6400000	0.1150000	0.0540000	0.6290000	0.5890000	0.3930000	0.2130000
	0.460	1.0200000	1.0400000	0.8150000	1.0700000	1.0100000	-0.2340000	0.0119000	6.2000000	-1.1400000	0.1220000	-0.0035100	5.2700000	0.0000000	-0.9470000	1221.1600000	-0.1830000	-0.0073200	-9.9000000	-9.9000000	105.6900000	265.4800000	0.1130000	0.0550000	0.6240000	0.5920000	0.3980000	0.2160000
	0.480	0.9970000	1.0200000	0.7890000	1.0400000	1.0200000	-0.2350000	0.0208000	6.2000000	-1.1400000	0.1210000	-0.0033600	5.3000000	0.0000000	-0.9590000	1212.7400000	-0.1790000	-0.0073800	-9.9000000	-9.9000000	105.5900000	265.2100000	0.1110000	0.0570000	0.6190000	0.5950000	0.4040000	0.2190000
	0.500	0.9700000	0.9910000	0.7620000	1.0100000	1.0400000	-0.2350000	0.0291000	6.2000000	-1.1500000	0.1200000	-0.0032200	5.3400000	0.0000000	-0.9690000	1203.9100000	-0.1750000	-0.0074400	-9.9000000	-9.9000000	105.5400000	265.0000000	0.1090000	0.0600000	0.6150000	0.5990000	0.4100000	0.2240000
	0.550	0.9050000	0.9280000	0.6980000	0.9420000	1.0800000	-0.2340000	0.0469000	6.2000000	-1.1500000	0.1180000	-0.0029000	5.4100000	0.0000000	-0.9890000	1194.5900000	-0.1660000	-0.0075800	-9.9000000	-9.9000000	105.6100000	264.7400000	0.1080000	0.0660000	0.6100000	0.6030000	0.4170000	0.2290000
	0.600	0.8420000	0.8670000	0.6390000	0.8740000	1.1300000	-0.2310000	0.0627000	6.2000000	-1.1600000	0.1170000	-0.0026100	5.4800000	0.0000000	-1.0000000	1184.9300000	-0.1580000	-0.0077300	-9.9000000	-9.9000000	105.8300000	264.8300000	0.1060000	0.0710000	0.6050000	0.6070000	0.4240000	0.2350000
	0.650	0.7820000	0.8090000	0.5820000	0.8090000	1.1900000	-0.2270000	0.0780000	6.2000000	-1.1700000	0.1150000	-0.0023600	5.5300000	0.0000000	-1.0100000	1175.1900000	-0.1510000	-0.0078700	0.0058300	0.0037600	106.2000000	265.2000000	0.1050000	0.0730000	0.5990000	0.6110000	0.4310000	0.2430000
	0.667	0.7630000	0.7900000	0.5640000	0.7890000	1.2000000	-0.2250000	0.0831000	6.2000000	-1.1700000	0.1140000	-0.0022800	5.5400000	0.0000000	-1.0100000	1165.6900000	-0.1490000	-0.0079200	0.0264000	0.0171000	106.7500000	265.3800000	0.1030000	0.0740000	0.5930000	0.6150000	0.4400000	0.2500000
	0.700	0.7250000	0.7530000	0.5290000	0.7500000	1.2400000	-0.2210000	0.0932000	6.2000000	-1.1700000	0.1130000	-0.0021300	5.5600000	0.0000000	-1.0100000	1156.4600000	-0.1450000	-0.0080000	0.0555000	0.0357000	107.4800000	265.7800000	0.1020000	0.0730000	0.5870000	0.6190000	0.4480000	0.2580000
	0.750	0.6690000	0.6970000	0.4750000	0.6920000	1.2900000	-0.2160000	0.1080000	6.2000000	-1.1800000	0.1110000	-0.0019300	5.6000000	0.0000000	-1.0200000	1147.5900000	-0.1390000	-0.0081200	0.0923000	0.0590000	108.3900000	266.5100000	0.1000000	0.0700000	0.5810000	0.6220000	0.4570000	0.2660000
	0.800	0.6130000	0.6420000	0.4220000	0.6350000	1.3300000	-0.2100000	0.1230000	6.2000000	-1.1800000	0.1090000	-0.0017500	5.6300000	0.0000000	-1.0200000	1139.2100000	-0.1320000	-0.0082200	0.1400000	0.0885000	109.6200000	267.3200000	0.0990000	0.0630000	0.5760000	0.6240000	0.4660000	0.2740000
	0.850	0.5590000	0.5870000	0.3680000	0.5800000	1.3800000	-0.2050000	0.1360000	6.2000000	-1.1900000	0.1070000	-0.0016000	5.6600000	0.0000000	-1.0300000	1131.3400000	-0.1250000	-0.0083000	0.1950000	0.1200000	111.0800000	268.1400000	0.0990000	0.0530000	0.5700000	0.6250000	0.4750000	0.2810000
	0.900	0.5030000	0.5310000	0.3140000	0.5240000	1.4200000	-0.2000000	0.1500000	6.2000000	-1.1900000	0.1050000	-0.0014600	5.6900000	0.0000000	-1.0400000	1123.9100000	-0.1180000	-0.0083600	0.2520000	0.1520000	112.7100000	268.9000000	0.0980000	0.0420000	0.5640000	0.6260000	0.4830000	0.2880000
	0.950	0.4470000	0.4750000	0.2590000	0.4670000	1.4600000	-0.1950000	0.1640000	6.2000000	-1.1900000	0.1040000	-0.0013300	5.7200000	0.0000000	-1.0400000	1116.8300000	-0.1120000	-0.0084100	0.3090000	0.1810000	114.5000000	269.5500000	0.0980000	0.0300000	0.5580000	0.6260000	0.4910000	0.2940000
	1.000	0.3930000	0.4220000	0.2070000	0.4120000	1.5000000	-0.1900000	0.1790000	6.2000000	-1.1900000	0.1020000	-0.0012100	5.7400000	0.0000000	-1.0500000	1109.9500000	-0.1050000	-0.0084400	0.3670000	0.2080000	116.3900000	270.0000000	0.0980000	0.0200000	0.5530000	0.6250000	0.4980000	0.2980000
	1.100	0.2850000	0.3140000	0.1020000	0.3020000	1.5700000	-0.1800000	0.2100000	6.2000000	-1.2000000	0.1000000	-0.0009940	5.8200000	0.0000000	-1.0600000	1103.0700000	-0.0938000	-0.0084700	0.4250000	0.2330000	118.3000000	270.1800000	0.0990000	0.0070000	0.5480000	0.6240000	0.5050000	0.3020000
	1.200	0.1730000	0.2030000	-0.0062000	0.1890000	1.6300000	-0.1710000	0.2440000	6.2000000	-1.2000000	0.0985000	-0.0008030	5.9200000	0.0000000	-1.0600000	1096.0400000	-0.0841000	-0.0084200	0.4810000	0.2560000	120.1900000	269.4200000	0.1000000	0.0020000	0.5430000	0.6230000	0.5110000	0.3060000
	1.300	0.0615000	0.0911000	-0.1130000	0.0743000	1.6800000	-0.1620000	0.2780000	6.2000000	-1.2000000	0.0974000	-0.0006350	6.0100000	0.0000000	-1.0600000	1088.6700000	-0.0758000	-0.0082900	0.5360000	0.2760000	122.0100000	267.8200000	0.1010000	0.0030000	0.5390000	0.6220000	0.5160000	0.3090000
	1.400	-0.0458000	-0.0157000	-0.2160000	-0.0361000	1.7200000	-0.1540000	0.3100000	6.2000000	-1.2000000	0.0967000	-0.0004900	6.1000000	0.0000000	-1.0500000	1080.7700000	-0.0685000	-0.0080600	0.5880000	0.2940000	123.7500000	265.4500000	0.1020000	0.0060000	0.5350000	0.6200000	0.5210000	0.3120000
	1.500	-0.1500000	-0.1190000	-0.3140000	-0.1440000	1.7600000	-0.1470000	0.3390000	6.2000000	-1.2100000	0.0964000	-0.0003650	6.1800000	0.0000000	-1.0500000	1072.3900000	-0.0620000	-0.0077100	0.6380000	0.3090000	125.3800000	262.4100000	0.1040000	0.0100000	0.5320000	0.6190000	0.5250000	0.3150000
	1.600	-0.2490000	-0.2170000	-0.4070000	-0.2470000	1.8000000	-0.1400000	0.3660000	6.2000000	-1.2100000	0.0963000	-0.0002590	6.2600000	0.0000000	-1.0400000	1061.7700000	-0.0559000	-0.0072300	0.6890000	0.3240000	126.9000000	258.7800000	0.1050000	0.0120000	0.5290000	0.6180000	0.5280000	0.3180000
	1.700	-0.3410000	-0.3080000	-0.4930000	-0.3450000	1.8300000	-0.1340000	0.3910000	6.2000000	-1.2100000	0.0963000	-0.0001710	6.3300000	0.0000000	-1.0400000	1049.2900000	-0.0503000	-0.0066600	0.7360000	0.3370000	128.1400000	254.6600000	0.1060000	0.0120000	0.5270000	0.6180000	0.5300000	0.3210000
	1.800	-0.4300000	-0.3960000	-0.5740000	-0.4380000	1.8600000	-0.1270000	0.4120000	6.2000000	-1.2100000	0.0962000	-0.0000990	6.4000000	0.0000000	-1.0400000	1036.4200000	-0.0451000	-0.0060300	0.7800000	0.3500000	129.1100000	250.1100000	0.1060000	0.0120000	0.5260000	0.6180000	0.5310000	0.3230000
	1.900	-0.5130000	-0.4770000	-0.6490000	-0.5270000	1.8900000	-0.1200000	0.4320000	6.2000000	-1.2100000	0.0963000	-0.0000420	6.4800000	0.0000000	-1.0400000	1023.1400000	-0.0404000	-0.0054000	0.8240000	0.3640000	129.8600000	245.2500000	0.1060000	0.0100000	0.5260000	0.6180000	0.5320000	0.3260000
	2.000	-0.5870000	-0.5500000	-0.7150000	-0.6070000	1.9200000	-0.1120000	0.4480000	6.2000000	-1.2200000	0.0964000	0.0000000	6.5400000	0.0000000	-1.0400000	1009.4900000	-0.0361000	-0.0047900	0.8710000	0.3820000	130.3700000	240.1400000	0.1050000	0.0080000	0.5260000	0.6180000	0.5320000	0.3290000
	2.200	-0.7210000	-0.6820000	-0.8300000	-0.7540000	1.9700000	-0.0980000	0.4800000	6.2000000	-1.2200000	0.0965000	0.0000000	6.6600000	0.0000000	-1.0400000	995.5200000		-0.0291000	-0.0037800	0.9200000	0.4040000	130.6700000	229.5500000	0.1030000	0.0050000	0.5270000	0.6190000	0.5330000	0.3320000
	2.400	-0.8480000	-0.8070000	-0.9330000	-0.8940000	2.0200000	-0.0838000	0.5190000	6.2000000	-1.2200000	0.0962000	0.0000000	6.7300000	0.0000000	-1.0300000	981.3300000		-0.0237000	-0.0030200	0.9690000	0.4270000	130.8100000	219.0500000	0.1000000	0.0030000	0.5280000	0.6190000	0.5330000	0.3350000
	2.500	-0.9100000	-0.8680000	-0.9820000	-0.9620000	2.0400000	-0.0763000	0.5390000	6.2000000	-1.2200000	0.0961000	0.0000000	6.7700000	0.0000000	-1.0300000	966.9400000		-0.0215000	-0.0027200	1.0200000	0.4510000	130.8100000	214.0400000	0.0970000	0.0020000	0.5300000	0.6190000	0.5340000	0.3370000
	2.600	-0.9690000	-0.9260000	-1.0300000	-1.0300000	2.0600000	-0.0689000	0.5580000	6.2000000	-1.2200000	0.0961000	0.0000000	6.8100000	0.0000000	-1.0300000	952.3400000		-0.0196000	-0.0024600	1.0600000	0.4740000	130.7200000	209.3200000	0.0940000	0.0010000	0.5310000	0.6200000	0.5350000	0.3400000
	2.800	-1.0800000	-1.0400000	-1.1300000	-1.1500000	2.1000000	-0.0552000	0.5940000	6.2000000	-1.2200000	0.0967000	0.0000000	6.8700000	0.0000000	-1.0200000	937.5200000		-0.0163000	-0.0020800	1.1000000	0.4950000	130.5700000	201.0800000	0.0910000	0.0000000	0.5320000	0.6190000	0.5360000	0.3420000
	3.000	-1.1900000	-1.1400000	-1.2300000	-1.2700000	2.1300000	-0.0433000	0.6270000	6.2000000	-1.2200000	0.0976000	0.0000000	6.9300000	0.0000000	-1.0100000	922.4300000		-0.0136000	-0.0018300	1.1300000	0.5160000	130.3600000	195.0000000	0.0880000	0.0000000	0.5340000	0.6190000	0.5370000	0.3440000
	3.200	-1.2900000	-1.2400000	-1.3300000	-1.3800000	2.1500000	-0.0344000	0.6580000	6.2000000	-1.2200000	0.0986000	-0.0000230	6.9900000	0.0000000	-1.0000000	908.7900000		-0.0110000	-0.0016700	1.1600000	0.5340000	130.1300000	191.6100000	0.0840000	0.0000000	0.5350000	0.6180000	0.5380000	0.3450000
	3.400	-1.3900000	-1.3300000	-1.4200000	-1.4800000	2.1700000	-0.0279000	0.6880000	6.2000000	-1.2200000	0.0996000	-0.0000400	7.0800000	0.0000000	-0.9950000	896.1500000		-0.0086700	-0.0015800	1.1900000	0.5510000	129.9000000	190.7300000	0.0810000	0.0000000	0.5350000	0.6180000	0.5400000	0.3460000
	3.500	-1.4300000	-1.3800000	-1.4600000	-1.5300000	2.1800000	-0.0250000	0.7020000	6.2000000	-1.2200000	0.1000000	-0.0000450	7.1200000	0.0000000	-0.9910000	883.1600000		-0.0075700	-0.0015500	1.2100000	0.5700000	129.7100000	191.1100000	0.0780000	0.0000000	0.5360000	0.6170000	0.5410000	0.3470000
	3.600	-1.4800000	-1.4200000	-1.5000000	-1.5800000	2.1800000	-0.0226000	0.7150000	6.2000000	-1.2200000	0.1000000	-0.0000490	7.1600000	0.0000000	-0.9870000	870.0500000		-0.0065400	-0.0015400	1.2300000	0.5890000	129.5600000	191.9800000	0.0750000	0.0000000	0.5360000	0.6160000	0.5420000	0.3480000
	3.800	-1.5600000	-1.5000000	-1.5900000	-1.6700000	2.1900000	-0.0184000	0.7400000	6.2000000	-1.2200000	0.1010000	-0.0000530	7.2400000	0.0000000	-0.9780000	857.0700000		-0.0047000	-0.0015200	1.2500000	0.6090000	129.4900000	195.0100000	0.0720000	0.0000000	0.5360000	0.6160000	0.5430000	0.3490000
	4.000	-1.6400000	-1.5700000	-1.6700000	-1.7500000	2.2000000	-0.0146000	0.7630000	6.2000000	-1.2200000	0.1020000	-0.0000520	7.3200000	0.0000000	-0.9690000	844.4800000		-0.0032100	-0.0015200	1.2700000	0.6290000	129.4900000	199.4500000	0.0700000	0.0000000	0.5360000	0.6160000	0.5430000	0.3490000
	4.200	-1.7100000	-1.6400000	-1.7500000	-1.8300000	2.2100000	-0.0122000	0.7860000	6.2000000	-1.2200000	0.1030000	-0.0000470	7.3900000	0.0000000	-0.9600000	832.4500000		-0.0021000	-0.0015200	1.2900000	0.6520000	129.5700000	204.9300000	0.0680000	0.0000000	0.5350000	0.6160000	0.5420000	0.3490000
	4.400	-1.7800000	-1.7100000	-1.8200000	-1.9000000	2.2200000	-0.0115000	0.8080000	6.2000000	-1.2200000	0.1030000	-0.0000390	7.4600000	0.0000000	-0.9500000	821.1800000		-0.0013200	-0.0015000	1.3000000	0.6740000	129.7100000	211.0900000	0.0660000	0.0000000	0.5340000	0.6170000	0.5400000	0.3470000
	4.600	-1.8500000	-1.7700000	-1.8900000	-1.9700000	2.2200000	-0.0118000	0.8310000	6.2000000	-1.2200000	0.1030000	-0.0000270	7.5200000	0.0000000	-0.9410000	810.7900000		-0.0008040	-0.0014800	1.3100000	0.6970000	129.8700000	217.5600000	0.0640000	0.0000000	0.5330000	0.6190000	0.5380000	0.3450000
	4.800	-1.9100000	-1.8300000	-1.9600000	-2.0300000	2.2300000	-0.0129000	0.8520000	6.2000000	-1.2200000	0.1030000	-0.0000140	7.6400000	0.0000000	-0.9300000	801.4100000		-0.0004710	-0.0014600	1.3200000	0.7190000	130.0500000	223.9900000	0.0630000	0.0000000	0.5310000	0.6210000	0.5350000	0.3410000
	5.000	-1.9700000	-1.8900000	-2.0200000	-2.0900000	2.2300000	-0.0149000	0.8730000	6.2000000	-1.2200000	0.1040000	0.0000000	7.7800000	0.0000000	-0.9200000	793.1300000		-0.0002550	-0.0014400	1.3300000	0.7380000	130.2200000	230.0000000	0.0610000	0.0000000	0.5280000	0.6220000	0.5320000	0.3350000
	5.500	-2.1100000	-2.0200000	-2.1900000	-2.2300000	2.2400000	-0.0195000	0.9150000	6.2000000	-1.2200000	0.1050000	0.0000000	8.0700000	0.0000000	-0.8920000	785.7300000		0.0000724	-0.0014000	1.3500000	0.7780000	130.3900000	241.8600000	0.0600000	0.0000000	0.5260000	0.6240000	0.5280000	0.3290000
	6.000	-2.2400000	-2.1600000	-2.3700000	-2.3600000	2.2400000	-0.0264000	0.9490000	6.2000000	-1.2200000	0.1080000	0.0000000	8.4800000	0.0000000	-0.8630000	779.9100000		0.0001880	-0.0013800	1.3500000	0.8030000	130.5300000	249.3400000	0.0590000	0.0000000	0.5240000	0.6250000	0.5240000	0.3210000
	6.500	-2.3700000	-2.2800000	-2.5300000	-2.4800000	2.2200000	-0.0395000	0.9760000	6.2000000	-1.2300000	0.1120000	0.0000000	8.9000000	0.0000000	-0.8340000	775.6000000		0.0001590	-0.0013700	1.3500000	0.8150000	130.6300000	252.9400000	0.0590000	0.0000000	0.5200000	0.6340000	0.5170000	0.3120000
	7.000	-2.4800000	-2.3900000	-2.6800000	-2.5900000	2.1700000	-0.0591000	0.9980000	6.2000000	-1.2400000	0.1190000	0.0000000	9.2000000	0.0000000	-0.8050000	772.6800000		0.0000559	-0.0013700	1.3400000	0.8160000	130.7000000	253.1200000	0.0590000	0.0000000	0.5150000	0.6360000	0.5140000	0.3020000
	7.500	-2.5900000	-2.4900000	-2.8200000	-2.6900000	2.1200000	-0.0816000	1.0100000	6.2000000	-1.2500000	0.1250000	0.0000000	9.4800000	0.0000000	-0.7770000	771.0100000		-0.0000546	-0.0013700	1.3300000	0.8090000	130.7200000	250.3900000	0.0580000	0.0000000	0.5120000	0.6340000	0.5110000	0.2700000
	8.000	-2.6900000	-2.5800000	-2.9400000	-2.7800000	2.0600000	-0.1040000	1.0200000	6.2000000	-1.2700000	0.1310000	0.0000000	9.5700000	0.0000000	-0.7500000	760.8100000		-0.0001170	-0.0013700	1.3100000	0.7950000	130.8700000	245.2300000	0.0590000	0.0000000	0.5100000	0.6300000	0.5070000	0.2780000
	8.500	-2.7800000	-2.6800000	-3.0600000	-2.8800000	2.0100000	-0.1210000	1.0300000	6.2000000	-1.2800000	0.1370000	0.0000000	9.6200000	0.0000000	-0.7250000	764.5000000		-0.0001310	-0.0013700	1.2800000	0.7770000	130.7100000	238.1300000	0.0590000	0.0000000	0.5090000	0.6220000	0.5030000	0.2650000
	9.000	-2.8800000	-2.7700000	-3.1700000	-2.9800000	1.9600000	-0.1340000	1.0500000	6.2000000	-1.3000000	0.1430000	0.0000000	9.6600000	0.0000000	-0.7020000	768.0700000		-0.0001080	-0.0013700	1.2500000	0.7540000	130.5000000	229.5600000	0.0600000	0.0000000	0.5090000	0.6130000	0.4980000	0.2520000
	9.500	-2.9800000	-2.8600000	-3.2800000	-3.0800000	1.9200000	-0.1440000	1.0600000	6.2000000	-1.3100000	0.1480000	0.0000000	9.6600000	0.0000000	-0.6790000	771.5500000		-0.0000600	-0.0013600	1.2200000	0.7290000	130.2600000	220.0200000	0.0600000	0.0000000	0.5090000	0.6040000	0.4920000	0.2390000
	10.000	-3.0700000	-2.9500000	-3.3800000	-3.1700000	1.8800000	-0.1510000	1.0700000	6.2000000	-1.3300000	0.1520000	0.0000000	9.6600000	0.0000000	-0.6560000	775.0000000		0.0000000	-0.0013600	1.1800000	0.7030000	130.0000000	210.0000000	0.0600000	0.0000000	0.5100000	0.6040000	0.4870000	0.2390000
    """)

    CONSTS = {
        "Mref": 4.5,
        "Rref": 1.0,
        "Vref": 760.0,
        "f1": 0.0,
        "f3": 0.1,
        "v1": 225.0,
        "v2": 300.0}

    #: coeffs for HA15
    COEFFS_HA15 = CoeffsTable(sa_damping=5, table="""
    IMT		C1		C2		C3		sigma	tau		phi
    pgv		0.166	0.0007	0.73	0.33	0.28	0.18
    pga		0.384	0.0017	0.63	0.41	0.30	0.27
    0.05	0.472	0.0022	0.44	0.47	0.36	0.32
    0.063	0.439	0.0023	0.52	0.45	0.33	0.30
    0.079	0.37	0.0024	0.61	0.42	0.32	0.28
    0.1		0.264	0.0023	0.71	0.40	0.31	0.25
    0.126	0.158	0.0022	0.81	0.38	0.30	0.22
    0.158	0.074	0.0021	0.88	0.36	0.30	0.20
    0.199	0.014	0.0020	0.89	0.35	0.30	0.18
    0.251	-0.016	0.0018	0.86	0.34	0.29	0.17
    0.316	-0.036	0.0016	0.80	0.33	0.29	0.15
    0.398	-0.053	0.0015	0.71	0.31	0.28	0.14
    0.5		-0.068	0.0014	0.63	0.30	0.27	0.14
    0.633	-0.077	0.0013	0.59	0.29	0.26	0.13
    0.794	-0.067	0.0011	0.57	0.28	0.25	0.12
    1.		-0.043	0.0009	0.54	0.28	0.25	0.13
    1.266	-0.009	0.0006	0.50	0.28	0.25	0.14
    1.587	0.02	0.0005	0.48	0.29	0.24	0.15
    2.		0.041	0.0004	0.44	0.28	0.24	0.15
    2.5		0.061	0.0004	0.36	0.28	0.23	0.16
    3.125	0.067	0.0005	0.27	0.28	0.23	0.16
    4.		0.052	0.0006	0.21	0.28	0.23	0.17
    5.		0.025	0.0006	0.22	0.29	0.22	0.18
    6.25	0.010	0.0006	0.27	0.29	0.22	0.20
    7.692	0.029	0.0006	0.27	0.28	0.21	0.19
    10.		0.065	0.0006	0.22	0.25	0.21	0.14
    """)

    # coeffs for ZR19 (more IMT values compared to the published paper)
    COEFFS_ZR19 = CoeffsTable(sa_damping=5, table="""
    IMT		a		Rb		Mb		b0		b1		c		Vc		Cadj	tau		phi		sigma
    pgv		-0.5339	13.94	2.925	0.2018	-0.2865	0.8682	515.96	0.1751	0.2716	0.513	0.5804
    pga		-0.6347	16.73	3.393	0.0908	-0.3217	0.4779	638.08	-0.2208	0.2959	0.5443	0.6195
    0.05	-0.7112	18.2	3.4		0.0854	-0.3108	0.1668	945.8	-0.2267	0.3147	0.5785	0.6586
    0.055	-0.6822	18.43	3.4		0.0867	-0.3146	0.2176	886.9	-0.2007	0.3156	0.5813	0.6614
    0.06	-0.6581	18.51	3.4		0.0876	-0.3167	0.2657	831.6	-0.1727	0.3146	0.5833	0.6627
    0.065	-0.6385	18.45	3.4		0.088	-0.3166	0.3088	783		-0.1414	0.3114	0.5844	0.6622
    0.07	-0.6164	18.45	3.4		0.0879	-0.3148	0.346	742.7	-0.1067	0.3072	0.5847	0.6605
    0.075	-0.5864	18.69	3.4		0.0872	-0.312	0.3777	710.8	-0.0697	0.3035	0.5843	0.6584
    0.08	-0.5478	19.21	3.4		0.086	-0.3088	0.4052	686.4	-0.0321	0.3011	0.5832	0.6564
    0.085	-0.507	19.81	3.4		0.0852	-0.3056	0.4306	667.5	0.0052	0.3		0.5819	0.6546
    0.09	-0.4712	20.28	3.39	0.086	-0.3029	0.4561	652		0.0423	0.2996	0.5809	0.6536
    0.095	-0.4467	20.42	3.37	0.0905	-0.3017	0.4838	637.8	0.0811	0.2995	0.5807	0.6534
    0.1		-0.4351	20.16	3.34	0.0986	-0.3019	0.5151	623.8	0.1222	0.2993	0.5817	0.6541
    0.133	-0.4354	19.48	3.3		0.1077	-0.3031	0.5509	609.3	0.1653	0.2986	0.5838	0.6557
    0.15	-0.4418	18.45	3.28	0.1135	-0.3032	0.5854	593.2	0.2079	0.2971	0.5869	0.6578
    0.2		-0.4515	17.13	3.27	0.1138	-0.3009	0.6158	563		0.248	0.2944	0.5897	0.6591
    0.25	-0.4627	15.69	3.29	0.1091	-0.2976	0.638	542.5	0.2818	0.2907	0.5894	0.6572
    0.3		-0.4789	14.27	3.36	0.0998	-0.3011	0.6878	527.8	0.3069	0.2865	0.5839	0.6504
    0.35	-0.5019	13.01	3.5		0.0848	-0.3225	0.7734	517		0.3249	0.2821	0.5729	0.6386
    0.4		-0.5353	11.92	3.73	0.0643	-0.3655	0.8714	510.8	0.3413	0.2773	0.5588	0.6239
    0.45	-0.581	10.97	4		0.0421	-0.4214	0.9505	510.9	0.3617	0.2714	0.5447	0.6085
    0.5		-0.6371	10.15	4.21	0.0244	-0.4728	0.9996	515		0.3861	0.264	0.5323	0.5942
    0.55	-0.6923	9.51	4.35	0.0142	-0.5134	1.0257	519.9	0.4107	0.2554	0.5224	0.5815
    0.6		-0.733	9.09	4.42	0.0104	-0.5482	1.0382	523.7	0.4307	0.2467	0.5148	0.5709
    0.65	-0.7523	8.87	4.47	0.0094	-0.5887	1.0429	526.3	0.4448	0.2389	0.5092	0.5625
    0.667	-0.7561	8.76	4.52	0.0091	-0.6355	1.0417	528.5	0.4542	0.2325	0.5051	0.556
    0.7		-0.7524	8.7		4.58	0.0087	-0.6825	1.0391	530.4	0.4616	0.2271	0.502	0.551
    0.75	-0.7452	8.64	4.62	0.0084	-0.7204	1.0398	531.8	0.4694	0.2226	0.4993	0.5466
    0.8		-0.7325	8.6		4.64	0.0083	-0.7468	1.0465	532.6	0.4785	0.2188	0.4963	0.5423
    0.85	-0.7131	8.58	4.66	0.0082	-0.7624	1.0576	533.2	0.4888	0.2161	0.4925	0.5379
    0.9		-0.687	8.62	4.67	0.008	-0.7724	1.0696	534		0.499	0.2153	0.4879	0.5333
    0.95	-0.6509	8.76	4.71	0.0076	-0.7704	1.081	534.9	0.5076	0.2176	0.4825	0.5293
    1		-0.603	9.02	4.69	0.0078	-0.7923	1.0909	536		0.5118	0.2242	0.4759	0.526
    1.5		-0.547	9.35	4.78	0.0059	-0.7244	1.0997	536.5	0.5093	0.2368	0.4677	0.5243
    2		-0.4981	9.59	4.89	0.0056	-0.5086	1.105	535.8	0.499	0.2548	0.459	0.525
    2.5		-0.4738	9.63	5.8		0		0		1.1049	533.5	0.4836	0.2754	0.4525	0.5297
    3		-0.4847	9.46	5.8		0		0		1.0961	530.3	0.4659	0.2926	0.4505	0.5372
    3.5		-0.5269	9.21	5.8		0		0		1.077	527.4	0.4487	0.3012	0.4535	0.5444
    4.0		-0.5844	9.06	5.8		0		0		1.0462	525.6	0.4316	0.2989	0.4589	0.5476
    4.4		-0.6387	9.14	5.8		0		0		1.0049	525		0.4151	0.2879	0.4634	0.5455
    5		-0.6766	9.45	5.8		0		0		0.9561	525.1	0.399	0.2726	0.4652	0.5392
    5.5		-0.6954	9.89	4.91	-0.0003	-0.0711	0.9036	525.4	0.3848	0.2573	0.4642	0.5308
    6		-0.6983	10.36	4.51	0.0042	-0.1059	0.8507	525.5	0.3732	0.2449	0.4614	0.5224
    6.5		-0.693	10.73	4.16	0.0151	-0.1418	0.7992	525.3	0.3653	0.2363	0.4582	0.5155
    7		-0.6839	10.95	3.86	0.0312	-0.1693	0.7499	525.1	0.3619	0.2311	0.4556	0.5108
    7.5		-0.676	11.03	3.63	0.0505	-0.1954	0.7043	524.8	0.3631	0.2284	0.4542	0.5084
    8		-0.6704	11.03	3.48	0.0684	-0.2226	0.6642	524.7	0.3681	0.227	0.4543	0.5078
    8.5		-0.6711	11.03	3.41	0.0846	-0.2569	0.6311	524.6	0.3755	0.226	0.4554	0.5084
    9		-0.6787	11.09	3.39	0.0982	-0.2937	0.6043	524.6	0.3831	0.2247	0.4572	0.5094
    9.5		-0.6941	11.22	3.39	0.1105	-0.3295	0.5816	524.6	0.3897	0.2227	0.4594	0.5106
    10		-0.7138	11.39	3.4		0.1218	-0.3627	0.5605	524.5	0.3951	0.2201	0.4618	0.5116
    """)
