"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(grantee);
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.streamName,
        });
        const shardCount = props.shardCount || 1;
        const retentionPeriodHours = (_b = (_a = props.retentionPeriod) === null || _a === void 0 ? void 0 : _a.toHours()) !== null && _b !== void 0 ? _b : 24;
        if (retentionPeriodHours < 24 || retentionPeriodHours > 168) {
            throw new Error(`retentionPeriod must be between 24 and 168 hours. Received ${retentionPeriodHours}`);
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * Import an existing Kinesis Stream provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo)
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).parseArn(attrs.streamArn).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = (_a = props.encryption) !== null && _a !== void 0 ? _a : (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption with a master key managed by Amazon Kinesis
     */
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
//# sourceMappingURL=data:application/json;base64,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