"""
Level 3: Audit Operation Models - Specific to Audit Operations

This module contains models that are specific to audit operations
within the Core API.

Three-Tier Architecture:
- Level 1: Global models shared across ALL APIs  
- Level 2: Core API models shared within Core API
- Level 3 (This file): Audit operation-specific models

Benefits:
- Perfect locality (audit models exactly where audit operations are)
- Clean imports (from .models import AuditResponse)
- Logical organization (operation-specific grouping)
"""

from typing import Optional, List, Dict, Any, Annotated
from datetime import datetime
from pydantic import BaseModel, Field, ConfigDict

# Import from Level 1 (global)
from ...models import BaseEntry, ContentInfo, UserInfo

# Import from Level 2 (Core API)
from ..models import CoreResponse


class AuditResponse(CoreResponse):
    """Response wrapper for audit operations."""
    entry: BaseEntry = Field(..., description="Audit data")


class AuditListResponse(BaseModel):
    """Response wrapper for audit list operations."""
    model_config = ConfigDict(extra='forbid')
    
    list: Dict[str, Any] = Field(..., description="List data with pagination")


# Operation-specific models will be added here based on analysis
# Example models for common patterns:


class CreateAuditRequest(BaseModel):
    """Request model for creating audit."""
    model_config = ConfigDict(extra='forbid')
    
    name: Annotated[str, Field(
        description="Audit name",
        min_length=1,
        max_length=255
    )]
    
    properties: Annotated[Optional[Dict[str, Any]], Field(
        description="Custom properties",
        default=None
    )]


class UpdateAuditRequest(BaseModel):
    """Request model for updating audit."""
    model_config = ConfigDict(extra='forbid')
    
    name: Annotated[Optional[str], Field(
        description="Updated name",
        min_length=1,
        max_length=255,
        default=None
    )]
    
    properties: Annotated[Optional[Dict[str, Any]], Field(
        description="Updated properties",
        default=None
    )]


# Export all models
__all__ = [
    'AuditResponse', 
    'AuditListResponse',
    'CreateAuditRequest',
    'UpdateAuditRequest'
]