from http import HTTPStatus
from typing import Any, Optional, Union, cast

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.rendition_entry import RenditionEntry
from ...types import Response


def _get_kwargs(
    shared_id: str,
    rendition_id: str,
) -> dict[str, Any]:
    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": f"/shared-links/{shared_id}/renditions/{rendition_id}",
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, RenditionEntry]]:
    if response.status_code == 200:
        response_200 = RenditionEntry.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if response.status_code == 501:
        response_501 = cast(Any, None)
        return response_501
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, RenditionEntry]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    shared_id: str,
    rendition_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Response[Union[Any, RenditionEntry]]:
    """Get shared link rendition information

     **Note:** this endpoint is available in Alfresco 5.2 and newer versions.

    Gets rendition information for the file with shared link identifier **sharedId**.

    This API method returns rendition information where the rendition status is CREATED,
    which means the rendition is available to view/download.

    **Note:** No authentication is required to call this endpoint.

    Args:
        shared_id (str):
        rendition_id (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, RenditionEntry]]
    """

    kwargs = _get_kwargs(
        shared_id=shared_id,
        rendition_id=rendition_id,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    shared_id: str,
    rendition_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Optional[Union[Any, RenditionEntry]]:
    """Get shared link rendition information

     **Note:** this endpoint is available in Alfresco 5.2 and newer versions.

    Gets rendition information for the file with shared link identifier **sharedId**.

    This API method returns rendition information where the rendition status is CREATED,
    which means the rendition is available to view/download.

    **Note:** No authentication is required to call this endpoint.

    Args:
        shared_id (str):
        rendition_id (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, RenditionEntry]
    """

    return sync_detailed(
        shared_id=shared_id,
        rendition_id=rendition_id,
        client=client,
    ).parsed


async def asyncio_detailed(
    shared_id: str,
    rendition_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Response[Union[Any, RenditionEntry]]:
    """Get shared link rendition information

     **Note:** this endpoint is available in Alfresco 5.2 and newer versions.

    Gets rendition information for the file with shared link identifier **sharedId**.

    This API method returns rendition information where the rendition status is CREATED,
    which means the rendition is available to view/download.

    **Note:** No authentication is required to call this endpoint.

    Args:
        shared_id (str):
        rendition_id (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, RenditionEntry]]
    """

    kwargs = _get_kwargs(
        shared_id=shared_id,
        rendition_id=rendition_id,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    shared_id: str,
    rendition_id: str,
    *,
    client: Union[AuthenticatedClient, Client],
) -> Optional[Union[Any, RenditionEntry]]:
    """Get shared link rendition information

     **Note:** this endpoint is available in Alfresco 5.2 and newer versions.

    Gets rendition information for the file with shared link identifier **sharedId**.

    This API method returns rendition information where the rendition status is CREATED,
    which means the rendition is available to view/download.

    **Note:** No authentication is required to call this endpoint.

    Args:
        shared_id (str):
        rendition_id (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, RenditionEntry]
    """

    return (
        await asyncio_detailed(
            shared_id=shared_id,
            rendition_id=rendition_id,
            client=client,
        )
    ).parsed
