"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnCodeDeployBlueGreenHook = exports.CfnTrafficRoutingType = void 0;
const cfn_hook_1 = require("./cfn-hook");
const util_1 = require("./util");
/**
 * The possible types of traffic shifting for the blue-green deployment configuration.
 *
 * The type of the {@link CfnTrafficRoutingConfig.type} property.
 */
var CfnTrafficRoutingType;
(function (CfnTrafficRoutingType) {
    CfnTrafficRoutingType["ALL_AT_ONCE"] = "AllAtOnce";
    CfnTrafficRoutingType["TIME_BASED_CANARY"] = "TimeBasedCanary";
    CfnTrafficRoutingType["TIME_BASED_LINEAR"] = "TimeBasedLinear";
})(CfnTrafficRoutingType = exports.CfnTrafficRoutingType || (exports.CfnTrafficRoutingType = {}));
/**
 * A CloudFormation Hook for CodeDeploy blue-green ECS deployments.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/blue-green.html#blue-green-template-reference
 */
class CfnCodeDeployBlueGreenHook extends cfn_hook_1.CfnHook {
    /**
     * Creates a new CodeDeploy blue-green ECS Hook.
     *
     * @param scope the scope to create the hook in (usually the containing Stack object).
     * @param id the identifier of the construct - will be used to generate the logical ID of the Hook.
     * @param props the properties of the Hook.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            type: 'AWS::CodeDeploy::BlueGreen',
        });
        this._serviceRole = props.serviceRole;
        this._applications = props.applications;
        this._trafficRoutingConfig = props.trafficRoutingConfig;
        this._additionalOptions = props.additionalOptions;
        this._lifecycleEventHooks = props.lifecycleEventHooks;
    }
    /**
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope, id, hookAttributes, options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        hookAttributes = hookAttributes || {};
        const hookProperties = options.parser.parseValue(hookAttributes.Properties);
        return new CfnCodeDeployBlueGreenHook(scope, id, {
            serviceRole: hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.ServiceRole,
            applications: (_a = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.Applications) === null || _a === void 0 ? void 0 : _a.map(applicationFromCloudFormation),
            trafficRoutingConfig: {
                type: (_b = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.Type,
                timeBasedCanary: {
                    stepPercentage: (_d = (_c = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _c === void 0 ? void 0 : _c.TimeBasedCanary) === null || _d === void 0 ? void 0 : _d.StepPercentage,
                    bakeTimeMins: (_f = (_e = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _e === void 0 ? void 0 : _e.TimeBasedCanary) === null || _f === void 0 ? void 0 : _f.BakeTimeMins,
                },
                timeBasedLinear: {
                    stepPercentage: (_h = (_g = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _g === void 0 ? void 0 : _g.TimeBasedLinear) === null || _h === void 0 ? void 0 : _h.StepPercentage,
                    bakeTimeMins: (_k = (_j = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.TrafficRoutingConfig) === null || _j === void 0 ? void 0 : _j.TimeBasedLinear) === null || _k === void 0 ? void 0 : _k.BakeTimeMins,
                },
            },
            additionalOptions: {
                terminationWaitTimeInMinutes: (_l = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.AdditionalOptions) === null || _l === void 0 ? void 0 : _l.TerminationWaitTimeInMinutes,
            },
            lifecycleEventHooks: {
                beforeInstall: (_m = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.BeforeInstall,
                afterInstall: (_o = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.AfterInstall,
                afterAllowTestTraffic: (_p = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.AfterAllowTestTraffic,
                beforeAllowTraffic: (_q = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.BeforeAllowTraffic,
                afterAllowTraffic: (_r = hookProperties === null || hookProperties === void 0 ? void 0 : hookProperties.LifecycleEventHooks) === null || _r === void 0 ? void 0 : _r.AfterAllowTraffic,
            },
        });
        function applicationFromCloudFormation(app) {
            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w;
            const target = findResource((_a = app === null || app === void 0 ? void 0 : app.Target) === null || _a === void 0 ? void 0 : _a.LogicalID);
            const taskDefinitions = (_c = (_b = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _b === void 0 ? void 0 : _b.TaskDefinitions) === null || _c === void 0 ? void 0 : _c.map((td) => findResource(td));
            const taskSets = (_e = (_d = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _d === void 0 ? void 0 : _d.TaskSets) === null || _e === void 0 ? void 0 : _e.map((ts) => findResource(ts));
            const prodTrafficRoute = findResource((_h = (_g = (_f = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _f === void 0 ? void 0 : _f.TrafficRouting) === null || _g === void 0 ? void 0 : _g.ProdTrafficRoute) === null || _h === void 0 ? void 0 : _h.LogicalID);
            const testTrafficRoute = findResource((_l = (_k = (_j = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _j === void 0 ? void 0 : _j.TrafficRouting) === null || _k === void 0 ? void 0 : _k.TestTrafficRoute) === null || _l === void 0 ? void 0 : _l.LogicalID);
            const targetGroups = (_p = (_o = (_m = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _m === void 0 ? void 0 : _m.TrafficRouting) === null || _o === void 0 ? void 0 : _o.TargetGroups) === null || _p === void 0 ? void 0 : _p.map((tg) => findResource(tg));
            return {
                target: {
                    type: (_q = app === null || app === void 0 ? void 0 : app.Target) === null || _q === void 0 ? void 0 : _q.Type,
                    logicalId: target === null || target === void 0 ? void 0 : target.logicalId,
                },
                ecsAttributes: {
                    taskDefinitions: taskDefinitions === null || taskDefinitions === void 0 ? void 0 : taskDefinitions.map(td => td === null || td === void 0 ? void 0 : td.logicalId),
                    taskSets: taskSets === null || taskSets === void 0 ? void 0 : taskSets.map(ts => ts === null || ts === void 0 ? void 0 : ts.logicalId),
                    trafficRouting: {
                        prodTrafficRoute: {
                            type: (_t = (_s = (_r = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _r === void 0 ? void 0 : _r.TrafficRouting) === null || _s === void 0 ? void 0 : _s.ProdTrafficRoute) === null || _t === void 0 ? void 0 : _t.Type,
                            logicalId: prodTrafficRoute === null || prodTrafficRoute === void 0 ? void 0 : prodTrafficRoute.logicalId,
                        },
                        testTrafficRoute: {
                            type: (_w = (_v = (_u = app === null || app === void 0 ? void 0 : app.ECSAttributes) === null || _u === void 0 ? void 0 : _u.TrafficRouting) === null || _v === void 0 ? void 0 : _v.TestTrafficRoute) === null || _w === void 0 ? void 0 : _w.Type,
                            logicalId: testTrafficRoute === null || testTrafficRoute === void 0 ? void 0 : testTrafficRoute.logicalId,
                        },
                        targetGroups: targetGroups === null || targetGroups === void 0 ? void 0 : targetGroups.map((tg) => tg === null || tg === void 0 ? void 0 : tg.logicalId),
                    },
                },
            };
        }
        function findResource(logicalId) {
            if (logicalId == null) {
                return undefined;
            }
            const ret = options.parser.finder.findResource(logicalId);
            if (!ret) {
                throw new Error(`Hook '${id}' references resource '${logicalId}' that was not found in the template`);
            }
            return ret;
        }
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    get serviceRole() {
        return this._serviceRole;
    }
    /**
     * The IAM Role for CloudFormation to use to perform blue-green deployments.
     */
    set serviceRole(serviceRole) {
        this._serviceRole = serviceRole;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    get applications() {
        return this._applications;
    }
    /**
     * Properties of the Amazon ECS applications being deployed.
     */
    set applications(value) {
        this._applications = value;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    get trafficRoutingConfig() {
        return this._trafficRoutingConfig;
    }
    /**
     * Traffic routing configuration settings.
     *
     * @default - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
     */
    set trafficRoutingConfig(value) {
        this._trafficRoutingConfig = value;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    get additionalOptions() {
        return this._additionalOptions;
    }
    /**
     * Additional options for the blue/green deployment.
     *
     * @default - no additional options
     */
    set additionalOptions(value) {
        this._additionalOptions = value;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     *
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    get lifecycleEventHooks() {
        return this._lifecycleEventHooks;
    }
    /**
     * Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.
     *
     * You can use the same function or a different one for deployment lifecycle events.
     * Following completion of the validation tests,
     * the Lambda {@link CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic}
     * function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.
     *
     * @default - no lifecycle event hooks
     */
    set lifecycleEventHooks(value) {
        this._lifecycleEventHooks = value;
    }
    /**
     *
     */
    renderProperties(_props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        return {
            ServiceRole: this.serviceRole,
            Applications: this.applications.map((app) => ({
                Target: {
                    Type: app.target.type,
                    LogicalID: app.target.logicalId,
                },
                ECSAttributes: {
                    TaskDefinitions: app.ecsAttributes.taskDefinitions,
                    TaskSets: app.ecsAttributes.taskSets,
                    TrafficRouting: {
                        ProdTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.prodTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.prodTrafficRoute.logicalId,
                        },
                        TestTrafficRoute: {
                            Type: app.ecsAttributes.trafficRouting.testTrafficRoute.type,
                            LogicalID: app.ecsAttributes.trafficRouting.testTrafficRoute.logicalId,
                        },
                        TargetGroups: app.ecsAttributes.trafficRouting.targetGroups,
                    },
                },
            })),
            TrafficRoutingConfig: util_1.undefinedIfAllValuesAreEmpty({
                Type: (_a = this.trafficRoutingConfig) === null || _a === void 0 ? void 0 : _a.type,
                TimeBasedCanary: util_1.undefinedIfAllValuesAreEmpty({
                    StepPercentage: (_c = (_b = this.trafficRoutingConfig) === null || _b === void 0 ? void 0 : _b.timeBasedCanary) === null || _c === void 0 ? void 0 : _c.stepPercentage,
                    BakeTimeMins: (_e = (_d = this.trafficRoutingConfig) === null || _d === void 0 ? void 0 : _d.timeBasedCanary) === null || _e === void 0 ? void 0 : _e.bakeTimeMins,
                }),
                TimeBasedLinear: util_1.undefinedIfAllValuesAreEmpty({
                    StepPercentage: (_g = (_f = this.trafficRoutingConfig) === null || _f === void 0 ? void 0 : _f.timeBasedLinear) === null || _g === void 0 ? void 0 : _g.stepPercentage,
                    BakeTimeMins: (_j = (_h = this.trafficRoutingConfig) === null || _h === void 0 ? void 0 : _h.timeBasedLinear) === null || _j === void 0 ? void 0 : _j.bakeTimeMins,
                }),
            }),
            AdditionalOptions: util_1.undefinedIfAllValuesAreEmpty({
                TerminationWaitTimeInMinutes: (_k = this.additionalOptions) === null || _k === void 0 ? void 0 : _k.terminationWaitTimeInMinutes,
            }),
            LifecycleEventHooks: util_1.undefinedIfAllValuesAreEmpty({
                BeforeInstall: (_l = this.lifecycleEventHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall,
                AfterInstall: (_m = this.lifecycleEventHooks) === null || _m === void 0 ? void 0 : _m.afterInstall,
                AfterAllowTestTraffic: (_o = this.lifecycleEventHooks) === null || _o === void 0 ? void 0 : _o.afterAllowTestTraffic,
                BeforeAllowTraffic: (_p = this.lifecycleEventHooks) === null || _p === void 0 ? void 0 : _p.beforeAllowTraffic,
                AfterAllowTraffic: (_q = this.lifecycleEventHooks) === null || _q === void 0 ? void 0 : _q.afterAllowTraffic,
            }),
        };
    }
}
exports.CfnCodeDeployBlueGreenHook = CfnCodeDeployBlueGreenHook;
//# sourceMappingURL=data:application/json;base64,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