from dataclasses import dataclass
from decimal import Decimal
from itertools import chain
from typing import TypedDict

from pydantic import BaseModel


class LineItemResponse(BaseModel):
    line_item: dict[str, Decimal | None]


@dataclass
class LineItemScore:
    """Represents a line item match with its similarity score."""

    name: str
    description: str
    score: float


class LineItemType(TypedDict):
    name: str
    aliases: set[str]
    dataitemid: int
    spgi_name: str
    description: str


# all of these values must be lower case keys
LINE_ITEMS: list[LineItemType] = [
    {
        "name": "revenue",
        "aliases": {"normal_revenue", "regular_revenue"},
        "dataitemid": 112,
        "spgi_name": "Revenue",
        "description": "Revenue recognized from primary business activities (excludes non-operating income).",
    },
    {
        "name": "finance_division_revenue",
        "aliases": set(),
        "dataitemid": 52,
        "spgi_name": "Finance Div. Revenue",
        "description": "Revenue generated by a company's finance or captive lending operations.",
    },
    {
        "name": "insurance_division_revenue",
        "aliases": set(),
        "dataitemid": 70,
        "spgi_name": "Insurance Div. Revenue",
        "description": "Revenue attributable to insurance activities within a diversified company.",
    },
    {
        "name": "revenue_from_sale_of_assets",
        "aliases": set(),
        "dataitemid": 104,
        "spgi_name": "Gain(Loss) on Sale Of Assets (Rev)",
        "description": "Revenue recognized from selling long-lived assets outside normal product sales.",
    },
    {
        "name": "revenue_from_sale_of_investments",
        "aliases": set(),
        "dataitemid": 106,
        "spgi_name": "Gain(Loss) on Sale Of Invest. (Rev)",
        "description": "Revenue recognized from disposing of investment securities or ownership stakes.",
    },
    {
        "name": "revenue_from_interest_and_investment_income",
        "aliases": set(),
        "dataitemid": 110,
        "spgi_name": "Interest And Invest. Income (Rev)",
        "description": "Interest and investment returns classified within revenue rather than other income.",
    },
    {
        "name": "other_revenue",
        "aliases": set(),
        "dataitemid": 90,
        "spgi_name": "Other Revenue",
        "description": "Miscellaneous revenue that does not fall into primary operating categories.",
    },
    {
        "name": "total_other_revenue",
        "aliases": set(),
        "dataitemid": 357,
        "spgi_name": "Other Revenue, Total",
        "description": "Aggregate of all non-core or miscellaneous revenue categories.",
    },
    {
        "name": "fees_and_other_income",
        "aliases": set(),
        "dataitemid": 168,
        "spgi_name": "Fees and Other Income",
        "description": "Fee-based revenue combined with ancillary income streams in one line item.",
    },
    {
        "name": "total_revenue",
        "aliases": set(),
        "dataitemid": 28,
        "spgi_name": "Total Revenue",
        "description": "Sum of operating and non-operating revenue streams for the period.",
    },
    {
        "name": "cost_of_goods_sold",
        "aliases": {"cogs"},
        "dataitemid": 34,
        "spgi_name": "Cost Of Goods Sold",
        "description": "Direct costs attributable to producing goods sold during the period.",
    },
    {
        "name": "finance_division_operating_expense",
        "aliases": {"operating_expense_finance_division"},
        "dataitemid": 51,
        "spgi_name": "Finance Div. Operating Exp.",
        "description": "Operating expenses incurred within the company's finance or captive lending division.",
    },
    {
        "name": "insurance_division_operating_expense",
        "aliases": {"operating_expense_insurance_division"},
        "dataitemid": 69,
        "spgi_name": "Insurance Div. Operating Exp.",
        "description": "Operating expenses tied to insurance operations inside a diversified company.",
    },
    {
        "name": "finance_division_interest_expense",
        "aliases": {"interest_expense_finance_division"},
        "dataitemid": 50,
        "spgi_name": "Interest Expense - Finance Division",
        "description": "Interest expense recognized by the finance or captive lending division.",
    },
    {
        "name": "cost_of_revenue",
        "aliases": {"cor"},
        "dataitemid": 1,
        "spgi_name": "Cost Of Revenue",
        "description": "Direct costs associated with delivering goods or services that generate revenue.",
    },
    {
        "name": "gross_profit",
        "aliases": set(),
        "dataitemid": 10,
        "spgi_name": "Gross Profit",
        "description": "Revenue minus cost_of_goods_sold or cost_of_revenue for the reported period.",
    },
    {
        "name": "selling_general_and_admin_expense",
        "aliases": {
            "selling_general_and_admin_cost",
            "selling_general_and_admin",
            "sg_and_a",
            "sga",
        },
        "dataitemid": 102,
        "spgi_name": "Selling General & Admin Exp.",
        "description": "Combined selling, general, and administrative operating expenses.",
    },
    {
        "name": "exploration_and_drilling_costs",
        "aliases": {
            "exploration_and_drilling_expense",
        },
        "dataitemid": 49,
        "spgi_name": "Exploration/Drilling Costs",
        "description": "Costs incurred for exploration and drilling activities in energy sectors.",
    },
    {
        "name": "provision_for_bad_debts",
        "aliases": {
            "provision_for_bad_debt",
        },
        "dataitemid": 95,
        "spgi_name": "Provision for Bad Debts",
        "description": "Allowance for credit losses on receivables recorded during the period.",
    },
    {
        "name": "pre_opening_costs",
        "aliases": {
            "pre_opening_expense",
        },
        "dataitemid": 96,
        "spgi_name": "Pre-Opening Costs",
        "description": "Costs associated with opening new locations or facilities before revenue starts.",
    },
    {
        "name": "total_selling_general_and_admin_expense",
        "aliases": {
            "total_selling_general_and_admin_cost",
            "total_selling_general_and_admin",
            "total_sga",
        },
        "dataitemid": 23,
        "spgi_name": "SG&A Exp., Total",
        "description": "Aggregate SG&A expenses including all subcategories.",
    },
    {
        "name": "research_and_development_expense",
        "aliases": {
            "research_and_development_cost",
            "r_and_d_expense",
            "r_and_d_cost",
            "rnd_expense",
            "rnd_cost",
        },
        "dataitemid": 100,
        "spgi_name": "R & D Exp.",
        "description": "Expenses incurred for research and development activities.",
    },
    {
        "name": "depreciation_and_amortization",
        "aliases": {
            "d_and_a",
            "dna",
        },
        "dataitemid": 41,
        "spgi_name": "Depreciation & Amort.",
        "description": "Combined depreciation and amortization expense for the period.",
    },
    {
        "name": "amortization_of_goodwill_and_intangibles",
        "aliases": set(),
        "dataitemid": 31,
        "spgi_name": "Amort. of Goodwill and Intangibles",
        "description": "Amortization expense tied to goodwill and other intangible assets.",
    },
    {
        "name": "impairment_of_oil_gas_and_mineral_properties",
        "aliases": {
            "impairment_of_oil_and_gas",
            "impairment_o_and_g",
        },
        "dataitemid": 71,
        "spgi_name": "Impair. of Oil, Gas & Mineral Prop.",
        "description": "Impairment charges related to oil, gas, or mineral property valuations.",
    },
    {
        "name": "total_depreciation_and_amortization",
        "aliases": {
            "total_d_and_a",
            "total_dna",
        },
        "dataitemid": 2,
        "spgi_name": "Depreciation & Amort., Total",
        "description": "Total depreciation and amortization expense reported for the period.",
    },
    {
        "name": "other_operating_expense",
        "aliases": set(),
        "dataitemid": 260,
        "spgi_name": "Other Operating Expense/(Income)",
        "description": "Operating expenses not classified under standard categories.",
    },
    {
        "name": "total_other_operating_expense",
        "aliases": set(),
        "dataitemid": 380,
        "spgi_name": "Other Operating Exp., Total",
        "description": "Aggregate of operating expenses not categorized elsewhere.",
    },
    {
        "name": "total_operating_expense",
        "aliases": {
            "operating_expense",
        },
        "dataitemid": 373,
        "spgi_name": "Total Operating Expenses",
        "description": "Sum of all operating expenses for the reporting period.",
    },
    {
        "name": "operating_income",
        "aliases": set(),
        "dataitemid": 21,
        "spgi_name": "Operating Income",
        "description": "Operating profit after subtracting operating expenses from operating revenue.",
    },
    {
        "name": "interest_expense",
        "aliases": set(),
        "dataitemid": 82,
        "spgi_name": "Interest Expense",
        "description": "Total interest expense recognized during the period.",
    },
    {
        "name": "interest_and_investment_income",
        "aliases": set(),
        "dataitemid": 65,
        "spgi_name": "Interest and Invest. Income",
        "description": "Income from interest-bearing assets and investment returns outside core operations.",
    },
    {
        "name": "net_interest_expense",
        "aliases": set(),
        "dataitemid": 368,
        "spgi_name": "Net Interest Exp.",
        "description": "Interest expense net of any interest income for the period.",
    },
    {
        "name": "income_from_affiliates",
        "aliases": set(),
        "dataitemid": 47,
        "spgi_name": "Income / (Loss) from Affiliates",
        "description": "Equity-method income earned from affiliated companies.",
    },
    {
        "name": "currency_exchange_gains",
        "aliases": set(),
        "dataitemid": 38,
        "spgi_name": "Currency Exchange Gains (Loss)",
        "description": "Gains arising from favorable foreign currency movements.",
    },
    {
        "name": "other_non_operating_income",
        "aliases": set(),
        "dataitemid": 85,
        "spgi_name": "Other Non-Operating Inc. (Exp.)",
        "description": "Non-operating income items not captured in other categories.",
    },
    {
        "name": "total_other_non_operating_income",
        "aliases": set(),
        "dataitemid": 371,
        "spgi_name": "Other Non-Operating Exp., Total",
        "description": "Aggregate of non-operating income items (e.g., investment income, gains).",
    },
    {
        "name": "ebt_excluding_unusual_items",
        "aliases": {
            "earnings_before_taxes_excluding_unusual_items",
        },
        "dataitemid": 4,
        "spgi_name": "EBT Excl Unusual Items",
        "description": "Earnings before taxes with unusual/non-recurring items removed.",
    },
    {
        "name": "restructuring_charges",
        "aliases": set(),
        "dataitemid": 98,
        "spgi_name": "Restructuring Charges",
        "description": "Charges related to restructuring initiatives such as layoffs or facility closures.",
    },
    {
        "name": "merger_charges",
        "aliases": set(),
        "dataitemid": 80,
        "spgi_name": "Merger & Related Restruct. Charges",
        "description": "Costs incurred during merger transactions.",
    },
    {
        "name": "merger_and_restructuring_charges",
        "aliases": set(),
        "dataitemid": 363,
        "spgi_name": "Merger & Restruct. Charges",
        "description": "Combined charges stemming from merger and restructuring activities.",
    },
    {
        "name": "impairment_of_goodwill",
        "aliases": set(),
        "dataitemid": 209,
        "spgi_name": "Impairment of Goodwill",
        "description": "Write-downs of goodwill carrying value due to impairment testing.",
    },
    {
        "name": "gain_from_sale_of_assets",
        "aliases": set(),
        "dataitemid": 62,
        "spgi_name": "Gain (Loss) On Sale Of Invest.",
        "description": "Gain recognized from disposing tangible or intangible assets.",
    },
    {
        "name": "gain_from_sale_of_investments",
        "aliases": set(),
        "dataitemid": 56,
        "spgi_name": "Gain (Loss) On Sale Of Assets",
        "description": "Gain realized from selling investment holdings.",
    },
    {
        "name": "asset_writedown",
        "aliases": set(),
        "dataitemid": 32,
        "spgi_name": "Asset Writedown",
        "description": "Reduction in asset carrying value due to impairment or obsolescence.",
    },
    {
        "name": "in_process_research_and_development_expense",
        "aliases": {
            "in_process_research_and_development_cost",
            "in_process_r_and_d_expense",
            "in_process_r_and_d_cost",
            "in_process_rnd_expense",
            "in_process_rnd_cost",
        },
        "dataitemid": 72,
        "spgi_name": "In Process R & D Exp.",
        "description": "Expense recognized for acquired in-process R&D projects.",
    },
    {
        "name": "insurance_settlements",
        "aliases": set(),
        "dataitemid": 73,
        "spgi_name": "Insurance Settlements",
        "description": "Proceeds or expenses arising from insurance claim settlements.",
    },
    {
        "name": "legal_settlements",
        "aliases": set(),
        "dataitemid": 77,
        "spgi_name": "Legal Settlements",
        "description": "Amounts paid or received to resolve legal actions.",
    },
    {
        "name": "other_unusual_items",
        "aliases": set(),
        "dataitemid": 87,
        "spgi_name": "Other Unusual Items",
        "description": "Unusual or infrequent items not captured elsewhere.",
    },
    {
        "name": "total_other_unusual_items",
        "aliases": set(),
        "dataitemid": 374,
        "spgi_name": "Other Unusual Items, Total",
        "description": "Aggregate impact of all unusual items except those categorized separately.",
    },
    {
        "name": "total_unusual_items",
        "aliases": {
            "unusual_items",
        },
        "dataitemid": 19,
        "spgi_name": "Total Unusual Items",
        "description": "Aggregate impact of all unusual or non-recurring items for the reporting period.",
    },
    {
        "name": "ebt_including_unusual_items",
        "aliases": {
            "earnings_before_taxes_including_unusual_items",
        },
        "dataitemid": 139,
        "spgi_name": "EBT Incl. Unusual Items",
        "description": "Earnings before taxes, including unusual or non-recurring items.",
    },
    {
        "name": "income_tax_expense",
        "aliases": {
            "income_taxes",
            "income_tax",
        },
        "dataitemid": 75,
        "spgi_name": "Income Tax Expense",
        "description": "Total income tax expense recognized for the period.",
    },
    {
        "name": "earnings_from_continued_operations",
        "aliases": {
            "continued_operations_earnings",
        },
        "dataitemid": 7,
        "spgi_name": "Earnings from Cont. Ops.",
        "description": "Earnings from ongoing operations excluding discontinued segments.",
    },
    {
        "name": "earnings_from_discontinued_operations",
        "aliases": {
            "discontinued_operations_earnings",
        },
        "dataitemid": 40,
        "spgi_name": "Earnings of Discontinued Ops.",
        "description": "Net earnings attributable to discontinued operations.",
    },
    {
        "name": "extraordinary_item_and_accounting_change",
        "aliases": set(),
        "dataitemid": 42,
        "spgi_name": "Extraord. Item & Account. Change",
        "description": "Combined impact of extraordinary items and accounting changes.",
    },
    {
        "name": "net_income_to_company",
        "aliases": set(),
        "dataitemid": 41571,
        "spgi_name": "Net Income to Company",
        "description": "Net income attributable to the parent company.",
    },
    {
        "name": "minority_interest_in_earnings",
        "aliases": {
            "net_income_to_minority_interest",
        },
        "dataitemid": 83,
        "spgi_name": "Minority Int. in Earnings",
        "description": "Earnings attributable to minority interest holders.",
    },
    {
        "name": "net_income",
        "aliases": set(),
        "dataitemid": 15,
        "spgi_name": "Net Income",
        "description": "Bottom-line profit attributable to common shareholders.",
    },
    {
        "name": "premium_on_redemption_of_preferred_stock",
        "aliases": set(),
        "dataitemid": 279,
        "spgi_name": "Premium on Redemption of Pref. Stock",
        "description": "Premium paid when redeeming preferred shares above par value.",
    },
    {
        "name": "preferred_stock_dividend",
        "aliases": set(),
        "dataitemid": 280,
        "spgi_name": "Preferred Stock Dividend",
        "description": "Dividends paid to preferred shareholders.",
    },
    {
        "name": "other_preferred_stock_adjustments",
        "aliases": set(),
        "dataitemid": 281,
        "spgi_name": "Other Pref. Stock Adjustments",
        "description": "Adjustments related to preferred stock outside dividends and redemption premiums.",
    },
    {
        "name": "other_adjustments_to_net_income",
        "aliases": set(),
        "dataitemid": 259,
        "spgi_name": "Other Adjustments to Net Income",
        "description": "Miscellaneous adjustments applied to net income calculations.",
    },
    {
        "name": "preferred_dividends_and_other_adjustments",
        "aliases": set(),
        "dataitemid": 97,
        "spgi_name": "Pref. Dividends and Other Adj.",
        "description": "Combined preferred dividend and adjustment line item.",
    },
    {
        "name": "net_income_allocable_to_general_partner",
        "aliases": set(),
        "dataitemid": 249,
        "spgi_name": "Net Income Allocable to General Partner",
        "description": "Net income portion allocated to the general partner.",
    },
    {
        "name": "net_income_to_common_shareholders_including_extra_items",
        "aliases": set(),
        "dataitemid": 16,
        "spgi_name": "NI to Common Incl. Extra Items",
        "description": "Net income available to common shareholders inclusive of extraordinary items.",
    },
    {
        "name": "net_income_to_common_shareholders_excluding_extra_items",
        "aliases": set(),
        "dataitemid": 379,
        "spgi_name": "NI to Common Excl. Extra Items",
        "description": "Net income to common shareholders excluding extraordinary items.",
    },
    {
        "name": "cash_and_equivalents",
        "aliases": {
            "cash",
            "cash_and_cash_equivalents",
        },
        "dataitemid": 1096,
        "spgi_name": "Cash And Equivalents",
        "description": "Cash on hand plus cash-equivalent short-term investments.",
    },
    {
        "name": "short_term_investments",
        "aliases": set(),
        "dataitemid": 1069,
        "spgi_name": "Short Term Investments",
        "description": "Short-term investments readily convertible to cash.",
    },
    {
        "name": "trading_asset_securities",
        "aliases": set(),
        "dataitemid": 1244,
        "spgi_name": "Trading Asset Securities",
        "description": "Trading securities held as short-term financial assets.",
    },
    {
        "name": "total_cash_and_short_term_investments",
        "aliases": {
            "cash_and_short_term_investments",
        },
        "dataitemid": 1002,
        "spgi_name": "Total Cash & ST Investments",
        "description": "Total cash plus short-term investment balance.",
    },
    {
        "name": "accounts_receivable",
        "aliases": {
            "short_term_accounts_receivable",
            "current_accounts_receivable",
        },
        "dataitemid": 1021,
        "spgi_name": "Accounts Receivable",
        "description": "Accounts receivable balance due from customers.",
    },
    {
        "name": "other_receivables",
        "aliases": {
            "short_term_other_receivables",
            "current_other_receivables",
        },
        "dataitemid": 1206,
        "spgi_name": "Other Receivables",
        "description": "Receivables that are not trade receivables (e.g., tax refunds, employee advances).",
    },
    {
        "name": "notes_receivable",
        "aliases": {
            "short_term_notes_receivable",
            "current_notes_receivable",
        },
        "dataitemid": 1048,
        "spgi_name": "Notes Receivable",
        "description": "Amounts owed to the company via promissory notes.",
    },
    {
        "name": "total_receivables",
        "aliases": {
            "short_term_total_receivables",
            "current_total_receivables",
            "total_receivable",
            "short_term_total_receivable",
            "current_total_receivable",
        },
        "dataitemid": 1001,
        "spgi_name": "Total Receivables",
        "description": "Aggregate receivable balance across categories.",
    },
    {
        "name": "inventory",
        "aliases": {
            "inventories",
        },
        "dataitemid": 1043,
        "spgi_name": "Inventory",
        "description": "Inventory balance reported for the period.",
    },
    {
        "name": "prepaid_expense",
        "aliases": {
            "prepaid_expenses",
        },
        "dataitemid": 1212,
        "spgi_name": "Prepaid Exp.",
        "description": "Prepaid expenses for goods or services yet to be received.",
    },
    {
        "name": "finance_division_loans_and_leases_short_term",
        "aliases": {
            "finance_division_short_term_loans_and_leases",
            "short_term_finance_division_loans_and_leases",
            "short_term_loans_and_leases_of_the_finance_division",
        },
        "dataitemid": 1032,
        "spgi_name": "Finance Div. Loans and Leases, ST",
        "description": "Short-term loans and leases held by the finance division.",
    },
    {
        "name": "finance_division_other_current_assets",
        "aliases": {
            "finance_division_other_short_term_assets",
            "other_current_assets_of_the_finance_division",
            "other_short_term_assets_of_the_finance_division",
        },
        "dataitemid": 1029,
        "spgi_name": "Finance Div. Other Curr. Assets",
        "description": "Other current assets specific to a company's finance division.",
    },
    {
        "name": "loans_held_for_sale",
        "aliases": set(),
        "dataitemid": 1185,
        "spgi_name": "Loans Held For Sale",
        "description": "Loans designated for sale rather than retention on the balance sheet.",
    },
    {
        "name": "deferred_tax_asset_current_portion",
        "aliases": {
            "current_deferred_tax_asset",
            "short_term_deferred_tax_asset",
        },
        "dataitemid": 1117,
        "spgi_name": "Deferred Tax Assets, Curr.",
        "description": "Current portion of deferred tax assets expected to be realized within a year.",
    },
    {
        "name": "restricted_cash",
        "aliases": set(),
        "dataitemid": 1104,
        "spgi_name": "Restricted Cash",
        "description": "Cash balances with usage restrictions (e.g., escrow, collateral).",
    },
    {
        "name": "other_current_assets",
        "aliases": set(),
        "dataitemid": 1055,
        "spgi_name": "Other Current Assets",
        "description": "Miscellaneous current assets not classified elsewhere.",
    },
    {
        "name": "total_current_assets",
        "aliases": {
            "current_assets",
            "total_short_term_assets",
            "short_term_assets",
        },
        "dataitemid": 1008,
        "spgi_name": "Total Current Assets",
        "description": "Sum of all assets classified as current for the reporting date.",
    },
    {
        "name": "gross_property_plant_and_equipment",
        "aliases": {
            "gppe",
            "gross_ppe",
        },
        "dataitemid": 1169,
        "spgi_name": "Gross Property, Plant & Equipment",
        "description": "Gross value of PP&E before accumulated depreciation.",
    },
    {
        "name": "accumulated_depreciation",
        "aliases": set(),
        "dataitemid": 1075,
        "spgi_name": "Accumulated Depreciation",
        "description": "Cumulative depreciation recorded against property, plant, and equipment.",
    },
    {
        "name": "net_property_plant_and_equipment",
        "aliases": {
            "property_plant_and_equipment",
            "nppe",
            "ppe",
            "net_ppe",
        },
        "dataitemid": 1004,
        "spgi_name": "Net Property, Plant & Equipment",
        "description": "Net property, plant, and equipment after accumulated depreciation.",
    },
    {
        "name": "long_term_investments",
        "aliases": {
            "non_current_investments",
        },
        "dataitemid": 1054,
        "spgi_name": "Long-term Investments",
        "description": "Non-current investments intended to be held longer than one year.",
    },
    {
        "name": "goodwill",
        "aliases": set(),
        "dataitemid": 1171,
        "spgi_name": "Goodwill",
        "description": "",
    },
    {
        "name": "other_intangibles",
        "aliases": set(),
        "dataitemid": 1040,
        "spgi_name": "Other Intangibles",
        "description": "Intangible assets other than goodwill (e.g., patents, trademarks).",
    },
    {
        "name": "finance_division_loans_and_leases_long_term",
        "aliases": {
            "finance_division_long_term_loans_and_leases",
            "long_term_finance_division_loans_and_leases",
            "long_term_loans_and_leases_of_the_finance_division",
        },
        "dataitemid": 1033,
        "spgi_name": "Finance Div. Loans and Leases, LT",
        "description": "",
    },
    {
        "name": "finance_division_other_non_current_assets",
        "aliases": {
            "finance_division_other_long_term_assets",
            "other_non_current_assets_of_the_finance_division",
            "other_long_term_assets_of_the_finance_division",
        },
        "dataitemid": 1034,
        "spgi_name": "Finance Div. Other LT Assets",
        "description": "",
    },
    {
        "name": "long_term_accounts_receivable",
        "aliases": {
            "non_current_accounts_receivable",
        },
        "dataitemid": 1088,
        "spgi_name": "Accounts Receivable Long-Term",
        "description": "",
    },
    {
        "name": "long_term_loans_receivable",
        "aliases": {
            "non_current_loans_receivable",
            "loans_receivable",
        },
        "dataitemid": 1050,
        "spgi_name": "Loans Receivable Long-Term",
        "description": "",
    },
    {
        "name": "long_term_deferred_tax_assets",
        "aliases": {
            "non_current_deferred_tax_assets",
        },
        "dataitemid": 1026,
        "spgi_name": "Deferred Tax Assets, LT",
        "description": "",
    },
    {
        "name": "long_term_deferred_charges",
        "aliases": {
            "non_current_deferred_charges",
        },
        "dataitemid": 1025,
        "spgi_name": "Deferred Charges, LT",
        "description": "Deferred charges expected to provide benefits beyond one year.",
    },
    {
        "name": "other_long_term_assets",
        "aliases": {
            "long_term_other_assets",
            "other_non_current_assets",
            "non_current_other_assets",
        },
        "dataitemid": 1060,
        "spgi_name": "Other Long-Term Assets",
        "description": "",
    },
    {
        "name": "total_assets",
        "aliases": {
            "assets",
        },
        "dataitemid": 1007,
        "spgi_name": "Total Assets",
        "description": "",
    },
    {
        "name": "accounts_payable",
        "aliases": set(),
        "dataitemid": 1018,
        "spgi_name": "Accounts Payable",
        "description": "",
    },
    {
        "name": "accrued_expenses",
        "aliases": set(),
        "dataitemid": 1016,
        "spgi_name": "Accrued Expenses",
        "description": "Accrued expenses and other short-term liabilities awaiting payment.",
    },
    {
        "name": "short_term_borrowings",
        "aliases": {
            "current_borrowings",
            "short_term_borrowing",
            "current_borrowing",
        },
        "dataitemid": 1046,
        "spgi_name": "Short-term Borrowings",
        "description": "Borrowings that mature within twelve months of the reporting date.",
    },
    {
        "name": "current_portion_of_long_term_debt",
        "aliases": {
            "current_portion_of_non_current_debt",
            "current_portion_of_lt_debt",
        },
        "dataitemid": 1297,
        "spgi_name": "Current Portion of Long Term Debt",
        "description": "Portion of long-term debt due within the next year.",
    },
    {
        "name": "current_portion_of_capital_leases",
        "aliases": {
            "current_portion_of_capitalized_leases",
            "current_portion_of_cap_leases",
            "current_portion_of_leases",
        },
        "dataitemid": 1090,
        "spgi_name": "Curr. Port. of Cap. Leases",
        "description": "Short-term portion of capital lease obligations due within 12 months.",
    },
    {
        "name": "current_portion_of_long_term_debt_and_capital_leases",
        "aliases": {
            "current_portion_of_lt_debt_and_cap_leases",
            "current_portion_of_long_term_debt_and_capitalized_leases",
            "current_portion_of_non_current_debt_and_capital_leases",
            "current_portion_of_non_current_debt_and_capitalized_leases",
            "total_current_portion_of_long_term_debt_and_capital_leases",
            "total_current_portion_of_lt_debt_and_cap_leases",
            "total_current_portion_of_long_term_debt_and_capitalized_leases",
            "total_current_portion_of_non_current_debt_and_capital_leases",
            "total_current_portion_of_non_current_debt_and_capitalized_leases",
        },
        "dataitemid": 1279,
        "spgi_name": "Curr. Port. of LT Debt/Cap. Leases",
        "description": "Combined current portion of long-term debt and capital leases.",
    },
    {
        "name": "finance_division_debt_current_portion",
        "aliases": set(),
        "dataitemid": 1030,
        "spgi_name": "Finance Div. Debt Current",
        "description": "Current portion of debt held within the finance division.",
    },
    {
        "name": "finance_division_other_current_liabilities",
        "aliases": set(),
        "dataitemid": 1031,
        "spgi_name": "Finance Div. Other Curr. Liab.",
        "description": "Other current liabilities associated with the finance division.",
    },
    {
        "name": "current_income_taxes_payable",
        "aliases": {
            "current_portion_of_income_taxes_payable",
        },
        "dataitemid": 1094,
        "spgi_name": "Curr. Income Taxes Payable",
        "description": "Current portion of income taxes owed but not yet paid (balance sheet liability).",
    },
    {
        "name": "current_unearned_revenue",
        "aliases": {
            "current_portion_of_unearned_revenue",
        },
        "dataitemid": 1074,
        "spgi_name": "Unearned Revenue, Current",
        "description": "Unearned revenue expected to be recognized within one year.",
    },
    {
        "name": "current_deferred_tax_liability",
        "aliases": set(),
        "dataitemid": 1119,
        "spgi_name": "Def. Tax Liability, Curr.",
        "description": "Deferred tax liabilities classified as current.",
    },
    {
        "name": "other_current_liability",
        "aliases": {
            "other_current_liabilities",
        },
        "dataitemid": 1057,
        "spgi_name": "Other Current Liabilities",
        "description": "Miscellaneous current liability category.",
    },
    {
        "name": "total_current_liabilities",
        "aliases": {
            "current_liabilities",
        },
        "dataitemid": 1009,
        "spgi_name": "Total Current Liabilities",
        "description": "Total balance of current liabilities reported.",
    },
    {
        "name": "long_term_debt",
        "aliases": {
            "non_current_debt",
        },
        "dataitemid": 1049,
        "spgi_name": "Long-Term Debt",
        "description": "Debt obligations with maturity dates beyond one year.",
    },
    {
        "name": "capital_leases",
        "aliases": {
            "long_term_leases",
            "capitalized_leases",
        },
        "dataitemid": 1183,
        "spgi_name": "Capital Leases",
        "description": "Long-term obligations arising from capital lease arrangements.",
    },
    {
        "name": "finance_division_debt_non_current_portion",
        "aliases": {
            "finance_division_debt_long_term_portion",
            "finance_division_non_current_debt",
            "finance_division_long_term_debt",
        },
        "dataitemid": 1035,
        "spgi_name": "Finance Div. Debt Non-Curr.",
        "description": "Non-current portion of debt held within the finance division.",
    },
    {
        "name": "finance_division_other_non_current_liabilities",
        "aliases": {
            "finance_division_other_long_term_liabilities",
        },
        "dataitemid": 1036,
        "spgi_name": "Finance Div. Other Non-Curr. Liab.",
        "description": "Other non-current liabilities reported by the finance division.",
    },
    {
        "name": "non_current_unearned_revenue",
        "aliases": {
            "long_term_unearned_revenue",
        },
        "dataitemid": 1256,
        "spgi_name": "Unearned Revenue, Non-Current",
        "description": "Revenue received in advance that will be recognized beyond one year.",
    },
    {
        "name": "pension_and_other_post_retirement_benefit",
        "aliases": set(),
        "dataitemid": 1213,
        "spgi_name": "Pension & Other Post-Retire. Benefits",
        "description": "Liabilities for pension and post-retirement benefit obligations.",
    },
    {
        "name": "non_current_deferred_tax_liability",
        "aliases": set(),
        "dataitemid": 1027,
        "spgi_name": "Def. Tax Liability, Non-Curr.",
        "description": "Deferred tax liabilities not expected to be settled within one year.",
    },
    {
        "name": "other_non_current_liabilities",
        "aliases": {
            "non_current_other_liabilities",
            "other_long_term_liabilities",
            "long_term_other_liabilities",
        },
        "dataitemid": 1062,
        "spgi_name": "Other Non-Current Liabilities",
        "description": "Miscellaneous non-current liability categories.",
    },
    {
        "name": "total_liabilities",
        "aliases": {
            "liabilities",
        },
        "dataitemid": 1276,
        "spgi_name": "Total Liabilities",
        "description": "Total liabilities including both current and non-current obligations.",
    },
    {
        "name": "preferred_stock_redeemable",
        "aliases": {
            "redeemable_preferred_stock",
        },
        "dataitemid": 1217,
        "spgi_name": "Pref. Stock, Redeemable",
        "description": "Redeemable preferred stock outstanding.",
    },
    {
        "name": "preferred_stock_non_redeemable",
        "aliases": {
            "non_redeemable_preferred_stock",
        },
        "dataitemid": 1216,
        "spgi_name": "Pref. Stock, Non-Redeem.",
        "description": "Non-redeemable preferred stock outstanding.",
    },
    {
        "name": "preferred_stock_convertible",
        "aliases": {
            "convertible_preferred_stock",
        },
        "dataitemid": 1214,
        "spgi_name": "Pref. Stock, Convertible",
        "description": "Preferred shares that can be converted into common stock.",
    },
    {
        "name": "preferred_stock_other",
        "aliases": {
            "other_preferred_stock",
        },
        "dataitemid": 1065,
        "spgi_name": "Pref. Stock, Other",
        "description": "Other preferred equity categories not classified elsewhere.",
    },
    {
        "name": "preferred_stock_additional_paid_in_capital",
        "aliases": {
            "additional_paid_in_capital_preferred_stock",
        },
        "dataitemid": 1085,
        "spgi_name": "Additional Paid In Capital - Preferred Stock",
        "description": "Additional paid-in capital attributable to preferred stock.",
    },
    {
        "name": "preferred_stock_equity_adjustment",
        "aliases": {
            "equity_adjustment_preferred_stock",
        },
        "dataitemid": 1215,
        "spgi_name": "Equity Adjustment - Preferred Stock",
        "description": "Equity adjustment related to preferred stock balances.",
    },
    {
        "name": "treasury_stock_preferred_stock_convertible",
        "aliases": {
            "treasury_preferred_stock_convertible",
            "treasury_stock_convertible_preferred_stock",
            "treasury_convertible_preferred_stock",
        },
        "dataitemid": 1249,
        "spgi_name": "Treasury Stock : Preferred Stock Convertible",
        "description": "",
    },
    {
        "name": "treasury_stock_preferred_stock_non_redeemable",
        "aliases": {
            "treasury_preferred_stock_non_redeemable",
            "treasury_stock_non_redeemable_preferred_stock",
            "treasury_non_redeemable_preferred_stock",
        },
        "dataitemid": 1250,
        "spgi_name": "Treasury Stock : Preferred Stock Non Redeemable",
        "description": "",
    },
    {
        "name": "treasury_stock_preferred_stock_redeemable",
        "aliases": {
            "treasury_preferred_stock_redeemable",
            "treasury_stock_redeemable_preferred_stock",
            "treasury_redeemable_preferred_stock",
        },
        "dataitemid": 1251,
        "spgi_name": "Treasury Stock : Preferred Stock Redeemable",
        "description": "Treasury shares held for redeemable preferred stock classes.",
    },
    {
        "name": "total_preferred_equity",
        "aliases": {
            "total_preferred_stock",
            "preferred_equity",
            "preferred_stock",
        },
        "dataitemid": 1005,
        "spgi_name": "Total Pref. Equity",
        "description": "Total preferred equity including all preferred share classes.",
    },
    {
        "name": "common_stock",
        "aliases": set(),
        "dataitemid": 1103,
        "spgi_name": "Common Stock",
        "description": "Par value of common stock issued and outstanding.",
    },
    {
        "name": "additional_paid_in_capital",
        "aliases": set(),
        "dataitemid": 1084,
        "spgi_name": "Additional Paid In Capital",
        "description": "Capital paid by shareholders above par value for common stock.",
    },
    {
        "name": "retained_earnings",
        "aliases": set(),
        "dataitemid": 1222,
        "spgi_name": "Retained Earnings",
        "description": "Accumulated retained earnings available to common shareholders.",
    },
    {
        "name": "treasury_stock",
        "aliases": set(),
        "dataitemid": 1248,
        "spgi_name": "Treasury Stock",
        "description": "Cost of company shares repurchased and held in treasury.",
    },
    {
        "name": "other_equity",
        "aliases": set(),
        "dataitemid": 1028,
        "spgi_name": "Comprehensive Inc. and Other",
        "description": "Other equity components not categorized elsewhere.",
    },
    {
        "name": "total_common_equity",
        "aliases": {
            "common_equity",
        },
        "dataitemid": 1006,
        "spgi_name": "Total Common Equity",
        "description": "Total equity attributable to common shareholders.",
    },
    {
        "name": "total_equity",
        "aliases": {
            "equity",
            "total_shareholders_equity",
            "shareholders_equity",
        },
        "dataitemid": 1275,
        "spgi_name": "Total Equity",
        "description": "Total shareholders' equity including preferred and common components.",
    },
    {
        "name": "total_liabilities_and_equity",
        "aliases": {
            "liabilities_and_equity",
        },
        "dataitemid": 1013,
        "spgi_name": "Total Liabilities And Equity",
        "description": "Total liabilities plus shareholders' equity to balance assets.",
    },
    {
        "name": "common_shares_outstanding",
        "aliases": set(),
        "dataitemid": 1100,
        "spgi_name": "Common Shares Outstanding",
        "description": "Weighted or period-end count of common shares outstanding.",
    },
    {
        "name": "adjustments_to_cash_flow_net_income",
        "aliases": set(),
        "dataitemid": 21523,
        "spgi_name": "Adjustments to Cash Flow Net Income",
        "description": "Adjustments reconciling net income to operating cash flow.",
    },
    {
        "name": "other_amortization",
        "aliases": set(),
        "dataitemid": 2014,
        "spgi_name": "Other Amortization",
        "description": "Amortization charges not captured in major categories.",
    },
    {
        "name": "total_other_non_cash_items",
        "aliases": set(),
        "dataitemid": 2179,
        "spgi_name": "Other Non-Cash Items, Total",
        "description": "Aggregate of non-cash items affecting operating cash flow.",
    },
    {
        "name": "net_decrease_in_loans_originated_and_sold",
        "aliases": set(),
        "dataitemid": 2033,
        "spgi_name": "Net (Increase)/Decrease in Loans Orig/Sold",
        "description": "Net decrease in loans originated and sold impacting cash flow.",
    },
    {
        "name": "provision_for_credit_losses",
        "aliases": set(),
        "dataitemid": 2112,
        "spgi_name": "Provision for Credit Losses",
        "description": "Provision recorded for credit losses within the period.",
    },
    {
        "name": "loss_on_equity_investments",
        "aliases": set(),
        "dataitemid": 2086,
        "spgi_name": "(Income) Loss on Equity Invest.",
        "description": "Loss recognized on equity method or other equity investments.",
    },
    {
        "name": "stock_based_compensation",
        "aliases": set(),
        "dataitemid": 2127,
        "spgi_name": "Stock-Based Compensation",
        "description": "Expense recognized for stock-based employee compensation.",
    },
    {
        "name": "tax_benefit_from_stock_options",
        "aliases": set(),
        "dataitemid": 2135,
        "spgi_name": "Tax Benefit from Stock Options",
        "description": "Tax benefit realized from employee stock option exercises.",
    },
    {
        "name": "net_cash_from_discontinued_operation",
        "aliases": {
            "cash_from_discontinued_operation",
        },
        "dataitemid": 2081,
        "spgi_name": "Net Cash From Discontinued Ops.",
        "description": "Net cash flow attributable to discontinued operations.",
    },
    {
        "name": "other_operating_activities",
        "aliases": set(),
        "dataitemid": 2047,
        "spgi_name": "Other Operating Activities",
        "description": "Other operating cash flow adjustments not listed elsewhere.",
    },
    {
        "name": "change_in_trading_asset_securities",
        "aliases": set(),
        "dataitemid": 2134,
        "spgi_name": "Change in Trad. Asset Securities",
        "description": "Cash impact from changes in trading asset securities.",
    },
    {
        "name": "change_in_accounts_receivable",
        "aliases": set(),
        "dataitemid": 2018,
        "spgi_name": "Change In Accounts Receivable",
        "description": "Operating cash flow impact from changes in accounts receivable.",
    },
    {
        "name": "change_in_inventories",
        "aliases": set(),
        "dataitemid": 2099,
        "spgi_name": "Change In Inventories",
        "description": "Change in inventory balances, typically from the cash flow statement.",
    },
    {
        "name": "change_in_accounts_payable",
        "aliases": set(),
        "dataitemid": 2017,
        "spgi_name": "Change in Acc. Payable",
        "description": "Operating cash flow impact from changes in accounts payable.",
    },
    {
        "name": "change_in_unearned_revenue",
        "aliases": set(),
        "dataitemid": 2139,
        "spgi_name": "Change in Unearned Rev.",
        "description": "Cash flow adjustment from changes in unearned revenue balances.",
    },
    {
        "name": "change_in_income_taxes",
        "aliases": set(),
        "dataitemid": 2101,
        "spgi_name": "Change in Inc. Taxes",
        "description": "Period-over-period change in income tax expense on the income statement.",
    },
    {
        "name": "change_in_deferred_taxes",
        "aliases": set(),
        "dataitemid": 2084,
        "spgi_name": "Change in Def. Taxes",
        "description": "Cash flow adjustment from changes in deferred tax balances.",
    },
    {
        "name": "change_in_other_net_operating_assets",
        "aliases": set(),
        "dataitemid": 2045,
        "spgi_name": "Change in Other Net Operating Assets",
        "description": "Change in other net operating assets affecting cash flow.",
    },
    {
        "name": "change_in_net_operating_assets",
        "aliases": set(),
        "dataitemid": 2010,
        "spgi_name": "Change in Net Operating Assets ",
        "description": "Aggregate change in net operating assets during the period.",
    },
    {
        "name": "cash_from_operations",
        "aliases": {
            "cash_from_operating_activities",
            "cash_flow_from_operations",
        },
        "dataitemid": 2006,
        "spgi_name": "Cash from Ops.",
        "description": "Operating cash flow generated during the reporting period.",
    },
    {
        "name": "capital_expenditure",
        "aliases": {
            "capital_expenditures",
            "capex",
        },
        "dataitemid": 2021,
        "spgi_name": "Capital Expenditure",
        "description": "Cash outflows for capital expenditures during the period.",
    },
    {
        "name": "sale_of_property_plant_and_equipment",
        "aliases": {
            "sale_of_ppe",
        },
        "dataitemid": 2042,
        "spgi_name": "Sale of Property, Plant, and Equipment",
        "description": "Cash inflows from selling property, plant, and equipment.",
    },
    {
        "name": "cash_acquisitions",
        "aliases": set(),
        "dataitemid": 2057,
        "spgi_name": "Cash Acquisitions",
        "description": "Cash paid for acquisitions of businesses or assets.",
    },
    {
        "name": "divestitures",
        "aliases": set(),
        "dataitemid": 2077,
        "spgi_name": "Divestitures",
        "description": "Cash proceeds or impact from divested businesses or assets (distinct from sale_of_real_estate).",
    },
    {
        "name": "sale_of_real_estate",
        "aliases": {
            "sale_of_real_properties",
            "sale_of_real_estate_properties",
        },
        "dataitemid": 2040,
        "spgi_name": "Sale (Purchase) of Real Estate properties",
        "description": "Proceeds from real estate sales (distinct from broader divestitures).",
    },
    {
        "name": "sale_of_intangible_assets",
        "aliases": {
            "sale_of_intangible_asset",
            "sale_of_intangibles",
        },
        "dataitemid": 2029,
        "spgi_name": "Sale (Purchase) of Intangible assets",
        "description": "Cash received from selling intangible assets.",
    },
    {
        "name": "net_cash_from_investments",
        "aliases": set(),
        "dataitemid": 2027,
        "spgi_name": "Net Cash from Investments",
        "description": "Net cash flow provided by investing activities overall.",
    },
    {
        "name": "net_decrease_in_investment_loans_originated_and_sold",
        "aliases": set(),
        "dataitemid": 2032,
        "spgi_name": "Net (Increase)/Decrease in Loans Orig/Sold",
        "description": "Net decrease in investment loans originated and sold.",
    },
    {
        "name": "other_investing_activities",
        "aliases": set(),
        "dataitemid": 2051,
        "spgi_name": "Other Investing Activities",
        "description": "Miscellaneous investing cash flow activities.",
    },
    {
        "name": "total_other_investing_activities",
        "aliases": set(),
        "dataitemid": 2177,
        "spgi_name": "Other Investing Activities, Total",
        "description": "Aggregate of other investing cash flow activities.",
    },
    {
        "name": "cash_from_investing",
        "aliases": {
            "cash_from_investing_activities",
            "cashflow_from_investing",
            "cashflow_from_investing_activities",
        },
        "dataitemid": 2005,
        "spgi_name": "Cash from Investing",
        "description": "Cash provided by investing activities.",
    },
    {
        "name": "short_term_debt_issued",
        "aliases": {
            "current_debt_issued",
        },
        "dataitemid": 2043,
        "spgi_name": "Short Term Debt Issued",
        "description": "Cash inflows from issuing short-term debt.",
    },
    {
        "name": "long_term_debt_issued",
        "aliases": {
            "non_current_debt_issued",
        },
        "dataitemid": 2034,
        "spgi_name": "Long-Term Debt Issued",
        "description": "Cash inflows from issuing long-term debt.",
    },
    {
        "name": "total_debt_issued",
        "aliases": set(),
        "dataitemid": 2161,
        "spgi_name": "Total Debt Issued",
        "description": "Aggregate debt issuance during the period.",
    },
    {
        "name": "short_term_debt_repaid",
        "aliases": {
            "current_debt_repaid",
        },
        "dataitemid": 2044,
        "spgi_name": "Short Term Debt Repaid",
        "description": "Cash outflows from repaying short-term debt.",
    },
    {
        "name": "long_term_debt_repaid",
        "aliases": {
            "non_current_debt_repaid",
        },
        "dataitemid": 2036,
        "spgi_name": "Long-Term Debt Repaid",
        "description": "Cash outflows from repaying long-term debt.",
    },
    {
        "name": "total_debt_repaid",
        "aliases": set(),
        "dataitemid": 2166,
        "spgi_name": "Total Debt Repaid",
        "description": "Aggregate debt repayments during the period.",
    },
    {
        "name": "issuance_of_common_stock",
        "aliases": set(),
        "dataitemid": 2169,
        "spgi_name": "Issuance of Common Stock",
        "description": "Cash inflows from issuing common stock.",
    },
    {
        "name": "repurchase_of_common_stock",
        "aliases": set(),
        "dataitemid": 2164,
        "spgi_name": "Repurchase of Common Stock",
        "description": "Cash outflows for repurchasing common stock.",
    },
    {
        "name": "issuance_of_preferred_stock",
        "aliases": set(),
        "dataitemid": 2181,
        "spgi_name": "Issuance of Preferred Stock",
        "description": "Cash inflows from issuing preferred stock.",
    },
    {
        "name": "repurchase_of_preferred_stock",
        "aliases": set(),
        "dataitemid": 2172,
        "spgi_name": "Repurchase of Preferred Stock",
        "description": "Cash outflows for repurchasing preferred stock.",
    },
    {
        "name": "common_dividends_paid",
        "aliases": set(),
        "dataitemid": 2074,
        "spgi_name": "Common Dividends Paid",
        "description": "Cash dividends paid to common shareholders.",
    },
    {
        "name": "preferred_dividends_paid",
        "aliases": set(),
        "dataitemid": 2116,
        "spgi_name": "Pref. Dividends Paid",
        "description": "Dividends paid to preferred shareholders.",
    },
    {
        "name": "total_dividends_paid",
        "aliases": {
            "dividends_paid",
        },
        "dataitemid": 2022,
        "spgi_name": "Total Dividends Paid",
        "description": "Total dividends paid during the period.",
    },
    {
        "name": "special_dividends_paid",
        "aliases": set(),
        "dataitemid": 2041,
        "spgi_name": "Special Dividend Paid",
        "description": "Cash outflows for special or one-time dividends.",
    },
    {
        "name": "other_financing_activities",
        "aliases": set(),
        "dataitemid": 2050,
        "spgi_name": "Other Financing Activities",
        "description": "Miscellaneous financing cash flow activities.",
    },
    {
        "name": "cash_from_financing",
        "aliases": {
            "cash_from_financing_activities",
            "cashflow_from_financing",
            "cashflow_from_financing_activities",
        },
        "dataitemid": 2004,
        "spgi_name": "Cash from Financing",
        "description": "Cash provided by financing activities.",
    },
    {
        "name": "foreign_exchange_rate_adjustments",
        "aliases": {
            "fx_adjustments",
            "foreign_exchange_adjustments",
        },
        "dataitemid": 2144,
        "spgi_name": "Foreign Exchange Rate Adj.",
        "description": "Adjustments from changes in foreign exchange rates.",
    },
    {
        "name": "miscellaneous_cash_flow_adjustments",
        "aliases": {
            "misc_cash_flow_adj",
        },
        "dataitemid": 2149,
        "spgi_name": "Misc. Cash Flow Adj.",
        "description": "Miscellaneous adjustments affecting total cash flow.",
    },
    {
        "name": "net_change_in_cash",
        "aliases": {
            "change_in_cash",
        },
        "dataitemid": 2093,
        "spgi_name": "Net Change in Cash",
        "description": "Net change in cash during the reporting period.",
    },
    {
        "name": "depreciation",
        "aliases": set(),
        "dataitemid": 2143,
        "spgi_name": "Depreciation (From Notes)",
        "description": "Depreciation expense recognized during the period.",
    },
    {
        "name": "depreciation_of_rental_assets",
        "aliases": set(),
        "dataitemid": 42409,
        "spgi_name": "Depreciation of Rental Assets",
        "description": "Depreciation charge specific to rental assets.",
    },
    {
        "name": "sale_proceeds_from_rental_assets",
        "aliases": set(),
        "dataitemid": 42411,
        "spgi_name": "Sale Proceeds from Rental Assets",
        "description": "Cash inflows from selling rental assets.",
    },
    {
        "name": "basic_eps",
        "aliases": {
            "basic_earning_per_share",
            "basic_eps_including_extra_items",
            "basic_earning_per_share_including_extra_items",
        },
        "dataitemid": 9,
        "spgi_name": "Basic EPS",
        "description": "Basic earnings per share; use sparingly when explicitly requested.",
    },
    {
        "name": "basic_eps_excluding_extra_items",
        "aliases": {
            "basic_earning_per_share_excluding_extra_items",
        },
        "dataitemid": 3064,
        "spgi_name": "Basic EPS Excl. Extra Items",
        "description": "Basic EPS calculated excluding extraordinary items.",
    },
    {
        "name": "basic_eps_from_accounting_change",
        "aliases": {
            "basic_earning_per_share_from_accounting_change",
        },
        "dataitemid": 145,
        "spgi_name": "Basic EPS - Accounting Change",
        "description": "",
    },
    {
        "name": "basic_eps_from_extraordinary_items",
        "aliases": {
            "basic_earning_per_share_from_extraordinary_items",
        },
        "dataitemid": 146,
        "spgi_name": "Basic EPS - Extraordinary Items",
        "description": "Basic EPS attributable solely to extraordinary items.",
    },
    {
        "name": "basic_eps_from_accounting_change_and_extraordinary_items",
        "aliases": {
            "basic_earning_per_share_from_accounting_change_and_extraordinary_items",
        },
        "dataitemid": 45,
        "spgi_name": "Basic EPS - Extraordinary Items & Accounting Change",
        "description": "Basic EPS from accounting changes and extraordinary items combined.",
    },
    {
        "name": "weighted_average_basic_shares_outstanding",
        "aliases": set(),
        "dataitemid": 3217,
        "spgi_name": "Weighted Avg. Basic Shares Out.",
        "description": "Weighted average basic shares used in EPS calculations.",
    },
    {
        "name": "diluted_eps",
        "aliases": {
            "diluted_earning_per_share",
            "diluted_eps_including_extra_items",
            "diluted_earning_per_share_including_extra_items",
        },
        "dataitemid": 8,
        "spgi_name": "Diluted EPS",
        "description": "Diluted earnings per share measure.",
    },
    {
        "name": "diluted_eps_excluding_extra_items",
        "aliases": {
            "diluted_earning_per_share_excluding_extra_items",
        },
        "dataitemid": 142,
        "spgi_name": "Diluted EPS Excl. Extra Items",
        "description": "Diluted EPS calculated excluding extraordinary items.",
    },
    {
        "name": "weighted_average_diluted_shares_outstanding",
        "aliases": set(),
        "dataitemid": 342,
        "spgi_name": "Weighted Avg. Diluted Shares Out.",
        "description": "Weighted average diluted shares used in EPS calculations.",
    },
    {
        "name": "normalized_basic_eps",
        "aliases": {
            "normalized_basic_earning_per_share",
        },
        "dataitemid": 4379,
        "spgi_name": "Normalized Basic EPS",
        "description": "Normalized basic earnings per share metric.",
    },
    {
        "name": "normalized_diluted_eps",
        "aliases": {
            "normalized_diluted_earning_per_share",
        },
        "dataitemid": 4380,
        "spgi_name": "Normalized Diluted EPS",
        "description": "Normalized diluted earnings per share metric.",
    },
    {
        "name": "dividends_per_share",
        "aliases": set(),
        "dataitemid": 3058,
        "spgi_name": "Dividends per share",
        "description": "Cash dividends declared per share.",
    },
    {
        "name": "distributable_cash_per_share",
        "aliases": set(),
        "dataitemid": 23317,
        "spgi_name": "Distributable Cash per Share",
        "description": "Distributable cash available per share.",
    },
    {
        "name": "diluted_eps_from_accounting_change_and_extraordinary_items",
        "aliases": {
            "diluted_earning_per_share_from_accounting_change_and_extraordinary_items",
        },
        "dataitemid": 44,
        "spgi_name": "Diluted EPS - Extraordinary Items & Accounting Change",
        "description": "Diluted EPS attributable to accounting changes and extraordinary items.",
    },
    {
        "name": "diluted_eps_from_accounting_change",
        "aliases": {
            "diluted_earning_per_share_from_accounting_change",
        },
        "dataitemid": 141,
        "spgi_name": "Diluted EPS - Accounting Change",
        "description": "Diluted EPS impact from accounting changes.",
    },
    {
        "name": "diluted_eps_from_extraordinary_items",
        "aliases": {
            "diluted_earning_per_share_from_extraordinary_items",
        },
        "dataitemid": 144,
        "spgi_name": "Diluted EPS - Extraordinary Items",
        "description": "Diluted EPS attributable to extraordinary items.",
    },
    {
        "name": "diluted_eps_from_discontinued_operations",
        "aliases": {
            "diluted_earning_per_share_from_discontinued_operations",
        },
        "dataitemid": 143,
        "spgi_name": "Diluted EPS - Discontinued Operations",
        "description": "Diluted EPS attributable to discontinued operations.",
    },
    {
        "name": "funds_from_operations",
        "aliases": {
            "ffo",
        },
        "dataitemid": 3074,
        "spgi_name": "FFO",
        "description": "Funds from operations metric (often for REITs).",
    },
    {
        "name": "ebitda",
        "aliases": {
            "earnings_before_interest_taxes_depreciation_and_amortization",
        },
        "dataitemid": 4051,
        "spgi_name": "EBITDA",
        "description": "Earnings before interest, taxes, depreciation, and amortization.",
    },
    {
        "name": "ebita",
        "aliases": {
            "earnings_before_interest_taxes_and_amortization",
        },
        "dataitemid": 100689,
        "spgi_name": "EBITA",
        "description": "Earnings before interest, taxes, and amortization.",
    },
    {
        "name": "ebit",
        "aliases": {
            "earnings_before_interest_and_taxes",
        },
        "dataitemid": 400,
        "spgi_name": "EBIT",
        "description": "Earnings before interest and taxes.",
    },
    {
        "name": "ebitdar",
        "aliases": {
            "earnings_before_interest_taxes_depreciation_amortization_and_rental_expense",
        },
        "dataitemid": 21674,
        "spgi_name": "EBITDAR",
        "description": "Earnings before interest, taxes, depreciation, amortization, and rent.",
    },
    {
        "name": "net_debt",
        "aliases": set(),
        "dataitemid": 4364,
        "spgi_name": "Net Debt",
        "description": "Net debt calculated as total debt minus cash and cash equivalents.",
    },
    {
        "name": "effective_tax_rate",
        "aliases": {
            "tax_rate",
        },
        "dataitemid": 4376,
        "spgi_name": "Effective Tax Rate %",
        "description": "Effective tax rate metric for the period.",
    },
    {
        "name": "current_ratio",
        "aliases": set(),
        "dataitemid": 4030,
        "spgi_name": "Current Ratio",
        "description": "Current assets divided by current liabilities.",
    },
    {
        "name": "quick_ratio",
        "aliases": set(),
        "dataitemid": 4121,
        "spgi_name": "Quick Ratio",
        "description": "Quick assets divided by current liabilities.",
    },
    {
        "name": "total_debt_to_capital",
        "aliases": set(),
        "dataitemid": 43907,
        "spgi_name": "Total Debt to Capital (%)",
        "description": "Total debt divided by total capital (debt plus equity).",
    },
    {
        "name": "net_working_capital",
        "aliases": set(),
        "dataitemid": 1311,
        "spgi_name": "Net Working Capital",
        "description": "Net working capital (current assets minus current liabilities).",
    },
    {
        "name": "working_capital",
        "aliases": set(),
        "dataitemid": 4165,
        "spgi_name": "Working Capital",
        "description": "Alternate label for net working capital.",
    },
    {
        "name": "change_in_net_working_capital",
        "aliases": set(),
        "dataitemid": 4421,
        "spgi_name": "Change In Net Working Capital",
        "description": "Change in net working capital over the period.",
    },
    {
        "name": "total_debt",
        "aliases": set(),
        "dataitemid": 4173,
        "spgi_name": "Total Debt",
        "description": "Total debt outstanding.",
    },
    {
        "name": "total_debt_to_equity_ratio",
        "aliases": {
            "debt_ratio",
            "total_debt_ratio",
            "total_debt_to_total_equity",
            "total_debt_to_equity",
        },
        "dataitemid": 4034,
        "spgi_name": "Total Debt/Equity",
        "description": "Total debt divided by total equity.",
    },
]
LINE_ITEM_NAMES_AND_ALIASES: list[str] = list(
    chain(*[[line_item["name"]] + list(line_item["aliases"]) for line_item in LINE_ITEMS])
)


def _get_line_item_to_descriptions_map() -> dict[str, str]:
    """Build line item to descriptions mapping from LINE_ITEMS data structure."""
    descriptors = {}

    for item in LINE_ITEMS:
        name = item["name"]
        description = item.get("description", "")
        if description:  # Only include items with descriptions
            descriptors[name] = description

        # Also include aliases with the same description
        for alias in item["aliases"]:
            if description:
                descriptors[alias] = description

    return descriptors


# Pre-computed constant to avoid recreating on each validation error
LINE_ITEM_TO_DESCRIPTIONS_MAP: dict[str, str] = _get_line_item_to_descriptions_map()
