from __future__ import annotations

import json
import logging
from typing import TYPE_CHECKING, Any

from pingsafe_cli.psgraph.common.pingsafe.integration_features.base_integration_feature import BaseIntegrationFeature
from pingsafe_cli.psgraph.common.pingsafe.platform_integration import bc_integration
from pingsafe_cli.psgraph.policies_3d.checks_parser import Policy3dParser
from pingsafe_cli.psgraph.policies_3d.runner import Policy3dRunner

if TYPE_CHECKING:
    from pingsafe_cli.psgraph.common.pingsafe.platform_integration import BcPlatformIntegration
    from pingsafe_cli.psgraph.common.output.report import Report
    from pingsafe_cli.psgraph.common.typing import _BaseRunner


class Policies3DIntegration(BaseIntegrationFeature):
    def __init__(self, bc_integration: BcPlatformIntegration) -> None:
        super().__init__(bc_integration=bc_integration, order=11)
        self.platform_policy_parser = Policy3dParser()

    def is_valid(self) -> bool:
        return (
            self.bc_integration.is_integration_configured()
            and not self.bc_integration.skip_download
            and not self.integration_feature_failures
        )

    def should_run_image_referencer(self) -> bool:
        return True

    def pre_scan(self) -> None:
        # not used
        pass

    def pre_runner(self, runner: _BaseRunner) -> None:
        # not used
        pass

    def post_runner(self, scan_report: Report) -> None:
        # not used
        pass

    @staticmethod
    def _convert_raw_check(policy: dict[str, Any]) -> dict[str, Any]:
        metadata = {
            'id': policy['id'],
            'name': policy['title'],
            'category': policy['category'],
            'guideline': policy['guideline'],
            'severity': policy['severity']
        }
        check = {
            'metadata': metadata,
            'definition': json.loads(policy['codescanner'])
        }
        return check

    def post_scan(self, scan_reports: list[Report]) -> Report | None:
        try:
            if not self.bc_integration.customer_run_config_response:
                logging.debug('In the post scan for 3d policies, but nothing was fetched from the platform')
                self.integration_feature_failures = True
                return None

            policies = self.bc_integration.customer_run_config_response.get('Policies3D')
            logging.debug(f'Got {len(policies)} 3d policies from the platform.')
            if not policies:
                return None

            runner = Policy3dRunner()
            report = runner.run_v2(raw_checks=policies, scan_reports=scan_reports)
            return report

        except Exception as e:
            self.integration_feature_failures = True
            logging.debug(f'Scanning without applying 3d policies from the platform.\n{e}')
            return None


integration = Policies3DIntegration(bc_integration)
