# Test RAFT member functionality
# 
import pytest
import numpy as np
from numpy.testing import assert_allclose
import yaml
import raft
from raft.raft_member import Member
from raft.helpers import getFromDict
import os
import matplotlib.pyplot as plt


'''
 Define files for testing
'''
# Name of the subfolder where the test data is located
test_dir = 'test_data'

# List of input file names to be tested
list_files = [
    'mem_srf_vert_circ_cyl.yaml',    
    'mem_srf_vert_rect_cyl.yaml',
    'mem_srf_pitch_circ_cyl.yaml',
    'mem_srf_pitch_rect_cyl.yaml',
    'mem_srf_inc_circ_cyl.yaml',
    'mem_srf_inc_rect_cyl.yaml',
    'mem_subm_horz_circ_cyl.yaml',
    'mem_subm_horz_rect_cyl.yaml',
    'mem_srf_vert_tap_circ_cyl.yaml',
    'mem_srf_vert_tap_rect_cyl.yaml',
]

# Get the directory of the current file
current_dir = os.path.dirname(os.path.abspath(__file__))

# To avoid problems with different platforms, get the full path of the file
list_files = [os.path.join(current_dir, test_dir, file) for file in list_files]


'''
 Desired values to compare with the results.
 Should be lists of the same length as list_files.
 List elements are indicated below.
 Should probably reduce the number of decimal places in the desired values.
 For example, it doesn't matter if the CG is off by one micrometer.
'''

# Basic inertia properties
# Format is: (mshell, mfill, COGx, COGy, COGz)
desired_inertiaBasic = [
    [4.22944765e+05, 2.310786205e+06,   0       ,  0       , -14.67635],
    [4.32731250e+05, 7.794637500e+05,   0       ,  0       , -12.88153],
    [4.22944765e+05, 2.310786205e+06,  -7.338177,  0       , -12.71057],
    [4.32732437e+05, 7.794658883e+05,  -6.440766,  0       , -11.15620],
    [4.22944707e+05, 2.310785885e+06,  -9.172721,  3.177500, -11.00727],
    [4.32731250e+05, 7.794637500e+05,  -8.050958,  2.788911,  -9.66115],
    [4.22944765e+05, 2.310786204e+06, -12.176354,  0       , -20.00000],
    [4.32731250e+05, 7.794637500e+05, -10.381539,  0       , -20.00000],
    [2.96708699e+05, 2.074252114e+06, 0.00000000,  0       , -15.51393],
    [3.10357186e+05, 7.464875164e+05, 0.00000000,  0       , -14.58884],
]




# 6x6 inertia matrix wrt to (0,0,0)
# Total value (sum of shell and ballast)
desired_inertiaMatrix = [
    np.array([[  2.73373097e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -4.01212027e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.73373097e+06,  0.00000000e+00,  4.01212027e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  2.73373097e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  4.01212027e+07,  0.00000000e+00,  7.32378508e+08,  0.00000000e+00,  0.00000000e+00],
              [ -4.01212027e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  7.32378508e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.76138073e+07]]),
    np.array([[  1.21219500e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.56149297e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.21219500e+06,  0.00000000e+00,  1.56149297e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  1.21219500e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.56149297e+07,  0.00000000e+00,  3.23452553e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.56149297e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.30580956e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  1.73816357e+07]]),
    np.array([[  2.73373847e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -3.47473748e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.73373847e+06,  0.00000000e+00,  3.47473748e+07,  0.00000000e+00, -2.00606564e+07],
              [  0.00000000e+00,  0.00000000e+00,  2.73373847e+06,  0.00000000e+00,  2.00606564e+07,  0.00000000e+00],
              [  0.00000000e+00,  3.47473748e+07,  0.00000000e+00,  5.58722538e+08,  0.00000000e+00, -3.00852508e+08],
              [ -3.47473748e+07,  0.00000000e+00,  2.00606564e+07,  0.00000000e+00,  7.32414215e+08,  0.00000000e+00],
              [  0.00000000e+00, -2.00606564e+07,  0.00000000e+00, -3.00852508e+08,  0.00000000e+00,  2.11305588e+08]]),
    np.array([[  1.21219833e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.35235319e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.21219833e+06,  0.00000000e+00,  1.35235319e+07,  0.00000000e+00, -7.80748629e+06],
              [  0.00000000e+00,  0.00000000e+00,  1.21219833e+06,  0.00000000e+00,  7.80748629e+06,  0.00000000e+00],
              [  0.00000000e+00,  1.35235319e+07,  0.00000000e+00,  2.46948823e+08,  0.00000000e+00, -1.32536819e+08],
              [ -1.35235319e+07,  0.00000000e+00,  7.80748629e+06,  0.00000000e+00,  3.30595201e+08,  0.00000000e+00],
              [  0.00000000e+00, -7.80748629e+06,  0.00000000e+00, -1.32536819e+08,  0.00000000e+00,  9.38996396e+07]]),
    np.array([[  2.73373059e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -3.00908979e+07, -8.68642940e+06],
              [  0.00000000e+00,  2.73373059e+06,  0.00000000e+00,  3.00908979e+07,  0.00000000e+00, -2.50757482e+07],
              [  0.00000000e+00,  0.00000000e+00,  2.73373059e+06,  8.68642940e+06,  2.50757482e+07,  0.00000000e+00],
              [  0.00000000e+00,  3.00908979e+07,  8.68642940e+06,  4.60985540e+08,  9.40124580e+07, -3.25670906e+08],
              [ -3.00908979e+07,  0.00000000e+00,  2.50757482e+07,  9.40124580e+07,  6.99811311e+08,  1.12814950e+08],
              [ -8.68642940e+06, -2.50757482e+07,  0.00000000e+00, -3.25670906e+08,  1.12814950e+08,  3.41572875e+08]]),
    np.array([[  1.21219483e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.17111957e+07, -3.38070373e+06],
              [  0.00000000e+00,  1.21219483e+06,  0.00000000e+00,  1.17111957e+07,  0.00000000e+00, -9.75932974e+06],
              [  0.00000000e+00,  0.00000000e+00,  1.21219483e+06,  3.38070373e+06,  9.75932974e+06,  0.00000000e+00],
              [  0.00000000e+00,  1.17111957e+07,  3.38070373e+06,  2.04657143e+08,  4.36209409e+07, -1.43470721e+08],
              [ -1.17111957e+07,  0.00000000e+00,  9.75932974e+06,  4.36209409e+07,  3.15470081e+08,  4.96994023e+07],
              [ -3.38070373e+06, -9.75932974e+06,  0.00000000e+00, -1.43470721e+08,  4.96994023e+07,  1.51287459e+08]]),
    np.array([[  2.73373097e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -5.46746194e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.73373097e+06,  0.00000000e+00,  5.46746194e+07,  0.00000000e+00, -3.32868753e+07],
              [  0.00000000e+00,  0.00000000e+00,  2.73373097e+06,  0.00000000e+00,  3.32868753e+07,  0.00000000e+00],
              [  0.00000000e+00,  5.46746194e+07,  0.00000000e+00,  1.13110620e+09,  0.00000000e+00, -6.65737506e+08],
              [ -5.46746194e+07,  0.00000000e+00,  3.32868753e+07,  0.00000000e+00,  1.64235070e+09,  0.00000000e+00],
              [  0.00000000e+00, -3.32868753e+07,  0.00000000e+00, -6.65737506e+08,  0.00000000e+00,  5.48858313e+08]]),
    np.array([[  1.21219500e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -2.42439000e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.21219500e+06,  0.00000000e+00,  2.42439000e+07,  0.00000000e+00, -1.25844422e+07],
              [  0.00000000e+00,  0.00000000e+00,  1.21219500e+06,  0.00000000e+00,  1.25844422e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.42439000e+07,  0.00000000e+00,  5.02259636e+08,  0.00000000e+00, -2.51688845e+08],
              [ -2.42439000e+07,  0.00000000e+00,  1.25844422e+07,  0.00000000e+00,  7.44960526e+08,  0.00000000e+00],
              [  0.00000000e+00, -1.25844422e+07,  0.00000000e+00, -2.51688845e+08,  0.00000000e+00,  2.52954123e+08]]),
    np.array([[  2.37096081e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -3.67829311e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.37096081e+06,  0.00000000e+00,  3.67829311e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  2.37096081e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  3.67829311e+07,  0.00000000e+00,  6.56510039e+08,  0.00000000e+00,  0.00000000e+00],
              [ -3.67829311e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  6.56510039e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  2.63582481e+07]]),
    np.array([[  1.05684470e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.54181373e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.05684470e+06,  0.00000000e+00,  1.54181373e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  1.05684470e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.54181373e+07,  0.00000000e+00,  3.01375065e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.54181373e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.06203665e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  1.15631538e+07]]),
]


# Some hydrostatic quantities
# Fvec[2], Fvec[3], Fvec[4], Cmat[2,2], Cmat[3,3], Cmat[4,4], r_center[0], r_center[1], r_center[2], xWP, yWP
# Other hydrostatic values, such as AWP and IWP, are directly included in the others
desired_hydrostatics = [
    [1.53244611e+07, 0.00000000e+00,  0.00000000e+00, 7.6622305e+05, -1.48598306e+08, -1.48598306e+08,  0.00000000, 0.00000000, -10.000000, 0       , 0            ],
    [1.18853055e+07, 0.00000000e+00,  0.00000000e+00, 5.9426527e+05, -1.17070259e+08, -1.14048296e+08,  0.00000000, 0.00000000, -10.000000, 0       , 0            ],
    [1.53248823e+07, 0.00000000e+00,  1.49977968e+08, 8.8475737e+05, -1.28074838e+08, -1.28074838e+08, -4.99987628, 0.00000000,  -8.660500, 2.47e-04, 0            ],
    [1.18856322e+07, 0.00000000e+00,  1.18856322e+08, 6.8619781e+05, -1.01598418e+08, -9.81307590e+07, -4.99987628, 0.00000000,  -8.660500, 2.47e-04, 0            ],
    [1.53244590e+07, 6.49309773e+07,  1.87440769e+08, 1.0216306e+06, -1.10287138e+08, -1.10287138e+08, -6.25000000, 2.16504286,  -7.500000, 0       , -1.428571e-05],
    [1.18853038e+07, 5.14645542e+07,  1.48566298e+08, 7.9235359e+05, -8.77296079e+07, -8.47423685e+07, -6.25000000, 2.16504285,  -7.500000, 0       , -1.428571e-05],
    [2.68178069e+07, 0.00000000e+00,  0.00000000e+00, 0.0000000e+00, -5.36356138e+08, -5.36356138e+08,  0.00000000, 0.00000000, -20.000000, 0       ,  0           ],
    [2.07992846e+07, 0.00000000e+00,  0.00000000e+00, 0.0000000e+00, -4.15985692e+08, -4.15985692e+08,  0.00000000, 0.00000000, -20.000000, 0       ,  0           ],
    [1.17548315e+07, 0.00000000e+00,  0.00000000e+00, 4.2580711e+05, -1.27460608e+08, -1.27460608e+08,  0.00000000, 0.00000000, -10.965322, 0       ,  0           ],
    [9.31432035e+06, 0.00000000e+00,  0.00000000e+00, 3.4807582e+05, -1.00704868e+08, -9.97855691e+07,  0.00000000, 0.00000000, -10.881043, 0       ,  0           ],
]


# Hydrodynamic added mass matrix
# Because those are analytical values, the test cases need to have a very fine discretization (dlsMax) otherwise the moments won't match
desired_Ahydro = [
    np.array([[  1.32780754e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.32780754e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.32780754e+06,  0.00000000e+00,  1.32780754e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  1.79514370e+05,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.32780754e+07,  0.00000000e+00,  1.77041005e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.32780754e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  1.77041005e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  1.81732500e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.81732500e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.51443750e+06,  0.00000000e+00,  1.51443750e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  9.34949911e+04,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.51443750e+07,  0.00000000e+00,  2.01925000e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.81732500e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  2.42310000e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  1.04071238e+06,  0.00000000e+00, -4.97211082e+05,  0.00000000e+00, -1.14996000e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.32777628e+06,  0.00000000e+00,  1.14997230e+07,  0.00000000e+00, -6.63901535e+06],
              [ -4.97211082e+05,  0.00000000e+00,  4.66578274e+05,  0.00000000e+00,  6.63922844e+06,  0.00000000e+00],
              [  0.00000000e+00,  1.14997230e+07,  0.00000000e+00,  1.32791120e+08,  0.00000000e+00, -7.66638631e+07],
              [ -1.14996000e+07,  0.00000000e+00,  6.63922844e+06,  0.00000000e+00,  1.77051217e+08,  0.00000000e+00],
              [  0.00000000e+00, -6.63901535e+06,  0.00000000e+00, -7.66638631e+07,  0.00000000e+00,  4.42600974e+07]]),
    np.array([[  1.38633778e+06,  0.00000000e+00, -7.46420401e+05,  0.00000000e+00, -1.57390950e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.51440185e+06,  0.00000000e+00,  1.31160664e+07,  0.00000000e+00, -7.57216203e+06],
              [ -7.46420401e+05,  0.00000000e+00,  5.24439434e+05,  0.00000000e+00,  9.08691434e+06,  0.00000000e+00],
              [  0.00000000e+00,  1.31160664e+07,  0.00000000e+00,  1.51455574e+08,  0.00000000e+00, -8.74393510e+07],
              [ -1.57390950e+07,  0.00000000e+00,  9.08691434e+06,  0.00000000e+00,  2.42323978e+08,  0.00000000e+00],
              [  0.00000000e+00, -7.57216203e+06,  0.00000000e+00, -8.74393510e+07,  0.00000000e+00,  5.04810746e+07]]),
    np.array([[  8.78749576e+05,  1.55270242e+05, -5.37873477e+05,  7.68390682e+00, -9.95852403e+06, -2.87475726e+06],
              [  1.55270242e+05,  1.27319044e+06,  1.86324290e+05,  9.95852136e+06, -3.55275059e-10, -8.29876780e+06],
              [ -5.37873477e+05,  1.86324290e+05,  6.81529301e+05,  2.87476007e+06,  8.29877002e+06, -7.68390682e+00],
              [  7.68390682e+00,  9.95852136e+06,  2.87476007e+06,  1.07883541e+08,  2.39562214e+07, -8.29874399e+07],
              [ -9.95852403e+06, -3.55275059e-10,  8.29877002e+06,  2.39562214e+07,  1.68741128e+08,  2.87474657e+07],
              [ -2.87475726e+06, -8.29876780e+06, -7.68390682e+00, -8.29874399e+07,  2.87474657e+07,  7.74548130e+07]]),
    np.array([[  1.11082965e+06,  1.39485495e+05, -8.07513003e+05, -7.02595984e+05, -1.33865034e+07, -3.27881716e+06],
              [  1.39485495e+05,  1.46517172e+06,  2.79729887e+05,  1.16016299e+07,  7.02607520e+05, -9.46520150e+06],
              [ -8.07513003e+05,  2.79729887e+05,  8.47173319e+05,  3.93458753e+06,  1.13582442e+07, -1.15359000e+01],
              [ -7.02595984e+05,  1.16016299e+07,  3.93458753e+06,  1.27374004e+08,  3.98141183e+07, -9.46517481e+07],
              [ -1.33865034e+07,  7.02607520e+05,  1.13582442e+07,  3.98141183e+07,  2.28516352e+08,  3.27880362e+07],
              [ -3.27881716e+06, -9.46520150e+06, -1.15359000e+01, -9.46517481e+07,  3.27880362e+07,  8.83414728e+07]]),
    np.array([[  3.59028744e+05,  0.00000000e+00, -1.42283334e-10,  0.00000000e+00, -7.18057487e+06,  0.00000000e+00],
              [  0.00000000e+00,  2.32366319e+06,  0.00000000e+00,  4.64732638e+07,  0.00000000e+00,  5.76255843e-09],
              [ -1.42283334e-10,  0.00000000e+00,  2.32366319e+06,  0.00000000e+00,  2.29920261e-09,  0.00000000e+00],
              [  0.00000000e+00,  4.64732638e+07,  0.00000000e+00,  9.29465275e+08,  0.00000000e+00,  1.49011612e-08],
              [ -7.18057487e+06,  0.00000000e+00,  2.29920261e-09,  0.00000000e+00,  3.80816845e+08,  0.00000000e+00],
              [  0.00000000e+00,  5.76255843e-09,  0.00000000e+00,  1.49011612e-08,  0.00000000e+00,  2.37205347e+08]]),
    np.array([[  0.00000000e+00,  0.00000000e+00, -1.94738359e-10,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  2.65026562e+06,  0.00000000e+00,  5.30053125e+07,  0.00000000e+00,  2.91038305e-11],
              [ -1.94738359e-10,  0.00000000e+00,  3.18031875e+06,  0.00000000e+00,  2.03726813e-10,  0.00000000e+00],
              [  0.00000000e+00,  5.30053125e+07,  0.00000000e+00,  1.06010625e+09,  0.00000000e+00,  4.65661287e-10],
              [  0.00000000e+00,  0.00000000e+00,  2.03726813e-10,  0.00000000e+00,  3.24654889e+08,  0.00000000e+00],
              [  0.00000000e+00,  2.91038305e-11,  0.00000000e+00,  4.65661287e-10,  0.00000000e+00,  2.70545741e+08]]),
    np.array([[  9.28158055e+05,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.05287510e+07,  0.00000000e+00],
              [  0.00000000e+00,  9.28158055e+05,  0.00000000e+00,  1.05287510e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  3.07272642e+05,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.05287510e+07,  0.00000000e+00,  1.49036850e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.05287510e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  1.49036850e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  1.30806698e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -1.46802574e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.09005582e+06,  0.00000000e+00,  1.22335478e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  2.95195567e+04,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  1.22335478e+07,  0.00000000e+00,  1.72325946e+08,  0.00000000e+00,  0.00000000e+00],
              [ -1.46802574e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  2.06791135e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
]

# Hydrodynamic inertial excitation matrices
desired_Ihydro = [
    np.array([[  2.88993405e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -2.88993405e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.88993405e+06,  0.00000000e+00,  2.88993405e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  1.79514370e+05,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  2.88993405e+07,  0.00000000e+00,  3.85324540e+08,  0.00000000e+00,  0.00000000e+00],
              [ -2.88993405e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.85324540e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  3.02887500e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -3.02887500e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.72598750e+06,  0.00000000e+00,  2.72598750e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  9.34949911e+04,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  2.72598750e+07,  0.00000000e+00,  3.63465000e+08,  0.00000000e+00,  0.00000000e+00],
              [ -3.02887500e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  4.03850000e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  2.21228183e+06,  0.00000000e+00, -1.17361454e+06,  0.00000000e+00, -2.50285185e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.88986602e+06,  0.00000000e+00,  2.50288089e+07,  0.00000000e+00, -1.44496216e+07],
              [ -1.17361454e+06,  0.00000000e+00,  8.57098566e+05,  0.00000000e+00,  1.44501246e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.50288089e+07,  0.00000000e+00,  2.89015967e+08,  0.00000000e+00, -1.66856643e+08],
              [ -2.50285185e+07,  0.00000000e+00,  1.44501246e+07,  0.00000000e+00,  3.85346767e+08,  0.00000000e+00],
              [  0.00000000e+00, -1.44496216e+07,  0.00000000e+00, -1.66856643e+08,  0.00000000e+00,  9.63308002e+07]]),
    np.array([[  2.29498055e+06,  0.00000000e+00, -1.27102363e+06,  0.00000000e+00, -2.62318182e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.72592333e+06,  0.00000000e+00,  2.36089195e+07,  0.00000000e+00, -1.36298917e+07],
              [ -1.27102363e+06,  0.00000000e+00,  8.27318142e+05,  0.00000000e+00,  1.51448688e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.36089195e+07,  0.00000000e+00,  2.72620032e+08,  0.00000000e+00, -1.57390832e+08],
              [ -2.62318182e+07,  0.00000000e+00,  1.51448688e+07,  0.00000000e+00,  4.03873296e+08,  0.00000000e+00],
              [  0.00000000e+00, -1.36298917e+07,  0.00000000e+00, -1.57390832e+08,  0.00000000e+00,  9.08659342e+07]]),
    np.array([[  1.83007517e+06,  3.66518981e+05, -1.26966273e+06,  1.81380390e+01, -2.16744346e+07, -6.25682580e+06],
              [  3.66518981e+05,  2.76116192e+06,  4.39822777e+05,  2.16744284e+07, -7.00239958e-10, -1.80620236e+07],
              [ -1.26966273e+06,  4.39822777e+05,  1.36453217e+06,  6.25683245e+06,  1.80620289e+07, -1.81380390e+01],
              [  1.81380390e+01,  2.16744284e+07,  6.25683245e+06,  2.34805354e+08,  5.21400112e+07, -1.80619722e+08],
              [ -2.16744346e+07, -7.00239958e-10,  1.80620289e+07,  5.21400112e+07,  3.67260102e+08,  6.25680135e+07],
              [ -6.25682580e+06, -1.80620236e+07, -1.81380390e+01, -1.80619722e+08,  6.25680135e+07,  1.68578122e+08]]),
    np.array([[  1.84865626e+06,  3.03325240e+05, -1.37507220e+06, -7.02587876e+05, -2.24730987e+07, -5.90187304e+06],
              [  3.03325240e+05,  2.61920875e+06,  4.76337581e+05,  2.06882224e+07,  7.02607520e+05, -1.70373620e+07],
              [ -1.37507220e+06,  4.76337581e+05,  1.37689490e+06,  6.55764639e+06,  1.89304070e+07, -1.96438885e+01],
              [ -7.02587876e+05,  2.06882224e+07,  6.55764639e+06,  2.25811692e+08,  6.16728285e+07, -1.70373138e+08],
              [ -2.24730987e+07,  7.02607520e+05,  1.89304070e+07,  6.16728285e+07,  3.82483179e+08,  5.90184883e+07],
              [ -5.90187304e+06, -1.70373620e+07, -1.96438885e+01, -1.70373138e+08,  5.90184883e+07,  1.59014651e+08]]),
    np.array([[  3.59028744e+05,  0.00000000e+00, -3.09675492e-10,  0.00000000e+00, -7.18057487e+06,  0.00000000e+00],
              [  0.00000000e+00,  5.05738459e+06,  0.00000000e+00,  1.01147692e+08,  0.00000000e+00, -1.25146471e-09],
              [ -3.09675492e-10,  0.00000000e+00,  5.05738459e+06,  0.00000000e+00,  1.30967237e-09,  0.00000000e+00],
              [  0.00000000e+00,  1.01147692e+08,  0.00000000e+00,  2.02295383e+09,  0.00000000e+00,  1.27591193e-07],
              [ -7.18057487e+06,  0.00000000e+00,  1.30967237e-09,  0.00000000e+00,  6.59881959e+08,  0.00000000e+00],
              [  0.00000000e+00, -1.25146471e-09,  0.00000000e+00,  2.00234354e-07,  0.00000000e+00,  5.16270462e+08]]),
    np.array([[  0.00000000e+00,  0.00000000e+00, -3.24563931e-10,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  4.77047812e+06,  0.00000000e+00,  9.54095625e+07,  0.00000000e+00, -1.51339918e-09],
              [ -3.24563931e-10,  0.00000000e+00,  5.30053125e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  9.54095625e+07,  0.00000000e+00,  1.90819125e+09,  0.00000000e+00,  9.31322575e-10],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  5.41091481e+08,  0.00000000e+00],
              [  0.00000000e+00, -1.51339918e-09,  0.00000000e+00,  9.31322575e-10,  0.00000000e+00,  4.86982333e+08]]),
    np.array([[  2.02010871e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -2.29155170e+07,  0.00000000e+00],
              [  0.00000000e+00,  2.02010871e+06,  0.00000000e+00,  2.29155170e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  3.07272642e+05,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  2.29155170e+07,  0.00000000e+00,  3.24374320e+08,  0.00000000e+00,  0.00000000e+00],
              [ -2.29155170e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.24374320e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
    np.array([[  2.18011163e+06,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00, -2.44670956e+07,  0.00000000e+00],
              [  0.00000000e+00,  1.96210047e+06,  0.00000000e+00,  2.20203860e+07,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  2.95195567e+04,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00],
              [  0.00000000e+00,  2.20203860e+07,  0.00000000e+00,  3.10186702e+08,  0.00000000e+00,  0.00000000e+00],
              [ -2.44670956e+07,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  3.44651891e+08,  0.00000000e+00],
              [  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00,  0.00000000e+00]]),
]


'''
 Aux functions
'''
# Function used to create member
# Not explicitly inside the fixture below so that we can also run this file as a script 
def create_member(file):
    with open(file) as f:
        design = yaml.load(f, Loader=yaml.FullLoader)

    if len(design['members']) != 1:
        raise ValueError(f'{file} should have only one member.')

    memData = design['members'][0]
    headings = getFromDict(memData, 'heading', shape=-1, default=0.)
    memData['headings'] = headings   # differentiating the list of headings/copies of a member from the individual member's heading
    
    # create member object
    if np.isscalar(headings):
        member = Member(memData, 0, heading=headings)
    else:
        raise ValueError(f'Cannot have more than one heading. Need a single member.')
    
    member.setPosition() # Set the initial position
    return member

# Define a fixture to loop member instances with the index to loop the desired values as well
# Could also zip the lists with the desired values, but I think the approach below is simpler
@pytest.fixture(params=enumerate(list_files))
def index_and_member(request):
    index, file = request.param
    member = create_member(file)
    return index, member


'''
 Test functions
'''
def test_inertia(index_and_member):
    index, member = index_and_member
    mass, cg, mshell, mfill, pfill = member.getInertia()
    assert_allclose([mshell, mfill[0], cg[0], cg[1], cg[2]], desired_inertiaBasic[index], rtol=1e-05, atol=1e-5, verbose=True)
    assert_allclose(member.M_struc, desired_inertiaMatrix[index], rtol=1e-05, atol=0, verbose=True)

def test_hydrostatics(index_and_member):
    index, member = index_and_member
    Fvec, Cmat, _, r_center, _, _, xWP, yWP = member.getHydrostatics(rho=1025, g=9.81)
    assert_allclose([Fvec[2], Fvec[3], Fvec[4], Cmat[2,2], Cmat[3,3], Cmat[4,4], r_center[0], r_center[1], r_center[2], xWP, yWP], desired_hydrostatics[index], rtol=1e-05, atol=1e-5, verbose=True)

def test_hydroConstants(index_and_member):
    index, member = index_and_member
    A_hydro, I_hydro = member.calcHydroConstants(sum_inertia=True, rho=1025, g=9.81)
    assert_allclose(A_hydro, desired_Ahydro[index], rtol=1e-05, atol=1e-7, verbose=True)
    assert_allclose(I_hydro, desired_Ihydro[index], rtol=1e-05, atol=1e-7, verbose=True)


'''
 To run as a script. Useful for debugging.
'''
if __name__ == "__main__":
    index = 9
    member = create_member(list_files[index])

    test_inertia((index, member))
    test_hydrostatics((index, member))
    test_hydroConstants((index, member))

    # This is what I used to generate the cases for inclined members
    # member.setPosition(r6=[0,0,0, 0, np.deg2rad(30), 0])
    # print(member.rA)
    # print(member.rB)

    # Plot member
    # fig = plt.figure()
    # ax = fig.add_subplot(111, projection='3d')
    # member.plot(ax, plot2d=False)
    # plt.show()


    