import unittest

import numpy.testing as npt
import numpy as np
import openmdao.api as om
from raft.omdao_raft import RAFT_Group


class TestSpider(unittest.TestCase):
    def testSpiderRegression(self):

        opt = {}

        opt['modeling'] = {}

        opt['modeling']['flag'] = True
        opt['modeling']['potential_model_override'] = 0
        opt['modeling']['potential_bem_members'] = ['spar']
        opt['modeling']['min_freq_BEM'] = 0.0999999
        opt['modeling']['min_freq'] = 0.1
        opt['modeling']['max_freq'] = 1.0
        opt['modeling']['xi_start'] = 0.0
        opt['modeling']['nIter'] = 15
        opt['modeling']['dls_max'] = 5
        opt['modeling']['trim_ballast'] = 0
        opt['modeling']['heave_tol'] = 1
        opt['modeling']['save_designs'] = False
        opt['modeling']['plot_designs'] = False
        opt['modeling']['runPyHAMS'] = True
        opt['modeling']['BEM_dir'] = 'none/BEM'
        opt['modeling']['model_potential'] = [True,False, False,False]
        opt['modeling']['nfreq'] = 10
        opt['modeling']['n_cases'] = 2

        opt['turbine'] = {}
        opt['turbine']['npts'] = 11
        opt['turbine']['scalar_thicknesses'] = False
        opt['turbine']['scalar_diameters'] = False
        opt['turbine']['scalar_coefficients'] = False
        opt['turbine']['shape'] = 'circ'
        opt['turbine']['PC_GS_n'] = 30
        opt['turbine']['n_span'] = 30
        opt['turbine']['n_aoa'] = 200
        opt['turbine']['n_Re'] = 1
        opt['turbine']['n_tab'] = 1
        opt['turbine']['n_pc'] = 20
        opt['turbine']['n_af'] = 7
        opt['turbine']['af_used_names'] = ['Cylinder', 'Cylinder', 'DU40_A17', 'DU35_A17', 'DU30_A17', 'DU25_A17', 'DU21_A17', 'NACA64_A17', 'NACA64_A17']


        opt['members'] = {}
        opt['members']['nmembers'] = 4
        opt['members']['npts'] = [6, 2, 2, 2]
        opt['members']['npts_lfill'] = [5, 1, 1, 1]
        opt['members']['npts_rho_fill'] = [5, 1, 1, 1]
        opt['members']['ncaps'] = [5, 2, 2, 2]
        opt['members']['nreps'] = [0, 0, 0, 0]
        opt['members']['shape'] = ['circ', 'circ', 'circ', 'circ']
        opt['members']['scalar_thicknesses'] = [False, False, False, False]
        opt['members']['scalar_diameters'] = [False, False, False, False]
        opt['members']['scalar_coefficients'] = [False, False, False, False]
        opt['members']['n_ballast_type'] = 2

        opt['mooring'] = {}
        opt['mooring']['nlines'] = 3
        opt['mooring']['nline_types'] = 1
        opt['mooring']['nconnections'] = 6

        opt['analysis'] = {}
        opt['analysis']['general'] = {}
        opt['analysis']['general']['folder_output'] = 'output'
        opt['analysis']['general']['fname_output'] = 'spiderfloat'

        prob = om.Problem()
        prob.model = RAFT_Group(modeling_options=opt['modeling'],
                                analysis_options=opt['analysis'],
                                turbine_options=opt['turbine'],
                                mooring_options=opt['mooring'],
                                member_options=opt['members'])
        prob.setup()

        # -------------------------
        # inputs
        # -------------------------
        prob['Fl_Kp'] = [0.0]
        prob['shear_exp'] = [0.11]
        prob['wind_reference_height'] = [90.0]
        prob['nBlades'] = 3
        prob['airfoils_name'] = ['DU40_A17', 'Cylinder', 'DU30_A17', 'DU21_A17', 'DU25_A17', 'DU35_A17', 'NACA64_A17']
        opt['modeling']['raft_dlcs'] = [[14.0, 0.0, 'IB_NTM', 'operating', 0, 'JONSWAP', 1.0, 7.0, 0.0], [16.0, 0.0, 'IB_NTM', 'operating', 0, 'JONSWAP', 2.0, 8.0, 0.0]]
        opt['modeling']['raft_dlcs_keys'] = ['wind_speed', 'wind_heading', 'turbulence',
                                              'turbine_status', 'yaw_misalign', 'wave_spectrum',
                                              'wave_period', 'wave_height', 'wave_heading']

        # -------------------------
        # turbine
        # -------------------------
        prob['turbine_Fthrust'] = [1159112.2383200827]
        prob['turbine_IrRNA'] = [24901633.358042654]
        prob['turbine_IxRNA'] = [37529686.66211496]
        prob['turbine_hHub'] = [90.0]
        prob['turbine_mRNA'] = [210369.7623886403]
        prob['turbine_overhang'] = [5.0]
        prob['turbine_tower_Ca'] = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        prob['turbine_tower_CaEnd'] = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        prob['turbine_tower_Cd'] = [0.7008455614091645, 0.7016469752391513, 0.7026995693220366, 0.7038669315998577, 0.7051082306017693, 0.7063666983271379, 0.7074097851470198, 0.7081055742541656, 0.7082150447201733, 0.7073650177679025, 0.7051730147379945]
        prob['turbine_tower_CdEnd'] = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        prob['turbine_tower_d'] = [6.5, 6.24, 5.97, 5.71, 5.45, 5.18, 4.92, 4.66, 4.4, 4.13, 3.8699999999999997]
        prob['turbine_tower_gamma'] = [0.0]
        prob['turbine_tower_rA'] = [0.0, 0.0, 10.0]
        prob['turbine_tower_rB'] = [0.0, 0.0, 87.7]
        prob['turbine_tower_rho_shell'] = [7800.0]
        prob['turbine_tower_stations'] = [0.0, 0.10000000000000003, 0.19999999999999998, 0.30000000000000004, 0.39999999999999997, 0.5, 0.6000000000000001, 0.7, 0.7999999999999999, 0.9, 1.0]
        prob['turbine_tower_t'] = [0.027, 0.0262, 0.0254, 0.0246, 0.0238, 0.023, 0.0222, 0.0214, 0.0206, 0.0198, 0.019000000000000003]
        prob['turbine_xCG_RNA'] = [-2.7914555962810814]
        prob['turbine_yaw_stiffness'] = [2105054739.1200264]
        prob['airfoils_aoa'] = [-3.141592653589793, -3.0892327760299634, -3.036872898470133, -2.9845130209103035, -2.9321531433504737, -2.8797932657906435, -2.827433388230814, -2.775073510670984, -2.722713633111154, -2.670353755551324, -2.617993877991494, -2.5656340004316642, -2.5132741228718345, -2.4609142453120043, -2.4085543677521746, -2.356194490192345, -2.3038346126325147, -2.251474735072685, -2.199114857512855, -2.146754979953025, -2.0943951023931953, -2.0420352248333655, -1.9896753472735356, -1.9373154697137056, -1.8849555921538759, -1.832595714594046, -1.780235837034216, -1.727875959474386, -1.6755160819145563, -1.6231562043547263, -1.5707963267948963, -1.5184364492350666, -1.4660765716752366, -1.4137166941154067, -1.361356816555577, -1.308996938995747, -1.256637061435917, -1.2042771838760873, -1.1519173063162573, -1.0995574287564276, -1.0471975511965974, -0.9948376736367677, -0.9424777960769379, -0.8901179185171078, -0.837758040957278, -0.7853981633974483, -0.7330382858376181, -0.6806784082777884, -0.6283185307179586, -0.5759586531581284, -0.5235987755982988, -0.513021022555909, -0.502443269513519, -0.4918655164711292, -0.4812877634287393, -0.4707100103863494, -0.46013225734395957, -0.4495545043015697, -0.4389767512591798, -0.42839899821678995, -0.41782124517440006, -0.40724349213201017, -0.39666573908962033, -0.38608798604723044, -0.37551023300484054, -0.3649324799624507, -0.3543547269200608, -0.3437769738776709, -0.3331992208352811, -0.3226214677928912, -0.3120437147505013, -0.30146596170811146, -0.29088820866572157, -0.2803104556233317, -0.26973270258094184, -0.25915494953855195, -0.24857719649616206, -0.23799944345377216, -0.22742169041138233, -0.21684393736899243, -0.20626618432660254, -0.1956884312842127, -0.1851106782418228, -0.17453292519943292, -0.16395517215704308, -0.1533774191146532, -0.1427996660722633, -0.13222191302987346, -0.12164415998748357, -0.11106640694509368, -0.10048865390270378, -0.08991090086031395, -0.07933314781792405, -0.06875539477553416, -0.058177641733144325, -0.04759988869075443, -0.03702213564836454, -0.026444382605974703, -0.015866629563584866, -0.0052888765211949185, 0.0052888765211949185, 0.015866629563584866, 0.026444382605974703, 0.03702213564836454, 0.04759988869075449, 0.058177641733144325, 0.06875539477553416, 0.07933314781792411, 0.08991090086031395, 0.10048865390270378, 0.11106640694509373, 0.12164415998748357, 0.1322219130298734, 0.14279966607226335, 0.1533774191146532, 0.16395517215704303, 0.17453292519943298, 0.1851106782418228, 0.19568843128421265, 0.2062661843266026, 0.21684393736899243, 0.22742169041138227, 0.23799944345377222, 0.24857719649616206, 0.2591549495385519, 0.26973270258094184, 0.2803104556233317, 0.2908882086657215, 0.30146596170811146, 0.3120437147505013, 0.32262146779289125, 0.3331992208352811, 0.3437769738776709, 0.35435472692006087, 0.3649324799624507, 0.37551023300484054, 0.3860879860472305, 0.39666573908962033, 0.40724349213201017, 0.4178212451744001, 0.42839899821678995, 0.4389767512591798, 0.44955450430156974, 0.46013225734395957, 0.4707100103863494, 0.48128776342873925, 0.4918655164711291, 0.5024432695135191, 0.513021022555909, 0.5235987755982988, 0.5759586531581287, 0.6283185307179586, 0.6806784082777885, 0.7330382858376183, 0.7853981633974483, 0.8377580409572781, 0.890117918517108, 0.9424777960769379, 0.9948376736367679, 1.0471975511965979, 1.0995574287564276, 1.1519173063162573, 1.2042771838760875, 1.2566370614359172, 1.3089969389957472, 1.3613568165555772, 1.413716694115407, 1.4660765716752369, 1.5184364492350668, 1.5707963267948966, 1.6231562043547263, 1.6755160819145565, 1.7278759594743862, 1.780235837034216, 1.8325957145940461, 1.8849555921538759, 1.937315469713706, 1.9896753472735358, 2.0420352248333655, 2.0943951023931957, 2.1467549799530254, 2.199114857512855, 2.2514747350726854, 2.303834612632515, 2.356194490192345, 2.408554367752175, 2.4609142453120048, 2.5132741228718345, 2.5656340004316642, 2.6179938779914944, 2.670353755551324, 2.722713633111154, 2.775073510670984, 2.827433388230814, 2.8797932657906435, 2.9321531433504737, 2.9845130209103035, 3.036872898470133, 3.0892327760299634, 3.141592653589793]
        prob['airfoils_cd'] = [[0.06162547426059073, 0.06296797797883891, 0.06920073530496863, 0.10217134189398058, 0.14436773261526759, 0.19916113989223314, 0.2613159042273307, 0.3249693802145587, 0.39401562162258014, 0.46416159470315793, 0.5351486407224958, 0.6061805126538875, 0.676562470269005, 0.7459237112253301, 0.8133302599499371, 0.8784285521720816, 0.9405318277556405, 0.9989244392895479, 1.0534728181523614, 1.1036870487468315, 1.149679982103966, 1.1913010999243736, 1.228220156950608, 1.260780828331162, 1.288288296902172, 1.3110192379399626, 1.3289930576565618, 1.3411305973405871, 1.3492052255262386, 1.3494226144757384, 1.3471516103075742, 1.338310712651388, 1.3238807582607919, 1.3038752677474148, 1.2794216371157345, 1.2498660023889852, 1.2158116332882771, 1.1774606977626247, 1.1347222059598447, 1.0882014690947377, 1.0377798353713914, 0.9838046057871134, 0.9265108853719466, 0.8659307526094487, 0.8024606800514815, 0.7362176599432407, 0.6674788785888516, 0.5974768059666742, 0.526777547771337, 0.4570586090701714, 0.3889926248891426, 0.3756716562058564, 0.3624538965235285, 0.34934519264831915, 0.33635139138638853, 0.32347833954389693, 0.3107318839270045, 0.29811787134187145, 0.2856421485946579, 0.2734043403371759, 0.26147470576409987, 0.2497148331704735, 0.23817487413336985, 0.22685657368838125, 0.21577434843464335, 0.20493921686687402, 0.1943553338962619, 0.1840818276208784, 0.17413153386004743, 0.16456595524049228, 0.15542077420422765, 0.14671543660700853, 0.13852743787669108, 0.13085733453698772, 0.12377022969838634, 0.117317607574238, 0.11150629514842382, 0.10636463448608106, 0.10197779600256834, 0.09838501231621694, 0.0956025683343815, 0.09354916704252768, 0.09199067839108818, 0.09067626355958172, 0.08933502269836993, 0.08771271019266315, 0.08554237688191808, 0.08257091343530584, 0.0785302965793859, 0.07306747975920203, 0.06620585770202356, 0.05777222551161496, 0.04910720454294258, 0.04079139702759626, 0.03365694596112839, 0.027741623940545457, 0.022885308754333377, 0.01906126379690795, 0.016118958250108492, 0.013913457470965201, 0.012443774095294876, 0.011481229177436842, 0.011061771174093743, 0.010930090914738285, 0.011093865401821676, 0.011359448119156493, 0.011719776087900712, 0.011961675038045764, 0.012218116287528598, 0.012556821522520736, 0.013257847786480047, 0.01466447814710148, 0.017032929527078012, 0.020503810673543777, 0.025559846162295063, 0.03231444630372062, 0.0409124654302864, 0.051389093733268366, 0.0637737311737497, 0.07808443335177134, 0.09434130721503471, 0.11256562506198056, 0.13273218388546476, 0.1546186897791256, 0.1779502838382692, 0.20245517346028657, 0.22786510326297255, 0.2539199551079846, 0.28034924478811474, 0.3068796582559082, 0.33327191567896575, 0.3594826041691513, 0.3855538864653276, 0.4115278610049022, 0.4374522672713143, 0.4633725610405744, 0.48933039504583203, 0.5153607451943794, 0.5415267567900305, 0.5678522995336853, 0.5943994961917812, 0.6211943692632529, 0.6482639799475812, 0.6756116625186395, 0.7031726391831338, 0.7308923622041218, 0.7587279539395325, 0.7867294775390098, 0.8148421038406662, 0.8429678857989144, 0.9820034794874484, 1.1173943075051989, 1.2470917701860833, 1.368058448936091, 1.4795347032857058, 1.5799798335512765, 1.6665768371878524, 1.7402345557044105, 1.79649243215625, 1.8357630631699153, 1.8566346249899217, 1.8607671534637866, 1.844346711505926, 1.8125354983935245, 1.756766805256532, 1.682954025879271, 1.5997344056868772, 1.5132185463464742, 1.4403067535719705, 1.3931188846134672, 1.3862640901799959, 1.3824643631319327, 1.3797825013923224, 1.3716121478932406, 1.354506133891423, 1.3319234751681994, 1.3036216873554227, 1.2694405722799318, 1.2304832262966465, 1.1864390186447098, 1.1379294954743198, 1.0853028395299462, 1.0286215190015136, 0.9685589134691722, 0.9051749936063894, 0.8389196060184211, 0.7703285841149609, 0.6996533712558838, 0.6276613541289572, 0.5546157606416738, 0.4809746484650379, 0.4074875184087239, 0.33434378664647546, 0.26805920827483054, 0.2041031839828012, 0.14802746565038263, 0.10496395151485671, 0.07156878026322783, 0.06373136782614273, 0.06162547426059073], [0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.49999999999999994, 0.49999999999999994, 0.5, 0.5, 0.5, 0.49999999999999994, 0.49999999999999994, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5000000000000001, 0.5000000000000002, 0.5000000000000002, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5000000000000001, 0.5000000000000002, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000002, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5000000000000001, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000002, 0.5000000000000002, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.49999999999999994, 0.49999999999999994, 0.49999999999999994, 0.5, 0.5, 0.5, 0.5, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001, 0.5000000000000001], [0.02308492094443014, 0.032086520954452094, 0.05037982267181087, 0.08443153586492445, 0.1274122116595934, 0.18092383788049274, 0.2412358022050174, 0.3046912263143485, 0.37553257817886976, 0.4484956633591017, 0.5230309081956682, 0.5975055986387536, 0.6712901424732032, 0.7439586856978176, 0.8146613466174393, 0.8830694609718447, 0.948537715616582, 1.010397937727545, 1.0685139235439762, 1.122406927131489, 1.1721609712963215, 1.2176092229665436, 1.2584269234578822, 1.294899957939489, 1.3263695442545094, 1.3530723622669172, 1.3749374939779317, 1.3911866269713253, 1.403152077500785, 1.4072267762646138, 1.4086794243950378, 1.4032237439686555, 1.3924364456345173, 1.3749386044885321, 1.3526322156191395, 1.3245849691159117, 1.2916131995781985, 1.253936169598917, 1.2114625207449412, 1.1649268222293299, 1.1141973330428045, 1.059702294540421, 1.0017278113537322, 0.9403335916999719, 0.8760289859057198, 0.8088639012909451, 0.739110321542582, 0.6672995624733133, 0.5936384191579316, 0.5202986692378802, 0.4482573057027471, 0.4341800484247276, 0.420202820978267, 0.4063374011814084, 0.39259556685219427, 0.3789890958086674, 0.36552976586887076, 0.35222935485084694, 0.3390996405726386, 0.3262561718762394, 0.3137784205703369, 0.3015101505227714, 0.28951280129678175, 0.27778873218137046, 0.2663606741992021, 0.255237123712147, 0.2444262597644654, 0.2339559639862967, 0.22381876003458168, 0.21403117653875087, 0.20447051943645106, 0.1950617416270463, 0.18561154058395232, 0.17605259206492893, 0.16621080277441788, 0.15596781104556234, 0.14518901693244773, 0.1338997682356868, 0.12225871694003211, 0.11063927066652338, 0.09947280171057794, 0.08891509963707576, 0.07994888723049592, 0.07150557847170264, 0.06143720491294779, 0.048316897233169655, 0.030484729617207627, 0.021029139058295423, 0.015917645008721646, 0.01373125521183282, 0.012242992246776965, 0.01106963650681019, 0.01015711465783964, 0.009458074924798091, 0.008931720611047248, 0.008586138543964882, 0.008364922062609546, 0.008313494342799394, 0.008301503373561742, 0.008408199464890671, 0.008567325912207812, 0.008776903599688923, 0.009001381704737634, 0.009225793794812847, 0.009435584891130177, 0.009589795782368443, 0.009702728752582714, 0.00977907599761941, 0.009867101884431077, 0.010016507483082837, 0.010263174268688911, 0.010669589511983355, 0.01127651841675432, 0.012116057575901954, 0.013373423686750218, 0.015353895911566287, 0.01840779959717143, 0.022887277720326078, 0.029148632509815146, 0.037534978464943226, 0.048387870500634166, 0.06204736444757632, 0.07865999336761255, 0.09762619724268821, 0.11815277528933966, 0.13945799652024224, 0.16076669204571353, 0.18132646344552153, 0.20034843363669405, 0.21713271472969245, 0.231996855627088, 0.24595609503791588, 0.26002601559620114, 0.27531664632301306, 0.29200326739207316, 0.30988912679293595, 0.328651683320085, 0.34798866606861084, 0.3674805964912859, 0.38706069046847175, 0.40665777994708496, 0.42628215664555674, 0.4459293474867355, 0.46558788472575235, 0.4852586208905108, 0.5049271111378852, 0.5245786219865183, 0.5442245201957029, 0.5638539798050259, 0.5834456820414533, 0.6797952055683814, 0.7743019578977194, 0.8664933397275966, 0.954971847172957, 1.0392750790168772, 1.1184952763398004, 1.1914460084664686, 1.2581061701995047, 1.316662546824552, 1.3669693362906266, 1.408502241020993, 1.4387539078347131, 1.4605183222482498, 1.46503781663669, 1.4661331169871241, 1.4576182922647358, 1.4436375899243634, 1.427358873603013, 1.412645488460162, 1.4007897064007269, 1.3946919489641414, 1.3888995942058053, 1.3820875020107655, 1.3697976329960628, 1.350252880877346, 1.3257658884524262, 1.2962077160299232, 1.2613473625638385, 1.2215690118543392, 1.1765813579063038, 1.126986276791735, 1.0732253683936697, 1.0154066133408657, 0.9542906597314205, 0.8899760568988211, 0.8229819163832265, 0.7538592714313873, 0.6828806617483839, 0.61082740658643, 0.5379791000190465, 0.46482967731169855, 0.39209730153281425, 0.31998117243104574, 0.2540748149540736, 0.19053295583335977, 0.1337377521792688, 0.08806919238153652, 0.05179241752670451, 0.032472604515130385, 0.02308492094443014], [0.018131403053539573, 0.024839050518760397, 0.04066577043123339, 0.07552517729752162, 0.11954879664761735, 0.17477996951468933, 0.23676451982494714, 0.3009168321414361, 0.37104300227081927, 0.44150565488277543, 0.5122931392195399, 0.5830444554921393, 0.6536116823724493, 0.7238914634733354, 0.7929557373153658, 0.8604010871191733, 0.9255019870042304, 0.9874038884241508, 1.0458727398500023, 1.0997793718047273, 1.1492338199232537, 1.194026389440358, 1.233936665136545, 1.2693289103346455, 1.2996142475740469, 1.3251103292164932, 1.3458085110558025, 1.3610313920432182, 1.3721372244998402, 1.3756882412716138, 1.376903802781469, 1.3717132408291532, 1.3615254046471248, 1.3452018984292273, 1.324148728373445, 1.2973936713912415, 1.2656575973605788, 1.2291408257252217, 1.1877046778529028, 1.1420559582483578, 1.0920143431147635, 1.0379522412122997, 0.9801926223235694, 0.9187853499426523, 0.8547467790857333, 0.7883582012165467, 0.7203968995205835, 0.6515783385003081, 0.5823027250416513, 0.5132776660028219, 0.443969971804437, 0.4297761144783814, 0.41554480291621165, 0.4012676780168206, 0.3869363806791004, 0.37254255180194373, 0.3580778322842431, 0.34353386302489086, 0.32890228492277973, 0.3141162630940959, 0.2991314530970754, 0.2840320576193813, 0.2687845655534953, 0.2533832718435734, 0.23781666902877172, 0.2220775434325951, 0.20615780230439731, 0.19004003787065893, 0.17373027413591113, 0.1572355767090207, 0.14080571418229232, 0.12460624138672222, 0.10906129078739651, 0.09430580787784766, 0.08070292734450227, 0.06858967416659702, 0.05808307104088824, 0.04922291770459472, 0.04194188968029668, 0.03597428126771214, 0.030967487164252263, 0.026797825249857767, 0.023195920191346718, 0.02007447234679635, 0.017378261144207412, 0.015095601503874088, 0.013168540348989769, 0.011596725531139468, 0.010314165976606407, 0.009287121674177968, 0.008470786745934177, 0.007816550214442013, 0.00728109245531156, 0.006822877977698641, 0.006404894322231979, 0.006034821963166246, 0.005709368817123515, 0.005442885969418003, 0.005245041940358229, 0.005101614292761501, 0.0050786177367499775, 0.00510014878952581, 0.005210677941226059, 0.005437747098201199, 0.005773983692004276, 0.006213457963336121, 0.0067819177339687, 0.0074690872820032065, 0.008312725442906265, 0.009359339388764388, 0.0106697067570557, 0.012320670948218872, 0.014386675980498396, 0.01687909226309213, 0.019941232440907605, 0.023591789352592946, 0.02790504056471937, 0.03294806325951423, 0.038788110911865216, 0.045489710877254466, 0.05311728653195913, 0.061735490542992474, 0.07141114631526907, 0.0820986657749856, 0.09351592763488899, 0.10534554875797442, 0.11727767479454516, 0.12913278571058956, 0.14108554469489887, 0.15339556893460582, 0.16628701695473264, 0.17979862587147818, 0.19388330828376488, 0.2085241083976403, 0.22361885966719483, 0.23916127556665212, 0.25510040392562416, 0.2713997377561375, 0.28799160411103897, 0.3048574709527854, 0.3219410613002134, 0.3392064667963104, 0.3566252670756121, 0.37414028581561254, 0.39174613455629254, 0.4093640666933012, 0.42691447556544215, 0.4444518312308718, 0.4619284842228619, 0.47924552723676506, 0.562554137878643, 0.6420571347670161, 0.718364915158614, 0.7919053708586888, 0.8629326096797632, 0.931987860895038, 0.9984152136637842, 1.062106228060969, 1.122110208181186, 1.1781951460277875, 1.2297784321948193, 1.2761201619410953, 1.3173905841570432, 1.3529740072774505, 1.3832723920450067, 1.4082958396603362, 1.4275080044725217, 1.4420926552930322, 1.4499966218466012, 1.453715455896014, 1.4510778058718212, 1.4449996043918656, 1.432086854002881, 1.414890036298525, 1.3922414422240361, 1.364759188254282, 1.3324794949032275, 1.2951016978179373, 1.2531956514756357, 1.206420429567057, 1.155020346361865, 1.0990948736734543, 1.038477650744873, 0.9736371195938702, 0.904822372347731, 0.8329996438944111, 0.7591513483585198, 0.683802086119663, 0.6082863713559491, 0.5330521748335265, 0.4590904621643909, 0.38652650227568963, 0.3153716041404965, 0.24879954890205797, 0.18347906793884897, 0.12479443330522738, 0.0777452777160933, 0.04012199744723111, 0.024439720656833953, 0.018131403053539573], [0.016774455834262427, 0.026839654107628143, 0.046204424693165436, 0.08086971120085673, 0.12441099273921623, 0.17827882632688677, 0.2388367771594735, 0.30254746937396493, 0.3735786644120182, 0.44676756552423413, 0.5215813806181616, 0.5964285983284542, 0.6706727682108446, 0.7438865417041464, 0.815210761928668, 0.8843107671174356, 0.9505334555939673, 1.013205281307807, 1.0721793743234518, 1.1269684462066152, 1.1776440297851076, 1.2240285072271555, 1.2657871931992764, 1.3031923235698224, 1.3355771434016308, 1.3631659606724051, 1.3858713155084346, 1.4029146878598118, 1.4155761744796778, 1.4204800946786436, 1.42237508322744, 1.417281226667955, 1.407007104259881, 1.389744401396634, 1.3676190183758496, 1.3396557567764842, 1.3067129466104652, 1.2690181650144399, 1.2264886194174063, 1.1798849980901605, 1.1290818972713152, 1.0745270748633573, 1.0165216152378416, 0.9551377579838212, 0.8909038206544002, 0.8238819733253259, 0.7543941092605924, 0.6828685745968401, 0.6094850344232461, 0.535583367070951, 0.4617286247449963, 0.44699983030283796, 0.43230149320148964, 0.4176421366872496, 0.40303028400641544, 0.3884744584052852, 0.3739831831301568, 0.3595649814273281, 0.345228376543097, 0.3310318400234824, 0.3170006791299569, 0.30306821802594414, 0.28916576112302894, 0.275261976221654, 0.2612560840004342, 0.24709524914154116, 0.23271146444025326, 0.21804103712022252, 0.20308257576393227, 0.18792483495471268, 0.1726422550258209, 0.15729953292246163, 0.14201290731136068, 0.1268256068154156, 0.11188668160617925, 0.09710832607731798, 0.08248678932583041, 0.06741518668277706, 0.05293795050333592, 0.04139323499588102, 0.034946637986306185, 0.032273762636458196, 0.030062338916306285, 0.027975709132811014, 0.025278087445548388, 0.02235231682783081, 0.019311849001460508, 0.01639327563456037, 0.013720231669898447, 0.011454453899257647, 0.009714968907947322, 0.008414208388043428, 0.00750443653169726, 0.006861059171122635, 0.006405255607510504, 0.006147716078475064, 0.006006196594055663, 0.006027617309487937, 0.006109344551722346, 0.006305792721127906, 0.0065602840069377, 0.00686556652849386, 0.007189317330215463, 0.007516002986766458, 0.007830218705337549, 0.008098966428113098, 0.0083147557084817, 0.00842240395036817, 0.008537598779507648, 0.008736490580937764, 0.009194497672756702, 0.010188170630769246, 0.01188127867977542, 0.014382261523525815, 0.018036704629151204, 0.022936729218034097, 0.029258363670547474, 0.03696351678833281, 0.04592241139237329, 0.05599333717101679, 0.06703763268314987, 0.07891849136397586, 0.09150603516999503, 0.10466232158747098, 0.1181827790591566, 0.13177803787138517, 0.14515284620820557, 0.15802602764718304, 0.17033620977880398, 0.18244303304242682, 0.19476524514681332, 0.2076566036711319, 0.22126237607963384, 0.2357510423983707, 0.2510642078034736, 0.2672273140382244, 0.2840622567717901, 0.3014312441307663, 0.31908330533176976, 0.33694127913324495, 0.3547862314460219, 0.3726223949672996, 0.39042501811918984, 0.40818416675278046, 0.4259163079197367, 0.44361082892492915, 0.46125428657017525, 0.47885377417185593, 0.49640492501616224, 0.5138964514279726, 0.5995695747520435, 0.6832959503833879, 0.7649074672753908, 0.8436224740918031, 0.9192286855308585, 0.9912621973203875, 1.0590792223014656, 1.1226320675991412, 1.1807688606326627, 1.2333144551061082, 1.2797001185265175, 1.3186605553096866, 1.3509196390792864, 1.3743420206296482, 1.3907952971489181, 1.4011110547596377, 1.405147558088399, 1.406232805072976, 1.4051985480442382, 1.4015305649323841, 1.3974133152858903, 1.3915791427856163, 1.3827108289931958, 1.3697777534447437, 1.351235367354727, 1.326066930060676, 1.2950366621615008, 1.2579755445456267, 1.2164597991233583, 1.170244473330438, 1.1199005138116631, 1.0656878937659144, 1.0076383924021137, 0.9462967710207697, 0.8817172792957342, 0.814262449529698, 0.7446085141943068, 0.673067423449282, 0.6005881005144406, 0.5275165644567039, 0.45448035632164113, 0.38227314797676615, 0.31113113432962874, 0.24636628378077213, 0.1841522827022918, 0.1285100017510995, 0.08352342792895816, 0.047686443832315926, 0.027449351825510133, 0.01677445583426243], [0.037735912391203906, 0.04508672354653929, 0.061589787365546655, 0.09501804639458958, 0.13748506441465366, 0.19086948783414803, 0.2512100575509591, 0.3145564814454104, 0.38519824246356116, 0.4576324070396615, 0.5312393293501028, 0.604245593894391, 0.6761310786625343, 0.7465167773885699, 0.8147024104143893, 0.880430828782791, 0.9431726119308478, 1.00235807454094, 1.0579016671267945, 1.1093180690870155, 1.1566776221698276, 1.199798172790401, 1.2383248720035476, 1.272553466132286, 1.3017769432120205, 1.32623003373337, 1.3458739944284275, 1.3597855099159464, 1.3696237540354566, 1.370439119928295, 1.3699678719083608, 1.3623607675001426, 1.3487728202475124, 1.329505698383122, 1.305697081575047, 1.2766236405358142, 1.2429377953538063, 1.2048459455824772, 1.1622550529097153, 1.1158027661841223, 1.0653653261335052, 1.0113096044998755, 0.9538825505617379, 0.8931216262406159, 0.829448392192381, 0.7629697858883991, 0.693978589620761, 0.623557393059364, 0.5522070331109916, 0.4814039936012458, 0.41173533742280066, 0.39799592052461513, 0.38433490185567276, 0.370757892018641, 0.3572705016161866, 0.34387834125097666, 0.3305870215256784, 0.31740215304295877, 0.30432934640548476, 0.29145013343204795, 0.27882164838367335, 0.26633233100026166, 0.2540232837597792, 0.24189679419291651, 0.22996595353495933, 0.21823683173083905, 0.2067149448383753, 0.1954186908297803, 0.1843506816914205, 0.17352955568537642, 0.16296474503173464, 0.15266235221621405, 0.1426440519171532, 0.13291051584889882, 0.12348052860875829, 0.11436714142135715, 0.10557971052963293, 0.09710938365349477, 0.08899749580308906, 0.08125255634112538, 0.0738659377146496, 0.06685706687335863, 0.060251716187876184, 0.05404426212821401, 0.04825015228976676, 0.04289340674931447, 0.03795714171269463, 0.033487062375027084, 0.029472891272303885, 0.025919381596011273, 0.022875552010384867, 0.020290336466531483, 0.018237895527460024, 0.01671919359204472, 0.015688858717377744, 0.015253603560889973, 0.015078938487321602, 0.014724897945774548, 0.014150248516433409, 0.013147469316074368, 0.012127918410087844, 0.011201083754966897, 0.010525714636202041, 0.010077436737014889, 0.009794329852012737, 0.009765845864656245, 0.009840912261653319, 0.010080880900249072, 0.010480577705821129, 0.010988765701359558, 0.011545400504487659, 0.01212994220852258, 0.012686555038962849, 0.013187221242346947, 0.013576668303979748, 0.013857920231509687, 0.014370249704897496, 0.0156964321787108, 0.01845239147665648, 0.023226504753696024, 0.030601798938667015, 0.0411609863379234, 0.05532996926188673, 0.07283870194474959, 0.09323756511618737, 0.11609222783890764, 0.14097609954595294, 0.16737437419161605, 0.19443964958325816, 0.22124551910011253, 0.24699783398618066, 0.27165864192687805, 0.29551711394693225, 0.31887577805533884, 0.34202660222520087, 0.36527333190337336, 0.3886091574829273, 0.41207146657596405, 0.4356182602279382, 0.4592396716945046, 0.482909997846514, 0.506613125007123, 0.5303362822906255, 0.5540579705092719, 0.5777850932576463, 0.6014957251214854, 0.6251664184455645, 0.6488140895773269, 0.6724234409187648, 0.6959634164634035, 0.8113623327775291, 0.9237257289600027, 1.0324105434339392, 1.1353405075487968, 1.2318633479218086, 1.3207188216851786, 1.3999383211334038, 1.4697870352205762, 1.5272360127272608, 1.5727932829961786, 1.606968629323977, 1.623808458304745, 1.6306089683890763, 1.6250145686794701, 1.60304797846683, 1.5679576377121174, 1.5243371182308398, 1.4764116580273037, 1.4341382304300174, 1.4061197373175007, 1.4026831110700113, 1.3994177850302667, 1.3942603735331467, 1.3834549261462465, 1.364356275640404, 1.3397395434737367, 1.3095716906259915, 1.273760989263376, 1.233391489583557, 1.1881900923782547, 1.1386908300264391, 1.0851793080931984, 1.0276811915587525, 0.9667684007093411, 0.9024819356693905, 0.8351983626315249, 0.7655418904172251, 0.6937967799622217, 0.6208484555579799, 0.5470085710807484, 0.4728377770084386, 0.3991257261800915, 0.3260950400141247, 0.2597456715921401, 0.1958593369474308, 0.13927635697390858, 0.09474667656150723, 0.05995047976497747, 0.04432275292601602, 0.037735912391203906], [0.01799702586935321, 0.02719080351167274, 0.04580377661345292, 0.08035682279318103, 0.12363111209621935, 0.1771397143037864, 0.23704375309746192, 0.29951998933448865, 0.36833644704426877, 0.43838600041161296, 0.5094186178647794, 0.5806148452625299, 0.6512926088701528, 0.7210800624033817, 0.7889618501107563, 0.8546098350154978, 0.9172998144084088, 0.9767191217452046, 1.0328340459076824, 1.0851657854579553, 1.1337131809743501, 1.1782324259466628, 1.2183068579981908, 1.254169124022018, 1.2850527406222845, 1.3111289032152034, 1.3322984502867161, 1.3475779559430878, 1.3585630516066574, 1.3602983274236828, 1.3605978656390612, 1.353950976556861, 1.3416194524768668, 1.3237564229398235, 1.301504220422807, 1.2741125522076573, 1.2421975338764975, 1.2059329350537487, 1.1651958352618654, 1.1205878172428598, 1.0719554579768504, 1.0196241764456133, 0.9638360097310406, 0.9046091276686951, 0.8425538443115655, 0.7777670792853001, 0.7106846976075316, 0.6416752569099377, 0.5709209615625105, 0.49894798785376726, 0.42593466629006393, 0.4111202738930524, 0.3962666651397454, 0.38138090896057614, 0.36647007428597783, 0.3515412300463835, 0.3366014451722263, 0.3216577885939393, 0.30671732924195555, 0.29177291455766247, 0.2768154289844779, 0.26186244766819267, 0.24691207744680038, 0.23196384197458061, 0.2170088812096405, 0.20204103423047182, 0.18705388202496465, 0.17203720002737527, 0.15698708299203304, 0.14189330205609482, 0.12674953151735405, 0.11154988429852442, 0.0962839240298354, 0.08094753212207392, 0.06542385504944928, 0.05010944341708549, 0.0350704423413494, 0.022632675026360457, 0.014480000081430446, 0.010473603402721049, 0.009368265110639287, 0.009762294984622763, 0.010644633504187803, 0.011596477761307088, 0.011370862702146215, 0.010834180712429387, 0.009733805211381157, 0.008950382723234441, 0.00842506463649037, 0.00815120574692995, 0.007993298574348382, 0.007897371803612881, 0.007734868592011331, 0.007476602339364554, 0.0070061664854308885, 0.006426355452338488, 0.005768658556668946, 0.005140893394622561, 0.004626051050776315, 0.004239597810249827, 0.0041866056437996485, 0.004303935465932876, 0.0046211998848805025, 0.005144437470145056, 0.005776005811945607, 0.006457013240359692, 0.007160940601768258, 0.007826629569999478, 0.008396049932544577, 0.008869261654680244, 0.009047066748052553, 0.009119726922389377, 0.009142340343687681, 0.00933986370804336, 0.01113810997253533, 0.015101695840401408, 0.02192715220479392, 0.03136421441539009, 0.04271254212065977, 0.05525584997614344, 0.06855302314388473, 0.08241205152546506, 0.0966596777412621, 0.11111531117165944, 0.12559619551181767, 0.139918897038972, 0.15393266388570664, 0.16761826357370568, 0.1809982051721452, 0.19409567317586932, 0.20693389689866434, 0.21953582391109078, 0.2319258152724767, 0.24411646174282037, 0.2561516195998421, 0.26803945629982306, 0.2798065113438651, 0.29147109090879597, 0.3030662618311441, 0.3146045831735406, 0.3261165162868672, 0.3376231657897159, 0.34913739006422684, 0.36072204457748164, 0.37234495250828903, 0.384045477714107, 0.3958723195971461, 0.4078229324898913, 0.41988430597378695, 0.43205201048271297, 0.493827681966947, 0.5576222143367818, 0.6228585490389984, 0.6887346142961741, 0.7549208810890833, 0.8207748380715546, 0.8856813639935024, 0.94934862482264, 1.0109268148656465, 1.0701564758888966, 1.1264430260666587, 1.1794948587685787, 1.2293121362995525, 1.2754341793202448, 1.3178828484648246, 1.3565657053718128, 1.3906904840367755, 1.420638636050457, 1.443216199562711, 1.4560896854765633, 1.4563164419364032, 1.4549934265702313, 1.4406276284880484, 1.422474360760216, 1.399117342669056, 1.3712329480256973, 1.3386147842914768, 1.3008537475558961, 1.2583304099494488, 1.2106617731777856, 1.1579610081505693, 1.1007919145049725, 1.0392500736717742, 0.9742875569581, 0.906046164514023, 0.8351544829120817, 0.7621985638656538, 0.6874759157303932, 0.6120904897038433, 0.5364812619242645, 0.46149649801497933, 0.387846974609099, 0.31576011415741145, 0.24922759546731077, 0.18460577938950845, 0.12674982855107822, 0.08031717722940304, 0.04333319338304656, 0.02569179247604264, 0.017997025869353207]]
        prob['airfoils_cl'] = [[0.012774299814956497, 0.14269327218499767, 0.2586370117479547, 0.3623386438927308, 0.44846540213334535, 0.5258962146947556, 0.5932747256952848, 0.6483731015052915, 0.6919714774537448, 0.724561884158482, 0.7475588397302069, 0.761476518094192, 0.7649770522520746, 0.7610743400910195, 0.7512774380781152, 0.7313798387745672, 0.7053181409441592, 0.6728915426068911, 0.634057340901292, 0.590171891142727, 0.5410148133866781, 0.48740716155951874, 0.42985986987276736, 0.3685763349494897, 0.3044320123794003, 0.23760988069898553, 0.16873016172346023, 0.09833293688442458, 0.02670874222112079, -0.04539424263359261, -0.117688842156067, -0.18961620380430833, -0.2606048613871396, -0.3303845295102284, -0.39818700075596947, -0.46373523804603245, -0.5264952231737257, -0.5858109650288182, -0.6415695381785984, -0.6928091067102785, -0.7393831898384677, -0.7808112224110497, -0.81611327995889, -0.8458394186133674, -0.8683507703076264, -0.8834865804009255, -0.8900593780814409, -0.890834276816703, -0.8807641250363609, -0.8634573989452705, -0.834899292600962, -0.8282489306690205, -0.8209342610459643, -0.8130379108983156, -0.804642507392596, -0.7958306776953279, -0.7866850489730332, -0.7772882483922339, -0.7677229031194519, -0.757642826424416, -0.746714949015, -0.7354598970264341, -0.723751051434922, -0.7115736689186716, -0.6989349299180345, -0.6858221810715147, -0.6722290584227956, -0.6581300051061376, -0.6434981088933348, -0.6283058740281965, -0.6125266628415562, -0.5961303038207688, -0.5790939409637361, -0.5613855416799043, -0.5429808773658764, -0.5238497699943111, -0.5039455731724146, -0.48331527043284084, -0.46184828498002933, -0.43951673403929203, -0.416384286969858, -0.39235389628802414, -0.3675495803246012, -0.3419926182048192, -0.31576519070347914, -0.28893931081761076, -0.2615473308199055, -0.2336907396717728, -0.2054051833434165, -0.17675892289432352, -0.14783043277550198, -0.11865391393879493, -0.08937954619790792, -0.059860095852091344, -0.030185078540683603, 0.0012276019638754424, 0.035077649579663424, 0.07314602872212428, 0.11636666891197389, 0.166292415723779, 0.22403376141165418, 0.2907425444002591, 0.3652231354797523, 0.44573809999042824, 0.5303855569486363, 0.6160593764113568, 0.70158900252098, 0.7838622877333025, 0.8621393122542008, 0.935702772940798, 1.0048242248664394, 1.069786819538331, 1.1306029864843783, 1.1879114452355026, 1.2413575000795158, 1.291427025620277, 1.3382500549164937, 1.3820187279231424, 1.4229244423480307, 1.4611558647009941, 1.4969023588186527, 1.5303538676614972, 1.5616804879844108, 1.5909828749291866, 1.6183431962266996, 1.643843995988398, 1.6675688299847076, 1.6896042718028865, 1.7100333113018582, 1.7289386078333222, 1.746403029450979, 1.762511085708032, 1.7773607847528972, 1.7909570443296023, 1.8035401559564967, 1.8150604270277528, 1.82564497776333, 1.8353774297611998, 1.8443376241529081, 1.8526135288184375, 1.860289296289324, 1.8674457505639261, 1.8741736088302692, 1.8802946201466608, 1.8860939669516341, 1.8914413019319458, 1.8961736822282051, 1.900525535294271, 1.9044326767258222, 1.9076799063903955, 1.9152886268953402, 1.913708174109812, 1.8942547863247823, 1.8634835437590251, 1.815671319614068, 1.7527231925405065, 1.6737503956327202, 1.5765429516184701, 1.463220505921252, 1.3342366508018952, 1.1953155117819012, 1.051696886081918, 0.9064407545832502, 0.7670097848150973, 0.6356478103389553, 0.5177141807138634, 0.41113935764915804, 0.3148901019407244, 0.22633041939349696, 0.14376557334508758, 0.06503676966750052, -0.011793380444871436, -0.08774937110969894, -0.1624196350704996, -0.23547116150071226, -0.30630966744662236, -0.3742863533863529, -0.43921290564776194, -0.5001720776223696, -0.5569541461155774, -0.6090250095165141, -0.6555259800300086, -0.6966907370648507, -0.7310982922380173, -0.7587989660341019, -0.7799816948279059, -0.7906161409454636, -0.7950150432209698, -0.7914675204942629, -0.7771749373511712, -0.75327399066682, -0.7192891181489044, -0.6737401420588566, -0.6161306932475592, -0.5456212440970902, -0.4645602813766505, -0.3746427178499516, -0.26680067154758935, -0.13878272406752298, 0.012774299814956529], [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0], [0.04964209610768981, 0.18728247933488665, 0.31002095168231864, 0.4197144359489547, 0.5106991363681169, 0.5924429394439092, 0.6635103121657981, 0.7215388854029396, 0.7673865118191947, 0.8015843789771776, 0.8256485921551736, 0.8400961981985308, 0.843702760386986, 0.8394444296466287, 0.8289938242178687, 0.8080256543206343, 0.7806321710826848, 0.7466342734001926, 0.7060153128499983, 0.6602081999837803, 0.6090118339637887, 0.5533041752551805, 0.49364207119826703, 0.43026327666183745, 0.3641026544735597, 0.2953770621220479, 0.22475499336493257, 0.15282746828955382, 0.0799153696412935, 0.006830498587953557, -0.06610873016717743, -0.1382744282075125, -0.2091146494597552, -0.2783623163152973, -0.3457249666850165, -0.4111544300960605, -0.4745009608088385, -0.5355831827593236, -0.5943839060863658, -0.6506998689248986, -0.7042851867756956, -0.7551355236093819, -0.8005802042448851, -0.8404406635564805, -0.871232633180789, -0.8902420264897877, -0.8935596412969572, -0.8920097340788381, -0.8704473942338138, -0.8496621277746017, -0.8360105155061717, -0.8364319170528584, -0.837651180659868, -0.8396008947776253, -0.842213647856555, -0.8454220283470817, -0.84915862469963, -0.8533560253646247, -0.8579468187924902, -0.8647688223719059, -0.8750541155037499, -0.8862978972438583, -0.8992564093183807, -0.9142976351807424, -0.9322863208005226, -0.9539001376819924, -0.980194821189284, -1.011774282526046, -1.0496252169726716, -1.0937918675176013, -1.1403857817161722, -1.186376193041567, -1.2242055045071811, -1.2507948917409322, -1.2548945730736596, -1.2543978298863578, -1.228670649367338, -1.1904373208481211, -1.1396299200975855, -1.0807080605203003, -1.01890919931456, -0.9559477002022148, -0.8986853259640485, -0.8479714431363805, -0.8072531609810798, -0.7699275458301977, -0.7340461901368132, -0.6922652607045104, -0.6422666207565266, -0.57699388892989, -0.5011537365373427, -0.41570664690505504, -0.3304839217503235, -0.2453385825423862, -0.16135472938184708, -0.0781178670652549, 0.004412087510226579, 0.08632850367660144, 0.16768663713800064, 0.24854436820941997, 0.32899470763543365, 0.4090743302507231, 0.4888761585658733, 0.5684593655987711, 0.6478796087988709, 0.7272610420728656, 0.8065225459389558, 0.8854077271924518, 0.9633183513944151, 1.0399017140545257, 1.1143275937629418, 1.1863230149999486, 1.255086353808457, 1.320492244052257, 1.3808451416281722, 1.433931118904791, 1.4770291648953342, 1.5076697555782053, 1.5226807865726688, 1.519900723239487, 1.4944307252610132, 1.4553800141978996, 1.4165183733181508, 1.3816306817072508, 1.3519703642070686, 1.3288607536830792, 1.313652451017612, 1.3081213075051374, 1.3113049316085135, 1.3270408235293274, 1.3509079157662678, 1.3774407560323114, 1.4012151392352752, 1.4143574610412732, 1.4198013580075086, 1.4161782030037942, 1.4075414481564543, 1.3952758921019044, 1.3829428714274528, 1.3711261556190608, 1.3601943086936434, 1.3499780863475288, 1.3404055208984913, 1.3316481834746832, 1.3233724277914574, 1.3156210763429435, 1.3084852682835109, 1.3018609160670271, 1.2956956372384731, 1.2899901097466615, 1.2672188915935685, 1.2500579077264622, 1.235201958883625, 1.2168362418596117, 1.1932919317470658, 1.1592008600230026, 1.116117928215041, 1.0632830045026074, 1.0033276572011578, 0.9360245506051608, 0.8628467007575726, 0.7847452125190102, 0.7022109958919858, 0.616772318620103, 0.528888208599961, 0.43967685289003583, 0.35018426166044636, 0.26096675744443065, 0.17347186560929934, 0.08816071259616412, 0.006090071507540515, -0.07261668492490235, -0.14792781747763964, -0.2196434818619227, -0.2878182273109148, -0.35235005179773804, -0.41309038969525097, -0.47015587339460335, -0.523243037447776, -0.5724612881034933, -0.6177031659307005, -0.6587596653581551, -0.6958766374831244, -0.7286411624213136, -0.7571192435331721, -0.781484500769481, -0.7994765967361901, -0.8126090987742801, -0.8133286154579451, -0.8109652027509235, -0.7965426029921661, -0.7709466282180923, -0.7322356166802069, -0.6798880600475492, -0.6111445924148585, -0.5291067917545871, -0.43653108136544194, -0.32075991128823245, -0.16266292116639633, 0.04964209610768984], [0.00921676639997223, 0.25147128101163213, 0.42715697353694854, 0.5546876910378238, 0.6186145451769408, 0.6628203413283077, 0.6938831543798423, 0.7134517023017004, 0.7246759439574989, 0.7452236741669606, 0.7739610979927354, 0.8035287327862465, 0.8185902307985563, 0.8234836678199766, 0.8178976438581758, 0.7972220924119334, 0.7681933393814395, 0.7313595201772793, 0.6879378650003364, 0.640678063092424, 0.5895108925722523, 0.5351887076834958, 0.4779475623199586, 0.4179054720071945, 0.35559236790475446, 0.29111792265922615, 0.2248560689793226, 0.15713675868656718, 0.08812841292293498, 0.018291229046321144, -0.052208796941995046, -0.12302767964464062, -0.19383377027396026, -0.2644577905479207, -0.33438885056352, -0.40319805540672643, -0.4701513375134536, -0.5342934556164555, -0.5953193454129955, -0.6519298216208483, -0.7037786439321458, -0.7501474414703292, -0.7896788104965823, -0.8229247704366778, -0.8477982094402521, -0.8636393568493153, -0.8684398610082152, -0.8681097937126746, -0.8524603187653123, -0.8276803298568334, -0.8011772881802078, -0.7988680629131615, -0.7966657055638412, -0.7946269276978027, -0.7928084408806018, -0.7912669566777939, -0.7900591866549354, -0.7892418423775815, -0.7888716354112882, -0.7909641475823819, -0.7966103767858694, -0.8033162866011881, -0.8116876984011139, -0.8220474468358867, -0.8343133901599069, -0.8487063083807073, -0.865377105214045, -0.884347838895094, -0.9058277072049403, -0.9296594497809416, -0.9544506849722145, -0.9791063516645745, -1.0010241439365108, -1.0192431184704995, -1.0314986908098287, -1.0355127518656941, -1.0305357603170158, -1.017892376784259, -0.9944162382564778, -0.9621539787041551, -0.9227768090607795, -0.8752741443050032, -0.8216545255892146, -0.761723349304874, -0.6965293875553347, -0.6267527740676063, -0.5526906468324642, -0.47545995568807276, -0.39539852210946713, -0.313267722121883, -0.22990068795983645, -0.14568091403114114, -0.06164335851299934, 0.021816481672126797, 0.1040343243562381, 0.18495768493122225, 0.26455507747655094, 0.34281648423562505, 0.4197409202863443, 0.4953257519807379, 0.5695427348756178, 0.6424046114802859, 0.7138832571217151, 0.7839748381155316, 0.8526817119727674, 0.9199764142679712, 0.9858608070331852, 1.050512195993086, 1.1128889581844994, 1.1726723039193037, 1.2270659125807475, 1.2752769358740255, 1.3148935324396747, 1.3453408448461388, 1.364215077491246, 1.3693098786701754, 1.3638059950393773, 1.3493859790032487, 1.329346943873698, 1.3068899015538749, 1.285125181540227, 1.2672098960171547, 1.2561217060831744, 1.2537185150941974, 1.2576023646138907, 1.2671227951481123, 1.2799168417313862, 1.2942746481784528, 1.3083502409040026, 1.3203021025754256, 1.3283020470769402, 1.330945142279679, 1.3275587115507608, 1.3163028993773243, 1.2978852389457352, 1.2748513241718629, 1.2483499131073517, 1.2205133648762037, 1.1937609228848125, 1.168433533849677, 1.1448640440159594, 1.1228770548320446, 1.102321295084218, 1.083637685576073, 1.0660249562552329, 1.049652442086331, 1.0348006827671679, 1.0211163752006545, 1.0085398801091126, 0.9972118120371214, 0.9555588067348608, 0.9334989299115172, 0.9225970611316626, 0.9149891095858532, 0.9076744656871922, 0.8912002398796282, 0.8671064572948797, 0.8338029412971538, 0.7939925117393026, 0.7470129898168785, 0.6941220881492408, 0.6359387376756891, 0.5726971960359782, 0.5056046080564932, 0.4348527829188079, 0.3612809890177059, 0.285590605079623, 0.2081628011746643, 0.1299662486331714, 0.051305704925849134, -0.027199448599835432, -0.10495654261613545, -0.18167263034204442, -0.25654380541281613, -0.329271391702217, -0.3992876583610322, -0.4659148900366639, -0.529002173109885, -0.587570955219782, -0.6414419160888952, -0.6900876415668283, -0.7325417712351244, -0.7691966154051664, -0.7984667176659584, -0.8211555317527629, -0.8376707343478791, -0.8463716831780459, -0.850436948653864, -0.8488220570043553, -0.8414971061320172, -0.8284180819857495, -0.807964855311958, -0.7763804676098537, -0.7308062694237145, -0.6671027522589141, -0.5881541248749942, -0.4956496360117654, -0.3712883394174779, -0.2066755550549938, 0.00921676639997221], [-0.002616468238648685, 0.23206900252153614, 0.4603181882842975, 0.6930017691262662, 0.7206625884778394, 0.7224940606574921, 0.724042517217225, 0.7286224145878468, 0.7432243241439088, 0.7743263584953606, 0.8182150197859032, 0.8475932097294956, 0.8562812804243328, 0.8559088044398866, 0.848310137238659, 0.8244348255650564, 0.7944906077302177, 0.7580456618158811, 0.7161403014998445, 0.6698705512386851, 0.6190869076774835, 0.5643711588155162, 0.5061264981235706, 0.4444953082081353, 0.3801568940966412, 0.3132402718889729, 0.24422198204879905, 0.17351535606415375, 0.10133400461399346, 0.028256928642579743, -0.045505643957431036, -0.11953465454539544, -0.19335604620539582, -0.2667402329440988, -0.33878318694595244, -0.40911395952502716, -0.4770562377919175, -0.5418208264847281, -0.6031957488410511, -0.6600595411911092, -0.7121652967033243, -0.7589060752760906, -0.7991450354022657, -0.8333728338966895, -0.859728529763199, -0.878025762689064, -0.8871600938577962, -0.8887790153727972, -0.8802828416325041, -0.8668810724656328, -0.8475396219516327, -0.8435937876761197, -0.8394828177921558, -0.8352478011813351, -0.8309298267252514, -0.8265699833054988, -0.822209359803671, -0.8178890451013624, -0.8136501280801666, -0.8094784245519336, -0.8054468041158873, -0.8014801358779907, -0.7986029911398755, -0.7967160777012372, -0.7973423384760688, -0.7995858282152296, -0.8057288987300519, -0.8150402254566712, -0.8284112530084973, -0.8440615358256713, -0.8615396926976412, -0.8799559859252027, -0.8980322647376757, -0.91523621784193, -0.9304950930790624, -0.942413431203417, -0.9512192040947905, -0.9546454762324963, -0.9527137252079912, -0.9439387871877649, -0.9284922926971095, -0.9033935665176519, -0.8700883610098411, -0.8270090759729221, -0.7758867897845587, -0.7175690934518452, -0.6523266842540123, -0.5818748509734252, -0.5064633903523231, -0.4271505141574296, -0.34499131063436694, -0.2604518711222642, -0.17488019238721864, -0.08877572034977002, -0.0030015220230726553, 0.08223883159210894, 0.16683214255734632, 0.25060668002555214, 0.3334594031529094, 0.4152833722544482, 0.49588434320030544, 0.5752071692546012, 0.6530583031678809, 0.7293241502406013, 0.8039201516564705, 0.8766298098879021, 0.947406478561816, 1.0162017990866998, 1.0821456415163473, 1.1449753361239645, 1.202455686496474, 1.254038085494264, 1.2977251991665586, 1.3332960105496208, 1.3585201510298999, 1.3726745888430492, 1.373869131445424, 1.365449536694834, 1.3499321528039856, 1.3308166793862883, 1.3113071429679923, 1.2940051518286881, 1.2810358200291048, 1.2747652274934762, 1.2753026195804178, 1.2812449121542835, 1.2909198454213517, 1.30255188375978, 1.3143512156984505, 1.3245481416714615, 1.331312267673752, 1.3332609626927117, 1.3309014885585173, 1.3244222189934274, 1.3147468755697793, 1.3027261076559569, 1.2886831474767493, 1.2733141100874439, 1.2575555916074548, 1.241726601610428, 1.2266684706057875, 1.21233017250945, 1.1987376570206512, 1.186065701896023, 1.1739501213850068, 1.1624510144609033, 1.1516818517781682, 1.1415289574255643, 1.131950201208858, 1.1229677217737193, 1.08593241400807, 1.058813919916643, 1.0381408865601018, 1.0199312884340919, 1.0021015494457755, 0.9800529933895374, 0.9519077003539834, 0.9140858918343574, 0.8688909985119098, 0.8157606398983058, 0.7563152277320806, 0.6913649547611471, 0.6212933911819628, 0.5475751859186863, 0.4705466871407976, 0.3912617908945412, 0.31064498280595226, 0.22920733957964814, 0.14824213451476245, 0.06818697762083922, -0.010000414060626444, -0.0860542379191686, -0.1598307396733716, -0.2309646749331465, -0.29937484355237, -0.36481519552244795, -0.4269771798942595, -0.48586307625290426, -0.540986283772392, -0.5923380694523884, -0.639671275942614, -0.682580587851279, -0.7212707371392048, -0.7550909114880705, -0.7839377276984641, -0.8082575653067923, -0.8229749922343437, -0.8309961991552474, -0.8276490324286447, -0.812153328474872, -0.7790834510508912, -0.7376085279356305, -0.6942866171984232, -0.6929948686289784, -0.7052200413905353, -0.7174452141520924, -0.6966272978689372, -0.4607460995828775, -0.2330895014007418, -0.0026164682386487174], [0.018044297553556108, 0.14940292936070623, 0.26676311810881936, 0.37184574202104037, 0.459327832609112, 0.5380583121024374, 0.6067099548761082, 0.6630759619069262, 0.7078512545269574, 0.7415826452276482, 0.7656879225541662, 0.7809072057292176, 0.7849042154407859, 0.7820136790001008, 0.7733331913378843, 0.754141826738946, 0.7288220468444292, 0.6970632530777141, 0.6588148047077906, 0.6154521326010057, 0.566732094013432, 0.5134825553972703, 0.4562102021298326, 0.39511120641012765, 0.33106447342347467, 0.2642449508526661, 0.19527277769327706, 0.12468428338095572, 0.05276690599345895, -0.019733548302731222, -0.09253337626831099, -0.16507537710710085, -0.23679362593452047, -0.3074160993521714, -0.3761831401057264, -0.4428166608744562, -0.5067844584573098, -0.5674418173978354, -0.6246619230483285, -0.6775064776028537, -0.7258186764907292, -0.7691114664534304, -0.8064504835211237, -0.8382989516438504, -0.8630739106199713, -0.8807531966420868, -0.8907376221990388, -0.8928570539234237, -0.8860643624134216, -0.8722595451946025, -0.8468339948534888, -0.840887678623592, -0.8342711249442187, -0.8270697405818078, -0.8193689323027982, -0.811254106873629, -0.8028106710607393, -0.7941240316305678, -0.7852795953495537, -0.7759325241196888, -0.7657463275718548, -0.7552394438916182, -0.7442860904730118, -0.732871091011862, -0.7209965852994233, -0.7086690832898925, -0.6958805534191188, -0.6827703398978145, -0.6693826674418567, -0.655865294345908, -0.6423042088145846, -0.6287792964016465, -0.6154382022137287, -0.6023341369623725, -0.5896011422462287, -0.5773472445419224, -0.5656446487095023, -0.5545814189202929, -0.544304876292798, -0.5349500473437488, -0.5264724063477177, -0.5189156752400378, -0.5115061410052029, -0.5039187729385417, -0.49540202085874857, -0.4854515729251999, -0.47344884352340466, -0.4588431723141543, -0.4410468983562104, -0.4193904220291376, -0.39350678462435895, -0.3624918912100299, -0.32610715179005945, -0.28357600470042815, -0.23417393194235994, -0.17790893307585603, -0.11410373121193833, -0.04430180276036432, 0.03116572155517389, 0.11151402019364313, 0.19556392121349686, 0.28299375013046174, 0.37260591794811077, 0.4637589517877755, 0.5558052372466824, 0.6475941527539673, 0.7386976877173175, 0.8280094715076933, 0.9150175575618789, 0.999336910944634, 1.0804222172467932, 1.158154002386628, 1.2319109123485366, 1.3018998307103988, 1.3671707207619306, 1.4276798936561483, 1.4828452381016506, 1.532080418305339, 1.5748107134493288, 1.6104184191799789, 1.6382642721582057, 1.6576593262751267, 1.668560626682865, 1.6713675752288517, 1.6689689757706465, 1.6609761289731089, 1.649404578269317, 1.6354173027523113, 1.6203394307362702, 1.6054923793991054, 1.5920483595632784, 1.5802864809739574, 1.57007933407109, 1.561421289135802, 1.5539913545454254, 1.5478721417445693, 1.5428924182757005, 1.5388872886448053, 1.5358520946201901, 1.5335646930318703, 1.5319838799345749, 1.5310217334041432, 1.5304632425865563, 1.530463811815848, 1.5308180284289807, 1.5313643646868758, 1.5320014973875913, 1.5329906515407428, 1.5342309311905247, 1.5354637098026827, 1.5401735956337035, 1.539835542103524, 1.5265280833035295, 1.5041039626805115, 1.46774989461565, 1.4197654497264338, 1.3595263122185595, 1.2855697852919836, 1.1996452453507955, 1.1019167180656488, 0.9962406649636701, 0.886022347987682, 0.7732746943174021, 0.6628802903211917, 0.556319453490864, 0.45716316234185894, 0.3642135709442855, 0.2770860039600689, 0.1941975069614218, 0.11473069390663591, 0.03743951062240684, -0.03865124723487228, -0.1140972612355698, -0.18832794902708228, -0.261010536014891, -0.3315447855565196, -0.3992650939437521, -0.4639725572177981, -0.5247288742103813, -0.5813131536288851, -0.6331773206764699, -0.6794387849370547, -0.7203340254201033, -0.7544068136156351, -0.7816971089344517, -0.8024675161162952, -0.8123002779886351, -0.8159168508309186, -0.8119305533472634, -0.7964342024747622, -0.7713665095596736, -0.7360579510850251, -0.6889909333230678, -0.6296048443647976, -0.557106534987666, -0.47386067904320894, -0.3815922724841882, -0.2711221527499519, -0.13912972807869703, 0.01804429755355611], [-0.0012980832577013347, 0.2292009629488962, 0.46455875051111, 0.716268381853667, 0.7343865752244809, 0.7049538117425548, 0.6755210482606289, 0.671874844651653, 0.7118206717912932, 0.7529485597627625, 0.7877773216487074, 0.804946230893374, 0.8089385361480931, 0.8023448568842237, 0.7890459031247791, 0.7657183040393372, 0.7377836940861604, 0.7047645561316509, 0.6667861899295056, 0.6245947200155327, 0.5780050697675925, 0.5274914386536655, 0.47335074608520666, 0.4156376691753185, 0.35490104656911975, 0.2911833611241573, 0.22484894255304483, 0.15619158287057033, 0.08533449345257137, 0.012792325755064407, -0.061045293690459676, -0.13538880942081147, -0.20934698970063492, -0.2824805732137822, -0.3536273534612184, -0.42233844840801643, -0.4878023463182307, -0.5489984801427885, -0.6057613758286579, -0.6565982452962243, -0.701260233755285, -0.7389547621249315, -0.7683863448981071, -0.7907574288449412, -0.8064028070096749, -0.8177427267136551, -0.8256938486044151, -0.8325858287457306, -0.8390856995702491, -0.8475788038261709, -0.8589638024794374, -0.8618048677741449, -0.8649427986574385, -0.868350745137283, -0.8720018572216433, -0.8758692849184841, -0.8799261782357707, -0.8841456871814676, -0.8885009617635398, -0.8932234754026177, -0.8985095478221862, -0.9040142029024567, -0.9099345096363662, -0.916406203320658, -0.9241535434115233, -0.9335826568446685, -0.9454673200891647, -0.9602424881442552, -0.9786189908026629, -1.0007042081482784, -1.02451692378966, -1.0483613843274824, -1.0685010051602222, -1.0826893955373307, -1.0838412255637169, -1.082920085697249, -1.0628213266424618, -1.0338106705270822, -0.9955703132584024, -0.9503000653788151, -0.9005512816901116, -0.8467158293833918, -0.7897970976690408, -0.7298153409295237, -0.6671113656264843, -0.6019822352642008, -0.5345505880245267, -0.4652539448549449, -0.39420923567016036, -0.3216807638627022, -0.24795735367106847, -0.17316139441533943, -0.0976458163980292, -0.021570932023350448, 0.05483916910998949, 0.13127599748168478, 0.20761493946949902, 0.28352015174166595, 0.35880633262699546, 0.43327916749898504, 0.5066024884015147, 0.5786674894475444, 0.6491393849285334, 0.7178113071540623, 0.7845327198500397, 0.8489140047234612, 0.9108862584851104, 0.9701070477222689, 1.0263314359482645, 1.0794971262870439, 1.1290897622207023, 1.175155362860969, 1.2172005279862064, 1.255816174860693, 1.2905611906484824, 1.3218636356009625, 1.349798480370815, 1.374506781666265, 1.3961305990136843, 1.414800097825901, 1.4306442432587214, 1.4437913069556252, 1.4543919691882305, 1.4625854069268043, 1.468500515771614, 1.4722603182423069, 1.4738546756247481, 1.4734038626065218, 1.470112139900382, 1.4639619840498446, 1.4546091317342857, 1.4417174447954193, 1.4249814626114456, 1.4042362155872725, 1.3802011093708269, 1.353257058113252, 1.3237822890331923, 1.2922080065674537, 1.2592528400711491, 1.2251340101666235, 1.1905120023093092, 1.1558089447278674, 1.1212799788901069, 1.0882785455561614, 1.0559185559451834, 1.0253221872471427, 0.9977828129557536, 0.9724194693930955, 0.9491434090957073, 0.9285049718966855, 0.8561284333799619, 0.8212568076313145, 0.8056291360658641, 0.7928895323124705, 0.7808848079601696, 0.7650390637211777, 0.7456937832988406, 0.7218311499055337, 0.6930480316216417, 0.6584610812542632, 0.6175064258502287, 0.5708770627198892, 0.518495719393311, 0.46164954500831784, 0.40037703618690584, 0.3354847948100945, 0.26758112120580313, 0.19697271207251588, 0.12457105350130862, 0.05067104425975604, -0.024054725316187096, -0.09896584832168753, -0.173724305280718, -0.2474695192061768, -0.31986007076960676, -0.3902686672405953, -0.45798358978992404, -0.5227763008072537, -0.5836507632076099, -0.6403551236234919, -0.692301759381108, -0.738544819769565, -0.7793231651433781, -0.813089114004263, -0.8398531686680367, -0.8601550788355445, -0.8682028075841017, -0.8700613439039103, -0.8644001498841054, -0.8443113226432083, -0.8142041835197227, -0.7835084195342937, -0.7582625355563639, -0.7358297514604144, -0.7180538029064889, -0.6945539243950556, -0.6474965775423123, -0.5097050515665223, -0.3051203769252373, -0.0012980832577012616]]
        prob['airfoils_cm'] = [[0.00037085425900144003, 0.05789068666447339, 0.10944205268392417, 0.15569918449408784, 0.19507878147596056, 0.2307373568338717, 0.2627043328445252, 0.29061030484228934, 0.3141404144065819, 0.3343575112310095, 0.351744183046338, 0.36656801137987616, 0.37898353201069507, 0.389368709427592, 0.3977114677049898, 0.4043744133861834, 0.4095105700795795, 0.41322783743125346, 0.41591949868740685, 0.4171591249818208, 0.4176916122685259, 0.41703207792602837, 0.41565329042575444, 0.41318811191431487, 0.4100491940633608, 0.4061107928112066, 0.40152487016880545, 0.39634605350131735, 0.3905917239655103, 0.3843951165497207, 0.377760180861634, 0.3707744751207791, 0.3634330709373342, 0.35573196738468327, 0.34767610157823114, 0.33925317105720204, 0.3304631189756253, 0.3213049704964981, 0.3117676420569007, 0.3018573539585031, 0.29156295234660873, 0.28088389362943783, 0.26981898699074613, 0.25835793743376406, 0.24650594427456407, 0.23425266225637845, 0.22159721083590359, 0.20853822999130642, 0.1950660104895885, 0.18118499556132284, 0.16688543831303493, 0.16394488716472386, 0.1609836148942787, 0.15800343702214772, 0.15500616906877923, 0.15199362655462142, 0.14896762500012267, 0.14592997992573112, 0.14288250685189507, 0.13981382511037682, 0.136714470305278, 0.13360203239923635, 0.13047217185388857, 0.127324805142607, 0.12416842382008624, 0.12097302118631227, 0.11774062777182148, 0.11422990349654172, 0.1103313784653508, 0.10577536525600711, 0.10046177667955168, 0.09417602124881466, 0.08713898031032849, 0.07934134419998852, 0.0709927683052058, 0.0621373319826026, 0.052828153689765144, 0.04319119584578551, 0.033247843990796985, 0.023109179059931782, 0.012758578714072153, 0.0021858611977625185, -0.00945760085804781, -0.022188624385069206, -0.036815899947827045, -0.049268640595802556, -0.058890245651517875, -0.05898523077715619, -0.056745304137224055, -0.05213815744307647, -0.05085541904427936, -0.052676662506736686, -0.05361806542016163, -0.054416731683135024, -0.056031135582720513, -0.05578354387509189, -0.051914539833571714, -0.048397199252834824, -0.046770522527351574, -0.05342936342274991, -0.062010780004164975, -0.07110245544478265, -0.07793366418455097, -0.0832277258318718, -0.08724222048350204, -0.09015152842157369, -0.09230739135094211, -0.09391950541638967, -0.09513384112014016, -0.09608408210350367, -0.0966081774336829, -0.09686804889072587, -0.09669193846847345, -0.09624834211334175, -0.09533927471299905, -0.09405979728306883, -0.09250727924923291, -0.09086665421765414, -0.08932536053128223, -0.0880729950111528, -0.08729544192260039, -0.0872177004241385, -0.08795853430696951, -0.08957326466803296, -0.09205662590450103, -0.09541662956226818, -0.09966271522417183, -0.10472829646191796, -0.1102893519564242, -0.1159655948884649, -0.12142234225751206, -0.12657532421543158, -0.1314515892368022, -0.13606206506327143, -0.14046320772999416, -0.1446574575130773, -0.14867834728452903, -0.15254695293373127, -0.1562988532834723, -0.1599504415103158, -0.16353609795605636, -0.16707712971975777, -0.17059283754544327, -0.17407961446890358, -0.17754737330784284, -0.18099375301011594, -0.1844152196283123, -0.18781277127707927, -0.1911871519247212, -0.19453808424783317, -0.210785980105559, -0.22646075099631718, -0.24161098233743097, -0.25620680408916585, -0.27026822802703887, -0.2838034945199159, -0.29681741653332483, -0.3093273709859146, -0.32132995800567804, -0.3328425182302101, -0.3438700881456854, -0.3544172333245589, -0.3645019194012123, -0.3741200035206051, -0.3832893993105802, -0.3920149906773047, -0.4003009772496097, -0.40816618140116595, -0.41560538000811365, -0.4226373157213831, -0.4292726756734257, -0.43549457922882145, -0.4413227311132443, -0.44660946100605026, -0.4513146733471013, -0.4553629623957365, -0.45856134583674807, -0.4610495507095331, -0.4622482940515648, -0.4627744297104864, -0.4619434132861477, -0.46015302368716005, -0.4567613721850611, -0.45216250280771847, -0.44596925306865653, -0.4382007385255059, -0.4286444839974412, -0.4169147640537381, -0.40294062796985575, -0.38631641663974964, -0.36684526985376253, -0.34435764086957305, -0.318434602711072, -0.2878041895243511, -0.25285271790714037, -0.21389528203634225, -0.17081084375904043, -0.12014086783452521, -0.06334457401672926, 0.00037085425900143835], [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0], [-1.8604272519832774e-07, 0.08250255519163899, 0.17022861183786944, 0.2676921429463001, 0.2774040655410892, 0.27590890169673815, 0.27441373785238715, 0.2768782339083548, 0.3035376538696982, 0.32453682909800013, 0.3413486757426983, 0.3548273351352087, 0.3664140105827809, 0.376725667396589, 0.3845086572781973, 0.3903082863832627, 0.3943009287800022, 0.3969065000601869, 0.3986649951625838, 0.39882407001737286, 0.39881973787632286, 0.3976930377590618, 0.3956453564653734, 0.39277823092945063, 0.38926378566626907, 0.3850110891581233, 0.38013346913843454, 0.374675973143719, 0.3686443648811435, 0.36214540769998316, 0.35518021856640447, 0.34782034865804956, 0.34010971682265384, 0.33206854647423173, 0.3237162289471403, 0.3150457115156353, 0.3060621385780459, 0.2967698878561025, 0.28716433422329835, 0.2772565391904764, 0.26704177280546204, 0.25652505500315026, 0.2457104471741165, 0.23459449144244549, 0.22318673711561002, 0.21148365200016092, 0.19950271115639195, 0.1872007624845397, 0.17455892715049307, 0.16125847923535686, 0.1471473997229389, 0.14415798979361077, 0.14112239087052397, 0.13803996111094707, 0.1349100586721486, 0.13173204171139702, 0.128505268385961, 0.12522909685310896, 0.12190288527010944, 0.11850819984007771, 0.11490184081671916, 0.1109921222088401, 0.10664362874317157, 0.10178171682637718, 0.09653615041197959, 0.09096850024711445, 0.0851701651173172, 0.07927330711042724, 0.07332169519017186, 0.06743779191260066, 0.061582756382804645, 0.05574728561049069, 0.04987950749280933, 0.04393555601836595, 0.03783454016214891, 0.03119556335544791, 0.02363449754228895, 0.01501965099151642, 0.005495240048834543, -0.004351843180740526, -0.013845759275179172, -0.022767513943594116, -0.029873725072084745, -0.03594586980762257, -0.041933006131934514, -0.04871306292818963, -0.056831172650869004, -0.0591392872954843, -0.060568840058096346, -0.0669434649300405, -0.07343661096526888, -0.07836131046765339, -0.08241701435081586, -0.08601391704663125, -0.08954472315946242, -0.09292449368040219, -0.09616045729747112, -0.0992558274473253, -0.10221683172070113, -0.10504891677286435, -0.10775607754960727, -0.110344924294947, -0.11281942016594015, -0.11518537989256404, -0.11744850098448203, -0.11961691717785786, -0.12168182934615743, -0.1236125184105281, -0.1253294452916634, -0.12684125054987874, -0.12784232446523724, -0.128399348609824, -0.12771068953759876, -0.1259463436053513, -0.12232074515442509, -0.11758362611197425, -0.11212757405226333, -0.10638604545930272, -0.1007967146043902, -0.09580666159531297, -0.09186657891752613, -0.08945681044031022, -0.08883804190884649, -0.08979200230187494, -0.09222272044469292, -0.09576693511806345, -0.10018926609996552, -0.10524864721062942, -0.11068455688047954, -0.1162504240145118, -0.12182761066746967, -0.12738837456406818, -0.13291607573337133, -0.13833735543618828, -0.14349883480876532, -0.1479149862831316, -0.15182403498832123, -0.15530408880487828, -0.15868683212144172, -0.16201661250221785, -0.16534049076791882, -0.16864663244508551, -0.1719354650360508, -0.17520190419430762, -0.17845336714254606, -0.18168832339891758, -0.1849040086662691, -0.18810116996476828, -0.19128033814981796, -0.19444124852384306, -0.2098225003493435, -0.22475175528519087, -0.23926318889633566, -0.2533278088922402, -0.2669571517970723, -0.28015298526987314, -0.29291379171712867, -0.3052497748972952, -0.3171520048470734, -0.3286305955110283, -0.33968458161726783, -0.35031183198174887, -0.36052400828490105, -0.3703103406841057, -0.3796823600046429, -0.38863873269781424, -0.3971763976198838, -0.4053091810559625, -0.41302640997676904, -0.42033322561684794, -0.42723253133078554, -0.43362974841246793, -0.4395256764918041, -0.4447738227436484, -0.44935956225706797, -0.45323069211488004, -0.456210823586161, -0.45847551049426966, -0.4593950542609165, -0.4597600981166992, -0.45881999876714635, -0.4568867705868882, -0.4535167810015536, -0.4490273274660211, -0.44308915095763024, -0.4357304276035835, -0.42677773881671804, -0.415913467557285, -0.4028632297271768, -0.3867213388584255, -0.3647599049319651, -0.34011332057873733, -0.31409664493282696, -0.30115146621493705, -0.295006891787009, -0.28766778505116636, -0.26454528345124734, -0.16953181938448328, -0.0825480838884032, -1.8604272519970356e-07], [2.6328884904914362e-05, 0.1381971249096504, 0.21824819695032566, 0.2637166839182505, 0.270415608198343, 0.2716058470114898, 0.2727415699437516, 0.27786531537930054, 0.30201137627713437, 0.32369452621112244, 0.3424922390436657, 0.3566529653355435, 0.368067242601738, 0.37746525140324066, 0.3849401317779571, 0.3908144525541236, 0.3952090472068166, 0.39817897912471434, 0.4001806462965743, 0.4007110834513769, 0.40087180257147104, 0.399959551999428, 0.3982285882769596, 0.3956198035732105, 0.39236561672749154, 0.3883596369105577, 0.38371516684224116, 0.37847148965880084, 0.37262809332554314, 0.36628944594207014, 0.3594499769534858, 0.35217517452326813, 0.3445131443689445, 0.33648295300925424, 0.3281542477609032, 0.3195205788791624, 0.3105917487750207, 0.30136509456395805, 0.2918337652508463, 0.28200259256816773, 0.2718646819633736, 0.261420295142471, 0.2506692753334402, 0.2396052830556648, 0.22823249019082892, 0.21654452428206855, 0.20454143312210354, 0.1922360649393193, 0.17963560287108607, 0.1665604192401164, 0.152471649676097, 0.14933433896076356, 0.14610778213809486, 0.14278535741182558, 0.13936044298569042, 0.13582641706342402, 0.13217665784876115, 0.12840454354543637, 0.1245034523571845, 0.12040070562569327, 0.1157619642023399, 0.11035198331133304, 0.10389124810200087, 0.09613123115512166, 0.08744476315982272, 0.07795230077695925, 0.06788977791842399, 0.05758208081694831, 0.04712899200120553, 0.03681840518051913, 0.026404641386980263, 0.015749042891016727, 0.004386081313664227, -0.007667161401578232, -0.020771518791845127, -0.03144954896647892, -0.03896273617703439, -0.0408779495500504, -0.03974276488729706, -0.0366504817800821, -0.03530189757797798, -0.03545340485117555, -0.03807599674067127, -0.04644644745851975, -0.05699531877482473, -0.0690393552210253, -0.08161869582586637, -0.09304248645406041, -0.102525369028302, -0.10894829449867799, -0.1134491141050414, -0.11666779888036626, -0.11934197304856427, -0.12166213440336034, -0.12381502446227553, -0.1258368185321159, -0.12775170617955284, -0.12956826920593298, -0.1312992805414803, -0.1329561233857192, -0.13456018259234398, -0.13610290457796073, -0.13753951656849664, -0.1387430794638843, -0.13973432800955563, -0.14004056264753884, -0.14012625177564006, -0.1395574330838514, -0.13829092192175127, -0.13632529139306268, -0.13386734055510524, -0.13086218672011557, -0.12744167558192615, -0.12366832071086475, -0.11956998928725886, -0.11524511239227737, -0.1107886715671765, -0.10632209733482106, -0.10196740940503819, -0.0978495540135114, -0.09409410727770015, -0.09082812399388668, -0.08816891493452622, -0.08620114082897735, -0.08494055062349258, -0.08441293415495073, -0.08459733805292986, -0.08552536016021928, -0.08718901674391442, -0.08959197269474246, -0.09271735253934167, -0.09641793002991937, -0.10048359503744708, -0.10469871949099574, -0.1089478825933295, -0.1131971072483771, -0.1174271651069164, -0.12161971324544088, -0.12574565397894386, -0.1298020092362855, -0.1337857591633483, -0.13770041255193774, -0.14155006074367224, -0.14531844241621342, -0.14903303345713712, -0.15269063059359353, -0.15628361508860772, -0.15981452534344834, -0.16328734102306863, -0.1667042002125977, -0.1828400235378734, -0.1978217872066428, -0.21188449300581927, -0.2252390367149256, -0.23801755360382246, -0.25044679679001297, -0.2625252303213085, -0.2742798708033924, -0.28569574286793126, -0.29677845725655116, -0.3075257877940818, -0.3179346787228811, -0.32801184156253776, -0.3377481061167311, -0.34715009440519073, -0.35621530024493303, -0.3649407804450507, -0.37333543236199845, -0.3813808488758216, -0.389053164883091, -0.3963026105411294, -0.40304658333366894, -0.40928371931396085, -0.41489122506242726, -0.41986176880410986, -0.4241463651174929, -0.4276082147542318, -0.4303666550368482, -0.43210942275956193, -0.4330027767168922, -0.43270081774467856, -0.431872933764268, -0.42950894951379404, -0.4261682715324198, -0.42141474582412075, -0.415089867601574, -0.40700567498650925, -0.3967253979141012, -0.3843067825046784, -0.3691102303266816, -0.3499319542290574, -0.327280302432205, -0.30088317923252905, -0.2900114429438742, -0.2854189891122763, -0.280381464633628, -0.26698195891054116, -0.21914223683618556, -0.13713110653060476, 2.632888490490884e-05], [-5.556297131044114e-07, 0.11057298082017548, 0.2274908204118889, 0.3567102161985245, 0.3594690163777208, 0.319028735091672, 0.27858845380562364, 0.2707065342417513, 0.29635782564652463, 0.31704549446456287, 0.3336358091573034, 0.34595167477218175, 0.35672711778391836, 0.3668282495293583, 0.37438395393667584, 0.3799252696846753, 0.38360616498495975, 0.3858579609015151, 0.38726168744386896, 0.38729587508238617, 0.38690227606925043, 0.3854824624075054, 0.3832094472002611, 0.38012042548089015, 0.37639502656354373, 0.3719567916952665, 0.36691393530370886, 0.3613138599914274, 0.35516544195745003, 0.3485733021393408, 0.3415421883269903, 0.3341436142204929, 0.32642193718964, 0.31839895824774345, 0.31008913652162745, 0.3014851108992356, 0.29258842977206406, 0.2834005324230888, 0.2739157133201641, 0.2641408492810213, 0.25406928048903554, 0.24370280540447475, 0.23303321572227423, 0.22205069253162052, 0.2107486593017974, 0.1991163636742011, 0.1871257178816131, 0.17483965403903873, 0.16226982321144306, 0.1501195457112718, 0.13800592126559721, 0.13549351438228704, 0.13302356194642964, 0.13055562977865534, 0.12804928369959426, 0.12546408952987667, 0.12275961309013277, 0.11989542020099275, 0.11683107668308687, 0.1133807066967645, 0.10942599204187578, 0.10513662856029513, 0.1004583977117157, 0.09535755458627647, 0.08983839424129282, 0.08387852512174668, 0.07745718406679789, 0.07058535943846994, 0.06324956983463553, 0.05552440765539992, 0.04744451921009444, 0.03904868868896783, 0.030410888814322738, 0.021552099080162746, 0.012534101081863787, 0.003423513961066673, -0.005736596652242856, -0.01475875434683446, -0.023522939153895216, -0.031333960794965525, -0.035228149346871386, -0.03623622362274906, -0.0393192336773043, -0.04520041432274912, -0.050587582666446515, -0.05777883033740838, -0.06836045328790671, -0.07675731396351623, -0.08395162099673711, -0.09059181190110335, -0.09809447832578383, -0.10687033008791907, -0.11262811573522172, -0.11672296976191013, -0.11964002131734343, -0.12226563976076203, -0.12477100467490906, -0.12714398337588034, -0.12942170339420317, -0.13163729590857112, -0.13385278210472573, -0.13606814060632835, -0.13824850757843832, -0.14026743520637916, -0.14207439867822977, -0.14347807160572112, -0.14441278390142637, -0.1444754914914647, -0.14441940941052045, -0.14320167045761015, -0.14133167710408184, -0.13852897241287823, -0.13484093595280397, -0.13039280892487684, -0.12535151359136457, -0.12020572068211051, -0.1152898744138353, -0.11076876763044248, -0.10673044037078813, -0.10326582556129367, -0.10046572463111778, -0.09840787926950101, -0.09713735263152157, -0.0967907634824816, -0.09715914379527152, -0.09836743234711633, -0.10023114339081989, -0.10267501839036189, -0.10560015839652166, -0.10890999049255608, -0.11250935351256165, -0.1163162309819257, -0.12028929885996109, -0.12440479503713436, -0.1286221018859301, -0.1329178082588883, -0.13726300327489388, -0.1416339631076425, -0.1459877405991483, -0.1503106039693882, -0.15456494636247395, -0.15875367593339285, -0.16287632567052246, -0.16691718320246277, -0.1709040022996136, -0.17483286475216137, -0.17869538365287646, -0.18249459245669183, -0.18623395244688484, -0.1899145889816434, -0.20731252530866634, -0.2234434777224668, -0.23852512354619176, -0.2526838603270497, -0.26606320693021185, -0.2788400072140643, -0.2911727167808485, -0.30314994922921257, -0.3147784960427098, -0.3260583976696464, -0.33697847671049214, -0.3475275096471934, -0.35771015664728845, -0.36750555837252524, -0.37691813817981373, -0.38593810260985584, -0.3945524785783647, -0.402769918573007, -0.41056686724581914, -0.41794456984976114, -0.42489651267167683, -0.4313321627085689, -0.43725443571881994, -0.4425219199557379, -0.4471236648739901, -0.4510101189239261, -0.4540120789095939, -0.4563004199052985, -0.4572858490770028, -0.4576911077082063, -0.45681789892863095, -0.4549999470387703, -0.45176078097803624, -0.4474382405490384, -0.4416950108734169, -0.43454355731517913, -0.4256512043150258, -0.4145109763526803, -0.4008982833547224, -0.3842722291818607, -0.3640237471692663, -0.339611456590784, -0.3099081914690557, -0.3133897204918505, -0.3386578306701346, -0.3639259408484189, -0.35715251572936935, -0.2276698727197008, -0.11061036770396891, -5.55629713106652e-07], [0.00040906107437695263, 0.058069816173762126, 0.10974488541029707, 0.1561114968777322, 0.19556896958866343, 0.2312937891597676, 0.26330530389071644, 0.29122291002967776, 0.3147393415073346, 0.3349069272237839, 0.3522136760496807, 0.36692274505239636, 0.379194880536038, 0.38941949207845844, 0.3975873342603854, 0.4040737280669798, 0.4090379540749483, 0.4125952828046121, 0.4151492917008358, 0.416229758165464, 0.41667155426318997, 0.41593033696172216, 0.41442634084618296, 0.4118748581708537, 0.4086472834475477, 0.4046235837815199, 0.3999501311262714, 0.39468047601157, 0.3888300175920895, 0.3825287709587171, 0.3757795624246074, 0.3686674337005617, 0.3611911376416861, 0.3533478806299063, 0.3451477098977642, 0.33658011002764965, 0.3276488385042223, 0.3183561595818624, 0.30869354137390215, 0.2986711882750558, 0.28828045477878567, 0.2775242994554303, 0.2664046548812471, 0.2549142373402885, 0.2430616323825608, 0.2308394769543901, 0.21825019229461434, 0.20529545146483336, 0.19196888399120218, 0.1782778954754791, 0.1642160568308371, 0.16132981308798805, 0.15842536396657222, 0.1555043226506305, 0.15256830232420365, 0.14961891617133258, 0.1466577773760581, 0.14368649912242104, 0.14070669459446222, 0.1377084763165975, 0.13468361145952448, 0.13164743658624584, 0.1285961980814457, 0.12552984233685674, 0.12245108341478293, 0.11935050355389129, 0.1162296569088021, 0.11301084455600571, 0.10966322999518978, 0.1061034226426045, 0.10228450929066248, 0.09814479603456149, 0.09362098015098788, 0.08865819911806917, 0.083195678816228, 0.07720125456074069, 0.07061409900091714, 0.06347067771481071, 0.055668894034384644, 0.047187473303917093, 0.03802578914560152, 0.02805799374801744, 0.016999432138984957, 0.004745814930318771, -0.008951483207208571, -0.023215466163952606, -0.037647327813644604, -0.05106098112337172, -0.06280751252482525, -0.0726077373404409, -0.07775592897103129, -0.08012452356569214, -0.07998913011548706, -0.07882800717488395, -0.06853376005277628, -0.06605314934203758, -0.07146589437783589, -0.07488742828682456, -0.07805494413876873, -0.08205434540789118, -0.08565646965346849, -0.08881865873961278, -0.09200560931388732, -0.09512609824371933, -0.09815760095382417, -0.10103259945254799, -0.1037334162099301, -0.10620751566001492, -0.10838906973459407, -0.1102813536774307, -0.11174420924600414, -0.1127561474325778, -0.11313661182811498, -0.11316470673571062, -0.11241224393744635, -0.1109746742340481, -0.10888258156558714, -0.10622756001882325, -0.10310873913028738, -0.09963066583495765, -0.09589602456766501, -0.09200783184462999, -0.08816495298948906, -0.08489605120149779, -0.08279086008276819, -0.08272082846135273, -0.08488691404087768, -0.08982731906377635, -0.09651260101519037, -0.10373916591004519, -0.11046123765850156, -0.11645269203940893, -0.12184489724059233, -0.12675285102910594, -0.13134669937578983, -0.1357309120721057, -0.13994157979935215, -0.1440134366268746, -0.14797056274040318, -0.15182662299081834, -0.1556094852988063, -0.1593368850035461, -0.16302534842049954, -0.16666727042201898, -0.17027840690322535, -0.17385588834221413, -0.17739463949025194, -0.18089612408149908, -0.18436200081291212, -0.18779257977303726, -0.20427082628251667, -0.21993757194159305, -0.23489595852731765, -0.24916550390925746, -0.26280518699566785, -0.2758743952231884, -0.28843815345587276, -0.30053950083939773, -0.3122442447559567, -0.32356116141721947, -0.3344948340794137, -0.345033192309697, -0.3551832746509782, -0.36492650366869267, -0.374269695973297, -0.38320532103948984, -0.3917228791018342, -0.3998331356143747, -0.4075128200009033, -0.414772312269757, -0.4216081094004227, -0.4279906739467117, -0.43393737579493097, -0.439306510029126, -0.44406751492893154, -0.44815286643363883, -0.45138132743555864, -0.45389446892881313, -0.4551549958568016, -0.4557214074694217, -0.45496395358625896, -0.45330699877446984, -0.45008293926412807, -0.44570432145375777, -0.4397953185041964, -0.43237933796229605, -0.42322586967249126, -0.4119355362464175, -0.3984117959419042, -0.38223293762895627, -0.36317672373992116, -0.3410663792286453, -0.31545956110683904, -0.28515032898297027, -0.2505019935499308, -0.21186456055708286, -0.1691604143611503, -0.11896233947304258, -0.06270043053271526, 0.00040906107437694704], [-1.1693462894286646e-07, 0.11268070205225389, 0.23178417847499982, 0.36337527300616873, 0.3663603087040572, 0.3258507356343704, 0.2853411625646839, 0.2776113280562874, 0.3047178080242563, 0.3258609934600095, 0.34282438489817885, 0.35706196825978825, 0.36913598127421077, 0.3794974902050352, 0.3873354656345861, 0.3933563173779664, 0.39802267433130983, 0.40132809363065713, 0.40367353592851496, 0.4044564915470801, 0.40473326029823686, 0.4038306823997621, 0.40208957238398946, 0.3993889666888625, 0.3960340401610914, 0.39192791848214936, 0.38721631079875124, 0.38196132025573476, 0.3761884047058922, 0.370018734047005, 0.36343861755446577, 0.3564750122942935, 0.3491329532329666, 0.34140482091418156, 0.3333129564396034, 0.32484939407603547, 0.3160251174303001, 0.3068484174633637, 0.2973149089498805, 0.2874426774752337, 0.2772271365586418, 0.2666779758032403, 0.25580233031686983, 0.2445976711640422, 0.23307477700012386, 0.22122905911623642, 0.20906507398918114, 0.19662075688827807, 0.18392680077730275, 0.17044014090782192, 0.1549253921660906, 0.15110912495285675, 0.14700133596125228, 0.1426142939243852, 0.13796026757536337, 0.13305152564729467, 0.127900336873287, 0.12251896998644823, 0.11691969371988632, 0.11078639018702408, 0.1038748581509624, 0.09659302277777579, 0.08884174818110639, 0.08061525320355804, 0.07208036604734297, 0.0633274247729396, 0.05446638992851955, 0.04563878360598553, 0.03695600780423872, 0.02885352065376051, 0.02158677588703837, 0.01567753537586061, 0.007899461129103905, -0.002386996866096861, -0.01426681597426834, -0.026741376685651845, -0.038901212414818614, -0.05001507152303662, -0.0594766476187754, -0.06678321191662677, -0.07097414683333719, -0.07295637788274872, -0.07352378125572734, -0.0738464384578254, -0.07520054855903437, -0.07712590074117615, -0.0793597764457424, -0.08109450370344097, -0.08244476259947302, -0.08342032066506773, -0.08426468037084632, -0.08504610291370517, -0.08600348447591735, -0.08719798810767806, -0.08881902431609111, -0.09076274772960242, -0.09304404174177452, -0.09554265231141501, -0.09823104402169604, -0.10105127628542157, -0.10392623273866934, -0.10682853172799553, -0.10969389603080201, -0.1124824987676838, -0.11516368828119045, -0.11768050063476447, -0.11998288175106105, -0.12215656266055638, -0.12307677993296706, -0.12314464602035673, -0.1224133834645201, -0.11991334347498317, -0.11798980459432779, -0.11642530772378472, -0.11538590664437756, -0.11477238311419129, -0.11457391427339063, -0.11473466300483998, -0.11525952863262943, -0.11608386647450171, -0.11604605634224115, -0.11420618967414947, -0.11197475624085673, -0.11113412036746524, -0.11115720396472871, -0.11126741301768613, -0.11086123142017083, -0.10997152842083999, -0.10893907287970017, -0.1080816993869581, -0.10775472708272504, -0.10821947414932379, -0.10985145521144367, -0.11241461782674914, -0.11513951364594648, -0.11704323806519397, -0.11919681301635954, -0.12287057989879545, -0.1290720637713571, -0.1374935107059502, -0.14175005804177507, -0.14482726460785614, -0.14730521544795044, -0.1499424285309044, -0.15274844282937555, -0.15561181061771592, -0.15846433255492998, -0.1613252699628873, -0.16419553395943554, -0.16705333559228203, -0.18100720388570377, -0.19469372058283782, -0.20812476708214578, -0.22126919069061327, -0.23412610892042135, -0.24668517599372866, -0.2589331015374286, -0.27086958560955776, -0.28247339935606836, -0.2937440699301045, -0.30466936078770773, -0.3152347279056135, -0.32544276420937013, -0.3352685965896385, -0.34471453822644316, -0.35376772736793904, -0.3624115140829081, -0.370653252471882, -0.3784620790677436, -0.3858448334145487, -0.39278788207729165, -0.39927136883653713, -0.40531253116455873, -0.41086154163967264, -0.415893622609862, -0.420351470908658, -0.42409166462303455, -0.42718536080227654, -0.42944420449579707, -0.4307004235475656, -0.4306999281592769, -0.4304948553602446, -0.42824980789244754, -0.4248641904498031, -0.4197454559420601, -0.4128877265555987, -0.4042192722388355, -0.39346842857563513, -0.38153352060677026, -0.3675456702404107, -0.3487702148959045, -0.32561071469254466, -0.2973170795209489, -0.3029141169139793, -0.3356932133434087, -0.3684723097728383, -0.3634403860977337, -0.23171650069208966, -0.11259484944328856, -1.169346289371737e-07]]
        prob['airfoils_position'] = [0.0, 0.022222764227642276, 0.17137860377801986, 0.23333333333333334, 0.3666666666666667, 0.47222384517197835, 0.6005347184880255, 0.7, 1.0]
        prob['airfoils_r_thick'] = [0.4, 1.0, 0.3, 0.21, 0.25, 0.35, 0.18]
        prob['blade_Rtip'] = [63.0]
        prob['blade_chord'] = [3.542, 3.5829065688610924, 3.8701677888975055, 4.113082662273744, 4.386184316700438, 4.5705983562554815, 4.6343256119198495, 4.646781924312676, 4.545529346958071, 4.426741607305162, 4.320291925514559, 4.205548041683577, 4.080209090674618, 3.9498741541952938, 3.8144707942245812, 3.6834791267944174, 3.5568910997434435, 3.4298965523912854, 3.3026551714743477, 3.1754137920031074, 3.0481724135647097, 2.9209310350823983, 2.793689656426132, 2.666448158844707, 2.5392068063848345, 2.416417405464169, 2.280796764281841, 2.1030278893097076, 1.5245990923819477, 1.419]
        prob['blade_precurve'] = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        prob['blade_precurveTip'] = [0.0]
        prob['blade_presweep'] = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        prob['blade_presweepTip'] = [0.0]
        prob['blade_r'] = [1.5, 3.620689655172414, 5.741379310344827, 7.862068965517241, 9.982758620689655, 12.10344827586207, 14.224137931034482, 16.344827586206897, 18.46551724137931, 20.586206896551726, 22.70689655172414, 24.827586206896548, 26.948275862068964, 29.06896551724138, 31.189655172413794, 33.310344827586206, 35.43103448275861, 37.551724137931025, 39.672413793103445, 41.793103448275865, 43.91379310344828, 46.03448275862069, 48.1551724137931, 50.27586206896552, 52.39655172413793, 54.51724137931034, 56.63793103448276, 58.75862068965517, 60.87931034482758, 63.0]
        prob['blade_theta'] = [13.308000180172, 13.308000180172, 13.308000180172, 13.308000180172, 13.308000180172, 13.27998884375364, 12.335570816951353, 11.299851335869668, 10.608203094087314, 9.975417328120727, 9.387115627540592, 8.785040906252279, 8.155968680944982, 7.51563610678596, 6.863446722622106, 6.233104839458657, 5.624189458687656, 5.012955961696406, 4.403133829816835, 3.821482956674538, 3.272695988895811, 2.8162941264888617, 2.4142451382835444, 2.003730085914454, 1.594414027181603, 1.1799586856042907, 0.7722910916991717, 0.3885197488983106, 0.13623584606856495, 0.10600005688817796]
        prob['gear_ratio'] = [96.0]
        prob['hub_radius'] = [1.5]
        prob['mu_air'] = [1.81e-05]
        prob['precone'] = [2.499814860155782]
        prob['rated_rotor_speed'] = [12.126090902239644]
        prob['rho_air'] = [1.225]
        prob['rho_water'] = [1025.0]
        prob['rotor_PC_GS_Ki'] = [-0.001052878214281871, -0.0009007028558454134, -0.0007869612073112069, -0.0006987254638421395, -0.0006282812504249029, -0.0005707402685070013, -0.0005228547466892069, -0.00048238250519572174, -0.0004477257220980474, -0.00041771497902817734, -0.00039147470258697113, -0.0003683363248133816, -0.000347780521258279, -0.000329397767908994, -0.0003128607792452187, -0.0002979048516854225, -0.00028431358775948384, -0.00027190835759670785, -0.00026054040398058983, -0.00025008484836107754, -0.00024043608442564813, -0.00023150419840723912, -0.00022321215869871322, -0.00021549358855808422, -0.00020829098548095728, -0.00020155428611512164, -0.00019523970093777277, -0.00018930876133192357, -0.00018372753522813583, -0.00017846597752015916]
        prob['rotor_PC_GS_Kp'] = [-0.01155677935786742, -0.009275261359111076, -0.00756996806455072, -0.006247076901837379, -0.005190928636632085, -0.004328234523336314, -0.003610301731656665, -0.0030035139549355306, -0.0024839155440709035, -0.002033973771805906, -0.0016405614374564378, -0.0012936549087327889, -0.0009854681150775675, -0.0007098611898564572, -0.0004619272426702107, -0.00023769765473716088, -3.3928045836751384e-05, 0.00015205972633535762, 0.0003224959326350152, 0.00047925283847056744, 0.0006239137665644245, 0.0007578267661671128, 0.0008821467482968329, 0.000997868878879897, 0.001105855275148019, 0.00120685652142918, 0.001301529140470962, 0.0013904498803286708, 0.0014741274740058203, 0.001553012378471194]
        prob['rotor_PC_GS_angles'] = [0.0, 0.0715621225776534, 0.08773665371850184, 0.10400288289241569, 0.12040369272119619, 0.1369853452701298, 0.15379823498271342, 0.17089776848091118, 0.18834540734482655, 0.20620991951915665, 0.22157752935892003, 0.23129844724009277, 0.2407994651904149, 0.2500952852438235, 0.25920175984272564, 0.2681356942806361, 0.27691470861377565, 0.2855571512182247, 0.29408205784466557, 0.3025091515736351, 0.31085888052390886, 0.3191524915482118, 0.3274121395049951, 0.33566103307153744, 0.3439236195205175, 0.3522258124900162, 0.3605952686230813, 0.36906172114856783, 0.37648737812646105, 0.3835216188484712]
        prob['rotor_TC_VS_Ki'] = [-154.91326410058622]
        prob['rotor_TC_VS_Kp'] = [-1492.2961374522677]
        prob['rotor_inertia'] = [35692016.048775055]
        prob['rotor_powercurve_omega_rpm'] = [6.899939740828794, 6.899939740828794, 6.899939740828794, 6.899939740828794, 7.007976800069685, 7.255399946448759, 7.3049692350849975, 7.650769631937398, 8.287490043856911, 9.205351183577271, 10.390255768228583, 11.824005037256768, 12.126090902239644, 12.126090902239644, 12.126090902239644, 12.126090902239644, 12.126090902239644, 12.126090902239644, 12.126090902239644, 12.126090902239644]
        prob['rotor_powercurve_pitch'] = [1.5586112272169546, 1.8124253382752327, 1.062365830470572, -0.7035977579761596, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 5.000882520589389e-18, 5.616929278126907, 9.473573880731745, 12.584867884300586, 15.421668799205388, 18.099376582959863, 20.648360999150785, 23.0697395321926]
        prob['rotor_powercurve_v'] = [3.0, 4.272046072021933, 5.304031475548528, 6.080106111341953, 6.588350372834852, 6.820958217466749, 6.867559376486633, 7.19265379938859, 7.7912484127754045, 8.6541494974404, 9.768103893234573, 11.116002552239781, 11.495708522428979, 12.677143313907282, 14.42754886624012, 16.340335009504354, 18.386123566365413, 20.53349359662874, 22.749463986449395, 25.0]
        prob['tilt'] = [4.999629720311564]

        # -------------------------
        # platform
        # -------------------------
        opt['members']['platform_member1_potMod'] = True
        opt['members']['platform_member2_potMod'] = False
        opt['members']['platform_member3_potMod'] = False
        opt['members']['platform_member4_potMod'] = False
        # member 1
        prob['platform_member1_Ca'] = [1.0, 1.0, 1.0, 1.0, 1.0, 1.0]
        prob['platform_member1_CaEnd'] = [0.6, 0.6, 0.6, 0.6, 0.6, 0.6]
        prob['platform_member1_Cd'] = [0.8, 0.8, 0.8, 0.8, 0.8, 0.8]
        prob['platform_member1_CdEnd'] = [0.6, 0.6, 0.6, 0.6, 0.6, 0.6]
        prob['platform_member1_cap_d_in'] = [0.0, 0.0, 0.0, 0.0, 0.0]
        prob['platform_member1_cap_stations'] = [0.0, 0.1, 0.5, 0.8308, 1.0]
        prob['platform_member1_cap_t'] = [0.1, 0.05, 0.05, 0.05, 0.05]
        prob['platform_member1_d'] = [9.4, 9.4, 9.4, 9.4, 6.5, 6.5]
        prob['platform_member1_gamma'] = [0.0]
        prob['platform_member1_heading'] = []
        prob['platform_member1_l_fill'] = [0.10324412152256533, 0.0, 0.054227176236731345, 0.0, 0.0]
        prob['platform_member1_rA'] = [0.0, 0.0, -120.0]
        prob['platform_member1_rB'] = [0.0, 0.0, 10.0]
        prob['platform_member1_rho_fill'] = [5000.0, 0.0, 1025.0, 0.0, 0.0]
        prob['platform_member1_rho_shell'] = [7800.0]
        prob['platform_member1_ring_h'] = [0.0]
        prob['platform_member1_ring_spacing'] = [0.0]
        prob['platform_member1_ring_t'] = [0.0]
        prob['platform_member1_s_ghostA'] = [0.0]
        prob['platform_member1_s_ghostB'] = [1.0]
        prob['platform_member1_stations'] = [0.0, 0.38462, 0.4, 0.8308, 0.8923, 1.0]
        prob['platform_member1_t'] = [0.08, 0.08, 0.08, 0.08, 0.08, 0.08]
        prob['platform_member2_Ca'] = [1.0, 1.0]
        prob['platform_member2_CaEnd'] = [0.6, 0.6]
        prob['platform_member2_Cd'] = [0.8, 0.8]
        prob['platform_member2_CdEnd'] = [0.6, 0.6]
        prob['platform_member2_cap_d_in'] = [0.0, 0.0]
        prob['platform_member2_cap_stations'] = [0.0, 1.0]
        prob['platform_member2_cap_t'] = [0.02, 0.02]
        prob['platform_member2_d'] = [1.0, 1.0]
        prob['platform_member2_gamma'] = [0.0]
        prob['platform_member2_heading'] = []
        prob['platform_member2_l_fill'] = [0.0]
        prob['platform_member2_rA'] = [5.2, 0.0, -70.0]
        prob['platform_member2_rB'] = [0.0, 0.0, -69.9994]
        prob['platform_member2_rho_fill'] = [0.0]
        prob['platform_member2_rho_shell'] = [7800.0]
        prob['platform_member2_ring_h'] = [0.0]
        prob['platform_member2_ring_spacing'] = [0.0]
        prob['platform_member2_ring_t'] = [0.0]
        prob['platform_member2_s_ghostA'] = [0.0]
        prob['platform_member2_s_ghostB'] = [0.09615385217057337]
        prob['platform_member2_stations'] = [0.0, 1.0]
        prob['platform_member2_t'] = [0.1, 0.1]
        prob['platform_member3_Ca'] = [1.0, 1.0]
        prob['platform_member3_CaEnd'] = [0.6, 0.6]
        prob['platform_member3_Cd'] = [0.8, 0.8]
        prob['platform_member3_CdEnd'] = [0.6, 0.6]
        prob['platform_member3_cap_d_in'] = [0.0, 0.0]
        prob['platform_member3_cap_stations'] = [0.0, 1.0]
        prob['platform_member3_cap_t'] = [0.02, 0.02]
        prob['platform_member3_d'] = [1.0, 1.0]
        prob['platform_member3_gamma'] = [0.0]
        prob['platform_member3_heading'] = []
        prob['platform_member3_l_fill'] = [0.0]
        prob['platform_member3_rA'] = [-2.5999999999999988, 4.503332099679082, -70.0]
        prob['platform_member3_rB'] = [0.0, 0.0, -69.9994]
        prob['platform_member3_rho_fill'] = [0.0]
        prob['platform_member3_rho_shell'] = [7800.0]
        prob['platform_member3_ring_h'] = [0.0]
        prob['platform_member3_ring_spacing'] = [0.0]
        prob['platform_member3_ring_t'] = [0.0]
        prob['platform_member3_s_ghostA'] = [0.0]
        prob['platform_member3_s_ghostB'] = [0.0961538521705736]
        prob['platform_member3_stations'] = [0.0, 1.0]
        prob['platform_member3_t'] = [0.1, 0.1]
        prob['platform_member4_Ca'] = [1.0, 1.0]
        prob['platform_member4_CaEnd'] = [0.6, 0.6]
        prob['platform_member4_Cd'] = [0.8, 0.8]
        prob['platform_member4_CdEnd'] = [0.6, 0.6]
        prob['platform_member4_cap_d_in'] = [0.0, 0.0]
        prob['platform_member4_cap_stations'] = [0.0, 1.0]
        prob['platform_member4_cap_t'] = [0.02, 0.02]
        prob['platform_member4_d'] = [1.0, 1.0]
        prob['platform_member4_gamma'] = [0.0]
        prob['platform_member4_heading'] = []
        prob['platform_member4_l_fill'] = [0.0]
        prob['platform_member4_rA'] = [-2.6000000000000023, -4.5033320996790795, -70.0]
        prob['platform_member4_rB'] = [0.0, 0.0, -69.9994]
        prob['platform_member4_rho_fill'] = [0.0]
        prob['platform_member4_rho_shell'] = [7800.0]
        prob['platform_member4_ring_h'] = [0.0]
        prob['platform_member4_ring_spacing'] = [0.0]
        prob['platform_member4_ring_t'] = [0.0]
        prob['platform_member4_s_ghostA'] = [0.0]
        prob['platform_member4_s_ghostB'] = [0.09615385217057337]
        prob['platform_member4_stations'] = [0.0, 1.0]
        prob['platform_member4_t'] = [0.1, 0.1]


        # -------------------------
        # mooring
        # -------------------------
        opt['mooring']['mooring_point1_name'] = 'line1_anchor'
        opt['mooring']['mooring_point2_name'] = 'line2_anchor'
        opt['mooring']['mooring_point3_name'] = 'line3_anchor'
        opt['mooring']['mooring_point4_name'] = 'line1_vessel'
        opt['mooring']['mooring_point5_name'] = 'line2_vessel'
        opt['mooring']['mooring_point6_name'] = 'line3_vessel'
        opt['mooring']['mooring_point1_type'] = 'fixed'
        opt['mooring']['mooring_point2_type'] = 'fixed'
        opt['mooring']['mooring_point3_type'] = 'fixed'
        opt['mooring']['mooring_point4_type'] = 'vessel'
        opt['mooring']['mooring_point5_type'] = 'vessel'
        opt['mooring']['mooring_point6_type'] = 'vessel'
        opt['mooring']['mooring_line1_endA'] = 'line1_anchor'
        opt['mooring']['mooring_line2_endA'] = 'line2_anchor'
        opt['mooring']['mooring_line3_endA'] = 'line3_anchor'
        opt['mooring']['mooring_line1_endB'] = 'line1_vessel'
        opt['mooring']['mooring_line2_endB'] = 'line2_vessel'
        opt['mooring']['mooring_line3_endB'] = 'line3_vessel'
        opt['mooring']['mooring_line1_type'] = 'main'
        opt['mooring']['mooring_line2_type'] = 'main'
        opt['mooring']['mooring_line3_type'] = 'main'
        opt['mooring']['mooring_line_type1_name'] = 'main'
        prob['mooring_line1_length'] = [902.2]
        prob['mooring_line2_length'] = [902.2]
        prob['mooring_line3_length'] = [902.2]
        prob['mooring_line_type1_breaking_load'] = [9911347.2]
        prob['mooring_line_type1_cost'] = [416.6761499999999]
        prob['mooring_line_type1_diameter'] = [0.09]
        prob['mooring_line_type1_mass_density'] = [161.18999999999997]
        prob['mooring_line_type1_stiffness'] = [691740000.0]
        prob['mooring_line_type1_tangential_added_mass'] = [0.0]
        prob['mooring_line_type1_tangential_drag'] = [0.1]
        prob['mooring_line_type1_transverse_added_mass'] = [1.0]
        prob['mooring_line_type1_transverse_drag'] = [1.6]
        prob['mooring_point1_location'] = [853.87, 0.0, -320.0]
        prob['mooring_point2_location'] = [-426.93499999999983, 739.4731115294188, -320.0]
        prob['mooring_point3_location'] = [-426.9350000000004, -739.4731115294184, -320.0]
        prob['mooring_point4_location'] = [5.2, 0.0, -70.0]
        prob['mooring_point5_location'] = [-2.5999999999999988, 4.503332099679082, -70.0]
        prob['mooring_point6_location'] = [-2.6000000000000023, -4.5033320996790795, -70.0]
        prob['mooring_water_depth'] = [320.0]

        try:
            prob.run_model()
            self.assertTrue(True)
        except Exception:
            self.assertTrue(False)


def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestSpider))
    return suite


if __name__ == "__main__":
    result = unittest.TextTestRunner().run(suite())

    if result.wasSuccessful():
        exit(0)
    else:
        exit(1)
