#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const [moduleFamily, moduleBaseName] = (namespace === 'AWS::Serverless' ? 'AWS::SAM' : namespace).split('::');
        const moduleName = `${moduleFamily}-${moduleBaseName.replace(/V\d+$/, '')}`.toLocaleLowerCase();
        const packagePath = path.join(root, moduleName);
        const lowcaseModuleName = moduleBaseName.toLocaleLowerCase();
        const packageName = `@aws-cdk/${moduleName}`;
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${moduleFamily}-${moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        // dotnet names
        const dotnetPackage = `Amazon.CDK.${moduleFamily}.${moduleBaseName}`;
        // java names
        const javaGroupId = 'software.amazon.awscdk';
        const javaPackage = moduleFamily === 'AWS'
            ? `services.${lowcaseModuleName}`
            : `${moduleFamily.toLocaleLowerCase()}.${lowcaseModuleName}`;
        const javaArtifactId = moduleFamily === 'AWS'
            ? lowcaseModuleName
            : `${moduleFamily.toLocaleLowerCase()}-${lowcaseModuleName}`;
        // python names
        const pythonDistName = `aws-cdk.${moduleName}`;
        const pythonModuleName = pythonDistName.replace(/-/g, '_');
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${packageName}...`);
        await write('package.json', {
            name: packageName,
            version,
            description: `The CDK Construct Library for ${namespace}`,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: dotnetPackage,
                        packageId: dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${javaGroupId}.${javaPackage}`,
                        maven: {
                            groupId: javaGroupId,
                            artifactId: javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: pythonDistName,
                        module: pythonModuleName,
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@types/jest': '^26.0.22',
                '@aws-cdk/assert-internal': version,
                'cdk-build-tools': version,
                'cfn2ts': version,
                'pkglint': version,
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assert-internal/jest';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await write('README.md', [
            `# ${namespace} Construct Library`,
            '<!--BEGIN STABILITY BANNER-->',
            '',
            '---',
            '',
            '![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)',
            '',
            '> All classes with the `Cfn` prefix in this module ([CFN Resources]) are always stable and safe to use.',
            '>',
            '> [CFN Resources]: https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib',
            '',
            '---',
            '',
            '<!--END STABILITY BANNER-->',
            '',
            'This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.',
            '',
            '```ts',
            `import ${lowcaseModuleName} = require('${packageName}');`,
            '```',
        ]);
        await write('.eslintrc.js', [
            "const baseConfig = require('cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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