"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const default_1 = require("../lib/default");
const aws_entities_1 = require("./aws-entities");
const fact_tables_1 = require("./fact-tables");
async function main() {
    checkRegions(fact_tables_1.APPMESH_ECR_ACCOUNTS);
    checkRegions(fact_tables_1.DLC_REPOSITORY_ACCOUNTS);
    checkRegions(fact_tables_1.ELBV2_ACCOUNTS);
    checkRegions(fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS);
    const lines = [
        "import { Fact, FactName } from './fact';",
        '',
        '/* eslint-disable quote-props */',
        '/* eslint-disable max-len */',
        '',
        '/**',
        ' * Built-in regional information, re-generated by `npm run build`.',
        ' *',
        ` * @generated ${new Date().toISOString()}`,
        ' */',
        'export class BuiltIns {',
        '  /**',
        '   * Registers all the built in regional data in the RegionInfo database.',
        '   */',
        '  public static register(): void {',
    ];
    const defaultMap = 'default';
    for (const region of aws_entities_1.AWS_REGIONS) {
        let partition = fact_tables_1.PARTITION_MAP[defaultMap].partition;
        let domainSuffix = fact_tables_1.PARTITION_MAP[defaultMap].domainSuffix;
        for (const key in fact_tables_1.PARTITION_MAP) {
            if (region.startsWith(key)) {
                partition = fact_tables_1.PARTITION_MAP[key].partition;
                domainSuffix = fact_tables_1.PARTITION_MAP[key].domainSuffix;
            }
        }
        registerFact(region, 'PARTITION', partition);
        registerFact(region, 'DOMAIN_SUFFIX', domainSuffix);
        registerFact(region, 'CDK_METADATA_RESOURCE_AVAILABLE', fact_tables_1.AWS_CDK_METADATA.has(region) ? 'YES' : 'NO');
        registerFact(region, 'S3_STATIC_WEBSITE_ENDPOINT', fact_tables_1.AWS_OLDER_REGIONS.has(region)
            ? `s3-website-${region}.${domainSuffix}`
            : `s3-website.${region}.${domainSuffix}`);
        registerFact(region, 'S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID', fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS[region] || '');
        registerFact(region, 'ELBV2_ACCOUNT', fact_tables_1.ELBV2_ACCOUNTS[region]);
        registerFact(region, 'DLC_REPOSITORY_ACCOUNT', fact_tables_1.DLC_REPOSITORY_ACCOUNTS[region]);
        registerFact(region, 'APPMESH_ECR_ACCOUNT', fact_tables_1.APPMESH_ECR_ACCOUNTS[region]);
        const vpcEndpointServiceNamePrefix = `${domainSuffix.split('.').reverse().join('.')}.vpce`;
        registerFact(region, 'VPC_ENDPOINT_SERVICE_NAME_PREFIX', vpcEndpointServiceNamePrefix);
        for (const service of aws_entities_1.AWS_SERVICES) {
            registerFact(region, ['servicePrincipal', service], default_1.Default.servicePrincipal(service, region, domainSuffix));
        }
    }
    lines.push('  }');
    lines.push('');
    lines.push('  private constructor() {}');
    lines.push('}');
    await fs.writeFile(path.resolve(__dirname, '..', 'lib', 'built-ins.generated.ts'), lines.join('\n'));
    function registerFact(region, name, value) {
        const factName = typeof name === 'string' ? name : `${name[0]}(${name.slice(1).map(s => JSON.stringify(s)).join(', ')})`;
        lines.push(`    Fact.register({ region: ${JSON.stringify(region)}, name: FactName.${factName}, value: ${JSON.stringify(value)} });`);
    }
}
/**
 * Verifies that the provided map of region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegions(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const region of Object.keys(map)) {
        if (!allRegions.has(region)) {
            throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
        }
    }
}
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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