"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// eslint-disable-next-line import/no-extraneous-dependencies
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
// eslint-disable-next-line import/no-extraneous-dependencies
const db_provision_pgsql_1 = require("@libreworks/db-provision-pgsql");
// eslint-disable-next-line import/no-extraneous-dependencies
const pg_1 = require("pg");
const util_1 = require("./util");
const adminSecretArn = process.env.ADMIN_SECRET_ARN;
const ownerSecretArn = process.env.OWNER_SECRET_ARN;
const databaseName = process.env.DB_NAME;
const encoding = process.env.DB_ENCODING || "UTF8";
const locale = process.env.DB_LOCALE;
const schemaName = process.env.SCHEMA_NAME;
const secretsManagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const ownerSecretArns = util_1.parseJsonArrayFromEnv("OWNER_SECRETS");
const readerSecretArns = util_1.parseJsonArrayFromEnv("READER_SECRETS");
const unprivilegedSecretArns = util_1.parseJsonArrayFromEnv("UNPRIVILEGED_SECRETS");
const handler = async () => {
    // Here's the first network request:
    // Load the admin secret, needed to create the catalog and its owner.
    const adminSecret = await util_1.fetchSecret(adminSecretArn, secretsManagerClient);
    // First we need to connect to the "postgres" database.
    const clientDefaults = {
        host: adminSecret.host,
        port: adminSecret.port,
        user: adminSecret.username,
        password: adminSecret.password,
        connectionTimeoutMillis: 40000,
        ssl: true,
    };
    const client = new pg_1.Client({
        ...clientDefaults,
        database: "postgres",
    });
    const serverClient = new db_provision_pgsql_1.ServerClient(client);
    // Here's our second network request.
    console.log("About to open a connection to database: postgres");
    await client.connect();
    const catalog = new db_provision_pgsql_1.Catalog(databaseName, encoding, locale);
    console.log(`About to create database: ${databaseName}`);
    await serverClient.createDatabase(catalog);
    console.log(`Database created: ${databaseName}`);
    // Here are the next set of network requests:
    // Load all other Secrets Manager secrets in one go.
    const [schemaOwnerSecret, ownerSecrets, readerSecrets, unprivilegedSecrets] = await Promise.all([
        util_1.fetchSecret(ownerSecretArn, secretsManagerClient),
        util_1.fetchAllSecrets(ownerSecretArns, secretsManagerClient),
        util_1.fetchAllSecrets(readerSecretArns, secretsManagerClient),
        util_1.fetchAllSecrets(unprivilegedSecretArns, secretsManagerClient),
    ]);
    // Map the secrets to the Logins for the sake of performance.
    const loginMap = new Map();
    // Create a Login object for each secret we have. Store it in the map.
    const loginSecrets = [
        schemaOwnerSecret,
        ...ownerSecrets,
        ...readerSecrets,
        ...unprivilegedSecrets,
    ];
    for (const secret of loginSecrets) {
        const login = new db_provision_pgsql_1.Login(secret.username, secret.password);
        loginMap.set(secret, login);
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all.
    for (const login of loginMap.values()) {
        console.log(`About to create user: ${login.username}`);
        await serverClient.createRole(login);
        console.log(`User created: ${login.username}`);
    }
    const adminRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_adm`);
    const readerRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_ro`);
    // Here is the next network request:
    // Create roles for our admin users and read-only users.
    for (const role of [adminRole, readerRole]) {
        console.log(`About to create role: ${role.name}`);
        await serverClient.createRole(role);
        console.log(`Role created: ${role.name}`);
    }
    const grants = [];
    // Create a Grant object for the schema owner and all readers.
    for (const secret of [...ownerSecrets, ...readerSecrets]) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT", "TEMP"));
    }
    // Create a Grant object for the users with no privileges.
    for (const secret of unprivilegedSecrets) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT"));
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all
    for (const grant of grants) {
        console.log(`About to grant privileges: ${grant.toSql()}`);
        await serverClient.createGrant(grant);
        console.log("Privileges granted");
    }
    console.log("About to close the connection");
    await client.end();
    console.log("Connection closed");
    const schemaClient = new pg_1.Client({
        ...clientDefaults,
        database: databaseName,
    });
    const databaseClient = new db_provision_pgsql_1.DatabaseClient(schemaClient);
    // Here is the next network request:
    console.log(`About to open a connection to database: ${databaseName}`);
    await schemaClient.connect();
    let owner = new db_provision_pgsql_1.Login(schemaOwnerSecret.username, schemaOwnerSecret.password);
    // Here is the next network request:
    const schema = catalog.createSchema(schemaName || owner.username, owner);
    console.log(`About to create schema: ${schema.name} (if it does not already exist)`);
    await schemaClient.query(schema.toSql());
    console.log(`Schema created: ${schema.name}`);
    console.log(`About to alter schema ${schema.name} owner to: ${owner.name}`);
    await schemaClient.query(schema.changeOwner(owner).toSql());
    console.log(`Schema altered: ${schema.name}`);
    const admins = ownerSecrets.map((secret) => loginMap.get(secret));
    const readers = readerSecrets.map((secret) => loginMap.get(secret));
    // Here are the next set of network requests:
    // Create the grants and default permissions for the owner secrets.
    console.log(`About to create schema grants for admins`);
    await databaseClient.createAdminGrants(schema, adminRole, admins);
    // Create the grants and default permissions for the reader secrets.
    console.log(`About to create schema grants for readers`);
    await databaseClient.createReaderGrants(schema, readerRole, readers);
    console.log("About to close the connection");
    await schemaClient.end();
    console.log("Connection closed");
};
module.exports = { handler };
//# sourceMappingURL=data:application/json;base64,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