"""SDK to interact with the Amazon-Fewsats Marketplace API"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_client.ipynb.

# %% auto 0
__all__ = ['Amazon']

# %% ../nbs/00_client.ipynb 3
import httpx
from typing import Dict, List
from .config import *
from .crypto import *

from fastcore.utils import *


# %% ../nbs/00_client.ipynb 5
class Amazon:
    """
    Client for the Amazon-Fewsats Marketplace API.
    
    This client provides methods to interact with the Amazon-Fewsats Marketplace API,
    including product search, cart management, and checkout functionality.
    """
    
    def __init__(self, base_url: str = "https://amazon-backend.replit.app/api/v1", private_key: str = ''):
        """
        Initialize the Amazon Marketplace client.
        
        Args:
            base_url: The base URL of the Amazon-Fewsats Backend API.
            private_key: The private key of the Amazon-Fewsats Backend API. If not provided,
                         a random UUID will be generated.
        """
        self.base_url = base_url

        cfg = get_cfg()

        if private_key: self.pk, self.pub = from_pk_hex(private_key) # if provided use the private key
        elif cfg.priv: self.pk, self.pub = from_pk_hex(cfg.priv) # if not provided use the private key from the config file
        else: 
            self.pk, self.pub = generate_keys()
            save_cfg({'priv': priv_key_hex(self.pk)})


        self.client = httpx.Client()
        self.client.headers.update({"user-id": self.pub}) # use public key as user id

    def _make_request(self, method: str, endpoint: str, timeout: int = 30, **kwargs) -> httpx.Response:
        """
        Make a request to the Amazon-Fewsats Backend API.
        
        Args:
            method: The HTTP method to use.
            endpoint: The API endpoint to call.
            **kwargs: Additional arguments to pass to httpx.
            
        Returns:
            The response from the API.
        """
        url = f"{self.base_url}/{endpoint}"
        return self.client.request(method, url, timeout=timeout, **kwargs)

# %% ../nbs/00_client.ipynb 8
@patch
def search(self: Amazon, query: str) -> Dict:
    """
    Search for products.
    
    Args:
        query: The search query.
        
    Returns:
        The search results.
    """
    params = {
        "q": query,
    }
    
    return self._make_request("GET", "search", params=params)

# %% ../nbs/00_client.ipynb 12
@patch    
def get_product_detail(self: Amazon, asin: str) -> Dict:
    """
    Get details for a specific product.
    
    Args:
        slug: The product slug.
        
    Returns:
        The product details.
    """
    return self.search(asin)

# %% ../nbs/00_client.ipynb 15
@patch
def get_cart(self: Amazon) -> Dict:
    """
    Get the current user's cart.
    
    Returns:
        The cart details.
    """
    return self._make_request("GET", "cart")


# %% ../nbs/00_client.ipynb 18
@patch
def buy_now(self: Amazon, asin: str, shipping_address: Dict,
            user: Dict, quantity: int = 1) -> Dict:
    """
    Purchase a product directly. 
    
    Args:
        asin: The product asin.
        quantity: The quantity to purchase.
        shipping_address: The shipping address.
        user: The user information.
        
    Example:
        shipping_address = {
            "name": "John Doe",
            "address": "123 Main St",
            "city": "New York",
            "state": "NY",
            "country": "US",
            "postalCode": "10001"
        }
        
        user = {
            "name": "John Doe",
            "email": "john@example.com",
        }
        
        a.buy_now(product_id, variant_id, shipping_address, user)
        
    Returns:
        The payment information.
    """

        
    request_data = {
        "external_id": asin,
        "quantity": quantity,
        "shipping_address": shipping_address,
        "user": user
    }
    
    return self._make_request("POST", "buy-now", json=request_data)

# %% ../nbs/00_client.ipynb 24
@patch
def get_user_orders(self: Amazon) -> List[Dict]:
    """
    Get all orders for the current user.
    
    Returns:
        A list of orders.
    """
    return self._make_request("GET", f"users/orders")

# %% ../nbs/00_client.ipynb 27
@patch
def get_order(self: Amazon, external_id: str) -> Dict:
    """
    Get details for a specific order.
    
    Args:
        external_id: The external ID of the order.
        
    Returns:
        The order details.
    """
    return self._make_request("GET", f"orders/{external_id}")

# %% ../nbs/00_client.ipynb 30
@patch
def get_user_info(self: Amazon) -> Dict:
    """
    Get the current user's profile and shipping addresses.
    
    Returns:
        Dict containing user profile info (firstname, lastname, email) and list of shipping addresses
    """
    return self._make_request("GET", "user/info")
