"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = exports.POLICY_DESC_SUFFIX_FROM_FILE = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
exports.POLICY_DESC_SUFFIX_FROM_FILE = ' - Created by CDK';
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param props A `StaticPolicyFromFileProps` object.
     */
    static fromFile(scope, id, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        let relativePath = path.basename(props.path);
        let enablePolicyValidation = (props.enablePolicyValidation == undefined) ? true : props.enablePolicyValidation;
        let policyDescription = props.description || (0, cedar_helpers_1.getPolicyDescription)(policyFileContents) || `${relativePath}${exports.POLICY_DESC_SUFFIX_FROM_FILE}`;
        return new Policy(scope, id, {
            definition: {
                static: {
                    statement: policyFileContents,
                    description: policyDescription,
                    enablePolicyValidation: enablePolicyValidation,
                },
            },
            policyStore: props.policyStore,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let cfnDefinitionAttr;
        let definitionProperty = props.definition;
        if (props.definition.static) {
            if (props.definition.static.enablePolicyValidation ?? true) {
                (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            }
            let description = props.definition.static.description || (0, cedar_helpers_1.getPolicyDescription)(props.definition.static.statement) || undefined;
            definitionProperty = {
                static: {
                    ...props.definition.static,
                    ...{
                        description,
                    },
                },
            };
            cfnDefinitionAttr = {
                static: {
                    description: description,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            cfnDefinitionAttr = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: cfnDefinitionAttr,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = definitionProperty.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = definitionProperty;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.2.1" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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