"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'Bucket with expiration days'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    expiration: core_1.Duration.days(30)
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationInDays: 30,
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Can use addLifecycleRule() to add a lifecycle rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const bucket = new lib_1.Bucket(stack, 'Bucket');
        bucket.addLifecycleRule({
            expiration: core_1.Duration.days(30)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationInDays: 30,
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Bucket with expiration date'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    expirationDate: new Date('2018-01-01')
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationDate: '2018-01-01T00:00:00',
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Bucket with transition rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    transitions: [{
                            storageClass: lib_1.StorageClass.GLACIER,
                            transitionAfter: core_1.Duration.days(30)
                        }]
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        Transitions: [{
                                StorageClass: 'GLACIER',
                                TransitionInDays: 30
                            }],
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Noncurrent rule on nonversioned bucket fails'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN: Fail because of lack of versioning
        test.throws(() => {
            new lib_1.Bucket(stack, 'Bucket1', {
                lifecycleRules: [{
                        noncurrentVersionExpiration: core_1.Duration.days(10)
                    }]
            });
        });
        // WHEN: Succeeds because versioning is enabled
        new lib_1.Bucket(stack, 'Bucket2', {
            versioned: true,
            lifecycleRules: [{
                    noncurrentVersionExpiration: core_1.Duration.days(10)
                }]
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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