"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const kendra = require("aws-cdk-lib/aws-kendra");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
test('Launch with minimal code and check  structure', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
        Environment: {
            Variables: {
                KENDRA_INDEX_ID: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        },
    });
    template.hasResourceProperties("AWS::Kendra::Index", {
        RoleArn: {
            "Fn::GetAtt": [
                "samplekendraindexrolesample4F9E7B66",
                "Arn",
            ],
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Allow Kendra index to write CloudWatch Logs",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "cloudwatch:PutMetricData",
                            Condition: {
                                StringEquals: {
                                    "cloudwatch:namespace": "AWS/Kendra"
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "logs:CreateLogGroup",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "logs:DescribeLogGroups",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                                "logs:DescribeLogStream"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*:log-stream:*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "AllowLogging"
            }
        ],
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Policy for Kendra S3 Data Source",
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344/*`
                        },
                        {
                            Action: "s3:ListBucket",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344`
                        },
                        {
                            Action: [
                                "kendra:BatchPutDocument",
                                "kendra:BatchDeleteDocument"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "samplekendraindexsample8A81A6C2",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "s3CrawlPolicy"
            }
        ]
    });
    // Confirm that Lambda function has QUERY access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ],
        },
        Roles: [
            {
                Ref: "sampletestfunctionname24334ServiceRole99395A01"
            }
        ]
    });
});
test('Check pattern properties on minimal launch', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const newConstruct = new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
    });
    expect(newConstruct.lambdaFunction).toBeDefined();
    expect(newConstruct.kendraDataSources).toBeDefined();
    expect(newConstruct.kendraDataSources.length).toEqual(1);
    expect(newConstruct.kendraIndex).toBeDefined();
});
test('Launch with VPC', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name12539';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        deployVpc: true,
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Check the VPC
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "10.0.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    // Is the Lambda function associated with the VPC
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "sampleReplaceDefaultSecurityGroupsecuritygroupE5725669",
                        "GroupId"
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    // Check that the Lambda function Policy has proper network access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                        "ec2:AssignPrivateIpAddresses",
                        "ec2:UnassignPrivateIpAddresses"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {},
                {},
                {}
            ],
        },
        Roles: [
            {
                Ref: "sampleLambdaFunctionServiceRole7A3C4AF5"
            }
        ]
    });
    // Check for the Kendra endpoint in the VPC
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kendra"
                ]
            ]
        },
        VpcId: {
            Ref: "Vpc8378EB38"
        },
        PrivateDnsEnabled: true,
        SecurityGroupIds: [
            {
                "Fn::GetAtt": [
                    "DefaultKENDRAsecuritygroup34536A79",
                    "GroupId"
                ]
            }
        ],
        SubnetIds: [
            {
                Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
            },
            {
                Ref: "VpcisolatedSubnet2Subnet39217055"
            }
        ],
        VpcEndpointType: "Interface"
    });
});
test('Launch with existing lambda', () => {
    const stack = new cdk.Stack();
    const testTimeout = 17;
    const testFunctionName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        timeout: cdk.Duration.seconds(testTimeout),
        functionName: testFunctionName
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingLambdaObj: existingFunction,
        kendraDataSourcesProps: [],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: testTimeout,
        FunctionName: testFunctionName,
    });
});
test('Confirm error with data source with no bucket name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', {
            lambdaFunctionProps: lambdaProps,
            kendraDataSourcesProps: [{
                    type: 'S3',
                    dataSourceConfiguration: {
                        s3Configuration: {}
                    }
                }
            ],
        });
    };
    expect(app).toThrowError(/Error - an S3 Kendra DataSource requires the DataSourceConfiguration.S3Configuration.bucketName prop/);
});
test('Launch with existing vpc', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingVpc: defaults.getTestVpc(stack),
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Launch with Read/Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["ReaD", "SubmitFeedBack", "wrITE"], // this also checks case sensitivity
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Read permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["READ"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with SubmitFeedback permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["SUBMITFEEDBACK"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with existing kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const existingRole = new iam.Role(stack, 'existing-role', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
    });
    const existingIndex = new kendra.CfnIndex(stack, 'existing-index', {
        edition: 'ENTERPRISE',
        name: 'existingIndex',
        roleArn: existingRole.roleArn
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        existingKendraIndexObj: existingIndex,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Make sure we didn't create an index anyway
    template.resourceCountIs("AWS::Kendra::Index", 1);
    template.hasResourceProperties("AWS::Kendra::Index", {
        Edition: 'ENTERPRISE'
    });
});
test('Launch with S3 data source with overridden defaults', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name223423';
    const testInclusionPattern = 'this-folder';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        inclusionPatterns: [testInclusionPattern],
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 1);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                InclusionPatterns: [testInclusionPattern],
                BucketName: testBucketName
            },
        }
    });
});
test('Launch with S3 data source and unsupported data source', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const nonImplementedSourceType = "WEBCRAWLER";
    const nonImplementedSourceName = "test-other-source";
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    // Create a role
    const fakeRole = new iam.Role(stack, 'fakeRole', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
        roleName: 'externalFakeRole'
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                name: nonImplementedSourceName,
                roleArn: fakeRole.roleArn,
                type: nonImplementedSourceType,
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Name: nonImplementedSourceName,
        Type: nonImplementedSourceType,
    });
});
test('Launch with multiple S3 data sources', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22342';
    const secondBucketName = 'second-bucket-name22342342';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'two-sources', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: secondBucketName,
                    }
                }
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources0B8E24996", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: secondBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources164176C5E", "Arn"]
        },
    });
});
test('Test with custom environment variable name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
        indexIdEnvironmentVariableName: "MY_VAR_NAME",
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                MY_VAR_NAME: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        }
    });
});
test('Confirm CheckVpcProps is being called', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const props = {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
        deployVpc: true,
        existingVpc: vpc
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'testBucketName',
                    }
                }
            }
        ],
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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