"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_compat_1 = require("./construct-compat");
const runtime_info_1 = require("./private/runtime-info");
const tree_metadata_1 = require("./private/tree-metadata");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends construct_compat_1.Construct {
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_compat_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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