"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk = require("../lib");
function createModernApp() {
    return new cdk.App({
        context: {
            [cxapi.DISABLE_VERSION_REPORTING]: 'true',
        }
    });
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.synth();
        // THEN
        test.same(app.synth(), session); // same session if we synth() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json', 'tree.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {
            Tree: {
                type: 'cdk:tree',
                properties: { file: 'tree.json' }
            }
        });
        test.deepEqual(readJson(session.directory, 'tree.json'), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: { id: 'Tree', path: 'Tree' }
                }
            }
        });
        test.done();
    },
    'synthesis respects disabling tree metadata'(test) {
        const app = new cdk.App({
            treeMetadata: false,
        });
        const assembly = app.synth();
        test.deepEqual(list(assembly.directory), ['cdk.out', 'manifest.json']);
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.assembly.outdir, 'foo.json', { bar: 123 });
                s.assembly.addArtifact('my-random-construct', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json'
                    }
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.ok(list(session.directory).includes('foo.json'));
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: cxapi.CLOUD_ASSEMBLY_VERSION,
            artifacts: {
                'Tree': {
                    type: 'cdk:tree',
                    properties: { file: 'tree.json' }
                },
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' }
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                }
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends cdk.Construct {
            constructor() {
                super(undefined, 'id');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.assembly.addArtifact('art', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    properties: {
                        templateFile: 'hey.json',
                        parameters: {
                            paramId: 'paramValue',
                            paramId2: 'paramValue2'
                        }
                    },
                    environment: 'aws://unknown-account/us-east-1'
                });
                writeJson(session.assembly.outdir, 'hey.json', { hello: 123 });
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
        }
        const root = new SynthesizeMe();
        const assembly = cdk.ConstructNode.synth(root.node, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStackByName('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.templateFile, 'hey.json');
        test.deepEqual(stack.parameters, { paramId: 'paramValue', paramId2: 'paramValue2' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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