# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
.. _`LARSIM`: http://www.larsim.de/en/the-model/
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

Version 1 of the L-Land model is a `LARSIM`_ type model applying relatively
simple approaches for calculating evapotranspiration (after Turc-Wendling)
and snowmelt (degree-day method). Hence, it requires only three meteorological
input variables (precipitation, temperature, and global radiation). We
created |lland_v1| on behalf of the `German Federal Institute of Hydrology
(BfG)`_ for large scale hydrological applications in central Europe.

The following list summarises the main components of |lland_v1|:

 * Simple routines for adjusting the meteorological input data
 * Reference evapotranspiration after Turc-Wendling :cite:`ref-DVWK`
 * Mixed precipitation within a definable temperature-range
 * An enhanced degree-day method for calculating snowmelt
 * A simple snow retention routine
 * Landuse and month specific potential evapotranspiration.
 * Actual soil evapotranspiration after ATV-DVWK-504 (2002) :cite:`ref-DVWK`
 * A soil routine based on the Xinanjiang model :cite:`ref-zhao1977flood`
 * One base flow, two interflow and two direct flow components
 * A freely configurable capillary rise routine
 * Options to limit the capacity of the base flow storage
 * Separate linear storages for modelling runoff concentration
 * Additional evaporation from water areas within the subcatchment
 * Optional evaporation from inflowing runoff

The following figure shows the general structure of L-Land Version 1.
Note that, besides water areas and sealed surfaces, all land-use types
rely on the same set of process equations:

.. image:: HydPy-L-Land_Version-1.png

As for all models implemented in HydPy, you can principally apply all L-Land
models on arbitrary simulation step sizes.  But application model |lland_v1|
one has to be aware that the Turc-Wendling equation for calculating reference
evaporation targets daily time steps only.

.. _lland_v1_integration_tests:

Integration tests
=================

.. how_to_understand_integration_tests::

We perform all integration tests over five days, including an extreme
precipitation event.  Despite the mentioned limitation of the Turc-Wendling
equation, we select an hourly simulation step size, which reveals more
details than a daily step size:

>>> from hydpy import pub
>>> pub.timegrids = "2000-01-01", "2000-01-05", "1h"

Next, we prepare a model instance:

>>> from hydpy.models.lland_v1 import *
>>> parameterstep("1h")

|lland_v1| provides a type of optional routing approach, which adds the
runoff from the upstream sub-catchments to the runoff generated by the actual
sub-catchment (see example :ref:`lland_v1_acker_routing`).  This approach
allows |lland_v1| to subtract water area evaporation not only from the runoff
generated within the actual sub-catchment but from the total runoff flowing
through it (see example :ref:`lland_v1_wasser_routing`).  The documentation
on method |Calc_QAH_V1| gives further information.

The optionality of this routing approach results in different configuration
possibilities of the related |Element| objects.  If you do not want to use
the routing approach (either generally or because the relevant catchment is
a headwater catchment), the element requires an outlet node only:

>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model

If you want to use the routing approach, you also need to define one or more
inlet nodes, providing the inflowing runoff (we rely on such an element
throughout the following examples but make our catchment effectively to a
headwater by setting the inflow to zero most of the time):

>>> inlet = Node("inlet")
>>> land = Element("land", inlets=inlet, outlets=outlet)
>>> land.model = model

We focus on a single hydrological response unit with a size of one
square kilometre at an altitude of 100 meters:

>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)
>>> hnn(100.0)

We initialise a test function object that prepares and runs the tests
and prints and plots their results:

>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(land)

.. _lland_v1_acker_summer:

acre (summer)
_____________

In the first example, arable land is the land-use class of our choice
(for all other land-use types, except the ones mentioned below, the
results were the same):

>>> lnk(ACKER)

The following set of control parameter values tries to configure
application model |lland_v1| in a manner that allows retracing the
influence of most of the different implemented methods on the shown results:

>>> kg(1.2)
>>> kt(0.8)
>>> ke(0.4)
>>> kf(0.6)
>>> fln(0.5)
>>> hinz(0.2)
>>> lai(4.0)
>>> treft(0.0)
>>> trefn(0.0)
>>> tgr(1.0)
>>> tsp(2.0)
>>> gtf(0.5)
>>> pwmax(1.4)
>>> grasref_r(5.0)
>>> wmax(200.0)
>>> fk(relative=0.5)
>>> pwp(relative=0.05)
>>> kapgrenz(option="0_WMax/10")
>>> kapmax(0.08)
>>> beta(0.005)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(1.0)
>>> dmin(0.1)
>>> bsf(0.4)
>>> volbmax(inf)
>>> gsbmax(1.0)
>>> gsbgrad1(inf)
>>> gsbgrad2(inf)
>>> a1(1.0)
>>> a2(1.0)
>>> tind(1.0)
>>> eqb(100.0)
>>> eqi1(20.0)
>>> eqi2(10.0)
>>> eqd1(5.0)
>>> eqd2(2.0)
>>> negq(False)

Initially, relative soil moisture is 10 %, but all other storages are empty
(this setting is not very realistic but makes it easier to understand results
of the different integration tests):

>>> test.inits = ((states.inzp, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.bowa, 20.0),
...               (states.qdgz1, 0.0),
...               (states.qdgz2, 0.0),
...               (states.qigz1, 0.0),
...               (states.qigz2, 0.0),
...               (states.qbgz, 0.0),
...               (states.qdga1, 0.0),
...               (states.qdga2, 0.0),
...               (states.qiga1, 0.0),
...               (states.qiga2, 0.0),
...               (states.qbga, 0.0))

The first input data set mimics an extreme precipitation event in summer
and set the inflow to zero:

>>> inputs.nied.series = (
...     0.0, 0.0,  0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0,
...     2.1, 10.4, 3.5, 3.4, 1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1,
...     20.7, 37.9, 8.2, 3.6, 7.5, 18.5, 15.4, 6.3, 1.9, 4.9, 2.7, 0.5,
...     0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6, 0.7, 0.3, 0.3, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.3, 0.0,
...     0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.teml.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2,
...     35.2, 37.1, 31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8,
...     19.6, 19.2, 19.2, 19.2, 18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8,
...     19.0, 19.2, 19.3, 19.0, 18.8, 18.7, 17.8, 17.4, 17.3, 16.8, 16.5,
...     16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3, 14.9, 15.7, 16.0,
...     16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3, 18.1,
...     16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1,
...     11.9, 12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8,
...     22.2, 20.1, 17.8, 15.2, 14.5, 12.4, 11.7, 11.9)
>>> inputs.glob.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 11.2, 105.5, 248.3, 401.3, 449.7,
...     493.4, 261.5, 363.6, 446.2, 137.6, 103.0, 63.7, 41.4, 7.9, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 6.1, 77.9, 196.7,
...     121.9, 156.6, 404.7, 217.9, 582.0, 263.9, 136.8, 146.6, 190.6,
...     103.5, 13.8, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     4.4, 26.1, 74.2, 287.1, 299.8, 363.5, 368.4, 317.8, 534.7, 319.4,
...     350.6, 215.4, 97.8, 13.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 17.0, 99.7, 239.4, 391.2, 525.6, 570.2, 559.1, 668.0,
...     593.4, 493.0, 391.2, 186.0, 82.4, 17.0, 0.0, 0.0, 0.0, 0.0)
>>> inlet.sequences.sim.series = 0.0

The following results show that all relevant model components, except the
snow routines, are activated at least once within the simulation period.
Take your time to select different time series and see, for example, how
the soil moisture content |BoWa| is varying over time.  One might realise
the "linear storage" type of relationship between input |Nied| and outflow
|lland_fluxes.QAH|.  This pattern is due to the dominance of the direct
runoff generation (|QDGZ|) based on the Xinanjiang model and modelling
runoff concentration via linear storages (inspectable through clicking on
|QDGZ1| and |QDGA1|):

.. integration-test::

    >>> test("lland_v1_acker_summer",
    ...      axis1=(inputs.nied, fluxes.qah),
    ...      axis2=states.bowa)
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000495 | 0.000138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.0 | 0.000138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 0.001462 | 0.000406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.0 | 0.000406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 0.002394 | 0.000665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.0 | 0.000665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 0.003291 | 0.000914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.0 | 0.000914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 0.004156 | 0.001154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.0 | 0.001154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  0.00499 | 0.001386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.0 | 0.001386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 0.005793 | 0.001609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.0 | 0.001609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 0.006566 | 0.001824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.0 | 0.001824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 0.007306 |  0.00203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.0 |  0.00203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 0.008003 | 0.002223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.0 | 0.002223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 0.008643 | 0.002401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.0 | 0.002401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 0.009219 | 0.002561 |      0.0 |  0.0 |  0.0 |   17.83797 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.0 | 0.002561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 0.009727 | 0.002702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.0 | 0.002702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  0.01018 | 0.002828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.0 | 0.002828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 0.010586 | 0.002941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.0 | 0.002941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 0.010942 | 0.003039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.0 | 0.003039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 0.011263 | 0.003129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.0 | 0.003129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 0.011564 | 0.003212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.0 | 0.003212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  0.01363 | 0.003786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.0 | 0.003786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 0.034046 | 0.009457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.0 | 0.009457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 0.060472 | 0.016798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.0 | 0.016798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 0.088924 | 0.024701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.0 | 0.024701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  0.17003 | 0.047231 | 0.780374 |  0.0 |  0.0 |   44.22507 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.0 | 0.047231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 0.210341 | 0.058428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.0 | 0.058428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 0.183571 | 0.050992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.0 | 0.050992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 0.185428 | 0.051508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.0 | 0.051508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 0.198953 | 0.055265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.0 | 0.055265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 0.291718 | 0.081033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.0 | 0.081033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 0.384279 | 0.106744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.0 | 0.106744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 0.401551 | 0.111542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.0 | 0.111542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 0.391634 | 0.108787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.0 | 0.108787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  0.34564 | 0.096011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.0 | 0.096011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   0.2946 | 0.081833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.0 | 0.081833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 0.252616 | 0.070171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.0 | 0.070171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 0.218815 | 0.060782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.0 | 0.060782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 0.191684 | 0.053246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.0 | 0.053246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 0.206673 | 0.057409 | 0.125043 |  0.0 |  0.0 |  68.927693 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.0 | 0.057409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 0.288705 | 0.080196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.0 | 0.080196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 0.323726 | 0.089924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.0 | 0.089924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 0.821088 |  0.22808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.0 |  0.22808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 2.868463 | 0.796795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.0 | 0.796795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.504942 | 3.791851 | 1.053292 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.600789 | 0.904153 | 0.064494 | 0.140463 | 0.594939 | 0.837653 | 2.869278 |  0.02348 | 0.006795 | 0.054646 |   0.0 | 1.053292 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.045681 |  2.93515 | 0.815319 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.043686 | 0.001996 | 0.067538 | 0.186956 | 0.625375 | 0.923811 | 1.903847 | 0.025555 | 0.021764 | 0.060173 |   0.0 | 0.815319 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.527094 | 2.481423 | 0.689284 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.604289 | 0.922805 | 0.068447 | 0.201685 | 0.634469 | 0.998043 | 1.351716 | 0.027625 | 0.038197 | 0.065843 |   0.0 | 0.689284 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.227705 | 3.418899 | 0.949694 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.861643 | 5.366061 | 0.071072 | 0.246232 |  0.66072 | 1.132039 | 2.129639 |  0.02968 |  0.05591 | 0.071631 |   0.0 | 0.949694 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  6.97553 | 4.804338 | 1.334538 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.856642 | 5.118889 |  0.07804 | 0.377966 | 0.730399 | 1.263826 | 3.350409 |  0.03187 | 0.080394 |  0.07784 |   0.0 | 1.334538 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 |  3.09949 |  4.84592 | 1.346089 |  0.78175 |  0.0 |  0.0 | 169.469597 | 1.677366 | 1.422124 |  0.08319 | 0.486735 | 0.781897 | 1.354495 | 3.258614 | 0.034249 | 0.113973 |  0.08459 |   0.0 | 1.346089 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 0.948023 | 3.857115 | 1.071421 | 0.782063 |  0.0 |  0.0 | 169.380125 | 0.948023 |      0.0 | 0.084735 | 0.521116 | 0.797348 | 1.344716 | 2.233012 | 0.036674 | 0.151109 | 0.091605 |   0.0 | 1.071421 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 2.488688 | 3.201733 |  0.88937 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.598182 | 0.890506 |  0.08469 |  0.52011 | 0.796901 | 1.333698 | 1.544122 | 0.039016 | 0.186271 | 0.098625 |   0.0 |  0.88937 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 2.838425 | 0.788451 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.351893 | 1.120521 | 0.041268 |  0.21912 | 0.105622 |   0.0 | 0.788451 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 0.249877 | 2.347956 |  0.65221 | 0.782239 |  0.0 |  0.0 | 170.639599 | 0.249877 |      0.0 | 0.085875 | 0.546974 | 0.808745 | 1.242434 | 0.699367 | 0.043439 | 0.250108 | 0.112609 |   0.0 |  0.65221 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 |  1.91484 |   0.5319 | 0.782169 |  0.0 |  0.0 | 169.345117 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.047903 | 0.424187 | 0.045495 | 0.277747 | 0.119508 |   0.0 |   0.5319 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 0.242047 | 1.621283 | 0.450356 | 0.781959 |  0.0 |  0.0 | 168.284125 | 0.242047 |      0.0 | 0.084673 | 0.519716 | 0.796726 | 0.888843 | 0.257283 | 0.047421 | 0.301457 | 0.126278 |   0.0 | 0.450356 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 1.523064 | 0.423073 | 0.781681 |  0.0 |  0.0 |  168.57882 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 0.857129 | 0.162134 | 0.049225 | 0.321653 | 0.132923 |   0.0 | 0.423073 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.189947 | 1.454205 | 0.403946 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.189947 |      0.0 | 0.084289 | 0.511125 | 0.792894 | 0.820771 | 0.103491 | 0.050931 | 0.339529 | 0.139482 |   0.0 | 0.403946 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 1.313091 | 0.364748 | 0.705926 |  0.0 |  0.0 | 166.216858 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.696563 |  0.06277 | 0.052545 | 0.355257 | 0.145956 |   0.0 | 0.364748 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 |      0.0 | 1.190388 | 0.330663 | 0.471181 |  0.0 |  0.0 | 164.867718 |      0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 | 0.577717 | 0.038072 |  0.05405 | 0.368242 | 0.152307 |   0.0 | 0.330663 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 1.088702 | 0.302417 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.472994 | 0.023092 | 0.055451 | 0.378635 | 0.158529 |   0.0 | 0.302417 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 |      0.0 | 1.009088 | 0.280302 |      0.0 |  0.0 |  0.0 | 161.377451 |      0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 | 0.387255 | 0.014006 | 0.056746 | 0.386466 | 0.164615 |   0.0 | 0.280302 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 1.028002 | 0.285556 |      0.0 |  0.0 |  0.0 | 161.294054 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.399248 | 0.008495 | 0.057936 |  0.39177 | 0.170553 |   0.0 | 0.285556 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.014983 | 1.041382 | 0.289273 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.014983 |      0.0 | 0.080647 | 0.431887 |  0.75647 | 0.405172 | 0.005153 | 0.059044 | 0.395629 | 0.176385 |   0.0 | 0.289273 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 0.976673 | 0.271298 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.333039 | 0.003125 | 0.060079 | 0.398311 | 0.182119 |   0.0 | 0.271298 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 |      0.0 | 0.922391 |  0.25622 |      0.0 |  0.0 |  0.0 | 155.593338 |      0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 | 0.272669 | 0.001896 | 0.061023 |  0.39909 | 0.187714 |   0.0 |  0.25622 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 0.877158 | 0.243655 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.223243 |  0.00115 |  0.06187 | 0.397747 | 0.193148 |   0.0 | 0.243655 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 |      0.0 | 0.838885 | 0.233024 |      0.0 |  0.0 |  0.0 | 151.294475 |      0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 | 0.182776 | 0.000697 | 0.062621 | 0.394374 | 0.198417 |   0.0 | 0.233024 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 0.806179 | 0.223939 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.149644 | 0.000423 | 0.063282 | 0.389303 | 0.203526 |   0.0 | 0.223939 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 |      0.0 | 0.778035 | 0.216121 |      0.0 |  0.0 |  0.0 | 148.167947 |      0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 | 0.122518 | 0.000257 | 0.063864 | 0.382909 | 0.208487 |   0.0 | 0.216121 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 0.753875 |  0.20941 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.100309 | 0.000156 | 0.064379 |  0.37571 | 0.213321 |   0.0 |  0.20941 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 |      0.0 | 0.733188 | 0.203663 |      0.0 |  0.0 |  0.0 | 145.983055 |      0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 | 0.082126 | 0.000094 | 0.064839 | 0.368084 | 0.218044 |   0.0 | 0.203663 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 0.715434 | 0.198732 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.067239 | 0.000057 | 0.065249 | 0.360222 | 0.222667 |   0.0 | 0.198732 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 |      0.0 | 0.700075 | 0.194465 |      0.0 |  0.0 |  0.0 | 143.898606 |      0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 | 0.055051 | 0.000035 | 0.065614 | 0.352185 |  0.22719 |   0.0 | 0.194465 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 | 0.686662 | 0.190739 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 | 0.045072 | 0.000021 | 0.065935 | 0.344017 | 0.231617 |   0.0 | 0.190739 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |      0.0 |  0.67483 | 0.187453 |      0.0 |  0.0 |  0.0 | 141.875122 |      0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 | 0.036902 | 0.000013 | 0.066216 | 0.335751 | 0.235948 |   0.0 | 0.187453 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 0.664287 | 0.184524 |      0.0 |  0.0 |  0.0 | 140.884763 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.030213 | 0.000008 | 0.066458 | 0.327422 | 0.240186 |   0.0 | 0.184524 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 |      0.0 | 0.654794 | 0.181887 |      0.0 |  0.0 |  0.0 | 139.908406 |      0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 | 0.024736 | 0.000005 | 0.066665 | 0.319056 | 0.244333 |   0.0 | 0.181887 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 0.666919 | 0.185255 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.041012 | 0.000003 | 0.066837 | 0.310678 | 0.248389 |   0.0 | 0.185255 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 |      0.0 | 0.674893 |  0.18747 | 0.766603 |  0.0 |  0.0 | 138.559658 |      0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 | 0.052999 | 0.000002 | 0.066984 | 0.302538 |  0.25237 |   0.0 |  0.18747 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 |      0.0 | 0.661405 | 0.183724 |  0.74963 |  0.0 |  0.0 | 137.632082 |      0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 | 0.043392 | 0.000001 | 0.067107 | 0.294626 | 0.256278 |   0.0 | 0.183724 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 |      0.0 | 0.649552 | 0.180431 | 0.687585 |  0.0 |  0.0 | 136.717337 |      0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 | 0.035526 | 0.000001 | 0.067202 | 0.286722 | 0.260101 |   0.0 | 0.180431 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 | 0.204485 | 0.658192 | 0.182831 | 0.521608 |  0.0 |  0.0 | 136.338253 | 0.204485 |      0.0 | 0.068359 | 0.200239 | 0.633587 | 0.048237 |      0.0 | 0.067269 | 0.278844 |  0.26384 |   0.0 | 0.182831 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.056289 | 0.668723 | 0.185757 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.056289 |      0.0 | 0.068169 | 0.197146 | 0.631691 | 0.062681 |      0.0 | 0.067318 | 0.271214 |  0.26751 |   0.0 | 0.185757 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 |      0.0 | 0.658587 | 0.182941 |      0.0 |  0.0 |  0.0 |  133.98234 |      0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 | 0.056251 |      0.0 |  0.06735 | 0.263872 | 0.271115 |   0.0 | 0.182941 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 |      0.0 | 0.644356 | 0.178988 |      0.0 |  0.0 |  0.0 | 132.218945 |      0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 | 0.046054 |      0.0 | 0.067352 | 0.256325 | 0.274625 |   0.0 | 0.178988 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 |      0.0 | 0.631272 | 0.175353 |      0.0 |  0.0 |  0.0 | 129.797091 |      0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 | 0.037706 |      0.0 | 0.067313 | 0.248236 | 0.278017 |   0.0 | 0.175353 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 |      0.0 | 0.618762 | 0.171878 |      0.0 |  0.0 |  0.0 | 127.414065 |      0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 | 0.030871 |      0.0 | 0.067224 | 0.239396 | 0.281271 |   0.0 | 0.171878 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |      0.0 | 0.606442 | 0.168456 |      0.0 |  0.0 |  0.0 | 124.635441 |      0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 | 0.025275 |      0.0 | 0.067082 | 0.229713 | 0.284373 |   0.0 | 0.168456 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 |      0.0 | 0.594112 | 0.165031 |      0.0 |  0.0 |  0.0 | 122.150124 |      0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 | 0.020693 |      0.0 | 0.066883 |  0.21922 | 0.287315 |   0.0 | 0.165031 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 |      0.0 | 0.581716 | 0.161588 |      0.0 |  0.0 |  0.0 | 119.985132 |      0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 | 0.016942 |      0.0 |  0.06663 | 0.208046 | 0.290098 |   0.0 | 0.161588 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 |      0.0 | 0.569468 | 0.158185 |      0.0 |  0.0 |  0.0 | 118.139837 |      0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 | 0.013871 |      0.0 | 0.066332 | 0.196528 | 0.292737 |   0.0 | 0.158185 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 |      0.0 | 0.557558 | 0.154877 |      0.0 |  0.0 |  0.0 | 116.932422 |      0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 | 0.011357 |      0.0 |    0.066 | 0.184951 |  0.29525 |   0.0 | 0.154877 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 |      0.0 | 0.546245 | 0.151735 |      0.0 |  0.0 |  0.0 | 116.038409 |      0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 | 0.009298 |      0.0 | 0.065647 | 0.173638 | 0.297662 |   0.0 | 0.151735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 |      0.0 | 0.535741 | 0.148817 |      0.0 |  0.0 |  0.0 | 115.332585 |      0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 | 0.007613 |      0.0 | 0.065286 | 0.162844 | 0.299998 |   0.0 | 0.148817 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016212 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 |      0.0 | 0.526093 | 0.146137 |      0.0 |  0.0 |  0.0 |  114.67801 |      0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 | 0.006233 |      0.0 | 0.064923 | 0.152666 | 0.302271 |   0.0 | 0.146137 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 |      0.0 | 0.517266 | 0.143685 |      0.0 |  0.0 |  0.0 | 114.031165 |      0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 | 0.005103 |      0.0 | 0.064561 | 0.143114 | 0.304487 |   0.0 | 0.143685 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 |      0.0 |  0.50918 | 0.141439 |      0.0 |  0.0 |  0.0 | 113.391444 |      0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 | 0.004178 |      0.0 | 0.064201 | 0.134151 | 0.306649 |   0.0 | 0.141439 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 |      0.0 | 0.501753 | 0.139376 |      0.0 |  0.0 |  0.0 | 112.758402 |      0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 | 0.003421 |      0.0 | 0.063843 | 0.125732 | 0.308758 |   0.0 | 0.139376 |

.. _lland_v1_acker_routing:

acre (routing)
______________

For all non-water areas, |lland_v1| effectively just adds the inflow to the
catchment-specific outflow:

.. integration-test:

    >>> inlet.sequences.sim.series = 0.3
    >>> test("lland_v1_acker_routing",
    ...      axis1=(inputs.nied, fluxes.qah),
    ...      axis2=states.bowa)
    |                date | nied | teml |  glob |  qz |  qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.3 | 1.08 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 1.080495 | 0.300138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.3 | 0.300138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.3 | 1.08 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 1.081462 | 0.300406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.3 | 0.300406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.3 | 1.08 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 1.082394 | 0.300665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.3 | 0.300665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.3 | 1.08 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 1.083291 | 0.300914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.3 | 0.300914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.3 | 1.08 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 1.084156 | 0.301154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.3 | 0.301154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.3 | 1.08 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  1.08499 | 0.301386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.3 | 0.301386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.3 | 1.08 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 1.085793 | 0.301609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.3 | 0.301609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.3 | 1.08 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 1.086566 | 0.301824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.3 | 0.301824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.3 | 1.08 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 1.087306 |  0.30203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.3 |  0.30203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.3 | 1.08 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 1.088003 | 0.302223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.3 | 0.302223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.3 | 1.08 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 1.088643 | 0.302401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.3 | 0.302401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.3 | 1.08 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 1.089219 | 0.302561 |      0.0 |  0.0 |  0.0 |   17.83797 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.3 | 0.302561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.3 | 1.08 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 1.089727 | 0.302702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.3 | 0.302702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.3 | 1.08 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  1.09018 | 0.302828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.3 | 0.302828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.3 | 1.08 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 1.090586 | 0.302941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.3 | 0.302941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.3 | 1.08 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 1.090942 | 0.303039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.3 | 0.303039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.3 | 1.08 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 1.091263 | 0.303129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.3 | 0.303129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.3 | 1.08 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 1.091564 | 0.303212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.3 | 0.303212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.3 | 1.08 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  1.09363 | 0.303786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.3 | 0.303786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.3 | 1.08 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 1.114046 | 0.309457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.3 | 0.309457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.3 | 1.08 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 1.140472 | 0.316798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.3 | 0.316798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.3 | 1.08 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 1.168924 | 0.324701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.3 | 0.324701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.3 | 1.08 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  1.25003 | 0.347231 | 0.780374 |  0.0 |  0.0 |   44.22507 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.3 | 0.347231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.3 | 1.08 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 1.290341 | 0.358428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.3 | 0.358428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.3 | 1.08 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 1.263571 | 0.350992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.3 | 0.350992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.3 | 1.08 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 1.265428 | 0.351508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.3 | 0.351508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.3 | 1.08 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 1.278953 | 0.355265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.3 | 0.355265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.3 | 1.08 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 1.371718 | 0.381033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.3 | 0.381033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.3 | 1.08 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 1.464279 | 0.406744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.3 | 0.406744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.3 | 1.08 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 1.481551 | 0.411542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.3 | 0.411542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.3 | 1.08 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 1.471634 | 0.408787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.3 | 0.408787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.3 | 1.08 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  1.42564 | 0.396011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.3 | 0.396011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.3 | 1.08 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   1.3746 | 0.381833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.3 | 0.381833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.3 | 1.08 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 1.332616 | 0.370171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.3 | 0.370171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.3 | 1.08 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 1.298815 | 0.360782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.3 | 0.360782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.3 | 1.08 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 1.271684 | 0.353246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.3 | 0.353246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.3 | 1.08 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 1.286673 | 0.357409 | 0.125043 |  0.0 |  0.0 |  68.927693 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.3 | 0.357409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.3 | 1.08 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 1.368705 | 0.380196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.3 | 0.380196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.3 | 1.08 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 1.403726 | 0.389924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.3 | 0.389924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.3 | 1.08 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 1.901088 |  0.52808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.3 |  0.52808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.3 | 1.08 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 3.948463 | 1.096795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.3 | 1.096795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.3 | 1.08 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.504942 | 4.871851 | 1.353292 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.600789 | 0.904153 | 0.064494 | 0.140463 | 0.594939 | 0.837653 | 2.869278 |  0.02348 | 0.006795 | 0.054646 |   0.3 | 1.353292 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.3 | 1.08 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.045681 |  4.01515 | 1.115319 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.043686 | 0.001996 | 0.067538 | 0.186956 | 0.625375 | 0.923811 | 1.903847 | 0.025555 | 0.021764 | 0.060173 |   0.3 | 1.115319 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.3 | 1.08 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.527094 | 3.561423 | 0.989284 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.604289 | 0.922805 | 0.068447 | 0.201685 | 0.634469 | 0.998043 | 1.351716 | 0.027625 | 0.038197 | 0.065843 |   0.3 | 0.989284 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.3 | 1.08 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.227705 | 4.498899 | 1.249694 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.861643 | 5.366061 | 0.071072 | 0.246232 |  0.66072 | 1.132039 | 2.129639 |  0.02968 |  0.05591 | 0.071631 |   0.3 | 1.249694 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.3 | 1.08 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  6.97553 | 5.884338 | 1.634538 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.856642 | 5.118889 |  0.07804 | 0.377966 | 0.730399 | 1.263826 | 3.350409 |  0.03187 | 0.080394 |  0.07784 |   0.3 | 1.634538 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.3 | 1.08 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 |  3.09949 |  5.92592 | 1.646089 |  0.78175 |  0.0 |  0.0 | 169.469597 | 1.677366 | 1.422124 |  0.08319 | 0.486735 | 0.781897 | 1.354495 | 3.258614 | 0.034249 | 0.113973 |  0.08459 |   0.3 | 1.646089 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.3 | 1.08 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 0.948023 | 4.937115 | 1.371421 | 0.782063 |  0.0 |  0.0 | 169.380125 | 0.948023 |      0.0 | 0.084735 | 0.521116 | 0.797348 | 1.344716 | 2.233012 | 0.036674 | 0.151109 | 0.091605 |   0.3 | 1.371421 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.3 | 1.08 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 2.488688 | 4.281733 |  1.18937 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.598182 | 0.890506 |  0.08469 |  0.52011 | 0.796901 | 1.333698 | 1.544122 | 0.039016 | 0.186271 | 0.098625 |   0.3 |  1.18937 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.3 | 1.08 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 3.918425 | 1.088451 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.351893 | 1.120521 | 0.041268 |  0.21912 | 0.105622 |   0.3 | 1.088451 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.3 | 1.08 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 0.249877 | 3.427956 |  0.95221 | 0.782239 |  0.0 |  0.0 | 170.639599 | 0.249877 |      0.0 | 0.085875 | 0.546974 | 0.808745 | 1.242434 | 0.699367 | 0.043439 | 0.250108 | 0.112609 |   0.3 |  0.95221 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.3 | 1.08 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 |  2.99484 |   0.8319 | 0.782169 |  0.0 |  0.0 | 169.345117 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.047903 | 0.424187 | 0.045495 | 0.277747 | 0.119508 |   0.3 |   0.8319 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.3 | 1.08 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 0.242047 | 2.701283 | 0.750356 | 0.781959 |  0.0 |  0.0 | 168.284125 | 0.242047 |      0.0 | 0.084673 | 0.519716 | 0.796726 | 0.888843 | 0.257283 | 0.047421 | 0.301457 | 0.126278 |   0.3 | 0.750356 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.3 | 1.08 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 2.603064 | 0.723073 | 0.781681 |  0.0 |  0.0 |  168.57882 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 0.857129 | 0.162134 | 0.049225 | 0.321653 | 0.132923 |   0.3 | 0.723073 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.3 | 1.08 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.189947 | 2.534205 | 0.703946 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.189947 |      0.0 | 0.084289 | 0.511125 | 0.792894 | 0.820771 | 0.103491 | 0.050931 | 0.339529 | 0.139482 |   0.3 | 0.703946 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.3 | 1.08 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 2.393091 | 0.664748 | 0.705926 |  0.0 |  0.0 | 166.216858 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.696563 |  0.06277 | 0.052545 | 0.355257 | 0.145956 |   0.3 | 0.664748 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.3 | 1.08 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 |      0.0 | 2.270388 | 0.630663 | 0.471181 |  0.0 |  0.0 | 164.867718 |      0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 | 0.577717 | 0.038072 |  0.05405 | 0.368242 | 0.152307 |   0.3 | 0.630663 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.3 | 1.08 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 2.168702 | 0.602417 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.472994 | 0.023092 | 0.055451 | 0.378635 | 0.158529 |   0.3 | 0.602417 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.3 | 1.08 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 |      0.0 | 2.089088 | 0.580302 |      0.0 |  0.0 |  0.0 | 161.377451 |      0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 | 0.387255 | 0.014006 | 0.056746 | 0.386466 | 0.164615 |   0.3 | 0.580302 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.3 | 1.08 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 2.108002 | 0.585556 |      0.0 |  0.0 |  0.0 | 161.294054 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.399248 | 0.008495 | 0.057936 |  0.39177 | 0.170553 |   0.3 | 0.585556 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.3 | 1.08 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.014983 | 2.121382 | 0.589273 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.014983 |      0.0 | 0.080647 | 0.431887 |  0.75647 | 0.405172 | 0.005153 | 0.059044 | 0.395629 | 0.176385 |   0.3 | 0.589273 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.3 | 1.08 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 2.056673 | 0.571298 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.333039 | 0.003125 | 0.060079 | 0.398311 | 0.182119 |   0.3 | 0.571298 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.3 | 1.08 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 |      0.0 | 2.002391 |  0.55622 |      0.0 |  0.0 |  0.0 | 155.593338 |      0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 | 0.272669 | 0.001896 | 0.061023 |  0.39909 | 0.187714 |   0.3 |  0.55622 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.3 | 1.08 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 1.957158 | 0.543655 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.223243 |  0.00115 |  0.06187 | 0.397747 | 0.193148 |   0.3 | 0.543655 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.3 | 1.08 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 |      0.0 | 1.918885 | 0.533024 |      0.0 |  0.0 |  0.0 | 151.294475 |      0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 | 0.182776 | 0.000697 | 0.062621 | 0.394374 | 0.198417 |   0.3 | 0.533024 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.3 | 1.08 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 1.886179 | 0.523939 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.149644 | 0.000423 | 0.063282 | 0.389303 | 0.203526 |   0.3 | 0.523939 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.3 | 1.08 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 |      0.0 | 1.858035 | 0.516121 |      0.0 |  0.0 |  0.0 | 148.167947 |      0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 | 0.122518 | 0.000257 | 0.063864 | 0.382909 | 0.208487 |   0.3 | 0.516121 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.3 | 1.08 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 1.833875 |  0.50941 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.100309 | 0.000156 | 0.064379 |  0.37571 | 0.213321 |   0.3 |  0.50941 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.3 | 1.08 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 |      0.0 | 1.813188 | 0.503663 |      0.0 |  0.0 |  0.0 | 145.983055 |      0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 | 0.082126 | 0.000094 | 0.064839 | 0.368084 | 0.218044 |   0.3 | 0.503663 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.3 | 1.08 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 1.795434 | 0.498732 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.067239 | 0.000057 | 0.065249 | 0.360222 | 0.222667 |   0.3 | 0.498732 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.3 | 1.08 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 |      0.0 | 1.780075 | 0.494465 |      0.0 |  0.0 |  0.0 | 143.898606 |      0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 | 0.055051 | 0.000035 | 0.065614 | 0.352185 |  0.22719 |   0.3 | 0.494465 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.3 | 1.08 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 | 1.766662 | 0.490739 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 | 0.045072 | 0.000021 | 0.065935 | 0.344017 | 0.231617 |   0.3 | 0.490739 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.3 | 1.08 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |      0.0 |  1.75483 | 0.487453 |      0.0 |  0.0 |  0.0 | 141.875122 |      0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 | 0.036902 | 0.000013 | 0.066216 | 0.335751 | 0.235948 |   0.3 | 0.487453 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.3 | 1.08 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 1.744287 | 0.484524 |      0.0 |  0.0 |  0.0 | 140.884763 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.030213 | 0.000008 | 0.066458 | 0.327422 | 0.240186 |   0.3 | 0.484524 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 |      0.0 | 1.734794 | 0.481887 |      0.0 |  0.0 |  0.0 | 139.908406 |      0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 | 0.024736 | 0.000005 | 0.066665 | 0.319056 | 0.244333 |   0.3 | 0.481887 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.3 | 1.08 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 1.746919 | 0.485255 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.041012 | 0.000003 | 0.066837 | 0.310678 | 0.248389 |   0.3 | 0.485255 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.3 | 1.08 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 |      0.0 | 1.754893 |  0.48747 | 0.766603 |  0.0 |  0.0 | 138.559658 |      0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 | 0.052999 | 0.000002 | 0.066984 | 0.302538 |  0.25237 |   0.3 |  0.48747 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 |      0.0 | 1.741405 | 0.483724 |  0.74963 |  0.0 |  0.0 | 137.632082 |      0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 | 0.043392 | 0.000001 | 0.067107 | 0.294626 | 0.256278 |   0.3 | 0.483724 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.3 | 1.08 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 |      0.0 | 1.729552 | 0.480431 | 0.687585 |  0.0 |  0.0 | 136.717337 |      0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 | 0.035526 | 0.000001 | 0.067202 | 0.286722 | 0.260101 |   0.3 | 0.480431 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.3 | 1.08 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 | 0.204485 | 1.738192 | 0.482831 | 0.521608 |  0.0 |  0.0 | 136.338253 | 0.204485 |      0.0 | 0.068359 | 0.200239 | 0.633587 | 0.048237 |      0.0 | 0.067269 | 0.278844 |  0.26384 |   0.3 | 0.482831 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.3 | 1.08 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.056289 | 1.748723 | 0.485757 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.056289 |      0.0 | 0.068169 | 0.197146 | 0.631691 | 0.062681 |      0.0 | 0.067318 | 0.271214 |  0.26751 |   0.3 | 0.485757 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.3 | 1.08 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 |      0.0 | 1.738587 | 0.482941 |      0.0 |  0.0 |  0.0 |  133.98234 |      0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 | 0.056251 |      0.0 |  0.06735 | 0.263872 | 0.271115 |   0.3 | 0.482941 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.3 | 1.08 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 |      0.0 | 1.724356 | 0.478988 |      0.0 |  0.0 |  0.0 | 132.218945 |      0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 | 0.046054 |      0.0 | 0.067352 | 0.256325 | 0.274625 |   0.3 | 0.478988 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.3 | 1.08 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 |      0.0 | 1.711272 | 0.475353 |      0.0 |  0.0 |  0.0 | 129.797091 |      0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 | 0.037706 |      0.0 | 0.067313 | 0.248236 | 0.278017 |   0.3 | 0.475353 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.3 | 1.08 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 |      0.0 | 1.698762 | 0.471878 |      0.0 |  0.0 |  0.0 | 127.414065 |      0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 | 0.030871 |      0.0 | 0.067224 | 0.239396 | 0.281271 |   0.3 | 0.471878 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.3 | 1.08 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |      0.0 | 1.686442 | 0.468456 |      0.0 |  0.0 |  0.0 | 124.635441 |      0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 | 0.025275 |      0.0 | 0.067082 | 0.229713 | 0.284373 |   0.3 | 0.468456 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.3 | 1.08 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 |      0.0 | 1.674112 | 0.465031 |      0.0 |  0.0 |  0.0 | 122.150124 |      0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 | 0.020693 |      0.0 | 0.066883 |  0.21922 | 0.287315 |   0.3 | 0.465031 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.3 | 1.08 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 |      0.0 | 1.661716 | 0.461588 |      0.0 |  0.0 |  0.0 | 119.985132 |      0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 | 0.016942 |      0.0 |  0.06663 | 0.208046 | 0.290098 |   0.3 | 0.461588 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.3 | 1.08 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 |      0.0 | 1.649468 | 0.458185 |      0.0 |  0.0 |  0.0 | 118.139837 |      0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 | 0.013871 |      0.0 | 0.066332 | 0.196528 | 0.292737 |   0.3 | 0.458185 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.3 | 1.08 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 |      0.0 | 1.637558 | 0.454877 |      0.0 |  0.0 |  0.0 | 116.932422 |      0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 | 0.011357 |      0.0 |    0.066 | 0.184951 |  0.29525 |   0.3 | 0.454877 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.3 | 1.08 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 |      0.0 | 1.626245 | 0.451735 |      0.0 |  0.0 |  0.0 | 116.038409 |      0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 | 0.009298 |      0.0 | 0.065647 | 0.173638 | 0.297662 |   0.3 | 0.451735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.3 | 1.08 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 |      0.0 | 1.615741 | 0.448817 |      0.0 |  0.0 |  0.0 | 115.332585 |      0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 | 0.007613 |      0.0 | 0.065286 | 0.162844 | 0.299998 |   0.3 | 0.448817 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.3 | 1.08 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016212 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 |      0.0 | 1.606093 | 0.446137 |      0.0 |  0.0 |  0.0 |  114.67801 |      0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 | 0.006233 |      0.0 | 0.064923 | 0.152666 | 0.302271 |   0.3 | 0.446137 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.3 | 1.08 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 |      0.0 | 1.597266 | 0.443685 |      0.0 |  0.0 |  0.0 | 114.031165 |      0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 | 0.005103 |      0.0 | 0.064561 | 0.143114 | 0.304487 |   0.3 | 0.443685 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.3 | 1.08 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 |      0.0 |  1.58918 | 0.441439 |      0.0 |  0.0 |  0.0 | 113.391444 |      0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 | 0.004178 |      0.0 | 0.064201 | 0.134151 | 0.306649 |   0.3 | 0.441439 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 |      0.0 | 1.581753 | 0.439376 |      0.0 |  0.0 |  0.0 | 112.758402 |      0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 | 0.003421 |      0.0 | 0.063843 | 0.125732 | 0.308758 |   0.3 | 0.439376 |

.. _lland_v1_wasser:

water
_____

L-Land defines three types of water areas. The first one, |WASSER|, is also
implemented in the original `LARSIM`_ model.  To represent the effects of
water areas in a straightforward manner, |lland_v1| adds precipitation
(|NKor|) to and tries to remove potential evaporation (|EvPo|) from the
total discharge (|lland_fluxes.QAH|.

In the following example (again with zero inflow), this simple approach has
the unfavourable side effect of discharge dropping to zero in periods with no
precipitation but relevant potential evaporation during the daytime.
Similar problems arise when |WASSER| is only one of many selected
land-use types, possibly even when the water area is below 1 % of the total
catchment area.  Hence, it seems advisable to use the land-use types |FLUSS|
and |SEE| instead, under most circumstances.  Note that sequence |EvI|
contains the actual evaporation values of |WASSER| (and other water-type)
response units:

.. integration-test::

    >>> lnk(WASSER)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_v1_wasser",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.409803 | 0.391612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.391612 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.675221 | 1.854228 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.854228 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.460115 | 0.961143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.961143 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039381 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.860309 | 1.627864 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.627864 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 12.700374 | 3.527882 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 3.527882 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.100505 | 0.027918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027918 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.820505 | 0.227918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.227918 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.580505 | 0.994585 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.994585 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.500604 | 0.694612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.694612 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  12.46067 | 3.461297 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 3.461297 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.180736 | 1.161315 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.161315 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.060802 | 1.128001 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.128001 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.40254 | 0.389595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.389595 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.645043 | 1.012512 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.012512 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.315689 |  1.47658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.47658 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.510795 | 0.141887 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.141887 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 24.417849 | 6.782736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 6.782736 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 45.032149 | 12.50893 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 12.50893 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   9.26443 | 2.573453 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 2.573453 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   4.00171 | 1.111586 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.111586 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.941679 |   2.4838 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   2.4838 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 22.181476 | 6.161521 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 6.161521 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  18.46151 | 5.128197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 5.128197 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   7.54175 | 2.094931 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 2.094931 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.262063 | 0.628351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.628351 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.862028 | 1.628341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 1.628341 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.222063 | 0.895018 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.895018 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.582239 | 0.161733 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.161733 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.222169 | 0.061713 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.061713 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.581959 | 0.161655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.161655 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.861681 | 0.794911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.794911 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061945 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.449028 |  0.12473 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.12473 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.145926 | 0.040535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.040535 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.058849 | 0.571903 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.571903 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 |     0.84 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033433 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.543283 |  0.42869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.42869 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124091 | 0.062046 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.561608 | 0.156002 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.156002 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 | 1.278351 | 0.639175 |     0.48 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |     0.12 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

.. _lland_v1_wasser_routing:

water (routing)
_______________

With given inflow, more water is available for evaporation, and the
differences between potential and actual evaporation are less pronounced
(hence, the routing approach offers the opportunity to capture the
evaporation effects of small lakes one does not want to represent by a
separate specialised lake model like |dam_v006|):

.. integration-test::

    >>> inlet.sequences.sim.series = 0.3
    >>> test("lland_v1_wasser_routing",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |  glob |  qz |  qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.3 | 1.08 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020141 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.059859 | 0.294405 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294405 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.3 | 1.08 |   0.0 | 20.2 |  0.0 |  0.0 | 0.039121 |  0.01956 |  0.01956 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.06044 | 0.294567 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294567 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.3 | 1.08 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060604 | 0.294612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294612 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.3 | 1.08 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060802 | 0.294667 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294667 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.3 | 1.08 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.060604 | 0.294612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294612 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.3 | 1.08 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041105 | 0.020552 | 0.020552 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.059448 | 0.294291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.294291 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.3 | 1.08 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.058382 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.021618 | 0.283783 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.283783 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.3 | 1.08 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 | 0.386575 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.693425 | 0.192618 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.192618 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.3 | 1.08 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.873907 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.206093 | 0.057248 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.057248 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.3 | 1.08 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.3 | 1.08 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.3 | 1.08 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.3 | 1.08 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 | 1.025236 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.054764 | 0.015212 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.015212 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.3 | 1.08 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.3 | 1.08 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.3 | 1.08 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.478302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.841698 | 0.233805 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.233805 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.3 | 1.08 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731933 | 0.365967 | 0.365967 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.714033 | 0.198343 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.198343 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.3 | 1.08 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | 0.227314 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.852686 | 0.236857 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.236857 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.3 | 1.08 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.489803 | 0.691612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.691612 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.3 | 1.08 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  7.755221 | 2.154228 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.154228 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.3 | 1.08 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.540115 | 1.261143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.261143 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.3 | 1.08 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039381 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.940309 | 1.927864 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.927864 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.3 | 1.08 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.780374 | 3.827882 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 3.827882 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.3 | 1.08 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.180505 | 0.327918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.327918 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.3 | 1.08 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.900505 | 0.527918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.527918 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.3 | 1.08 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.660505 | 1.294585 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.294585 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.3 | 1.08 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.580604 | 0.994612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.994612 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.3 | 1.08 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  13.54067 | 3.761297 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 3.761297 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.3 | 1.08 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.260736 | 1.461315 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.461315 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.3 | 1.08 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.140802 | 1.428001 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.428001 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.3 | 1.08 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   2.48254 | 0.689595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.689595 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.3 | 1.08 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.947078 | 0.263077 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.263077 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.3 | 1.08 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.470892 | 0.130803 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.130803 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.3 | 1.08 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.693848 | 0.192736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.192736 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.3 | 1.08 |  0.48 | 20.0 |  0.0 |  0.0 | 0.984401 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0678 | 0.296611 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.296611 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.3 | 1.08 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 |      1.2 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.3 | 1.08 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.725043 | 1.312512 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.312512 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.3 | 1.08 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.395689 |  1.77658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  1.77658 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.3 | 1.08 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.590795 | 0.441887 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.441887 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.3 | 1.08 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 25.497849 | 7.082736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 7.082736 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.3 | 1.08 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 46.112149 | 12.80893 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 12.80893 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.3 | 1.08 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  10.34443 | 2.873453 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.873453 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.3 | 1.08 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.08171 | 1.411586 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.411586 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.3 | 1.08 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 10.021679 |   2.7838 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |   2.7838 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.3 | 1.08 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 23.261476 | 6.461521 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 6.461521 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.3 | 1.08 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  19.54151 | 5.428197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 5.428197 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.3 | 1.08 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   8.62175 | 2.394931 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 2.394931 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.3 | 1.08 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.342063 | 0.928351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.928351 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.3 | 1.08 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.942028 | 1.928341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.928341 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.3 | 1.08 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.302063 | 1.195018 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.195018 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.3 | 1.08 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.662239 | 0.461733 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.461733 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.3 | 1.08 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.302169 | 0.361713 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.361713 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.3 | 1.08 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.661959 | 0.461655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.461655 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.3 | 1.08 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.941681 | 1.094911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 1.094911 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.3 | 1.08 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061945 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.529028 |  0.42473 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  0.42473 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.3 | 1.08 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.225926 | 0.340535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.340535 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.3 | 1.08 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.845255 | 0.234793 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.234793 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.3 | 1.08 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.239541 | 0.066539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.066539 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.3 | 1.08 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.568057 | 0.157794 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.157794 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.3 | 1.08 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.138849 | 0.871903 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.871903 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.3 | 1.08 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.844795 | 0.234665 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.234665 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.3 | 1.08 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.508217 | 0.141171 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.141171 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.3 | 1.08 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 |     1.44 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.3 | 1.08 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.109687 | 0.030468 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.030468 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.3 | 1.08 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.018616 | 0.005171 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.005171 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.3 | 1.08 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.422793 | 0.117442 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.117442 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.3 | 1.08 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.778757 | 0.216321 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.216321 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.3 | 1.08 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.025048 | 0.284735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.284735 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.3 | 1.08 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062487 | 0.295135 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295135 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.3 | 1.08 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062846 | 0.295235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295235 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.3 | 1.08 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063137 | 0.295316 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295316 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.3 | 1.08 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063357 | 0.295377 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295377 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.3 | 1.08 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063542 | 0.295428 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295428 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.3 | 1.08 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0631 | 0.295306 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295306 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.3 | 1.08 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033433 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.623283 |  0.72869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |  0.72869 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.3 | 1.08 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.06332 | 0.295367 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295367 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.3 | 1.08 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.017954 | 0.282765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.282765 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.3 | 1.08 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.641608 | 0.456002 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.456002 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.3 | 1.08 |  0.48 | 12.2 |  0.0 |  0.0 | 1.278351 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.920825 | 0.255785 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.255785 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.3 | 1.08 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.16167 | 0.044908 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.044908 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.3 | 1.08 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.128993 | 0.035831 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.035831 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.3 | 1.08 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.3 | 1.08 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.3 | 1.08 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143047 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.3 | 1.08 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.3 | 1.08 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.3 | 1.08 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 |     1.08 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 |      0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.3 | 1.08 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179367 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.490317 | 0.136199 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.136199 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.3 | 1.08 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.818154 | 0.227265 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.227265 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.3 | 1.08 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.01409 | 0.281692 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.281692 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.3 | 1.08 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062098 | 0.295027 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295027 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.3 | 1.08 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.062846 | 0.295235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295235 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.3 | 1.08 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |    1.0631 | 0.295306 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295306 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.3 | 1.08 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.063027 | 0.295285 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.3 | 0.295285 |

.. _lland_v1_wasser_negq:

water (negative runoff)
_______________________

In the integration test :ref:`lland_v1_wasser`, discharge is zero in
all periods with evaporation exceeding precipitation but never negative.
This non-negativity is due to trimming flux sequence |lland_fluxes.QAH|,
which we can disable by setting parameter |NegQ| to |True| (negative values
might be problematic for the models downstream, so set |NegQ| to |True|
only for good reasons).  When allowing negative |lland_fluxes.QAH| values,
potential evaporation (|EvPo|) and actual water evaporation (|EvI|) are
always identical:

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_v1_wasser_negq",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |        qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020141 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.020141 | -0.005595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005595 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 | 0.039121 |  0.01956 |  0.01956 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.01956 | -0.005433 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005433 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019396 | -0.005388 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005388 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019198 | -0.005333 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005333 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.019396 | -0.005388 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005388 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041105 | 0.020552 | 0.020552 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.020552 | -0.005709 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.005709 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.058382 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.058382 | -0.016217 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.016217 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 | 0.386575 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.386575 | -0.107382 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.107382 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.873907 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.873907 | -0.242752 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.242752 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 | 1.463511 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.463511 | -0.406531 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.406531 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 | 1.652872 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.652872 | -0.459131 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.459131 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 | 1.873973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.873973 | -0.520548 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.520548 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 | 1.025236 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.025236 | -0.284788 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.284788 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 | 1.325006 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.325006 | -0.368057 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.368057 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 | 1.479524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.479524 | -0.410979 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.410979 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.478302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.238302 | -0.066195 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.066195 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731933 | 0.365967 | 0.365967 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.365967 | -0.101657 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.101657 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | 0.227314 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.227314 | -0.063143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.063143 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.409803 |  0.391612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.391612 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  6.675221 |  1.854228 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.854228 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.460115 |  0.961143 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.961143 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039381 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.860309 |  1.627864 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.627864 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 12.700374 |  3.527882 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  3.527882 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.100505 |  0.027918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.027918 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.820505 |  0.227918 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.227918 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.580505 |  0.994585 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.994585 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.500604 |  0.694612 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.694612 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  12.46067 |  3.461297 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  3.461297 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.180736 |  1.161315 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.161315 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  4.060802 |  1.128001 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.128001 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   1.40254 |  0.389595 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.389595 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.132922 | -0.036923 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.036923 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.609108 | -0.169197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.169197 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.386152 | -0.107264 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.107264 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 | 0.984401 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0122 | -0.003389 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.003389 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 | 1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.123189 | -0.311997 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.311997 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.645043 |  1.012512 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.012512 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.315689 |   1.47658 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   1.47658 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.510795 |  0.141887 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.141887 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 24.417849 |  6.782736 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.782736 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 45.032149 |  12.50893 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  12.50893 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   9.26443 |  2.573453 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.573453 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   4.00171 |  1.111586 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.111586 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.941679 |    2.4838 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |    2.4838 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 22.181476 |  6.161521 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  6.161521 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  18.46151 |  5.128197 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  5.128197 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   7.54175 |  2.094931 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  2.094931 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.262063 |  0.628351 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.628351 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  5.862028 |  1.628341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  1.628341 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  3.222063 |  0.895018 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.895018 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.582239 |  0.161733 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.161733 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.222169 |  0.061713 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.061713 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.581959 |  0.161655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.161655 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.861681 |  0.794911 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.794911 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061945 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.449028 |   0.12473 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.12473 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.145926 |  0.040535 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.040535 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.234745 | -0.065207 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.065207 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.840459 | -0.233461 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.233461 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.511943 | -0.142206 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.142206 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.058849 |  0.571903 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.571903 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.235205 | -0.065335 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.065335 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.571783 | -0.158829 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.158829 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 | 1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.197919 | -0.332755 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.332755 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.970313 | -0.269532 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.269532 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.061384 | -0.294829 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.294829 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.657207 | -0.182558 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.182558 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.301243 | -0.083679 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.083679 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.054952 | -0.015265 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.015265 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017513 | -0.004865 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004865 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017154 | -0.004765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004765 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016863 | -0.004684 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004684 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016643 | -0.004623 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004623 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016458 | -0.004572 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004572 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0169 | -0.004694 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004694 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033433 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.543283 |   0.42869 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.42869 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.01668 | -0.004633 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004633 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.062046 | -0.017235 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017235 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.561608 |  0.156002 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.156002 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 | 1.278351 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.159175 | -0.044215 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.044215 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.91833 | -0.255092 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.255092 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.951007 | -0.264169 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.264169 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 | 1.678286 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.678286 | -0.466191 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.466191 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 | 1.678257 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.678257 | -0.466183 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.466183 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143047 | 2.143047 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.143047 | -0.595291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.595291 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 | 1.876679 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.876679 |   -0.5213 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   -0.5213 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 | 1.572383 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.572383 | -0.436773 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.436773 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 | 1.259666 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.259666 | -0.349907 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.349907 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179367 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.589683 | -0.163801 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.163801 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.261846 | -0.072735 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.072735 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.06591 | -0.018308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.018308 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017902 | -0.004973 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004973 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.017154 | -0.004765 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004765 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   -0.0169 | -0.004694 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004694 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.016973 | -0.004715 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.004715 |

.. _lland_v1_see:

lakes
_____

As an alternative for water type |WASSER|, HydPy-L offers water type |SEE|
for representing lakes not directly connected to the stream network but the
groundwater.  In some agreement with the implementation of "internal lakes"
in the HBV96 model (see |hland|), precipitation and evaporation values are
directly added and removed from the input of the linear storage for base
flow (|QBGZ|).  Hence, defining |SEE| areas results in reduced responsiveness
of a catchment.

Note that we again disable the |NegQ| option.  However, in contrast to
integration test :ref:`lland_v1_wasser`, the actual evaporation (|EvI|)
is identical with the potential one (|EvPo|) during and after the
precipitation event.  Only at the beginning of the simulation period, we
can see the result slight correction, because we are starting our test
with zero groundwater storage (method |Calc_QAH_V1| gives more information
on the adjustment of |lland_fluxes.QAH| and |EvI|):

.. integration-test::

    >>> lnk(SEE)
    >>> negq(False)
    >>> test("lland_v1_see",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |      evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 | 0.020041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020141 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0001 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 | 0.039121 |  0.01956 | 0.019263 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01956 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000297 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.018909 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000488 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.018523 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019198 |   0.0 |   0.0 |   0.0 |   0.0 | -0.000675 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.018536 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.019396 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00086 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041105 | 0.020552 | 0.019502 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.020552 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00105 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 | 0.056949 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058382 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001433 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 |  0.38294 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.003635 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 | 0.864033 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.873907 |   0.0 |   0.0 |   0.0 |   0.0 | -0.009874 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 | 1.442102 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.463511 |   0.0 |   0.0 |   0.0 |   0.0 | -0.021409 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |  1.61617 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.652872 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036702 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 | 1.820088 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.873973 |   0.0 |   0.0 |   0.0 |   0.0 | -0.053885 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 |  0.95747 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.025236 |   0.0 |   0.0 |   0.0 |   0.0 | -0.067766 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 | 1.246219 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.325006 |   0.0 |   0.0 |   0.0 |   0.0 | -0.078787 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 | 1.387567 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.479524 |   0.0 |   0.0 |   0.0 |   0.0 | -0.091957 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | 0.378724 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.238302 |   0.0 |   0.0 |   0.0 |   0.0 | -0.099578 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731933 | 0.365967 | 0.264372 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.365967 |   0.0 |   0.0 |   0.0 |   0.0 | -0.101594 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 |  0.12378 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.227314 |   0.0 |   0.0 |   0.0 |   0.0 | -0.103534 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 |  0.05359 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.409803 |   0.0 |   0.0 |   0.0 |   0.0 | -0.096607 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 |  -0.0106 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  6.675221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.055379 |   0.0 |      0.0 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 | 0.015455 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.460115 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.00443 |   0.0 |      0.0 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039381 | 0.019691 | 0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.042004 | 0.011668 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.860309 |   0.0 |   0.0 |   0.0 |   0.0 |  0.042004 |   0.0 | 0.011668 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.133983 | 0.037218 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 12.700374 |   0.0 |   0.0 |   0.0 |   0.0 |  0.133983 |   0.0 | 0.037218 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.196231 | 0.054509 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.100505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.196231 |   0.0 | 0.054509 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.198867 | 0.055241 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.820505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.198867 |   0.0 | 0.055241 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.218806 |  0.06078 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.580505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.218806 |   0.0 |  0.06078 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.246874 | 0.068576 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.500604 |   0.0 |   0.0 |   0.0 |   0.0 |  0.246874 |   0.0 | 0.068576 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.318934 | 0.088593 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  12.46067 |   0.0 |   0.0 |   0.0 |   0.0 |  0.318934 |   0.0 | 0.088593 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.398484 |  0.11069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.180736 |   0.0 |   0.0 |   0.0 |   0.0 |  0.398484 |   0.0 |  0.11069 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.435521 | 0.120978 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  4.060802 |   0.0 |   0.0 |   0.0 |   0.0 |  0.435521 |   0.0 | 0.120978 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.458346 | 0.127318 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   1.40254 |   0.0 |   0.0 |   0.0 |   0.0 |  0.458346 |   0.0 | 0.127318 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.460089 | 0.127802 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.132922 |   0.0 |   0.0 |   0.0 |   0.0 |  0.460089 |   0.0 | 0.127802 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 | 0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.451815 | 0.125504 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.609108 |   0.0 |   0.0 |   0.0 |   0.0 |  0.451815 |   0.0 | 0.125504 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 | 0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.44237 | 0.122881 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.386152 |   0.0 |   0.0 |   0.0 |   0.0 |   0.44237 |   0.0 | 0.122881 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 | 0.984401 |   0.4922 |   0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.435989 | 0.121108 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0122 |   0.0 |   0.0 |   0.0 |   0.0 |  0.435989 |   0.0 | 0.121108 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 | 1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.425993 | 0.118332 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.123189 |   0.0 |   0.0 |   0.0 |   0.0 |  0.425993 |   0.0 | 0.118332 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.434341 |  0.12065 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.645043 |   0.0 |   0.0 |   0.0 |   0.0 |  0.434341 |   0.0 |  0.12065 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 | 1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.474613 | 0.131837 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.315689 |   0.0 |   0.0 |   0.0 |   0.0 |  0.474613 |   0.0 | 0.131837 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 | 0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.498838 | 0.138566 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.510795 |   0.0 |   0.0 |   0.0 |   0.0 |  0.498838 |   0.0 | 0.138566 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.618095 | 0.171693 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 24.417849 |   0.0 |   0.0 |   0.0 |   0.0 |  0.618095 |   0.0 | 0.171693 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.957635 |  0.26601 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 45.032149 |   0.0 |   0.0 |   0.0 |   0.0 |  0.957635 |   0.0 |  0.26601 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.21794 | 0.338317 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   9.26443 |   0.0 |   0.0 |   0.0 |   0.0 |   1.21794 |   0.0 | 0.338317 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.271778 | 0.353272 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   4.00171 |   0.0 |   0.0 |   0.0 |   0.0 |  1.271778 |   0.0 | 0.353272 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.323559 | 0.367655 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.941679 |   0.0 |   0.0 |   0.0 |   0.0 |  1.323559 |   0.0 | 0.367655 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.465339 | 0.407039 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 22.181476 |   0.0 |   0.0 |   0.0 |   0.0 |  1.465339 |   0.0 | 0.407039 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.65293 | 0.459147 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  18.46151 |   0.0 |   0.0 |   0.0 |   0.0 |   1.65293 |   0.0 | 0.459147 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.765761 | 0.490489 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   7.54175 |   0.0 |   0.0 |   0.0 |   0.0 |  1.765761 |   0.0 | 0.490489 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.796922 | 0.499145 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.262063 |   0.0 |   0.0 |   0.0 |   0.0 |  1.796922 |   0.0 | 0.499145 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.819491 | 0.505414 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  5.862028 |   0.0 |   0.0 |   0.0 |   0.0 |  1.819491 |   0.0 | 0.505414 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.846559 | 0.512933 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  3.222063 |   0.0 |   0.0 |   0.0 |   0.0 |  1.846559 |   0.0 | 0.512933 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.84709 | 0.513081 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.582239 |   0.0 |   0.0 |   0.0 |   0.0 |   1.84709 |   0.0 | 0.513081 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.83271 | 0.509086 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.222169 |   0.0 |   0.0 |   0.0 |   0.0 |   1.83271 |   0.0 | 0.509086 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.818478 | 0.505133 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.581959 |   0.0 |   0.0 |   0.0 |   0.0 |  1.818478 |   0.0 | 0.505133 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.817535 | 0.504871 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.861681 |   0.0 |   0.0 |   0.0 |   0.0 |  1.817535 |   0.0 | 0.504871 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061945 | 0.030972 | 0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.815901 | 0.504417 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.449028 |   0.0 |   0.0 |   0.0 |   0.0 |  1.815901 |   0.0 | 0.504417 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.80079 |  0.50022 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.145926 |   0.0 |   0.0 |   0.0 |   0.0 |   1.80079 |   0.0 |  0.50022 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.782427 | 0.495119 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.234745 |   0.0 |   0.0 |   0.0 |   0.0 |  1.782427 |   0.0 | 0.495119 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 | 0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.759337 | 0.488705 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.840459 |   0.0 |   0.0 |   0.0 |   0.0 |  1.759337 |   0.0 | 0.488705 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 | 0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.735106 | 0.481974 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.511943 |   0.0 |   0.0 |   0.0 |   0.0 |  1.735106 |   0.0 | 0.481974 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 | 1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.725559 | 0.479322 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  2.058849 |   0.0 |   0.0 |   0.0 |   0.0 |  1.725559 |   0.0 | 0.479322 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 | 1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.717443 | 0.477067 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.235205 |   0.0 |   0.0 |   0.0 |   0.0 |  1.717443 |   0.0 | 0.477067 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 | 0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.696336 | 0.471205 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.571783 |   0.0 |   0.0 |   0.0 |   0.0 |  1.696336 |   0.0 | 0.471205 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 | 1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.670648 | 0.464069 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.197919 |   0.0 |   0.0 |   0.0 |   0.0 |  1.670648 |   0.0 | 0.464069 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 | 0.970313 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.64324 | 0.456455 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.970313 |   0.0 |   0.0 |   0.0 |   0.0 |   1.64324 |   0.0 | 0.456455 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 | 1.061384 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.61678 | 0.449106 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.061384 |   0.0 |   0.0 |   0.0 |   0.0 |   1.61678 |   0.0 | 0.449106 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 | 0.657207 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.592146 | 0.442263 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.657207 |   0.0 |   0.0 |   0.0 |   0.0 |  1.592146 |   0.0 | 0.442263 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | 0.301243 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.571539 | 0.436539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.301243 |   0.0 |   0.0 |   0.0 |   0.0 |  1.571539 |   0.0 | 0.436539 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 | 0.054952 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.554132 | 0.431703 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.054952 |   0.0 |   0.0 |   0.0 |   0.0 |  1.554132 |   0.0 | 0.431703 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | 0.017513 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.538308 | 0.427308 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017513 |   0.0 |   0.0 |   0.0 |   0.0 |  1.538308 |   0.0 | 0.427308 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.522829 | 0.423008 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 |  1.522829 |   0.0 | 0.423008 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | 0.016863 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.507507 | 0.418752 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016863 |   0.0 |   0.0 |   0.0 |   0.0 |  1.507507 |   0.0 | 0.418752 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | 0.016643 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.49234 | 0.414539 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016643 |   0.0 |   0.0 |   0.0 |   0.0 |   1.49234 |   0.0 | 0.414539 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | 0.016458 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.477327 | 0.410369 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016458 |   0.0 |   0.0 |   0.0 |   0.0 |  1.477327 |   0.0 | 0.410369 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.462461 | 0.406239 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 |  1.462461 |   0.0 | 0.406239 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.447741 |  0.40215 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.447741 |   0.0 |  0.40215 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033433 | 0.016717 | 0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.440942 | 0.400262 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.543283 |   0.0 |   0.0 |   0.0 |   0.0 |  1.440942 |   0.0 | 0.400262 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.434187 | 0.398385 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.01668 |   0.0 |   0.0 |   0.0 |   0.0 |  1.434187 |   0.0 | 0.398385 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.419749 | 0.394375 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.419749 |   0.0 | 0.394375 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.405228 | 0.390341 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.062046 |   0.0 |   0.0 |   0.0 |   0.0 |  1.405228 |   0.0 | 0.390341 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.393737 | 0.387149 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.561608 |   0.0 |   0.0 |   0.0 |   0.0 |  1.393737 |   0.0 | 0.387149 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 | 1.278351 | 0.639175 | 0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.381865 | 0.383851 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.159175 |   0.0 |   0.0 |   0.0 |   0.0 |  1.381865 |   0.0 | 0.383851 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |  1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.362748 | 0.378541 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.91833 |   0.0 |   0.0 |   0.0 |   0.0 |  1.362748 |   0.0 | 0.378541 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 | 1.431007 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.339888 | 0.372191 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.951007 |   0.0 |   0.0 |   0.0 |   0.0 |  1.339888 |   0.0 | 0.372191 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 | 1.678286 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.313469 | 0.364853 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678286 |   0.0 |   0.0 |   0.0 |   0.0 |  1.313469 |   0.0 | 0.364853 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 | 1.678257 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.283701 | 0.356584 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.678257 |   0.0 |   0.0 |   0.0 |   0.0 |  1.283701 |   0.0 | 0.356584 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143047 | 2.143047 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.251913 | 0.347754 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.143047 |   0.0 |   0.0 |   0.0 |   0.0 |  1.251913 |   0.0 | 0.347754 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 | 1.876679 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.21946 | 0.338739 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.876679 |   0.0 |   0.0 |   0.0 |   0.0 |   1.21946 |   0.0 | 0.338739 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 | 1.572383 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.190169 | 0.330603 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.572383 |   0.0 |   0.0 |   0.0 |   0.0 |  1.190169 |   0.0 | 0.330603 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 | 1.259666 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.16424 |   0.3234 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.259666 |   0.0 |   0.0 |   0.0 |   0.0 |   1.16424 |   0.0 |   0.3234 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179367 | 0.589683 | 0.589683 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.14346 | 0.317628 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.589683 |   0.0 |   0.0 |   0.0 |   0.0 |   1.14346 |   0.0 | 0.317628 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | 0.261846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.127849 | 0.313291 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.261846 |   0.0 |   0.0 |   0.0 |   0.0 |  1.127849 |   0.0 | 0.313291 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 |  0.06591 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.114997 | 0.309722 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06591 |   0.0 |   0.0 |   0.0 |   0.0 |  1.114997 |   0.0 | 0.309722 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | 0.017902 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.103486 | 0.306524 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017902 |   0.0 |   0.0 |   0.0 |   0.0 |  1.103486 |   0.0 | 0.306524 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | 0.017154 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.092332 | 0.303426 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.017154 |   0.0 |   0.0 |   0.0 |   0.0 |  1.092332 |   0.0 | 0.303426 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 |   0.0169 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.081294 | 0.300359 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   -0.0169 |   0.0 |   0.0 |   0.0 |   0.0 |  1.081294 |   0.0 | 0.300359 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 1.070366 | 0.297324 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.016973 |   0.0 |   0.0 |   0.0 |   0.0 |  1.070366 |   0.0 | 0.297324 |

.. _lland_v1_fluss:

streams
_______

The second alternative for water type |WASSER| is water type |FLUSS| for
representing streams.  Precipitation and evaporation values of |FLUSS|
areas are directly added and removed from the (not yet separated) input
of the linear storages for direct flow (|QDGZ|).  In contrast to water
type |SEE| (see integration test :ref:`lland_v1_see`), using water type
|FLUSS| increases the responsiveness of a catchment, but to a lesser extent
than using water type |WASSER| (see test :ref:`lland_v1_wasser`).

Using |FLUSS| instead of |WASSER| reduces the discussed problem during low
flow conditions.  However, for catchments with a very dense stream network,
it may persist.  Compare the series |EvPo| and |EvI| to see how method
|Calc_QAH_V1| adjusts the actual evaporation in the most extreme case of a
"stream network only" catchment, resulting even in some condensation
periods with negative |EvI| values:

.. integration-test::

    >>> lnk(FLUSS)
    >>> test("lland_v1_fluss",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor | nbes | sbes |      et0 |     evpo |       evi | evb | wgtf | wnied | schmpot | schm | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |       qah |       qa | inzp | wats | waes | bowa |     qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |     qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |  0.0 |  0.0 | 0.040283 | 0.020141 |  0.018255 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.020141 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020141 |       0.0 |   0.0 |   0.0 |  0.0 | -0.001886 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |  0.0 |  0.0 | 0.039121 |  0.01956 |  0.014419 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.01956 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01956 |       0.0 |   0.0 |   0.0 |  0.0 | -0.005141 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 |  0.011657 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019396 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 | -0.007739 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 |  0.009364 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019198 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019198 |       0.0 |   0.0 |   0.0 |  0.0 | -0.009834 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 |  0.007846 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.019396 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.019396 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01155 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |  0.0 |  0.0 | 0.041105 | 0.020552 |  0.007472 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.020552 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.020552 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.01308 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |  0.0 |  0.0 | 0.116763 | 0.058382 |  0.040404 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.058382 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.058382 |       0.0 |   0.0 |   0.0 |  0.0 | -0.017978 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |  0.0 |  0.0 |  0.77315 | 0.386575 |  0.330537 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.386575 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386575 |       0.0 |   0.0 |   0.0 |  0.0 | -0.056038 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |  0.0 |  0.0 | 1.747814 | 0.873907 |  0.712312 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.873907 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.873907 |       0.0 |   0.0 |   0.0 |  0.0 | -0.161595 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |  0.0 |  0.0 | 2.927022 | 1.463511 |  1.117577 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.463511 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.463511 |       0.0 |   0.0 |   0.0 |  0.0 | -0.345934 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |  0.0 |  0.0 | 3.305745 | 1.652872 |  1.086622 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.652872 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.652872 |       0.0 |   0.0 |   0.0 |  0.0 | -0.566251 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |  0.0 |  0.0 | 3.747947 | 1.873973 |  1.090045 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.873973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.873973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.783929 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |  0.0 |  0.0 | 2.050471 | 1.025236 |  0.123203 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.025236 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.025236 |       0.0 |   0.0 |   0.0 |  0.0 | -0.902033 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |  0.0 |  0.0 | 2.650012 | 1.325006 |  0.372566 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.325006 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.325006 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.95244 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |  0.0 |  0.0 | 2.959048 | 1.479524 |  0.445078 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.479524 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.479524 |       0.0 |   0.0 |   0.0 |  0.0 | -1.034446 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |  0.0 |  0.0 | 0.956604 | 0.478302 | -0.520578 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.238302 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.238302 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.99888 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |  0.0 |  0.0 | 0.731933 | 0.365967 |    -0.507 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.365967 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.365967 |       0.0 |   0.0 |   0.0 |  0.0 | -0.872967 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |  0.0 |  0.0 | 0.454628 | 0.227314 | -0.540764 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.227314 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.227314 |       0.0 |   0.0 |   0.0 |  0.0 | -0.768078 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |  0.0 |  0.0 | 0.300394 | 0.150197 | -0.352311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.409803 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  1.290681 |  0.119122 |   0.0 |   0.0 |  0.0 | -0.527888 |   0.02538 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  0.0 |  0.0 | 0.089558 | 0.044779 |  0.044779 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  6.675221 |  0.919074 | 0.255298 |  0.0 |  0.0 |  0.0 |  0.0 |  1.850192 |  4.825029 |   0.0 |   0.0 |  0.0 | -0.145837 |  1.064911 |   0.0 |   0.0 |  0.0 |   0.0 | 0.255298 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  0.0 |  0.0 |  0.03977 | 0.019885 |  0.019885 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.460115 |  2.091991 | 0.581109 |  0.0 |  0.0 |  0.0 |  0.0 |  1.710992 |  1.749123 |   0.0 |   0.0 |  0.0 |  0.202945 |  1.889046 |   0.0 |   0.0 |  0.0 |   0.0 | 0.581109 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  0.0 |  0.0 | 0.039381 | 0.019691 |  0.019691 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.860309 |  2.807552 | 0.779876 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829361 |  4.030949 |   0.0 |   0.0 |  0.0 |  0.487393 |  2.320159 |   0.0 |   0.0 |  0.0 |   0.0 | 0.779876 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 |  0.0 |  0.0 | 0.039251 | 0.019626 |  0.019626 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 12.700374 |  5.170333 | 1.436204 |  0.0 |  0.0 |  0.0 |  0.0 |  1.921262 | 10.779112 |   0.0 |   0.0 |  0.0 |  0.739258 |  4.431075 |   0.0 |   0.0 |  0.0 |   0.0 | 1.436204 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.100505 |  5.415219 | 1.504228 |  0.0 |  0.0 |  0.0 |  0.0 |  0.100505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.782998 |  4.632221 |   0.0 |   0.0 |  0.0 |   0.0 | 1.504228 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.820505 |  3.536298 | 0.982305 |  0.0 |  0.0 |  0.0 |  0.0 |  0.820505 |       0.0 |   0.0 |   0.0 |  0.0 |  0.726714 |  2.809584 |   0.0 |   0.0 |  0.0 |   0.0 | 0.982305 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  0.0 |  0.0 |  0.03899 | 0.019495 |  0.019495 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.580505 |  2.928372 | 0.813437 |  0.0 |  0.0 |  0.0 |  0.0 |   1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 |  0.828023 |  2.100349 |   0.0 |   0.0 |  0.0 |   0.0 | 0.813437 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  0.0 |  0.0 | 0.038793 | 0.019396 |  0.019396 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.500604 |  2.779855 | 0.772182 |  0.0 |  0.0 |  0.0 |  0.0 |  1.600097 |  0.900507 |   0.0 |   0.0 |  0.0 |  0.978544 |  1.801312 |   0.0 |   0.0 |  0.0 |   0.0 | 0.772182 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 |  0.0 |  0.0 | 0.038661 |  0.01933 |   0.01933 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  12.46067 |  4.622021 | 1.283895 |  0.0 |  0.0 |  0.0 |  0.0 |  1.919747 | 10.540922 |   0.0 |   0.0 |  0.0 |  1.121149 |  3.500872 |   0.0 |   0.0 |  0.0 |   0.0 | 1.283895 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |  0.0 |  0.0 | 0.038528 | 0.019264 |  0.019264 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.180736 |  5.791671 | 1.608797 |  0.0 |  0.0 |  0.0 |  0.0 |  1.760808 |  2.419928 |   0.0 |   0.0 |  0.0 |  1.251025 |  4.540646 |   0.0 |   0.0 |  0.0 |   0.0 | 1.608797 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  0.0 |  0.0 | 0.038396 | 0.019198 |  0.019198 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  4.060802 |  5.024932 | 1.395814 |  0.0 |  0.0 |  0.0 |  0.0 |  1.753743 |  2.307059 |   0.0 |   0.0 |  0.0 |  1.342771 |  3.682161 |   0.0 |   0.0 |  0.0 |   0.0 | 1.395814 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  0.0 |  0.0 | 0.074919 |  0.03746 |   0.03746 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   1.40254 |  4.047727 | 1.124369 |  0.0 |  0.0 |  0.0 |  0.0 |  1.287008 |  0.115532 |   0.0 |   0.0 |  0.0 |  1.373556 |  2.674171 |   0.0 |   0.0 |  0.0 |   0.0 | 1.124369 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |  0.0 |  0.0 | 0.505845 | 0.252922 |  0.252922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.132922 |  2.867695 | 0.796582 |  0.0 |  0.0 |  0.0 |  0.0 | -0.132922 |       0.0 |   0.0 |   0.0 |  0.0 |  1.224886 |  1.642809 |   0.0 |   0.0 |  0.0 |   0.0 | 0.796582 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |  0.0 |  0.0 | 1.218215 | 0.609108 |  0.609108 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.609108 |  1.930575 | 0.536271 |  0.0 |  0.0 |  0.0 |  0.0 | -0.609108 |       0.0 |   0.0 |   0.0 |  0.0 |   0.93416 |  0.996414 |   0.0 |   0.0 |  0.0 |   0.0 | 0.536271 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |  0.0 |  0.0 | 0.772303 | 0.386152 |  0.386152 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.386152 |   1.27965 | 0.355458 |  0.0 |  0.0 |  0.0 |  0.0 | -0.386152 |       0.0 |   0.0 |   0.0 |  0.0 |  0.675294 |  0.604356 |   0.0 |   0.0 |  0.0 |   0.0 | 0.355458 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |  0.0 |  0.0 | 0.984401 |   0.4922 |    0.4922 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0122 |  0.884469 | 0.245686 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0122 |       0.0 |   0.0 |   0.0 |  0.0 |  0.517909 |   0.36656 |   0.0 |   0.0 |  0.0 |   0.0 | 0.245686 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |  0.0 |  0.0 | 2.486378 | 1.243189 |  1.243189 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.123189 |  0.540098 | 0.150027 |  0.0 |  0.0 |  0.0 |  0.0 | -1.123189 |       0.0 |   0.0 |   0.0 |  0.0 |  0.317768 |   0.22233 |   0.0 |   0.0 |  0.0 |   0.0 | 0.150027 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |  0.0 |  0.0 | 1.349913 | 0.674957 |  0.674957 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.645043 |  0.867169 |  0.24088 |  0.0 |  0.0 |  0.0 |  0.0 |  1.725655 |  1.919388 |   0.0 |   0.0 |  0.0 |  0.323372 |  0.543797 |   0.0 |   0.0 |  0.0 |   0.0 |  0.24088 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  0.0 |  0.0 | 3.528622 | 1.764311 |  1.764311 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.315689 |  2.008267 | 0.557852 |  0.0 |  0.0 |  0.0 |  0.0 |  1.811878 |  3.503811 |   0.0 |   0.0 |  0.0 |  0.585638 |   1.42263 |   0.0 |   0.0 |  0.0 |   0.0 | 0.557852 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |  0.0 |  0.0 |  1.61841 | 0.809205 |  0.809205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.510795 |   2.18105 | 0.605847 |  0.0 |  0.0 |  0.0 |  0.0 |  0.510795 |       0.0 |   0.0 |   0.0 |  0.0 |  0.686066 |  1.494984 |   0.0 |   0.0 |  0.0 |   0.0 | 0.605847 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |  0.0 |  0.0 | 0.844303 | 0.422151 |  0.422151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 24.417849 |  6.481785 | 1.800496 |  0.0 |  0.0 |  0.0 |  0.0 |  1.959046 | 22.458802 |   0.0 |   0.0 |  0.0 |  0.789929 |  5.691856 |   0.0 |   0.0 |  0.0 |   0.0 | 1.800496 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 |  0.0 |  0.0 | 0.895703 | 0.447851 |  0.447851 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 45.032149 |  17.68086 |  4.91135 |  0.0 |  0.0 |  0.0 |  0.0 |  1.977794 | 43.054355 |   0.0 |   0.0 |  0.0 |   1.00361 | 16.677251 |   0.0 |   0.0 |  0.0 |   0.0 |  4.91135 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  0.0 |  0.0 | 1.151139 |  0.57557 |   0.57557 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   9.26443 | 20.625552 |  5.72932 |  0.0 |  0.0 |  0.0 |  0.0 |   1.89206 |   7.37237 |   0.0 |   0.0 |  0.0 |   1.17217 | 19.453382 |   0.0 |   0.0 |  0.0 |   0.0 |  5.72932 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |  0.0 |  0.0 | 0.636581 |  0.31829 |   0.31829 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   4.00171 | 14.898206 | 4.138391 |  0.0 |  0.0 |  0.0 |  0.0 |  1.750107 |  2.251603 |   0.0 |   0.0 |  0.0 |  1.289369 | 13.608837 |   0.0 |   0.0 |  0.0 |   0.0 | 4.138391 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |  0.0 |  0.0 | 0.116642 | 0.058321 |  0.058321 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.941679 | 11.549032 | 3.208064 |  0.0 |  0.0 |  0.0 |  0.0 |  1.888164 |  7.053515 |   0.0 |   0.0 |  0.0 |  1.385817 | 10.163215 |   0.0 |   0.0 |  0.0 |   0.0 | 3.208064 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 |  0.0 |  0.0 | 0.037049 | 0.018524 |  0.018524 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 22.181476 | 13.229438 | 3.674844 |  0.0 |  0.0 |  0.0 |  0.0 |  1.954917 | 20.226558 |   0.0 |   0.0 |  0.0 |  1.483128 | 11.746309 |   0.0 |   0.0 |  0.0 |   0.0 | 3.674844 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 |  0.0 |  0.0 |  0.03698 |  0.01849 |   0.01849 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  18.46151 |  15.86018 | 4.405606 |  0.0 |  0.0 |  0.0 |  0.0 |  1.945833 | 16.515677 |   0.0 |   0.0 |  0.0 |  1.567798 | 14.292382 |   0.0 |   0.0 |  0.0 |   0.0 | 4.405606 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |  0.0 |  0.0 | 0.036499 |  0.01825 |   0.01825 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   7.54175 | 14.486291 |  4.02397 |  0.0 |  0.0 |  0.0 |  0.0 |  1.867405 |  5.674346 |   0.0 |   0.0 |  0.0 |  1.628979 | 12.857312 |   0.0 |   0.0 |  0.0 |   0.0 |  4.02397 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 |  0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.262063 | 10.615291 | 2.948692 |  0.0 |  0.0 |  0.0 |  0.0 |  1.557926 |  0.704138 |   0.0 |   0.0 |  0.0 |  1.643215 |  8.972076 |   0.0 |   0.0 |  0.0 |   0.0 | 2.948692 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  0.0 |  0.0 | 0.035943 | 0.017972 |  0.017972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  5.862028 |  8.081246 | 2.244791 |  0.0 |  0.0 |  0.0 |  0.0 |  1.829411 |  4.032618 |   0.0 |   0.0 |  0.0 |   1.65318 |  6.428066 |   0.0 |   0.0 |  0.0 |   0.0 | 2.244791 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  0.0 |  0.0 | 0.035873 | 0.017937 |  0.017937 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  3.222063 |  6.624871 | 1.840242 |  0.0 |  0.0 |  0.0 |  0.0 |   1.68964 |  1.532423 |   0.0 |   0.0 |  0.0 |  1.672035 |  4.952836 |   0.0 |   0.0 |  0.0 |   0.0 | 1.840242 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.0 |  0.0 | 0.035522 | 0.017761 |  0.017761 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582239 |  4.852022 | 1.347784 |  0.0 |  0.0 |  0.0 |  0.0 |  0.582239 |       0.0 |   0.0 |   0.0 |  0.0 |  1.571514 |  3.280508 |   0.0 |   0.0 |  0.0 |   0.0 | 1.347784 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.0 |  0.0 | 0.035663 | 0.017831 |  0.017831 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222169 |  3.348196 | 0.930054 |  0.0 |  0.0 |  0.0 |  0.0 |  0.222169 |       0.0 |   0.0 |   0.0 |  0.0 |  1.358467 |  1.989729 |   0.0 |   0.0 |  0.0 |   0.0 | 0.930054 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.0 |  0.0 | 0.036083 | 0.018041 |  0.018041 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.581959 |  2.393018 | 0.664727 |  0.0 |  0.0 |  0.0 |  0.0 |  0.581959 |       0.0 |   0.0 |   0.0 |  0.0 |  1.186187 |  1.206832 |   0.0 |   0.0 |  0.0 |   0.0 | 0.664727 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  0.0 |  0.0 | 0.036637 | 0.018319 |  0.018319 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.861681 |  2.166761 | 0.601878 |  0.0 |  0.0 |  0.0 |  0.0 |  1.650555 |  1.211126 |   0.0 |   0.0 |  0.0 |  1.176737 |  0.990025 |   0.0 |   0.0 |  0.0 |   0.0 | 0.601878 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.0 |  0.0 | 0.061945 | 0.030972 |  0.030972 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.449028 |  1.969075 | 0.546965 |  0.0 |  0.0 |  0.0 |  0.0 |  0.449028 |       0.0 |   0.0 |   0.0 |  0.0 |  1.150098 |  0.818977 |   0.0 |   0.0 |  0.0 |   0.0 | 0.546965 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.0 |  0.0 | 0.188148 | 0.094074 |  0.094074 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.145926 |  1.491364 | 0.414268 |  0.0 |  0.0 |  0.0 |  0.0 |  0.145926 |       0.0 |   0.0 |   0.0 |  0.0 |  0.994629 |  0.496735 |   0.0 |   0.0 |  0.0 |   0.0 | 0.414268 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 |  0.46949 | 0.234745 |  0.234745 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.234745 |  1.106419 | 0.307339 |  0.0 |  0.0 |  0.0 |  0.0 | -0.234745 |       0.0 |   0.0 |   0.0 |  0.0 |  0.805134 |  0.301285 |   0.0 |   0.0 |  0.0 |   0.0 | 0.307339 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |  0.0 |  0.0 | 1.680918 | 0.840459 |  0.840459 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.840459 |  0.742647 | 0.206291 |  0.0 |  0.0 |  0.0 |  0.0 | -0.840459 |       0.0 |   0.0 |   0.0 |  0.0 |  0.559908 |  0.182738 |   0.0 |   0.0 |  0.0 |   0.0 | 0.206291 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |  0.0 |  0.0 | 1.743886 | 0.871943 |  0.871943 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.511943 |  0.447668 | 0.124352 |  0.0 |  0.0 |  0.0 |  0.0 | -0.511943 |       0.0 |   0.0 |   0.0 |  0.0 |  0.336832 |  0.110837 |   0.0 |   0.0 |  0.0 |   0.0 | 0.124352 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |  0.0 |  0.0 | 2.122301 | 1.061151 |  1.061151 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  2.058849 |  0.555989 | 0.154441 |  0.0 |  0.0 |  0.0 |  0.0 |  1.514292 |  0.544558 |   0.0 |   0.0 |  0.0 |  0.372739 |   0.18325 |   0.0 |   0.0 |  0.0 |   0.0 | 0.154441 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |  0.0 |  0.0 | 2.150411 | 1.075205 |  1.075205 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.235205 |   0.62521 | 0.173669 |  0.0 |  0.0 |  0.0 |  0.0 | -0.235205 |       0.0 |   0.0 |   0.0 |  0.0 |  0.415821 |  0.209389 |   0.0 |   0.0 |  0.0 |   0.0 | 0.173669 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |  0.0 |  0.0 | 1.863566 | 0.931783 |  0.931783 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.571783 |  0.393289 | 0.109247 |  0.0 |  0.0 |  0.0 |  0.0 | -0.571783 |       0.0 |   0.0 |   0.0 |  0.0 |  0.266288 |  0.127001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.109247 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |  0.0 |  0.0 | 3.115838 | 1.557919 |  1.557919 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.197919 |  0.132761 | 0.036878 |  0.0 |  0.0 |  0.0 |  0.0 | -1.197919 |       0.0 |   0.0 |   0.0 |  0.0 |  0.055731 |   0.07703 |   0.0 |   0.0 |  0.0 |   0.0 | 0.036878 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |  0.0 |  0.0 | 1.940627 | 0.970313 |  0.866834 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.970313 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.970313 |       0.0 |   0.0 |   0.0 |  0.0 | -0.150201 |  0.046721 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |  0.0 |  0.0 | 2.122768 | 1.061384 |  0.782331 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.061384 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.061384 |       0.0 |   0.0 |   0.0 |  0.0 | -0.307391 |  0.028338 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |  0.0 |  0.0 | 1.314414 | 0.657207 |  0.268181 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.657207 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.657207 |       0.0 |   0.0 |   0.0 |  0.0 | -0.406214 |  0.017188 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |  0.0 |  0.0 | 0.602486 | 0.301243 | -0.106706 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.301243 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.301243 |       0.0 |   0.0 |   0.0 |  0.0 | -0.418374 |  0.010425 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 | 0.109905 | 0.054952 |   -0.3128 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.054952 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.054952 |       0.0 |   0.0 |   0.0 |  0.0 | -0.374076 |  0.006323 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |  0.0 |  0.0 | 0.035026 | 0.017513 | -0.291374 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017513 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017513 |       0.0 |   0.0 |   0.0 |  0.0 | -0.312722 |  0.003835 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | -0.239696 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017154 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 | -0.259176 |  0.002326 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |  0.0 |  0.0 | 0.033727 | 0.016863 | -0.197004 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016863 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016863 |       0.0 |   0.0 |   0.0 |  0.0 | -0.215278 |  0.001411 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |  0.0 |  0.0 | 0.033286 | 0.016643 | -0.161792 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016643 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016643 |       0.0 |   0.0 |   0.0 |  0.0 | -0.179291 |  0.000856 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |  0.0 |  0.0 | 0.032916 | 0.016458 | -0.132813 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016458 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016458 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.14979 |  0.000519 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 | -0.108448 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0169 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.125663 |  0.000315 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |  -0.08879 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.105954 |  0.000191 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |  0.0 |  0.0 | 0.033433 | 0.016717 |  0.016717 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.543283 |  0.079252 | 0.022014 |  0.0 |  0.0 |  0.0 |  0.0 |  1.352031 |  0.191252 |   0.0 |   0.0 |  0.0 |  0.038388 |  0.040864 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022014 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |  0.0 |  0.0 |  0.03336 |  0.01668 |   0.01668 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.01668 |  0.207615 | 0.057671 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.01668 |       0.0 |   0.0 |   0.0 |  0.0 |  0.148326 |  0.059289 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057671 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 |  0.016973 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |  0.154349 | 0.042875 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 |  0.118388 |  0.035961 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042875 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |  0.0 |  0.0 | 0.124091 | 0.062046 |  0.062046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.062046 |  0.111441 | 0.030956 |  0.0 |  0.0 |  0.0 |  0.0 | -0.062046 |       0.0 |   0.0 |   0.0 |  0.0 |   0.08963 |  0.021811 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030956 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.0 |  0.0 | 0.556783 | 0.278392 |  0.278392 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.561608 |  0.133773 | 0.037159 |  0.0 |  0.0 |  0.0 |  0.0 |  0.561608 |       0.0 |   0.0 |   0.0 |  0.0 |  0.120543 |  0.013229 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037159 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.0 |  0.0 | 1.278351 | 0.639175 |  0.639175 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.159175 |  0.141015 | 0.039171 |  0.0 |  0.0 |  0.0 |  0.0 | -0.159175 |       0.0 |   0.0 |   0.0 |  0.0 |  0.132991 |  0.008024 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039171 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |  0.0 |  0.0 |  2.07666 |  1.03833 |   1.03833 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.91833 |  0.013799 | 0.003833 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.91833 |       0.0 |   0.0 |   0.0 |  0.0 |  0.008932 |  0.004867 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003833 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |  0.0 |  0.0 | 2.862014 | 1.431007 |  1.271747 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.951007 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.951007 |       0.0 |   0.0 |   0.0 |  0.0 | -0.162212 |  0.002952 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |  0.0 |  0.0 | 3.356573 | 1.678286 |  1.306768 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.678286 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678286 |       0.0 |   0.0 |   0.0 |  0.0 | -0.373309 |   0.00179 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |  0.0 |  0.0 | 3.356514 | 1.678257 |  1.069485 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.678257 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.678257 |       0.0 |   0.0 |   0.0 |  0.0 | -0.609858 |  0.001086 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |  0.0 |  0.0 | 4.286095 | 2.143047 |  1.296651 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.143047 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -2.143047 |       0.0 |   0.0 |   0.0 |  0.0 | -0.847056 |  0.000659 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |  0.0 |  0.0 | 3.753358 | 1.876679 |  0.820046 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.876679 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.876679 |       0.0 |   0.0 |   0.0 |  0.0 | -1.057033 |  0.000399 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |  0.0 |  0.0 | 3.144766 | 1.572383 |  0.395514 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.572383 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.572383 |       0.0 |   0.0 |   0.0 |  0.0 | -1.177111 |  0.000242 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |  0.0 |  0.0 | 2.519332 | 1.259666 |  0.040338 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.259666 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -1.259666 |       0.0 |   0.0 |   0.0 |  0.0 | -1.219474 |  0.000147 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |  0.0 |  0.0 | 1.179367 | 0.589683 | -0.574241 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.589683 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.589683 |       0.0 |   0.0 |   0.0 |  0.0 | -1.164013 |  0.000089 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |  0.0 |  0.0 | 0.523693 | 0.261846 | -0.767302 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.261846 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.261846 |       0.0 |   0.0 |   0.0 |  0.0 | -1.029202 |  0.000054 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |  0.0 |  0.0 |  0.13182 |  0.06591 | -0.805811 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.06591 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -0.06591 |       0.0 |   0.0 |   0.0 |  0.0 | -0.871754 |  0.000033 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |  0.0 |  0.0 | 0.035803 | 0.017902 | -0.703261 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017902 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017902 |       0.0 |   0.0 |   0.0 |  0.0 | -0.721183 |   0.00002 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |  0.0 |  0.0 | 0.034308 | 0.017154 | -0.576463 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.017154 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.017154 |       0.0 |   0.0 |   0.0 |  0.0 |  -0.59363 |  0.000012 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |  0.0 |  0.0 |   0.0338 |   0.0169 | -0.472201 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   -0.0169 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -0.0169 |       0.0 |   0.0 |   0.0 |  0.0 | -0.489109 |  0.000007 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |  0.0 |  0.0 | 0.033946 | 0.016973 | -0.386541 | 0.0 |  0.0 |   0.0 |     0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.016973 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -0.016973 |       0.0 |   0.0 |   0.0 |  0.0 | -0.403519 |  0.000004 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

.. _lland_v1_vers:

sealed surfaces
_______________

For sealed surfaces, we assume retention processes below the surface to
be negligible.  All water reaching the sealed surface becomes direct
discharge immediately:

.. integration-test::

    >>> lnk(VERS)
    >>> test("lland_v1_vers",
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi | evb |   wgtf |    wnied |   schmpot | schm |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |       qah |       qa |     inzp | wats | waes | bowa |    qdgz1 |     qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |     qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.0 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.0 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.0 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.0 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.0 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.0 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.0 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.0 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.0 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.0 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 | 0.0 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.0 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 | 0.0 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.0 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.0 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 | 0.0 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 | 0.0 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.0 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 | 0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |  0.0 |      0.76 |  0.071177 | 0.019771 | 0.649803 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071177 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019771 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 | 0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 |  6.569803 |  0.0 |  0.0 | 0.0 |  0.0 |  6.569803 |  1.303993 |  0.36222 | 0.755221 |  0.0 |  0.0 |  0.0 | 1.847788 |  4.722014 |   0.0 |   0.0 |  0.0 | 0.297915 |  1.006079 |   0.0 |   0.0 |  0.0 |   0.0 |  0.36222 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 | 0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 |  3.435221 |  0.0 |  0.0 | 0.0 |  0.0 |  3.435221 |  2.395771 | 0.665492 | 0.780115 |  0.0 |  0.0 |  0.0 | 1.708898 |  1.726323 |   0.0 |   0.0 |  0.0 | 0.565852 |  1.829919 |   0.0 |   0.0 |  0.0 |   0.0 | 0.665492 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 | 0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 |  5.860115 |  0.0 |  0.0 | 0.0 |  0.0 |  5.860115 |  3.064476 | 0.851243 | 0.780309 |  0.0 |  0.0 |  0.0 | 1.829355 |   4.03076 |   0.0 |   0.0 |  0.0 | 0.784332 |  2.280144 |   0.0 |   0.0 |  0.0 |   0.0 | 0.851243 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 | 0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 12.700309 |  0.0 |  0.0 | 0.0 |  0.0 | 12.700309 |  5.389127 |  1.49698 | 0.780374 |  0.0 |  0.0 |  0.0 | 1.921262 | 10.779048 |   0.0 |   0.0 |  0.0 |  0.98237 |  4.406757 |   0.0 |   0.0 |  0.0 |   0.0 |  1.49698 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 |  0.100374 |  0.0 |  0.0 | 0.0 |  0.0 |  0.100374 |  5.599489 | 1.555414 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.100374 |       0.0 |   0.0 |   0.0 |  0.0 | 0.982029 |   4.61746 |   0.0 |   0.0 |  0.0 |   0.0 | 1.555414 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 |  0.820505 |  0.0 |  0.0 | 0.0 |  0.0 |  0.820505 |  3.690286 | 1.025079 | 0.780505 |  0.0 |  0.0 |  0.0 | 0.820505 |       0.0 |   0.0 |   0.0 |  0.0 | 0.889655 |  2.800631 |   0.0 |   0.0 |  0.0 |   0.0 | 1.025079 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 | 0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  3.580505 |  0.0 |  0.0 | 0.0 |  0.0 |  3.580505 |  3.056347 | 0.848985 | 0.780505 |  0.0 |  0.0 |  0.0 |  1.72071 |  1.859795 |   0.0 |   0.0 |  0.0 | 0.961428 |  2.094919 |   0.0 |   0.0 |  0.0 |   0.0 | 0.848985 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 | 0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  2.500505 |  0.0 |  0.0 | 0.0 |  0.0 |  2.500505 |  2.885765 | 0.801601 | 0.780604 |  0.0 |  0.0 |  0.0 | 1.600081 |  0.900424 |   0.0 |   0.0 |  0.0 | 1.087765 |     1.798 |   0.0 |   0.0 |  0.0 |   0.0 | 0.801601 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 | 0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 | 12.460604 |  0.0 |  0.0 | 0.0 |  0.0 | 12.460604 |  4.709405 | 1.308168 |  0.78067 |  0.0 |  0.0 |  0.0 | 1.919747 | 10.540857 |   0.0 |   0.0 |  0.0 |  1.21057 |  3.498835 |   0.0 |   0.0 |  0.0 |   0.0 | 1.308168 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 | 0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 |   4.18067 |  0.0 |  0.0 | 0.0 |  0.0 |   4.18067 |  5.863622 | 1.628784 | 0.780736 |  0.0 |  0.0 |  0.0 | 1.760804 |  2.419866 |   0.0 |   0.0 |  0.0 | 1.324236 |  4.539385 |   0.0 |   0.0 |  0.0 |   0.0 | 1.628784 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 | 0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 |  4.060736 |  0.0 |  0.0 | 0.0 |  0.0 |  4.060736 |  5.084082 | 1.412245 | 0.780802 |  0.0 |  0.0 |  0.0 | 1.753739 |  2.306997 |   0.0 |   0.0 |  0.0 | 1.402711 |  3.681371 |   0.0 |   0.0 |  0.0 |   0.0 | 1.412245 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 | 0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 |  1.420802 |  0.0 |  0.0 | 0.0 |  0.0 |  1.420802 |  4.099108 | 1.138641 |  0.76254 |  0.0 |  0.0 |  0.0 | 1.296172 |   0.12463 |   0.0 |   0.0 |  0.0 | 1.423489 |  2.675619 |   0.0 |   0.0 |  0.0 |   0.0 | 1.138641 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 | 0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 |   0.08254 |  0.0 |  0.0 | 0.0 |  0.0 |   0.08254 |  2.932078 | 0.814466 | 0.547078 |  0.0 |  0.0 |  0.0 |  0.08254 |       0.0 |   0.0 |   0.0 |  0.0 | 1.286749 |  1.645329 |   0.0 |   0.0 |  0.0 |   0.0 | 0.814466 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.0 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  2.058675 | 0.571854 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 1.060733 |  0.997943 |   0.0 |   0.0 |  0.0 |   0.0 | 0.571854 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.0 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.473737 | 0.409372 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.868455 |  0.605283 |   0.0 |   0.0 |  0.0 |   0.0 | 0.409372 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.0 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.078153 | 0.299487 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.71103 |  0.367123 |   0.0 |   0.0 |  0.0 |   0.0 | 0.299487 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.0 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.804814 | 0.223559 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.582143 |  0.222671 |   0.0 |   0.0 |  0.0 |   0.0 | 0.223559 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 | 0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 |      3.52 |  0.0 |  0.0 | 0.0 |  0.0 |      3.52 |  1.156758 | 0.321322 | 0.125043 |  0.0 |  0.0 |  0.0 | 1.715909 |  1.804091 |   0.0 |   0.0 |  0.0 | 0.637319 |  0.519439 |   0.0 |   0.0 |  0.0 |   0.0 | 0.321322 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.0 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 |  6.405043 |  0.0 |  0.0 | 0.0 |  0.0 |  6.405043 |  2.457156 | 0.682543 |      0.0 |  0.0 |  0.0 |  0.0 | 1.843873 |   4.56117 |   0.0 |   0.0 |  0.0 | 0.844819 |  1.612337 |   0.0 |   0.0 |  0.0 |   0.0 | 0.682543 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.0 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 |      0.52 |  0.0 |  0.0 | 0.0 |  0.0 |      0.52 |  2.702734 |  0.75076 |      0.0 |  0.0 |  0.0 |  0.0 |     0.52 |       0.0 |   0.0 |   0.0 |  0.0 | 0.901931 |  1.800804 |   0.0 |   0.0 |  0.0 |   0.0 |  0.75076 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 | 0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 |     24.04 |  0.0 |  0.0 | 0.0 |  0.0 |     24.04 |  6.764387 | 1.878996 | 0.377849 |  0.0 |  0.0 |  0.0 | 1.958403 | 22.081597 |   0.0 |   0.0 |  0.0 |  0.96741 |  5.796977 |   0.0 |   0.0 |  0.0 |   0.0 | 1.878996 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 | 0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 45.057849 |  0.0 |  0.0 | 0.0 |  0.0 | 45.057849 | 17.827296 | 4.952027 | 0.352149 |  0.0 |  0.0 |  0.0 | 1.977806 | 43.080042 |   0.0 |   0.0 |  0.0 | 1.148864 | 16.678432 |   0.0 |   0.0 |  0.0 |   0.0 | 4.952027 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 | 0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 |  9.392149 |  0.0 |  0.0 | 0.0 |  0.0 |  9.392149 | 20.776864 | 5.771351 |  0.22443 |  0.0 |  0.0 |  0.0 | 1.893528 |   7.49862 |   0.0 |   0.0 |  0.0 | 1.291233 | 19.485632 |   0.0 |   0.0 |  0.0 |   0.0 | 5.771351 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 | 0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 |   3.74443 |  0.0 |  0.0 | 0.0 |  0.0 |   3.74443 | 14.985386 | 4.162607 |  0.48171 |  0.0 |  0.0 |  0.0 | 1.732937 |  2.011494 |   0.0 |   0.0 |  0.0 |  1.38537 | 13.600016 |   0.0 |   0.0 |  0.0 |   0.0 | 4.162607 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 | 0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 |   8.68171 |  0.0 |  0.0 | 0.0 |  0.0 |   8.68171 | 11.522469 | 3.200686 | 0.741679 |  0.0 |  0.0 |  0.0 | 1.884815 |  6.796894 |   0.0 |   0.0 |  0.0 | 1.462598 | 10.059871 |   0.0 |   0.0 |  0.0 |   0.0 | 3.200686 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 | 0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 22.141679 |  0.0 |  0.0 | 0.0 |  0.0 | 22.141679 |  13.17456 |   3.6596 | 0.781476 |  0.0 |  0.0 |  0.0 | 1.954836 | 20.186843 |   0.0 |   0.0 |  0.0 |  1.54569 |  11.62887 |   0.0 |   0.0 |  0.0 |   0.0 |   3.6596 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 | 0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 | 18.461476 |  0.0 |  0.0 | 0.0 |  0.0 | 18.461476 | 15.832992 | 4.398053 |  0.78151 |  0.0 |  0.0 |  0.0 | 1.945833 | 16.515642 |   0.0 |   0.0 |  0.0 | 1.619013 | 14.213979 |   0.0 |   0.0 |  0.0 |   0.0 | 4.398053 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 | 0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |   7.54151 |  0.0 |  0.0 | 0.0 |  0.0 |   7.54151 | 14.480611 | 4.022392 |  0.78175 |  0.0 |  0.0 |  0.0 | 1.867401 |  5.674109 |   0.0 |   0.0 |  0.0 |  1.67091 | 12.809702 |   0.0 |   0.0 |  0.0 |   0.0 | 4.022392 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 |   2.26175 |  0.0 |  0.0 | 0.0 |  0.0 |   2.26175 | 10.620641 | 2.950178 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.557865 |  0.703886 |   0.0 |   0.0 |  0.0 | 1.677538 |  8.943102 |   0.0 |   0.0 |  0.0 |   0.0 | 2.950178 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 | 0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 |  5.862063 |  0.0 |  0.0 | 0.0 |  0.0 |  5.862063 |  8.091731 | 2.247703 | 0.782028 |  0.0 |  0.0 |  0.0 | 1.829412 |  4.032652 |   0.0 |   0.0 |  0.0 | 1.681277 |  6.410455 |   0.0 |   0.0 |  0.0 |   0.0 | 2.247703 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 | 0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 |  3.222028 |  0.0 |  0.0 | 0.0 |  0.0 |  3.222028 |  6.637192 | 1.843664 | 0.782063 |  0.0 |  0.0 |  0.0 | 1.689637 |  1.532392 |   0.0 |   0.0 |  0.0 | 1.695038 |  4.942153 |   0.0 |   0.0 |  0.0 |   0.0 | 1.843664 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 | 0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 |  0.582063 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582063 |  4.864354 | 1.351209 | 0.782239 |  0.0 |  0.0 |  0.0 | 0.582063 |       0.0 |   0.0 |   0.0 |  0.0 | 1.590331 |  3.274023 |   0.0 |   0.0 |  0.0 |   0.0 | 1.351209 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 | 0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 |  0.222239 |  0.0 |  0.0 | 0.0 |  0.0 |  0.222239 |   3.35966 | 0.933239 | 0.782169 |  0.0 |  0.0 |  0.0 | 0.222239 |       0.0 |   0.0 |   0.0 |  0.0 | 1.373864 |  1.985796 |   0.0 |   0.0 |  0.0 |   0.0 | 0.933239 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 | 0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 |  0.582169 |  0.0 |  0.0 | 0.0 |  0.0 |  0.582169 |  2.403264 | 0.667573 | 0.781959 |  0.0 |  0.0 |  0.0 | 0.582169 |       0.0 |   0.0 |   0.0 |  0.0 | 1.198819 |  1.204446 |   0.0 |   0.0 |  0.0 |   0.0 | 0.667573 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 | 0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 |  2.861959 |  0.0 |  0.0 | 0.0 |  0.0 |  2.861959 |   2.17573 | 0.604369 | 0.781681 |  0.0 |  0.0 |  0.0 | 1.650589 |   1.21137 |   0.0 |   0.0 |  0.0 |   1.1871 |  0.988629 |   0.0 |   0.0 |  0.0 |   0.0 | 0.604369 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 | 0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 |  0.461681 |  0.0 |  0.0 | 0.0 |  0.0 |  0.461681 |  1.977946 | 0.549429 | 0.769028 |  0.0 |  0.0 |  0.0 | 0.461681 |       0.0 |   0.0 |   0.0 |  0.0 | 1.159771 |  0.818175 |   0.0 |   0.0 |  0.0 |   0.0 | 0.549429 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 | 0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 |  0.209028 |  0.0 |  0.0 | 0.0 |  0.0 |  0.209028 |  1.505815 | 0.418282 | 0.705926 |  0.0 |  0.0 |  0.0 | 0.209028 |       0.0 |   0.0 |   0.0 |  0.0 | 1.009567 |  0.496248 |   0.0 |   0.0 |  0.0 |   0.0 | 0.418282 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 | 0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  1.145867 | 0.318296 | 0.471181 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.844877 |   0.30099 |   0.0 |   0.0 |  0.0 |   0.0 | 0.318296 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.0 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.874287 | 0.242857 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.691727 |  0.182559 |   0.0 |   0.0 |  0.0 |   0.0 | 0.242857 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.0 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.677066 | 0.188074 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.566338 |  0.110728 |   0.0 |   0.0 |  0.0 |   0.0 | 0.188074 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.0 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 |      2.32 |  0.0 |  0.0 | 0.0 |  0.0 |      2.32 |  0.837794 | 0.232721 |      0.0 |  0.0 |  0.0 |  0.0 | 1.568966 |  0.751034 |   0.0 |   0.0 |  0.0 | 0.610618 |  0.227176 |   0.0 |   0.0 |  0.0 |   0.0 | 0.232721 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.0 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 |      0.04 |  0.0 |  0.0 | 0.0 |  0.0 |      0.04 |  0.914426 | 0.254007 |      0.0 |  0.0 |  0.0 |  0.0 |     0.04 |       0.0 |   0.0 |   0.0 |  0.0 | 0.641144 |  0.273282 |   0.0 |   0.0 |  0.0 |   0.0 | 0.254007 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.0 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.694183 | 0.192828 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.528429 |  0.165754 |   0.0 |   0.0 |  0.0 |   0.0 | 0.192828 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 0.0 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.533176 | 0.148104 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.432641 |  0.100535 |   0.0 |   0.0 |  0.0 |   0.0 | 0.148104 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 | 0.0 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.415194 | 0.115332 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.354216 |  0.060977 |   0.0 |   0.0 |  0.0 |   0.0 | 0.115332 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 0.0 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.326992 | 0.090831 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.290008 |  0.036985 |   0.0 |   0.0 |  0.0 |   0.0 | 0.090831 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 | 0.0 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.259871 | 0.072186 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.237438 |  0.022432 |   0.0 |   0.0 |  0.0 |   0.0 | 0.072186 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.0 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.208004 | 0.057779 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.194398 |  0.013606 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057779 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.0 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.167412 | 0.046503 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.15916 |  0.008252 |   0.0 |   0.0 |  0.0 |   0.0 | 0.046503 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 | 0.0 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.135314 | 0.037587 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.130309 |  0.005005 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037587 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.109724 | 0.030479 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.106688 |  0.003036 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030479 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.0 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |   0.08919 | 0.024775 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.087349 |  0.001841 |   0.0 |   0.0 |  0.0 |   0.0 | 0.024775 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.0 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.072632 | 0.020176 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.071515 |  0.001117 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020176 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.0 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.059229 | 0.016452 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.058552 |  0.000677 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016452 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.048349 |  0.01343 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.047938 |  0.000411 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01343 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.039497 | 0.010972 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.039248 |  0.000249 |   0.0 |   0.0 |  0.0 |   0.0 | 0.010972 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 | 0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 |      0.76 |  0.0 |  0.0 | 0.0 |  0.0 |      0.76 |  0.103462 | 0.028739 | 0.783283 |  0.0 |  0.0 |  0.0 |     0.76 |       0.0 |   0.0 |   0.0 |  0.0 | 0.103311 |  0.000151 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028739 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 | 0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.151263 | 0.042018 | 0.766603 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151171 |  0.000092 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042018 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.123824 | 0.034396 |  0.74963 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.123769 |  0.000056 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034396 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 | 0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.101367 | 0.028157 | 0.687585 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.101333 |  0.000034 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028157 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 | 0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 |  0.727585 |  0.0 |  0.0 | 0.0 |  0.0 |  0.727585 |  0.151126 | 0.041979 | 0.521608 |  0.0 |  0.0 |  0.0 | 0.727585 |       0.0 |   0.0 |   0.0 |  0.0 | 0.151106 |   0.00002 |   0.0 |   0.0 |  0.0 |   0.0 | 0.041979 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 | 0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 |  0.201608 |  0.0 |  0.0 | 0.0 |  0.0 |  0.201608 |  0.206356 | 0.057321 | 0.160825 |  0.0 |  0.0 |  0.0 | 0.201608 |       0.0 |   0.0 |   0.0 |  0.0 | 0.206344 |  0.000012 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057321 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.0 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.186612 | 0.051837 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.186604 |  0.000008 |   0.0 |   0.0 |  0.0 |   0.0 | 0.051837 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.0 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.152783 |  0.04244 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.152779 |  0.000005 |   0.0 |   0.0 |  0.0 |   0.0 |  0.04244 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.125087 | 0.034746 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.125084 |  0.000003 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034746 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 0.0 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.102412 | 0.028448 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.102411 |  0.000002 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028448 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 0.0 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.083848 | 0.023291 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.083847 |  0.000001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023291 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 0.0 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.068648 | 0.019069 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.068648 |  0.000001 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019069 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 0.0 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.056204 | 0.015612 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.056204 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015612 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 0.0 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.046016 | 0.012782 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.046016 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012782 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.0 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.037675 | 0.010465 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.037675 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.010465 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.0 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.030846 | 0.008568 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.030845 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.008568 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.0 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.025254 | 0.007015 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.025254 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.007015 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.0 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.020676 | 0.005743 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.020676 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.005743 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.0 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.016928 | 0.004702 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.016928 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004702 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.0 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |   0.01386 |  0.00385 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 |  0.01386 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00385 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.011347 | 0.003152 |      0.0 |  0.0 |  0.0 |  0.0 |      0.0 |       0.0 |   0.0 |   0.0 |  0.0 | 0.011347 |       0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003152 |

.. _lland_v1_acker_gw_vol:

acre (limited groundwater volume)
_________________________________

Usually, HydPy-L represents groundwater processes by a simple linear runoff
concentration storage, capturing the aspect of the relative slowness of
groundwater but leaving other relevant characteristics aside.  The control
parameter |VolBMax| offers the chance to explicitly account for another
groundwater characteristic, namely the aquifer's limited size.

In this example, we set the maximum capacity of the groundwater storage to
25 mm, which corresponds to a maximum outflow of 0.25 mm/h (see the
documentation on the derived parameter |QBGAMax|).  As soon as groundwater
discharge (|QBGA|) reaches this threshold, it converts all recharge excess
(|QBGZ|) into direct runoff (|QDGZ|, see the documentation on method
|Update_QDGZ_QBGZ_QBGA_V1| for further information).  This additional
generation of direct runoff leads to a small increase in total runoff
(|QAH|) at the end of the simulation period:

.. integration-test::

    >>> lnk(ACKER)
    >>> volbmax(25.0)
    >>> test("lland_v1_acker_gw_vol",
    ...      axis1=(states.qbgz, states.qbga, fluxes.qdgz, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000495 | 0.000138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.0 | 0.000138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 0.001462 | 0.000406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.0 | 0.000406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 0.002394 | 0.000665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.0 | 0.000665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 0.003291 | 0.000914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.0 | 0.000914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 0.004156 | 0.001154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.0 | 0.001154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  0.00499 | 0.001386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.0 | 0.001386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 0.005793 | 0.001609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.0 | 0.001609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 0.006566 | 0.001824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.0 | 0.001824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 0.007306 |  0.00203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.0 |  0.00203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 0.008003 | 0.002223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.0 | 0.002223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 0.008643 | 0.002401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.0 | 0.002401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 0.009219 | 0.002561 |      0.0 |  0.0 |  0.0 |   17.83797 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.0 | 0.002561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 0.009727 | 0.002702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.0 | 0.002702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  0.01018 | 0.002828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.0 | 0.002828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 0.010586 | 0.002941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.0 | 0.002941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 0.010942 | 0.003039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.0 | 0.003039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 0.011263 | 0.003129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.0 | 0.003129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 0.011564 | 0.003212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.0 | 0.003212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  0.01363 | 0.003786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.0 | 0.003786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 0.034046 | 0.009457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.0 | 0.009457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 0.060472 | 0.016798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.0 | 0.016798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 0.088924 | 0.024701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.0 | 0.024701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  0.17003 | 0.047231 | 0.780374 |  0.0 |  0.0 |   44.22507 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.0 | 0.047231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 0.210341 | 0.058428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.0 | 0.058428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 0.183571 | 0.050992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.0 | 0.050992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 0.185428 | 0.051508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.0 | 0.051508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 0.198953 | 0.055265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.0 | 0.055265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 0.291718 | 0.081033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.0 | 0.081033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 0.384279 | 0.106744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.0 | 0.106744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 0.401551 | 0.111542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.0 | 0.111542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 0.391634 | 0.108787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.0 | 0.108787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  0.34564 | 0.096011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.0 | 0.096011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   0.2946 | 0.081833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.0 | 0.081833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 0.252616 | 0.070171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.0 | 0.070171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 0.218815 | 0.060782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.0 | 0.060782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 0.191684 | 0.053246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.0 | 0.053246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 0.206673 | 0.057409 | 0.125043 |  0.0 |  0.0 |  68.927693 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.0 | 0.057409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 0.288705 | 0.080196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.0 | 0.080196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 0.323726 | 0.089924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.0 | 0.089924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 0.821088 |  0.22808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.0 |  0.22808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 2.868463 | 0.796795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.0 | 0.796795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.504942 | 3.791851 | 1.053292 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.600789 | 0.904153 | 0.064494 | 0.140463 | 0.594939 | 0.837653 | 2.869278 |  0.02348 | 0.006795 | 0.054646 |   0.0 | 1.053292 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.045681 |  2.93515 | 0.815319 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.043686 | 0.001996 | 0.067538 | 0.186956 | 0.625375 | 0.923811 | 1.903847 | 0.025555 | 0.021764 | 0.060173 |   0.0 | 0.815319 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.527094 | 2.481423 | 0.689284 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.604289 | 0.922805 | 0.068447 | 0.201685 | 0.634469 | 0.998043 | 1.351716 | 0.027625 | 0.038197 | 0.065843 |   0.0 | 0.689284 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.227705 | 3.418899 | 0.949694 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.861643 | 5.366061 | 0.071072 | 0.246232 |  0.66072 | 1.132039 | 2.129639 |  0.02968 |  0.05591 | 0.071631 |   0.0 | 0.949694 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  6.97553 | 4.804338 | 1.334538 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.856642 | 5.118889 |  0.07804 | 0.377966 | 0.730399 | 1.263826 | 3.350409 |  0.03187 | 0.080394 |  0.07784 |   0.0 | 1.334538 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 |  3.09949 |  4.84592 | 1.346089 |  0.78175 |  0.0 |  0.0 | 169.469597 | 1.677366 | 1.422124 |  0.08319 | 0.486735 | 0.781897 | 1.354495 | 3.258614 | 0.034249 | 0.113973 |  0.08459 |   0.0 | 1.346089 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 0.948023 | 3.857115 | 1.071421 | 0.782063 |  0.0 |  0.0 | 169.380125 | 0.948023 |      0.0 | 0.084735 | 0.521116 | 0.797348 | 1.344716 | 2.233012 | 0.036674 | 0.151109 | 0.091605 |   0.0 | 1.071421 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 2.488688 | 3.201733 |  0.88937 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.598182 | 0.890506 |  0.08469 |  0.52011 | 0.796901 | 1.333698 | 1.544122 | 0.039016 | 0.186271 | 0.098625 |   0.0 |  0.88937 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 2.838425 | 0.788451 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.351893 | 1.120521 | 0.041268 |  0.21912 | 0.105622 |   0.0 | 0.788451 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 0.249877 | 2.347956 |  0.65221 | 0.782239 |  0.0 |  0.0 | 170.639599 | 0.249877 |      0.0 | 0.085875 | 0.546974 | 0.808745 | 1.242434 | 0.699367 | 0.043439 | 0.250108 | 0.112609 |   0.0 |  0.65221 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 |  1.91484 |   0.5319 | 0.782169 |  0.0 |  0.0 | 169.345117 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.047903 | 0.424187 | 0.045495 | 0.277747 | 0.119508 |   0.0 |   0.5319 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 0.242047 | 1.621283 | 0.450356 | 0.781959 |  0.0 |  0.0 | 168.284125 | 0.242047 |      0.0 | 0.084673 | 0.519716 | 0.796726 | 0.888843 | 0.257283 | 0.047421 | 0.301457 | 0.126278 |   0.0 | 0.450356 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 1.523064 | 0.423073 | 0.781681 |  0.0 |  0.0 |  168.57882 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 0.857129 | 0.162134 | 0.049225 | 0.321653 | 0.132923 |   0.0 | 0.423073 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.189947 | 1.454205 | 0.403946 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.189947 |      0.0 | 0.084289 | 0.511125 | 0.792894 | 0.820771 | 0.103491 | 0.050931 | 0.339529 | 0.139482 |   0.0 | 0.403946 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 1.313091 | 0.364748 | 0.705926 |  0.0 |  0.0 | 166.216858 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.696563 |  0.06277 | 0.052545 | 0.355257 | 0.145956 |   0.0 | 0.364748 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 |      0.0 | 1.190388 | 0.330663 | 0.471181 |  0.0 |  0.0 | 164.867718 |      0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 | 0.577717 | 0.038072 |  0.05405 | 0.368242 | 0.152307 |   0.0 | 0.330663 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 1.088702 | 0.302417 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.472994 | 0.023092 | 0.055451 | 0.378635 | 0.158529 |   0.0 | 0.302417 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 |      0.0 | 1.009088 | 0.280302 |      0.0 |  0.0 |  0.0 | 161.377451 |      0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 | 0.387255 | 0.014006 | 0.056746 | 0.386466 | 0.164615 |   0.0 | 0.280302 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 1.028002 | 0.285556 |      0.0 |  0.0 |  0.0 | 161.294054 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.399248 | 0.008495 | 0.057936 |  0.39177 | 0.170553 |   0.0 | 0.285556 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.014983 | 1.041382 | 0.289273 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.014983 |      0.0 | 0.080647 | 0.431887 |  0.75647 | 0.405172 | 0.005153 | 0.059044 | 0.395629 | 0.176385 |   0.0 | 0.289273 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 0.976673 | 0.271298 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.333039 | 0.003125 | 0.060079 | 0.398311 | 0.182119 |   0.0 | 0.271298 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 |      0.0 | 0.922391 |  0.25622 |      0.0 |  0.0 |  0.0 | 155.593338 |      0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 | 0.272669 | 0.001896 | 0.061023 |  0.39909 | 0.187714 |   0.0 |  0.25622 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 0.877158 | 0.243655 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.223243 |  0.00115 |  0.06187 | 0.397747 | 0.193148 |   0.0 | 0.243655 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 |      0.0 | 0.838885 | 0.233024 |      0.0 |  0.0 |  0.0 | 151.294475 |      0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 | 0.182776 | 0.000697 | 0.062621 | 0.394374 | 0.198417 |   0.0 | 0.233024 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 0.806179 | 0.223939 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.149644 | 0.000423 | 0.063282 | 0.389303 | 0.203526 |   0.0 | 0.223939 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 |      0.0 | 0.778035 | 0.216121 |      0.0 |  0.0 |  0.0 | 148.167947 |      0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 | 0.122518 | 0.000257 | 0.063864 | 0.382909 | 0.208487 |   0.0 | 0.216121 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 0.753875 |  0.20941 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.100309 | 0.000156 | 0.064379 |  0.37571 | 0.213321 |   0.0 |  0.20941 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 |      0.0 | 0.733188 | 0.203663 |      0.0 |  0.0 |  0.0 | 145.983055 |      0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 | 0.082126 | 0.000094 | 0.064839 | 0.368084 | 0.218044 |   0.0 | 0.203663 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 0.715434 | 0.198732 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.067239 | 0.000057 | 0.065249 | 0.360222 | 0.222667 |   0.0 | 0.198732 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 |      0.0 | 0.700075 | 0.194465 |      0.0 |  0.0 |  0.0 | 143.898606 |      0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 | 0.055051 | 0.000035 | 0.065614 | 0.352185 |  0.22719 |   0.0 | 0.194465 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 | 0.686662 | 0.190739 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 | 0.045072 | 0.000021 | 0.065935 | 0.344017 | 0.231617 |   0.0 | 0.190739 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |      0.0 |  0.67483 | 0.187453 |      0.0 |  0.0 |  0.0 | 141.875122 |      0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 | 0.036902 | 0.000013 | 0.066216 | 0.335751 | 0.235948 |   0.0 | 0.187453 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 0.664287 | 0.184524 |      0.0 |  0.0 |  0.0 | 140.884763 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.030213 | 0.000008 | 0.066458 | 0.327422 | 0.240186 |   0.0 | 0.184524 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 |      0.0 | 0.654794 | 0.181887 |      0.0 |  0.0 |  0.0 | 139.908406 |      0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 | 0.024736 | 0.000005 | 0.066665 | 0.319056 | 0.244333 |   0.0 | 0.181887 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 0.666919 | 0.185255 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.041012 | 0.000003 | 0.066837 | 0.310678 | 0.248389 |   0.0 | 0.185255 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 | 0.475631 | 0.717067 | 0.199185 | 0.766603 |  0.0 |  0.0 | 138.559658 | 0.475631 |      0.0 |  0.06975 | 0.223431 | 0.171871 | 0.097544 | 0.000002 | 0.066984 | 0.302538 |     0.25 |   0.0 | 0.199185 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 | 0.314929 | 0.762763 | 0.211879 |  0.74963 |  0.0 |  0.0 | 137.632082 | 0.314929 |      0.0 |  0.06928 | 0.215498 | 0.327869 | 0.151029 | 0.000001 | 0.067107 | 0.294626 |     0.25 |   0.0 | 0.211879 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 |  0.46577 | 0.798791 | 0.221886 | 0.687585 |  0.0 |  0.0 | 136.717337 |  0.46577 |      0.0 | 0.068816 | 0.207769 |  0.17239 | 0.194866 | 0.000001 | 0.067202 | 0.286722 |     0.25 |   0.0 | 0.221886 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 |  0.51072 | 0.844296 | 0.234527 | 0.521608 |  0.0 |  0.0 | 136.338253 |  0.51072 |      0.0 | 0.068359 | 0.200239 | 0.327352 | 0.248182 |      0.0 | 0.067269 | 0.278844 |     0.25 |   0.0 | 0.234527 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.515075 | 0.884713 | 0.245754 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.515075 |      0.0 | 0.068169 | 0.197146 | 0.172906 |  0.29618 |      0.0 | 0.067318 | 0.271214 |     0.25 |   0.0 | 0.245754 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 | 0.301095 | 0.897041 | 0.249178 |      0.0 |  0.0 |  0.0 |  133.98234 | 0.301095 |      0.0 | 0.067793 | 0.191061 | 0.326838 | 0.315819 |      0.0 |  0.06735 | 0.263872 |     0.25 |   0.0 | 0.249178 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 | 0.446494 | 0.900444 | 0.250123 |      0.0 |  0.0 |  0.0 | 132.218945 | 0.446494 |      0.0 | 0.066991 | 0.178288 | 0.173418 | 0.326767 |      0.0 | 0.067352 | 0.256325 |     0.25 |   0.0 | 0.250123 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 | 0.284768 | 0.898873 | 0.249687 |      0.0 |  0.0 |  0.0 | 129.797091 | 0.284768 |      0.0 | 0.066109 | 0.164592 | 0.326327 | 0.333324 |      0.0 | 0.067313 | 0.248236 |     0.25 |   0.0 | 0.249687 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 | 0.425059 | 0.894281 | 0.248411 |      0.0 |  0.0 |  0.0 | 127.414065 | 0.425059 |      0.0 | 0.064899 | 0.146387 | 0.173927 | 0.337661 |      0.0 | 0.067224 | 0.239396 |     0.25 |   0.0 | 0.248411 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |  0.26125 | 0.884956 | 0.245821 |      0.0 |  0.0 |  0.0 | 124.635441 |  0.26125 |      0.0 | 0.063707 | 0.129182 |  0.32582 | 0.338162 |      0.0 | 0.067082 | 0.229713 |     0.25 |   0.0 | 0.245821 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 | 0.398745 |   0.8732 | 0.242556 |      0.0 |  0.0 |  0.0 | 122.150124 | 0.398745 |      0.0 | 0.062318 | 0.110048 | 0.174432 | 0.337097 |      0.0 | 0.066883 |  0.21922 |     0.25 |   0.0 | 0.242556 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 | 0.235434 | 0.857653 | 0.238237 |      0.0 |  0.0 |  0.0 | 119.985132 | 0.235434 |      0.0 | 0.061075 | 0.093822 | 0.325316 | 0.332977 |      0.0 |  0.06663 | 0.208046 |     0.25 |   0.0 | 0.238237 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 | 0.374991 | 0.841225 | 0.233674 |      0.0 |  0.0 |  0.0 | 118.139837 | 0.374991 |      0.0 | 0.059993 | 0.080409 | 0.174934 | 0.328366 |      0.0 | 0.066332 | 0.196528 |     0.25 |   0.0 | 0.233674 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 | 0.215883 | 0.822868 | 0.228574 |      0.0 |  0.0 |  0.0 | 116.932422 | 0.215883 |      0.0 |  0.05907 | 0.069533 | 0.324816 | 0.321917 |      0.0 |    0.066 | 0.184951 |     0.25 |   0.0 | 0.228574 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 | 0.359229 | 0.805406 | 0.223724 |      0.0 |  0.0 |  0.0 | 116.038409 | 0.359229 |      0.0 | 0.058466 | 0.062708 | 0.175433 | 0.316121 |      0.0 | 0.065647 | 0.173638 |     0.25 |   0.0 | 0.223724 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 | 0.205873 | 0.787703 | 0.218806 |      0.0 |  0.0 |  0.0 | 115.332585 | 0.205873 |      0.0 | 0.058019 | 0.057808 | 0.324319 | 0.309573 |      0.0 | 0.065286 | 0.162844 |     0.25 |   0.0 | 0.218806 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016212 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 | 0.350734 | 0.771931 | 0.214425 |      0.0 |  0.0 |  0.0 |  114.67801 | 0.350734 |      0.0 | 0.057666 | 0.054034 | 0.175929 | 0.304342 |      0.0 | 0.064923 | 0.152666 |     0.25 |   0.0 | 0.214425 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 | 0.199565 | 0.756269 | 0.210075 |      0.0 |  0.0 |  0.0 | 114.031165 | 0.199565 |      0.0 | 0.057339 | 0.050611 | 0.323825 | 0.298594 |      0.0 | 0.064561 | 0.143114 |     0.25 |   0.0 | 0.210075 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 | 0.343735 | 0.742497 | 0.206249 |      0.0 |  0.0 |  0.0 | 113.391444 | 0.343735 |      0.0 | 0.057016 | 0.047302 | 0.176421 | 0.294145 |      0.0 | 0.064201 | 0.134151 |     0.25 |   0.0 | 0.206249 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 | 0.193623 |  0.72865 | 0.202403 |      0.0 |  0.0 |  0.0 | 112.758402 | 0.193623 |      0.0 | 0.056696 | 0.044105 | 0.323334 | 0.289076 |      0.0 | 0.063843 | 0.125732 |     0.25 |   0.0 | 0.202403 |


.. _lland_v1_acker_gw_rise:

acre (limited groundwater rise)
_______________________________

Similar to restricting an aquifer's capacity (:ref:`lland_v1_acker_gw_vol`),
HydPy-L allows to limit the speed of the rise of the water table (see
the documentation on method |Update_QDGZ_QBGZ_QBGA_V2| for further information).
After setting the related threshold parameters to 0.4 mm/h (|GSBGrad1|) and
0.6 mm/h (|GSBGrad2|), we see an increase in runoff due to the redirection
of groundwater recharge to direct runoff at the end of the event:

.. integration-test::

    >>> volbmax(inf)
    >>> gsbgrad1(0.4)
    >>> gsbgrad2(0.6)
    >>> test("lland_v1_acker_gw_rise",
    ...      axis1=(states.qbgz, states.qbga, fluxes.qdgz, fluxes.qah))
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |      qdb |     qib1 |     qib2 |      qbb |     qkap |     qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000495 | 0.000138 |      0.0 |  0.0 |  0.0 |  19.935025 |      0.0 |      0.0 |     0.01 |      0.0 |     0.05 |      0.0 |      0.0 | 0.000246 |      0.0 | 0.000249 |   0.0 | 0.000138 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.004816 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |      0.0 | 0.009968 |      0.0 | 0.049675 |  0.00026 |      0.0 | 0.001462 | 0.000406 |      0.0 |  0.0 |  0.0 |  19.870826 |      0.0 |      0.0 | 0.009968 |      0.0 | 0.049415 |      0.0 |      0.0 | 0.000721 |      0.0 | 0.000741 |   0.0 | 0.000406 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004761 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009935 |      0.0 | 0.049354 | 0.000517 |      0.0 | 0.002394 | 0.000665 |      0.0 |  0.0 |  0.0 |  19.807293 |      0.0 |      0.0 | 0.009935 |      0.0 | 0.048837 |      0.0 |      0.0 | 0.001171 |      0.0 | 0.001223 |   0.0 | 0.000665 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004698 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.009904 |      0.0 | 0.049036 | 0.000771 |      0.0 | 0.003291 | 0.000914 |      0.0 |  0.0 |  0.0 |  19.744426 |      0.0 |      0.0 | 0.009904 |      0.0 | 0.048266 |      0.0 |      0.0 | 0.001598 |      0.0 | 0.001694 |   0.0 | 0.000914 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004732 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |      0.0 | 0.009872 |      0.0 | 0.048722 | 0.001022 |      0.0 | 0.004156 | 0.001154 |      0.0 |  0.0 |  0.0 |  19.682122 |      0.0 |      0.0 | 0.009872 |      0.0 |   0.0477 |      0.0 |      0.0 | 0.002002 |      0.0 | 0.002154 |   0.0 | 0.001154 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.004999 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |      0.0 | 0.009841 |      0.0 | 0.048411 | 0.001272 |      0.0 |  0.00499 | 0.001386 |      0.0 |  0.0 |  0.0 |  19.620143 |      0.0 |      0.0 | 0.009841 |      0.0 | 0.047139 |      0.0 |      0.0 | 0.002385 |      0.0 | 0.002605 |   0.0 | 0.001386 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 | 0.014156 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |      0.0 |  0.00981 |      0.0 | 0.048101 | 0.001519 |      0.0 | 0.005793 | 0.001609 |      0.0 |  0.0 |  0.0 |  19.549595 |      0.0 |      0.0 |  0.00981 |      0.0 | 0.046581 |      0.0 |      0.0 | 0.002748 |      0.0 | 0.003045 |   0.0 | 0.001609 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 | 0.093411 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |      0.0 | 0.009775 |      0.0 | 0.047748 | 0.001802 |      0.0 | 0.006566 | 0.001824 |      0.0 |  0.0 |  0.0 |  19.400463 |      0.0 |      0.0 | 0.009775 |      0.0 | 0.045946 |      0.0 |      0.0 | 0.003091 |      0.0 | 0.003475 |   0.0 | 0.001824 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.209617 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |      0.0 |   0.0097 |      0.0 | 0.047002 | 0.002398 |      0.0 | 0.007306 |  0.00203 |      0.0 |  0.0 |  0.0 |  19.136542 |      0.0 |      0.0 |   0.0097 |      0.0 | 0.044604 |      0.0 |      0.0 | 0.003416 |      0.0 | 0.003891 |   0.0 |  0.00203 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.346436 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |      0.0 | 0.009568 |      0.0 | 0.045683 | 0.003454 |      0.0 | 0.008003 | 0.002223 |      0.0 |  0.0 |  0.0 |  18.738309 |      0.0 |      0.0 | 0.009568 |      0.0 | 0.042229 |      0.0 |      0.0 | 0.003719 |      0.0 | 0.004284 |   0.0 | 0.002223 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 |   0.3834 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |      0.0 | 0.009369 |      0.0 | 0.043692 | 0.005047 |      0.0 | 0.008643 | 0.002401 |      0.0 |  0.0 |  0.0 |  18.306895 |      0.0 |      0.0 | 0.009369 |      0.0 | 0.038645 |      0.0 |      0.0 | 0.003999 |      0.0 | 0.004644 |   0.0 | 0.002401 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.425009 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |      0.0 | 0.009153 |      0.0 | 0.041534 | 0.006772 |      0.0 | 0.009219 | 0.002561 |      0.0 |  0.0 |  0.0 |   17.83797 |      0.0 |      0.0 | 0.009153 |      0.0 | 0.034762 |      0.0 |      0.0 | 0.004256 |      0.0 | 0.004963 |   0.0 | 0.002561 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.22675 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |      0.0 | 0.008919 |      0.0 |  0.03919 | 0.008648 |      0.0 | 0.009727 | 0.002702 |      0.0 |  0.0 |  0.0 |   17.57176 |      0.0 |      0.0 | 0.008919 |      0.0 | 0.030542 |      0.0 |      0.0 | 0.004489 |      0.0 | 0.005238 |   0.0 | 0.002702 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.288809 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |      0.0 | 0.008786 |      0.0 | 0.037859 | 0.009713 |      0.0 |  0.01018 | 0.002828 |      0.0 |  0.0 |  0.0 |   17.24602 |      0.0 |      0.0 | 0.008786 |      0.0 | 0.028146 |      0.0 |      0.0 | 0.004702 |      0.0 | 0.005478 |   0.0 | 0.002828 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.316685 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |      0.0 | 0.008623 |      0.0 |  0.03623 | 0.011016 |      0.0 | 0.010586 | 0.002941 |      0.0 |  0.0 |  0.0 |  16.895498 |      0.0 |      0.0 | 0.008623 |      0.0 | 0.025214 |      0.0 |      0.0 | 0.004897 |      0.0 | 0.005689 |   0.0 | 0.002941 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 |     0.05 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |      0.0 | 0.008448 |      0.0 | 0.034477 | 0.012418 |      0.0 | 0.010942 | 0.003039 |      0.0 |  0.0 |  0.0 |  16.814991 |      0.0 |      0.0 | 0.008448 |      0.0 | 0.022059 |      0.0 |      0.0 | 0.005074 |      0.0 | 0.005868 |   0.0 | 0.003039 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 |  0.07643 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |      0.0 | 0.008407 |      0.0 | 0.034075 |  0.01274 |      0.0 | 0.011263 | 0.003129 |      0.0 |  0.0 |  0.0 |  16.708818 |      0.0 |      0.0 | 0.008407 |      0.0 | 0.021335 |      0.0 |      0.0 | 0.005238 |      0.0 | 0.006025 |   0.0 | 0.003129 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.047182 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |      0.0 | 0.008354 |      0.0 | 0.033544 | 0.013165 |      0.0 | 0.011564 | 0.003212 |      0.0 |  0.0 |  0.0 |  16.632903 |      0.0 |      0.0 | 0.008354 |      0.0 | 0.020379 |      0.0 |      0.0 | 0.005391 |      0.0 | 0.006173 |   0.0 | 0.003212 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |  0.01905 | 0.008316 |      0.0 | 0.033165 | 0.013468 |  0.01905 |  0.01363 | 0.003786 | 0.649803 |  0.0 |  0.0 |   17.34584 |  0.01905 |      0.0 | 0.008316 |      0.0 | 0.019696 | 0.001784 |      0.0 | 0.005535 |      0.0 | 0.006311 |   0.0 | 0.003786 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 | 0.200321 | 0.008673 |      0.0 | 0.036729 | 0.010617 | 0.200321 | 0.034046 | 0.009457 | 0.755221 |  0.0 |  0.0 |  23.680536 | 0.200321 |      0.0 | 0.008673 |      0.0 | 0.026113 | 0.021891 |      0.0 | 0.005679 |      0.0 | 0.006476 |   0.0 | 0.009457 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 | 0.130408 |  0.01184 |      0.0 | 0.068403 |      0.0 | 0.130408 | 0.060472 | 0.016798 | 0.780115 |  0.0 |  0.0 |  26.905106 | 0.130408 |      0.0 |  0.01184 |      0.0 | 0.068403 | 0.047687 |      0.0 | 0.005903 |      0.0 | 0.006882 |   0.0 | 0.016798 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 | 0.263195 | 0.013453 |      0.0 | 0.084526 |      0.0 | 0.263195 | 0.088924 | 0.024701 | 0.780309 |  0.0 |  0.0 |  32.404048 | 0.263195 |      0.0 | 0.013453 |      0.0 | 0.084526 | 0.075118 |      0.0 | 0.006232 |      0.0 | 0.007574 |   0.0 | 0.024701 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 | 0.751065 | 0.016202 |      0.0 |  0.11202 |      0.0 | 0.751065 |  0.17003 | 0.047231 | 0.780374 |  0.0 |  0.0 |   44.22507 | 0.751065 |      0.0 | 0.016202 |      0.0 |  0.11202 | 0.154901 |      0.0 | 0.006652 |      0.0 | 0.008477 |   0.0 | 0.047231 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 | 0.006925 | 0.022113 |      0.0 | 0.171125 |      0.0 | 0.006925 | 0.210341 | 0.058428 | 0.780505 |  0.0 |  0.0 |  44.125281 | 0.006925 |      0.0 | 0.022113 |      0.0 | 0.171125 | 0.193276 |      0.0 | 0.007263 |      0.0 | 0.009802 |   0.0 | 0.058428 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 | 0.056938 | 0.022063 |      0.0 | 0.170626 |      0.0 | 0.056938 | 0.183571 | 0.050992 | 0.780505 |  0.0 |  0.0 |  44.696159 | 0.056938 |      0.0 | 0.022063 |      0.0 | 0.170626 |  0.16418 |      0.0 | 0.007986 |      0.0 | 0.011405 |   0.0 | 0.050992 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |  0.25986 | 0.022348 |      0.0 | 0.173481 |      0.0 |  0.25986 | 0.185428 | 0.051508 | 0.780505 |  0.0 |  0.0 |  47.820975 |  0.25986 |      0.0 | 0.022348 |      0.0 | 0.173481 | 0.163745 |      0.0 |  0.00868 |      0.0 | 0.013003 |   0.0 | 0.051508 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.19283 |  0.02391 |      0.0 | 0.189105 |      0.0 |  0.19283 | 0.198953 | 0.055265 | 0.780604 |  0.0 |  0.0 |  49.915634 |  0.19283 |      0.0 |  0.02391 |      0.0 | 0.189105 |  0.17489 |      0.0 | 0.009385 |      0.0 | 0.014678 |   0.0 | 0.055265 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.10822 | 0.024958 |      0.0 | 0.199578 |      0.0 |  1.10822 | 0.291718 | 0.081033 |  0.78067 |  0.0 |  0.0 |  61.043482 | 1.097652 | 0.010568 | 0.024958 |      0.0 | 0.199578 | 0.262882 | 0.002252 | 0.010119 |      0.0 | 0.016466 |   0.0 | 0.081033 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 | 0.427763 | 0.030522 |      0.0 | 0.255217 |      0.0 | 0.427763 | 0.384279 | 0.106744 | 0.780736 |  0.0 |  0.0 |   64.51065 | 0.427763 |      0.0 | 0.030522 |      0.0 | 0.255217 | 0.351463 | 0.003272 | 0.010979 |      0.0 | 0.018565 |   0.0 | 0.106744 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 | 0.441557 | 0.032255 |      0.0 | 0.272553 |      0.0 | 0.441557 | 0.401551 | 0.111542 | 0.780802 |  0.0 |  0.0 |   67.82502 | 0.441557 |      0.0 | 0.032255 |      0.0 | 0.272553 | 0.366585 | 0.001985 | 0.011975 |      0.0 | 0.021006 |   0.0 | 0.111542 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 | 0.160295 | 0.033913 |      0.0 | 0.289125 |      0.0 | 0.160295 | 0.391634 | 0.108787 |  0.76254 |  0.0 |  0.0 |   68.76249 | 0.160295 |      0.0 | 0.033913 |      0.0 | 0.289125 | 0.353834 | 0.001204 | 0.013005 |      0.0 | 0.023591 |   0.0 | 0.108787 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 | 0.009367 | 0.034381 |      0.0 | 0.293812 |      0.0 | 0.009367 |  0.34564 | 0.096011 | 0.547078 |  0.0 |  0.0 |   68.50747 | 0.009367 |      0.0 | 0.034381 |      0.0 | 0.293812 | 0.304616 |  0.00073 | 0.014036 |      0.0 | 0.026257 |   0.0 | 0.096011 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 | 0.043569 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 |      0.0 |      0.0 |   0.2946 | 0.081833 |      0.0 |  0.0 |  0.0 |   68.13711 |      0.0 |      0.0 | 0.034254 |      0.0 | 0.292537 | 0.250219 | 0.000443 | 0.015025 |      0.0 | 0.028913 |   0.0 | 0.081833 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.270284 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 |      0.0 |      0.0 | 0.252616 | 0.070171 |      0.0 |  0.0 |  0.0 |  67.542071 |      0.0 |      0.0 | 0.034069 |      0.0 | 0.290686 | 0.204862 | 0.000269 | 0.015958 |      0.0 | 0.031527 |   0.0 | 0.070171 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.008491 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 |      0.0 |      0.0 | 0.218815 | 0.060782 |      0.0 |  0.0 |  0.0 |  67.212099 |      0.0 |      0.0 | 0.033771 |      0.0 |  0.28771 | 0.167727 | 0.000163 | 0.016834 |      0.0 | 0.034091 |   0.0 | 0.060782 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.779235 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 |      0.0 |      0.0 | 0.191684 | 0.053246 |      0.0 |  0.0 |  0.0 |  66.113197 |      0.0 |      0.0 | 0.033606 |      0.0 |  0.28606 | 0.137323 | 0.000099 | 0.017656 |      0.0 | 0.036606 |   0.0 | 0.053246 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 | 0.391882 | 0.033057 |      0.0 | 0.280566 |      0.0 | 0.391882 | 0.206673 | 0.057409 | 0.125043 |  0.0 |  0.0 |  68.927693 | 0.391882 |      0.0 | 0.033057 |      0.0 | 0.280566 | 0.149132 |  0.00006 | 0.018421 |      0.0 | 0.039061 |   0.0 | 0.057409 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.679976 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 | 0.763787 | 0.034464 |      0.0 | 0.294638 |      0.0 | 0.763787 | 0.288705 | 0.080196 |      0.0 |  0.0 |  0.0 |   73.55987 | 0.763787 |      0.0 | 0.034464 |      0.0 | 0.294638 | 0.227966 | 0.000036 | 0.019169 |      0.0 | 0.041534 |   0.0 | 0.080196 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.006758 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 | 0.064088 |  0.03678 |      0.0 | 0.317799 |      0.0 | 0.064088 | 0.323726 | 0.089924 |      0.0 |  0.0 |  0.0 |  73.654444 | 0.064088 |      0.0 |  0.03678 |      0.0 | 0.317799 | 0.259564 | 0.000022 | 0.019972 |      0.0 | 0.044168 |   0.0 | 0.089924 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 | 3.471682 | 0.036827 |      0.0 | 0.318272 |      0.0 | 3.471682 | 0.821088 |  0.22808 | 0.377849 |  0.0 |  0.0 |  93.867662 | 1.711955 | 1.759727 | 0.036827 |      0.0 | 0.318272 | 0.378459 | 0.374943 | 0.020793 |      0.0 | 0.046893 |   0.0 |  0.22808 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 9.471536 | 0.046934 |      0.0 | 0.419338 |      0.0 | 9.471536 | 2.868463 | 0.796795 | 0.352149 |  0.0 |  0.0 | 128.987703 | 1.894421 | 7.577116 | 0.046934 |      0.0 | 0.419338 |  0.63727 | 2.159274 | 0.021823 |      0.0 | 0.050097 |   0.0 | 0.796795 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 | 2.504942 | 0.064494 | 0.140463 | 0.594939 |      0.0 | 2.668282 |  3.82292 | 1.061922 |  0.22443 |  0.0 |  0.0 | 135.075014 | 1.625227 | 1.043055 | 0.064494 | 0.140463 | 0.431599 | 0.839941 | 2.898873 |  0.02348 | 0.006795 | 0.053832 |   0.0 | 1.061922 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 | 1.045681 | 0.067538 | 0.186956 | 0.625375 |      0.0 | 1.272225 | 3.007362 | 0.835378 |  0.48171 |  0.0 |  0.0 | 136.893894 | 1.213975 | 0.058249 | 0.067538 | 0.186956 | 0.398832 | 0.943774 | 1.958841 | 0.025555 | 0.021764 | 0.057427 |   0.0 | 0.835378 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 | 2.527094 | 0.068447 | 0.201685 | 0.634469 |      0.0 | 2.708331 | 2.586897 | 0.718583 | 0.741679 |  0.0 |  0.0 | 142.143908 | 1.630769 | 1.077562 | 0.068447 | 0.201685 | 0.453233 | 1.031788 | 1.428193 | 0.027625 | 0.038197 | 0.061095 |   0.0 | 0.718583 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 | 7.227705 | 0.071072 | 0.246232 |  0.66072 |      0.0 | 7.536861 | 3.581202 | 0.994778 | 0.781476 |  0.0 |  0.0 | 156.079859 | 1.867319 | 5.669542 | 0.071072 | 0.246232 | 0.351563 | 1.162518 | 2.268604 |  0.02968 |  0.05591 | 0.064491 |   0.0 | 0.994778 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  6.97553 |  0.07804 | 0.377966 | 0.730399 |      0.0 |  7.24735 | 5.016139 | 1.393372 |  0.78151 |  0.0 |  0.0 | 166.379399 | 1.862019 | 5.385331 |  0.07804 | 0.377966 |  0.45858 | 1.289781 | 3.546214 |  0.03187 | 0.080394 |  0.06788 |   0.0 | 1.393372 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.09949 |  0.08319 | 0.486735 | 0.781897 |      0.0 | 3.685419 | 5.139055 | 1.427515 |  0.78175 |  0.0 |  0.0 | 169.469597 |  1.72866 | 1.956759 |  0.08319 | 0.486735 | 0.195968 |  1.38102 | 3.539354 | 0.034249 | 0.113973 | 0.070459 |   0.0 | 1.427515 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 | 0.948023 | 0.084735 | 0.521116 | 0.797348 |      0.0 | 1.045992 | 4.142085 | 1.150579 | 0.782063 |  0.0 |  0.0 | 169.380125 |  1.04397 | 0.002022 | 0.084735 | 0.521116 | 0.699379 | 1.379913 | 2.500173 | 0.036674 | 0.151109 | 0.074217 |   0.0 | 1.150579 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 | 2.488688 |  0.08469 |  0.52011 | 0.796901 |      0.0 | 3.285588 |  3.53784 | 0.982733 | 0.782028 |  0.0 |  0.0 |   171.3518 | 1.695641 | 1.589948 |  0.08469 |  0.52011 |      0.0 | 1.380048 | 1.855553 | 0.039016 | 0.186271 | 0.076952 |   0.0 | 0.982733 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 | 1.389949 | 0.085676 | 0.542438 | 0.806759 |      0.0 | 1.389949 | 3.174573 | 0.881826 | 0.782063 |  0.0 |  0.0 | 171.749006 | 1.280549 |   0.1094 | 0.085676 | 0.542438 | 0.806759 | 1.398381 | 1.435598 | 0.041268 |  0.21912 | 0.080207 |   0.0 | 0.881826 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 | 0.249877 | 0.085875 | 0.546974 | 0.808745 |      0.0 | 1.058622 | 2.624057 | 0.728905 | 0.782239 |  0.0 |  0.0 | 170.639599 | 1.055376 | 0.003246 | 0.085875 | 0.546974 |      0.0 | 1.355933 | 0.891162 | 0.043439 | 0.250108 | 0.083415 |   0.0 | 0.728905 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 | 0.093866 |  0.08532 | 0.534337 | 0.803198 |      0.0 | 0.093866 | 2.162335 | 0.600649 | 0.782169 |  0.0 |  0.0 | 169.345117 | 0.093866 |      0.0 |  0.08532 | 0.534337 | 0.803198 | 1.211402 | 0.541103 | 0.045495 | 0.277747 | 0.086588 |   0.0 | 0.600649 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 | 0.242047 | 0.084673 | 0.519716 | 0.796726 |      0.0 | 1.038772 | 1.864284 | 0.517857 | 0.781959 |  0.0 |  0.0 | 168.284125 | 1.037325 | 0.001447 | 0.084673 | 0.519716 |      0.0 | 1.097186 | 0.328504 | 0.047421 | 0.301457 | 0.089716 |   0.0 | 0.517857 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 | 1.183866 | 0.084142 | 0.507834 | 0.791421 |      0.0 | 1.183866 | 1.766622 | 0.490728 | 0.781681 |  0.0 |  0.0 |  168.57882 |  1.15531 | 0.028556 | 0.084142 | 0.507834 | 0.791421 | 1.097385 | 0.205593 | 0.049225 | 0.321653 | 0.092767 |   0.0 | 0.490728 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 | 0.189947 | 0.084289 | 0.511125 | 0.792894 |      0.0 | 0.982841 | 1.707818 | 0.474394 | 0.769028 |  0.0 |  0.0 | 167.462246 | 0.982841 |      0.0 | 0.084289 | 0.511125 |      0.0 | 1.091732 |  0.12985 | 0.050931 | 0.339529 | 0.095775 |   0.0 | 0.474394 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 | 0.084679 | 0.083731 | 0.498693 | 0.787311 |      0.0 | 0.084679 | 1.573183 | 0.436995 | 0.705926 |  0.0 |  0.0 | 166.216858 | 0.084679 |      0.0 | 0.083731 | 0.498693 | 0.787311 | 0.987877 | 0.078758 | 0.052545 | 0.355257 | 0.098745 |   0.0 | 0.436995 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.083108 | 0.484948 | 0.781084 |      0.0 | 0.781084 | 1.461111 | 0.405864 | 0.471181 |  0.0 |  0.0 | 164.867718 | 0.781084 |      0.0 | 0.083108 | 0.484948 |      0.0 | 0.889376 | 0.047769 |  0.05405 | 0.368242 | 0.101673 |   0.0 | 0.405864 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.362296 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 |      0.0 |      0.0 | 1.364175 | 0.378937 |      0.0 |  0.0 |  0.0 | 163.178446 |      0.0 |      0.0 | 0.082434 | 0.470203 | 0.774339 | 0.796595 | 0.028973 | 0.055451 | 0.378635 |  0.10452 |   0.0 | 0.378937 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.501557 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |      0.0 | 0.081589 | 0.451956 | 0.765892 |      0.0 | 0.765892 | 1.292037 | 0.358899 |      0.0 |  0.0 |  0.0 | 161.377451 | 0.765892 |      0.0 | 0.081589 | 0.451956 |      0.0 | 0.723925 | 0.017573 | 0.056746 | 0.386466 | 0.107326 |   0.0 | 0.358899 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 | 0.255451 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 | 0.877601 | 0.080689 | 0.432769 | 0.756887 |      0.0 | 0.877601 | 1.312389 | 0.364553 |      0.0 |  0.0 |  0.0 | 161.294054 | 0.877601 |      0.0 | 0.080689 | 0.432769 | 0.756887 | 0.741995 | 0.010659 | 0.057936 |  0.39177 |  0.11003 |   0.0 | 0.364553 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269179 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 | 0.014983 | 0.080647 | 0.431887 |  0.75647 |      0.0 | 0.771453 | 1.330468 | 0.369574 |      0.0 |  0.0 |  0.0 | 159.780888 | 0.771453 |      0.0 | 0.080647 | 0.431887 |      0.0 | 0.756635 | 0.006465 | 0.059044 | 0.395629 | 0.112695 |   0.0 | 0.369574 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.558494 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 |      0.0 |      0.0 | 1.264688 | 0.351302 |      0.0 |  0.0 |  0.0 | 157.977606 |      0.0 |      0.0 |  0.07989 | 0.415993 | 0.748904 | 0.687071 | 0.003921 | 0.060079 | 0.398311 | 0.115306 |   0.0 | 0.351302 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.168078 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |      0.0 | 0.078989 | 0.397313 | 0.739888 |      0.0 | 0.739888 | 1.212189 | 0.336719 |      0.0 |  0.0 |  0.0 | 155.593338 | 0.739888 |      0.0 | 0.078989 | 0.397313 |      0.0 |  0.63182 | 0.002378 | 0.061023 |  0.39909 | 0.117878 |   0.0 | 0.336719 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 |  0.94389 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 |      0.0 |      0.0 | 1.163509 | 0.323197 |      0.0 |  0.0 |  0.0 | 153.470627 |      0.0 |      0.0 | 0.077797 | 0.373058 | 0.727967 | 0.582116 | 0.001443 |  0.06187 | 0.397747 | 0.120333 |   0.0 | 0.323197 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.030166 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |      0.0 | 0.076735 | 0.351897 | 0.717353 |      0.0 | 0.714903 | 1.124183 | 0.312273 |      0.0 |  0.0 |  0.0 | 151.294475 | 0.714903 |      0.0 | 0.076735 | 0.351897 | 0.002451 |  0.54355 | 0.000875 | 0.062621 | 0.394374 | 0.122763 |   0.0 | 0.312273 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 |  0.63633 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 |      0.0 |      0.0 | 1.085848 | 0.301625 |      0.0 |  0.0 |  0.0 | 149.545391 |      0.0 |      0.0 | 0.075647 | 0.330635 | 0.706472 | 0.507657 | 0.000531 | 0.063282 | 0.389303 | 0.125075 |   0.0 | 0.301625 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.291076 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |      0.0 | 0.074773 | 0.313868 | 0.697727 |      0.0 | 0.607519 | 1.047415 | 0.290948 |      0.0 |  0.0 |  0.0 | 148.167947 | 0.607519 |      0.0 | 0.074773 | 0.313868 | 0.090208 | 0.472531 | 0.000322 | 0.063864 | 0.382909 | 0.127788 |   0.0 | 0.290948 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053009 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 |      0.0 |      0.0 | 1.010796 | 0.280777 |      0.0 |  0.0 |  0.0 | 147.049144 |      0.0 |      0.0 | 0.074084 | 0.300871 |  0.69084 | 0.440104 | 0.000195 | 0.064379 |  0.37571 | 0.130408 |   0.0 | 0.280777 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 |  0.01687 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |      0.0 | 0.073525 | 0.290449 | 0.685246 |      0.0 | 0.530557 | 0.976369 | 0.271214 |      0.0 |  0.0 |  0.0 | 145.983055 | 0.530557 |      0.0 | 0.073525 | 0.290449 | 0.154689 | 0.410015 | 0.000118 | 0.064839 | 0.368084 | 0.133312 |   0.0 | 0.271214 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016501 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 |      0.0 |      0.0 | 0.943862 | 0.262184 |      0.0 |  0.0 |  0.0 | 144.933013 |      0.0 |      0.0 | 0.072992 | 0.280633 | 0.679915 | 0.382177 | 0.000072 | 0.065249 | 0.360222 | 0.136142 |   0.0 | 0.262184 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016199 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.072467 | 0.271076 | 0.674665 |      0.0 | 0.467026 | 0.913681 |   0.2538 |      0.0 |  0.0 |  0.0 | 143.898606 | 0.467026 |      0.0 | 0.072467 | 0.271076 | 0.207639 | 0.356639 | 0.000044 | 0.065614 | 0.352185 | 0.139199 |   0.0 |   0.2538 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.015965 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |      0.0 |      0.0 |  0.88507 | 0.245853 |      0.0 |  0.0 |  0.0 |  142.87943 |      0.0 |      0.0 | 0.071949 | 0.261769 | 0.669493 |  0.33291 | 0.000026 | 0.065935 | 0.344017 | 0.142182 |   0.0 | 0.245853 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015765 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |      0.0 |  0.07144 | 0.252706 | 0.664397 |      0.0 |  0.40576 | 0.857929 | 0.238314 |      0.0 |  0.0 |  0.0 | 141.875122 |  0.40576 |      0.0 |  0.07144 | 0.252706 | 0.258637 | 0.310564 | 0.000016 | 0.066216 | 0.335751 | 0.145381 |   0.0 | 0.238314 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016165 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 |      0.0 |      0.0 | 0.832215 | 0.231171 |      0.0 |  0.0 |  0.0 | 140.884763 |      0.0 |      0.0 | 0.070938 |  0.24388 | 0.659376 | 0.289819 |  0.00001 | 0.066458 | 0.327422 | 0.148505 |   0.0 | 0.231171 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016211 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.070442 |  0.23528 | 0.654424 |      0.0 | 0.346376 | 0.807287 | 0.224247 |      0.0 |  0.0 |  0.0 | 139.908406 | 0.346376 |      0.0 | 0.070442 |  0.23528 | 0.308048 | 0.269724 | 0.000006 | 0.066665 | 0.319056 | 0.151838 |   0.0 | 0.224247 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 | 0.221667 | 0.069954 | 0.226902 | 0.649542 |      0.0 | 0.221667 | 0.804551 | 0.223486 | 0.783283 |  0.0 |  0.0 | 139.500341 | 0.221667 |      0.0 | 0.069954 | 0.226902 | 0.649542 | 0.271939 | 0.000004 | 0.066837 | 0.310678 | 0.155094 |   0.0 | 0.223486 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |      0.0 |  0.06975 | 0.223431 | 0.647502 |      0.0 | 0.294507 | 0.797708 | 0.221585 | 0.766603 |  0.0 |  0.0 | 138.559658 | 0.294507 |      0.0 |  0.06975 | 0.223431 | 0.352994 | 0.269648 | 0.000002 | 0.066984 | 0.302538 | 0.158536 |   0.0 | 0.221585 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 |      0.0 |      0.0 | 0.770222 | 0.213951 |  0.74963 |  0.0 |  0.0 | 137.632082 |      0.0 |      0.0 |  0.06928 | 0.215498 | 0.642798 | 0.246572 | 0.000001 | 0.067107 | 0.294626 | 0.161915 |   0.0 | 0.213951 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |      0.0 | 0.068816 | 0.207769 |  0.63816 |      0.0 | 0.243061 |  0.74403 | 0.206675 | 0.687585 |  0.0 |  0.0 | 136.717337 | 0.243061 |      0.0 | 0.068816 | 0.207769 | 0.395099 |  0.22464 | 0.000001 | 0.067202 | 0.286722 | 0.165466 |   0.0 | 0.206675 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 | 0.204485 | 0.068359 | 0.200239 | 0.633587 |      0.0 | 0.204485 | 0.739419 | 0.205394 | 0.521608 |  0.0 |  0.0 | 136.338253 | 0.204485 |      0.0 | 0.068359 | 0.200239 | 0.633587 | 0.224366 |      0.0 | 0.067269 | 0.278844 | 0.168939 |   0.0 | 0.205394 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 | 0.056289 | 0.068169 | 0.197146 | 0.631691 |      0.0 | 0.250179 | 0.736161 | 0.204489 | 0.160825 |  0.0 |  0.0 | 135.586566 | 0.250179 |      0.0 | 0.068169 | 0.197146 | 0.437801 | 0.225042 |      0.0 | 0.067318 | 0.271214 | 0.172587 |   0.0 | 0.204489 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.717438 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 |      0.0 |      0.0 | 0.713563 | 0.198212 |      0.0 |  0.0 |  0.0 |  133.98234 |      0.0 |      0.0 | 0.067793 | 0.191061 | 0.627933 | 0.206168 |      0.0 |  0.06735 | 0.263872 | 0.176173 |   0.0 | 0.198212 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.898204 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |      0.0 | 0.066991 | 0.178288 | 0.619912 |      0.0 | 0.141065 |  0.68561 | 0.190447 |      0.0 |  0.0 |  0.0 | 132.218945 | 0.141065 |      0.0 | 0.066991 | 0.178288 | 0.478847 | 0.182008 |      0.0 | 0.067352 | 0.256325 | 0.179925 |   0.0 | 0.190447 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 | 1.580057 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 |      0.0 |      0.0 | 0.660482 | 0.183467 |      0.0 |  0.0 |  0.0 | 129.797091 |      0.0 |      0.0 | 0.066109 | 0.164592 | 0.611095 | 0.161375 |      0.0 | 0.067313 | 0.248236 | 0.183558 |   0.0 | 0.183467 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 | 1.572755 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |      0.0 | 0.064899 | 0.146387 | 0.598985 |      0.0 | 0.058015 | 0.631639 | 0.175455 |      0.0 |  0.0 |  0.0 | 127.414065 | 0.058015 |      0.0 | 0.064899 | 0.146387 |  0.54097 | 0.137556 |      0.0 | 0.067224 | 0.239396 | 0.187463 |   0.0 | 0.175455 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 1.998665 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 |      0.0 |      0.0 | 0.605709 | 0.168252 |      0.0 |  0.0 |  0.0 | 124.635441 |      0.0 |      0.0 | 0.063707 | 0.129182 |  0.58707 | 0.117704 |      0.0 | 0.067082 | 0.229713 |  0.19121 |   0.0 | 0.168252 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.739773 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 |      0.0 |      0.0 | 0.577551 | 0.160431 |      0.0 |  0.0 |  0.0 | 122.150124 |      0.0 |      0.0 | 0.062318 | 0.110048 | 0.573177 | 0.096368 |      0.0 | 0.066883 |  0.21922 |  0.19508 |   0.0 | 0.160431 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.449345 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 |      0.0 |      0.0 | 0.552355 | 0.153432 |      0.0 |  0.0 |  0.0 | 119.985132 |      0.0 |      0.0 | 0.061075 | 0.093822 | 0.560751 | 0.078899 |      0.0 |  0.06663 | 0.208046 |  0.19878 |   0.0 | 0.153432 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.154968 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 |      0.0 |      0.0 | 0.529785 | 0.147162 |      0.0 |  0.0 |  0.0 | 118.139837 |      0.0 |      0.0 | 0.059993 | 0.080409 | 0.549926 | 0.064597 |      0.0 | 0.066332 | 0.196528 | 0.202328 |   0.0 | 0.147162 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.538112 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 |      0.0 |      0.0 |  0.50958 |  0.14155 |      0.0 |  0.0 |  0.0 | 116.932422 |      0.0 |      0.0 |  0.05907 | 0.069533 | 0.540699 | 0.052888 |      0.0 |    0.066 | 0.184951 | 0.205741 |   0.0 |  0.14155 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.238177 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 |      0.0 |      0.0 | 0.491629 | 0.136564 |      0.0 |  0.0 |  0.0 | 116.038409 |      0.0 |      0.0 | 0.058466 | 0.062708 | 0.534662 | 0.043301 |      0.0 | 0.065647 | 0.173638 | 0.209043 |   0.0 | 0.136564 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.059805 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 |      0.0 |      0.0 | 0.475843 | 0.132179 |      0.0 |  0.0 |  0.0 | 115.332585 |      0.0 |      0.0 | 0.058019 | 0.057808 | 0.530192 | 0.035452 |      0.0 | 0.065286 | 0.162844 | 0.212261 |   0.0 | 0.132179 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016212 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 |      0.0 |      0.0 | 0.462022 | 0.128339 |      0.0 |  0.0 |  0.0 |  114.67801 |      0.0 |      0.0 | 0.057666 | 0.054034 | 0.526663 | 0.029025 |      0.0 | 0.064923 | 0.152666 | 0.215407 |   0.0 | 0.128339 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.015506 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 |      0.0 |      0.0 | 0.449927 |  0.12498 |      0.0 |  0.0 |  0.0 | 114.031165 |      0.0 |      0.0 | 0.057339 | 0.050611 |  0.52339 | 0.023764 |      0.0 | 0.064561 | 0.143114 | 0.218488 |   0.0 |  0.12498 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015247 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 |      0.0 |      0.0 | 0.439314 | 0.122032 |      0.0 |  0.0 |  0.0 | 113.391444 |      0.0 |      0.0 | 0.057016 | 0.047302 | 0.520156 | 0.019456 |      0.0 | 0.064201 | 0.134151 | 0.221505 |   0.0 | 0.122032 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015284 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |      0.0 |      0.0 | 0.429965 | 0.119435 |      0.0 |  0.0 |  0.0 | 112.758402 |      0.0 |      0.0 | 0.056696 | 0.044105 | 0.516957 |  0.01593 |      0.0 | 0.063843 | 0.125732 | 0.224461 |   0.0 | 0.119435 |


.. _lland_v1_acker_qkap:

acre (capillary rise)
_____________________

Now we focus on the capillary rise (|QKap|), being configurable via the
parameters |KapMax| and |KapGrenz|.  Please first read the documentation
on parameter |KapGrenz| to understand its keyword argument `option`.

In integration test :ref:`lland_v1_acker_summer`, capillary rise occurs
only for a short period where the soil water content (|BoWa|) is less than
10 % of its capacity (|WMax|), due to to the selected option `0_WMax/10`.
Additionally, there is no deep percolation (|QBB|) during this short period,
due to parameter |RBeta| set to |False|. (A note for `LARSIM`_ users: this
combination of parameter settings corresponds to the option
`KOPPELUNG BODEN/GRUNDWASSER`.)

Now we assign `FK` to parameter |KapGrenz| and |True| to parameter |RBeta|.
Consequently, capillary rise and deep percolation exclude each other, and
there is an abrupt transition from capillary rise to deep percolation when
the soil moisture exceeds |FK| (which agrees with the `LARSIM`_ option
`KAPILLARER AUFSTIEG`):

.. integration-test::

    >>> gsbgrad2(inf)
    >>> gsbgrad1(inf)
    >>> kapgrenz(option="FK")
    >>> rbeta(True)
    >>> test("lland_v1_acker_qkap",
    ...      axis1=(fluxes.qbb, fluxes.qkap),
    ...      axis2=states.bowa)
    |                date | nied | teml |  glob |  qz | qzh |  nkor | tkor |      nbes | sbes |      et0 |     evpo |      evi |      evb |   wgtf |    wnied |   schmpot | schm |      wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |      qah |       qa |     inzp | wats | waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |      qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 | 21.2 |   0.0 | 0.0 | 0.0 |   0.0 | 22.0 |       0.0 |  0.0 | 0.040283 | 0.020141 |      0.0 | 0.004975 |  3.674 |      0.0 |      11.0 |  0.0 |       0.0 |       0.0 |     0.01 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.065025 |      0.0 |      0.0 |     0.01 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000246 |      0.0 | -0.000399 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | 19.4 |   0.0 | 0.0 | 0.0 |   0.0 | 20.2 |       0.0 |  0.0 | 0.039121 |  0.01956 |      0.0 | 0.004846 | 3.3734 |      0.0 |      10.1 |  0.0 |       0.0 |       0.0 | 0.010033 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.130146 |      0.0 |      0.0 | 0.010033 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000722 |      0.0 | -0.001191 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004821 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 | 0.010065 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.195261 |      0.0 |      0.0 | 0.010065 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001177 |      0.0 | -0.001975 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | 18.3 |   0.0 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 0.038396 | 0.019198 |      0.0 | 0.004786 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 | 0.010098 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.260377 |      0.0 |      0.0 | 0.010098 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001612 |      0.0 | -0.002751 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | 18.9 |   0.0 | 0.0 | 0.0 |   0.0 | 19.7 |       0.0 |  0.0 | 0.038793 | 0.019396 |      0.0 | 0.004851 | 3.2899 |      0.0 |      9.85 |  0.0 |       0.0 |       0.0 |  0.01013 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.325396 |      0.0 |      0.0 |  0.01013 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002026 |      0.0 |  -0.00352 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | 22.5 |   0.0 | 0.0 | 0.0 |   0.0 | 23.3 |       0.0 |  0.0 | 0.041105 | 0.020552 |      0.0 | 0.005156 | 3.8911 |      0.0 |     11.65 |  0.0 |       0.0 |       0.0 | 0.010163 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.390078 |      0.0 |      0.0 | 0.010163 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002422 |      0.0 | -0.004281 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | 25.1 |  11.2 | 0.0 | 0.0 |   0.0 | 25.9 |       0.0 |  0.0 | 0.116763 | 0.058382 |      0.0 |  0.01469 | 4.3253 |      0.0 |     12.95 |  0.0 |       0.0 |       0.0 | 0.010195 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  20.445193 |      0.0 |      0.0 | 0.010195 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002801 |      0.0 | -0.005034 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | 28.3 | 105.5 | 0.0 | 0.0 |   0.0 | 29.1 |       0.0 |  0.0 |  0.77315 | 0.386575 |      0.0 |  0.09752 | 4.8597 |      0.0 |     14.55 |  0.0 |       0.0 |       0.0 | 0.010223 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   20.41745 |      0.0 |      0.0 | 0.010223 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003162 |      0.0 |  -0.00578 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | 27.8 | 248.3 | 0.0 | 0.0 |   0.0 | 28.6 |       0.0 |  0.0 | 1.747814 | 0.873907 |      0.0 | 0.220171 | 4.7762 |      0.0 |      14.3 |  0.0 |       0.0 |       0.0 | 0.010209 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   20.26707 |      0.0 |      0.0 | 0.010209 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003506 |      0.0 | -0.006519 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | 31.4 | 401.3 | 0.0 | 0.0 |   0.0 | 32.2 |       0.0 |  0.0 | 2.927022 | 1.463511 |      0.0 | 0.366109 | 5.3774 |      0.0 |      16.1 |  0.0 |       0.0 |       0.0 | 0.010134 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.970828 |      0.0 |      0.0 | 0.010134 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003831 |      0.0 |  -0.00725 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | 32.2 | 449.7 | 0.0 | 0.0 |   0.0 | 33.0 |       0.0 |  0.0 | 3.305745 | 1.652872 |      0.0 | 0.407671 |  5.511 |      0.0 |      16.5 |  0.0 |       0.0 |       0.0 | 0.009985 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.633171 |      0.0 |      0.0 | 0.009985 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004135 |      0.0 | -0.007974 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | 35.2 | 493.4 | 0.0 | 0.0 |   0.0 | 36.0 |       0.0 |  0.0 | 3.747947 | 1.873973 |      0.0 | 0.454685 |  6.012 |      0.0 |      18.0 |  0.0 |       0.0 |       0.0 | 0.009817 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.248669 |      0.0 |      0.0 | 0.009817 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004416 |      0.0 |  -0.00869 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | 37.1 | 261.5 | 0.0 | 0.0 |   0.0 | 37.9 |       0.0 |  0.0 | 2.050471 | 1.025236 |      0.0 |  0.24406 | 6.3293 |      0.0 |     18.95 |  0.0 |       0.0 |       0.0 | 0.009624 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  19.074984 |      0.0 |      0.0 | 0.009624 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004675 |      0.0 |   -0.0094 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | 31.2 | 363.6 | 0.0 | 0.0 |   0.0 | 32.0 |       0.0 |  0.0 | 2.650012 | 1.325006 |      0.0 | 0.312677 |  5.344 |      0.0 |      16.0 |  0.0 |       0.0 |       0.0 | 0.009537 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   18.83277 |      0.0 |      0.0 | 0.009537 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004914 |      0.0 | -0.010102 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | 24.3 | 446.2 | 0.0 | 0.0 |   0.0 | 25.1 |       0.0 |  0.0 | 2.959048 | 1.479524 |      0.0 | 0.344861 | 4.1917 |      0.0 |     12.55 |  0.0 |       0.0 |       0.0 | 0.009416 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.558493 |      0.0 |      0.0 | 0.009416 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005136 |      0.0 | -0.010798 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | 25.4 | 137.6 | 0.0 | 0.0 |  0.24 | 26.2 |       0.0 |  0.0 | 0.956604 | 0.478302 |     0.24 | 0.054764 | 4.3754 |      0.0 |      13.1 |  0.0 |       0.0 |       0.0 | 0.009279 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |   18.57445 |      0.0 |      0.0 | 0.009279 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005342 |      0.0 | -0.011487 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | 25.9 | 103.0 | 0.0 | 0.0 |   0.0 | 26.7 |       0.0 |  0.0 | 0.731933 | 0.365967 |      0.0 | 0.084172 | 4.4589 |      0.0 |     13.35 |  0.0 |       0.0 |       0.0 | 0.009287 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.560991 |      0.0 |      0.0 | 0.009287 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005534 |      0.0 | -0.012168 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | 23.7 |  63.7 | 0.0 | 0.0 |   0.0 | 24.5 |       0.0 |  0.0 | 0.454628 | 0.227314 |      0.0 | 0.052246 | 4.0915 |      0.0 |     12.25 |  0.0 |       0.0 |       0.0 |  0.00928 |      0.0 |      0.0 | 0.08 |       0.0 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  18.579465 |      0.0 |      0.0 |  0.00928 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005717 |      0.0 | -0.012843 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | 21.6 |  41.4 | 0.0 | 0.0 |  1.56 | 22.4 |      0.76 |  0.0 | 0.300394 | 0.150197 | 0.150197 |      0.0 | 3.7408 | 0.071276 | 11.413401 |  0.0 |      0.76 |   0.02131 |  0.00929 |      0.0 |      0.0 | 0.08 |   0.02131 |      0.0 |      0.0 | 0.649803 |  0.0 |  0.0 |  19.388865 |  0.02131 |      0.0 |  0.00929 |      0.0 |    -0.08 | 0.001996 |      0.0 | 0.005891 |      0.0 | -0.013511 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 | 21.2 |   7.9 | 0.0 | 0.0 |  6.72 | 22.0 |  6.569803 |  0.0 | 0.089558 | 0.044779 | 0.044779 |      0.0 |  3.674 | 0.605142 | 12.811802 |  0.0 |  6.569803 |  0.221019 | 0.009694 |      0.0 |      0.0 | 0.08 |  0.221019 | 0.016094 | 0.004471 | 0.755221 |  0.0 |  0.0 |  25.807955 | 0.221019 |      0.0 | 0.009694 |      0.0 |    -0.08 |   0.0242 |      0.0 | 0.006067 |      0.0 | -0.014173 |   0.0 | 0.004471 |
    | 2000-01-01 20:00:00 |  2.9 | 20.4 |   0.0 | 0.0 | 0.0 |  3.48 | 21.2 |  3.435221 |  0.0 |  0.03977 | 0.019885 | 0.019885 |      0.0 | 3.5404 | 0.304911 | 11.512906 |  0.0 |  3.435221 |  0.141928 | 0.012904 |      0.0 |      0.0 | 0.08 |  0.141928 | 0.043965 | 0.012212 | 0.780115 |  0.0 |  0.0 |  29.168344 | 0.141928 |      0.0 | 0.012904 |      0.0 |    -0.08 |  0.05247 |      0.0 | 0.006322 |      0.0 | -0.014828 |   0.0 | 0.012212 |
    | 2000-01-01 21:00:00 |  4.9 | 19.8 |   0.0 | 0.0 | 0.0 |  5.88 | 20.6 |  5.860115 |  0.0 | 0.039381 | 0.019691 | 0.019691 |      0.0 | 3.4402 | 0.505424 | 11.813245 |  0.0 |  5.860115 |   0.28445 | 0.014584 |      0.0 |      0.0 | 0.08 |   0.28445 | 0.073242 | 0.020345 | 0.780309 |  0.0 |  0.0 |  34.809425 |  0.28445 |      0.0 | 0.014584 |      0.0 |    -0.08 | 0.082034 |      0.0 | 0.006685 |      0.0 | -0.015476 |   0.0 | 0.020345 |
    | 2000-01-01 22:00:00 | 10.6 | 19.6 |   0.0 | 0.0 | 0.0 | 12.72 | 20.4 | 12.700309 |  0.0 | 0.039251 | 0.019626 | 0.019626 |      0.0 | 3.4068 | 1.084743 | 13.447732 |  0.0 | 12.700309 |   0.80163 | 0.017405 |      0.0 |      0.0 | 0.08 |   0.80163 | 0.158182 |  0.04394 | 0.780374 |  0.0 |  0.0 |  46.770699 |  0.80163 |      0.0 | 0.017405 |      0.0 |    -0.08 | 0.167162 |      0.0 | 0.007139 |      0.0 | -0.016118 |   0.0 |  0.04394 |
    | 2000-01-01 23:00:00 |  0.1 | 19.2 |   0.0 | 0.0 | 0.0 |  0.12 | 20.0 |  0.100374 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.008405 | 10.025165 |  0.0 |  0.100374 |  0.007364 | 0.023385 |      0.0 |      0.0 | 0.08 |  0.007364 | 0.198818 | 0.055227 | 0.780505 |  0.0 |  0.0 |  46.920325 | 0.007364 |      0.0 | 0.023385 |      0.0 |    -0.08 | 0.207785 |      0.0 | 0.007787 |      0.0 | -0.016754 |   0.0 | 0.055227 |
    | 2000-01-02 00:00:00 |  0.7 | 19.2 |   0.0 | 0.0 | 0.0 |  0.84 | 20.0 |  0.820505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.068706 | 10.205706 |  0.0 |  0.820505 |  0.060883 |  0.02346 |      0.0 |      0.0 | 0.08 |  0.060883 | 0.167633 | 0.046565 | 0.780505 |  0.0 |  0.0 |  47.736487 | 0.060883 |      0.0 |  0.02346 |      0.0 |    -0.08 | 0.176467 |      0.0 |  0.00855 |      0.0 | -0.017383 |   0.0 | 0.046565 |
    | 2000-01-02 01:00:00 |  3.0 | 19.2 |   0.0 | 0.0 | 0.0 |   3.6 | 20.0 |  3.580505 |  0.0 |  0.03899 | 0.019495 | 0.019495 |      0.0 |   3.34 | 0.299817 | 10.897656 |  0.0 |  3.580505 |   0.27871 | 0.023868 |      0.0 |      0.0 | 0.08 |   0.27871 | 0.167196 | 0.046443 | 0.780505 |  0.0 |  0.0 |  51.094414 |  0.27871 |      0.0 | 0.023868 |      0.0 |    -0.08 | 0.175916 |      0.0 | 0.009287 |      0.0 | -0.018006 |   0.0 | 0.046443 |
    | 2000-01-02 02:00:00 |  2.1 | 18.9 |   0.0 | 0.0 | 0.0 |  2.52 | 19.7 |  2.500505 |  0.0 | 0.038793 | 0.019396 | 0.019396 |      0.0 | 3.2899 | 0.206242 |  10.46749 |  0.0 |  2.500505 |  0.207202 | 0.025547 |      0.0 |      0.0 | 0.08 |  0.207202 | 0.179268 | 0.049797 | 0.780604 |  0.0 |  0.0 |   53.44217 | 0.207202 |      0.0 | 0.025547 |      0.0 |    -0.08 | 0.187852 |      0.0 | 0.010039 |      0.0 | -0.018623 |   0.0 | 0.049797 |
    | 2000-01-02 03:00:00 | 10.4 | 18.7 |   0.0 | 0.0 | 0.0 | 12.48 | 19.5 | 12.460604 |  0.0 | 0.038661 |  0.01933 |  0.01933 |      0.0 | 3.2565 | 1.017316 | 12.795857 |  0.0 | 12.460604 |  1.187264 | 0.026721 |      0.0 |      0.0 | 0.08 |  1.187264 | 0.278264 | 0.077295 |  0.78067 |  0.0 |  0.0 |  64.768788 | 1.157728 | 0.029537 | 0.026721 |      0.0 |    -0.08 |  0.28038 | 0.006293 | 0.010824 |      0.0 | -0.019234 |   0.0 | 0.077295 |
    | 2000-01-02 04:00:00 |  3.5 | 18.5 |   0.0 | 0.0 | 0.0 |   4.2 | 19.3 |   4.18067 |  0.0 | 0.038528 | 0.019264 | 0.019264 |      0.0 | 3.2231 |  0.33782 | 10.661437 |  0.0 |   4.18067 |  0.457075 | 0.032384 |      0.0 |      0.0 | 0.08 |  0.457075 | 0.374843 | 0.104123 | 0.780736 |  0.0 |  0.0 |  68.539999 | 0.457075 |      0.0 | 0.032384 |      0.0 |    -0.08 | 0.373797 | 0.009146 | 0.011739 |      0.0 | -0.019839 |   0.0 | 0.104123 |
    | 2000-01-02 05:00:00 |  3.4 | 18.3 |   0.0 | 0.0 | 0.0 |  4.08 | 19.1 |  4.060736 |  0.0 | 0.038396 | 0.019198 | 0.019198 |      0.0 | 3.1897 | 0.324728 | 10.522241 |  0.0 |  4.060736 |  0.472944 |  0.03427 |      0.0 |      0.0 | 0.08 |  0.472944 | 0.388281 | 0.107856 | 0.780802 |  0.0 |  0.0 |   72.17352 | 0.472944 |      0.0 |  0.03427 |      0.0 |    -0.08 | 0.390379 | 0.005547 | 0.012792 |      0.0 | -0.020437 |   0.0 | 0.107856 |
    | 2000-01-02 06:00:00 |  1.2 | 18.5 |   6.1 | 0.0 | 0.0 |  1.44 | 19.3 |  1.420802 |  0.0 | 0.074919 |  0.03746 |  0.03746 |      0.0 | 3.2231 | 0.114808 |  9.993737 |  0.0 |  1.420802 |  0.172327 | 0.036087 |      0.0 |      0.0 | 0.08 |  0.172327 | 0.373411 | 0.103725 |  0.76254 |  0.0 |  0.0 |  73.465909 | 0.172327 |      0.0 | 0.036087 |      0.0 |    -0.08 | 0.377192 | 0.003364 | 0.013884 |      0.0 |  -0.02103 |   0.0 | 0.103725 |
    | 2000-01-02 07:00:00 |  0.1 | 18.8 |  77.9 | 0.0 | 0.0 |  0.12 | 19.6 |   0.08254 |  0.0 | 0.505845 | 0.252922 | 0.252922 |      0.0 | 3.2732 | 0.006773 |   9.82028 |  0.0 |   0.08254 |  0.010126 | 0.036733 |      0.0 |      0.0 | 0.08 |  0.010126 | 0.320272 | 0.088965 | 0.547078 |  0.0 |  0.0 |  73.581591 | 0.010126 |      0.0 | 0.036733 |      0.0 |    -0.08 | 0.324865 | 0.002041 | 0.014983 |      0.0 | -0.021617 |   0.0 | 0.088965 |
    | 2000-01-02 08:00:00 |  0.0 | 18.8 | 196.7 | 0.0 | 0.0 |   0.0 | 19.6 |       0.0 |  0.0 | 1.218215 | 0.609108 | 0.547078 |  0.04555 | 3.2732 |      0.0 |       9.8 |  0.0 |       0.0 |       0.0 | 0.036791 |      0.0 |      0.0 | 0.08 |       0.0 |  0.26195 | 0.072764 |      0.0 |  0.0 |  0.0 |  73.579249 |      0.0 |      0.0 | 0.036791 |      0.0 |    -0.08 | 0.266864 | 0.001238 | 0.016045 |      0.0 | -0.022198 |   0.0 | 0.072764 |
    | 2000-01-02 09:00:00 |  0.0 | 19.0 | 121.9 | 0.0 | 0.0 |   0.0 | 19.8 |       0.0 |  0.0 | 0.772303 | 0.386152 |      0.0 | 0.283556 | 3.3066 |      0.0 |       9.9 |  0.0 |       0.0 |       0.0 |  0.03679 |      0.0 |      0.0 | 0.08 |       0.0 | 0.213525 | 0.059313 |      0.0 |  0.0 |  0.0 |  73.338903 |      0.0 |      0.0 |  0.03679 |      0.0 |    -0.08 |  0.21849 | 0.000751 | 0.017057 |      0.0 | -0.022773 |   0.0 | 0.059313 |
    | 2000-01-02 10:00:00 |  0.4 | 19.2 | 156.6 | 0.0 | 0.0 |  0.48 | 20.0 |       0.0 |  0.0 | 0.984401 |   0.4922 |     0.48 | 0.008941 |   3.34 |      0.0 |      10.0 |  0.0 |       0.0 |       0.0 | 0.036669 |      0.0 |      0.0 | 0.08 |       0.0 | 0.174014 | 0.048337 |      0.0 |  0.0 |  0.0 |  73.373293 |      0.0 |      0.0 | 0.036669 |      0.0 |    -0.08 | 0.178885 | 0.000455 | 0.018016 |      0.0 | -0.023342 |   0.0 | 0.048337 |
    | 2000-01-02 11:00:00 |  0.1 | 19.3 | 404.7 | 0.0 | 0.0 |  0.12 | 20.1 |       0.0 |  0.0 | 2.486378 | 1.243189 |     0.12 | 0.823379 | 3.3567 |      0.0 |     10.05 |  0.0 |       0.0 |       0.0 | 0.036687 |      0.0 |      0.0 | 0.08 |       0.0 | 0.141755 | 0.039376 |      0.0 |  0.0 |  0.0 |  72.593227 |      0.0 |      0.0 | 0.036687 |      0.0 |    -0.08 | 0.146458 | 0.000276 | 0.018927 |      0.0 | -0.023906 |   0.0 | 0.039376 |
    | 2000-01-02 12:00:00 |  3.6 | 19.0 | 217.9 | 0.0 | 0.0 |  4.32 | 19.8 |      3.52 |  0.0 | 1.349913 | 0.674957 | 0.674957 |      0.0 | 3.3066 | 0.291803 | 10.773662 |  0.0 |      3.52 |  0.436285 | 0.036297 |      0.0 |      0.0 | 0.08 |  0.436285 | 0.156256 | 0.043405 | 0.125043 |  0.0 |  0.0 |  75.720646 | 0.436285 |      0.0 | 0.036297 |      0.0 |    -0.08 |  0.16077 | 0.000168 | 0.019783 |      0.0 | -0.024464 |   0.0 | 0.043405 |
    | 2000-01-02 13:00:00 |  5.9 | 18.8 | 582.0 | 0.0 | 0.0 |  7.08 | 19.6 |  6.405043 |  0.0 | 3.528622 | 1.764311 |      0.8 | 0.720288 | 3.2732 | 0.525606 | 11.373671 |  0.0 |  6.405043 |  0.850271 |  0.03786 |      0.0 |      0.0 | 0.08 |  0.850271 | 0.245195 |  0.06811 |      0.0 |  0.0 |  0.0 |   80.59727 | 0.850271 |      0.0 |  0.03786 |      0.0 |    -0.08 | 0.249483 | 0.000102 | 0.020627 |      0.0 | -0.025017 |   0.0 |  0.06811 |
    | 2000-01-02 14:00:00 |  1.1 | 18.7 | 263.9 | 0.0 | 0.0 |  1.32 | 19.5 |      0.52 |  0.0 |  1.61841 | 0.809205 |      0.8 | 0.007124 | 3.2565 | 0.042454 |  9.877108 |  0.0 |      0.52 |  0.071497 | 0.040299 |      0.0 |      0.0 | 0.08 |  0.071497 | 0.281478 | 0.078188 |      0.0 |  0.0 |  0.0 |   81.07835 | 0.071497 |      0.0 | 0.040299 |      0.0 |    -0.08 | 0.285453 | 0.000062 | 0.021527 |      0.0 | -0.025564 |   0.0 | 0.078188 |
    | 2000-01-02 15:00:00 | 20.7 | 17.8 | 136.8 | 0.0 | 0.0 | 24.84 | 18.6 |     24.04 |  0.0 | 0.844303 | 0.422151 | 0.422151 |      0.0 | 3.1062 | 1.872102 | 14.905097 |  0.0 |     24.04 |  3.847472 | 0.040539 |      0.0 |      0.0 | 0.08 |  3.847472 | 0.848321 | 0.235645 | 0.377849 |  0.0 |  0.0 | 101.310339 | 1.740089 | 2.107383 | 0.040539 |      0.0 |    -0.08 | 0.402939 | 0.449039 | 0.022449 |      0.0 | -0.026105 |   0.0 | 0.235645 |
    | 2000-01-02 16:00:00 | 37.9 | 17.4 | 146.6 | 0.0 | 0.0 | 45.48 | 18.2 | 45.057849 |  0.0 | 0.895703 | 0.447851 | 0.447851 |      0.0 | 3.0394 | 3.433397 | 19.379632 |  0.0 | 45.057849 | 10.314783 | 0.050655 |  0.00135 | 0.456552 |  0.0 | 10.314783 | 3.105022 | 0.862506 | 0.352149 |  0.0 |  0.0 | 135.544849 | 1.903052 | 8.411731 | 0.050655 |  0.00135 | 0.456552 | 0.660585 | 2.444759 |  0.02358 | 0.000065 | -0.023968 |   0.0 | 0.862506 |
    | 2000-01-02 17:00:00 |  8.2 | 17.3 | 190.6 | 0.0 | 0.0 |  9.84 | 18.1 |  9.392149 |  0.0 | 1.151139 |  0.57557 |  0.57557 |      0.0 | 3.0227 | 0.711747 |  11.18098 |  0.0 |  9.392149 |  2.700025 | 0.067772 | 0.190725 | 0.627724 |  0.0 |  2.700025 |  4.10496 | 1.140267 |  0.22443 |  0.0 |  0.0 | 141.350751 | 1.629633 | 1.070392 | 0.067772 | 0.190725 | 0.627724 | 0.860199 | 3.228424 | 0.025321 | 0.009348 | -0.018334 |   0.0 | 1.140267 |
    | 2000-01-02 18:00:00 |  3.6 | 16.8 | 103.5 | 0.0 | 0.0 |  4.32 | 17.6 |   3.74443 |  0.0 | 0.636581 |  0.31829 |  0.31829 |      0.0 | 2.9392 | 0.275918 |  9.626103 |  0.0 |   3.74443 |  1.124141 | 0.070675 | 0.239314 | 0.656754 |  0.0 |  1.124141 | 3.149877 | 0.874966 |  0.48171 |  0.0 |  0.0 | 143.004297 | 1.110432 | 0.013709 | 0.070675 | 0.239314 | 0.656754 | 0.951049 | 2.154166 | 0.027463 | 0.028959 |  -0.01176 |   0.0 | 0.874966 |
    | 2000-01-02 19:00:00 |  7.5 | 16.5 |  13.8 | 0.0 | 0.0 |   9.0 | 17.3 |   8.68171 |  0.0 | 0.116642 | 0.058321 | 0.058321 |      0.0 | 2.8891 |  0.62883 | 10.532726 |  0.0 |   8.68171 |  2.710266 | 0.071502 | 0.253811 | 0.665021 |  0.0 |  2.710266 | 2.641882 | 0.733856 | 0.741679 |  0.0 |  0.0 | 147.985407 | 1.631032 | 1.079233 | 0.071502 | 0.253811 | 0.665021 | 1.028696 | 1.538984 | 0.029591 | 0.049678 | -0.005067 |   0.0 | 0.733856 |
    | 2000-01-02 20:00:00 | 18.5 | 16.3 |   0.0 | 0.0 | 0.0 |  22.2 | 17.1 | 22.141679 |  0.0 | 0.037049 | 0.018524 | 0.018524 |      0.0 | 2.8557 | 1.585218 |  13.29616 |  0.0 | 22.141679 |  7.724215 | 0.073993 | 0.299162 | 0.689927 |  0.0 |  7.724215 | 3.640366 | 1.011213 | 0.781476 |  0.0 |  0.0 | 161.339789 | 1.870537 | 5.853678 | 0.073993 | 0.299162 | 0.689927 | 1.160312 | 2.375336 | 0.031696 | 0.071298 |  0.001724 |   0.0 | 1.011213 |
    | 2000-01-02 21:00:00 | 15.4 | 16.2 |   0.0 | 0.0 | 0.0 | 18.48 | 17.0 | 18.461476 |  0.0 |  0.03698 |  0.01849 |  0.01849 |      0.0 |  2.839 | 1.314007 | 12.434152 |  0.0 | 18.461476 |  7.433494 |  0.08067 |  0.43237 | 0.756699 |  0.0 |  7.433494 | 5.113928 | 1.420535 |  0.78151 |  0.0 |  0.0 | 171.098031 | 1.865474 |  5.56802 |  0.08067 |  0.43237 | 0.756699 |  1.28858 | 3.683094 | 0.033923 | 0.099426 |  0.008905 |   0.0 | 1.420535 |
    | 2000-01-02 22:00:00 |  6.3 | 15.5 |   0.0 | 0.0 | 0.0 |  7.56 | 16.3 |   7.54151 |  0.0 | 0.036499 |  0.01825 |  0.01825 |      0.0 | 2.7221 | 0.514669 |  9.690926 |  0.0 |   7.54151 |  3.297279 | 0.085549 | 0.539546 |  0.80549 |  0.0 |  3.297279 | 5.145996 | 1.429443 |  0.78175 |  0.0 |  0.0 | 173.911677 |  1.69672 | 1.600559 | 0.085549 | 0.539546 |  0.80549 | 1.377348 | 3.579442 | 0.036323 | 0.136294 |  0.016588 |   0.0 | 1.429443 |
    | 2000-01-02 23:00:00 |  1.9 | 14.6 |   0.0 | 0.0 | 0.0 |  2.28 | 15.4 |   2.26175 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 |  0.14583 |  8.136618 |  0.0 |   2.26175 |  1.006677 | 0.086956 | 0.571889 | 0.819558 |  0.0 |  1.006677 | 4.069916 | 1.130532 | 0.782063 |  0.0 |  0.0 | 173.688346 | 1.006633 | 0.000044 | 0.086956 | 0.571889 | 0.819558 | 1.370611 | 2.459805 | 0.038758 | 0.176233 |  0.024508 |   0.0 | 1.130532 |
    | 2000-01-03 00:00:00 |  4.9 | 14.7 |   0.0 | 0.0 | 0.0 |  5.88 | 15.5 |  5.862063 |  0.0 | 0.035943 | 0.017972 | 0.017972 |      0.0 | 2.5885 | 0.380421 |  8.888985 |  0.0 |  5.862063 |  2.637943 | 0.086844 | 0.569299 | 0.818442 |  0.0 |  2.637943 | 3.358086 | 0.932802 | 0.782028 |  0.0 |  0.0 | 175.437882 | 1.620917 | 1.017026 | 0.086844 | 0.569299 | 0.818442 | 1.362163 | 1.708644 | 0.041106 |  0.21376 |  0.032414 |   0.0 | 0.932802 |
    | 2000-01-03 01:00:00 |  2.7 | 14.6 |   0.0 | 0.0 | 0.0 |  3.24 | 15.4 |  3.222028 |  0.0 | 0.035873 | 0.017937 | 0.017937 |      0.0 | 2.5718 | 0.207746 |  8.321993 |  0.0 |  3.222028 |   1.47069 | 0.087719 | 0.589694 | 0.827189 |  0.0 |   1.47069 | 2.965027 | 0.823619 | 0.782063 |  0.0 |  0.0 | 175.684618 | 1.320047 | 0.150643 | 0.087719 | 0.589694 | 0.827189 |  1.38089 | 1.251921 | 0.043358 |  0.24858 |  0.040278 |   0.0 | 0.823619 |
    | 2000-01-03 02:00:00 |  0.5 | 14.1 |   0.0 | 0.0 | 0.0 |   0.6 | 14.9 |  0.582063 |  0.0 | 0.035522 | 0.017761 | 0.017761 |      0.0 | 2.4883 | 0.036311 |  7.558716 |  0.0 |  0.582063 |  0.263876 | 0.087842 | 0.592589 | 0.828423 |  0.0 |  0.263876 | 2.432273 | 0.675631 | 0.782239 |  0.0 |  0.0 | 174.493951 | 0.263876 |      0.0 | 0.087842 | 0.592589 | 0.828423 | 1.270946 | 0.786505 | 0.045525 | 0.281182 |  0.048114 |   0.0 | 0.675631 |
    | 2000-01-03 03:00:00 |  0.2 | 14.3 |   0.0 | 0.0 | 0.0 |  0.24 | 15.1 |  0.222239 |  0.0 | 0.035663 | 0.017831 | 0.017831 |      0.0 | 2.5217 |  0.01405 |  7.592066 |  0.0 |  0.222239 |  0.098934 | 0.087247 | 0.578661 |  0.82247 |  0.0 |  0.098934 | 1.963553 | 0.545431 | 0.782169 |  0.0 |  0.0 | 173.128878 | 0.098934 |      0.0 | 0.087247 | 0.578661 |  0.82247 | 1.072948 |  0.47704 | 0.047574 | 0.310142 |  0.055849 |   0.0 | 0.545431 |
    | 2000-01-03 04:00:00 |  0.5 | 14.9 |   0.0 | 0.0 | 0.0 |   0.6 | 15.7 |  0.582169 |  0.0 | 0.036083 | 0.018041 | 0.018041 |      0.0 | 2.6219 | 0.038268 |  7.964573 |  0.0 |  0.582169 |  0.254662 | 0.086564 | 0.562828 | 0.815644 |  0.0 |  0.254662 | 1.648177 | 0.457827 | 0.781959 |  0.0 |  0.0 | 171.991347 | 0.254662 |      0.0 | 0.086564 | 0.562828 | 0.815644 | 0.910974 | 0.289339 | 0.049492 | 0.334929 |  0.063443 |   0.0 | 0.457827 |
    | 2000-01-03 05:00:00 |  2.4 | 15.7 |   0.0 | 0.0 | 0.0 |  2.88 | 16.5 |  2.861959 |  0.0 | 0.036637 | 0.018319 | 0.018319 |      0.0 | 2.7555 |  0.19771 |  8.841947 |  0.0 |  2.861959 |  1.243591 | 0.085996 | 0.549747 | 0.809957 |  0.0 |  1.243591 | 1.543981 | 0.428884 | 0.781681 |  0.0 |  0.0 | 172.164015 | 1.195877 | 0.047714 | 0.085996 | 0.549747 | 0.809957 | 0.880153 | 0.185659 | 0.051286 | 0.355984 |  0.070899 |   0.0 | 0.428884 |
    | 2000-01-03 06:00:00 |  0.4 | 16.0 |   4.4 | 0.0 | 0.0 |  0.48 | 16.8 |  0.461681 |  0.0 | 0.061945 | 0.030972 | 0.030972 |      0.0 | 2.8056 | 0.032474 |  8.497227 |  0.0 |  0.461681 |  0.199221 | 0.086082 | 0.551726 |  0.81082 |  0.0 |  0.199221 | 1.471016 | 0.408616 | 0.769028 |  0.0 |  0.0 | 170.977847 | 0.199221 |      0.0 | 0.086082 | 0.551726 |  0.81082 | 0.844044 | 0.121216 | 0.052981 | 0.374519 |  0.078257 |   0.0 | 0.408616 |
    | 2000-01-03 07:00:00 |  0.2 | 16.7 |  26.1 | 0.0 | 0.0 |  0.24 | 17.5 |  0.209028 |  0.0 | 0.188148 | 0.094074 | 0.094074 |      0.0 | 2.9225 | 0.015315 |  8.795854 |  0.0 |  0.209028 |  0.088682 | 0.085489 | 0.538179 | 0.804889 |  0.0 |  0.088682 |  1.32115 | 0.366986 | 0.705926 |  0.0 |  0.0 | 169.669636 | 0.088682 |      0.0 | 0.085489 | 0.538179 | 0.804889 | 0.716805 | 0.073521 | 0.054581 | 0.390727 |  0.085517 |   0.0 | 0.366986 |
    | 2000-01-03 08:00:00 |  0.0 | 17.1 |  74.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 |  0.46949 | 0.234745 | 0.234745 |      0.0 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 | 0.084835 | 0.523369 | 0.798348 |  0.0 |       0.0 | 1.191989 | 0.331108 | 0.471181 |  0.0 |  0.0 | 168.263084 |      0.0 |      0.0 | 0.084835 | 0.523369 | 0.798348 |  0.59464 | 0.044593 | 0.056072 | 0.404042 |  0.092642 |   0.0 | 0.331108 |
    | 2000-01-03 09:00:00 |  0.0 | 16.2 | 287.1 | 0.0 | 0.0 |   0.0 | 17.0 |       0.0 |  0.0 | 1.680918 | 0.840459 | 0.471181 | 0.363261 |  2.839 |      0.0 |       8.5 |  0.0 |       0.0 |       0.0 | 0.084132 |   0.5076 | 0.791315 |  0.0 |       0.0 | 1.085618 | 0.301561 |      0.0 |  0.0 |  0.0 | 166.516777 |      0.0 |      0.0 | 0.084132 |   0.5076 | 0.791315 |  0.48685 | 0.027047 | 0.057458 | 0.414635 |  0.099629 |   0.0 | 0.301561 |
    | 2000-01-03 10:00:00 |  0.3 | 15.9 | 299.8 | 0.0 | 0.0 |  0.36 | 16.7 |       0.0 |  0.0 | 1.743886 | 0.871943 |     0.36 | 0.502927 | 2.7889 |      0.0 |      8.35 |  0.0 |       0.0 |       0.0 | 0.083258 | 0.488247 | 0.782584 |  0.0 |       0.0 | 1.002754 | 0.278543 |      0.0 |  0.0 |  0.0 | 164.659761 |      0.0 |      0.0 | 0.083258 | 0.488247 | 0.782584 | 0.398599 | 0.016405 | 0.058737 | 0.422546 |  0.106468 |   0.0 | 0.278543 |
    | 2000-01-03 11:00:00 |  2.6 | 16.3 | 363.5 | 0.0 | 0.0 |  3.12 | 17.1 |      2.32 |  0.0 | 2.122301 | 1.061151 |      0.8 |  0.25617 | 2.8557 | 0.166099 |  9.047302 |  0.0 |      2.32 |  0.914244 |  0.08233 | 0.467944 | 0.773299 |  0.0 |  0.914244 | 1.022793 | 0.284109 |      0.0 |  0.0 |  0.0 | 164.485774 | 0.914244 |      0.0 |  0.08233 | 0.467944 | 0.773299 | 0.411968 |  0.00995 |  0.05991 | 0.427816 |  0.113149 |   0.0 | 0.284109 |
    | 2000-01-03 12:00:00 |  0.7 | 16.3 | 368.4 | 0.0 | 0.0 |  0.84 | 17.1 |      0.04 |  0.0 | 2.150411 | 1.075205 |      0.8 | 0.269918 | 2.8557 | 0.002864 |  8.558574 |  0.0 |      0.04 |  0.015591 | 0.082243 | 0.466056 | 0.772429 |  0.0 |  0.015591 | 1.037145 | 0.288096 |      0.0 |  0.0 |  0.0 | 162.919538 | 0.015591 |      0.0 | 0.082243 | 0.466056 | 0.772429 | 0.418853 | 0.006035 | 0.061001 | 0.431543 |  0.119713 |   0.0 | 0.288096 |
    | 2000-01-03 13:00:00 |  0.3 | 16.4 | 317.8 | 0.0 | 0.0 |  0.36 | 17.2 |       0.0 |  0.0 | 1.863566 | 0.931783 |     0.36 | 0.560059 | 2.8724 |      0.0 |       8.6 |  0.0 |       0.0 |       0.0 |  0.08146 |  0.44918 | 0.764598 |  0.0 |       0.0 | 0.970152 | 0.269487 |      0.0 |  0.0 |  0.0 | 161.064241 |      0.0 |      0.0 |  0.08146 |  0.44918 | 0.764598 | 0.344294 |  0.00366 | 0.062018 | 0.434011 |  0.126169 |   0.0 | 0.269487 |
    | 2000-01-03 14:00:00 |  0.3 | 16.5 | 534.7 | 0.0 | 0.0 |  0.36 | 17.3 |       0.0 |  0.0 | 3.115838 | 1.557919 |     0.36 | 1.171448 | 2.8891 |      0.0 |      8.65 |  0.0 |       0.0 |       0.0 | 0.080532 |  0.42946 | 0.755321 |  0.0 |       0.0 | 0.914023 | 0.253895 |      0.0 |  0.0 |  0.0 | 158.627479 |      0.0 |      0.0 | 0.080532 |  0.42946 | 0.755321 | 0.281884 |  0.00222 | 0.062944 | 0.434501 |  0.132475 |   0.0 | 0.253895 |
    | 2000-01-03 15:00:00 |  0.0 | 18.4 | 319.4 | 0.0 | 0.0 |   0.0 | 19.2 |       0.0 |  0.0 | 1.940627 | 0.970313 |      0.0 | 0.946734 | 3.2064 |      0.0 |       9.6 |  0.0 |       0.0 |       0.0 | 0.079314 | 0.404012 | 0.743137 |  0.0 |       0.0 | 0.867307 | 0.240919 |      0.0 |  0.0 |  0.0 | 156.454282 |      0.0 |      0.0 | 0.079314 | 0.404012 | 0.743137 | 0.230787 | 0.001347 | 0.063771 |  0.43279 |  0.138612 |   0.0 | 0.240919 |
    | 2000-01-03 16:00:00 |  0.0 | 18.3 | 350.6 | 0.0 | 0.0 |   0.0 | 19.1 |       0.0 |  0.0 | 2.122768 | 1.061384 |      0.0 | 1.033386 | 3.1897 |      0.0 |      9.55 |  0.0 |       0.0 |       0.0 | 0.078227 | 0.381758 | 0.732271 |  0.0 |       0.0 | 0.827819 |  0.22995 |      0.0 |  0.0 |  0.0 | 154.228639 |      0.0 |      0.0 | 0.078227 | 0.381758 | 0.732271 | 0.188952 | 0.000817 | 0.064503 | 0.428975 |  0.144573 |   0.0 |  0.22995 |
    | 2000-01-03 17:00:00 |  0.0 | 18.1 | 215.4 | 0.0 | 0.0 |   0.0 | 18.9 |       0.0 |  0.0 | 1.314414 | 0.657207 |      0.0 | 0.638397 | 3.1563 |      0.0 |      9.45 |  0.0 |       0.0 |       0.0 | 0.077114 | 0.359406 | 0.721143 |  0.0 |       0.0 | 0.794106 | 0.220585 |      0.0 |  0.0 |  0.0 | 152.432578 |      0.0 |      0.0 | 0.077114 | 0.359406 | 0.721143 | 0.154701 | 0.000495 | 0.065145 |   0.4234 |  0.150365 |   0.0 | 0.220585 |
    | 2000-01-03 18:00:00 |  0.0 | 16.7 |  97.8 | 0.0 | 0.0 |   0.0 | 17.5 |       0.0 |  0.0 | 0.602486 | 0.301243 |      0.0 | 0.292049 | 2.9225 |      0.0 |      8.75 |  0.0 |       0.0 |       0.0 | 0.076216 | 0.341699 | 0.712163 |  0.0 |       0.0 | 0.765119 | 0.212533 |      0.0 |  0.0 |  0.0 | 151.010451 |      0.0 |      0.0 | 0.076216 | 0.341699 | 0.712163 | 0.126659 |   0.0003 | 0.065706 | 0.416454 |     0.156 |   0.0 | 0.212533 |
    | 2000-01-03 19:00:00 |  0.0 | 15.2 |  13.1 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 | 0.109905 | 0.054952 |      0.0 | 0.053189 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 | 0.075505 | 0.327892 | 0.705052 |  0.0 |       0.0 | 0.740253 | 0.205626 |      0.0 |  0.0 |  0.0 | 149.848812 |      0.0 |      0.0 | 0.075505 | 0.327892 | 0.705052 | 0.103699 | 0.000182 | 0.066201 | 0.408672 |  0.161498 |   0.0 | 0.205626 |
    | 2000-01-03 20:00:00 |  0.0 | 13.4 |   0.0 | 0.0 | 0.0 |   0.0 | 14.2 |       0.0 |  0.0 | 0.035026 | 0.017513 |      0.0 | 0.016928 | 2.3714 |      0.0 |       7.1 |  0.0 |       0.0 |       0.0 | 0.074924 | 0.316756 | 0.699244 |  0.0 |       0.0 | 0.718977 | 0.199716 |      0.0 |  0.0 |  0.0 | 148.740959 |      0.0 |      0.0 | 0.074924 | 0.316756 | 0.699244 | 0.084902 | 0.000111 | 0.066641 | 0.400446 |  0.166878 |   0.0 | 0.199716 |
    | 2000-01-03 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 | 0.016559 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.07437 | 0.306255 | 0.693705 |  0.0 |       0.0 | 0.700731 | 0.194648 |      0.0 |  0.0 |  0.0 | 147.650069 |      0.0 |      0.0 |  0.07437 | 0.306255 | 0.693705 | 0.069512 | 0.000067 | 0.067031 | 0.391974 |  0.172147 |   0.0 | 0.194648 |
    | 2000-01-03 22:00:00 |  0.0 | 11.6 |   0.0 | 0.0 | 0.0 |   0.0 | 12.4 |       0.0 |  0.0 | 0.033727 | 0.016863 |      0.0 | 0.016256 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 | 0.073825 | 0.296031 |  0.68825 |  0.0 |       0.0 |  0.68496 | 0.190267 |      0.0 |  0.0 |  0.0 | 146.575706 |      0.0 |      0.0 | 0.073825 | 0.296031 |  0.68825 | 0.056911 | 0.000041 | 0.067376 | 0.383322 |  0.177309 |   0.0 | 0.190267 |
    | 2000-01-03 23:00:00 |  0.0 | 11.0 |   0.0 | 0.0 | 0.0 |   0.0 | 11.8 |       0.0 |  0.0 | 0.033286 | 0.016643 |      0.0 | 0.016022 | 1.9706 |      0.0 |       5.9 |  0.0 |       0.0 |       0.0 | 0.073288 | 0.286076 | 0.682879 |  0.0 |       0.0 | 0.671197 | 0.186444 |      0.0 |  0.0 |  0.0 | 145.517441 |      0.0 |      0.0 | 0.073288 | 0.286076 | 0.682879 | 0.046595 | 0.000025 | 0.067677 | 0.374534 |  0.182367 |   0.0 | 0.186444 |
    | 2000-01-04 00:00:00 |  0.0 | 10.5 |   0.0 | 0.0 | 0.0 |   0.0 | 11.3 |       0.0 |  0.0 | 0.032916 | 0.016458 |      0.0 | 0.015822 | 1.8871 |      0.0 |      5.65 |  0.0 |       0.0 |       0.0 | 0.072759 | 0.276382 | 0.677587 |  0.0 |       0.0 | 0.659069 | 0.183075 |      0.0 |  0.0 |  0.0 | 144.474892 |      0.0 |      0.0 | 0.072759 | 0.276382 | 0.677587 | 0.038149 | 0.000015 | 0.067938 | 0.365647 |   0.18732 |   0.0 | 0.183075 |
    | 2000-01-04 01:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.016224 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 | 0.072237 | 0.266941 | 0.672374 |  0.0 |       0.0 | 0.648271 | 0.180075 |      0.0 |  0.0 |  0.0 | 143.447115 |      0.0 |      0.0 | 0.072237 | 0.266941 | 0.672374 | 0.031234 | 0.000009 |  0.06816 | 0.356696 |  0.192173 |   0.0 | 0.180075 |
    | 2000-01-04 02:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.016271 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.071724 | 0.257741 | 0.667236 |  0.0 |       0.0 | 0.638558 | 0.177377 |      0.0 |  0.0 |  0.0 | 142.434143 |      0.0 |      0.0 | 0.071724 | 0.257741 | 0.667236 | 0.025572 | 0.000006 | 0.068346 | 0.347709 |  0.196925 |   0.0 | 0.177377 |
    | 2000-01-04 03:00:00 |  1.3 | 11.2 |   0.0 | 0.0 | 0.0 |  1.56 | 12.0 |      0.76 |  0.0 | 0.033433 | 0.016717 | 0.016717 |      0.0 |  2.004 | 0.038184 |  6.114322 |  0.0 |      0.76 |  0.228252 | 0.071217 |  0.24878 | 0.662171 |  0.0 |  0.228252 | 0.651109 | 0.180864 | 0.783283 |  0.0 |  0.0 | 141.983723 | 0.228252 |      0.0 | 0.071217 |  0.24878 | 0.662171 | 0.042313 | 0.000003 | 0.068499 | 0.338714 |   0.20158 |   0.0 | 0.180864 |
    | 2000-01-04 04:00:00 |  0.0 | 11.1 |   0.0 | 0.0 | 0.0 |   0.0 | 11.9 |       0.0 |  0.0 |  0.03336 |  0.01668 |  0.01668 |      0.0 | 1.9873 |      0.0 |      5.95 |  0.0 |       0.0 |       0.0 | 0.070992 |  0.24483 | 0.659919 |  0.0 |       0.0 | 0.659385 | 0.183163 | 0.766603 |  0.0 |  0.0 | 141.007983 |      0.0 |      0.0 | 0.070992 |  0.24483 | 0.659919 | 0.054642 | 0.000002 | 0.068626 | 0.329965 |  0.206151 |   0.0 | 0.183163 |
    | 2000-01-04 05:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 | 0.016973 |      0.0 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.070504 | 0.236344 |  0.65504 |  0.0 |       0.0 | 0.645562 | 0.179323 |  0.74963 |  0.0 |  0.0 | 140.046095 |      0.0 |      0.0 | 0.070504 | 0.236344 |  0.65504 | 0.044737 | 0.000001 | 0.068729 | 0.321453 |  0.210642 |   0.0 | 0.179323 |
    | 2000-01-04 06:00:00 |  0.0 | 12.2 |  17.0 | 0.0 | 0.0 |   0.0 | 13.0 |       0.0 |  0.0 | 0.124091 | 0.062046 | 0.062046 |      0.0 |  2.171 |      0.0 |       6.5 |  0.0 |       0.0 |       0.0 | 0.070023 | 0.228078 |  0.65023 |  0.0 |       0.0 | 0.633425 | 0.175951 | 0.687585 |  0.0 |  0.0 | 139.097764 |      0.0 |      0.0 | 0.070023 | 0.228078 |  0.65023 | 0.036627 | 0.000001 | 0.068804 | 0.312954 |   0.21504 |   0.0 | 0.175951 |
    | 2000-01-04 07:00:00 |  0.7 | 11.8 |  99.7 | 0.0 | 0.0 |  0.84 | 12.6 |  0.727585 |  0.0 | 0.556783 | 0.278392 | 0.278392 |      0.0 | 2.1042 | 0.038383 |  6.414918 |  0.0 |  0.727585 |  0.210201 | 0.069549 | 0.220024 | 0.645489 |  0.0 |  0.210201 |  0.64236 | 0.178433 | 0.521608 |  0.0 |  0.0 | 138.680086 | 0.210201 |      0.0 | 0.069549 | 0.220024 | 0.645489 | 0.049674 |      0.0 | 0.068852 | 0.304487 |  0.219347 |   0.0 | 0.178433 |
    | 2000-01-04 08:00:00 |  0.4 | 11.4 | 239.4 | 0.0 | 0.0 |  0.48 | 12.2 |  0.201608 |  0.0 | 1.278351 | 0.639175 | 0.639175 |      0.0 | 2.0374 | 0.010298 |  6.130832 |  0.0 |  0.201608 |  0.057838 |  0.06934 | 0.216508 |   0.6434 |  0.0 |  0.057838 | 0.653239 | 0.181455 | 0.160825 |  0.0 |  0.0 | 137.894608 | 0.057838 |      0.0 |  0.06934 | 0.216508 |   0.6434 | 0.064503 |      0.0 |  0.06888 | 0.296279 |  0.223576 |   0.0 | 0.181455 |
    | 2000-01-04 09:00:00 |  0.1 | 11.6 | 391.2 | 0.0 | 0.0 |  0.12 | 12.4 |       0.0 |  0.0 |  2.07666 |  1.03833 | 0.280825 | 0.720177 | 2.0708 |      0.0 |       6.2 |  0.0 |       0.0 |       0.0 | 0.068947 | 0.209946 | 0.639473 |  0.0 |       0.0 | 0.642876 | 0.178577 |      0.0 |  0.0 |  0.0 | 136.256064 |      0.0 |      0.0 | 0.068947 | 0.209946 | 0.639473 | 0.057878 |      0.0 | 0.068893 | 0.288371 |  0.227734 |   0.0 | 0.178577 |
    | 2000-01-04 10:00:00 |  0.4 | 13.0 | 525.6 | 0.0 | 0.0 |  0.48 | 13.8 |       0.0 |  0.0 | 2.862014 | 1.431007 |     0.48 | 0.901722 | 2.3046 |      0.0 |       6.9 |  0.0 |       0.0 |       0.0 | 0.068128 | 0.196478 |  0.63128 |  0.0 |       0.0 | 0.628309 |  0.17453 |      0.0 |  0.0 |  0.0 | 134.458457 |      0.0 |      0.0 | 0.068128 | 0.196478 |  0.63128 | 0.047387 |      0.0 | 0.068876 | 0.280256 |   0.23179 |   0.0 |  0.17453 |
    | 2000-01-04 11:00:00 |  0.0 | 17.1 | 570.2 | 0.0 | 0.0 |   0.0 | 17.9 |       0.0 |  0.0 | 3.356573 | 1.678286 |      0.0 |  1.58643 | 2.9893 |      0.0 |      8.95 |  0.0 |       0.0 |       0.0 | 0.067229 | 0.182048 | 0.622292 |  0.0 |       0.0 |  0.61492 | 0.170811 |      0.0 |  0.0 |  0.0 | 132.000457 |      0.0 |      0.0 | 0.067229 | 0.182048 | 0.622292 | 0.038797 |      0.0 | 0.068817 | 0.271585 |   0.23572 |   0.0 | 0.170811 |
    | 2000-01-04 12:00:00 |  0.0 | 18.2 | 559.1 | 0.0 | 0.0 |   0.0 | 19.0 |       0.0 |  0.0 | 3.356514 | 1.678257 |      0.0 |  1.57939 |  3.173 |      0.0 |       9.5 |  0.0 |       0.0 |       0.0 |    0.066 | 0.162921 | 0.610002 |  0.0 |       0.0 | 0.602119 | 0.167255 |      0.0 |  0.0 |  0.0 | 129.582144 |      0.0 |      0.0 |    0.066 | 0.162921 | 0.610002 | 0.031764 |      0.0 | 0.068709 |  0.26214 |  0.239506 |   0.0 | 0.167255 |
    | 2000-01-04 13:00:00 |  0.0 | 22.4 | 668.0 | 0.0 | 0.0 |   0.0 | 23.2 |       0.0 |  0.0 | 4.286095 | 2.143047 |      0.0 | 2.007477 | 3.8744 |      0.0 |      11.6 |  0.0 |       0.0 |       0.0 | 0.064791 | 0.144806 | 0.597911 |  0.0 |       0.0 | 0.589507 | 0.163752 |      0.0 |  0.0 |  0.0 | 126.767159 |      0.0 |      0.0 | 0.064791 | 0.144806 | 0.597911 | 0.026006 |      0.0 | 0.068548 | 0.251821 |  0.243132 |   0.0 | 0.163752 |
    | 2000-01-04 14:00:00 |  0.0 | 21.4 | 593.4 | 0.0 | 0.0 |   0.0 | 22.2 |       0.0 |  0.0 | 3.753358 | 1.876679 |      0.0 | 1.747864 | 3.7074 |      0.0 |      11.1 |  0.0 |       0.0 |       0.0 | 0.063384 | 0.124637 | 0.583836 |  0.0 |       0.0 | 0.576876 | 0.160243 |      0.0 |  0.0 |  0.0 | 124.247439 |      0.0 |      0.0 | 0.063384 | 0.124637 | 0.583836 | 0.021292 |      0.0 |  0.06833 | 0.240662 |  0.246592 |   0.0 | 0.160243 |
    | 2000-01-04 15:00:00 |  0.0 | 21.8 | 493.0 | 0.0 | 0.0 |   0.0 | 22.6 |       0.0 |  0.0 | 3.144766 | 1.572383 |      0.0 | 1.456406 | 3.7742 |      0.0 |      11.3 |  0.0 |       0.0 |       0.0 | 0.062124 | 0.107459 | 0.571237 |  0.0 |       0.0 | 0.564164 | 0.156712 |      0.0 |  0.0 |  0.0 | 122.050213 |      0.0 |      0.0 | 0.062124 | 0.107459 | 0.571237 | 0.017433 |      0.0 | 0.068058 |  0.22879 |  0.249885 |   0.0 | 0.156712 |
    | 2000-01-04 16:00:00 |  0.0 | 22.2 | 391.2 | 0.0 | 0.0 |   0.0 | 23.0 |       0.0 |  0.0 | 2.519332 | 1.259666 |      0.0 | 1.160821 |  3.841 |      0.0 |      11.5 |  0.0 |       0.0 |       0.0 | 0.061025 | 0.093188 | 0.560251 |  0.0 |       0.0 | 0.551594 | 0.153221 |      0.0 |  0.0 |  0.0 | 120.174928 |      0.0 |      0.0 | 0.061025 | 0.093188 | 0.560251 | 0.014273 |      0.0 | 0.067741 | 0.216553 |  0.253027 |   0.0 | 0.153221 |
    | 2000-01-04 17:00:00 |  0.0 | 20.1 | 186.0 | 0.0 | 0.0 |   0.0 | 20.9 |       0.0 |  0.0 | 1.179367 | 0.589683 |      0.0 | 0.540929 | 3.4903 |      0.0 |     10.45 |  0.0 |       0.0 |       0.0 | 0.060087 | 0.081557 | 0.550875 |  0.0 |       0.0 | 0.539364 | 0.149823 |      0.0 |  0.0 |  0.0 |  118.94148 |      0.0 |      0.0 | 0.060087 | 0.081557 | 0.550875 | 0.011685 |      0.0 | 0.067391 | 0.204251 |  0.256038 |   0.0 | 0.149823 |
    | 2000-01-04 18:00:00 |  0.0 | 17.8 |  82.4 | 0.0 | 0.0 |   0.0 | 18.6 |       0.0 |  0.0 | 0.523693 | 0.261846 |      0.0 | 0.239446 | 3.1062 |      0.0 |       9.3 |  0.0 |       0.0 |       0.0 | 0.059471 | 0.074193 | 0.544707 |  0.0 |       0.0 | 0.527746 | 0.146596 |      0.0 |  0.0 |  0.0 | 118.023663 |      0.0 |      0.0 | 0.059471 | 0.074193 | 0.544707 | 0.009567 |      0.0 | 0.067019 | 0.192219 |   0.25894 |   0.0 | 0.146596 |
    | 2000-01-04 19:00:00 |  0.0 | 15.2 |  17.0 | 0.0 | 0.0 |   0.0 | 16.0 |       0.0 |  0.0 |  0.13182 |  0.06591 |      0.0 | 0.060127 |  2.672 |      0.0 |       8.0 |  0.0 |       0.0 |       0.0 | 0.059012 | 0.068866 | 0.540118 |  0.0 |       0.0 | 0.516963 | 0.143601 |      0.0 |  0.0 |  0.0 | 117.295539 |      0.0 |      0.0 | 0.059012 | 0.068866 | 0.540118 | 0.007833 |      0.0 |  0.06664 | 0.180729 |  0.261761 |   0.0 | 0.143601 |
    | 2000-01-04 20:00:00 |  0.0 | 14.5 |   0.0 | 0.0 | 0.0 |   0.0 | 15.3 |       0.0 |  0.0 | 0.035803 | 0.017902 |      0.0 | 0.016299 | 2.5551 |      0.0 |      7.65 |  0.0 |       0.0 |       0.0 | 0.058648 | 0.064736 | 0.536478 |  0.0 |       0.0 | 0.507069 | 0.140852 |      0.0 |  0.0 |  0.0 | 116.619378 |      0.0 |      0.0 | 0.058648 | 0.064736 | 0.536478 | 0.006413 |      0.0 | 0.066259 | 0.169884 |  0.264513 |   0.0 | 0.140852 |
    | 2000-01-04 21:00:00 |  0.0 | 12.4 |   0.0 | 0.0 | 0.0 |   0.0 | 13.2 |       0.0 |  0.0 | 0.034308 | 0.017154 |      0.0 |  0.01559 | 2.2044 |      0.0 |       6.6 |  0.0 |       0.0 |       0.0 |  0.05831 | 0.060977 | 0.533097 |  0.0 |       0.0 | 0.498028 | 0.138341 |      0.0 |  0.0 |  0.0 | 115.951405 |      0.0 |      0.0 |  0.05831 | 0.060977 | 0.533097 | 0.005251 |      0.0 | 0.065879 | 0.159696 |  0.267202 |   0.0 | 0.138341 |
    | 2000-01-04 22:00:00 |  0.0 | 11.7 |   0.0 | 0.0 | 0.0 |   0.0 | 12.5 |       0.0 |  0.0 |   0.0338 |   0.0169 |      0.0 | 0.015331 | 2.0875 |      0.0 |      6.25 |  0.0 |       0.0 |       0.0 | 0.057976 | 0.057338 | 0.529757 |  0.0 |       0.0 | 0.489758 | 0.136044 |      0.0 |  0.0 |  0.0 | 115.291003 |      0.0 |      0.0 | 0.057976 | 0.057338 | 0.529757 | 0.004299 |      0.0 | 0.065502 | 0.150126 |  0.269831 |   0.0 | 0.136044 |
    | 2000-01-04 23:00:00 |  0.0 | 11.9 |   0.0 | 0.0 | 0.0 |   0.0 | 12.7 |       0.0 |  0.0 | 0.033946 | 0.016973 |      0.0 | 0.015369 | 2.1209 |      0.0 |      6.35 |  0.0 |       0.0 |       0.0 | 0.057646 | 0.053814 | 0.526455 |  0.0 |       0.0 | 0.482173 | 0.133937 |      0.0 |  0.0 |  0.0 |  114.63772 |      0.0 |      0.0 | 0.057646 | 0.053814 | 0.526455 |  0.00352 |      0.0 | 0.065127 | 0.141125 |  0.272401 |   0.0 | 0.133937 |

.. _lland_v1_acker_winter:

acre (winter)
_____________

In the next example, we modify the input temperature series |TemL|
to demonstrate the functioning of the snow routine.  For simplicity,
|TemL| increases linearly from -20 to +20 °C.  The ice content of the
snow layer (|WATS|) starts to melt when the temperature crosses the
threshold temperature |TGr|. However, snow layer's actual water
release (|WaDa|) starts one day later when the liquid water
content of the snow layer finally exceeds its holding capacity:

.. integration-test::

    >>> inputs.teml.series = numpy.linspace(-20.0, 20.0, 96)
    >>> test("lland_v1_acker_winter",
    ...      axis1=(inputs.nied, fluxes.wada),
    ...      axis2=(states.waes, states.wats))
    |                date | nied |       teml |  glob |  qz | qzh |  nkor |       tkor |      nbes |      sbes |      et0 |     evpo |      evi |      evb |      wgtf |     wnied |   schmpot |     schm |      wada |      qdb |     qib1 |     qib2 |      qbb | qkap |     qdgz |      qah |       qa |     inzp |       wats |       waes |       bowa |    qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |    qdga1 |    qdga2 |    qiga1 |    qiga2 |      qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |  0.0 |      -20.0 |   0.0 | 0.0 | 0.0 |   0.0 |      -19.2 |       0.0 |       0.0 | 0.003581 |  0.00179 |      0.0 | 0.000442 |   -3.2064 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 |     0.01 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.069558 |      0.0 |      0.0 |     0.01 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000246 |      0.0 | -0.000399 |   0.0 |      0.0 |
    | 2000-01-01 01:00:00 |  0.0 | -19.578947 |   0.0 | 0.0 | 0.0 |   0.0 | -18.778947 |       0.0 |       0.0 | 0.004103 | 0.002051 |      0.0 | 0.000508 | -3.136084 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010035 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.139015 |      0.0 |      0.0 | 0.010035 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.000722 |      0.0 | -0.001191 |   0.0 |      0.0 |
    | 2000-01-01 02:00:00 |  0.0 | -19.157895 |   0.0 | 0.0 | 0.0 |   0.0 | -18.357895 |       0.0 |       0.0 |  0.00462 |  0.00231 |      0.0 | 0.000574 | -3.065768 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01007 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.208371 |      0.0 |      0.0 |  0.01007 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001177 |      0.0 | -0.001975 |   0.0 |      0.0 |
    | 2000-01-01 03:00:00 |  0.0 | -18.736842 |   0.0 | 0.0 | 0.0 |   0.0 | -17.936842 |       0.0 |       0.0 | 0.005134 | 0.002567 |      0.0 |  0.00064 | -2.995453 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010104 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.277626 |      0.0 |      0.0 | 0.010104 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.001612 |      0.0 | -0.002751 |   0.0 |      0.0 |
    | 2000-01-01 04:00:00 |  0.0 | -18.315789 |   0.0 | 0.0 | 0.0 |   0.0 | -17.515789 |       0.0 |       0.0 | 0.005643 | 0.002822 |      0.0 | 0.000706 | -2.925137 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010139 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.346781 |      0.0 |      0.0 | 0.010139 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002027 |      0.0 |  -0.00352 |   0.0 |      0.0 |
    | 2000-01-01 05:00:00 |  0.0 | -17.894737 |   0.0 | 0.0 | 0.0 |   0.0 | -17.094737 |       0.0 |       0.0 | 0.006149 | 0.003074 |      0.0 | 0.000772 | -2.854821 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010173 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.415836 |      0.0 |      0.0 | 0.010173 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002423 |      0.0 | -0.004281 |   0.0 |      0.0 |
    | 2000-01-01 06:00:00 |  0.0 | -17.473684 |  11.2 | 0.0 | 0.0 |   0.0 | -16.673684 |       0.0 |       0.0 | 0.018182 | 0.009091 |      0.0 |  0.00229 | -2.784505 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010208 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.483338 |      0.0 |      0.0 | 0.010208 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.002802 |      0.0 | -0.005034 |   0.0 |      0.0 |
    | 2000-01-01 07:00:00 |  0.0 | -17.052632 | 105.5 | 0.0 | 0.0 |   0.0 | -16.252632 |       0.0 |       0.0 | 0.123904 | 0.061952 |      0.0 | 0.015656 | -2.714189 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010242 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |   20.53744 |      0.0 |      0.0 | 0.010242 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.003164 |      0.0 |  -0.00578 |   0.0 |      0.0 |
    | 2000-01-01 08:00:00 |  0.0 | -16.631579 | 248.3 | 0.0 | 0.0 |   0.0 | -15.831579 |       0.0 |       0.0 | 0.301406 | 0.150703 |      0.0 | 0.038182 | -2.643874 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010269 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.568989 |      0.0 |      0.0 | 0.010269 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00351 |      0.0 | -0.006519 |   0.0 |      0.0 |
    | 2000-01-01 09:00:00 |  0.0 | -16.210526 | 401.3 | 0.0 | 0.0 |   0.0 | -15.410526 |       0.0 |       0.0 | 0.513333 | 0.256667 |      0.0 | 0.065125 | -2.573558 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010284 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |   20.57358 |      0.0 |      0.0 | 0.010284 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00384 |      0.0 |  -0.00725 |   0.0 |      0.0 |
    | 2000-01-01 10:00:00 |  0.0 | -15.789474 | 449.7 | 0.0 | 0.0 |   0.0 | -14.989474 |       0.0 |       0.0 | 0.608577 | 0.304289 |      0.0 | 0.077224 | -2.503242 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010287 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.566069 |      0.0 |      0.0 | 0.010287 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004154 |      0.0 | -0.007974 |   0.0 |      0.0 |
    | 2000-01-01 11:00:00 |  0.0 | -15.368421 | 493.4 | 0.0 | 0.0 |   0.0 | -14.568421 |       0.0 |       0.0 | 0.704187 | 0.352093 |      0.0 | 0.089325 | -2.432926 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010283 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |   20.54646 |      0.0 |      0.0 | 0.010283 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004453 |      0.0 |  -0.00869 |   0.0 |      0.0 |
    | 2000-01-01 12:00:00 |  0.0 | -14.947368 | 261.5 | 0.0 | 0.0 |   0.0 | -14.147368 |       0.0 |       0.0 | 0.397337 | 0.198669 |      0.0 | 0.050356 | -2.362611 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010273 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.565831 |      0.0 |      0.0 | 0.010273 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.004738 |      0.0 |   -0.0094 |   0.0 |      0.0 |
    | 2000-01-01 13:00:00 |  0.0 | -14.526316 | 363.6 | 0.0 | 0.0 |   0.0 | -13.726316 |       0.0 |       0.0 | 0.575929 | 0.287965 |      0.0 | 0.073055 | -2.292295 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010283 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.562493 |      0.0 |      0.0 | 0.010283 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005008 |      0.0 | -0.010102 |   0.0 |      0.0 |
    | 2000-01-01 14:00:00 |  0.0 | -14.105263 | 446.2 | 0.0 | 0.0 |   0.0 | -13.305263 |       0.0 |       0.0 | 0.737491 | 0.368746 |      0.0 | 0.093534 | -2.221979 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010281 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.538677 |      0.0 |      0.0 | 0.010281 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005265 |      0.0 | -0.010798 |   0.0 |      0.0 |
    | 2000-01-01 15:00:00 |  0.2 | -13.684211 | 137.6 | 0.0 | 0.0 |  0.24 | -12.884211 |       0.0 |       0.0 | 0.245131 | 0.122566 | 0.122566 |      0.0 | -2.151663 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010269 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.117434 |        0.0 |        0.0 |  20.608408 |      0.0 |      0.0 | 0.010269 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005509 |      0.0 | -0.011487 |   0.0 |      0.0 |
    | 2000-01-01 16:00:00 |  0.0 | -13.263158 | 103.0 | 0.0 | 0.0 |   0.0 | -12.463158 |       0.0 |       0.0 | 0.194116 | 0.097058 | 0.097058 |      0.0 | -2.081347 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010304 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.020376 |        0.0 |        0.0 |  20.678104 |      0.0 |      0.0 | 0.010304 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005742 |      0.0 | -0.012168 |   0.0 |      0.0 |
    | 2000-01-01 17:00:00 |  0.0 | -12.842105 |  63.7 | 0.0 | 0.0 |   0.0 | -12.042105 |       0.0 |       0.0 | 0.129421 |  0.06471 | 0.020376 | 0.011306 | -2.011032 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010339 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |  20.736459 |      0.0 |      0.0 | 0.010339 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.005966 |      0.0 | -0.012843 |   0.0 |      0.0 |
    | 2000-01-01 18:00:00 |  1.3 | -12.421053 |  41.4 | 0.0 | 0.0 |  1.56 | -11.621053 |      0.76 |      0.76 | 0.091669 | 0.045835 | 0.045835 |      0.0 | -1.940716 | -0.018459 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010368 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.754165 |       0.76 |       0.76 |   20.80609 |      0.0 |      0.0 | 0.010368 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00618 |      0.0 | -0.013511 |   0.0 |      0.0 |
    | 2000-01-01 19:00:00 |  5.6 |      -12.0 |   7.9 | 0.0 | 0.0 |  6.72 |      -11.2 |  6.674165 |  6.674165 |  0.02851 | 0.014255 | 0.014255 |      0.0 |   -1.8704 | -0.156229 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010403 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.785745 |   7.434165 |   7.434165 |  20.875687 |      0.0 |      0.0 | 0.010403 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006385 |      0.0 | -0.014173 |   0.0 |      0.0 |
    | 2000-01-01 20:00:00 |  2.9 | -11.578947 |   0.0 | 0.0 | 0.0 |  3.48 | -10.778947 |  3.465745 |  3.465745 | 0.013274 | 0.006637 | 0.006637 |      0.0 | -1.800084 | -0.078076 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010438 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.793363 |   10.89991 |   10.89991 |   20.94525 |      0.0 |      0.0 | 0.010438 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006582 |      0.0 | -0.014828 |   0.0 |      0.0 |
    | 2000-01-01 21:00:00 |  4.9 | -11.157895 |   0.0 | 0.0 | 0.0 |  5.88 | -10.357895 |  5.873363 |  5.873363 |  0.01372 |  0.00686 |  0.00686 |      0.0 | -1.729768 | -0.127147 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010473 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.79314 |  16.773273 |  16.773273 |  21.014777 |      0.0 |      0.0 | 0.010473 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006771 |      0.0 | -0.015476 |   0.0 |      0.0 |
    | 2000-01-01 22:00:00 | 10.6 | -10.736842 |   0.0 | 0.0 | 0.0 | 12.72 |  -9.936842 |  12.71314 |  12.71314 | 0.014164 | 0.007082 | 0.007082 |      0.0 | -1.659453 | -0.264026 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010507 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.792918 |  29.486413 |  29.486413 |   21.08427 |      0.0 |      0.0 | 0.010507 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.006952 |      0.0 | -0.016118 |   0.0 |      0.0 |
    | 2000-01-01 23:00:00 |  0.1 | -10.315789 |   0.0 | 0.0 | 0.0 |  0.12 |  -9.515789 |  0.112918 |  0.112918 | 0.014604 | 0.007302 | 0.007302 |      0.0 | -1.589137 | -0.002246 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010542 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.792698 |  29.599331 |  29.599331 |  21.153727 |      0.0 |      0.0 | 0.010542 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007126 |      0.0 | -0.016754 |   0.0 |      0.0 |
    | 2000-01-02 00:00:00 |  0.7 |  -9.894737 |   0.0 | 0.0 | 0.0 |  0.84 |  -9.094737 |  0.832698 |  0.832698 |  0.01504 |  0.00752 |  0.00752 |      0.0 | -1.518821 | -0.015828 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010577 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.79248 |  30.432029 |  30.432029 |  21.223151 |      0.0 |      0.0 | 0.010577 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007294 |      0.0 | -0.017383 |   0.0 |      0.0 |
    | 2000-01-02 01:00:00 |  3.0 |  -9.473684 |   0.0 | 0.0 | 0.0 |   3.6 |  -8.673684 |   3.59248 |   3.59248 | 0.015474 | 0.007737 | 0.007737 |      0.0 | -1.448505 | -0.065124 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010612 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.792263 |  34.024509 |  34.024509 |  21.292539 |      0.0 |      0.0 | 0.010612 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007455 |      0.0 | -0.018006 |   0.0 |      0.0 |
    | 2000-01-02 02:00:00 |  2.1 |  -9.052632 |   0.0 | 0.0 | 0.0 |  2.52 |  -8.252632 |  2.512263 |  2.512263 | 0.015904 | 0.007952 | 0.007952 |      0.0 | -1.378189 | -0.043332 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010646 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.792048 |  36.536772 |  36.536772 |  21.361893 |      0.0 |      0.0 | 0.010646 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007609 |      0.0 | -0.018623 |   0.0 |      0.0 |
    | 2000-01-02 03:00:00 | 10.4 |  -8.631579 |   0.0 | 0.0 | 0.0 | 12.48 |  -7.831579 | 12.472048 | 12.472048 | 0.016331 | 0.008166 | 0.008166 |      0.0 | -1.307874 | -0.204142 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010681 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.791834 |   49.00882 |   49.00882 |  21.431212 |      0.0 |      0.0 | 0.010681 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007758 |      0.0 | -0.019234 |   0.0 |      0.0 |
    | 2000-01-02 04:00:00 |  3.5 |  -8.210526 |   0.0 | 0.0 | 0.0 |   4.2 |  -7.410526 |  4.191834 |  4.191834 | 0.016756 | 0.008378 | 0.008378 |      0.0 | -1.237558 | -0.064923 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010716 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.791622 |  53.200655 |  53.200655 |  21.500496 |      0.0 |      0.0 | 0.010716 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.007902 |      0.0 | -0.019839 |   0.0 |      0.0 |
    | 2000-01-02 05:00:00 |  3.4 |  -7.789474 |   0.0 | 0.0 | 0.0 |  4.08 |  -6.989474 |  4.071622 |  4.071622 | 0.017176 | 0.008588 | 0.008588 |      0.0 | -1.167242 | -0.059478 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01075 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.791412 |  57.272277 |  57.272277 |  21.569746 |      0.0 |      0.0 |  0.01075 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00804 |      0.0 | -0.020437 |   0.0 |      0.0 |
    | 2000-01-02 06:00:00 |  1.2 |  -7.368421 |   6.1 | 0.0 | 0.0 |  1.44 |  -6.568421 |  1.431412 |  1.431412 | 0.034213 | 0.017106 | 0.017106 |      0.0 | -1.096926 |  -0.01965 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010785 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.782894 |  58.703689 |  58.703689 |  21.638961 |      0.0 |      0.0 | 0.010785 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008173 |      0.0 |  -0.02103 |   0.0 |      0.0 |
    | 2000-01-02 07:00:00 |  0.1 |  -6.947368 |  77.9 | 0.0 | 0.0 |  0.12 |  -6.147368 |  0.102894 |  0.102894 | 0.235237 | 0.117619 | 0.117619 |      0.0 | -1.026611 | -0.001322 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010819 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.682381 |  58.806582 |  58.806582 |  21.708142 |      0.0 |      0.0 | 0.010819 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008301 |      0.0 | -0.021617 |   0.0 |      0.0 |
    | 2000-01-02 08:00:00 |  0.0 |  -6.526316 | 196.7 | 0.0 | 0.0 |   0.0 |  -5.726316 |       0.0 |       0.0 | 0.579476 | 0.289738 | 0.289738 |      0.0 | -0.956295 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010854 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.392643 |  58.806582 |  58.806582 |  21.777287 |      0.0 |      0.0 | 0.010854 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008425 |      0.0 | -0.022198 |   0.0 |      0.0 |
    | 2000-01-02 09:00:00 |  0.0 |  -6.105263 | 121.9 | 0.0 | 0.0 |   0.0 |  -5.305263 |       0.0 |       0.0 |  0.37425 | 0.187125 | 0.187125 |      0.0 | -0.885979 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010889 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.205518 |  58.806582 |  58.806582 |  21.846399 |      0.0 |      0.0 | 0.010889 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008544 |      0.0 | -0.022773 |   0.0 |      0.0 |
    | 2000-01-02 10:00:00 |  0.4 |  -5.684211 | 156.6 | 0.0 | 0.0 |  0.48 |  -4.884211 |       0.0 |       0.0 | 0.485677 | 0.242839 | 0.242839 |      0.0 | -0.815663 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010923 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.44268 |  58.806582 |  58.806582 |  21.915476 |      0.0 |      0.0 | 0.010923 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008659 |      0.0 | -0.023342 |   0.0 |      0.0 |
    | 2000-01-02 11:00:00 |  0.1 |  -5.263158 | 404.7 | 0.0 | 0.0 |  0.12 |  -4.463158 |       0.0 |       0.0 | 1.250324 | 0.625162 |  0.56268 | 0.016845 | -0.745347 |       0.0 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010958 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  58.806582 |  58.806582 |  21.967673 |      0.0 |      0.0 | 0.010958 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008771 |      0.0 | -0.023906 |   0.0 |      0.0 |
    | 2000-01-02 12:00:00 |  3.6 |  -4.842105 | 217.9 | 0.0 | 0.0 |  4.32 |  -4.042105 |      3.52 |      3.52 | 0.696177 | 0.348089 | 0.348089 |      0.0 | -0.675032 | -0.029737 |       0.0 |      0.0 |       0.0 |      0.0 | 0.010984 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.451911 |  62.326582 |  62.326582 |  22.036689 |      0.0 |      0.0 | 0.010984 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008878 |      0.0 | -0.024464 |   0.0 |      0.0 |
    | 2000-01-02 13:00:00 |  5.9 |  -4.421053 | 582.0 | 0.0 | 0.0 |  7.08 |  -3.621053 |  6.731911 |  6.731911 | 1.862191 | 0.931095 |      0.8 | 0.035529 | -0.604716 | -0.050947 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011018 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  69.058493 |  69.058493 |  22.070142 |      0.0 |      0.0 | 0.011018 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.008981 |      0.0 | -0.025017 |   0.0 |      0.0 |
    | 2000-01-02 14:00:00 |  1.1 |       -4.0 | 263.9 | 0.0 | 0.0 |  1.32 |       -3.2 |      0.52 |      0.52 |  0.87208 |  0.43604 |  0.43604 |      0.0 |   -0.5344 | -0.003478 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011035 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.36396 |  69.578493 |  69.578493 |  22.139107 |      0.0 |      0.0 | 0.011035 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009081 |      0.0 | -0.025564 |   0.0 |      0.0 |
    | 2000-01-02 15:00:00 | 20.7 |  -3.578947 | 136.8 | 0.0 | 0.0 | 24.84 |  -2.778947 |  24.40396 |  24.40396 | 0.470795 | 0.235398 | 0.235398 |      0.0 | -0.464084 | -0.141738 |       0.0 |      0.0 |       0.0 |      0.0 |  0.01107 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.564602 |  93.982453 |  93.982453 |  22.208038 |      0.0 |      0.0 |  0.01107 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009177 |      0.0 | -0.026105 |   0.0 |      0.0 |
    | 2000-01-02 16:00:00 | 37.9 |  -3.157895 | 146.6 | 0.0 | 0.0 | 45.48 |  -2.357895 | 45.244602 | 45.244602 | 0.512226 | 0.256113 | 0.256113 |      0.0 | -0.393768 | -0.222965 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011104 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.543887 | 139.227056 | 139.227056 |  22.276934 |      0.0 |      0.0 | 0.011104 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009271 |      0.0 | -0.026642 |   0.0 |      0.0 |
    | 2000-01-02 17:00:00 |  8.2 |  -2.736842 | 190.6 | 0.0 | 0.0 |  9.84 |  -1.936842 |  9.583887 |  9.583887 | 0.671271 | 0.335635 | 0.335635 |      0.0 | -0.323453 | -0.038796 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011138 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.464365 | 148.810943 | 148.810943 |  22.345795 |      0.0 |      0.0 | 0.011138 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009361 |      0.0 | -0.027173 |   0.0 |      0.0 |
    | 2000-01-02 18:00:00 |  3.6 |  -2.315789 | 103.5 | 0.0 | 0.0 |  4.32 |  -1.515789 |  3.984365 |  3.984365 | 0.381104 | 0.190552 | 0.190552 |      0.0 | -0.253137 | -0.012622 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011173 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.609448 | 152.795307 | 152.795307 |  22.414622 |      0.0 |      0.0 | 0.011173 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009448 |      0.0 | -0.027698 |   0.0 |      0.0 |
    | 2000-01-02 19:00:00 |  7.5 |  -1.894737 |  13.8 | 0.0 | 0.0 |   9.0 |  -1.094737 |  8.809448 |  8.809448 | 0.071409 | 0.035704 | 0.035704 |      0.0 | -0.182821 | -0.020156 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011207 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.764296 | 161.604755 | 161.604755 |  22.483415 |      0.0 |      0.0 | 0.011207 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009533 |      0.0 | -0.028219 |   0.0 |      0.0 |
    | 2000-01-02 20:00:00 | 18.5 |  -1.473684 |   0.0 | 0.0 | 0.0 |  22.2 |  -0.673684 | 22.164296 | 22.164296 | 0.023144 | 0.011572 | 0.011572 |      0.0 | -0.112505 | -0.031207 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011242 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.788428 | 183.769051 | 183.769051 |  22.552173 |      0.0 |      0.0 | 0.011242 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009616 |      0.0 | -0.028734 |   0.0 |      0.0 |
    | 2000-01-02 21:00:00 | 15.4 |  -1.052632 |   0.0 | 0.0 | 0.0 | 18.48 |  -0.252632 | 18.468428 | 18.468428 |  0.02352 |  0.01176 |  0.01176 |      0.0 | -0.042189 | -0.009751 |       0.0 |      0.0 |       0.0 |      0.0 | 0.011276 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.78824 | 202.237479 | 202.237479 |  22.620897 |      0.0 |      0.0 | 0.011276 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009696 |      0.0 | -0.029244 |   0.0 |      0.0 |
    | 2000-01-02 22:00:00 |  6.3 |  -0.631579 |   0.0 | 0.0 | 0.0 |  7.56 |   0.168421 |   7.54824 |  6.912599 | 0.023893 | 0.011947 | 0.011947 |      0.0 |  0.028126 |  0.002881 |  0.092838 | 0.092838 |       0.0 |      0.0 |  0.01131 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.788053 |  209.05724 | 209.785719 |  22.689587 |      0.0 |      0.0 |  0.01131 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009774 |      0.0 | -0.029749 |   0.0 |      0.0 |
    | 2000-01-02 23:00:00 |  1.9 |  -0.210526 |   0.0 | 0.0 | 0.0 |  2.28 |   0.589474 |  2.268053 |  1.599575 | 0.024264 | 0.012132 | 0.012132 |      0.0 |  0.098442 |   0.00362 |  0.305577 | 0.305577 |       0.0 |      0.0 | 0.011345 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787868 | 210.351238 | 212.053773 |  22.758242 |      0.0 |      0.0 | 0.011345 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.00985 |      0.0 | -0.030249 |   0.0 |      0.0 |
    | 2000-01-03 00:00:00 |  4.9 |   0.210526 |   0.0 | 0.0 | 0.0 |  5.88 |   1.010526 |  5.867868 |  2.903051 | 0.024632 | 0.012316 | 0.012316 |      0.0 |  0.168758 |  0.018675 |  0.561176 | 0.561176 |       0.0 |      0.0 | 0.011379 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787684 | 212.693113 | 217.921641 |  22.826863 |      0.0 |      0.0 | 0.011379 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009923 |      0.0 | -0.030744 |   0.0 |      0.0 |
    | 2000-01-03 01:00:00 |  2.7 |   0.631579 |   0.0 | 0.0 | 0.0 |  3.24 |   1.431579 |  3.227684 |  0.917342 | 0.024998 | 0.012499 | 0.012499 |      0.0 |  0.239074 |  0.016592 |  0.765467 | 0.765467 |       0.0 |      0.0 | 0.011413 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787501 | 212.844987 | 221.149325 |  22.895449 |      0.0 |      0.0 | 0.011413 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.009995 |      0.0 | -0.031234 |   0.0 |      0.0 |
    | 2000-01-03 02:00:00 |  0.5 |   1.052632 |   0.0 | 0.0 | 0.0 |   0.6 |   1.852632 |  0.587501 |   0.04329 | 0.025361 | 0.012681 | 0.012681 |      0.0 |  0.309389 |  0.004389 |  0.939456 | 0.939456 |       0.0 |      0.0 | 0.011448 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787319 | 211.948821 | 221.736826 |  22.964002 |      0.0 |      0.0 | 0.011448 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010065 |      0.0 | -0.031719 |   0.0 |      0.0 |
    | 2000-01-03 03:00:00 |  0.2 |   1.473684 |   0.0 | 0.0 | 0.0 |  0.24 |   2.273684 |  0.227319 |       0.0 | 0.025722 | 0.012861 | 0.012861 |      0.0 |  0.379705 |  0.002164 |  1.143321 | 1.143321 |       0.0 |      0.0 | 0.011482 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.787139 |   210.8055 | 221.964145 |   23.03252 |      0.0 |      0.0 | 0.011482 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010133 |      0.0 |   -0.0322 |   0.0 |      0.0 |
    | 2000-01-03 04:00:00 |  0.5 |   1.894737 |   0.0 | 0.0 | 0.0 |   0.6 |   2.694737 |  0.587139 |       0.0 | 0.026081 |  0.01304 |  0.01304 |      0.0 |  0.450021 |  0.006624 |  1.367202 | 1.367202 |       0.0 |      0.0 | 0.011516 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |  0.78696 | 209.438298 | 222.551284 |  23.101003 |      0.0 |      0.0 | 0.011516 |      0.0 |    -0.08 |      0.0 |      0.0 |   0.0102 |      0.0 | -0.032675 |   0.0 |      0.0 |
    | 2000-01-03 05:00:00 |  2.4 |   2.315789 |   0.0 | 0.0 | 0.0 |  2.88 |   3.115789 |   2.86696 |       0.0 | 0.026437 | 0.013219 | 0.013219 |      0.0 |  0.520337 |    0.0374 |  1.669871 | 1.669871 |       0.0 |      0.0 | 0.011551 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.786781 | 207.768427 | 225.418243 |  23.169453 |      0.0 |      0.0 | 0.011551 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010265 |      0.0 | -0.033146 |   0.0 |      0.0 |
    | 2000-01-03 06:00:00 |  0.4 |   2.736842 |   4.4 | 0.0 | 0.0 |  0.48 |   3.536842 |  0.466781 |       0.0 | 0.045043 | 0.022522 | 0.022522 |      0.0 |  0.590653 |  0.006912 |  1.789116 | 1.789116 |       0.0 |      0.0 | 0.011585 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.777478 | 205.979311 | 225.885025 |  23.237868 |      0.0 |      0.0 | 0.011585 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010329 |      0.0 | -0.033612 |   0.0 |      0.0 |
    | 2000-01-03 07:00:00 |  0.2 |   3.157895 |  26.1 | 0.0 | 0.0 |  0.24 |   3.957895 |  0.217478 |       0.0 | 0.136832 | 0.068416 | 0.068416 |      0.0 |  0.660968 |  0.003604 |  1.989737 | 1.989737 |       0.0 |      0.0 | 0.011619 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.731584 | 203.989574 | 226.102503 |  23.306249 |      0.0 |      0.0 | 0.011619 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010391 |      0.0 | -0.034074 |   0.0 |      0.0 |
    | 2000-01-03 08:00:00 |  0.0 |   3.578947 |  74.2 | 0.0 | 0.0 |   0.0 |   4.378947 |       0.0 |       0.0 |  0.34334 |  0.17167 |  0.17167 |      0.0 |  0.731284 |       0.0 |  2.189474 | 2.189474 |       0.0 |      0.0 | 0.011653 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 | 0.559914 |   201.8001 | 226.102503 |  23.374596 |      0.0 |      0.0 | 0.011653 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010452 |      0.0 | -0.034531 |   0.0 |      0.0 |
    | 2000-01-03 09:00:00 |  0.0 |        4.0 | 287.1 | 0.0 | 0.0 |   0.0 |        4.8 |       0.0 |       0.0 |  1.26536 |  0.63268 | 0.559914 | 0.020856 |    0.8016 |       0.0 |       2.4 |      2.4 |       0.0 |      0.0 | 0.011687 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |   199.4001 | 226.102503 |  23.422052 |      0.0 |      0.0 | 0.011687 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010511 |      0.0 | -0.034983 |   0.0 |      0.0 |
    | 2000-01-03 10:00:00 |  0.3 |   4.421053 | 299.8 | 0.0 | 0.0 |  0.36 |   5.221053 |       0.0 |       0.0 | 1.336439 | 0.668219 |     0.36 | 0.088512 |  0.871916 |       0.0 |  2.610526 | 2.610526 |       0.0 |      0.0 | 0.011711 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 196.789574 | 226.102503 |  23.401829 |      0.0 |      0.0 | 0.011711 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010569 |      0.0 | -0.035431 |   0.0 |      0.0 |
    | 2000-01-03 11:00:00 |  2.6 |   4.842105 | 363.5 | 0.0 | 0.0 |  3.12 |   5.642105 |      2.32 |       0.0 | 1.634016 | 0.817008 |      0.8 |  0.00488 |  0.942232 |  0.054804 |  2.985136 | 2.985136 |       0.0 |      0.0 | 0.011701 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 193.804438 | 228.422503 |  23.465248 |      0.0 |      0.0 | 0.011701 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010624 |      0.0 | -0.035875 |   0.0 |      0.0 |
    | 2000-01-03 12:00:00 |  0.7 |   5.263158 | 368.4 | 0.0 | 0.0 |  0.84 |   6.063158 |      0.04 |       0.0 | 1.675394 | 0.837697 |      0.8 | 0.010845 |  1.012547 |  0.001015 |  3.034619 | 3.034619 |       0.0 |      0.0 | 0.011733 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 190.769819 | 228.462503 |  23.522671 |      0.0 |      0.0 | 0.011733 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010678 |      0.0 | -0.036314 |   0.0 |      0.0 |
    | 2000-01-03 13:00:00 |  0.3 |   5.684211 | 317.8 | 0.0 | 0.0 |  0.36 |   6.484211 |       0.0 |       0.0 | 1.466203 | 0.733102 |     0.36 |  0.10758 |  1.082863 |       0.0 |  3.242105 | 3.242105 |       0.0 |      0.0 | 0.011761 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 187.527714 | 228.462503 |  23.483329 |      0.0 |      0.0 | 0.011761 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.01073 |      0.0 | -0.036749 |   0.0 |      0.0 |
    | 2000-01-03 14:00:00 |  0.3 |   6.105263 | 534.7 | 0.0 | 0.0 |  0.36 |   6.905263 |       0.0 |       0.0 | 2.475085 | 1.237543 |     0.36 | 0.252631 |  1.153179 |       0.0 |  3.452632 | 3.452632 |       0.0 |      0.0 | 0.011742 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 184.075082 | 228.462503 |  23.298956 |      0.0 |      0.0 | 0.011742 |      0.0 |    -0.08 |      0.0 |      0.0 |  0.01078 |      0.0 | -0.037179 |   0.0 |      0.0 |
    | 2000-01-03 15:00:00 |  0.0 |   6.526316 | 319.4 | 0.0 | 0.0 |   0.0 |   7.326316 |       0.0 |       0.0 | 1.507196 | 0.753598 |      0.0 | 0.215336 |  1.223495 |       0.0 |  3.663158 | 3.663158 |       0.0 |      0.0 | 0.011649 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 180.411924 | 228.462503 |  23.151971 |      0.0 |      0.0 | 0.011649 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010824 |      0.0 | -0.037605 |   0.0 |      0.0 |
    | 2000-01-03 16:00:00 |  0.0 |   6.947368 | 350.6 | 0.0 | 0.0 |   0.0 |   7.747368 |       0.0 |       0.0 | 1.669823 | 0.834911 |      0.0 | 0.237144 |  1.293811 |       0.0 |  3.873684 | 3.873684 |       0.0 |      0.0 | 0.011576 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 |  176.53824 | 228.462503 |  22.983251 |      0.0 |      0.0 | 0.011576 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010863 |      0.0 | -0.038027 |   0.0 |      0.0 |
    | 2000-01-03 17:00:00 |  0.0 |   7.368421 | 215.4 | 0.0 | 0.0 |   0.0 |   8.168421 |       0.0 |       0.0 | 1.048853 | 0.524427 |      0.0 | 0.147926 |  1.364126 |       0.0 |  4.084211 | 4.084211 |       0.0 |      0.0 | 0.011492 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 172.454029 | 228.462503 |  22.903834 |      0.0 |      0.0 | 0.011492 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010895 |      0.0 | -0.038444 |   0.0 |      0.0 |
    | 2000-01-03 18:00:00 |  0.0 |   7.789474 |  97.8 | 0.0 | 0.0 |   0.0 |   8.589474 |       0.0 |       0.0 |  0.49817 | 0.249085 |      0.0 | 0.070029 |  1.434442 |       0.0 |  4.294737 | 4.294737 |       0.0 |      0.0 | 0.011452 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 168.159293 | 228.462503 |  22.902352 |      0.0 |      0.0 | 0.011452 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010924 |      0.0 | -0.038858 |   0.0 |      0.0 |
    | 2000-01-03 19:00:00 |  0.0 |   8.210526 |  13.1 | 0.0 | 0.0 |   0.0 |   9.010526 |       0.0 |       0.0 | 0.094438 | 0.047219 |      0.0 | 0.013275 |  1.504758 |       0.0 |  4.505263 | 4.505263 |       0.0 |      0.0 | 0.011451 |      0.0 |      0.0 | 0.08 |      0.0 |      0.0 |      0.0 |      0.0 | 163.654029 | 228.462503 |  22.957627 |      0.0 |      0.0 | 0.011451 |      0.0 |    -0.08 |      0.0 |      0.0 | 0.010949 |      0.0 | -0.039267 |   0.0 |      0.0 |
    | 2000-01-03 20:00:00 |  0.0 |   8.631579 |   0.0 | 0.0 | 0.0 |   0.0 |   9.431579 |       0.0 |       0.0 | 0.031507 | 0.015754 |      0.0 | 0.004439 |  1.575074 |       0.0 |  4.715789 | 4.715789 |  5.948967 | 0.230433 | 0.011479 |      0.0 |      0.0 | 0.08 | 0.230433 |      0.0 |      0.0 |      0.0 |  158.93824 | 222.513536 |  28.740243 | 0.230433 |      0.0 | 0.011479 |      0.0 |    -0.08 | 0.021581 |      0.0 | 0.010974 |      0.0 | -0.039673 |   0.0 |      0.0 |
    | 2000-01-03 21:00:00 |  0.0 |   9.052632 |   0.0 | 0.0 | 0.0 |   0.0 |   9.852632 |       0.0 |       0.0 | 0.031828 | 0.015914 |      0.0 | 0.005533 |  1.645389 |       0.0 |  4.926316 | 4.926316 |  6.896842 | 0.335537 |  0.01437 |      0.0 |      0.0 | 0.08 | 0.335537 | 0.040279 | 0.011189 |      0.0 | 154.011924 | 215.616694 |  35.361644 | 0.335537 |      0.0 |  0.01437 |      0.0 |    -0.08 | 0.069283 |      0.0 |  0.01107 |      0.0 | -0.040074 |   0.0 | 0.011189 |
    | 2000-01-03 22:00:00 |  0.0 |   9.473684 |   0.0 | 0.0 | 0.0 |   0.0 |  10.273684 |       0.0 |       0.0 | 0.032147 | 0.016073 |      0.0 |  0.00674 |  1.715705 |       0.0 |  5.136842 | 5.136842 |  7.191579 | 0.429268 | 0.017681 |      0.0 |      0.0 | 0.08 | 0.429268 | 0.097166 | 0.026991 |      0.0 | 148.875082 | 208.425115 |  42.179535 | 0.429268 |      0.0 | 0.017681 |      0.0 |    -0.08 | 0.126325 |      0.0 | 0.011312 |      0.0 | -0.040471 |   0.0 | 0.026991 |
    | 2000-01-03 23:00:00 |  0.0 |   9.894737 |   0.0 | 0.0 | 0.0 |   0.0 |  10.694737 |       0.0 |       0.0 | 0.032464 | 0.016232 |      0.0 | 0.007924 |  1.786021 |       0.0 |  5.347368 | 5.347368 |  7.486316 |  0.53446 |  0.02109 |      0.0 |      0.0 | 0.08 |  0.53446 | 0.161933 | 0.044981 |      0.0 | 143.527714 | 200.938799 |  49.182377 |  0.53446 |      0.0 |  0.02109 |      0.0 |    -0.08 | 0.191091 |      0.0 | 0.011707 |      0.0 | -0.040864 |   0.0 | 0.044981 |
    | 2000-01-04 00:00:00 |  0.0 |  10.315789 |   0.0 | 0.0 | 0.0 |   0.0 |  11.115789 |       0.0 |       0.0 | 0.032779 | 0.016389 |      0.0 | 0.009067 |  1.856337 |       0.0 |  5.557895 | 5.557895 |  7.781053 | 0.651987 | 0.024591 |      0.0 |      0.0 | 0.08 | 0.651987 | 0.235337 | 0.065371 |      0.0 | 137.969819 | 193.157746 |  56.357784 | 0.651987 |      0.0 | 0.024591 |      0.0 |    -0.08 |  0.26434 |      0.0 | 0.012251 |      0.0 | -0.041254 |   0.0 | 0.065371 |
    | 2000-01-04 01:00:00 |  0.0 |  10.736842 |   0.0 | 0.0 | 0.0 |   0.0 |  11.536842 |       0.0 |       0.0 | 0.033092 | 0.016546 |      0.0 | 0.010157 |  1.926653 |       0.0 |  5.768421 | 5.768421 |  8.075789 | 0.782802 | 0.028179 |      0.0 |      0.0 | 0.08 | 0.782802 | 0.318161 | 0.088378 |      0.0 | 132.201398 | 185.081957 |  63.692436 | 0.782802 |      0.0 | 0.028179 |      0.0 |    -0.08 |  0.34686 |      0.0 | 0.012941 |      0.0 | -0.041639 |   0.0 | 0.088378 |
    | 2000-01-04 02:00:00 |  0.0 |  11.157895 |   0.0 | 0.0 | 0.0 |   0.0 |  11.957895 |       0.0 |       0.0 | 0.033402 | 0.016701 |      0.0 | 0.011179 |  1.996968 |       0.0 |  5.978947 | 5.978947 |  8.370526 | 0.927958 | 0.031846 |      0.0 |      0.0 | 0.08 | 0.927958 |  0.41123 | 0.114231 |      0.0 |  126.22245 | 176.711431 |  71.171978 | 0.927958 |      0.0 | 0.031846 |      0.0 |    -0.08 | 0.439477 |      0.0 | 0.013774 |      0.0 | -0.042021 |   0.0 | 0.114231 |
    | 2000-01-04 03:00:00 |  1.3 |  11.578947 |   0.0 | 0.0 | 0.0 |  1.56 |  12.378947 |      0.76 |       0.0 | 0.033711 | 0.016856 | 0.016856 |      0.0 |  2.067284 |  0.039389 |  6.307406 | 6.307406 |  9.590368 | 1.212635 | 0.035586 |      0.0 |      0.0 | 0.08 | 1.212635 | 0.531485 | 0.147635 | 0.783144 | 119.915044 | 167.881062 |  79.594126 |  1.17535 | 0.037285 | 0.035586 |      0.0 |    -0.08 | 0.551193 | 0.007944 | 0.014747 |      0.0 | -0.042399 |   0.0 | 0.147635 |
    | 2000-01-04 04:00:00 |  0.0 |       12.0 |   0.0 | 0.0 | 0.0 |   0.0 |       12.8 |       0.0 |       0.0 | 0.034019 | 0.017009 | 0.017009 |      0.0 |    2.1376 |       0.0 |       6.4 |      6.4 |      8.96 |  1.28123 | 0.039797 |      0.0 |      0.0 | 0.08 |  1.28123 | 0.666259 | 0.185072 | 0.766135 | 113.515044 | 158.921062 |  87.313098 |   1.2195 |  0.06173 | 0.039797 |      0.0 |    -0.08 | 0.668468 | 0.024697 | 0.015867 |      0.0 | -0.042773 |   0.0 | 0.185072 |
    | 2000-01-04 05:00:00 |  0.0 |  12.421053 |   0.0 | 0.0 | 0.0 |   0.0 |  13.221053 |       0.0 |       0.0 | 0.034324 | 0.017162 | 0.017162 |      0.0 |  2.207916 |       0.0 |  6.610526 | 6.610526 |  9.254737 | 1.478215 | 0.043657 |      0.0 |      0.0 | 0.08 | 1.478215 | 0.811158 | 0.225322 | 0.748973 | 106.904518 | 149.666325 |  95.125963 | 1.323509 | 0.154707 | 0.043657 |      0.0 |    -0.08 | 0.778094 | 0.059078 | 0.017129 |      0.0 | -0.043144 |   0.0 | 0.225322 |
    | 2000-01-04 06:00:00 |  0.0 |  12.842105 |  17.0 | 0.0 | 0.0 |   0.0 |  13.642105 |       0.0 |       0.0 | 0.125774 | 0.062887 | 0.062887 |      0.0 |  2.278232 |       0.0 |  6.821053 | 6.821053 |  9.549474 | 1.695307 | 0.047563 |      0.0 |      0.0 | 0.08 | 1.695307 | 0.984585 | 0.273496 | 0.686086 | 100.083465 | 140.116852 | 103.012567 | 1.410136 | 0.285171 | 0.047563 |      0.0 |    -0.08 | 0.885074 | 0.124502 | 0.018519 |      0.0 |  -0.04351 |   0.0 | 0.273496 |
    | 2000-01-04 07:00:00 |  0.7 |  13.263158 |  99.7 | 0.0 | 0.0 |  0.84 |  14.063158 |  0.726086 |       0.0 | 0.574133 | 0.287067 | 0.287067 |      0.0 |  2.348547 |  0.042752 |  7.159578 | 7.159578 | 10.749495 | 2.122076 | 0.051506 | 0.004706 | 0.465063 |  0.0 | 2.122076 | 1.223838 | 0.339955 | 0.512933 |  92.923887 | 130.093442 | 111.118711 | 1.528763 | 0.593313 | 0.051506 | 0.004706 | 0.465063 | 0.991361 | 0.253373 | 0.020032 | 0.000228 | -0.041157 |   0.0 | 0.339955 |
    | 2000-01-04 08:00:00 |  0.4 |  13.684211 | 239.4 | 0.0 | 0.0 |  0.48 |  14.484211 |  0.192933 |       0.0 | 1.341074 | 0.670537 | 0.670537 |      0.0 |  2.418863 |    0.0117 |  7.277135 | 7.277135 | 10.380922 | 2.257544 | 0.055559 | 0.033368 | 0.505594 |  0.0 | 2.257544 | 1.489181 | 0.413661 | 0.129463 |  85.646752 | 119.905453 | 118.647569 | 1.557041 | 0.700503 | 0.055559 | 0.033368 | 0.505594 | 1.091424 | 0.409967 | 0.021667 |  0.00204 | -0.035918 |   0.0 | 0.413661 |
    | 2000-01-04 09:00:00 |  0.1 |  14.105263 | 391.2 | 0.0 | 0.0 |  0.12 |  14.905263 |       0.0 |       0.0 | 2.187425 | 1.093712 | 0.249463 | 0.771439 |  2.489179 |       0.0 |  7.452632 | 7.452632 | 10.433684 | 2.480667 | 0.059324 | 0.072473 | 0.543238 |  0.0 | 2.480667 | 1.742875 | 0.484132 |      0.0 |  78.194121 | 109.471769 | 125.154113 | 1.596883 | 0.883784 | 0.059324 | 0.072473 | 0.543238 | 1.179558 | 0.563334 | 0.023413 | 0.006913 | -0.030342 |   0.0 | 0.484132 |
    | 2000-01-04 10:00:00 |  0.4 |  14.526316 | 525.6 | 0.0 | 0.0 |  0.48 |  15.326316 |       0.0 |       0.0 | 2.951108 | 1.475554 |     0.48 | 0.923991 |  2.559495 |       0.0 |  7.663158 | 7.663158 | 10.728421 | 2.754019 | 0.062577 | 0.113542 | 0.575771 |  0.0 | 2.754019 |    2.014 | 0.559444 |      0.0 |  70.530963 |  98.743348 | 131.452634 | 1.636894 | 1.117125 | 0.062577 | 0.113542 | 0.575771 | 1.258953 | 0.739137 | 0.025244 | 0.015139 | -0.024473 |   0.0 | 0.559444 |
    | 2000-01-04 11:00:00 |  0.0 |  14.947368 | 570.2 | 0.0 | 0.0 |   0.0 |  15.747368 |       0.0 |       0.0 |  3.22475 | 1.612375 |      0.0 | 1.515834 |  2.629811 |       0.0 |  7.873684 | 7.873684 | 11.023158 | 3.045113 | 0.065726 | 0.158755 | 0.607263 |  0.0 | 3.045113 | 2.308691 | 0.641303 |      0.0 |  62.657279 |   87.72019 |   137.0831 | 1.671605 | 1.373508 | 0.065726 | 0.158755 | 0.607263 | 1.330713 | 0.942489 | 0.027142 |  0.02669 | -0.018344 |   0.0 | 0.641303 |
    | 2000-01-04 12:00:00 |  0.0 |  15.368421 | 559.1 | 0.0 | 0.0 |   0.0 |  16.168421 |       0.0 |       0.0 |  3.18828 |  1.59414 |      0.0 | 1.513593 |  2.700126 |       0.0 |  8.084211 | 8.084211 | 11.317895 | 3.337761 | 0.068542 | 0.203239 | 0.635416 |  0.0 | 3.337761 | 2.622026 | 0.728341 |      0.0 |  54.573068 |  76.402295 | 142.642445 | 1.700398 | 1.637363 | 0.068542 | 0.203239 | 0.635416 | 1.395203 | 1.168299 | 0.029093 |  0.04141 | -0.011978 |   0.0 | 0.728341 |
    | 2000-01-04 13:00:00 |  0.0 |  15.789474 | 668.0 | 0.0 | 0.0 |   0.0 |  16.589474 |       0.0 |       0.0 | 3.832542 | 1.916271 |      0.0 | 1.834963 |  2.770442 |       0.0 |  8.294737 | 8.294737 | 11.612632 | 3.653288 | 0.071321 | 0.250614 | 0.663212 |  0.0 | 3.653288 | 2.952312 | 0.820087 |      0.0 |  46.278331 |  64.789664 | 147.781678 | 1.726274 | 1.927014 | 0.071321 | 0.250614 | 0.663212 | 1.452949 | 1.414575 | 0.031085 | 0.059101 | -0.005398 |   0.0 | 0.820087 |
    | 2000-01-04 14:00:00 |  0.0 |  16.210526 | 593.4 | 0.0 | 0.0 |   0.0 |  17.010526 |       0.0 |       0.0 | 3.435464 | 1.717732 |      0.0 | 1.656179 |  2.840758 |       0.0 |  8.505263 | 8.505263 | 11.907368 | 3.978557 | 0.073891 | 0.297259 | 0.688908 |  0.0 | 3.978557 | 3.299405 | 0.916501 |      0.0 |  37.773068 |  52.882295 | 152.994252 | 1.748653 | 2.229905 | 0.073891 | 0.297259 | 0.688908 |  1.50459 | 1.680738 | 0.033111 | 0.079583 |  0.001383 |   0.0 | 0.916501 |
    | 2000-01-04 15:00:00 |  0.0 |  16.631579 | 493.0 | 0.0 | 0.0 |   0.0 |  17.431579 |       0.0 |       0.0 | 2.882666 | 1.441333 |      0.0 | 1.398212 |  2.911074 |       0.0 |  8.715789 | 8.715789 | 12.202105 | 4.337227 | 0.076497 | 0.347205 | 0.714971 |  0.0 | 4.337227 | 3.665817 | 1.018283 |      0.0 |  29.057279 |   40.68019 | 158.322245 | 1.769438 | 2.567789 | 0.076497 | 0.347205 | 0.714971 | 1.550778 | 1.968809 | 0.035164 | 0.102713 |  0.008353 |   0.0 | 1.018283 |
    | 2000-01-04 16:00:00 |  0.0 |  17.052632 | 391.2 | 0.0 | 0.0 |   0.0 |  17.852632 |       0.0 |       0.0 | 2.312692 | 1.156346 |      0.0 | 1.127916 |  2.981389 |       0.0 |  8.926316 | 8.926316 | 12.496842 | 4.738146 | 0.079161 | 0.400861 | 0.741611 |  0.0 | 4.738146 | 4.059332 | 1.127592 |      0.0 |  20.130963 |  28.183348 | 163.731393 | 1.788947 | 2.949199 | 0.079161 | 0.400861 | 0.741611 | 1.592241 | 2.285753 | 0.037245 | 0.128575 |  0.015517 |   0.0 | 1.127592 |
    | 2000-01-04 17:00:00 |  0.0 |  17.473684 | 186.0 | 0.0 | 0.0 |   0.0 |  18.273684 |       0.0 |       0.0 | 1.127749 | 0.563874 |      0.0 | 0.552693 |  3.051705 |       0.0 |  9.136842 | 9.136842 | 12.791579 | 5.188971 | 0.081866 | 0.457902 | 0.768657 |  0.0 | 5.188971 | 4.488039 | 1.246678 |      0.0 |  10.994121 |  15.391769 | 169.472882 | 1.807284 | 3.381688 | 0.081866 | 0.457902 | 0.768657 | 1.629615 | 2.638945 | 0.039356 | 0.157246 |  0.022877 |   0.0 | 1.246678 |
    | 2000-01-04 18:00:00 |  0.0 |  17.894737 |  82.4 | 0.0 | 0.0 |   0.0 |  18.694737 |       0.0 |       0.0 | 0.524564 | 0.262282 |      0.0 | 0.258239 |  3.122021 |       0.0 |  9.347368 | 9.347368 | 13.086316 | 5.722149 | 0.084736 | 0.521153 | 0.797364 |  0.0 | 5.722149 | 4.965325 | 1.379257 |      0.0 |   1.646752 |   2.305453 | 175.175556 |  1.82524 | 3.896909 | 0.084736 | 0.521153 | 0.797364 | 1.663503 | 3.040965 |   0.0415 | 0.188917 |   0.03044 |   0.0 | 1.379257 |
    | 2000-01-04 19:00:00 |  0.0 |  18.315789 |  17.0 | 0.0 | 0.0 |   0.0 |  19.115789 |       0.0 |       0.0 | 0.139502 | 0.069751 |      0.0 | 0.068942 |  3.192337 |       0.0 |  9.557895 | 1.646752 |  2.305453 | 1.044653 | 0.087588 | 0.586621 | 0.825878 |  0.0 | 1.044653 | 4.473009 | 1.242502 |      0.0 |        0.0 |        0.0 | 174.867328 | 1.042744 | 0.001909 | 0.087588 | 0.586621 | 0.825878 | 1.619537 | 2.547879 | 0.043679 |   0.2237 |  0.038214 |   0.0 | 1.242502 |
    | 2000-01-04 20:00:00 |  0.0 |  18.736842 |   0.0 | 0.0 | 0.0 |   0.0 |  19.536842 |       0.0 |       0.0 | 0.038685 | 0.019342 |      0.0 | 0.019114 |  3.262653 |       0.0 |  9.768421 |      0.0 |       0.0 |      0.0 | 0.087434 | 0.583017 | 0.824337 |  0.0 |      0.0 | 3.312958 | 0.920266 |      0.0 |        0.0 |        0.0 | 173.353427 |      0.0 |      0.0 | 0.087434 | 0.583017 | 0.824337 | 1.417325 | 1.545711 | 0.045816 | 0.258063 |  0.046043 |   0.0 | 0.920266 |
    | 2000-01-04 21:00:00 |  0.0 |  19.157895 |   0.0 | 0.0 | 0.0 |   0.0 |  19.957895 |       0.0 |       0.0 | 0.038962 | 0.019481 |      0.0 | 0.019232 |  3.332968 |       0.0 |  9.978947 |      0.0 |       0.0 |      0.0 | 0.086677 | 0.565423 | 0.816767 |  0.0 |      0.0 | 2.487641 | 0.691011 |      0.0 |        0.0 |        0.0 | 171.865328 |      0.0 |      0.0 | 0.086677 | 0.565423 | 0.816767 | 1.160408 | 0.937521 | 0.047827 | 0.288135 |   0.05375 |   0.0 | 0.691011 |
    | 2000-01-04 22:00:00 |  0.0 |  19.578947 |   0.0 | 0.0 | 0.0 |   0.0 |  20.378947 |       0.0 |       0.0 | 0.039237 | 0.019619 |      0.0 | 0.019349 |  3.403284 |       0.0 | 10.189474 |      0.0 |       0.0 |      0.0 | 0.085933 | 0.548304 | 0.809327 |  0.0 |      0.0 |   1.9434 | 0.539833 |      0.0 |        0.0 |        0.0 | 170.402416 |      0.0 |      0.0 | 0.085933 | 0.548304 | 0.809327 | 0.950062 | 0.568635 | 0.049704 | 0.313694 |  0.061305 |   0.0 | 0.539833 |
    | 2000-01-04 23:00:00 |  0.0 |       20.0 |   0.0 | 0.0 | 0.0 |   0.0 |       20.8 |       0.0 |       0.0 | 0.039511 | 0.019756 |      0.0 | 0.019464 |    3.4736 |       0.0 |      10.4 |      0.0 |       0.0 |      0.0 | 0.085201 | 0.531648 | 0.802012 |  0.0 |      0.0 | 1.578118 | 0.438366 |      0.0 |        0.0 |        0.0 | 168.964091 |      0.0 |      0.0 | 0.085201 | 0.531648 | 0.802012 | 0.777845 | 0.344895 | 0.051453 | 0.335215 |  0.068711 |   0.0 | 0.438366 |
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools

# ...from lland
from hydpy.models.lland import lland_model
from hydpy.models.lland import lland_masks
from hydpy.models.lland.lland_constants import *


class Model(modeltools.AdHocModel):
    """Turc-Wendling/degree-day version of HydPy-L-Land."""

    INLET_METHODS = (lland_model.Pick_QZ_V1,)
    RECEIVER_METHODS = ()
    ADD_METHODS = ()
    RUN_METHODS = (
        lland_model.Calc_QZH_V1,
        lland_model.Calc_NKor_V1,
        lland_model.Calc_TKor_V1,
        lland_model.Calc_ET0_V1,
        lland_model.Calc_EvPo_V1,
        lland_model.Calc_NBes_Inzp_V1,
        lland_model.Calc_EvI_Inzp_V1,
        lland_model.Calc_SNRatio_V1,
        lland_model.Calc_SBes_V1,
        lland_model.Calc_WATS_V1,
        lland_model.Calc_WGTF_V1,
        lland_model.Calc_WNied_V1,
        lland_model.Calc_SchmPot_V1,
        lland_model.Calc_Schm_WATS_V1,
        lland_model.Calc_WaDa_WAeS_V1,
        lland_model.Calc_EvB_V1,
        lland_model.Calc_QKap_V1,
        lland_model.Calc_QBB_V1,
        lland_model.Calc_QIB1_V1,
        lland_model.Calc_QIB2_V1,
        lland_model.Calc_QDB_V1,
        lland_model.Calc_BoWa_V1,
        lland_model.Calc_QBGZ_V1,
        lland_model.Calc_QIGZ1_V1,
        lland_model.Calc_QIGZ2_V1,
        lland_model.Calc_QDGZ_V1,
        lland_model.Calc_QBGA_V1,
        lland_model.Update_QDGZ_QBGZ_QBGA_V1,
        lland_model.Update_QDGZ_QBGZ_QBGA_V2,
        lland_model.Calc_QIGA1_V1,
        lland_model.Calc_QIGA2_V1,
        lland_model.Calc_QDGZ1_QDGZ2_V1,
        lland_model.Calc_QDGA1_V1,
        lland_model.Calc_QDGA2_V1,
        lland_model.Calc_QAH_V1,
        lland_model.Calc_QA_V1,
    )
    OUTLET_METHODS = (lland_model.Pass_QA_V1,)
    SENDER_METHODS = ()
    SUBMODELS = ()


class Masks(masktools.Masks):
    """Masks applicable to |lland_v1|."""

    # pylint: disable=no-member
    # bug of pylint 2.4?
    CLASSES = lland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
cythonizer.finalise()
