# -*- coding: utf-8 -*-
# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN
.. _`LARSIM`: http://www.larsim.de/en/the-model/

One can understand version 3 of the L-Land model as an extended, more
complicated version of the application model |lland_v1|, using precipitation,
temperature, air pressure, wind speed, relative humidity and sunshine duration
as meteorological input parameters.  It relies on the equations after Penman
and Penman-Monteith for calculating evaporation and evapotranspiration, which
require much more input data than the simple Turc-Wendling equation but bring
flexibility regarding the simulation time step (Turc-Wendling is limited to
daily time steps, Penman and Penman-Monteith work for any time step size).
The second main difference is that |lland_v3| models the energy
balance of snow processes in more detail (following Knauf, 2006), taking not
only the air temperature but also forcings like the soil heat flux or global
radiation into account explicitly.  The third difference is that |lland_v3|
models the soil temperature to simulate the effect of soil water freezing on
runoff generation.  We created |lland_v3| on behalf of the `German Federal
Institute of Hydrology (BfG)`_ in the context of optimising the control of
the Kiel Canal (Nord-Ostsee-Kanal).

The following list summarises the main components of |lland_v3|:

 * Multiple routines for adjusting the meteorological input data
 * Transformation of sunshine duration into global radiation
   :cite:`ref-Thompson1981`
 * Evaporation from water surfaces after Penman :cite:`ref-penman1948natural`
 * Evapotranspiration (interception evaporation) after Penman-Monteith
   :cite:`ref-Thompson1981`
 * Actual soil evapotranspiration after Wigmosta
   :cite:`ref-wigmosta1994distributed`
 * Surface roughnesses after Quast and Böhm :cite:`ref-bohm1998ermittlung`
 * Mixed precipitation within a definable temperature-range
 * Energy balance of the snow-layer after Knauf :cite:`ref-LUBW2006a`
 * Dynamical adjustment of the snow albedo
 * Optional refinement of the snow-surface temperature
 * Optional (re)freezing of liquid water within the snow-layer
 * A simple snow retention routine
 * A soil routine based on the Xinanjiang model :cite:`ref-zhao1977flood`
 * Frost-sealing of the upper soil-layer
 * One base flow, two interflow and two direct flow components
 * A freely configurable capillary rise routine
 * Options to limit the capacity of the base flow storage
 * Separate linear storages for modelling runoff concentration
 * Additional evaporation from water areas within the subcatchment
 * Optional evaporation from inflowing runoff

Some notes for `LARSIM`_ users: |lland_v3| is similar to many `LARSIM`_
models used for forecasting (relying on a combination of options equivalent
to the features listed above) but not identical.  Often, we strive for more
flexibility. One example is that you are allowed to modify parameter
|KTSchnee| to control if and how |lland_v3| adjusts the snow surface
temperature.  However, there are also differences due to technical reasons.
One relevant difference is that *HydPy* does not include "future values"
into the current simulation time step. For example, `LARSIM`_ fills
nightly gaps regarding cloudiness with the average value of the particular
day, while |lland_v3| uses the average over the 24 hours preceding the
current simulation step.  Hence, do not expect the results of |lland_v3|
and `LARSIM`_ to be identical. When switching from one model system to the
other, the need to fine-tune some parameters via calibration might arise.

.. note::

    Some details of |lland_v3| are still under discussion and might change
    in the future.

Integration tests
=================

.. how_to_understand_integration_tests::

Many of the following integration tests are similar to the integration tests
performed for the application model |lland_v1|.  When there are no substantial
differences, we do not repeat our explanations, so please note the respective
sections of the documentation on |lland_v1|.  Some additional tests are
covering structural differences between land-use types that only exist for
|lland_v3| and others are showing the different possible snow-related
configurations of |lland_v3|.  Furthermore, we perform some tests both on a
daily and an hourly basis to point out some differences and to make sure
|lland_v3| works well for different simulation step sizes.

Note that, while our daily simulations always cover an entire month, our
hourly simulations cover only a period three days, focussing on the "most
interesting" part of the daily simulation.  To ensure the best possible
comparability, we use the same input data (we aggregate the hourly values
to daily values), the same parameter values (possibly adapted to the change
in the time step size), and the same initial conditions (we extract the
required initial conditions of the respective hourly simulation from the
longer daily simulation).

daily simulation
________________

The following general setup is identical with the one of |lland_v1|,
except that we now start with a different period and with a daily
simulation time step:

>>> from hydpy import pub, Timegrid
>>> pub.timegrids = "1997-08-01", "1997-09-01", "1d"
>>> from hydpy.models.lland_v3 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> inlet = Node("inlet")
>>> land = Element("land", inlets=inlet, outlets=outlet)
>>> land.model = model
>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)
>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.nied
>>> IntegrationTest.plotting_options.axis2 = fluxes.qah
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

.. _lland_v3_acker_summer_daily:

acre (summer)
-------------

First, we set the values of those parameters also required by the application
model |lland_v1| (see integration test :ref:`lland_v1_acker_summer`):

>>> lnk(ACKER)
>>> kg(0.94)
>>> kt(0.0)
>>> hinz(0.2)
>>> lai.acker_aug = 3.5
>>> tgr(0.0)
>>> trefn(0.0)
>>> tsp(4.0)
>>> pwmax(1.43)
>>> wmax(309.0)
>>> fk(199.0)
>>> pwp(119.4)
>>> kapgrenz(option="0_WMax/10")
>>> kapmax(0.08)
>>> beta(0.001/24)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(r_dmax=0.55)
>>> dmin(r_dmin=10.0)
>>> bsf(0.3)
>>> volbmax(inf)
>>> gsbmax(1.0)
>>> gsbgrad1(inf)
>>> gsbgrad2(inf)
>>> a1(0.0)
>>> a2(0.5)
>>> tind(0.191056)
>>> eqb(35000.0)
>>> eqi1(400.0)
>>> eqi2(400.0)
>>> eqd1(200.0)
>>> eqd2(50.0)
>>> negq(False)

Next, we set the values of the parameters specific to |lland_v3|, beginning
with those required for calculating the evaporation via Penman and the
evapotranspiration via Penman-Monteith:

>>> latitude(54.1)
>>> longitude(9.7)
>>> measuringheightwindspeed(10.0)
>>> angstromconstant(0.25)
>>> angstromfactor(0.5)
>>> angstromalternative(0.15)
>>> emissivity(0.95)
>>> py(119.4)
>>> cropheight.acker_aug = 0.4
>>> albedo.acker_aug = 0.24
>>> wg2z.aug = -0.25
>>> surfaceresistance.acker = 40.0

The following parameters are related to the energy accounting snow
modelling approach after Knauf :cite:`ref-LUBW2006a`:

>>> p1strahl(0.5)
>>> p2strahl(1.0/35.0)
>>> p1wind(0.6)
>>> p2wind(1.0/70.0)
>>> turb0(0.0072)
>>> turb1(0.00576)
>>> albedo0snow(0.9)
>>> snowagingfactor(0.35)
>>> refreezeflag(True)
>>> ktschnee(inf)

Now we set the values of the parameters controlling the frost sealing
of the upper soil layer:

>>> fvf(0.3)
>>> bsff(2.0)

Due to preparing a simulation with a daily simulation step size, we only need
to define the initial values of the state sequences:

>>> test.inits = (
...     (states.inzp, 0.0),
...     (states.wats, 0.0),
...     (states.waes, 0.0),
...     (states.esnow, 0.0),
...     (states.taus, 0.0),
...     (states.ebdn, 30.0),
...     (states.bowa, 72.0),
...     (states.qdgz1, 0.0),
...     (states.qdgz2, 0.0),
...     (states.qigz1, 0.0),
...     (states.qigz2, 0.0),
...     (states.qbgz, 0.0),
...     (states.qdga1, 0.0),
...     (states.qdga2, 0.0),
...     (states.qiga1, 0.0),
...     (states.qiga2, 0.0),
...     (states.qbga, 0.0))

Finally, we prepare the input time series:

>>> inputs.nied.series = (
...     13.3, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0,
...     2.7, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 0.3, 0.0, 0.1, 10.5, 6.5,
...     15.0, 9.2, 0.0, 0.0)
>>> inputs.teml.series = (
...     16.1, 17.3, 18.9, 17.8, 18.5, 19.9, 20.8, 20.9, 22.3, 23.0, 23.1,
...     23.2, 23.2, 21.7, 20.8, 19.5, 21.0, 21.6, 21.1, 22.4, 22.5, 19.6,
...     20.3, 24.1, 23.7, 21.1, 20.9, 20.2, 16.1, 17.0, 18.2)
>>> inputs.sunshineduration.series = (
...     6.3, 1.7, 4.5, 12.4, 13.9, 13.0, 13.8, 12.3, 13.1, 12.8, 13.1, 13.3,
...     12.7, 10.2, 9.4, 10.3, 11.1, 11.0, 8.5, 11.3, 12.4, 0.1, 6.7, 10.4,
...     6.5, 4.9, 6.6, 0.3, 0.1, 5.0, 3.8)
>>> inputs.windspeed.series = (
...     2.4, 1.6, 1.5, 2.8, 3.0, 3.5, 3.8, 3.3, 1.8, 1.9, 3.0, 2.5, 1.6, 2.7,
...     3.2, 1.3, 1.9, 2.0, 2.6, 3.6, 2.5, 2.1, 1.5, 2.2, 1.9, 2.3, 2.1, 2.6,
...     2.6, 2.1, 2.1)
>>> inputs.relativehumidity.series = (
...     86.2, 85.8, 82.4, 77.6, 74.9, 77.9, 73.5, 80.1, 72.9, 68.5, 68.6,
...     66.0, 69.0, 75.6, 81.4, 79.0, 75.9, 70.0, 70.3, 69.0, 62.1, 84.5,
...     83.6, 76.5, 76.6, 84.2, 85.8, 86.5, 89.6, 78.3, 78.5)
>>> inputs.atmosphericpressure.series = (
...     100.7, 101.3, 101.6, 101.7, 101.8, 101.8, 101.9, 101.9, 101.9, 102.0,
...     102.1, 102.2, 101.9, 101.5, 101.6, 101.9, 101.9, 101.8, 101.9, 101.9,
...     101.7, 101.5, 101.5, 101.1, 100.8, 100.6, 100.5, 100.0, 100.4, 101.1,
...     101.4)
>>> inlet.sequences.sim.series = 0.0

The following results illustrate the behaviour of |lland_v3| for relatively
dry and hot summer condions. Compared to the result table of integration
test :ref:`lland_v1_acker_summer` of the application model |lland_v1|,
there are much more columns, because of to the higher number of input,
flux, and state sequences.  These are mainly due to the increased data
requirements and the more complex calculations of the Penman-Monteith
equation and the Knauf :cite:`ref-LUBW2006a` approach.  Additionally,
|lland_v3| calculates many daily values (containing the averages or sums
over the last 24 hours), required for the Penman equation, which we generally
apply based on the input data of the previous 24-hours. (Note that we could
save some computation time by not calculating daily values that are never
required.  First, land-use type |ACKER| does only rely on the Penman-Monteith
equation.  Second, the values of the "normal" sequences are already daily
values, due to the selected simulation step size of one day.  Maybe we
perform some related optimisations later.):

.. integration-test::

    >>> conditions_acker_summer = test(
    ...     "lland_v3_acker_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah),
    ...     axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |     g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |     evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |      ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | -0.25 | 10.933333 |   -2.6784 |             12.492674 |                 12.492674 |                  12.492674 |                  1.877708 |                      0.0 |    10.614966 |              0.0 |         10.614966 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 11.802 |  0.0 | 3.623966 |   0.7 | 1.720207 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 0.196005 | 0.002269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   32.4284 |  81.32045 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.002269 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | -0.25 |    19.028 |  0.895795 |              8.382718 |                  8.382718 |                   8.382718 |                   0.96939 |                      0.0 |     7.413327 |              0.0 |          7.413327 |             88.271017 |                 100.0 |                98.302128 |               94.620659 |    0.0 |  0.0 | 2.555713 | 0.094 | 1.697264 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.263667 | 0.003052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.282605 | 79.623186 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.003052 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | -0.25 | 15.208683 | -1.913579 |             10.795247 |                 10.795247 |                  10.795247 |                  1.386217 |                      0.0 |      9.40903 |              0.0 |           9.40903 |             94.155751 |                 100.0 |               100.629116 |               95.574305 |    0.0 |  0.0 | 3.286521 |   0.0 | 2.329549 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.140696 | 0.001628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.946184 | 77.293637 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.001628 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | -0.25 | 20.753946 |  1.531325 |             17.664962 |                 17.664962 |                  17.664962 |                  3.108707 |                      0.0 |    14.556255 |              0.0 |         14.556255 |             50.440581 |                 100.0 |               103.850807 |               96.853335 |    0.0 |  0.0 | 5.847436 | 0.094 | 3.314416 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.075077 | 0.000869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.164858 | 73.979222 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000869 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | -0.25 | 14.816194 | -1.909685 |             18.901295 |                 18.901295 |                  18.901295 |                  3.414868 |                      0.0 |    15.486427 |              0.0 |         15.486427 |             47.077876 |                 100.0 |               108.496391 |               98.617204 |    0.0 |  0.0 | 6.638616 |   0.0 | 3.732358 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.040062 | 0.000464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.824543 | 70.246864 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000464 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | -0.25 | 20.348477 |  0.232491 |             18.036211 |                 18.036211 |                  18.036211 |                  2.845741 |                      0.0 |     15.19047 |              0.0 |          15.19047 |             40.352465 |                 100.0 |               113.828249 |              100.519671 |    0.0 |  0.0 | 6.829116 |   0.0 | 3.652462 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.021377 | 0.000247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342053 | 66.594402 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000247 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | -0.25 | 18.740175 | -1.067813 |             18.651721 |                 18.651721 |                  18.651721 |                  3.051087 |                      0.0 |    15.600634 |              0.0 |         15.600634 |             37.166744 |                 100.0 |               119.168431 |              102.297453 |    0.0 |  0.0 | 7.945284 |   0.0 | 4.128761 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.011407 | 0.000132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.159866 | 62.465641 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000132 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | -0.25 | 21.466219 |  0.293528 |             17.267745 |                 17.267745 |                  17.267745 |                   2.45312 |                      0.0 |    14.814625 |              0.0 |         14.814625 |             42.798069 |                 100.0 |               125.382697 |              104.231161 |    0.0 |  0.0 | 6.428125 |   0.0 | 3.518477 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.006087 |  0.00007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.616338 | 58.947163 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.00007 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | -0.25 | 19.654459 | -1.371448 |             17.877445 |                 17.877445 |                  17.877445 |                  2.688326 |                      0.0 |    15.189119 |              0.0 |         15.189119 |             78.463126 |                 100.0 |               130.861333 |              105.804695 |    0.0 |  0.0 | 6.036736 |   0.0 | 4.091355 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.003248 | 0.000038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.737786 | 54.855808 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000038 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | -0.25 |  23.39262 |  0.203534 |              17.53355 |                  17.53355 |                   17.53355 |                   2.76572 |                      0.0 |     14.76783 |              0.0 |          14.76783 |             74.333488 |                 100.0 |               137.498449 |               107.59179 |    0.0 |  0.0 | 6.494046 |   0.0 | 4.345849 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.001733 |  0.00002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.284252 | 50.509959 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.00002 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | -0.25 | 21.880839 | -0.632013 |             17.704758 |                 17.704758 |                  17.704758 |                  2.799361 |                      0.0 |    14.905396 |              0.0 |         14.905396 |             47.077876 |                 100.0 |               144.953402 |                109.4495 |    0.0 |  0.0 | 7.914762 |   0.0 | 4.548032 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000925 | 0.000011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.666265 | 45.961927 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000011 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | -0.25 | 23.154216 | -0.023734 |             17.786952 |                 17.786952 |                  17.786952 |                  2.986484 |                      0.0 |    14.800468 |              0.0 |         14.800468 |             56.493451 |                 100.0 |               153.359832 |              111.375646 |    0.0 |  0.0 | 7.566718 | 0.094 | 4.554485 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000494 | 0.000006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.439999 | 41.407442 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000006 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | -0.25 | 22.399998 | -0.414721 |             17.184083 |                 17.184083 |                  17.184083 |                  2.708378 |                      0.0 |    14.475704 |              0.0 |         14.475704 |             88.271017 |                 100.0 |               162.655293 |                113.3188 |    0.0 |  0.0 | 5.996364 |   0.0 | 4.137143 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000263 | 0.000003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.604721 | 37.270299 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | -0.25 | 22.949068 |  0.647517 |             14.968904 |                 14.968904 |                  14.968904 |                  2.239353 |                      0.0 |    12.729551 |              0.0 |         12.729551 |             52.308751 |                 100.0 |               172.213709 |              115.125967 |    0.0 |  0.0 | 5.983925 |   0.0 | 3.428814 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000141 | 0.000002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.707203 | 33.841485 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000002 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | -0.25 | 19.957345 | -0.436832 |             14.206821 |                 14.206821 |                  14.206821 |                  2.021579 |                      0.0 |    12.185243 |              0.0 |         12.185243 |             44.135508 |                 100.0 |               181.325786 |              116.675553 |  1.838 |  0.0 | 5.470984 |   0.7 |  2.50575 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049891 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049891 | 0.012919 |  0.00015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.894036 | 33.123844 | 0.049891 |   0.0 |   0.0 |   0.0 |       0.0 | 0.012919 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.00015 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | -0.25 |  20.58012 |  0.559934 |             14.881951 |                 14.881951 |                  14.881951 |                  2.487815 |                      0.0 |    12.394137 |              0.0 |         12.394137 |            108.641252 |                 100.0 |               183.415178 |              116.867739 |    0.0 |  0.0 |  4.08134 |   0.0 | 2.836595 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.017318 |   0.0002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.084102 | 30.287249 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017318 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |   0.0002 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | -0.25 |  17.88034 | -1.617232 |             15.465744 |                 15.465744 |                  15.465744 |                  2.520953 |                      0.0 |     12.94479 |              0.0 |          12.94479 |             74.333488 |                 100.0 |               192.486718 |              118.223505 |    0.0 |  0.0 | 5.175424 |   0.0 | 3.277008 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.038074 |      0.0 | 0.009173 | 0.000106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.451334 | 27.048315 |      0.0 |   0.0 |   0.0 |   0.0 | -0.038074 | 0.009241 |   0.0 |   0.0 |   0.0 | -0.000068 |   0.0 | 0.000106 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | -0.25 | 22.437779 |  0.434305 |             15.289825 |                 15.289825 |                  15.289825 |                  2.702703 |                      0.0 |    12.587122 |              0.0 |         12.587122 |             70.616813 |                 100.0 |               204.997694 |              119.967773 |    0.0 |  0.0 | 5.738005 |   0.0 | 3.581497 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239328 |      0.0 | 0.004366 | 0.000051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.767029 | 23.706145 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239328 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |   0.0 | 0.000051 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | -0.25 | 20.156763 | -0.488974 |             13.113987 |                 13.113987 |                  13.113987 |                  2.307856 |                      0.0 |    10.806131 |              0.0 |         10.806131 |             54.320626 |                 100.0 |                221.73546 |              122.070817 |    0.0 |  0.0 | 5.867083 |   0.0 | 3.296906 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446997 |      0.0 | 0.000839 |  0.00001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.006003 | 20.856236 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446997 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001793 |   0.0 |  0.00001 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | -0.25 | 20.953343 | -0.749947 |             15.350398 |                 15.350398 |                  15.350398 |                  2.688476 |                      0.0 |    12.661922 |              0.0 |         12.661922 |             39.231563 |                 100.0 |               241.240755 |              124.200232 |    0.0 |  0.0 | 7.829385 |   0.0 | 3.892052 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.624079 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.50595 | 17.588263 |      0.0 |   0.0 |   0.0 |   0.0 | -0.624079 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003705 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | -0.25 | 22.619833 |  0.062121 |             16.161308 |                 16.161308 |                  16.161308 |                  3.308913 |                      0.0 |    12.852395 |              0.0 |         12.852395 |             56.493451 |                 100.0 |               274.870616 |              127.357125 |    0.0 |  0.0 |  7.32284 |   0.0 |  4.18324 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827137 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.193828 | 14.232159 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827137 | 0.000749 |   0.0 |   0.0 |   0.0 | -0.006292 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | -0.25 | 21.579428 |  1.026136 |              5.948207 |                  5.948207 |                   5.948207 |                   0.60663 |                      0.0 |     5.341577 |              0.0 |          5.341577 |             67.254108 |                 100.0 |               337.482269 |              131.866514 |   1.18 |  0.0 |  2.56747 |   0.7 | 1.075772 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013304 |  0.0 |  0.0 | 0.0 | 1.035672 | 0.013304 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.917693 | 15.358755 | 0.013304 |   0.0 |   0.0 |   0.0 | -1.035672 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009606 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | -0.25 | 17.325643 | -1.541907 |             11.290596 |                 11.290596 |                  11.290596 |                  1.641158 |                      0.0 |     9.649438 |              0.0 |          9.649438 |             94.155751 |                 100.0 |               311.359555 |              129.743335 |    0.0 |  0.0 | 3.374286 | 0.282 | 2.014946 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96567 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   33.2096 | 14.309479 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96567 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013157 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | -0.25 | 21.631999 | -1.279412 |             14.219527 |                 14.219527 |                  14.219527 |                  1.868809 |                      0.0 |    12.350718 |              0.0 |         12.350718 |             64.197103 |                 100.0 |               335.458668 |              131.071625 |    0.0 |  0.0 | 5.573178 |   0.0 |  3.36219 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030867 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 34.239011 | 11.978156 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030867 | 0.002572 |   0.0 |   0.0 |   0.0 | -0.016686 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | -0.25 | 25.063372 |  0.706772 |             10.960427 |                 10.960427 |                  10.960427 |                  1.401603 |                      0.0 |     9.558824 |              0.0 |          9.558824 |             74.333488 |                 100.0 |               419.640495 |              135.153792 |    0.0 |  0.0 | 4.467819 | 0.094 | 2.719425 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175726 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.28224 | 10.434457 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175726 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020579 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | -0.25 | 21.874132 |   0.40131 |              9.585999 |                  9.585999 |                   9.585999 |                  1.261689 |                      0.0 |      8.32431 |              0.0 |           8.32431 |             61.405925 |                 100.0 |               522.198911 |              138.404806 |   9.17 |  0.0 | 3.569973 |   0.7 | 1.597438 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104569 |  0.0 |  0.0 | 0.0 | 1.271645 | 0.104569 | 0.002764 | 0.000032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.63093 | 19.174095 | 0.104569 |   0.0 |   0.0 |   0.0 | -1.271645 | 0.027653 |   0.0 |   0.0 |   0.0 |  -0.02489 |   0.0 | 0.000032 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | -0.25 | 19.703098 | -0.620474 |             10.870274 |                 10.870274 |                  10.870274 |                  1.498964 |                      0.0 |      9.37131 |              0.0 |           9.37131 |             67.254108 |                 100.0 |               256.494997 |              123.965055 |   5.41 |  0.0 | 3.567075 |   0.7 | 1.722338 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090739 |  0.0 |  0.0 | 0.0 |   0.7286 | 0.090739 | 0.031583 | 0.000366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.001403 | 23.499618 | 0.090739 |   0.0 |   0.0 |   0.0 |   -0.7286 | 0.059966 |   0.0 |   0.0 |   0.0 | -0.028383 |   0.0 | 0.000366 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | -0.25 | 20.938011 |  0.382585 |              5.758296 |                  5.758296 |                   5.758296 |                  0.589077 |                      0.0 |     5.169219 |              0.0 |          5.169219 |             54.320626 |                 100.0 |                222.95161 |              120.113963 |   13.4 |  0.0 | 2.621374 |   0.7 | 1.072031 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312931 |  0.0 |  0.0 | 0.0 |  0.45983 | 0.312931 | 0.101111 |  0.00117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.368818 | 35.974485 | 0.312931 |   0.0 |   0.0 |   0.0 |  -0.45983 | 0.131521 |   0.0 |   0.0 |   0.0 |  -0.03041 |   0.0 |  0.00117 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | -0.25 | 18.829395 |  1.414918 |               5.53921 |                   5.53921 |                    5.53921 |                  0.708412 |                      0.0 |     4.830799 |              0.0 |          4.830799 |             54.320626 |                 100.0 |                 175.5055 |              113.153351 |  7.948 |  0.0 | 2.027722 |   0.7 | 0.717235 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249156 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249156 | 0.168586 | 0.001951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   30.7039 | 42.956093 | 0.249156 |   0.0 |   0.0 |   0.0 |       0.0 |  0.19971 |   0.0 |   0.0 |   0.0 | -0.031124 |   0.0 | 0.001951 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | -0.25 | 13.279667 |  -1.92862 |              9.344591 |                  9.344591 |                   9.344591 |                  1.881434 |                      0.0 |     7.463158 |              0.0 |          7.463158 |             67.254108 |                 100.0 |               159.374248 |              110.061771 |    0.0 |  0.0 | 3.393609 |   0.0 | 2.035968 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.127614 | 0.001477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.382521 | 40.920125 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158627 |   0.0 |   0.0 |   0.0 | -0.031013 |   0.0 | 0.001477 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | -0.25 | 18.875069 |  0.349956 |              8.321079 |                  8.321079 |                   8.321079 |                  1.511299 |                      0.0 |      6.80978 |              0.0 |           6.80978 |             67.254108 |                 100.0 |               163.717417 |              110.731287 |    0.0 |  0.0 | 3.322101 |   0.0 | 2.018298 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.053743 | 0.000622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.782565 | 38.901827 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084645 |   0.0 |   0.0 |   0.0 | -0.030902 |   0.0 | 0.000622 |

.. _lland_v3_acker_routing_daily:

:ref:`acre (routing) <lland_v1_acker_routing>`
----------------------------------------------

The following calculation shows the possibility to route inflowing runoff,
discussed in the documentation of |lland_v1|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_v3_acker_routing_daily",
    ...     axis1=(inputs.nied, fluxes.qah),
    ...     axis2=states.bowa)
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |   qz |   qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |     g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |     evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |      ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.02 | 1.728 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | -0.25 | 10.933333 |   -2.6784 |             12.492674 |                 12.492674 |                  12.492674 |                  1.877708 |                      0.0 |    10.614966 |              0.0 |         10.614966 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 11.802 |  0.0 | 3.623966 |   0.7 | 1.720207 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 1.924005 | 0.022269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   32.4284 |  81.32045 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.022269 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.02 | 1.728 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | -0.25 |    19.028 |  0.895795 |              8.382718 |                  8.382718 |                   8.382718 |                   0.96939 |                      0.0 |     7.413327 |              0.0 |          7.413327 |             88.271017 |                 100.0 |                98.302128 |               94.620659 |    0.0 |  0.0 | 2.555713 | 0.094 | 1.697264 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.991667 | 0.023052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.282605 | 79.623186 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.023052 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.02 | 1.728 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | -0.25 | 15.208683 | -1.913579 |             10.795247 |                 10.795247 |                  10.795247 |                  1.386217 |                      0.0 |      9.40903 |              0.0 |           9.40903 |             94.155751 |                 100.0 |               100.629116 |               95.574305 |    0.0 |  0.0 | 3.286521 |   0.0 | 2.329549 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.868696 | 0.021628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.946184 | 77.293637 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.021628 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.02 | 1.728 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | -0.25 | 20.753946 |  1.531325 |             17.664962 |                 17.664962 |                  17.664962 |                  3.108707 |                      0.0 |    14.556255 |              0.0 |         14.556255 |             50.440581 |                 100.0 |               103.850807 |               96.853335 |    0.0 |  0.0 | 5.847436 | 0.094 | 3.314416 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.803077 | 0.020869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.164858 | 73.979222 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020869 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.02 | 1.728 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | -0.25 | 14.816194 | -1.909685 |             18.901295 |                 18.901295 |                  18.901295 |                  3.414868 |                      0.0 |    15.486427 |              0.0 |         15.486427 |             47.077876 |                 100.0 |               108.496391 |               98.617204 |    0.0 |  0.0 | 6.638616 |   0.0 | 3.732358 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.768062 | 0.020464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.824543 | 70.246864 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020464 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.02 | 1.728 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | -0.25 | 20.348477 |  0.232491 |             18.036211 |                 18.036211 |                  18.036211 |                  2.845741 |                      0.0 |     15.19047 |              0.0 |          15.19047 |             40.352465 |                 100.0 |               113.828249 |              100.519671 |    0.0 |  0.0 | 6.829116 |   0.0 | 3.652462 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.749377 | 0.020247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342053 | 66.594402 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020247 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.02 | 1.728 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | -0.25 | 18.740175 | -1.067813 |             18.651721 |                 18.651721 |                  18.651721 |                  3.051087 |                      0.0 |    15.600634 |              0.0 |         15.600634 |             37.166744 |                 100.0 |               119.168431 |              102.297453 |    0.0 |  0.0 | 7.945284 |   0.0 | 4.128761 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.739407 | 0.020132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.159866 | 62.465641 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020132 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.02 | 1.728 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | -0.25 | 21.466219 |  0.293528 |             17.267745 |                 17.267745 |                  17.267745 |                   2.45312 |                      0.0 |    14.814625 |              0.0 |         14.814625 |             42.798069 |                 100.0 |               125.382697 |              104.231161 |    0.0 |  0.0 | 6.428125 |   0.0 | 3.518477 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.734087 |  0.02007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.616338 | 58.947163 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02007 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.02 | 1.728 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | -0.25 | 19.654459 | -1.371448 |             17.877445 |                 17.877445 |                  17.877445 |                  2.688326 |                      0.0 |    15.189119 |              0.0 |         15.189119 |             78.463126 |                 100.0 |               130.861333 |              105.804695 |    0.0 |  0.0 | 6.036736 |   0.0 | 4.091355 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.731248 | 0.020038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.737786 | 54.855808 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020038 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.02 | 1.728 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | -0.25 |  23.39262 |  0.203534 |              17.53355 |                  17.53355 |                   17.53355 |                   2.76572 |                      0.0 |     14.76783 |              0.0 |          14.76783 |             74.333488 |                 100.0 |               137.498449 |               107.59179 |    0.0 |  0.0 | 6.494046 |   0.0 | 4.345849 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.729733 |  0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.284252 | 50.509959 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02002 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.02 | 1.728 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | -0.25 | 21.880839 | -0.632013 |             17.704758 |                 17.704758 |                  17.704758 |                  2.799361 |                      0.0 |    14.905396 |              0.0 |         14.905396 |             47.077876 |                 100.0 |               144.953402 |                109.4495 |    0.0 |  0.0 | 7.914762 |   0.0 | 4.548032 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728925 | 0.020011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.666265 | 45.961927 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020011 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.02 | 1.728 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | -0.25 | 23.154216 | -0.023734 |             17.786952 |                 17.786952 |                  17.786952 |                  2.986484 |                      0.0 |    14.800468 |              0.0 |         14.800468 |             56.493451 |                 100.0 |               153.359832 |              111.375646 |    0.0 |  0.0 | 7.566718 | 0.094 | 4.554485 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728494 | 0.020006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.439999 | 41.407442 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020006 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.02 | 1.728 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | -0.25 | 22.399998 | -0.414721 |             17.184083 |                 17.184083 |                  17.184083 |                  2.708378 |                      0.0 |    14.475704 |              0.0 |         14.475704 |             88.271017 |                 100.0 |               162.655293 |                113.3188 |    0.0 |  0.0 | 5.996364 |   0.0 | 4.137143 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728263 | 0.020003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.604721 | 37.270299 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.02 | 1.728 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | -0.25 | 22.949068 |  0.647517 |             14.968904 |                 14.968904 |                  14.968904 |                  2.239353 |                      0.0 |    12.729551 |              0.0 |         12.729551 |             52.308751 |                 100.0 |               172.213709 |              115.125967 |    0.0 |  0.0 | 5.983925 |   0.0 | 3.428814 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728141 | 0.020002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.707203 | 33.841485 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020002 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.02 | 1.728 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | -0.25 | 19.957345 | -0.436832 |             14.206821 |                 14.206821 |                  14.206821 |                  2.021579 |                      0.0 |    12.185243 |              0.0 |         12.185243 |             44.135508 |                 100.0 |               181.325786 |              116.675553 |  1.838 |  0.0 | 5.470984 |   0.7 |  2.50575 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049891 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049891 | 1.740919 |  0.02015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.894036 | 33.123844 | 0.049891 |   0.0 |   0.0 |   0.0 |       0.0 | 0.012919 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02015 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.02 | 1.728 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | -0.25 |  20.58012 |  0.559934 |             14.881951 |                 14.881951 |                  14.881951 |                  2.487815 |                      0.0 |    12.394137 |              0.0 |         12.394137 |            108.641252 |                 100.0 |               183.415178 |              116.867739 |    0.0 |  0.0 |  4.08134 |   0.0 | 2.836595 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.745318 |   0.0202 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.084102 | 30.287249 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017318 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |   0.0202 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.02 | 1.728 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | -0.25 |  17.88034 | -1.617232 |             15.465744 |                 15.465744 |                  15.465744 |                  2.520953 |                      0.0 |     12.94479 |              0.0 |          12.94479 |             74.333488 |                 100.0 |               192.486718 |              118.223505 |    0.0 |  0.0 | 5.175424 |   0.0 | 3.277008 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.038074 |      0.0 | 1.737173 | 0.020106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.451334 | 27.048315 |      0.0 |   0.0 |   0.0 |   0.0 | -0.038074 | 0.009241 |   0.0 |   0.0 |   0.0 | -0.000068 |  0.02 | 0.020106 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.02 | 1.728 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | -0.25 | 22.437779 |  0.434305 |             15.289825 |                 15.289825 |                  15.289825 |                  2.702703 |                      0.0 |    12.587122 |              0.0 |         12.587122 |             70.616813 |                 100.0 |               204.997694 |              119.967773 |    0.0 |  0.0 | 5.738005 |   0.0 | 3.581497 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239328 |      0.0 | 1.732366 | 0.020051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.767029 | 23.706145 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239328 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |  0.02 | 0.020051 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.02 | 1.728 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | -0.25 | 20.156763 | -0.488974 |             13.113987 |                 13.113987 |                  13.113987 |                  2.307856 |                      0.0 |    10.806131 |              0.0 |         10.806131 |             54.320626 |                 100.0 |                221.73546 |              122.070817 |    0.0 |  0.0 | 5.867083 |   0.0 | 3.296906 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446997 |      0.0 | 1.728839 |  0.02001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.006003 | 20.856236 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446997 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001793 |  0.02 |  0.02001 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.02 | 1.728 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | -0.25 | 20.953343 | -0.749947 |             15.350398 |                 15.350398 |                  15.350398 |                  2.688476 |                      0.0 |    12.661922 |              0.0 |         12.661922 |             39.231563 |                 100.0 |               241.240755 |              124.200232 |    0.0 |  0.0 | 7.829385 |   0.0 | 3.892052 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.624079 |      0.0 | 1.725699 | 0.019973 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.50595 | 17.588263 |      0.0 |   0.0 |   0.0 |   0.0 | -0.624079 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003705 |  0.02 | 0.019973 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.02 | 1.728 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | -0.25 | 22.619833 |  0.062121 |             16.161308 |                 16.161308 |                  16.161308 |                  3.308913 |                      0.0 |    12.852395 |              0.0 |         12.852395 |             56.493451 |                 100.0 |               274.870616 |              127.357125 |    0.0 |  0.0 |  7.32284 |   0.0 |  4.18324 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827137 |      0.0 | 1.722458 | 0.019936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.193828 | 14.232159 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827137 | 0.000749 |   0.0 |   0.0 |   0.0 | -0.006292 |  0.02 | 0.019936 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.02 | 1.728 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | -0.25 | 21.579428 |  1.026136 |              5.948207 |                  5.948207 |                   5.948207 |                   0.60663 |                      0.0 |     5.341577 |              0.0 |          5.341577 |             67.254108 |                 100.0 |               337.482269 |              131.866514 |   1.18 |  0.0 |  2.56747 |   0.7 | 1.075772 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013304 |  0.0 |  0.0 | 0.0 | 1.035672 | 0.013304 | 1.722219 | 0.019933 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.917693 | 15.358755 | 0.013304 |   0.0 |   0.0 |   0.0 | -1.035672 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009606 |  0.02 | 0.019933 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.02 | 1.728 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | -0.25 | 17.325643 | -1.541907 |             11.290596 |                 11.290596 |                  11.290596 |                  1.641158 |                      0.0 |     9.649438 |              0.0 |          9.649438 |             94.155751 |                 100.0 |               311.359555 |              129.743335 |    0.0 |  0.0 | 3.374286 | 0.282 | 2.014946 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96567 |      0.0 | 1.719664 | 0.019904 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   33.2096 | 14.309479 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96567 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013157 |  0.02 | 0.019904 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.02 | 1.728 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | -0.25 | 21.631999 | -1.279412 |             14.219527 |                 14.219527 |                  14.219527 |                  1.868809 |                      0.0 |    12.350718 |              0.0 |         12.350718 |             64.197103 |                 100.0 |               335.458668 |              131.071625 |    0.0 |  0.0 | 5.573178 |   0.0 |  3.36219 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030867 |      0.0 | 1.713886 | 0.019837 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 34.239011 | 11.978156 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030867 | 0.002572 |   0.0 |   0.0 |   0.0 | -0.016686 |  0.02 | 0.019837 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.02 | 1.728 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | -0.25 | 25.063372 |  0.706772 |             10.960427 |                 10.960427 |                  10.960427 |                  1.401603 |                      0.0 |     9.558824 |              0.0 |          9.558824 |             74.333488 |                 100.0 |               419.640495 |              135.153792 |    0.0 |  0.0 | 4.467819 | 0.094 | 2.719425 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175726 |      0.0 | 1.708793 | 0.019778 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.28224 | 10.434457 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175726 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020579 |  0.02 | 0.019778 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.02 | 1.728 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | -0.25 | 21.874132 |   0.40131 |              9.585999 |                  9.585999 |                   9.585999 |                  1.261689 |                      0.0 |      8.32431 |              0.0 |           8.32431 |             61.405925 |                 100.0 |               522.198911 |              138.404806 |   9.17 |  0.0 | 3.569973 |   0.7 | 1.597438 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104569 |  0.0 |  0.0 | 0.0 | 1.271645 | 0.104569 | 1.730764 | 0.020032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.63093 | 19.174095 | 0.104569 |   0.0 |   0.0 |   0.0 | -1.271645 | 0.027653 |   0.0 |   0.0 |   0.0 |  -0.02489 |  0.02 | 0.020032 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.02 | 1.728 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | -0.25 | 19.703098 | -0.620474 |             10.870274 |                 10.870274 |                  10.870274 |                  1.498964 |                      0.0 |      9.37131 |              0.0 |           9.37131 |             67.254108 |                 100.0 |               256.494997 |              123.965055 |   5.41 |  0.0 | 3.567075 |   0.7 | 1.722338 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090739 |  0.0 |  0.0 | 0.0 |   0.7286 | 0.090739 | 1.759583 | 0.020366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.001403 | 23.499618 | 0.090739 |   0.0 |   0.0 |   0.0 |   -0.7286 | 0.059966 |   0.0 |   0.0 |   0.0 | -0.028383 |  0.02 | 0.020366 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.02 | 1.728 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | -0.25 | 20.938011 |  0.382585 |              5.758296 |                  5.758296 |                   5.758296 |                  0.589077 |                      0.0 |     5.169219 |              0.0 |          5.169219 |             54.320626 |                 100.0 |                222.95161 |              120.113963 |   13.4 |  0.0 | 2.621374 |   0.7 | 1.072031 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312931 |  0.0 |  0.0 | 0.0 |  0.45983 | 0.312931 | 1.829111 |  0.02117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.368818 | 35.974485 | 0.312931 |   0.0 |   0.0 |   0.0 |  -0.45983 | 0.131521 |   0.0 |   0.0 |   0.0 |  -0.03041 |  0.02 |  0.02117 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.02 | 1.728 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | -0.25 | 18.829395 |  1.414918 |               5.53921 |                   5.53921 |                    5.53921 |                  0.708412 |                      0.0 |     4.830799 |              0.0 |          4.830799 |             54.320626 |                 100.0 |                 175.5055 |              113.153351 |  7.948 |  0.0 | 2.027722 |   0.7 | 0.717235 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249156 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249156 | 1.896586 | 0.021951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   30.7039 | 42.956093 | 0.249156 |   0.0 |   0.0 |   0.0 |       0.0 |  0.19971 |   0.0 |   0.0 |   0.0 | -0.031124 |  0.02 | 0.021951 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.02 | 1.728 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | -0.25 | 13.279667 |  -1.92862 |              9.344591 |                  9.344591 |                   9.344591 |                  1.881434 |                      0.0 |     7.463158 |              0.0 |          7.463158 |             67.254108 |                 100.0 |               159.374248 |              110.061771 |    0.0 |  0.0 | 3.393609 |   0.0 | 2.035968 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.855614 | 0.021477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.382521 | 40.920125 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158627 |   0.0 |   0.0 |   0.0 | -0.031013 |  0.02 | 0.021477 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.02 | 1.728 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | -0.25 | 18.875069 |  0.349956 |              8.321079 |                  8.321079 |                   8.321079 |                  1.511299 |                      0.0 |      6.80978 |              0.0 |           6.80978 |             67.254108 |                 100.0 |               163.717417 |              110.731287 |    0.0 |  0.0 | 3.322101 |   0.0 | 2.018298 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.781743 | 0.020622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.782565 | 38.901827 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084645 |   0.0 |   0.0 |   0.0 | -0.030902 |  0.02 | 0.020622 |

.. _lland_v3_acker_heavy_rain_daily:

acre (heavy rain)
-----------------

Integration test :ref:`lland_v3_acker_summer_daily` deals with dry summer
conditions.  To show how |lland_v3| works for warm but wet conditions,
we set all values of the precipitation input time series to 20 mm.
Now, the soil water content (|BoWa|) rises from its initial value of
72 mm and nearly reaches its maximum value of 309 mm (|WMax|), resulting
in a relevant response of all runoff components:

.. integration-test::

    >>> lnk(ACKER)
    >>> inlet.sequences.sim.series = 0.0
    >>> nied = inputs.nied.series.copy()
    >>> inputs.nied.series = 20.0
    >>> conditions_acker_heavy_rain = test(
    ...     "lland_v3_acker_heavy_rain_daily",
    ...     axis1=(inputs.nied, fluxes.qah),
    ...     axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |     g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |       qah |       qa | inzp | wats | waes | esnow | taus |      ebdn |       bowa |     qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |     qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 18.8 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | -0.25 | 10.933333 |   -2.6784 |             12.492674 |                 12.492674 |                  12.492674 |                  1.877708 |                      0.0 |    10.614966 |              0.0 |         10.614966 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 18.1 |  0.0 | 3.623966 | 0.7 | 1.720207 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.217805 |      0.0 |      0.0 |      0.0 |  0.0 |  1.217805 |   0.31352 | 0.003629 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   32.4284 |  87.161988 |  1.217805 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 | 18.8 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | -0.25 |    19.028 |  0.895795 |              8.382718 |                  8.382718 |                   8.382718 |                   0.96939 |                      0.0 |     7.413327 |              0.0 |          7.413327 |             88.271017 |                 100.0 |                90.407167 |               91.066405 | 18.1 |  0.0 | 2.555713 | 0.7 | 1.294549 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.480868 |      0.0 |      0.0 |      0.0 |  0.0 |  1.480868 |  0.802993 | 0.009294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.282605 |  102.48657 |  1.480868 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802993 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 | 18.8 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | -0.25 | 15.208683 | -1.913579 |             10.795247 |                 10.795247 |                  10.795247 |                  1.386217 |                      0.0 |      9.40903 |              0.0 |           9.40903 |             94.155751 |                 100.0 |                70.327061 |                80.36502 | 18.1 |  0.0 | 3.286521 | 0.7 | 1.922457 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.761522 |      0.0 |      0.0 |      0.0 |  0.0 |  1.761522 |    1.1914 | 0.013789 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.946184 | 116.902592 |  1.761522 |      0.0 |      0.0 |      0.0 |      0.0 |    1.1914 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 | 18.8 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | -0.25 | 20.753946 |  1.531325 |             17.664962 |                 17.664962 |                  17.664962 |                  3.108707 |                      0.0 |    14.556255 |              0.0 |         14.556255 |             50.440581 |                 100.0 |                51.993585 |               67.804653 | 18.1 |  0.0 | 5.847436 | 0.7 | 3.397259 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.040913 |      0.0 |      0.0 |      0.0 |  0.0 |  2.040913 |  1.529227 | 0.017699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.164858 |  129.56442 |  2.040913 |      0.0 |      0.0 |      0.0 |      0.0 |  1.529227 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 | 18.8 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | -0.25 | 14.816194 | -1.909685 |             18.901295 |                 18.901295 |                  18.901295 |                  3.414868 |                      0.0 |    15.486427 |              0.0 |         15.486427 |             47.077876 |                 100.0 |                 48.85611 |               65.355461 | 18.1 |  0.0 | 6.638616 | 0.7 | 3.917199 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.300279 | 0.101438 |      0.0 | 0.010164 |  0.0 |  2.300279 |  1.849046 | 0.021401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.824543 |  141.33534 |  2.300279 |      0.0 | 0.101438 |      0.0 | 0.010164 |  1.834644 |      0.0 | 0.014384 |      0.0 | 0.000018 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 | 18.8 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | -0.25 | 20.348477 |  0.232491 |             18.036211 |                 18.036211 |                  18.036211 |                  2.845741 |                      0.0 |     15.19047 |              0.0 |          15.19047 |             40.352465 |                 100.0 |                 48.85611 |               65.425338 | 18.1 |  0.0 | 6.829116 | 0.7 | 3.913663 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.554684 | 0.110653 |      0.0 | 0.021935 |  0.0 |  2.554684 |  2.156535 |  0.02496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342053 | 152.834404 |  2.554684 |      0.0 | 0.110653 |      0.0 | 0.021935 |  2.117307 |      0.0 | 0.039153 |      0.0 | 0.000076 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 | 18.8 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | -0.25 | 18.740175 | -1.067813 |             18.651721 |                 18.651721 |                  18.651721 |                  3.051087 |                      0.0 |    15.600634 |              0.0 |         15.600634 |             37.166744 |                 100.0 |                 48.85611 |               65.496024 | 18.1 |  0.0 | 7.945284 | 0.7 | 4.551539 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.817223 | 0.119656 |      0.0 | 0.033434 |  0.0 |  2.817223 |  2.448759 | 0.028342 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.159866 | 163.412552 |  2.817223 |      0.0 | 0.119656 |      0.0 | 0.033434 |  2.388884 |      0.0 |   0.0597 |      0.0 | 0.000175 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 | 18.8 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | -0.25 | 21.466219 |  0.293528 |             17.267745 |                 17.267745 |                  17.267745 |                   2.45312 |                      0.0 |    14.814625 |              0.0 |         14.814625 |             42.798069 |                 100.0 |                 48.85611 |               65.567492 | 18.1 |  0.0 | 6.428125 | 0.7 | 3.767989 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.072608 | 0.127938 |      0.0 | 0.044013 |  0.0 |  3.072608 |   2.73175 | 0.031617 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.616338 | 174.500005 |  3.072608 |      0.0 | 0.127938 |      0.0 | 0.044013 |  2.654405 |      0.0 | 0.077033 |      0.0 | 0.000313 |   0.0 | 0.031617 |
    | 1997-09-08 | 20.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 | 18.8 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | -0.25 | 19.654459 | -1.371448 |             17.877445 |                 17.877445 |                  17.877445 |                  2.688326 |                      0.0 |    15.189119 |              0.0 |         15.189119 |             78.463126 |                 100.0 |                 48.85611 |               65.639719 | 18.1 |  0.0 | 6.036736 | 0.7 | 4.121081 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.356591 | 0.136618 |      0.0 |   0.0551 |  0.0 |  3.356591 |  3.015033 | 0.034896 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.737786 | 184.930615 |  3.356591 |      0.0 | 0.136618 |      0.0 |   0.0551 |   2.92256 |      0.0 | 0.091983 |      0.0 | 0.000489 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 | 18.8 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | -0.25 |  23.39262 |  0.203534 |              17.53355 |                  17.53355 |                   17.53355 |                   2.76572 |                      0.0 |     14.76783 |              0.0 |          14.76783 |             74.333488 |                 100.0 |                 48.85611 |               65.712681 | 18.1 |  0.0 | 6.494046 | 0.7 | 4.450439 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.641259 | 0.144785 |      0.0 | 0.065531 |  0.0 |  3.641259 |  3.304149 | 0.038242 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.284252 | 194.728603 |  3.641259 |      0.0 | 0.144785 |      0.0 | 0.065531 |  3.198274 |      0.0 | 0.105171 |      0.0 | 0.000703 |   0.0 | 0.038242 |
    | 1997-11-08 | 20.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 | 18.8 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | -0.25 | 21.880839 | -0.632013 |             17.704758 |                 17.704758 |                  17.704758 |                  2.799361 |                      0.0 |    14.905396 |              0.0 |         14.905396 |             47.077876 |                 100.0 |                 48.85611 |               65.786357 | 18.1 |  0.0 | 7.914762 | 0.7 | 4.993103 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.926571 | 0.152455 |      0.0 | 0.075329 |  0.0 |  3.926571 |  3.596218 | 0.041623 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.666265 | 203.681145 |  3.926571 |      0.0 | 0.152455 |      0.0 | 0.075329 |   3.47833 |      0.0 | 0.116935 |      0.0 | 0.000953 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 | 18.8 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | -0.25 | 23.154216 | -0.023734 |             17.786952 |                 17.786952 |                  17.786952 |                  2.986484 |                      0.0 |    14.800468 |              0.0 |         14.800468 |             56.493451 |                 100.0 |                 48.85611 |               65.860724 | 18.1 |  0.0 | 7.566718 | 0.7 | 4.979876 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.204866 | 0.159465 | 0.009557 | 0.084281 |  0.0 |  4.204866 |  3.889124 | 0.045013 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.439999 |   212.3431 |  4.204866 |      0.0 | 0.159465 | 0.009557 | 0.084281 |  3.759031 |      0.0 | 0.127502 | 0.001355 | 0.001236 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 | 18.8 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | -0.25 | 22.399998 | -0.414721 |             17.184083 |                 17.184083 |                  17.184083 |                  2.708378 |                      0.0 |    14.475704 |              0.0 |         14.475704 |             88.271017 |                 100.0 |                 48.85611 |               65.935761 | 18.1 |  0.0 | 5.996364 | 0.7 |  4.19851 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.492755 | 0.166246 | 0.045992 | 0.092943 |  0.0 |  4.492755 |  4.188438 | 0.048477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.604721 | 221.446655 |  4.492755 |      0.0 | 0.166246 | 0.045992 | 0.092943 |  4.041079 |      0.0 | 0.137078 | 0.008732 | 0.001549 |   0.0 | 0.048477 |
    | 1997-14-08 | 20.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 | 18.8 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | -0.25 | 22.949068 |  0.647517 |             14.968904 |                 14.968904 |                  14.968904 |                  2.239353 |                      0.0 |    12.729551 |              0.0 |         12.729551 |             52.308751 |                 100.0 |                 48.85611 |               66.011448 | 18.1 |  0.0 | 5.983925 | 0.7 | 3.702096 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.818552 | 0.173373 | 0.100351 | 0.102047 |  0.0 |  4.818552 |  4.509936 | 0.052198 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.707203 | 230.650235 |  4.818552 |      0.0 | 0.173373 | 0.100351 | 0.102047 |  4.335611 |      0.0 |  0.14595 | 0.026482 | 0.001893 |   0.0 | 0.052198 |
    | 1997-15-08 | 20.0 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 | 18.8 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | -0.25 | 19.957345 | -0.436832 |             14.206821 |                 14.206821 |                  14.206821 |                  2.021579 |                      0.0 |    12.185243 |              0.0 |         12.185243 |             44.135508 |                 100.0 |                 48.85611 |               66.087764 | 18.1 |  0.0 | 5.470984 | 0.7 | 3.155298 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.176929 | 0.180579 |  0.16802 |  0.11125 |  0.0 |  5.176929 |  4.865729 | 0.056316 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.894036 | 239.958158 |  5.176929 |      0.0 | 0.180579 |  0.16802 |  0.11125 |  4.653113 |      0.0 | 0.154363 | 0.055986 | 0.002268 |   0.0 | 0.056316 |
    | 1997-16-08 | 20.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 | 18.8 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | -0.25 |  20.58012 |  0.559934 |             14.881951 |                 14.881951 |                  14.881951 |                  2.487815 |                      0.0 |    12.394137 |              0.0 |         12.394137 |            108.641252 |                 100.0 |                 48.85611 |                66.16469 | 18.1 |  0.0 |  4.08134 | 0.7 | 2.708461 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.575829 | 0.187866 | 0.247347 | 0.120558 |  0.0 |  5.575829 |  5.262676 | 0.060911 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.084102 | 249.218097 |  5.575829 |      0.0 | 0.187866 | 0.247347 | 0.120558 |   5.00011 |      0.0 | 0.162462 | 0.097429 | 0.002675 |   0.0 | 0.060911 |
    | 1997-17-08 | 20.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 | 18.8 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | -0.25 |  17.88034 | -1.617232 |             15.465744 |                 15.465744 |                  15.465744 |                  2.520953 |                      0.0 |     12.94479 |              0.0 |          12.94479 |             74.333488 |                 100.0 |                 48.85611 |               66.242207 | 18.1 |  0.0 | 5.175424 | 0.7 | 3.378706 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.018699 | 0.195116 | 0.335804 | 0.129818 |  0.0 |  6.018699 |  5.706462 | 0.066047 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.451334 | 257.259955 |  6.018699 |      0.0 | 0.195116 | 0.335804 | 0.129818 |  5.382634 |      0.0 | 0.170336 | 0.150377 | 0.003114 |   0.0 | 0.066047 |
    | 1997-18-08 | 20.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 | 18.8 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | -0.25 | 22.437779 |  0.434305 |             15.289825 |                 15.289825 |                  15.289825 |                  2.702703 |                      0.0 |    12.587122 |              0.0 |         12.587122 |             70.616813 |                 100.0 |                 48.85611 |               66.320298 | 18.1 |  0.0 | 5.738005 | 0.7 | 3.779831 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.451503 | 0.201412 | 0.419614 |  0.13786 |  0.0 |  6.451503 |  6.184438 | 0.071579 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.767029 | 264.369735 |  6.451503 |      0.0 | 0.201412 | 0.419614 |  0.13786 |  5.790712 |      0.0 | 0.177908 | 0.212236 | 0.003582 |   0.0 | 0.071579 |
    | 1997-19-08 | 20.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 | 18.8 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | -0.25 | 20.156763 | -0.488974 |             13.113987 |                 13.113987 |                  13.113987 |                  2.307856 |                      0.0 |    10.806131 |              0.0 |         10.806131 |             54.320626 |                 100.0 |                 48.85611 |               66.398945 | 18.1 |  0.0 | 5.867083 | 0.7 |  3.62847 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.883173 | 0.206978 | 0.498724 |  0.14497 |  0.0 |  6.883173 |  6.678482 | 0.077297 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.006003 | 271.107419 |  6.883173 |      0.0 | 0.206978 | 0.498724 |  0.14497 |  6.210029 |      0.0 | 0.185032 | 0.279345 | 0.004076 |   0.0 | 0.077297 |
    | 1997-20-08 | 20.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 | 18.8 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | -0.25 | 20.953343 | -0.749947 |             15.350398 |                 15.350398 |                  15.350398 |                  2.688476 |                      0.0 |    12.661922 |              0.0 |         12.661922 |             39.231563 |                 100.0 |                 48.85611 |                66.47813 | 18.1 |  0.0 | 7.829385 | 0.7 | 4.625035 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.348865 | 0.212253 | 0.577784 | 0.151707 |  0.0 |  7.348865 |  7.189836 | 0.083216 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.50595 | 276.291774 |  7.348865 |      0.0 | 0.212253 | 0.577784 | 0.151707 |  6.643867 |      0.0 | 0.191695 | 0.349681 | 0.004593 |   0.0 | 0.083216 |
    | 1997-21-08 | 20.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 | 18.8 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | -0.25 | 22.619833 |  0.062121 |             16.161308 |                 16.161308 |                  16.161308 |                  3.308913 |                      0.0 |    12.852395 |              0.0 |         12.852395 |             56.493451 |                 100.0 |                 48.85611 |               66.557838 | 18.1 |  0.0 |  7.32284 | 0.7 | 4.757012 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.757458 | 0.216312 | 0.641202 | 0.156892 |  0.0 |  7.757458 |  7.700952 | 0.089131 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.193828 | 280.862897 |  7.757458 |      0.0 | 0.216312 | 0.641202 | 0.156892 |  7.077861 |      0.0 | 0.197811 |  0.42015 | 0.005129 |   0.0 | 0.089131 |
    | 1997-22-08 | 20.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 | 18.8 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | -0.25 | 21.579428 |  1.026136 |              5.948207 |                  5.948207 |                   5.948207 |                   0.60663 |                      0.0 |     5.341577 |              0.0 |          5.341577 |             67.254108 |                 100.0 |                 48.85611 |               66.638052 | 18.1 |  0.0 |  2.56747 | 0.7 | 1.361228 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.166601 | 0.219891 | 0.698918 | 0.161463 |  0.0 |  8.166601 |  8.197047 | 0.094873 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.917693 | 288.354796 |  8.166601 |      0.0 | 0.219891 | 0.698918 | 0.161463 |   7.50015 |      0.0 | 0.203305 | 0.487911 | 0.005681 |   0.0 | 0.094873 |
    | 1997-23-08 | 20.0 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 | 18.8 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | -0.25 | 17.325643 | -1.541907 |             11.290596 |                 11.290596 |                  11.290596 |                  1.641158 |                      0.0 |     9.649438 |              0.0 |          9.649438 |             94.155751 |                 100.0 |                 48.85611 |               66.718758 | 18.1 |  0.0 | 3.374286 | 0.7 | 2.097563 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.984192 | 0.225757 | 0.797025 | 0.168955 |  0.0 |  8.984192 |  8.795013 | 0.101794 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   33.2096 | 294.181304 |  8.984192 |      0.0 | 0.225757 | 0.797025 | 0.168955 |  8.021462 |      0.0 | 0.208607 | 0.558691 | 0.006253 |   0.0 | 0.101794 |
    | 1997-24-08 | 20.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 | 18.8 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | -0.25 | 21.631999 | -1.279412 |             14.219527 |                 14.219527 |                  14.219527 |                  1.868809 |                      0.0 |    12.350718 |              0.0 |         12.350718 |             64.197103 |                 100.0 |                 48.85611 |                66.79994 | 18.1 |  0.0 | 5.573178 | 0.7 | 3.649927 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  9.834859 | 0.230318 | 0.876239 | 0.174781 |  0.0 |  9.834859 |   9.54435 | 0.110467 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 34.239011 | 297.515179 |  9.834859 |      0.0 | 0.230318 | 0.876239 | 0.174781 |   8.68947 |      0.0 | 0.213876 | 0.634158 | 0.006846 |   0.0 | 0.110467 |
    | 1997-25-08 | 20.0 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 | 18.8 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | -0.25 | 25.063372 |  0.706772 |             10.960427 |                 10.960427 |                  10.960427 |                  1.401603 |                      0.0 |     9.558824 |              0.0 |          9.558824 |             74.333488 |                 100.0 |                 48.85611 |               66.881585 | 18.1 |  0.0 | 4.467819 | 0.7 | 2.895981 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 10.482817 | 0.232928 | 0.922678 | 0.178115 |  0.0 | 10.482817 | 10.322599 | 0.119475 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.28224 |  300.90266 | 10.482817 |      0.0 | 0.232928 | 0.922678 | 0.178115 |  9.390481 |      0.0 | 0.218677 | 0.705987 | 0.007454 |   0.0 | 0.119475 |
    | 1997-26-08 | 20.0 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 | 18.8 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | -0.25 | 21.874132 |   0.40131 |              9.585999 |                  9.585999 |                   9.585999 |                  1.261689 |                      0.0 |      8.32431 |              0.0 |           8.32431 |             61.405925 |                 100.0 |                 48.85611 |               66.963678 | 18.1 |  0.0 | 3.569973 | 0.7 | 2.071555 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  11.39404 |  0.23558 | 0.970674 | 0.181503 |  0.0 |  11.39404 | 11.136685 | 0.128897 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.63093 | 304.149308 |  11.39404 |      0.0 |  0.23558 | 0.970674 | 0.181503 | 10.134526 |      0.0 | 0.222894 | 0.771194 | 0.008071 |   0.0 | 0.128897 |
    | 1997-27-08 | 20.0 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 | 18.8 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | -0.25 | 19.703098 | -0.620474 |             10.870274 |                 10.870274 |                  10.870274 |                  1.498964 |                      0.0 |      9.37131 |              0.0 |           9.37131 |             67.254108 |                 100.0 |                 48.85611 |               67.046207 | 18.1 |  0.0 | 3.567075 | 0.7 | 2.108966 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 13.249308 | 0.238122 | 1.017431 | 0.184749 |  0.0 | 13.249308 | 12.737269 | 0.147422 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.001403 | 305.450732 |      12.0 | 1.249308 | 0.238122 | 1.017431 | 0.184749 | 10.877952 | 0.792359 | 0.226674 | 0.831586 | 0.008698 |   0.0 | 0.147422 |
    | 1997-28-08 | 20.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 | 18.8 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | -0.25 | 20.938011 |  0.382585 |              5.758296 |                  5.758296 |                   5.758296 |                  0.589077 |                      0.0 |     5.169219 |              0.0 |          5.169219 |             54.320626 |                 100.0 |                 48.85611 |               67.129158 | 18.1 |  0.0 | 2.621374 | 0.7 | 1.331712 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.550732 | 0.239141 | 1.036378 | 0.186051 |  0.0 | 14.550732 | 14.562531 | 0.168548 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.368818 | 306.206718 |      12.0 | 2.550732 | 0.239141 | 1.036378 | 0.186051 | 11.401263 | 2.037672 | 0.229904 | 0.884361 | 0.009331 |   0.0 | 0.168548 |
    | 1997-29-08 | 20.0 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 | 18.8 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | -0.25 | 18.829395 |  1.414918 |               5.53921 |                   5.53921 |                    5.53921 |                  0.708412 |                      0.0 |     4.830799 |              0.0 |          4.830799 |             54.320626 |                 100.0 |                 48.85611 |               67.212519 | 18.1 |  0.0 | 2.027722 | 0.7 | 0.881862 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.306718 | 0.239733 | 1.047438 | 0.186807 |  0.0 | 15.306718 |  15.83846 | 0.183316 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   30.7039 |  306.64416 |      12.0 | 3.306718 | 0.239733 | 1.047438 | 0.186807 | 11.680507 |  2.98861 | 0.232477 |   0.9269 | 0.009966 |   0.0 | 0.183316 |
    | 1997-30-08 | 20.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 | 18.8 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | -0.25 | 13.279667 |  -1.92862 |              9.344591 |                  9.344591 |                   9.344591 |                  1.881434 |                      0.0 |     7.463158 |              0.0 |          7.463158 |             67.254108 |                 100.0 |                 48.85611 |               67.296277 | 18.1 |  0.0 | 3.393609 | 0.7 | 1.913447 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  15.74416 | 0.240076 | 1.053855 | 0.187244 |  0.0 |  15.74416 | 16.593262 | 0.192052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.382521 | 305.605378 |      12.0 |  3.74416 | 0.240076 | 1.053855 | 0.187244 | 11.829515 | 3.558369 | 0.234481 | 0.960296 |   0.0106 |   0.0 | 0.192052 |
    | 1997-31-08 | 20.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 | 18.8 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | -0.25 | 18.875069 |  0.349956 |              8.321079 |                  8.321079 |                   8.321079 |                  1.511299 |                      0.0 |      6.80978 |              0.0 |           6.80978 |             67.254108 |                 100.0 |                 48.85611 |               67.380423 | 18.1 |  0.0 | 3.322101 | 0.7 | 1.882222 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.705378 | 0.239262 | 1.038637 | 0.186205 |  0.0 | 14.705378 | 16.209748 | 0.187613 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.782565 | 305.653673 |      12.0 | 2.705378 | 0.239262 | 1.038637 | 0.186205 | 11.909027 | 3.070262 | 0.235874 | 0.983354 | 0.011231 |   0.0 | 0.187613 |

>>> inputs.nied.series = nied

.. _lland_v3_wasser_daily:

water
-----

L-Land defines three types of water areas, |WASSER|, |FLUSS| and |SEE|.
|lland_v3| calculates their evaporation via the Penman equation based on
daily input data.  Besides that, there are no structural differences to
|lland_v1|,  so please read the detailed discussions in integration test
:ref:`lland_v1_wasser` for better understanding the following test
results for water type |WASSER|:

.. integration-test::

    >>> lnk(WASSER)
    >>> surfaceresistance.wasser_aug = 0.0
    >>> cropheight.wasser_aug = 0.05
    >>> albedo.wasser_aug = 0.7
    >>> conditions_wasser = test(
    ...     "lland_v3_wasser_daily",
    ...     axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | 0.0 | nan | 0.0 |              4.931319 |                  4.931319 |                   4.931319 |                  1.877708 |                      0.0 |      3.05361 |              0.0 |           3.05361 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 11.424803 | 0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | 0.0 | nan | 0.0 |              3.308967 |                  3.308967 |                   3.308967 |                   0.96939 |                      0.0 |     2.339577 |              0.0 |          2.339577 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | 0.0 | nan | 0.0 |              4.261282 |                  4.261282 |                   4.261282 |                  1.386217 |                      0.0 |     2.875064 |              0.0 |          2.875064 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | 0.0 | nan | 0.0 |              6.973011 |                  6.973011 |                   6.973011 |                  3.108707 |                      0.0 |     3.864304 |              0.0 |          3.864304 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | 0.0 | nan | 0.0 |              7.461037 |                  7.461037 |                   7.461037 |                  3.414868 |                      0.0 |     4.046169 |              0.0 |          4.046169 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | 0.0 | nan | 0.0 |              7.119557 |                  7.119557 |                   7.119557 |                  2.845741 |                      0.0 |     4.273816 |              0.0 |          4.273816 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | 0.0 | nan | 0.0 |              7.362521 |                  7.362521 |                   7.362521 |                  3.051087 |                      0.0 |     4.311434 |              0.0 |          4.311434 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | 0.0 | nan | 0.0 |              6.816215 |                  6.816215 |                   6.816215 |                   2.45312 |                      0.0 |     4.363095 |              0.0 |          4.363095 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | 0.0 | nan | 0.0 |              7.056886 |                  7.056886 |                   7.056886 |                  2.688326 |                      0.0 |      4.36856 |              0.0 |           4.36856 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | 0.0 | nan | 0.0 |              6.921138 |                  6.921138 |                   6.921138 |                   2.76572 |                      0.0 |     4.155419 |              0.0 |          4.155419 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | 0.0 | nan | 0.0 |               6.98872 |                   6.98872 |                    6.98872 |                  2.799361 |                      0.0 |     4.189359 |              0.0 |          4.189359 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | 0.0 | nan | 0.0 |              7.021165 |                  7.021165 |                   7.021165 |                  2.986484 |                      0.0 |     4.034682 |              0.0 |          4.034682 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | 0.0 | nan | 0.0 |               6.78319 |                   6.78319 |                    6.78319 |                  2.708378 |                      0.0 |     4.074812 |              0.0 |          4.074812 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | 0.0 | nan | 0.0 |              5.908778 |                  5.908778 |                   5.908778 |                  2.239353 |                      0.0 |     3.669425 |              0.0 |          3.669425 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | 0.0 | nan | 0.0 |              5.607956 |                  5.607956 |                   5.607956 |                  2.021579 |                      0.0 |     3.586377 |              0.0 |          3.586377 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.008458 | 0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | 0.0 | nan | 0.0 |              5.874454 |                  5.874454 |                   5.874454 |                  2.487815 |                      0.0 |      3.38664 |              0.0 |           3.38664 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | 0.0 | nan | 0.0 |              6.104899 |                  6.104899 |                   6.104899 |                  2.520953 |                      0.0 |     3.583945 |              0.0 |          3.583945 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | 0.0 | nan | 0.0 |              6.035457 |                  6.035457 |                   6.035457 |                  2.702703 |                      0.0 |     3.332754 |              0.0 |          3.332754 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | 0.0 | nan | 0.0 |              5.176574 |                  5.176574 |                   5.176574 |                  2.307856 |                      0.0 |     2.868718 |              0.0 |          2.868718 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | 0.0 | nan | 0.0 |              6.059368 |                  6.059368 |                   6.059368 |                  2.688476 |                      0.0 |     3.370892 |              0.0 |          3.370892 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | 0.0 | nan | 0.0 |              6.379464 |                  6.379464 |                   6.379464 |                  3.308913 |                      0.0 |     3.070551 |              0.0 |          3.070551 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | 0.0 | nan | 0.0 |              2.347976 |                  2.347976 |                   2.347976 |                   0.60663 |                      0.0 |     1.741346 |              0.0 |          1.741346 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.073812 | 0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | 0.0 | nan | 0.0 |              4.456814 |                  4.456814 |                   4.456814 |                  1.641158 |                      0.0 |     2.815656 |              0.0 |          2.815656 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |    0.282 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | 0.0 | nan | 0.0 |              5.612971 |                  5.612971 |                   5.612971 |                  1.868809 |                      0.0 |     3.744162 |              0.0 |          3.744162 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | 0.0 | nan | 0.0 |              4.326484 |                  4.326484 |                   4.326484 |                  1.401603 |                      0.0 |     2.924881 |              0.0 |          2.924881 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | 0.0 | nan | 0.0 |              3.783947 |                  3.783947 |                   3.783947 |                  1.261689 |                      0.0 |     2.522258 |              0.0 |          2.522258 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.792904 |  0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | 0.0 | nan | 0.0 |              4.290898 |                  4.290898 |                   4.290898 |                  1.498964 |                      0.0 |     2.791934 |              0.0 |          2.791934 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.01244 | 0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | 0.0 | nan | 0.0 |              2.273012 |                  2.273012 |                   2.273012 |                  0.589077 |                      0.0 |     1.683934 |              0.0 |          1.683934 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.305141 | 0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | 0.0 | nan | 0.0 |               2.18653 |                   2.18653 |                    2.18653 |                  0.708412 |                      0.0 |     1.478119 |              0.0 |          1.478119 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.040224 | 0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | 0.0 | nan | 0.0 |              3.688655 |                  3.688655 |                   3.688655 |                  1.881434 |                      0.0 |     1.807221 |              0.0 |          1.807221 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | 0.0 | nan | 0.0 |              3.284636 |                  3.284636 |                   3.284636 |                  1.511299 |                      0.0 |     1.773338 |              0.0 |          1.773338 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

.. _lland_v3_wasser_routing_daily:

water (routing)
---------------

The following calculation shows the possibility to subtract evaporation
from inflowing runoff, discussed in the integration test
:ref:`lland_v1_wasser_routing` of |lland_v1|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_v3_wasser_routing_daily",
    ...     axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |   qz |   qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.02 | 1.728 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | 0.0 | nan | 0.0 |              4.931319 |                  4.931319 |                   4.931319 |                  1.877708 |                      0.0 |      3.05361 |              0.0 |           3.05361 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.152803 | 0.152232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.152232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.02 | 1.728 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | 0.0 | nan | 0.0 |              3.308967 |                  3.308967 |                   3.308967 |                   0.96939 |                      0.0 |     2.339577 |              0.0 |          2.339577 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.95655 | 0.011071 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.011071 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.02 | 1.728 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | 0.0 | nan | 0.0 |              4.261282 |                  4.261282 |                   4.261282 |                  1.386217 |                      0.0 |     2.875064 |              0.0 |          2.875064 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.635635 | 0.007357 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.007357 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.02 | 1.728 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | 0.0 | nan | 0.0 |              6.973011 |                  6.973011 |                   6.973011 |                  3.108707 |                      0.0 |     3.864304 |              0.0 |          3.864304 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.256106 | 0.002964 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002964 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.02 | 1.728 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | 0.0 | nan | 0.0 |              7.461037 |                  7.461037 |                   7.461037 |                  3.414868 |                      0.0 |     4.046169 |              0.0 |          4.046169 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.02 | 1.728 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | 0.0 | nan | 0.0 |              7.119557 |                  7.119557 |                   7.119557 |                  2.845741 |                      0.0 |     4.273816 |              0.0 |          4.273816 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.02 | 1.728 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | 0.0 | nan | 0.0 |              7.362521 |                  7.362521 |                   7.362521 |                  3.051087 |                      0.0 |     4.311434 |              0.0 |          4.311434 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.02 | 1.728 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | 0.0 | nan | 0.0 |              6.816215 |                  6.816215 |                   6.816215 |                   2.45312 |                      0.0 |     4.363095 |              0.0 |          4.363095 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.02 | 1.728 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | 0.0 | nan | 0.0 |              7.056886 |                  7.056886 |                   7.056886 |                  2.688326 |                      0.0 |      4.36856 |              0.0 |           4.36856 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.02 | 1.728 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | 0.0 | nan | 0.0 |              6.921138 |                  6.921138 |                   6.921138 |                   2.76572 |                      0.0 |     4.155419 |              0.0 |          4.155419 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.02 | 1.728 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | 0.0 | nan | 0.0 |               6.98872 |                   6.98872 |                    6.98872 |                  2.799361 |                      0.0 |     4.189359 |              0.0 |          4.189359 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.02 | 1.728 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | 0.0 | nan | 0.0 |              7.021165 |                  7.021165 |                   7.021165 |                  2.986484 |                      0.0 |     4.034682 |              0.0 |          4.034682 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |    1.822 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.02 | 1.728 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | 0.0 | nan | 0.0 |               6.78319 |                   6.78319 |                    6.78319 |                  2.708378 |                      0.0 |     4.074812 |              0.0 |          4.074812 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.02 | 1.728 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | 0.0 | nan | 0.0 |              5.908778 |                  5.908778 |                   5.908778 |                  2.239353 |                      0.0 |     3.669425 |              0.0 |          3.669425 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.055264 |  0.00064 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.00064 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.02 | 1.728 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | 0.0 | nan | 0.0 |              5.607956 |                  5.607956 |                   5.607956 |                  2.021579 |                      0.0 |     3.586377 |              0.0 |          3.586377 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.736458 | 0.031672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.031672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.02 | 1.728 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | 0.0 | nan | 0.0 |              5.874454 |                  5.874454 |                   5.874454 |                  2.487815 |                      0.0 |      3.38664 |              0.0 |           3.38664 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.444099 |  0.00514 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.00514 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.02 | 1.728 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | 0.0 | nan | 0.0 |              6.104899 |                  6.104899 |                   6.104899 |                  2.520953 |                      0.0 |     3.583945 |              0.0 |          3.583945 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.217257 | 0.002515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002515 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.02 | 1.728 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | 0.0 | nan | 0.0 |              6.035457 |                  6.035457 |                   6.035457 |                  2.702703 |                      0.0 |     3.332754 |              0.0 |          3.332754 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.133428 | 0.001544 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.001544 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.02 | 1.728 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | 0.0 | nan | 0.0 |              5.176574 |                  5.176574 |                   5.176574 |                  2.307856 |                      0.0 |     2.868718 |              0.0 |          2.868718 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.186433 | 0.002158 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002158 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.02 | 1.728 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | 0.0 | nan | 0.0 |              6.059368 |                  6.059368 |                   6.059368 |                  2.688476 |                      0.0 |     3.370892 |              0.0 |          3.370892 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.02 | 1.728 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | 0.0 | nan | 0.0 |              6.379464 |                  6.379464 |                   6.379464 |                  3.308913 |                      0.0 |     3.070551 |              0.0 |          3.070551 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.02 | 1.728 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | 0.0 | nan | 0.0 |              2.347976 |                  2.347976 |                   2.347976 |                   0.60663 |                      0.0 |     1.741346 |              0.0 |          1.741346 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.801812 | 0.032428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.032428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.02 | 1.728 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | 0.0 | nan | 0.0 |              4.456814 |                  4.456814 |                   4.456814 |                  1.641158 |                      0.0 |     2.815656 |              0.0 |          2.815656 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.92579 | 0.010715 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.010715 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.02 | 1.728 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | 0.0 | nan | 0.0 |              5.612971 |                  5.612971 |                   5.612971 |                  1.868809 |                      0.0 |     3.744162 |              0.0 |          3.744162 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.055768 | 0.000645 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.000645 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.02 | 1.728 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | 0.0 | nan | 0.0 |              4.326484 |                  4.326484 |                   4.326484 |                  1.401603 |                      0.0 |     2.924881 |              0.0 |          2.924881 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.447898 | 0.005184 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.005184 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.02 | 1.728 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | 0.0 | nan | 0.0 |              3.783947 |                  3.783947 |                   3.783947 |                  1.261689 |                      0.0 |     2.522258 |              0.0 |          2.522258 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 10.520904 |  0.12177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.12177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.02 | 1.728 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | 0.0 | nan | 0.0 |              4.290898 |                  4.290898 |                   4.290898 |                  1.498964 |                      0.0 |     2.791934 |              0.0 |          2.791934 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   6.74044 | 0.078014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.078014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.02 | 1.728 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | 0.0 | nan | 0.0 |              2.273012 |                  2.273012 |                   2.273012 |                  0.589077 |                      0.0 |     1.683934 |              0.0 |          1.683934 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 15.033141 | 0.173995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.173995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.02 | 1.728 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | 0.0 | nan | 0.0 |               2.18653 |                   2.18653 |                    2.18653 |                  0.708412 |                      0.0 |     1.478119 |              0.0 |          1.478119 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  9.768224 | 0.113058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.113058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.02 | 1.728 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | 0.0 | nan | 0.0 |              3.688655 |                  3.688655 |                   3.688655 |                  1.881434 |                      0.0 |     1.807221 |              0.0 |          1.807221 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.82735 | 0.009576 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.009576 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.02 | 1.728 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | 0.0 | nan | 0.0 |              3.284636 |                  3.284636 |                   3.284636 |                  1.511299 |                      0.0 |     1.773338 |              0.0 |          1.773338 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.815922 | 0.009444 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.009444 |

.. _lland_v3_wasser_negq_daily:

water (negative runoff)
-----------------------

The following calculation shows the possibility to calculate negative
discharge values, discussed in the integration test :ref:`lland_v1_wasser_negq`
of |lland_v1|:

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_v3_wasser_negq_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |        qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |    outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | 0.0 | nan | 0.0 |              4.931319 |                  4.931319 |                   4.931319 |                  1.877708 |                      0.0 |      3.05361 |              0.0 |           3.05361 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 11.424803 |  0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | 0.0 | nan | 0.0 |              3.308967 |                  3.308967 |                   3.308967 |                   0.96939 |                      0.0 |     2.339577 |              0.0 |          2.339577 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.77145 | -0.008929 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.008929 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | 0.0 | nan | 0.0 |              4.261282 |                  4.261282 |                   4.261282 |                  1.386217 |                      0.0 |     2.875064 |              0.0 |          2.875064 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.092365 | -0.012643 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.012643 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | 0.0 | nan | 0.0 |              6.973011 |                  6.973011 |                   6.973011 |                  3.108707 |                      0.0 |     3.864304 |              0.0 |          3.864304 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.471894 | -0.017036 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017036 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | 0.0 | nan | 0.0 |              7.461037 |                  7.461037 |                   7.461037 |                  3.414868 |                      0.0 |     4.046169 |              0.0 |          4.046169 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.729733 |  -0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.02002 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | 0.0 | nan | 0.0 |              7.119557 |                  7.119557 |                   7.119557 |                  2.845741 |                      0.0 |     4.273816 |              0.0 |          4.273816 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.826087 | -0.021135 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021135 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | 0.0 | nan | 0.0 |              7.362521 |                  7.362521 |                   7.362521 |                  3.051087 |                      0.0 |     4.311434 |              0.0 |          4.311434 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.041956 | -0.023634 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.023634 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | 0.0 | nan | 0.0 |              6.816215 |                  6.816215 |                   6.816215 |                   2.45312 |                      0.0 |     4.363095 |              0.0 |          4.363095 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.800053 | -0.020834 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.020834 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | 0.0 | nan | 0.0 |              7.056886 |                  7.056886 |                   7.056886 |                  2.688326 |                      0.0 |      4.36856 |              0.0 |           4.36856 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |  1.82186 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -1.82186 | -0.021086 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021086 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | 0.0 | nan | 0.0 |              6.921138 |                  6.921138 |                   6.921138 |                   2.76572 |                      0.0 |     4.155419 |              0.0 |          4.155419 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 | 1.891469 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.891469 | -0.021892 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021892 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | 0.0 | nan | 0.0 |               6.98872 |                   6.98872 |                    6.98872 |                  2.799361 |                      0.0 |     4.189359 |              0.0 |          4.189359 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 | 2.106575 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.106575 | -0.024382 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.024382 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | 0.0 | nan | 0.0 |              7.021165 |                  7.021165 |                   7.021165 |                  2.986484 |                      0.0 |     4.034682 |              0.0 |          4.034682 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 | 2.036929 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.942929 | -0.022488 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.022488 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | 0.0 | nan | 0.0 |               6.78319 |                   6.78319 |                    6.78319 |                  2.708378 |                      0.0 |     4.074812 |              0.0 |          4.074812 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | 1.808559 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.808559 | -0.020932 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.020932 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | 0.0 | nan | 0.0 |              5.908778 |                  5.908778 |                   5.908778 |                  2.239353 |                      0.0 |     3.669425 |              0.0 |          3.669425 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.672736 |  -0.01936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.01936 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | 0.0 | nan | 0.0 |              5.607956 |                  5.607956 |                   5.607956 |                  2.021579 |                      0.0 |     3.586377 |              0.0 |          3.586377 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.008458 |  0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | 0.0 | nan | 0.0 |              5.874454 |                  5.874454 |                   5.874454 |                  2.487815 |                      0.0 |      3.38664 |              0.0 |           3.38664 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.283901 |  -0.01486 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.01486 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | 0.0 | nan | 0.0 |              6.104899 |                  6.104899 |                   6.104899 |                  2.520953 |                      0.0 |     3.583945 |              0.0 |          3.583945 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.510743 | -0.017485 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017485 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | 0.0 | nan | 0.0 |              6.035457 |                  6.035457 |                   6.035457 |                  2.702703 |                      0.0 |     3.332754 |              0.0 |          3.332754 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.594572 | -0.018456 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.018456 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | 0.0 | nan | 0.0 |              5.176574 |                  5.176574 |                   5.176574 |                  2.307856 |                      0.0 |     2.868718 |              0.0 |          2.868718 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.541567 | -0.017842 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017842 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | 0.0 | nan | 0.0 |              6.059368 |                  6.059368 |                   6.059368 |                  2.688476 |                      0.0 |     3.370892 |              0.0 |          3.370892 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 | 1.939383 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.939383 | -0.022447 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.022447 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | 0.0 | nan | 0.0 |              6.379464 |                  6.379464 |                   6.379464 |                  3.308913 |                      0.0 |     3.070551 |              0.0 |          3.070551 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 | 1.826811 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.826811 | -0.021144 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021144 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | 0.0 | nan | 0.0 |              2.347976 |                  2.347976 |                   2.347976 |                   0.60663 |                      0.0 |     1.741346 |              0.0 |          1.741346 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.073812 |  0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | 0.0 | nan | 0.0 |              4.456814 |                  4.456814 |                   4.456814 |                  1.641158 |                      0.0 |     2.815656 |              0.0 |          2.815656 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.80221 | -0.009285 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.009285 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | 0.0 | nan | 0.0 |              5.612971 |                  5.612971 |                   5.612971 |                  1.868809 |                      0.0 |     3.744162 |              0.0 |          3.744162 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.672232 | -0.019355 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.019355 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | 0.0 | nan | 0.0 |              4.326484 |                  4.326484 |                   4.326484 |                  1.401603 |                      0.0 |     2.924881 |              0.0 |          2.924881 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.280102 | -0.014816 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.014816 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | 0.0 | nan | 0.0 |              3.783947 |                  3.783947 |                   3.783947 |                  1.261689 |                      0.0 |     2.522258 |              0.0 |          2.522258 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.792904 |   0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | 0.0 | nan | 0.0 |              4.290898 |                  4.290898 |                   4.290898 |                  1.498964 |                      0.0 |     2.791934 |              0.0 |          2.791934 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.01244 |  0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | 0.0 | nan | 0.0 |              2.273012 |                  2.273012 |                   2.273012 |                  0.589077 |                      0.0 |     1.683934 |              0.0 |          1.683934 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.305141 |  0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | 0.0 | nan | 0.0 |               2.18653 |                   2.18653 |                    2.18653 |                  0.708412 |                      0.0 |     1.478119 |              0.0 |          1.478119 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.040224 |  0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | 0.0 | nan | 0.0 |              3.688655 |                  3.688655 |                   3.688655 |                  1.881434 |                      0.0 |     1.807221 |              0.0 |          1.807221 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.90065 | -0.010424 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.010424 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | 0.0 | nan | 0.0 |              3.284636 |                  3.284636 |                   3.284636 |                  1.511299 |                      0.0 |     1.773338 |              0.0 |          1.773338 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.912078 | -0.010556 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.010556 |

.. _lland_v3_see_daily:

lakes
-----

The following example focusses on water-type |SEE| (for further information,
see integration test :ref:`lland_v1_see` of |lland_v1|):

.. integration-test::

    >>> lnk(SEE)
    >>> surfaceresistance.see_aug = 0.0
    >>> cropheight.see_aug = 0.05
    >>> albedo.see_aug = 0.7
    >>> negq(False)
    >>> test("lland_v3_see_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | 0.0 | nan | 0.0 |              4.931319 |                  4.931319 |                   4.931319 |                  1.877708 |                      0.0 |      3.05361 |              0.0 |           3.05361 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.020478 | 0.000237 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 11.424803 |   0.0 |   0.0 |   0.0 |   0.0 |  0.020478 |   0.0 | 0.000237 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | 0.0 | nan | 0.0 |              3.308967 |                  3.308967 |                   3.308967 |                   0.96939 |                      0.0 |     2.339577 |              0.0 |          2.339577 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.039475 | 0.000457 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.77145 |   0.0 |   0.0 |   0.0 |   0.0 |  0.039475 |   0.0 | 0.000457 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | 0.0 | nan | 0.0 |              4.261282 |                  4.261282 |                   4.261282 |                  1.386217 |                      0.0 |     2.875064 |              0.0 |          2.875064 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.035994 | 0.000417 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.092365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035994 |   0.0 | 0.000417 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | 0.0 | nan | 0.0 |              6.973011 |                  6.973011 |                   6.973011 |                  3.108707 |                      0.0 |     3.864304 |              0.0 |          3.864304 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.031272 | 0.000362 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.471894 |   0.0 |   0.0 |   0.0 |   0.0 |  0.031272 |   0.0 | 0.000362 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | 0.0 | nan | 0.0 |              7.461037 |                  7.461037 |                   7.461037 |                  3.414868 |                      0.0 |     4.046169 |              0.0 |          4.046169 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.025424 | 0.000294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.729733 |   0.0 |   0.0 |   0.0 |   0.0 |  0.025424 |   0.0 | 0.000294 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | 0.0 | nan | 0.0 |              7.119557 |                  7.119557 |                   7.119557 |                  2.845741 |                      0.0 |     4.273816 |              0.0 |          4.273816 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.018963 | 0.000219 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826087 |   0.0 |   0.0 |   0.0 |   0.0 |  0.018963 |   0.0 | 0.000219 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | 0.0 | nan | 0.0 |              7.362521 |                  7.362521 |                   7.362521 |                  3.051087 |                      0.0 |     4.311434 |              0.0 |          4.311434 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011966 | 0.000138 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.041956 |   0.0 |   0.0 |   0.0 |   0.0 |  0.011966 |   0.0 | 0.000138 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | 0.0 | nan | 0.0 |              6.816215 |                  6.816215 |                   6.816215 |                   2.45312 |                      0.0 |     4.363095 |              0.0 |          4.363095 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.005041 | 0.000058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.800053 |   0.0 |   0.0 |   0.0 |   0.0 |  0.005041 |   0.0 | 0.000058 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | 0.0 | nan | 0.0 |              7.056886 |                  7.056886 |                   7.056886 |                  2.688326 |                      0.0 |      4.36856 |              0.0 |           4.36856 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 | 1.820396 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -1.82186 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001465 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | 0.0 | nan | 0.0 |              6.921138 |                  6.921138 |                   6.921138 |                   2.76572 |                      0.0 |     4.155419 |              0.0 |          4.155419 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 | 1.883357 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.891469 |   0.0 |   0.0 |   0.0 |   0.0 | -0.008111 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | 0.0 | nan | 0.0 |               6.98872 |                   6.98872 |                    6.98872 |                  2.799361 |                      0.0 |     4.189359 |              0.0 |          4.189359 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 | 2.091331 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.106575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.015244 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | 0.0 | nan | 0.0 |              7.021165 |                  7.021165 |                   7.021165 |                  2.986484 |                      0.0 |     4.034682 |              0.0 |          4.034682 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 | 2.014486 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.942929 |   0.0 |   0.0 |   0.0 |   0.0 | -0.022443 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | 0.0 | nan | 0.0 |               6.78319 |                   6.78319 |                    6.78319 |                  2.708378 |                      0.0 |     4.074812 |              0.0 |          4.074812 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | 1.779476 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.808559 |   0.0 |   0.0 |   0.0 |   0.0 | -0.029083 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | 0.0 | nan | 0.0 |              5.908778 |                  5.908778 |                   5.908778 |                  2.239353 |                      0.0 |     3.669425 |              0.0 |          3.669425 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.637521 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672736 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035215 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | 0.0 | nan | 0.0 |              5.607956 |                  5.607956 |                   5.607956 |                  2.021579 |                      0.0 |     3.586377 |              0.0 |          3.586377 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.493266 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.008458 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036276 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | 0.0 | nan | 0.0 |              5.874454 |                  5.874454 |                   5.874454 |                  2.487815 |                      0.0 |      3.38664 |              0.0 |           3.38664 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.247259 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.283901 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036642 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | 0.0 | nan | 0.0 |              6.104899 |                  6.104899 |                   6.104899 |                  2.520953 |                      0.0 |     3.583945 |              0.0 |          3.583945 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.469226 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.510743 |   0.0 |   0.0 |   0.0 |   0.0 | -0.041517 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | 0.0 | nan | 0.0 |              6.035457 |                  6.035457 |                   6.035457 |                  2.702703 |                      0.0 |     3.332754 |              0.0 |          3.332754 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.547641 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.594572 |   0.0 |   0.0 |   0.0 |   0.0 | -0.046931 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | 0.0 | nan | 0.0 |              5.176574 |                  5.176574 |                   5.176574 |                  2.307856 |                      0.0 |     2.868718 |              0.0 |          2.868718 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.489186 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.541567 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.05238 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | 0.0 | nan | 0.0 |              6.059368 |                  6.059368 |                   6.059368 |                  2.688476 |                      0.0 |     3.370892 |              0.0 |          3.370892 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 | 1.880954 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.939383 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058428 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | 0.0 | nan | 0.0 |              6.379464 |                  6.379464 |                   6.379464 |                  3.308913 |                      0.0 |     3.070551 |              0.0 |          3.070551 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 | 1.761846 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826811 |   0.0 |   0.0 |   0.0 |   0.0 | -0.064965 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | 0.0 | nan | 0.0 |              2.347976 |                  2.347976 |                   2.347976 |                   0.60663 |                      0.0 |     1.741346 |              0.0 |          1.741346 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.740109 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.073812 |   0.0 |   0.0 |   0.0 |   0.0 | -0.066078 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | 0.0 | nan | 0.0 |              4.456814 |                  4.456814 |                   4.456814 |                  1.641158 |                      0.0 |     2.815656 |              0.0 |          2.815656 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 | 1.018853 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.80221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.065357 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | 0.0 | nan | 0.0 |              5.612971 |                  5.612971 |                   5.612971 |                  1.868809 |                      0.0 |     3.744162 |              0.0 |          3.744162 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.602675 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672232 |   0.0 |   0.0 |   0.0 |   0.0 | -0.069557 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | 0.0 | nan | 0.0 |              4.326484 |                  4.326484 |                   4.326484 |                  1.401603 |                      0.0 |     2.924881 |              0.0 |          2.924881 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.299507 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.280102 |   0.0 |   0.0 |   0.0 |   0.0 | -0.074595 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | 0.0 | nan | 0.0 |              3.783947 |                  3.783947 |                   3.783947 |                  1.261689 |                      0.0 |     2.522258 |              0.0 |          2.522258 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.016237 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.792904 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06086 |   0.0 |      0.0 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | 0.0 | nan | 0.0 |              4.290898 |                  4.290898 |                   4.290898 |                  1.498964 |                      0.0 |     2.791934 |              0.0 |          2.791934 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 | 1.061645 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   5.01244 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035916 |   0.0 |      0.0 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | 0.0 | nan | 0.0 |              2.273012 |                  2.273012 |                   2.273012 |                  0.589077 |                      0.0 |     1.683934 |              0.0 |          1.683934 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.791893 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 13.305141 |   0.0 |   0.0 |   0.0 |   0.0 | -0.002966 |   0.0 |      0.0 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | 0.0 | nan | 0.0 |               2.18653 |                   2.18653 |                    2.18653 |                  0.708412 |                      0.0 |     1.478119 |              0.0 |          1.478119 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.035276 | 0.000408 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.040224 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035276 |   0.0 | 0.000408 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | 0.0 | nan | 0.0 |              3.688655 |                  3.688655 |                   3.688655 |                  1.881434 |                      0.0 |     1.807221 |              0.0 |          1.807221 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.047929 | 0.000555 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.90065 |   0.0 |   0.0 |   0.0 |   0.0 |  0.047929 |   0.0 | 0.000555 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | 0.0 | nan | 0.0 |              3.284636 |                  3.284636 |                   3.284636 |                  1.511299 |                      0.0 |     1.773338 |              0.0 |          1.773338 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.04451 | 0.000515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.912078 |   0.0 |   0.0 |   0.0 |   0.0 |   0.04451 |   0.0 | 0.000515 |

.. _lland_v3_fluss_daily:

streams
-------

The following example focusses on water-type |FLUSS| (for further information,
see integration test :ref:`lland_v1_fluss` of |lland_v1|):

.. integration-test::

    >>> lnk(FLUSS)
    >>> surfaceresistance.fluss_aug = 0.0
    >>> cropheight.fluss_aug = 0.05
    >>> albedo.fluss_aug = 0.7
    >>> test("lland_v3_fluss_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |       evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa |     qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |    qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | 0.0 | nan | 0.0 |              4.931319 |                  4.931319 |                   4.931319 |                  1.877708 |                      0.0 |      3.05361 |              0.0 |           3.05361 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 |  1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 11.424803 | 2.941276 | 0.034043 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 11.424803 |      0.0 |   0.0 |   0.0 |  0.0 |  2.941276 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034043 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | 0.0 | nan | 0.0 |              3.308967 |                  3.308967 |                   3.308967 |                   0.96939 |                      0.0 |     2.339577 |              0.0 |          2.339577 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |   0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.77145 | 3.758016 | 0.043496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.77145 |      0.0 |   0.0 |   0.0 |  0.0 |  3.758016 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043496 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | 0.0 | nan | 0.0 |              4.261282 |                  4.261282 |                   4.261282 |                  1.386217 |                      0.0 |     2.875064 |              0.0 |          2.875064 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 |  1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.092365 | 1.562905 | 0.018089 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.092365 |      0.0 |   0.0 |   0.0 |  0.0 |  1.562905 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018089 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | 0.0 | nan | 0.0 |              6.973011 |                  6.973011 |                   6.973011 |                  3.108707 |                      0.0 |     3.864304 |              0.0 |          3.864304 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 |  1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.471894 | 0.226808 | 0.002625 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.471894 |      0.0 |   0.0 |   0.0 |  0.0 |  0.226808 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002625 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | 0.0 | nan | 0.0 |              7.461037 |                  7.461037 |                   7.461037 |                  3.414868 |                      0.0 |     4.046169 |              0.0 |          4.046169 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |  1.097905 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.729733 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.729733 |      0.0 |   0.0 |   0.0 |  0.0 | -0.631827 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | 0.0 | nan | 0.0 |              7.119557 |                  7.119557 |                   7.119557 |                  2.845741 |                      0.0 |     4.273816 |              0.0 |          4.273816 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |  0.657403 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826087 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.826087 |      0.0 |   0.0 |   0.0 |  0.0 | -1.168684 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | 0.0 | nan | 0.0 |              7.362521 |                  7.362521 |                   7.362521 |                  3.051087 |                      0.0 |     4.311434 |              0.0 |          4.311434 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |  0.511091 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.041956 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.041956 |      0.0 |   0.0 |   0.0 |  0.0 | -1.530864 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | 0.0 | nan | 0.0 |              6.816215 |                  6.816215 |                   6.816215 |                   2.45312 |                      0.0 |     4.363095 |              0.0 |          4.363095 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |  0.093098 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.800053 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.800053 |      0.0 |   0.0 |   0.0 |  0.0 | -1.706955 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | 0.0 | nan | 0.0 |              7.056886 |                  7.056886 |                   7.056886 |                  2.688326 |                      0.0 |      4.36856 |              0.0 |           4.36856 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |  0.065871 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -1.82186 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -1.82186 |      0.0 |   0.0 |   0.0 |  0.0 | -1.755989 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | 0.0 | nan | 0.0 |              6.921138 |                  6.921138 |                   6.921138 |                   2.76572 |                      0.0 |     4.155419 |              0.0 |          4.155419 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |  0.086838 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.891469 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.891469 |      0.0 |   0.0 |   0.0 |  0.0 | -1.804631 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | 0.0 | nan | 0.0 |               6.98872 |                   6.98872 |                    6.98872 |                  2.799361 |                      0.0 |     4.189359 |              0.0 |          4.189359 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |  0.206065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.106575 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.106575 |      0.0 |   0.0 |   0.0 |  0.0 | -1.900509 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | 0.0 | nan | 0.0 |              7.021165 |                  7.021165 |                   7.021165 |                  2.986484 |                      0.0 |     4.034682 |              0.0 |          4.034682 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |  0.082443 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.942929 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.942929 |      0.0 |   0.0 |   0.0 |  0.0 | -1.954486 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | 0.0 | nan | 0.0 |               6.78319 |                   6.78319 |                    6.78319 |                  2.708378 |                      0.0 |     4.074812 |              0.0 |          4.074812 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | -0.105944 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.808559 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.808559 |      0.0 |   0.0 |   0.0 |  0.0 | -1.914503 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | 0.0 | nan | 0.0 |              5.908778 |                  5.908778 |                   5.908778 |                  2.239353 |                      0.0 |     3.669425 |              0.0 |          3.669425 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | -0.157388 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672736 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.672736 |      0.0 |   0.0 |   0.0 |  0.0 | -1.830125 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | 0.0 | nan | 0.0 |              5.607956 |                  5.607956 |                   5.607956 |                  2.021579 |                      0.0 |     3.586377 |              0.0 |          3.586377 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 |  0.463086 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.008458 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.008458 |      0.0 |   0.0 |   0.0 |  0.0 | -1.066456 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | 0.0 | nan | 0.0 |              5.874454 |                  5.874454 |                   5.874454 |                  2.487815 |                      0.0 |      3.38664 |              0.0 |           3.38664 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 |  0.595002 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.283901 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.283901 |      0.0 |   0.0 |   0.0 |  0.0 | -0.688899 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | 0.0 | nan | 0.0 |              6.104899 |                  6.104899 |                   6.104899 |                  2.520953 |                      0.0 |     3.583945 |              0.0 |          3.583945 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 |  0.485942 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.510743 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.510743 |      0.0 |   0.0 |   0.0 |  0.0 | -1.024801 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | 0.0 | nan | 0.0 |              6.035457 |                  6.035457 |                   6.035457 |                  2.702703 |                      0.0 |     3.332754 |              0.0 |          3.332754 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 |  0.321551 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.594572 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.594572 |      0.0 |   0.0 |   0.0 |  0.0 |  -1.27302 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | 0.0 | nan | 0.0 |              5.176574 |                  5.176574 |                   5.176574 |                  2.307856 |                      0.0 |     2.868718 |              0.0 |          2.868718 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 |  0.132224 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.541567 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.541567 |      0.0 |   0.0 |   0.0 |  0.0 | -1.409342 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | 0.0 | nan | 0.0 |              6.059368 |                  6.059368 |                   6.059368 |                  2.688476 |                      0.0 |     3.370892 |              0.0 |          3.370892 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |  0.365956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.939383 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.939383 |      0.0 |   0.0 |   0.0 |  0.0 | -1.573427 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | 0.0 | nan | 0.0 |              6.379464 |                  6.379464 |                   6.379464 |                  3.308913 |                      0.0 |     3.070551 |              0.0 |          3.070551 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |  0.111688 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826811 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.826811 |      0.0 |   0.0 |   0.0 |  0.0 | -1.715123 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | 0.0 | nan | 0.0 |              2.347976 |                  2.347976 |                   2.347976 |                   0.60663 |                      0.0 |     1.741346 |              0.0 |          1.741346 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 |  -0.21427 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.073812 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.073812 |      0.0 |   0.0 |   0.0 |  0.0 | -1.020458 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | 0.0 | nan | 0.0 |              4.456814 |                  4.456814 |                   4.456814 |                  1.641158 |                      0.0 |     2.815656 |              0.0 |          2.815656 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  0.557521 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.80221 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.80221 |      0.0 |   0.0 |   0.0 |  0.0 | -0.526689 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | 0.0 | nan | 0.0 |              5.612971 |                  5.612971 |                   5.612971 |                  1.868809 |                      0.0 |     3.744162 |              0.0 |          3.744162 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 |  0.793059 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672232 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.672232 |      0.0 |   0.0 |   0.0 |  0.0 | -0.879173 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | 0.0 | nan | 0.0 |              4.326484 |                  4.326484 |                   4.326484 |                  1.401603 |                      0.0 |     2.924881 |              0.0 |          2.924881 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 |  0.226008 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.280102 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.280102 |      0.0 |   0.0 |   0.0 |  0.0 | -1.148095 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | 0.0 | nan | 0.0 |              3.783947 |                  3.783947 |                   3.783947 |                  1.261689 |                      0.0 |     2.522258 |              0.0 |          2.522258 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 |  1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.792904 |   1.3836 | 0.016014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.792904 |      0.0 |   0.0 |   0.0 |  0.0 |    1.3836 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016014 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | 0.0 | nan | 0.0 |              4.290898 |                  4.290898 |                   4.290898 |                  1.498964 |                      0.0 |     2.791934 |              0.0 |          2.791934 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |   1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   5.01244 |  3.86595 | 0.044745 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   5.01244 |      0.0 |   0.0 |   0.0 |  0.0 |   3.86595 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044745 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | 0.0 | nan | 0.0 |              2.273012 |                  2.273012 |                   2.273012 |                  0.589077 |                      0.0 |     1.683934 |              0.0 |          1.683934 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 |  0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.305141 | 7.027354 | 0.081335 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |      12.0 | 1.305141 |   0.0 |   0.0 |  0.0 |  6.199583 |  0.82777 |   0.0 |   0.0 |  0.0 |   0.0 | 0.081335 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | 0.0 | nan | 0.0 |               2.18653 |                   2.18653 |                    2.18653 |                  0.708412 |                      0.0 |     1.478119 |              0.0 |          1.478119 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 |  0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.040224 |  8.32407 | 0.096343 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.040224 |      0.0 |   0.0 |   0.0 |  0.0 |  7.885403 | 0.438667 |   0.0 |   0.0 |  0.0 |   0.0 | 0.096343 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | 0.0 | nan | 0.0 |              3.688655 |                  3.688655 |                   3.688655 |                  1.881434 |                      0.0 |     1.807221 |              0.0 |          1.807221 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |   0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.90065 | 5.691378 | 0.065872 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.90065 |      0.0 |   0.0 |   0.0 |  0.0 |  5.655812 | 0.035566 |   0.0 |   0.0 |  0.0 |   0.0 | 0.065872 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | 0.0 | nan | 0.0 |              3.284636 |                  3.284636 |                   3.284636 |                  1.511299 |                      0.0 |     1.773338 |              0.0 |          1.773338 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 |  0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.912078 | 2.597892 | 0.030068 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.912078 |      0.0 |   0.0 |   0.0 |  0.0 |  2.595008 | 0.002884 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030068 |

.. _lland_v3_vers_daily:

sealed surfaces
---------------

As also shown in the integration test :ref:`lland_v1_vers` of |lland_v1|,
sealed surfaces route the not intercepted water to the linear storages for
direct discharge immediately:

.. integration-test::

    >>> lnk(VERS)
    >>> surfaceresistance.vers_aug = 500
    >>> cropheight.vers_aug = 5.0
    >>> albedo.vers_aug = 0.10
    >>> lai.vers_aug = 10.0
    >>> test("lland_v3_vers_daily",
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |     g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |      evpo |   evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |    qdb | qib1 | qib2 | qbb | qkap |   qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |      ebdn | bowa |  qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | -0.25 | 10.933333 |   -2.6784 |             14.793956 |                 14.793956 |                  14.793956 |                  1.877708 |                      0.0 |    12.916248 |              0.0 |         12.916248 |             16.038785 |                   nan |                    500.0 |                   500.0 | 10.502 |  0.0 |  7.049585 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 10.502 | 10.502 |  0.0 |  0.0 | 0.0 |  0.0 | 10.502 | 2.703704 | 0.031293 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   32.4284 |  0.0 | 10.502 |   0.0 |   0.0 |   0.0 |  0.0 | 2.703704 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.031293 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | -0.25 |    19.028 |  0.895795 |              9.926902 |                  9.926902 |                   9.926902 |                   0.96939 |                      0.0 |     8.957512 |              0.0 |          8.957512 |             24.058177 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  5.020895 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 3.637039 | 0.042095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.282605 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 3.637039 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042095 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | -0.25 | 15.208683 | -1.913579 |             12.783845 |                 12.783845 |                  12.783845 |                  1.386217 |                      0.0 |    11.397628 |              0.0 |         11.397628 |             25.662055 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  6.335547 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 1.940764 | 0.022463 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.946184 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.940764 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022463 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | -0.25 | 20.753946 |  1.531325 |             20.919034 |                 20.919034 |                  20.919034 |                  3.108707 |                      0.0 |    17.810327 |              0.0 |         17.810327 |              13.74753 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 12.230738 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 1.035613 | 0.011986 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.164858 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.035613 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011986 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | -0.25 | 14.816194 | -1.909685 |             22.383112 |                 22.383112 |                  22.383112 |                  3.414868 |                      0.0 |    18.968244 |              0.0 |         18.968244 |             12.831028 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 14.271962 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.552615 | 0.006396 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.824543 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.552615 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.006396 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | -0.25 | 20.348477 |  0.232491 |             21.358671 |                 21.358671 |                  21.358671 |                  2.845741 |                      0.0 |     18.51293 |              0.0 |          18.51293 |             10.998024 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 14.791696 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.294881 | 0.003413 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342053 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.294881 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003413 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | -0.25 | 18.740175 | -1.067813 |             22.087564 |                 22.087564 |                  22.087564 |                  3.051087 |                      0.0 |    19.036477 |              0.0 |         19.036477 |             10.129759 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 18.203529 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.157352 | 0.001821 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.159866 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.157352 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001821 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | -0.25 | 21.466219 |  0.293528 |             20.448645 |                 20.448645 |                  20.448645 |                   2.45312 |                      0.0 |    17.995525 |              0.0 |         17.995525 |             11.664571 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.456812 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.083965 | 0.000972 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.616338 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083965 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000972 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | -0.25 | 19.654459 | -1.371448 |             21.170659 |                 21.170659 |                  21.170659 |                  2.688326 |                      0.0 |    18.482332 |              0.0 |         18.482332 |             21.385046 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  11.84053 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.044804 | 0.000519 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.737786 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044804 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000519 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | -0.25 |  23.39262 |  0.203534 |             20.763414 |                 20.763414 |                  20.763414 |                   2.76572 |                      0.0 |    17.997695 |              0.0 |         17.997695 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.388795 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.023908 | 0.000277 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.284252 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.023908 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000277 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | -0.25 | 21.880839 | -0.632013 |              20.96616 |                  20.96616 |                   20.96616 |                  2.799361 |                      0.0 |    18.166799 |              0.0 |         18.166799 |             12.831028 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 18.009869 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.012758 | 0.000148 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.666265 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.012758 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000148 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | -0.25 | 23.154216 | -0.023734 |             21.063496 |                 21.063496 |                  21.063496 |                  2.986484 |                      0.0 |    18.077012 |              0.0 |         18.077012 |             15.397233 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 16.851837 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.006808 | 0.000079 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.439999 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006808 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000079 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | -0.25 | 22.399998 | -0.414721 |             20.349571 |                 20.349571 |                  20.349571 |                  2.708378 |                      0.0 |    17.641193 |              0.0 |         17.641193 |             24.058177 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 11.954049 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.003633 | 0.000042 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.604721 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003633 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000042 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | -0.25 | 22.949068 |  0.647517 |             17.726334 |                 17.726334 |                  17.726334 |                  2.239353 |                      0.0 |    15.486981 |              0.0 |         15.486981 |             14.256697 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.005903 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.001938 | 0.000022 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.707203 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001938 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000022 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | -0.25 | 19.957345 | -0.436832 |             16.823868 |                 16.823868 |                  16.823868 |                  2.021579 |                      0.0 |    14.802289 |              0.0 |         14.802289 |             12.029088 |                   nan |                    500.0 |                   500.0 |  0.538 |  0.0 | 11.728768 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.538 |  0.538 |  0.0 |  0.0 | 0.0 |  0.0 |  0.538 | 0.139541 | 0.001615 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.894036 |  0.0 |  0.538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.139541 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001615 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | -0.25 |  20.58012 |  0.559934 |             17.623363 |                 17.623363 |                  17.623363 |                  2.487815 |                      0.0 |    15.135549 |              0.0 |         15.135549 |             29.610064 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |   7.56081 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.186871 | 0.002163 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.084102 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.186871 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002163 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | -0.25 |  17.88034 | -1.617232 |             18.314696 |                 18.314696 |                  18.314696 |                  2.520953 |                      0.0 |    15.793743 |              0.0 |         15.793743 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  10.39177 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.099717 | 0.001154 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.451334 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.099717 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001154 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | -0.25 | 22.437779 |  0.434305 |             18.106372 |                 18.106372 |                  18.106372 |                  2.702703 |                      0.0 |    15.403668 |              0.0 |         15.403668 |             19.246541 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 12.298568 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |  0.05321 | 0.000616 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.767029 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05321 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000616 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | -0.25 | 20.156763 | -0.488974 |             15.529721 |                 15.529721 |                  15.529721 |                  2.307856 |                      0.0 |    13.221865 |              0.0 |         13.221865 |             14.805032 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.759102 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.028393 | 0.000329 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.006003 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.028393 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000329 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | -0.25 | 20.953343 | -0.749947 |             18.178103 |                 18.178103 |                  18.178103 |                  2.688476 |                      0.0 |    15.489627 |              0.0 |         15.489627 |             10.692523 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 19.233472 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.015151 | 0.000175 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.50595 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015151 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000175 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | -0.25 | 22.619833 |  0.062121 |             19.138391 |                 19.138391 |                  19.138391 |                  3.308913 |                      0.0 |    15.829478 |              0.0 |         15.829478 |             15.397233 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  17.26069 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.008085 | 0.000094 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.193828 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008085 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000094 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | -0.25 | 21.579428 |  1.026136 |              7.043929 |                  7.043929 |                   7.043929 |                   0.60663 |                      0.0 |     6.437299 |              0.0 |          6.437299 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  5.830469 |  1.88 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.004314 |  0.00005 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.917693 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00005 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | -0.25 | 17.325643 | -1.541907 |             13.370443 |                 13.370443 |                  13.370443 |                  1.641158 |                      0.0 |    11.729284 |              0.0 |         11.729284 |             25.662055 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  6.371493 | 0.282 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.002302 | 0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   33.2096 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002302 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000027 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | -0.25 | 21.631999 | -1.279412 |             16.838913 |                 16.838913 |                  16.838913 |                  1.868809 |                      0.0 |    14.970104 |              0.0 |         14.970104 |             17.496856 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 11.504016 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.001228 | 0.000014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 34.239011 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001228 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000014 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | -0.25 | 25.063372 |  0.706772 |             12.979453 |                 12.979453 |                  12.979453 |                  1.401603 |                      0.0 |     11.57785 |              0.0 |          11.57785 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  9.461825 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.000655 | 0.000008 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.28224 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000655 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000008 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | -0.25 | 21.874132 |   0.40131 |             11.351841 |                 11.351841 |                  11.351841 |                  1.261689 |                      0.0 |    10.090152 |              0.0 |         10.090152 |             16.736123 |                   nan |                    500.0 |                   500.0 |   7.87 |  0.0 |  7.472563 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   7.87 |   7.87 |  0.0 |  0.0 | 0.0 |  0.0 |   7.87 | 2.026454 | 0.023454 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.63093 |  0.0 |   7.87 |   0.0 |   0.0 |   0.0 |  0.0 | 2.026454 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023454 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | -0.25 | 19.703098 | -0.620474 |             12.872693 |                 12.872693 |                  12.872693 |                  1.498964 |                      0.0 |    11.373729 |              0.0 |         11.373729 |             18.330039 |                   nan |                    500.0 |                   500.0 |   4.11 |  0.0 |  6.960765 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   4.11 |   4.11 |  0.0 |  0.0 | 0.0 |  0.0 |   4.11 |  3.78382 | 0.043794 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.001403 |  0.0 |   4.11 |   0.0 |   0.0 |   0.0 |  0.0 |  3.78382 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043794 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | -0.25 | 20.938011 |  0.382585 |              6.819035 |                  6.819035 |                   6.819035 |                  0.589077 |                      0.0 |     6.229957 |              0.0 |          6.229957 |             14.805032 |                   nan |                    500.0 |                   500.0 |   12.1 |  0.0 |  6.075158 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   12.1 |   12.1 |  0.0 |  0.0 | 0.0 |  0.0 |   12.1 | 6.030624 | 0.069799 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.368818 |  0.0 |   12.0 |   0.1 |   0.0 |   0.0 |  0.0 | 5.967201 | 0.063424 |   0.0 |   0.0 |  0.0 |   0.0 | 0.069799 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | -0.25 | 18.829395 |  1.414918 |              6.559591 |                  6.559591 |                   6.559591 |                  0.708412 |                      0.0 |     5.851179 |              0.0 |          5.851179 |             14.805032 |                   nan |                    500.0 |                   500.0 |  6.648 |  0.0 |  4.534236 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  6.648 |  6.648 |  0.0 |  0.0 | 0.0 |  0.0 |  6.648 | 7.436588 | 0.086072 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   30.7039 |  0.0 |  6.648 |   0.0 |   0.0 |   0.0 |  0.0 | 7.402977 | 0.033611 |   0.0 |   0.0 |  0.0 |   0.0 | 0.086072 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | -0.25 | 13.279667 |  -1.92862 |             11.065964 |                 11.065964 |                  11.065964 |                  1.881434 |                      0.0 |      9.18453 |              0.0 |           9.18453 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  7.752367 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 5.342084 |  0.06183 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.382521 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 5.339359 | 0.002725 |   0.0 |   0.0 |  0.0 |   0.0 |  0.06183 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | -0.25 | 18.875069 |  0.349956 |              9.853909 |                  9.853909 |                   9.853909 |                  1.511299 |                      0.0 |      8.34261 |              0.0 |           8.34261 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  7.711096 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 2.849362 | 0.032979 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.782565 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849141 | 0.000221 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032979 |

.. _lland_v3_nadelw_summer_daily:

conifers (summer)
-----------------

The following tests focus on functionalities of |lland_v3| not implemented
by application model |lland_v1|.  We start with the land-use type |NADELW|.
For |lland_v1|, there is no structural difference between |NADELW| and,
for example, |ACKER|.  Different simulation results only are based solely
on different parameterisations.  In contrast, |lland_v3| handles |NADELW|
unlike all other land use types (even unlike |LAUBW| and |MISCHW|) regarding
the calculation of the surface resistance, relevant for the Penman-Monteith
equation (see method |Calc_LanduseSurfaceResistance_V1| for more details).
The calculated soil evapotranspiration (|EvB|) is similar to the one
calculated with the same input data for land-use type |ACKER| in the
integration test :ref:`lland_v3_acker_summer_daily`:

.. integration-test::

    >>> lnk(NADELW)
    >>> surfaceresistance.nadelw = 56.0
    >>> cropheight.nadelw = 10.0
    >>> albedo.nadelw = 0.12
    >>> lai.nadelw = 11.0
    >>> conditions_nadelw_summer = test(
    ...     "lland_v3_nadelw_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah, states.bowa),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |     g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |      evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |      ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |             86.2 |       2.4 |               100.7 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |             1.579769 |                  1.579769 |      99.120231 |   1.205601 |         0.307429 | 4.507941 | 20.198726 |         0.971564 |                  36.46712 |                15.690785 |                     15.690785 |            16.437729 | 100.0 |       16.437729 |               16.437729 | -0.25 | 10.933333 |   -2.6784 |             14.465201 |                  2.686395 |                  14.465201 |                  1.877708 |                      0.0 |    12.587493 |              0.0 |         12.587493 |             39.166667 |                 100.0 |               211.364559 |              122.747678 | 10.302 |  0.0 |  4.686428 |   2.2 | 1.110404 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 10.302 | 0.657815 |  0.0 |  0.0 | 0.0 |      0.0 | 0.657815 | 0.169352 |  0.00196 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   32.4284 | 80.533782 | 0.657815 |   0.0 |   0.0 |   0.0 |       0.0 | 0.169352 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.00196 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |             85.8 |       1.6 |               101.3 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           0.567985 |          1.6 |                 1.977914 |                      1.977914 |                          0.0 |                      0.125103 |                           0.125103 |              1.69705 |                   1.69705 |       99.60295 |   1.207284 |         0.302714 | 4.538923 | 20.167744 |         0.971855 |                 36.236444 |                15.628821 |                     15.628821 |            11.029892 | 100.0 |       11.029892 |               11.029892 | -0.25 |    19.028 |  0.895795 |              9.706305 |                  1.802599 |                   9.706305 |                   0.96939 |                      0.0 |     8.736914 |              0.0 |          8.736914 |                 58.75 |                 100.0 |               181.457925 |               115.35055 |    0.0 |  0.0 |  3.312002 | 0.094 | 1.809101 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.227814 | 0.002637 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.282605 | 78.724681 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.227814 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.002637 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |             82.4 |       1.5 |               101.6 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                 2.187158 |                      2.187158 |                          0.0 |                      0.136594 |                           0.136594 |             1.802218 |                  1.802218 |      99.797782 |   1.203775 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                15.566134 |                     15.566134 |            14.204272 | 100.0 |       14.204272 |               14.204272 | -0.25 | 15.208683 | -1.913579 |             12.499759 |                  2.321384 |                  12.499759 |                  1.386217 |                      0.0 |    11.113542 |              0.0 |         11.113542 |             62.666667 |                 100.0 |               184.734569 |              115.873944 |    0.0 |  0.0 |   4.24805 |   0.0 | 2.495048 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.121564 | 0.001407 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.946184 | 76.229633 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.121564 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.001407 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |             77.6 |       2.8 |               101.7 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           0.993973 |          2.8 |                 2.041337 |                      2.041337 |                          0.0 |                      0.128603 |                           0.128603 |             1.584077 |                  1.584077 |     100.115923 |    1.21051 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                15.502754 |                     15.502754 |            23.243371 | 100.0 |       23.243371 |               23.243371 | -0.25 | 20.753946 |  1.531325 |             20.454167 |                  3.798631 |                  20.454167 |                  3.108707 |                      0.0 |     17.34546 |              0.0 |          17.34546 |             33.571429 |                 100.0 |               209.622626 |              121.261768 |    0.0 |  0.0 |  7.655528 | 0.094 |  3.24309 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.064868 | 0.000751 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.164858 | 72.986543 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.064868 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000751 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |             74.9 |       3.0 |               101.8 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 2.133112 |                      2.133112 |                          0.0 |                      0.133641 |                           0.133641 |             1.597701 |                  1.597701 |     100.202299 |   1.208738 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                15.438712 |                     15.438712 |            24.870125 | 100.0 |       24.870125 |               24.870125 | -0.25 | 14.816194 | -1.909685 |              21.88571 |                  4.064489 |                   21.88571 |                  3.414868 |                      0.0 |    18.470842 |              0.0 |         18.470842 |             31.333333 |                 100.0 |               223.524712 |              123.686516 |    0.0 |  0.0 |  8.731502 |   0.0 | 3.618105 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.034614 | 0.000401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.824543 | 69.368437 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.034614 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000401 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |             77.9 |       3.5 |               101.8 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           1.242466 |          3.5 |                 2.327535 |                      2.327535 |                          0.0 |                      0.144219 |                           0.144219 |              1.81315 |                   1.81315 |       99.98685 |   1.201995 |         0.282971 | 4.666316 |  20.04035 |         0.973104 |                 35.273753 |                15.374034 |                     15.374034 |            23.731856 | 100.0 |       23.731856 |               23.731856 | -0.25 | 20.348477 |  0.232491 |             20.884034 |                  3.878463 |                  20.884034 |                  2.845741 |                      0.0 |    18.038293 |              0.0 |         18.038293 |             26.857143 |                 100.0 |               217.806222 |              122.172235 |    0.0 |  0.0 |  8.967691 |   0.0 | 3.538589 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.018471 | 0.000214 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342053 | 65.829848 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.018471 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000214 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |             73.5 |       3.8 |               101.9 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           1.348964 |          3.8 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             1.808493 |                  1.808493 |     100.091507 |   1.199521 |         0.277822 | 4.698958 | 20.007708 |         0.973436 |                  35.02347 |                 15.30875 |                      15.30875 |            24.541738 | 100.0 |       24.541738 |               24.541738 | -0.25 | 18.740175 | -1.067813 |             21.596729 |                  4.010821 |                  21.596729 |                  3.051087 |                      0.0 |    18.545642 |              0.0 |         18.545642 |             24.736842 |                 100.0 |                249.89699 |              127.503607 |    0.0 |  0.0 | 10.546527 |   0.0 | 3.943139 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.009856 | 0.000114 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.159866 | 61.886709 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.009856 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000114 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |             80.1 |       3.3 |               101.9 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           1.171468 |          3.3 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             1.983047 |                  1.983047 |      99.916953 |   1.198331 |         0.272591 | 4.731891 | 19.974776 |         0.973776 |                 34.769499 |                15.242885 |                     15.242885 |            22.720717 | 100.0 |       22.720717 |               22.720717 | -0.25 | 21.466219 |  0.293528 |             19.994231 |                  3.713214 |                  19.994231 |                   2.45312 |                      0.0 |    17.541111 |              0.0 |         17.541111 |             28.484848 |                 100.0 |               234.560208 |              124.470702 |    0.0 |  0.0 |  8.375029 |   0.0 | 3.447428 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.005259 | 0.000061 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.616338 | 58.439281 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.005259 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000061 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |             72.9 |       1.8 |               101.9 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           0.638983 |          1.8 |                 2.696965 |                      2.696965 |                          0.0 |                      0.163997 |                           0.163997 |             1.966088 |                  1.966088 |      99.933912 |   1.192729 |         0.267279 |   4.7651 | 19.941567 |         0.974124 |                 34.511921 |                15.176467 |                     15.176467 |            23.522954 | 100.0 |       23.522954 |               23.522954 | -0.25 | 19.654459 | -1.371448 |               20.7002 |                  3.844323 |                    20.7002 |                  2.688326 |                      0.0 |    18.011873 |              0.0 |         18.011873 |             52.222222 |                 100.0 |                290.49321 |              132.398807 |    0.0 |  0.0 |  7.801371 |   0.0 | 4.238315 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.002806 | 0.000032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.737786 | 54.200966 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.002806 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000032 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |             68.5 |       1.9 |               102.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                 2.813915 |                      2.813915 |                          0.0 |                      0.170178 |                           0.170178 |             1.927532 |                  1.927532 |     100.072468 |   1.191257 |         0.261887 | 4.798573 | 19.908094 |         0.974479 |                 34.250819 |                15.109521 |                     15.109521 |             23.07046 | 100.0 |        23.07046 |                23.07046 | -0.25 |  23.39262 |  0.203534 |             20.302005 |                  3.770372 |                  20.302005 |                   2.76572 |                      0.0 |    17.536286 |              0.0 |         17.536286 |             49.473684 |                 100.0 |               348.466656 |              138.318012 |    0.0 |  0.0 |  8.457407 |   0.0 | 4.453782 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.001498 | 0.000017 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.284252 | 49.747184 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001498 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000017 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |             68.6 |       3.0 |               102.1 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 2.830978 |                      2.830978 |                          0.0 |                      0.171077 |                           0.171077 |             1.942051 |                  1.942051 |     100.157949 |   1.191966 |         0.256418 | 4.832298 | 19.874369 |         0.974842 |                 33.986274 |                15.042071 |                     15.042071 |            23.295734 | 100.0 |       23.295734 |               23.295734 | -0.25 | 21.880839 | -0.632013 |             20.500246 |                  3.807188 |                  20.500246 |                  2.799361 |                      0.0 |    17.700884 |              0.0 |         17.700884 |             31.333333 |                 100.0 |               368.725681 |              139.570111 |    0.0 |  0.0 | 10.478761 |   0.0 | 4.442334 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000799 | 0.000009 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.666265 |  45.30485 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000799 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000009 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |             66.0 |       2.5 |               102.2 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           0.887476 |          2.5 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |             1.879766 |                  1.879766 |     100.320234 |   1.193016 |         0.250873 | 4.866263 | 19.840404 |         0.975213 |                  33.71837 |                14.974141 |                     14.974141 |            23.403884 | 100.0 |       23.403884 |               23.403884 | -0.25 | 23.154216 | -0.023734 |             20.595418 |                  3.824863 |                  20.595418 |                  2.986484 |                      0.0 |    17.608935 |              0.0 |         17.608935 |                  37.6 |                 100.0 |               419.711234 |              142.937942 |    0.0 |  0.0 |  9.985603 | 0.094 | 4.546047 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000426 | 0.000005 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.439999 | 40.758803 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000426 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000005 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |             69.0 |       1.6 |               101.9 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           0.567985 |          1.6 |                  2.84813 |                       2.84813 |                          0.0 |                       0.17198 |                            0.17198 |              1.96521 |                   1.96521 |       99.93479 |    1.18911 |         0.245254 | 4.900456 | 19.806211 |          0.97559 |                 33.447192 |                14.905755 |                     14.905755 |            22.610635 | 100.0 |       22.610635 |               22.610635 | -0.25 | 22.399998 | -0.414721 |             19.897359 |                  3.695224 |                  19.897359 |                  2.708378 |                      0.0 |    17.188981 |              0.0 |         17.188981 |                 58.75 |                 100.0 |               411.253273 |              141.649314 |    0.0 |  0.0 |  7.775599 |   0.0 | 4.347844 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000228 | 0.000003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.604721 | 36.410959 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000228 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |             75.6 |       2.7 |               101.5 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 2.600118 |                      2.600118 |                          0.0 |                       0.15885 |                            0.15885 |             1.965689 |                  1.965689 |      99.534311 |   1.190432 |         0.239562 | 4.934866 |   19.7718 |         0.975975 |                 33.172825 |                14.836934 |                     14.836934 |            19.695927 | 100.0 |       19.695927 |               19.695927 | -0.25 | 22.949068 |  0.647517 |             17.332416 |                  3.218877 |                  17.332416 |                  2.239353 |                      0.0 |    15.093063 |              0.0 |         15.093063 |             34.814815 |                 100.0 |               357.548831 |              136.707247 |    0.0 |  0.0 |  7.857065 |   0.0 | 3.459696 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.000121 | 0.000001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.707203 | 32.951263 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000121 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000001 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |             81.4 |       3.2 |               101.6 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           1.135969 |          3.2 |                 2.460535 |                      2.460535 |                          0.0 |                      0.151385 |                           0.151385 |             2.002875 |                  2.002875 |      99.597125 |   1.195095 |         0.233799 | 4.969483 | 19.737183 |         0.976367 |                 32.895357 |                  14.7677 |                       14.7677 |            18.693186 | 100.0 |       18.693186 |               18.693186 | -0.25 | 19.957345 | -0.436832 |             16.450004 |                  3.055001 |                  16.450004 |                  2.021579 |                      0.0 |    14.428425 |              0.0 |         14.428425 |                29.375 |                 100.0 |               333.908335 |              133.915514 |  0.338 |  0.0 |  7.162052 |   2.2 | 1.984501 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.338 | 0.008727 |  0.0 |  0.0 | 0.0 |      0.0 | 0.008727 | 0.002312 | 0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.894036 | 31.296034 | 0.008727 |   0.0 |   0.0 |   0.0 |       0.0 | 0.002312 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000027 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |             79.0 |       1.3 |               101.9 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                 2.270468 |                      2.270468 |                          0.0 |                      0.141127 |                           0.141127 |             1.793669 |                  1.793669 |     100.106331 |   1.204916 |         0.227967 | 5.004296 | 19.702371 |         0.976766 |                 32.614875 |                14.698075 |                     14.698075 |            19.581515 | 100.0 |       19.581515 |               19.581515 | -0.25 |  20.58012 |  0.559934 |             17.231733 |                  3.200179 |                  17.231733 |                  2.487815 |                      0.0 |    14.743918 |              0.0 |         14.743918 |             72.307692 |                 100.0 |               354.695699 |              135.231698 |    0.0 |  0.0 |  5.274274 |   0.0 | 3.081998 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.003057 | 0.000035 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.084102 | 28.214037 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003057 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 | 0.000035 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |             75.9 |       1.9 |               101.9 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                 2.490975 |                      2.490975 |                          0.0 |                      0.153018 |                           0.153018 |              1.89065 |                   1.89065 |      100.00935 |   1.198338 |         0.222067 | 5.039294 | 19.667372 |         0.977172 |                 32.331467 |                14.628078 |                     14.628078 |            20.349663 | 100.0 |       20.349663 |               20.349663 | -0.25 |  17.88034 | -1.617232 |             17.907703 |                  3.325716 |                  17.907703 |                  2.520953 |                      0.0 |     15.38675 |              0.0 |          15.38675 |             49.473684 |                 100.0 |               400.426126 |               138.20342 |    0.0 |  0.0 |  6.725557 |   0.0 | 3.429544 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.166895 |      0.0 | 0.001332 | 0.000015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.451334 | 24.951387 |      0.0 |   0.0 |   0.0 |   0.0 | -0.166895 | 0.001631 |   0.0 |   0.0 |   0.0 | -0.000299 |   0.0 | 0.000015 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |             70.0 |       2.0 |               101.8 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           0.709981 |          2.0 |                 2.584275 |                      2.584275 |                          0.0 |                      0.158006 |                           0.158006 |             1.808993 |                  1.808993 |      99.991007 |   1.195081 |         0.216102 | 5.074469 | 19.632198 |         0.977584 |                 32.045225 |                14.557728 |                     14.557728 |            20.118191 | 100.0 |       20.118191 |               20.118191 | -0.25 | 22.437779 |  0.434305 |             17.704008 |                  3.287887 |                  17.704008 |                  2.702703 |                      0.0 |    15.001304 |              0.0 |         15.001304 |                  47.0 |                 100.0 |               491.324867 |              142.949964 |    0.0 |  0.0 |   7.53971 |   0.0 | 3.733858 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.369623 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.767029 | 21.587152 |      0.0 |   0.0 |   0.0 |   0.0 | -0.369623 |  0.00087 |   0.0 |   0.0 |   0.0 | -0.001259 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |             70.3 |       2.6 |               101.9 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |             1.761941 |                  1.761941 |     100.138059 |   1.198506 |         0.210072 |  5.10981 | 19.596856 |         0.978003 |                 31.756238 |                14.487046 |                     14.487046 |            17.255246 | 100.0 |       17.255246 |               17.255246 | -0.25 | 20.156763 | -0.488974 |             15.184616 |                      2.82 |                  15.184616 |                  2.307856 |                      0.0 |     12.87676 |              0.0 |          12.87676 |             36.153846 |                 100.0 |               525.389413 |              143.816091 |    0.0 |  0.0 |  7.832014 |   0.0 |  3.37772 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.578662 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.006003 | 18.788095 |      0.0 |   0.0 |   0.0 |   0.0 | -0.578662 | 0.000464 |   0.0 |   0.0 |   0.0 | -0.002954 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |             69.0 |       3.6 |               101.9 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           1.277965 |          3.6 |                 2.713408 |                      2.713408 |                          0.0 |                      0.164868 |                           0.164868 |             1.872252 |                  1.872252 |     100.027748 |   1.192743 |          0.20398 | 5.145309 | 19.561357 |         0.978429 |                   31.4646 |                14.416048 |                     14.416048 |            20.197892 | 100.0 |       20.197892 |               20.197892 | -0.25 | 20.953343 | -0.749947 |             17.774145 |                  3.300913 |                  17.774145 |                  2.688476 |                      0.0 |    15.085669 |              0.0 |         15.085669 |             26.111111 |                 100.0 |               629.540329 |              146.963401 |    0.0 |  0.0 | 10.542903 |   0.0 | 3.855581 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.752584 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.50595 | 15.685099 |      0.0 |   0.0 |   0.0 |   0.0 | -0.752584 | 0.000248 |   0.0 |   0.0 |   0.0 | -0.005328 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |             62.1 |       2.5 |               101.7 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           0.887476 |          2.5 |                 2.729939 |                      2.729939 |                          0.0 |                      0.165743 |                           0.165743 |             1.695292 |                  1.695292 |     100.004708 |   1.190771 |         0.197828 | 5.180957 | 19.525709 |          0.97886 |                 31.170402 |                14.344752 |                     14.344752 |            21.264879 | 100.0 |       21.264879 |               21.264879 | -0.25 | 22.619833 |  0.062121 |             18.713094 |                  3.475289 |                  18.713094 |                  3.308913 |                      0.0 |    15.404181 |              0.0 |         15.404181 |                  37.6 |                 100.0 |               884.502325 |              152.120949 |    0.0 |  0.0 |  9.796451 |   0.0 |  4.29127 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.945392 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.193828 | 12.339221 |      0.0 |   0.0 |   0.0 |   0.0 | -0.945392 | 0.000132 |   0.0 |   0.0 |   0.0 | -0.008351 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |             84.5 |       2.1 |               101.5 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                 2.284619 |                      2.284619 |                          0.0 |                      0.141895 |                           0.141895 |             1.930503 |                  1.930503 |      99.569497 |   1.199129 |         0.191617 | 5.216746 | 19.489921 |         0.979299 |                  30.87374 |                14.273175 |                     14.273175 |             7.826588 | 100.0 |        7.826588 |                7.826588 | -0.25 | 21.579428 |  1.026136 |              6.887397 |                  1.279088 |                   6.887397 |                   0.60663 |                      0.0 |     6.280768 |              0.0 |          6.280768 |             44.761905 |                 100.0 |               696.001395 |              147.247892 |    0.0 |  0.0 |  3.397063 |  1.88 | 0.698098 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.153291 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.917693 | 12.794413 |      0.0 |   0.0 |   0.0 |   0.0 | -1.153291 | 0.000071 |   0.0 |   0.0 |   0.0 | -0.012081 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |             83.6 |       1.5 |               101.5 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                  2.38585 |                       2.38585 |                          0.0 |                      0.147368 |                           0.147368 |             1.994571 |                  1.994571 |      99.505429 |   1.195981 |          0.18535 | 5.252667 |    19.454 |         0.979743 |                 30.574709 |                14.201333 |                     14.201333 |            14.856047 | 100.0 |       14.856047 |               14.856047 | -0.25 | 17.325643 | -1.541907 |             13.073322 |                  2.427903 |                  13.073322 |                  1.641158 |                      0.0 |    11.432163 |              0.0 |         11.432163 |             62.666667 |                 100.0 |               667.674311 |              145.668066 |    0.0 |  0.0 |  4.351914 | 0.282 | 2.206487 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.125007 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   33.2096 | 11.712934 |      0.0 |   0.0 |   0.0 |   0.0 | -1.125007 | 0.000038 |   0.0 |   0.0 |   0.0 | -0.016118 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |             76.5 |       2.2 |               101.1 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           0.780979 |          2.2 |                 3.006626 |                      3.006626 |                          0.0 |                      0.180287 |                           0.180287 |             2.300069 |                  2.300069 |      98.799931 |   1.174651 |         0.179027 | 5.288712 | 19.417955 |         0.980193 |                 30.273404 |                14.129243 |                     14.129243 |            18.709904 | 100.0 |       18.709904 |               18.709904 | -0.25 | 21.631999 | -1.279412 |             16.464715 |                  3.057733 |                  16.464715 |                  1.868809 |                      0.0 |    14.595906 |              0.0 |         14.595906 |             42.727273 |                 100.0 |               938.453811 |              150.290353 |    0.0 |  0.0 |  7.236094 |   0.0 | 3.494179 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.192206 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 34.239011 |  9.410961 |      0.0 |   0.0 |   0.0 |   0.0 | -1.192206 |  0.00002 |   0.0 |   0.0 |   0.0 | -0.020212 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |             76.6 |       1.9 |               100.8 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                  2.93526 |                       2.93526 |                          0.0 |                      0.176554 |                           0.176554 |             2.248409 |                  2.248409 |      98.551591 |   1.172942 |         0.172651 | 5.324874 | 19.381792 |         0.980649 |                 29.969924 |                14.056918 |                     14.056918 |            14.421614 | 100.0 |       14.421614 |               14.421614 | -0.25 | 25.063372 |  0.706772 |              12.69102 |                  2.356904 |                   12.69102 |                  1.401603 |                      0.0 |    11.289418 |              0.0 |         11.289418 |             49.473684 |                 100.0 |              1353.717528 |               153.70968 |    0.0 |  0.0 |  5.830325 | 0.094 | 2.901447 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.335241 |      0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  33.28224 |  7.844755 |      0.0 |   0.0 |   0.0 |   0.0 | -1.335241 | 0.000011 |   0.0 |   0.0 |   0.0 | -0.024667 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |             84.2 |       2.3 |               100.6 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |           0.816478 |          2.3 |                 2.506318 |                      2.506318 |                          0.0 |                       0.15384 |                            0.15384 |              2.11032 |                   2.11032 |       98.48968 |   1.181557 |         0.166225 | 5.361147 |  19.34552 |         0.981111 |                 29.664367 |                13.984374 |                     13.984374 |            12.613157 | 100.0 |       12.613157 |               12.613157 | -0.25 | 21.874132 |   0.40131 |             11.099578 |                   2.06135 |                  11.099578 |                  1.261689 |                      0.0 |     9.837889 |              0.0 |          9.837889 |             40.869565 |                 100.0 |              1694.004243 |              154.756666 |   7.67 |  0.0 |  4.656548 |   2.2 | 1.080941 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   7.67 | 0.067757 |  0.0 |  0.0 | 0.0 | 1.432559 | 0.067757 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.63093 | 15.798616 | 0.067757 |   0.0 |   0.0 |   0.0 | -1.432559 |  0.01745 |   0.0 |   0.0 |   0.0 | -0.029537 |   0.0 |      0.0 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |             85.8 |       2.1 |               100.5 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                 2.475714 |                      2.475714 |                          0.0 |                        0.1522 |                             0.1522 |             2.124163 |                  2.124163 |      98.375837 |   1.181114 |         0.159748 | 5.397522 | 19.309145 |         0.981578 |                 29.356832 |                13.911624 |                     13.911624 |            14.302992 | 100.0 |       14.302992 |               14.302992 | -0.25 | 19.703098 | -0.620474 |             12.586633 |                  2.337518 |                  12.586633 |                  1.498964 |                      0.0 |    11.087669 |              0.0 |         11.087669 |             44.761905 |                 100.0 |               514.362016 |              137.508257 |   3.91 |  0.0 |  4.603863 |   2.2 | 1.172642 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   3.91 | 0.052955 |  0.0 |  0.0 | 0.0 | 0.938338 | 0.052955 | 0.003425 |  0.00004 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 33.001403 | 19.421358 | 0.052955 |   0.0 |   0.0 |   0.0 | -0.938338 | 0.037102 |   0.0 |   0.0 |   0.0 | -0.033677 |   0.0 |  0.00004 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |             86.5 |       2.6 |               100.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                 2.371153 |                      2.371153 |                          0.0 |                      0.146575 |                           0.146575 |             2.051047 |                  2.051047 |      97.948953 |   1.178323 |         0.153225 | 5.433993 | 19.272674 |         0.982051 |                 29.047419 |                13.838682 |                     13.838682 |             7.576705 | 100.0 |        7.576705 |                7.576705 | -0.25 | 20.938011 |  0.382585 |                6.6675 |                   1.23825 |                     6.6675 |                  0.589077 |                      0.0 |     6.078423 |              0.0 |          6.078423 |             36.153846 |                 100.0 |               423.391318 |               132.52466 |   11.9 |  0.0 |  3.480236 |   2.2 | 0.567552 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   11.9 |   0.2321 |  0.0 |  0.0 | 0.0 | 0.713236 |   0.2321 | 0.054101 | 0.000626 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.368818 | 31.234942 |   0.2321 |   0.0 |   0.0 |   0.0 | -0.713236 | 0.090616 |   0.0 |   0.0 |   0.0 | -0.036515 |   0.0 | 0.000626 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |             89.6 |       2.6 |               100.4 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                 1.832679 |                      1.832679 |                          0.0 |                      0.117031 |                           0.117031 |              1.64208 |                   1.64208 |       98.75792 |   1.201704 |         0.146656 | 5.470553 | 19.236114 |         0.982529 |                  28.73623 |                13.765561 |                     13.765561 |             7.288435 | 100.0 |        7.288435 |                7.288435 | -0.25 | 18.829395 |  1.414918 |              6.413823 |                  1.191138 |                   6.413823 |                  0.708412 |                      0.0 |     5.705411 |              0.0 |          5.705411 |             36.153846 |                 100.0 |               347.072847 |              127.000616 |  6.448 |  0.0 |  2.685024 |   2.2 | 0.203792 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  6.448 | 0.173145 |  0.0 |  0.0 | 0.0 |      0.0 | 0.173145 | 0.103764 | 0.001201 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   30.7039 | 37.306005 | 0.173145 |   0.0 |   0.0 |   0.0 |       0.0 | 0.141425 |   0.0 |   0.0 |   0.0 | -0.037661 |   0.0 | 0.001201 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |             78.3 |       2.1 |               101.1 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                 1.940693 |                      1.940693 |                          0.0 |                      0.123042 |                           0.123042 |             1.519563 |                  1.519563 |      99.580437 |   1.206937 |         0.140044 | 5.507197 |  19.19947 |         0.983012 |                 28.423368 |                13.692273 |                     13.692273 |            12.295515 | 100.0 |       12.295515 |               12.295515 | -0.25 | 13.279667 |  -1.92862 |             10.820053 |                  2.009438 |                  10.820053 |                  1.881434 |                      0.0 |      8.93862 |              0.0 |           8.93862 |             44.761905 |                 100.0 |               346.875642 |              126.428035 |    0.0 |  0.0 |  4.533079 |   0.0 | 2.160946 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.074117 | 0.000858 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.382521 | 35.145059 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.111643 |   0.0 |   0.0 |   0.0 | -0.037526 |   0.0 | 0.000858 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |             78.5 |       2.1 |               101.4 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                 2.093348 |                      2.093348 |                          0.0 |                      0.131461 |                           0.131461 |             1.643278 |                  1.643278 |      99.756722 |   1.204994 |          0.13339 | 5.543918 | 19.162749 |           0.9835 |                 28.108935 |                13.618831 |                     13.618831 |            10.948788 | 100.0 |       10.948788 |               10.948788 | -0.25 | 18.875069 |  0.349956 |              9.634933 |                  1.789345 |                   9.634933 |                  1.511299 |                      0.0 |     8.123635 |              0.0 |          8.123635 |             44.761905 |                 100.0 |               345.936719 |              125.805103 |    0.0 |  0.0 |   4.43981 |   0.0 | 2.165797 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 0.022183 | 0.000257 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.782565 | 32.979262 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.059574 |   0.0 |   0.0 |   0.0 | -0.037392 |   0.0 | 0.000257 |

.. _lland_v3_acker_winter_daily:

acre (winter)
-------------

None of the examples discussed so far dealt with snow processes.  Thus,
we now switch from a warm August to a snowy December and adapt the initial
conditions and the input time series accordingly:

>>> pub.timegrids = "2010-12-01", "2011-01-01", "1d"
>>> lnk(ACKER)
>>> lai.acker_dec = 0.3
>>> cropheight.acker_dec = 0.05
>>> albedo.acker_dec = 0.1
>>> wg2z.dec = 0.7416
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

>>> test.inits = (
...     (states.inzp, 0.0),
...     (states.wats, 0.0),
...     (states.waes, 0.0),
...     (states.esnow, 0.0),
...     (states.taus, 0.0),
...     (states.ebdn, 0.0),
...     (states.bowa, 72.0),
...     (states.qdgz1, 0.0),
...     (states.qdgz2, 0.0),
...     (states.qigz1, 0.0),
...     (states.qigz2, 0.0),
...     (states.qbgz, 0.0),
...     (states.qdga1, 0.0),
...     (states.qdga2, 0.0),
...     (states.qiga1, 0.0),
...     (states.qiga2, 0.0),
...     (states.qbga, 0.0))

>>> inputs.nied.series = (
...     0.0, 11.1, 0.4, 1.2, 6.4, 0.0, 0.0, 0.0, 0.7, 3.3, 9.4, 0.2, 2.9, 0.0,
...     0.0, 6.7, 0.0, 0.1, 0.0, 0.2, 0.2, 2.4, 7.0, 0.2, 0.0, 1.2, 0.9, 0.0,
...     0.0, 0.2, 0.9)
>>> inputs.teml.series = (
...     -5.8, -3.0, -5.6, -3.8, 0.2, -1.1, -2.9, -2.9, -2.1, -3.1, 5.1, 0.2,
...     -3.5, -3.9, -6.3, -3.5, -7.3, -5.8, -7.9, -5.7, -5.6, -4.8, -1.1,
...     -2.9, -9.1, -8.6, -5.3, -7.7, -9.2, -7.4, 1.7)
>>> inputs.sunshineduration.series = (
...     3.5, 0.0, 0.1, 3.7, 0.0, 0.1, 0.0, 0.0, 4.4, 0.1, 0.2, 0.7, 0.0, 1.2,
...     3.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.8, 0.0, 0.0, 0.0, 3.2, 4.3, 0.0, 3.4,
...     0.0, 0.0, 4.0)
>>> inputs.windspeed.series = (
...     6.9, 5.2, 2.0, 4.3, 4.0, 2.4, 1.9, 1.8, 2.7, 3.0, 7.4, 5.7, 2.2, 3.1,
...     2.8, 3.7, 0.9, 1.3, 1.9, 2.4, 2.3, 4.6, 8.6, 7.6, 3.8, 1.5, 1.3, 3.0,
...     2.7, 1.4, 4.2)
>>> inputs.relativehumidity.series = (
...     70.3, 85.0, 95.9, 90.6, 96.5, 99.0, 99.2, 99.1, 96.9, 95.0, 90.6,
...     78.7, 90.5, 88.1, 92.8, 98.1, 96.6, 97.7, 95.8, 97.5, 95.0, 93.8,
...     94.9, 85.4, 87.2, 92.7, 97.8, 95.4, 94.9, 96.0, 96.5)
>>> inputs.atmosphericpressure.series = (
...     101.5, 100.7, 100.5, 100.8, 99.6, 99.8, 100.4, 100.6, 100.9, 101.8,
...     100.4, 101.2, 102.0, 102.5, 102.3, 100.0, 99.1, 99.7, 99.7, 99.7,
...     100.4, 100.7, 100.4, 100.8, 101.8, 101.9, 101.7, 102.2, 102.4, 102.1,
...     101.6)
>>> inlet.sequences.sim.series = 0.0

When comparing the following results with those of integration test
:ref:`lland_v1_acker_winter` on |lland_v1|, it is not immediately clear what
differences are due to the much higher complexities of the snow module of
|lland_v3|.  At least, one can see that |lland_v3| allows for the (re)freezing
of liquid water within the snow-layer.  Refreezing  occurs around December 7th,
where the frozen water equivalent (|WATS|) grows until it reaches to total
water equivalent (|WAeS|) (a note for `LARSIM`_ users: currently, `LARSIM`_
does not implement such a freezing feature; you can disable it in |lland_v3|
through setting |RefreezeFlag| to |False|):

.. integration-test::

    >>> conditions_acker_winter = test(
    ...     "lland_v3_acker_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada),
    ...     axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |      g |         tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes |    sbes |      evpo | evi |     evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |  wlatsnow | wsenssnow |     wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |      ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |             70.3 |       6.9 |               101.5 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |            5.53098 |          6.9 |                 0.395811 |                      0.395811 |                          0.0 |                      0.030356 |                           0.030356 |             0.278255 |                  0.278255 |     101.221745 |   1.321192 |        -0.389384 | 8.532819 | 16.173848 |         1.028719 |                  5.761386 |                 7.641029 |                      7.641029 |             2.759858 | 100.0 |        2.759858 |                2.759858 | 0.7416 |        0.0 |   3.00672 |              2.483872 |                  2.483872 |                   2.483872 |                   4.01525 |                      0.0 |    -1.531378 |              0.0 |         -1.531378 |             30.870386 |                 100.0 |               111.309423 |               99.512695 |    0.0 |     0.0 |  1.808179 | 0.0 | 0.55374 |       0.0 |       0.0 |          nan |          0.1 |       0.0 |       0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |  -2.26512 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |             85.0 |       5.2 |               100.7 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |           4.168274 |          5.2 |                 0.489349 |                      0.489349 |                     0.381881 |                      0.036626 |                           0.036626 |             0.415947 |                  0.415947 |     100.284053 |   1.296511 |        -0.391536 | 8.548511 | 16.158156 |         1.028994 |                  5.687365 |                 7.609644 |                      7.609644 |             0.853105 | 100.0 |        0.853105 |                0.853105 | 0.7416 |    -7.5504 | -0.665713 |              0.106229 |                  0.106229 |                   0.106229 |                  1.277835 |                 2.406897 |    -1.171606 |        -2.300668 |         -1.171606 |             40.962628 |                 100.0 |               112.102145 |               99.530051 | 10.374 |  10.374 |  0.779906 | 0.0 |     0.0 | -0.168357 | -0.065045 |    -6.266232 |     0.875479 |       0.0 |       0.0 | 0.406773 |      0.0 | -0.449083 |  -2.44648 | -0.594896 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 10.542357 | 10.542357 | -0.135862 | 0.084415 | -0.857807 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |             95.9 |       2.0 |               100.5 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 0.401924 |                      0.401924 |                     0.357079 |                      0.030771 |                           0.030771 |             0.385445 |                  0.385445 |     100.114555 |   1.306656 |        -0.393572 | 8.563421 | 16.143245 |          1.02926 |                  5.617593 |                 7.579824 |                      7.579824 |             1.441454 | 100.0 |        1.441454 |                1.441454 | 0.7416 |  -2.859357 |  2.216454 |              0.200535 |                  0.200535 |                   0.200535 |                  1.329454 |                 3.241865 |    -1.128919 |         -3.04133 |         -1.128919 |            106.502833 |                 100.0 |               112.102145 |               99.527226 |  0.376 |   0.376 |  0.039557 | 0.0 |     0.0 | -0.073821 | -0.004401 |    -7.134925 |      0.86088 |       0.0 |       0.0 | 0.487468 |      0.0 | -0.196913 | -0.605411 |  2.239006 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 10.992177 | 10.992177 | -0.162814 |   0.1418 | -2.332662 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |             90.6 |       4.3 |               100.8 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           3.446842 |          4.3 |                 0.460812 |                      0.460812 |                     0.323611 |                       0.03473 |                            0.03473 |             0.417496 |                  0.417496 |     100.382504 |   1.301648 |        -0.395492 | 8.577535 | 16.129131 |         1.029518 |                  5.552051 |                 7.551596 |                      7.551596 |             2.748162 | 100.0 |        2.748162 |                2.748162 | 0.7416 |  -7.775538 |  0.321724 |              0.408995 |                  0.408995 |                   0.408995 |                  3.727317 |                 4.828741 |    -3.318322 |        -4.419745 |         -3.318322 |             49.536202 |                 100.0 |               112.102145 |               99.524551 |  1.128 |   1.128 |  0.092108 | 0.0 |     0.0 | -0.402208 | -0.008959 |    -8.396148 |     0.851175 |       0.0 |       0.0 | 0.636779 |      0.0 |  -1.07287 |  -2.98424 |  0.362635 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 12.522386 | 12.522386 | -0.212684 | 0.183432 | -1.912786 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |             96.5 |       4.0 |                99.6 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           3.206365 |          4.0 |                 0.619748 |                      0.619748 |                     0.491255 |                      0.045128 |                           0.045128 |             0.598057 |                  0.598057 |      99.001943 |   1.266438 |        -0.397294 | 8.590839 | 16.115828 |         1.029767 |                  5.490726 |                 7.524989 |                      7.524989 |             0.823609 | 100.0 |        0.823609 |                0.823609 | 0.7416 |  -6.375953 | -1.777027 |              0.126225 |                  0.126225 |                   0.126225 |                  1.140917 |                 1.372429 |    -1.014692 |        -1.246204 |         -1.014692 |             53.251417 |                 100.0 |               112.102145 |               99.522031 |  6.016 |  2.7072 |  0.134777 | 0.0 |     0.0 | -0.434119 |  0.003902 |    -2.948046 |     0.846742 |       0.0 |       0.0 | 0.403221 | 0.403221 |  -1.15799 | -1.939347 | -1.851133 | 0.977326 |  0.28655 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 15.998885 | 18.972504 |       0.0 | 0.203476 |  0.605841 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |             99.0 |       2.4 |                99.8 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 0.563475 |                      0.563475 |                     0.515157 |                      0.041489 |                           0.041489 |             0.557841 |                  0.557841 |      99.242159 |   1.275245 |        -0.398979 | 8.603319 | 16.103347 |         1.030007 |                  5.433602 |                 7.500028 |                      7.500028 |             1.394624 | 100.0 |        1.394624 |                1.394624 | 0.7416 |        0.0 |  1.198162 |               0.26017 |                   0.26017 |                    0.26017 |                  1.217715 |                 2.425437 |    -0.957545 |        -2.165267 |         -0.957545 |             88.752361 |                 100.0 |               112.102145 |               99.519666 |    0.0 |     0.0 |  0.000586 | 0.0 |     0.0 | -0.123565 |       0.0 |    -2.311269 |     0.813448 |       0.0 |       0.0 | 0.317541 | 0.317541 | -0.329604 | -0.531443 |  1.304221 | 0.994413 | 0.296657 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 16.422692 | 19.096069 |       0.0 | 0.378868 |  0.149279 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |             99.2 |       1.9 |               100.4 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 0.493024 |                      0.493024 |                     0.462365 |                      0.036869 |                           0.036869 |             0.489079 |                  0.489079 |      99.910921 |   1.291808 |        -0.400545 | 8.614964 | 16.091703 |         1.030238 |                  5.380663 |                 7.476739 |                      7.476739 |             0.807099 | 100.0 |        0.807099 |                0.807099 | 0.7416 |        0.0 |  1.946783 |              0.171372 |                  0.171372 |                   0.171372 |                  1.193664 |                 2.797224 |    -1.022292 |        -2.625852 |         -1.022292 |            112.108246 |                 100.0 |               112.102145 |                99.51746 |    0.0 |     0.0 | -0.015911 | 0.0 |     0.0 |  -0.06757 |       0.0 |    -3.755368 |     0.787669 |       0.0 |       0.0 | 0.511768 | 0.511768 | -0.180239 | -0.327899 |  2.117714 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 16.994381 | 19.163639 |       0.0 |  0.55426 | -1.055904 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |             99.1 |       1.8 |               100.6 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 0.493024 |                      0.493024 |                     0.422362 |                      0.036869 |                           0.036869 |             0.488586 |                  0.488586 |     100.111414 |   1.294388 |        -0.401992 | 8.625761 | 16.080906 |         1.030461 |                  5.331896 |                 7.455145 |                      7.455145 |             0.799784 | 100.0 |        0.799784 |                0.799784 | 0.7416 |   -3.51968 |  0.741623 |              0.186256 |                  0.186256 |                   0.186256 |                  1.194208 |                 2.345793 |    -1.007952 |        -2.159537 |         -1.007952 |            118.336481 |                 100.0 |               112.102145 |               99.515415 |    0.0 |     0.0 | -0.012641 | 0.0 |     0.0 |  -0.16266 |       0.0 |    -4.950281 |     0.767118 |       0.0 |       0.0 | 0.661033 | 0.661033 | -0.433888 | -0.763241 |  0.962408 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 17.805273 | 19.326299 |       0.0 | 0.729652 | -1.055927 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |             96.9 |       2.7 |               100.9 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 0.523311 |                      0.523311 |                     0.389718 |                      0.038864 |                           0.038864 |             0.507088 |                  0.507088 |     100.392912 |   1.294334 |        -0.403321 | 8.635699 | 16.070967 |         1.030674 |                  5.287288 |                 7.435268 |                      7.435268 |             2.886262 | 100.0 |        2.886262 |                2.886262 | 0.7416 |  -3.519758 |  1.286849 |              0.707155 |                  0.707155 |                   0.707155 |                  4.005891 |                 5.077714 |    -3.298735 |        -4.370559 |         -3.298735 |             78.890988 |                 100.0 |               112.102145 |               99.513532 |  0.658 |   0.658 | -0.235007 | 0.0 |     0.0 | -0.365518 | -0.002888 |    -6.002106 |     0.754993 |       0.0 |       0.0 | 0.807885 | 0.807885 |    -0.975 | -1.841764 |  1.553795 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 19.623632 | 20.349817 |       0.0 |  0.85302 | -1.601177 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |             95.0 |       3.0 |               101.8 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 0.485698 |                      0.485698 |                     0.402698 |                      0.036384 |                           0.036384 |             0.461413 |                  0.461413 |     101.338587 |   1.310959 |         -0.40453 | 8.644768 | 16.061898 |         1.030879 |                  5.246826 |                  7.41713 |                       7.41713 |             1.347076 | 100.0 |        1.347076 |                1.347076 | 0.7416 |  -5.337255 |  0.123184 |               0.31535 |                   0.31535 |                    0.31535 |                  1.286731 |                 2.472681 |    -0.971381 |        -2.157331 |         -0.971381 |             71.001889 |                 100.0 |               112.102145 |               99.511814 |  3.102 |   3.102 |  0.137189 | 0.0 |     0.0 | -0.195734 | -0.020098 |    -5.574878 |       0.7659 |       0.0 |       0.0 | 0.843525 | 0.726185 | -0.522111 | -1.250397 |  0.384823 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 23.647551 | 23.647551 | -0.039192 | 0.741291 |  -0.98276 |  71.44626 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |             90.6 |       7.4 |               100.4 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           5.931775 |          7.4 |                 0.879017 |                      0.879017 |                      0.61078 |                      0.061412 |                           0.061412 |              0.79639 |                   0.79639 |       99.60361 |   1.253213 |        -0.405619 | 8.652959 | 16.053708 |         1.031074 |                  5.210498 |                 7.400749 |                      7.400749 |             1.373029 | 100.0 |        1.373029 |                1.373029 | 0.7416 |  -3.275868 |  -1.69821 |               0.34539 |                   0.34539 |                    0.34539 |                   1.16859 |                -0.065197 |      -0.8232 |         0.410586 |           -0.8232 |              28.78455 |                 100.0 |               112.102145 |               99.510262 |  8.836 |     0.0 |  1.033005 | 0.0 |     0.0 | -1.215856 |  0.188672 |          0.0 |     0.748447 | 21.462327 | 21.462327 |      0.0 |      0.0 | -3.243236 | -5.063324 | -8.717146 |  0.94547 | 0.268174 | 30.457573 | 9.778899 |  0.0 |  0.0 | 0.0 |  0.0 | 9.778899 | 2.517544 | 0.029138 | 0.06 |  2.267017 |  3.241834 |       0.0 | 0.927231 |  1.457049 | 92.124934 | 9.778899 |   0.0 |   0.0 |   0.0 |  0.0 | 2.517544 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029138 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |             78.7 |       5.7 |               101.2 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |            4.56907 |          5.7 |                 0.619748 |                      0.619748 |                     0.517645 |                      0.045128 |                           0.045128 |             0.487741 |                  0.487741 |     100.712259 |    1.28736 |        -0.406588 | 8.660262 | 16.046405 |          1.03126 |                  5.178293 |                 7.386143 |                      7.386143 |             1.539952 | 100.0 |        1.539952 |                1.539952 | 0.7416 |        0.0 |  1.164579 |              0.407516 |                  0.407516 |                   0.407516 |                  1.733999 |                 3.137362 |    -1.326483 |        -2.729846 |         -1.326483 |             37.369415 |                 100.0 |                83.817217 |               98.586605 |  0.188 |  0.0846 |  1.429349 | 0.0 |     0.0 |  0.158718 |  0.000066 |    -2.246487 |     0.735371 |       0.0 |       0.0 | 0.061533 | 0.061533 |  0.423374 | -1.968023 |  1.185197 |   0.9613 | 0.277229 |  0.067022 | 0.022382 |  0.0 |  0.0 | 0.0 |  0.0 | 0.022382 | 3.392378 | 0.039264 | 0.06 |  2.299254 |  3.204093 |       0.0 | 1.094821 |  1.034071 | 92.169574 | 0.022382 |   0.0 |   0.0 |   0.0 |  0.0 | 3.392378 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039264 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |             90.5 |       2.2 |               102.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 0.471336 |                      0.471336 |                     0.432007 |                      0.035431 |                           0.035431 |             0.426559 |                  0.426559 |     101.573441 |   1.315658 |        -0.407436 | 8.666669 | 16.039997 |         1.031437 |                  5.150203 |                 7.373328 |                      7.373328 |              0.77253 | 100.0 |         0.77253 |                 0.77253 | 0.7416 |        0.0 |  2.412276 |              0.193285 |                  0.193285 |                   0.193285 |                  1.254261 |                 3.092635 |    -1.060976 |        -2.899351 |         -1.060976 |             96.820757 |                 100.0 |                83.758368 |               98.584459 |  2.726 |   2.726 |  0.193889 | 0.0 |     0.0 |  0.014973 | -0.019941 |    -4.653311 |     0.749803 |       0.0 |       0.0 | 0.199105 | 0.199105 |  0.039939 | -0.480453 |  2.458837 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 1.814887 | 0.021006 | 0.06 |  5.211168 |   5.91512 |       0.0 | 0.911369 | -0.636606 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.814887 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021006 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |             88.1 |       3.1 |               102.5 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |           2.484933 |          3.1 |                 0.457351 |                      0.457351 |                     0.383823 |                      0.034499 |                           0.034499 |             0.402926 |                  0.402926 |     102.097074 |   1.324197 |        -0.408164 | 8.672175 | 16.034492 |         1.031605 |                  5.126218 |                 7.362317 |                      7.362317 |             1.699321 | 100.0 |        1.699321 |                1.699321 | 0.7416 |  -2.122018 |    2.1142 |              0.448703 |                  0.448703 |                   0.448703 |                  2.107297 |                 4.010007 |    -1.658594 |        -3.561304 |         -1.658594 |             68.711505 |                 100.0 |                83.758368 |               98.584802 |    0.0 |     0.0 |  0.281648 | 0.0 |     0.0 | -0.065079 |       0.0 |    -6.200337 |     0.735952 |       0.0 |       0.0 | 0.256899 | 0.256899 | -0.173596 | -1.187703 |  2.200005 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.968444 | 0.011209 | 0.06 |  5.528228 |    5.9802 |       0.0 | 1.086761 | -2.009206 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968444 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011209 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |             92.8 |       2.8 |               102.3 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 0.380889 |                      0.380889 |                     0.282608 |                       0.02934 |                            0.02934 |             0.353465 |                  0.353465 |     101.946535 |    1.33374 |         -0.40877 | 8.676772 | 16.029895 |         1.031763 |                  5.106331 |                 7.353123 |                      7.353123 |             2.318249 | 100.0 |        2.318249 |                2.318249 | 0.7416 |  -6.697353 |  1.768945 |              0.639262 |                  0.639262 |                   0.639262 |                  3.397964 |                 4.986924 |    -2.758702 |        -4.347662 |         -2.758702 |             76.073452 |                 100.0 |                83.758368 |               98.585088 |    0.0 |     0.0 | -0.028526 | 0.0 |     0.0 | -0.225845 |       0.0 |   -10.109669 |     0.724248 |       0.0 |       0.0 | 0.406999 | 0.406999 | -0.602429 |  -1.84035 |  1.904882 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.516772 | 0.005981 | 0.06 |  6.159374 |  6.206044 |       0.0 | 1.262153 | -3.036551 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516772 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.005981 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |             98.1 |       3.7 |               100.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           2.965888 |          3.7 |                 0.471336 |                      0.471336 |                     0.357858 |                      0.035431 |                           0.035431 |             0.462381 |                  0.462381 |      99.537619 |   1.289645 |        -0.409256 | 8.680456 | 16.026211 |         1.031912 |                  5.090537 |                 7.345755 |                      7.345755 |             0.763581 | 100.0 |        0.763581 |                0.763581 | 0.7416 | -10.121835 | -1.562975 |              0.163977 |                  0.163977 |                   0.163977 |                  1.211777 |                 1.915561 |      -1.0478 |        -1.751584 |           -1.0478 |             57.569099 |                 100.0 |                83.758368 |               98.585317 |  6.298 |   6.298 |  0.035293 | 0.0 |     0.0 |  -0.40193 |  -0.04607 |    -7.106837 |     0.785252 |       0.0 |       0.0 | 0.558149 | 0.046671 | -1.072127 |  -2.10208 | -1.422623 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.275755 | 0.003192 | 0.06 | 12.905974 | 12.905974 | -0.170834 | 0.573017 | -0.731975 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275755 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003192 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |             96.6 |       0.9 |                99.1 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.721432 |          0.9 |                 0.352531 |                      0.352531 |                     0.326115 |                      0.027395 |                           0.027395 |             0.340545 |                  0.340545 |      98.759455 |   1.296889 |         -0.40962 | 8.683222 | 16.023444 |         1.032051 |                  5.078832 |                 7.340222 |                      7.340222 |             0.761825 | 100.0 |        0.761825 |                0.761825 | 0.7416 |  -2.439918 |  3.036773 |              0.169175 |                  0.169175 |                   0.169175 |                  1.287913 |                 3.600107 |    -1.118738 |        -3.430932 |         -1.118738 |            236.672963 |                 100.0 |                83.758368 |               98.585489 |    0.0 |     0.0 | -0.001256 | 0.0 |     0.0 | -0.025948 |       0.0 |    -8.297891 |     0.777934 |       0.0 |       0.0 | 0.670129 |      0.0 | -0.069214 | -0.271956 |  3.089762 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.147146 | 0.001703 | 0.06 | 12.931922 | 12.931922 | -0.223823 | 0.632681 | -3.027148 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147146 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001703 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |             97.7 |       1.3 |                99.7 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 0.395811 |                      0.395811 |                     0.283712 |                      0.030356 |                           0.030356 |             0.386708 |                  0.386708 |      99.313292 |   1.297204 |        -0.409863 | 8.685068 | 16.021598 |         1.032182 |                  5.071211 |                  7.33653 |                       7.33653 |             0.760682 | 100.0 |        0.760682 |                0.760682 | 0.7416 | -10.090494 | -0.015431 |              0.172921 |                  0.172921 |                   0.172921 |                  1.257034 |                 2.130829 |    -1.084113 |        -1.957908 |         -1.084113 |            163.850513 |                 100.0 |                83.758368 |               98.585603 |  0.094 |   0.094 | -0.002818 | 0.0 |     0.0 | -0.215326 | -0.001139 |   -10.060727 |     0.772676 |       0.0 |       0.0 | 0.820045 |      0.0 | -0.574371 | -1.350035 |  0.033501 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.078519 | 0.000909 | 0.06 | 13.241248 | 13.241248 | -0.273895 | 0.678422 | -2.270117 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078519 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000909 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |             95.8 |       1.9 |                99.7 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  0.33643 |                       0.33643 |                     0.272279 |                      0.026283 |                           0.026283 |               0.3223 |                    0.3223 |        99.3777 |   1.307793 |        -0.409985 | 8.685992 | 16.020675 |         1.032302 |                  5.067673 |                 7.334683 |                      7.334683 |             0.760151 | 100.0 |        0.760151 |                0.760151 | 0.7416 |  -7.567057 |  1.559857 |              0.176614 |                  0.176614 |                   0.176614 |                  1.301361 |                 3.118579 |    -1.124747 |        -2.941965 |         -1.124747 |            112.108246 |                 100.0 |                83.758368 |               98.585661 |    0.0 |     0.0 |  0.032531 | 0.0 |     0.0 | -0.126518 |       0.0 |    -10.57604 |      0.76766 |       0.0 |       0.0 | 0.876298 |      0.0 | -0.337479 |  -1.02584 |  1.578645 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.041898 | 0.000485 | 0.06 | 13.367765 | 13.367765 | -0.292684 | 0.724518 | -3.088374 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041898 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000485 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |             97.5 |       2.4 |                99.7 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 0.398857 |                      0.398857 |                     0.293041 |                      0.030563 |                           0.030563 |             0.388886 |                  0.388886 |      99.311114 |   1.296708 |        -0.409985 | 8.685992 | 16.020675 |         1.032413 |                  5.068219 |                 7.334683 |                      7.334683 |             1.336154 | 100.0 |        1.336154 |                1.336154 | 0.7416 | -10.294579 | -0.332219 |              0.314734 |                  0.314734 |                   0.314734 |                  1.393316 |                 2.435899 |    -1.078582 |        -2.121165 |         -1.078582 |             88.752361 |                 100.0 |                83.758368 |               98.585661 |  0.188 |   0.188 |  0.020944 | 0.0 |     0.0 |  -0.27746 |  -0.00224 |    -9.653725 |     0.764448 |       0.0 |       0.0 | 0.818877 |      0.0 |  -0.74011 | -1.734692 | -0.353637 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.022357 | 0.000259 | 0.06 | 13.833225 | 13.833225 | -0.273505 | 0.755386 | -2.014555 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022357 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000259 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |             95.0 |       2.3 |               100.4 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 0.401924 |                      0.401924 |                     0.312843 |                      0.030771 |                           0.030771 |             0.381828 |                  0.381828 |     100.018172 |   1.305372 |        -0.409863 | 8.685068 | 16.021598 |         1.032515 |                  5.072848 |                  7.33653 |                       7.33653 |             1.544792 | 100.0 |        1.544792 |                1.544792 | 0.7416 |  -6.715183 |  1.094569 |              0.369069 |                  0.369069 |                   0.369069 |                  1.820398 |                 3.345756 |     -1.45133 |        -2.976687 |          -1.45133 |             92.611159 |                 100.0 |                83.758368 |               98.585603 |  0.188 |   0.188 |  0.034932 | 0.0 |     0.0 |  -0.19466 |   -0.0022 |     -8.82662 |     0.761089 |       0.0 |       0.0 | 0.774426 |      0.0 | -0.519245 |  -1.37992 |  1.077522 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.01193 | 0.000138 | 0.06 | 14.215885 | 14.215885 | -0.258658 | 0.788873 | -2.367524 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.01193 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000138 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |             93.8 |       4.6 |               100.7 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |            3.68732 |          4.6 |                 0.427219 |                      0.427219 |                     0.344223 |                      0.032479 |                           0.032479 |             0.400731 |                  0.400731 |     100.299269 |   1.305282 |         -0.40962 | 8.683222 | 16.023444 |         1.032607 |                  5.081565 |                 7.340222 |                      7.340222 |             0.762235 | 100.0 |        0.762235 |                0.762235 | 0.7416 |  -7.891745 | -0.147882 |              0.171942 |                  0.171942 |                   0.171942 |                  1.259735 |                 2.598961 |    -1.087793 |        -2.427019 |         -1.087793 |              46.30558 |                 100.0 |                83.758368 |               98.585489 |  2.256 |   2.256 |  0.238205 | 0.0 |     0.0 | -0.254479 | -0.022632 |    -7.606479 |     0.774424 |       0.0 |       0.0 | 0.784019 |      0.0 | -0.678809 |  -1.91552 |  -0.16731 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.006366 | 0.000074 | 0.06 | 16.726363 | 16.726363 | -0.261862 | 0.662934 | -1.478042 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006366 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000074 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |             94.9 |       8.6 |               100.4 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           6.893685 |          8.6 |                 0.563475 |                      0.563475 |                     0.487656 |                      0.041489 |                           0.041489 |             0.534738 |                  0.534738 |      99.865262 |    1.28304 |        -0.409256 | 8.680456 | 16.026211 |         1.032689 |                  5.094372 |                 7.345755 |                      7.345755 |             0.764156 | 100.0 |        0.764156 |                0.764156 | 0.7416 |  -4.926805 | -0.974852 |              0.143863 |                  0.143863 |                   0.143863 |                  1.179808 |                 2.170447 |    -1.035945 |        -2.026585 |         -1.035945 |             24.768101 |                 100.0 |                83.758368 |               98.585317 |   6.58 |  5.0995 |  0.461634 | 0.0 |     0.0 | -0.349728 | -0.018542 |    -3.046303 |     0.811736 |       0.0 |       0.0 | 0.472584 | 0.472584 | -0.932883 | -2.191115 | -1.097414 | 0.991077 |  0.29467 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.003397 | 0.000039 | 0.06 | 22.633051 | 23.656092 |       0.0 | 0.389279 |  0.238411 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003397 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000039 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |             85.4 |       7.6 |               100.8 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           6.092093 |          7.6 |                 0.493024 |                      0.493024 |                     0.451022 |                      0.036869 |                           0.036869 |             0.421042 |                  0.421042 |     100.378958 |   1.297296 |         -0.40877 | 8.676772 | 16.029895 |         1.032762 |                  5.111276 |                 7.353123 |                      7.353123 |             0.766691 | 100.0 |        0.766691 |                0.766691 | 0.7416 |        0.0 |  2.117452 |              0.162963 |                  0.162963 |                   0.162963 |                  1.273508 |                 3.158372 |    -1.110545 |        -2.995409 |         -1.110545 |             28.027061 |                 100.0 |                83.758368 |               98.585088 |  0.188 |   0.188 |  1.132438 | 0.0 |     0.0 |   0.20077 | -0.001139 |    -4.084591 |     0.787447 |       0.0 |       0.0 | 0.635671 | 0.635671 |  0.535543 | -1.202325 |  2.328626 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.001813 | 0.000021 | 0.06 | 23.259214 | 23.643322 |       0.0 | 0.555966 | -1.137441 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000021 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |             87.2 |       3.8 |               101.8 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                  0.30617 |                       0.30617 |                     0.260819 |                      0.024175 |                           0.024175 |             0.266981 |                  0.266981 |     101.533019 |   1.341718 |        -0.408164 | 8.672175 | 16.034492 |         1.032825 |                  5.132282 |                 7.362317 |                      7.362317 |             2.398433 | 100.0 |        2.398433 |                2.398433 | 0.7416 |  -3.791471 |  3.795094 |              0.558983 |                  0.558983 |                   0.558983 |                  3.730222 |                  6.17159 |    -3.171238 |        -5.612606 |         -3.171238 |             56.054123 |                 100.0 |                83.758368 |               98.584802 |    0.0 |     0.0 |  0.146109 | 0.0 |     0.0 | -0.024146 |       0.0 |   -11.112254 |     0.766938 |       0.0 |       0.0 | 1.670829 | 0.384108 | -0.064407 | -1.195048 |  4.353151 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000967 | 0.000011 | 0.06 | 23.667468 | 23.667468 | -0.429765 | 0.731358 | -4.190935 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000967 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000011 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |             92.7 |       1.5 |               101.9 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 0.318472 |                      0.318472 |                     0.181906 |                      0.025034 |                           0.025034 |             0.295223 |                  0.295223 |     101.604777 |   1.340358 |        -0.407436 | 8.666669 | 16.039997 |         1.032879 |                  5.157401 |                 7.373328 |                      7.373328 |             2.793204 | 100.0 |        2.793204 |                2.793204 | 0.7416 | -13.969784 |  0.796549 |               0.64132 |                   0.64132 |                    0.64132 |                  4.423241 |                 4.808909 |    -3.781921 |        -4.167588 |         -3.781921 |            142.003778 |                 100.0 |                83.758368 |               98.584459 |  1.128 |   1.128 |  -0.16868 | 0.0 |     0.0 | -0.253474 | -0.020275 |   -15.506336 |       0.7704 |       0.0 |       0.0 | 2.405921 |      0.0 | -0.676129 | -2.341372 |  1.150087 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000516 | 0.000006 | 0.06 | 25.048942 | 25.048942 | -0.803578 | 0.699031 | -4.245884 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000006 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |             97.8 |       1.3 |               101.7 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                  0.41125 |                       0.41125 |                     0.253736 |                      0.031402 |                           0.031402 |             0.402202 |                  0.402202 |     101.297798 |   1.320718 |        -0.406588 | 8.660262 | 16.046405 |         1.032922 |                  5.186639 |                 7.386143 |                      7.386143 |             0.777996 | 100.0 |        0.777996 |                0.777996 | 0.7416 | -14.152947 | -1.399016 |              0.176319 |                  0.176319 |                   0.176319 |                  1.247575 |                  1.36221 |    -1.071255 |         -1.18589 |         -1.071255 |            163.850513 |                 100.0 |                83.758368 |               98.584061 |  0.846 |   0.846 | -0.002438 | 0.0 |     0.0 | -0.310389 | -0.009371 |   -11.454227 |     0.773367 |       0.0 |       0.0 | 1.856011 |      0.0 | -0.827947 | -1.950001 | -1.592057 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000275 | 0.000003 | 0.06 | 26.205331 | 26.205331 | -0.619908 | 0.672265 | -2.105268 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000275 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000003 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |             95.4 |       3.0 |               102.2 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 0.341723 |                      0.341723 |                     0.255303 |                      0.026649 |                           0.026649 |             0.326003 |                  0.326003 |     101.873997 |   1.339598 |        -0.405619 | 8.652959 | 16.053708 |         1.032956 |                   5.22001 |                 7.400749 |                      7.400749 |             2.504073 | 100.0 |        2.504073 |                2.504073 | 0.7416 |  -7.017558 |  2.260351 |              0.579004 |                  0.579004 |                   0.579004 |                    3.6896 |                 5.329433 |    -3.110597 |         -4.75043 |         -3.110597 |             71.001889 |                 100.0 |                83.758368 |               98.583607 |    0.0 |     0.0 | -0.122157 | 0.0 |     0.0 | -0.235684 |       0.0 |   -11.377804 |     0.768775 |       0.0 |       0.0 | 1.865726 |      0.0 | -0.628675 | -1.858158 |  2.263596 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000147 | 0.000002 | 0.06 | 26.441015 | 26.441015 | -0.623153 |  0.71405 | -3.624019 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |             94.9 |       2.7 |               102.4 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 0.303761 |                      0.303761 |                     0.225291 |                      0.024006 |                           0.024006 |              0.28827 |                   0.28827 |      102.11173 |   1.350039 |         -0.40453 | 8.644768 | 16.061898 |         1.032981 |                  5.257524 |                  7.41713 |                       7.41713 |             0.788629 | 100.0 |        0.788629 |                0.788629 | 0.7416 | -12.080062 |  0.434923 |              0.185809 |                  0.185809 |                   0.185809 |                  1.325064 |                 2.990028 |    -1.139254 |        -2.804219 |         -1.139254 |             78.890988 |                 100.0 |                83.758368 |               98.583098 |    0.0 |     0.0 |  0.072905 | 0.0 |     0.0 | -0.196129 |       0.0 |   -12.919034 |     0.764389 |       0.0 |       0.0 | 2.137509 |      0.0 | -0.523164 | -1.755355 |  0.525699 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000078 | 0.000001 | 0.06 | 26.637144 | 26.637144 | -0.713928 | 0.755957 | -3.317342 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |             96.0 |       1.4 |               102.1 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           1.122228 |          1.4 |                 0.349801 |                      0.349801 |                     0.247548 |                      0.027207 |                           0.027207 |             0.335809 |                  0.335809 |     101.764191 |   1.336727 |        -0.403321 | 8.635699 | 16.070967 |         1.032995 |                  5.299194 |                 7.435268 |                      7.435268 |             0.794879 | 100.0 |        0.794879 |                0.794879 | 0.7416 | -11.057806 |  0.364008 |               0.18917 |                   0.18917 |                    0.18917 |                  1.292505 |                 2.434873 |    -1.103335 |        -2.245703 |         -1.103335 |            152.146905 |                 100.0 |                83.758368 |               98.582534 |  0.188 |   0.188 |  0.018935 | 0.0 |     0.0 | -0.190974 | -0.002908 |   -11.759981 |     0.762015 |       0.0 |       0.0 | 1.974006 |      0.0 | -0.509415 | -1.429798 |   0.30649 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000042 |      0.0 | 0.06 | 27.016118 | 27.016118 | -0.659318 | 0.779513 |  -2.93975 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |             96.5 |       4.2 |               101.6 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |           3.366683 |          4.2 |                 0.690794 |                      0.690794 |                     0.446985 |                      0.049663 |                           0.049663 |             0.666616 |                  0.666616 |     100.933384 |   1.284547 |        -0.401992 | 8.625761 | 16.080906 |            1.033 |                  5.345035 |                 7.455145 |                      7.455145 |             2.770177 | 100.0 |        2.770177 |                2.770177 | 0.7416 |  -9.799165 | -2.900775 |              0.669518 |                  0.669518 |                   0.669518 |                  3.475519 |                 3.601606 |    -2.806001 |        -2.932088 |         -2.806001 |             50.715635 |                 100.0 |                83.758368 |               98.581916 |  0.846 | 0.06345 | -0.128252 | 0.0 |     0.0 | -0.924858 |  0.005795 |    -4.203534 |     0.758312 |       0.0 |       0.0 | 0.753524 | 0.753524 | -2.467012 | -3.767696 | -3.302621 | 0.973704 |  0.28443 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000022 |      0.0 | 0.06 | 28.756986 | 28.786976 |       0.0 | 0.817531 |  0.702625 | 92.169574 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

.. _lland_v3_nadelw_winter_daily:

conifers (winter)
-----------------

Next, we repeat the integration test :ref:`lland_v3_acker_winter_daily` but
select the land use type |NADELW| instead of |ACKER|.  Like for |LAUBW| and
|MISCHW|, |lland_v3| now modifies the global radiation, the net longwave
radiation, and the wind speed to take the shadowing effects of the tree-canopies
into account.  Here, this net-effect of these modifications is that the total
water equivalent's peak amounts are two to three times smaller than for land
use type |ACKER|.  Apart from this, snow evaporation (|EvS|) coincides with
interception evaporation (|EvI|) and soil evaporation (|EvB|), which never
happens non-forest sites:

.. integration-test::

    >>> lnk(NADELW)
    >>> conditions_nadelw_winter = test(
    ...     "lland_v3_nadelw_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada),
    ...     axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |      g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |     nbes |     sbes |     evpo |      evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |  wlatsnow | wsenssnow |     wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |      qah |       qa |     inzp |      wats |      waes |     esnow |     taus |      ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |             70.3 |       6.9 |               101.5 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |           2.449434 |          6.9 |                 0.395811 |                      0.395811 |                          0.0 |                      0.030356 |                           0.030356 |             0.278255 |                  0.278255 |     101.221745 |   1.321192 |        -0.389384 | 8.532819 | 16.173848 |         1.028719 |                  5.761386 |                 7.641029 |                      7.641029 |             2.759858 | 100.0 |        2.759858 |                2.759858 | 0.7416 |       0.0 |   3.00672 |              2.428675 |                   0.45104 |                   2.428675 |                   4.01525 |                      0.0 |    -1.586575 |              0.0 |         -1.586575 |             13.623188 |                 100.0 |             27827.355778 |              101.116829 |      0.0 |      0.0 | 4.142976 |      0.0 | 0.674992 |       0.0 |       0.0 |          nan |         0.12 |       0.0 |       0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      nan |  -2.26512 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |             85.0 |       5.2 |               100.7 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |            1.84595 |          5.2 |                 0.489349 |                      0.489349 |                     0.400893 |                      0.036626 |                           0.036626 |             0.415947 |                  0.415947 |     100.284053 |   1.296511 |        -0.391536 | 8.548511 | 16.158156 |         1.028994 |                  5.687365 |                 7.609644 |                      7.609644 |             0.853105 | 100.0 |        0.853105 |                0.853105 | 0.7416 |   -7.5504 | -0.993695 |              0.106229 |                  0.019728 |                   0.106229 |                  1.277835 |                 0.311864 |    -1.171606 |        -0.292136 |         -1.171606 |             18.076923 |                 100.0 |              2039.690308 |               99.509579 |    8.234 |    8.234 | 1.794481 | 1.794481 |      0.0 |  -0.04251 | -0.051627 |    -5.633549 |     0.875479 |       0.0 |       0.0 | 0.290264 |      0.0 | -0.113393 | -1.127117 | -0.948375 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.405519 |   8.27651 |   8.27651 | -0.096948 | 0.084415 | -0.529825 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |             95.9 |       2.0 |               100.5 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           0.709981 |          2.0 |                 0.401924 |                      0.401924 |                     0.416597 |                      0.030771 |                           0.030771 |             0.385445 |                  0.385445 |     100.114555 |   1.306656 |        -0.393572 | 8.563421 | 16.143245 |          1.02926 |                  5.617593 |                 7.579824 |                      7.579824 |             1.441454 | 100.0 |        1.441454 |                1.441454 | 0.7416 | -1.766082 |  1.744215 |              0.202598 |                  0.037625 |                   0.202598 |                  1.329454 |                 1.497933 |    -1.126856 |        -1.460308 |         -1.126856 |                  47.0 |                 100.0 |             28069.018521 |              100.749738 |      0.0 |      0.0 | 0.129568 | 0.129568 |      0.0 |  0.055691 |       0.0 |    -5.130694 |     0.859449 |       0.0 |       0.0 | 0.265719 |      0.0 |  0.148552 |  0.127157 |  1.736017 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.651951 |  8.220819 |  8.220819 |  -0.08875 | 0.147756 |  -1.53244 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |             90.6 |       4.3 |               100.8 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           1.526459 |          4.3 |                 0.460812 |                      0.460812 |                     0.401412 |                       0.03473 |                            0.03473 |             0.417496 |                  0.417496 |     100.382504 |   1.301648 |        -0.395492 | 8.577535 | 16.129131 |         1.029518 |                  5.552051 |                 7.551596 |                      7.551596 |             2.748162 | 100.0 |        2.748162 |                2.748162 | 0.7416 | -5.108132 |  0.263613 |              0.427177 |                  0.079333 |                   0.427177 |                  3.727317 |                 1.156608 |     -3.30014 |        -1.077275 |          -3.30014 |             21.860465 |                 100.0 |              3347.213388 |                99.76651 |      0.0 |      0.0 |  0.58437 |  0.58437 |      0.0 | -0.040732 |       0.0 |    -5.616644 |     0.844559 |       0.0 |       0.0 | 0.288929 |      0.0 |  -0.10865 |  -0.69726 |  0.271365 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.195582 |  8.261551 |  8.261551 | -0.096502 | 0.213597 | -1.054452 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |             96.5 |       4.0 |                99.6 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           1.419962 |          4.0 |                 0.619748 |                      0.619748 |                     0.532291 |                      0.045128 |                           0.045128 |             0.598057 |                  0.598057 |      99.001943 |   1.266438 |        -0.397294 | 8.590839 | 16.115828 |         1.029767 |                  5.490726 |                 7.524989 |                      7.524989 |             0.823609 | 100.0 |        0.823609 |                0.823609 | 0.7416 | -3.514841 | -0.852339 |                0.1347 |                  0.025016 |                     0.1347 |                  1.140917 |                 0.333623 |    -1.006217 |        -0.308607 |         -1.006217 |                  23.5 |                 100.0 |               763.990266 |               96.624998 | 5.011582 | 2.255212 | 0.342571 | 0.342571 |      0.0 |  -0.16016 |  0.003251 |    -1.870669 |     0.836452 |       0.0 |       0.0 | 0.187741 | 0.187741 | -0.427218 | -0.764275 | -0.882885 |  0.97981 | 0.288008 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.857429 | 10.833669 | 13.433292 |       0.0 | 0.252724 |  0.539486 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |             99.0 |       2.4 |                99.8 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                 0.563475 |                      0.563475 |                     0.547452 |                      0.041489 |                           0.041489 |             0.557841 |                  0.557841 |      99.242159 |   1.275245 |        -0.398979 | 8.603319 | 16.103347 |         1.030007 |                  5.433602 |                 7.500028 |                      7.500028 |             1.394624 | 100.0 |        1.394624 |                1.394624 | 0.7416 |       0.0 |  0.773005 |              0.271156 |                  0.050358 |                   0.271156 |                  1.217715 |                 1.040137 |    -0.946558 |        -0.989779 |         -0.946558 |             39.166667 |                 100.0 |              1010.452619 |               97.419093 |      0.0 |      0.0 |  0.03349 |  0.03349 |      0.0 | -0.019917 |       0.0 |    -1.491136 |      0.80557 |       0.0 |       0.0 | 0.149678 | 0.149678 | -0.053127 | -0.113655 |  0.822997 | 0.980985 | 0.288699 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.823939 | 10.999632 | 13.453209 |       0.0 | 0.428116 |  0.508082 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |             99.2 |       1.9 |               100.4 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                 0.493024 |                      0.493024 |                     0.503686 |                      0.036869 |                           0.036869 |             0.489079 |                  0.489079 |      99.910921 |   1.291808 |        -0.400545 | 8.614964 | 16.091703 |         1.030238 |                  5.380663 |                 7.476739 |                      7.476739 |             0.807099 | 100.0 |        0.807099 |                0.807099 | 0.7416 |       0.0 |  1.354861 |              0.176396 |                  0.032759 |                   0.176396 |                  1.193664 |                 1.329952 |    -1.017268 |        -1.297192 |         -1.017268 |             49.473684 |                 100.0 |               1874.96551 |               98.634658 |      0.0 |      0.0 |  0.00082 |  0.00082 |      0.0 |  0.025639 |       0.0 |    -2.613543 |     0.781444 |       0.0 |       0.0 | 0.260273 | 0.260273 |   0.06839 |  0.076209 |  1.441792 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.823119 | 11.238446 |  13.42757 |       0.0 | 0.603508 | -0.105179 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |             99.1 |       1.8 |               100.6 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           0.638983 |          1.8 |                 0.493024 |                      0.493024 |                      0.49875 |                      0.036869 |                           0.036869 |             0.488586 |                  0.488586 |     100.111414 |   1.294388 |        -0.401992 | 8.625761 | 16.080906 |         1.030461 |                  5.331896 |                 7.455145 |                      7.455145 |             0.799784 | 100.0 |        0.799784 |                0.799784 | 0.7416 | -0.350597 |  1.241498 |              0.190288 |                  0.035339 |                   0.190288 |                  1.194208 |                 1.279419 |    -1.003919 |         -1.24408 |         -1.003919 |             52.222222 |                 100.0 |              1875.428739 |                98.51545 |      0.0 |      0.0 | 0.005396 | 0.005396 |      0.0 |  0.017512 |       0.0 |    -2.745463 |     0.762076 |       0.0 |       0.0 | 0.268412 | 0.268412 |  0.046713 |  0.040355 |  1.331148 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.817722 | 11.491851 | 13.410058 |       0.0 |   0.7789 | -0.605077 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |             96.9 |       2.7 |               100.9 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 0.523311 |                      0.523311 |                     0.479126 |                      0.038864 |                           0.038864 |             0.507088 |                  0.507088 |     100.392912 |   1.294334 |        -0.403321 | 8.635699 | 16.070967 |         1.030674 |                  5.287288 |                 7.435268 |                      7.435268 |             2.886262 | 100.0 |        2.886262 |                2.886262 | 0.7416 | -2.016925 |  0.655677 |              0.726842 |                  0.134985 |                   0.726842 |                  4.005891 |                 1.407552 |    -3.279049 |        -1.272567 |         -3.279049 |             34.814815 |                 100.0 |              1366.137242 |               97.824519 | 0.275722 | 0.275722 | -0.14603 | -0.14603 |      0.0 | -0.056327 |  -0.00121 |    -3.281734 |     0.748172 |       0.0 |       0.0 | 0.320562 | 0.320562 |  -0.15025 | -0.360783 |  0.761535 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |  2.34603 | 12.137888 | 13.742107 |       0.0 |  0.93048 | -0.519155 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |             95.0 |       3.0 |               101.8 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 0.485698 |                      0.485698 |                     0.472483 |                      0.036384 |                           0.036384 |             0.461413 |                  0.461413 |     101.338587 |   1.310959 |         -0.40453 | 8.644768 | 16.061898 |         1.030879 |                  5.246826 |                  7.41713 |                       7.41713 |             1.347076 | 100.0 |        1.347076 |                1.347076 | 0.7416 | -1.730516 |  0.900543 |              0.319875 |                  0.059405 |                   0.319875 |                  1.286731 |                 1.129022 |    -0.966856 |        -1.069617 |         -0.966856 |             31.333333 |                 100.0 |              2093.665722 |               98.468592 |  3.24803 |  3.24803 | 0.332893 | 0.332893 |      0.0 |  0.023373 | -0.021044 |    -3.467675 |     0.762541 |       0.0 |       0.0 | 0.403591 | 0.403591 |  0.062346 | -0.117664 |  1.014299 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.867107 | 15.767788 | 16.966765 |       0.0 | 0.774233 | -0.678098 | 71.325008 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |             90.6 |       7.4 |               100.4 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           2.626929 |          7.4 |                 0.879017 |                      0.879017 |                      0.61078 |                      0.061412 |                           0.061412 |              0.79639 |                   0.79639 |       99.60361 |   1.253213 |        -0.405619 | 8.652959 | 16.053708 |         1.031074 |                  5.210498 |                 7.400749 |                      7.400749 |             1.373029 | 100.0 |        1.373029 |                1.373029 | 0.7416 | -2.260327 | -1.171754 |              0.349316 |                  0.064873 |                   0.349316 |                   1.16859 |                -1.000249 |    -0.819274 |         1.065122 |         -0.819274 |             12.702703 |                 100.0 |               405.842399 |               92.890929 | 8.503107 |      0.0 |  2.30022 |      2.2 | 0.020632 | -0.656354 |  0.119173 |          0.0 |     0.745587 | 13.463057 | 13.463057 |      0.0 |      0.0 | -1.750792 | -2.733328 | -5.549242 |  0.95377 | 0.272903 | 22.734524 | 7.340048 |  0.0 |  0.0 | 0.0 |  0.0 | 7.340048 |  1.88967 | 0.021871 |      0.0 |  2.371819 |  3.391701 |       0.0 | 0.961549 |  1.235255 | 86.698853 | 7.340048 |   0.0 |   0.0 |   0.0 |  0.0 |  1.88967 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021871 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |             78.7 |       5.7 |               101.2 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |           2.023445 |          5.7 |                 0.619748 |                      0.619748 |                     0.556893 |                      0.045128 |                           0.045128 |             0.487741 |                  0.487741 |     100.712259 |    1.28736 |        -0.406588 | 8.660262 | 16.046405 |          1.03126 |                  5.178293 |                 7.386143 |                      7.386143 |             1.539952 | 100.0 |        1.539952 |                1.539952 | 0.7416 |       0.0 |  0.652909 |               0.41208 |                  0.076529 |                    0.41208 |                  1.733999 |                 0.850749 |    -1.321918 |         -0.77422 |         -1.321918 |             16.491228 |                 100.0 |               655.980194 |               95.337563 |      0.0 |      0.0 | 3.264426 |    0.188 | 0.683853 |  0.206471 |       0.0 |     -1.25947 |     0.732407 |       0.0 |       0.0 | 0.034794 | 0.034794 |  0.550751 | -0.660441 |   0.66453 | 0.950451 | 0.271007 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 2.541996 | 0.029421 |      0.0 |   2.26011 |   3.18523 |       0.0 | 1.136941 |  1.323946 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.541996 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029421 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |             90.5 |       2.2 |               102.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           0.780979 |          2.2 |                 0.471336 |                      0.471336 |                     0.481086 |                      0.035431 |                           0.035431 |             0.426559 |                  0.426559 |     101.573441 |   1.315658 |        -0.407436 | 8.666669 | 16.039997 |         1.031437 |                  5.150203 |                 7.373328 |                      7.373328 |              0.77253 | 100.0 |         0.77253 |                 0.77253 | 0.7416 |       0.0 |  1.673029 |              0.211479 |                  0.039275 |                   0.211479 |                  1.254261 |                 1.393742 |    -1.042782 |        -1.354467 |         -1.042782 |             42.727273 |                 100.0 |              2194.505356 |               98.289962 |    0.526 |    0.526 | 0.463478 | 0.463478 |      0.0 |  0.101011 | -0.003848 |    -3.227294 |     0.726252 |       0.0 |       0.0 | 0.093691 | 0.093691 |  0.269442 |  0.076566 |  1.700474 |  0.98531 | 0.291251 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 1.356437 |   0.0157 | 1.736522 |  2.801419 |  3.610219 |       0.0 | 1.230059 |  0.392517 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.356437 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0157 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |             88.1 |       3.1 |               102.5 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |            1.10047 |          3.1 |                 0.457351 |                      0.457351 |                     0.464648 |                      0.034499 |                           0.034499 |             0.402926 |                  0.402926 |     102.097074 |   1.324197 |        -0.408164 | 8.672175 | 16.034492 |         1.031605 |                  5.126218 |                 7.362317 |                      7.362317 |             1.699321 | 100.0 |        1.699321 |                1.699321 | 0.7416 |       0.0 |  1.912884 |              0.482688 |                  0.089642 |                   0.482688 |                  2.107297 |                 1.615415 |    -1.624609 |        -1.525773 |         -1.624609 |             30.322581 |                 100.0 |              3007.346718 |               98.587687 |      0.0 |      0.0 |  0.74779 |  0.74779 |      0.0 |  0.132324 |       0.0 |    -3.689978 |     0.715953 |       0.0 |       0.0 | 0.102096 | 0.102096 |  0.352969 |  0.068242 |  1.946984 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.72381 | 0.008377 | 0.988732 |  2.797093 |  3.477895 |       0.0 | 1.405451 | -0.778767 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.72381 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.008377 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |             92.8 |       2.8 |               102.3 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           0.993973 |          2.8 |                 0.380889 |                      0.380889 |                     0.384339 |                       0.02934 |                            0.02934 |             0.353465 |                  0.353465 |     101.946535 |    1.33374 |         -0.40877 | 8.676772 | 16.029895 |         1.031763 |                  5.106331 |                 7.353123 |                      7.353123 |             2.318249 | 100.0 |        2.318249 |                2.318249 | 0.7416 | -2.595891 |  1.859484 |              0.679124 |                  0.126123 |                   0.679124 |                  3.397964 |                 1.834473 |     -2.71884 |         -1.70835 |          -2.71884 |             33.571429 |                 100.0 |             22986.147223 |               99.386062 |      0.0 |      0.0 | 0.185123 | 0.185123 |      0.0 |  0.063193 |       0.0 |    -6.182859 |     0.707053 |       0.0 |       0.0 | 0.160982 | 0.160982 |  0.168564 |  0.036338 |  1.913252 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.386233 |  0.00447 | 0.803609 |  2.904328 |  3.414702 |       0.0 | 1.580843 | -1.896651 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.386233 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00447 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |             98.1 |       3.7 |               100.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           1.313465 |          3.7 |                 0.471336 |                      0.471336 |                     0.410569 |                      0.035431 |                           0.035431 |             0.462381 |                  0.462381 |      99.537619 |   1.289645 |        -0.409256 | 8.680456 | 16.026211 |         1.031912 |                  5.090537 |                 7.345755 |                      7.345755 |             0.763581 | 100.0 |        0.763581 |                0.763581 | 0.7416 | -6.322172 | -0.518643 |              0.190798 |                  0.035434 |                   0.190798 |                  1.211777 |                 0.547842 |    -1.020979 |        -0.512408 |         -1.020979 |             25.405405 |                 100.0 |              2155.023275 |               98.113406 | 4.901609 | 4.901609 | 0.125412 | 0.125412 |      0.0 | -0.121146 | -0.035855 |    -5.321703 |     0.750127 |       0.0 |       0.0 | 0.293988 | 0.293988 | -0.323152 | -0.645563 | -0.456306 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.206098 | 0.002385 | 2.074588 |  8.217918 |  8.437457 |       0.0 | 0.907623 | -0.636409 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.206098 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002385 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |             96.6 |       0.9 |                99.1 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.319491 |          0.9 |                 0.352531 |                      0.352531 |                     0.386658 |                      0.027395 |                           0.027395 |             0.340545 |                  0.340545 |      98.759455 |   1.296889 |         -0.40962 | 8.683222 | 16.023444 |         1.032051 |                  5.078832 |                 7.340222 |                      7.340222 |             0.761825 | 100.0 |        0.761825 |                0.761825 | 0.7416 | -2.121362 |   2.06495 |              0.200952 |                   0.03732 |                   0.200952 |                  1.287913 |                 1.777299 |    -1.086961 |         -1.73998 |         -1.086961 |            104.444444 |                 100.0 |             22986.147223 |               99.310938 |      0.0 |      0.0 | 0.027234 | 0.027234 |      0.0 |  0.066013 |       0.0 |    -6.104677 |     0.736223 |       0.0 |       0.0 | 0.330724 | 0.219538 |  0.176087 |  0.259345 |  2.175412 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.109976 | 0.001273 | 2.047354 |  8.371443 |  8.371443 | -0.037136 | 1.083015 | -1.959759 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.109976 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001273 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |             97.7 |       1.3 |                99.7 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                 0.395811 |                      0.395811 |                     0.358783 |                      0.030356 |                           0.030356 |             0.386708 |                  0.386708 |      99.313292 |   1.297204 |        -0.409863 | 8.685068 | 16.021598 |         1.032182 |                  5.071211 |                  7.33653 |                       7.33653 |             0.760682 | 100.0 |        0.760682 |                0.760682 | 0.7416 |  -6.53253 |  0.280465 |              0.205822 |                  0.038224 |                   0.205822 |                  1.257034 |                 0.822912 |    -1.051212 |        -0.784688 |         -1.051212 |             72.307692 |                 100.0 |             22986.147223 |                99.28946 |      0.0 |      0.0 | 0.029397 | 0.029397 |      0.0 | -0.043592 |       0.0 |    -7.073551 |     0.729424 |       0.0 |       0.0 | 0.370542 |      0.0 |  -0.11628 | -0.301317 |   0.36709 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.058685 | 0.000679 | 2.111958 |  8.415036 |  8.415036 | -0.123761 | 1.181072 | -1.498624 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.058685 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000679 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |             95.8 |       1.9 |                99.7 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                  0.33643 |                       0.33643 |                     0.339349 |                      0.026283 |                           0.026283 |               0.3223 |                    0.3223 |        99.3777 |   1.307793 |        -0.409985 | 8.685992 | 16.020675 |         1.032302 |                  5.067673 |                 7.334683 |                      7.334683 |             0.760151 | 100.0 |        0.760151 |                0.760151 | 0.7416 | -4.995414 |  1.448382 |              0.208451 |                  0.038712 |                   0.208451 |                  1.301361 |                 1.391065 |    -1.092909 |        -1.352352 |         -1.092909 |             49.473684 |                 100.0 |             22986.147223 |               99.278718 |      0.0 |      0.0 | 0.112247 | 0.112247 |      0.0 |  0.029927 |       0.0 |    -7.789361 |     0.725776 |       0.0 |       0.0 | 0.410164 |      0.0 |  0.079829 |  0.029434 |  1.461616 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.031315 | 0.000362 |  1.99971 |  8.385109 |  8.385109 | -0.136995 | 1.237594 | -2.205406 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.031315 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000362 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |             97.5 |       2.4 |                99.7 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                 0.398857 |                      0.398857 |                     0.351395 |                      0.030563 |                           0.030563 |             0.388886 |                  0.388886 |      99.311114 |   1.296708 |        -0.409985 | 8.685992 | 16.020675 |         1.032413 |                  5.068219 |                 7.334683 |                      7.334683 |             1.336154 | 100.0 |        1.336154 |                1.336154 | 0.7416 | -7.351355 | -0.005088 |              0.370787 |                   0.06886 |                   0.370787 |                  1.393316 |                 0.724167 |    -1.022529 |        -0.655307 |         -1.022529 |             39.166667 |                 100.0 |             22986.147223 |               99.278718 |      0.0 |      0.0 |  0.09212 |  0.09212 |      0.0 |  -0.07188 |       0.0 |     -7.34154 |     0.722497 |       0.0 |       0.0 | 0.385208 |      0.0 | -0.191736 | -0.476994 | -0.013423 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.01671 | 0.000193 | 2.095591 |  8.456988 |  8.456988 |  -0.12866 | 1.290963 | -1.458718 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.01671 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000193 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |             95.0 |       2.3 |               100.4 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |           0.816478 |          2.3 |                 0.401924 |                      0.401924 |                     0.376955 |                      0.030771 |                           0.030771 |             0.381828 |                  0.381828 |     100.018172 |   1.305372 |        -0.409863 | 8.685068 | 16.021598 |         1.032515 |                  5.072848 |                  7.33653 |                       7.33653 |             1.544792 | 100.0 |        1.544792 |                1.544792 | 0.7416 | -4.862394 |  0.815086 |              0.432758 |                  0.080369 |                   0.432758 |                  1.820398 |                 1.143627 |    -1.387641 |        -1.063258 |         -1.387641 |             40.869565 |                 100.0 |             22986.147223 |                99.28946 | 0.083591 | 0.083591 | 0.166883 | 0.166883 |      0.0 | -0.009185 | -0.000978 |    -6.434706 |      0.71986 |       0.0 |       0.0 | 0.343887 |      0.0 | -0.024501 |  -0.23845 |  0.800307 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.008917 | 0.000103 | 2.033117 |  8.549764 |  8.549764 | -0.114858 | 1.335751 | -1.532205 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008917 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000103 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |             93.8 |       4.6 |               100.7 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |           1.632956 |          4.6 |                 0.427219 |                      0.427219 |                     0.393866 |                      0.032479 |                           0.032479 |             0.400731 |                  0.400731 |     100.299269 |   1.305282 |         -0.40962 | 8.683222 | 16.023444 |         1.032607 |                  5.081565 |                 7.340222 |                      7.340222 |             0.762235 | 100.0 |        0.762235 |                0.762235 | 0.7416 | -5.107348 |  0.392372 |              0.202934 |                  0.037688 |                   0.202934 |                  1.259735 |                 0.896934 |    -1.056801 |        -0.859246 |         -1.056801 |             20.434783 |                 100.0 |             16306.259519 |               99.258272 | 2.089117 | 2.089117 | 0.580802 | 0.580802 |      0.0 | -0.018053 | -0.020958 |    -5.864239 |     0.733764 |       0.0 |       0.0 | 0.390398 |      0.0 | -0.048156 | -0.424141 |  0.386948 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.004758 | 0.000055 | 1.619198 | 10.656935 | 10.656935 | -0.130393 | 1.117444 | -1.182976 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004758 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000055 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |             94.9 |       8.6 |               100.4 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           3.052917 |          8.6 |                 0.563475 |                      0.563475 |                     0.501973 |                      0.041489 |                           0.041489 |             0.534738 |                  0.534738 |      99.865262 |    1.28304 |        -0.409256 | 8.680456 | 16.026211 |         1.032689 |                  5.094372 |                 7.345755 |                      7.345755 |             0.764156 | 100.0 |        0.764156 |                0.764156 | 0.7416 | -3.943255 |  -0.66566 |              0.173225 |                   0.03217 |                   0.173225 |                  1.179808 |                 0.589788 |    -1.006584 |        -0.557618 |         -1.006584 |             10.930233 |                 100.0 |               837.172765 |               96.048177 | 5.999198 | 4.649378 | 1.086523 | 1.086523 |      0.0 | -0.128595 | -0.016905 |    -2.659188 |     0.773312 |       0.0 |       0.0 | 0.299689 | 0.299689 |  -0.34302 |  -0.92746 | -0.712863 | 0.991606 | 0.294985 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.002539 | 0.000029 | 1.113477 | 15.726489 | 16.784727 |       0.0 | 0.672752 |  0.224284 | 86.014999 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002539 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000029 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |             85.4 |       7.6 |               100.8 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           2.697927 |          7.6 |                 0.493024 |                      0.493024 |                     0.485964 |                      0.036869 |                           0.036869 |             0.421042 |                  0.421042 |     100.378958 |   1.297296 |         -0.40877 | 8.676772 | 16.029895 |         1.032762 |                  5.111276 |                 7.353123 |                      7.353123 |             0.766691 | 100.0 |        0.766691 |                0.766691 | 0.7416 |       0.0 |  1.603251 |              0.187501 |                  0.034822 |                   0.187501 |                  1.273508 |                 1.234986 |    -1.086007 |        -1.200164 |         -1.086007 |             12.368421 |                 100.0 |              1589.621439 |               97.678815 |      0.0 |      0.0 | 2.588482 | 1.301477 | 0.210134 |  0.233783 |       0.0 |    -3.092691 |     0.755441 |       0.0 |       0.0 | 0.345372 | 0.345372 |  0.623604 | -0.105163 |  1.718605 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.001355 | 0.000016 |      0.0 | 15.848007 | 16.550944 |       0.0 | 0.848144 | -0.637367 | 85.804866 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001355 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000016 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |             87.2 |       3.8 |               101.8 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           1.348964 |          3.8 |                  0.30617 |                       0.30617 |                     0.340577 |                      0.024175 |                           0.024175 |             0.266981 |                  0.266981 |     101.533019 |   1.341718 |        -0.408164 | 8.672175 | 16.034492 |         1.032825 |                  5.132282 |                 7.362317 |                      7.362317 |             2.398433 | 100.0 |        2.398433 |                2.398433 | 0.7416 | -2.124557 |  2.912631 |               0.62199 |                  0.115512 |                    0.62199 |                  3.730222 |                 2.354503 |    -3.108232 |        -2.238991 |         -3.108232 |             24.736842 |                 100.0 |             23056.721884 |               99.440068 |      0.0 |      0.0 | 0.605427 |      0.0 | 0.151701 |  0.174464 |       0.0 |    -7.743059 |     0.740668 |       0.0 |       0.0 | 0.836097 | 0.702937 |  0.465375 |  0.487523 |  3.191888 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000723 | 0.000008 |      0.0 |  16.37648 |  16.37648 | -0.044475 | 1.023536 | -2.808398 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000723 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000008 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |             92.7 |       1.5 |               101.9 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                 0.318472 |                      0.318472 |                     0.280307 |                      0.025034 |                           0.025034 |             0.295223 |                  0.295223 |     101.604777 |   1.340358 |        -0.407436 | 8.666669 | 16.039997 |         1.032879 |                  5.157401 |                 7.373328 |                      7.373328 |             2.793204 | 100.0 |        2.793204 |                2.793204 | 0.7416 | -9.361328 |  0.441105 |              0.748282 |                  0.138967 |                   0.748282 |                  4.423241 |                 1.347085 |     -3.67496 |        -1.208118 |          -3.67496 |             62.666667 |                 100.0 |             23107.702809 |               99.504632 |      0.0 |      0.0 | -0.08797 | -0.08797 |      0.0 | -0.024251 |       0.0 |   -10.212225 |     0.732106 |       0.0 |       0.0 | 1.046504 |      0.0 | -0.064687 |  -0.39727 |  0.746162 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000386 | 0.000004 |  1.21597 | 16.400731 | 16.400731 | -0.349532 | 1.141307 | -2.507903 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000386 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000004 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |             97.8 |       1.3 |               101.7 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                  0.41125 |                       0.41125 |                     0.342527 |                      0.031402 |                           0.031402 |             0.402202 |                  0.402202 |     101.297798 |   1.320718 |        -0.406588 | 8.660262 | 16.046405 |         1.032922 |                  5.186639 |                 7.386143 |                      7.386143 |             0.777996 | 100.0 |        0.777996 |                0.777996 | 0.7416 | -8.359677 | -0.357601 |              0.210783 |                  0.039145 |                   0.210783 |                  1.247575 |                  0.40411 |    -1.036791 |        -0.364964 |         -1.036791 |             72.307692 |                 100.0 |             23107.702809 |               99.579551 |      0.0 |      0.0 | 0.030242 | 0.030242 |      0.0 | -0.093158 |       0.0 |    -7.669861 |     0.729069 |       0.0 |       0.0 | 0.787137 |      0.0 | -0.248494 |   -0.5607 | -0.444229 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000206 | 0.000002 | 2.031728 | 16.493888 | 16.493888 | -0.262904 | 1.186455 | -1.408703 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000206 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |             95.4 |       3.0 |               102.2 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 0.341723 |                      0.341723 |                     0.337331 |                      0.026649 |                           0.026649 |             0.326003 |                  0.326003 |     101.873997 |   1.339598 |        -0.405619 | 8.652959 | 16.053708 |         1.032956 |                   5.22001 |                 7.400749 |                      7.400749 |             2.504073 | 100.0 |        2.504073 |                2.504073 | 0.7416 | -4.695675 |  1.643355 |              0.687144 |                  0.127612 |                   0.687144 |                    3.6896 |                 1.768446 |    -3.002456 |        -1.640833 |         -3.002456 |             31.333333 |                 100.0 |             23107.702809 |               99.665075 |      0.0 |      0.0 | 0.014786 | 0.014786 |      0.0 |  0.023919 |       0.0 |    -7.865728 |     0.725589 |       0.0 |       0.0 | 0.811824 |      0.0 |  0.063803 | -0.053036 |    1.6516 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.00011 | 0.000001 | 2.016941 | 16.469969 | 16.469969 | -0.271149 | 1.240568 | -2.310458 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00011 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |             94.9 |       2.7 |               102.4 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 0.303761 |                      0.303761 |                     0.293214 |                      0.024006 |                           0.024006 |              0.28827 |                   0.28827 |      102.11173 |   1.350039 |         -0.40453 | 8.644768 | 16.061898 |         1.032981 |                  5.257524 |                  7.41713 |                       7.41713 |             0.788629 | 100.0 |        0.788629 |                0.788629 | 0.7416 | -7.701526 |   1.00817 |              0.218977 |                  0.040667 |                   0.218977 |                  1.325064 |                 1.219422 |    -1.106087 |        -1.178755 |         -1.106087 |             34.814815 |                 100.0 |             23107.702809 |               99.761167 |      0.0 |      0.0 | 0.205733 | 0.205733 |      0.0 |   0.00996 |       0.0 |    -9.646298 |     0.722332 |       0.0 |       0.0 | 0.994153 |      0.0 |  0.026567 | -0.136255 |  1.069068 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000059 | 0.000001 | 1.811208 | 16.460009 | 16.460009 | -0.332047 | 1.293707 | -2.577028 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000059 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |             96.0 |       1.4 |               102.1 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           0.496987 |          1.4 |                 0.349801 |                      0.349801 |                     0.308153 |                      0.027207 |                           0.027207 |             0.335809 |                  0.335809 |     101.764191 |   1.336727 |        -0.403321 | 8.635699 | 16.070967 |         1.032995 |                  5.299194 |                 7.435268 |                      7.435268 |             0.794879 | 100.0 |        0.794879 |                0.794879 | 0.7416 | -8.590092 |  0.221944 |              0.222899 |                  0.041396 |                   0.222899 |                  1.292505 |                  0.74989 |    -1.069606 |        -0.708494 |         -1.069606 |             67.142857 |                 100.0 |             23107.702809 |               99.867784 |      0.0 |      0.0 | 0.077249 | 0.077249 |      0.0 | -0.044068 |       0.0 |    -9.018226 |     0.719581 |       0.0 |       0.0 | 0.928862 |      0.0 |  -0.11755 | -0.390807 |  0.200137 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000031 |      0.0 | 1.921959 | 16.504078 | 16.504078 |  -0.31024 | 1.340593 | -2.057372 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000031 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |             96.5 |       4.2 |               101.6 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |            1.49096 |          4.2 |                 0.690794 |                      0.690794 |                      0.51298 |                      0.049663 |                           0.049663 |             0.666616 |                  0.666616 |     100.933384 |   1.284547 |        -0.401992 | 8.625761 | 16.080906 |            1.033 |                  5.345035 |                 7.455145 |                      7.455145 |             2.770177 | 100.0 |        2.770177 |                2.770177 | 0.7416 | -6.857907 | -2.327478 |              0.783955 |                  0.145592 |                   0.783955 |                  3.475519 |                 0.164918 |    -2.691564 |        -0.019326 |         -2.691564 |             22.380952 |                 100.0 |               488.756044 |                94.23896 | 0.567959 | 0.042597 | 0.111449 | 0.111449 |      0.0 | -0.384103 |  0.003891 |    -2.368172 |     0.717002 |       0.0 |       0.0 | 0.260798 | 0.260798 | -1.024575 | -1.541472 | -2.546721 | 0.962137 | 0.277712 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000017 |      0.0 | 2.088551 | 17.185623 | 17.456139 |       0.0 | 1.386332 |  1.011706 | 85.653164 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000017 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

.. _lland_v3_acker_winter_ktschnee_daily:

acre (snow surface temperature)
-------------------------------

In integration test :ref:`lland_v3_acker_winter_daily`, |lland_v3| assumes
that the snow-layer has a constant temperature over its complete depth.
For less simplified modelling of the processes within the snow layer, one
can enable an iterative adjustment of the snow surface temperature (see
method |Return_TempSSurface_V1|).  Therefore, one must set the thermal
conductivity of the top snow layer (|KTSchnee|) to a value smaller than
|numpy.inf|.  The lower the conductivity, the higher the possible differences
between the bulk and the surface temperature of the snow layer. Setting
zero values (which is not advisable) would prevent any thermal exchange,
setting |numpy.inf| values results in an infinitely fast exchange, avoiding
any temperature differences within the snow layer.

We show what happens when we use the default |KTSchnee| value used by
`LARSIM`_.  In integration test :ref:`lland_v3_acker_winter_daily`, with
:math:`KTSchnee = inf`, nearly the complete frozen water equivalent melts
on December 11th, where the average air temperature is 5 °C.  In our next
test, with :math:`KTSchnee = 0.018 MJ/m³/K/h`, the limited thermal
conductivity of the top snow layer weakens the flux of energy into the
snow layer and fewer snow melts.  The snow surface temperature rises
faster than the bulk temperature, and the higher surface temperature results
in smaller sensible (|WSensSnow|) and latent (|WLatSnow|) heat fluxes
from the atmosphere into the snow layer and to larger longwave radiation
losses (|NetLongwaveRadiationSnow|):

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(0.018)
    >>> conditions_acker_winter_ktschnee = test(
    ...     "lland_v3_acker_winter_ktschnee_daily",
    ...     axis1=(inputs.nied, fluxes.wada),
    ...     axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |    sp | globalradiation | adjustedglobalradiation |      g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes |    sbes |      evpo |  evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |  wlatsnow | wsenssnow |     wsurf |      sff |      fvg |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |      ebdn |      bowa |     qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |             70.3 |       6.9 |               101.5 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |            5.53098 |          6.9 |                 0.395811 |                      0.395811 |                          0.0 |                      0.030356 |                           0.030356 |             0.278255 |                  0.278255 |     101.221745 |   1.321192 |        -0.389384 | 8.532819 | 16.173848 |         1.028719 |                  5.761386 |                 7.641029 |                      7.641029 |             2.759858 | 100.0 |        2.759858 |                2.759858 | 0.7416 |       0.0 |   3.00672 |              2.483872 |                  2.483872 |                   2.483872 |                   4.01525 |                      0.0 |    -1.531378 |              0.0 |         -1.531378 |             30.870386 |                 100.0 |               111.309423 |               99.512695 |    0.0 |     0.0 |  1.808179 |  0.0 |  0.55374 |       0.0 |       0.0 |          nan |          0.1 |       0.0 |       0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |  -2.26512 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |             85.0 |       5.2 |               100.7 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |           4.168274 |          5.2 |                 0.489349 |                      0.489349 |                     0.387464 |                      0.036626 |                           0.036626 |             0.415947 |                  0.415947 |     100.284053 |   1.296511 |        -0.391536 | 8.548511 | 16.158156 |         1.028994 |                  5.687365 |                 7.609644 |                      7.609644 |             0.853105 | 100.0 |        0.853105 |                0.853105 | 0.7416 |   -7.5504 | -0.391921 |              0.106229 |                  0.106229 |                   0.106229 |                  1.277835 |                  2.47724 |    -1.171606 |        -2.371011 |         -1.171606 |             40.962628 |                 100.0 |               112.102145 |               99.530051 | 10.374 |  10.374 |  0.779906 |  0.0 |      0.0 | -0.140764 | -0.065045 |     -6.07759 |     0.875479 |       0.0 |       0.0 | 0.441058 |      0.0 |  -0.37548 | -2.305184 | -0.309653 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 10.514764 | 10.514764 | -0.147313 | 0.084415 | -1.131599 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |             95.9 |       2.0 |               100.5 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 0.401924 |                      0.401924 |                     0.324336 |                      0.030771 |                           0.030771 |             0.385445 |                  0.385445 |     100.114555 |   1.306656 |        -0.393572 | 8.563421 | 16.143245 |          1.02926 |                  5.617593 |                 7.579824 |                      7.579824 |             1.441454 | 100.0 |        1.441454 |                1.441454 | 0.7416 | -3.771996 |  1.092655 |              0.199691 |                  0.199691 |                   0.199691 |                  1.329454 |                 2.790301 |    -1.129764 |         -2.59061 |         -1.129764 |            106.502833 |                 100.0 |               112.102145 |               99.527226 |  0.376 |   0.376 |  0.039468 |  0.0 |      0.0 | -0.159032 | -0.004401 |    -8.367644 |     0.861466 |       0.0 |       0.0 | 0.400697 |      0.0 | -0.424211 | -1.091625 |  1.074774 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 11.049796 | 11.049796 | -0.133833 | 0.139381 | -1.482654 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |             90.6 |       4.3 |               100.8 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           3.446842 |          4.3 |                 0.460812 |                      0.460812 |                     0.331298 |                       0.03473 |                            0.03473 |             0.417496 |                  0.417496 |     100.382504 |   1.301648 |        -0.395492 | 8.577535 | 16.129131 |         1.029518 |                  5.552051 |                 7.551596 |                      7.551596 |             2.748162 | 100.0 |        2.748162 |                2.748162 | 0.7416 |  -4.94218 |   0.73294 |              0.411175 |                  0.411175 |                   0.411175 |                  3.727317 |                 4.937862 |    -3.316142 |        -4.526687 |         -3.316142 |             49.536202 |                 100.0 |               112.102145 |               99.524551 |  1.128 |   1.128 |  0.092386 |  0.0 |      0.0 | -0.369279 | -0.008959 |    -8.096584 |     0.850382 |       0.0 |       0.0 | 0.484345 |      0.0 | -0.985032 | -2.789736 |   0.75192 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 12.547075 | 12.547075 | -0.161771 | 0.186969 | -1.473994 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |             96.5 |       4.0 |                99.6 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           3.206365 |          4.0 |                 0.619748 |                      0.619748 |                     0.523186 |                      0.045128 |                           0.045128 |             0.598057 |                  0.598057 |      99.001943 |   1.266438 |        -0.397294 | 8.590839 | 16.115828 |         1.029767 |                  5.490726 |                 7.524989 |                      7.524989 |             0.823609 | 100.0 |        0.823609 |                0.823609 | 0.7416 | -4.913313 | -0.666543 |              0.128375 |                  0.128375 |                   0.128375 |                  1.140917 |                 1.700543 |    -1.012542 |        -1.572169 |         -1.012542 |             53.251417 |                 100.0 |               112.102145 |               99.522031 |  6.016 |  2.7072 |  0.135099 |  0.0 |      0.0 | -0.304329 |  0.003902 |     -2.10322 |     0.844132 |       0.0 |       0.0 |  0.49672 |  0.49672 | -0.811782 | -1.418894 | -0.658507 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.009222 | 18.867404 |       0.0 | 0.215598 | -0.065851 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |             99.0 |       2.4 |                99.8 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 0.563475 |                      0.563475 |                     0.493464 |                      0.041489 |                           0.041489 |             0.557841 |                  0.557841 |      99.242159 |   1.275245 |        -0.398979 | 8.603319 | 16.103347 |         1.030007 |                  5.433602 |                 7.500028 |                      7.500028 |             1.394624 | 100.0 |        1.394624 |                1.394624 | 0.7416 | -0.219502 |  0.594934 |              0.262946 |                  0.262946 |                   0.262946 |                  1.217715 |                 2.201607 |    -0.954768 |        -1.938661 |         -0.954768 |             88.752361 |                 100.0 |               112.102145 |               99.519666 |    0.0 |     0.0 |  0.000956 |  0.0 |      0.0 | -0.186363 |       0.0 |    -2.888059 |     0.811457 |       0.0 |       0.0 | 0.185939 | 0.185939 | -0.497115 | -0.784508 |  0.657038 | 0.996975 | 0.298188 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.355129 | 19.053767 |       0.0 |  0.39099 |  0.080815 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |             99.2 |       1.9 |               100.4 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 0.493024 |                      0.493024 |                     0.429691 |                      0.036869 |                           0.036869 |             0.489079 |                  0.489079 |      99.910921 |   1.291808 |        -0.400545 | 8.614964 | 16.091703 |         1.030238 |                  5.380663 |                 7.476739 |                      7.476739 |             0.807099 | 100.0 |        0.807099 |                0.807099 | 0.7416 |       0.0 |  1.062327 |              0.172638 |                  0.172638 |                   0.172638 |                  1.193664 |                 2.428099 |    -1.021026 |        -2.255461 |         -1.021026 |            112.108246 |                 100.0 |               112.102145 |                99.51746 |    0.0 |     0.0 | -0.015761 |  0.0 |      0.0 | -0.150213 |       0.0 |    -4.724078 |       0.7861 |       0.0 |       0.0 | 0.279046 | 0.279046 | -0.400685 | -0.699247 |  1.155529 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.765312 |  19.20398 |       0.0 | 0.566382 | -0.239912 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |             99.1 |       1.8 |               100.6 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 0.493024 |                      0.493024 |                     0.424215 |                      0.036869 |                           0.036869 |             0.488586 |                  0.488586 |     100.111414 |   1.294388 |        -0.401992 | 8.625761 | 16.080906 |         1.030461 |                  5.331896 |                 7.455145 |                      7.455145 |             0.799784 | 100.0 |        0.799784 |                0.799784 | 0.7416 | -0.799708 |  0.901798 |              0.187269 |                  0.187269 |                   0.187269 |                  1.194208 |                 2.367544 |    -1.006938 |        -2.180275 |         -1.006938 |            118.336481 |                 100.0 |               112.102145 |               99.515415 |    0.0 |     0.0 | -0.012523 |  0.0 |      0.0 | -0.158109 |       0.0 |     -4.89276 |      0.76585 |       0.0 |       0.0 | 0.344018 | 0.344018 | -0.421747 | -0.741828 |    1.0167 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 17.250194 | 19.362088 |       0.0 | 0.741774 |  -0.40011 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |             96.9 |       2.7 |               100.9 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 0.523311 |                      0.523311 |                     0.381225 |                      0.038864 |                           0.038864 |             0.507088 |                  0.507088 |     100.392912 |   1.294334 |        -0.403321 | 8.635699 | 16.070967 |         1.030674 |                  5.287288 |                 7.435268 |                      7.435268 |             2.886262 | 100.0 |        2.886262 |                2.886262 | 0.7416 | -1.333701 |   1.08274 |              0.710165 |                  0.710165 |                   0.710165 |                  4.005891 |                 4.970876 |    -3.295726 |        -4.260711 |         -3.295726 |             78.890988 |                 100.0 |               112.102145 |               99.513532 |  0.658 |   0.658 | -0.234616 |  0.0 |      0.0 | -0.391967 | -0.002888 |    -6.288533 |      0.75395 |       0.0 |       0.0 | 0.474106 | 0.474106 | -1.045552 | -1.976956 |  1.238204 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 18.742201 | 20.412055 |       0.0 | 0.864457 | -0.741251 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |             95.0 |       3.0 |               101.8 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 0.485698 |                      0.485698 |                     0.412172 |                      0.036384 |                           0.036384 |             0.461413 |                  0.461413 |     101.338587 |   1.310959 |         -0.40453 | 8.644768 | 16.061898 |         1.030879 |                  5.246826 |                  7.41713 |                       7.41713 |             1.347076 | 100.0 |        1.347076 |                1.347076 | 0.7416 | -2.470836 |  0.568272 |               0.31656 |                   0.31656 |                    0.31656 |                  1.286731 |                 2.587081 |    -0.970171 |         -2.27052 |         -0.970171 |             71.001889 |                 100.0 |               112.102145 |               99.511814 |  3.102 |   3.102 |  0.137342 |  0.0 |      0.0 | -0.164151 | -0.020098 |    -5.270657 |     0.765002 |       0.0 |       0.0 | 0.562244 | 0.562244 | -0.437864 | -1.096694 |  0.735963 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 22.562863 | 23.678206 |       0.0 | 0.749982 | -0.567922 |  71.44626 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |             90.6 |       7.4 |               100.4 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           5.931775 |          7.4 |                 0.879017 |                      0.879017 |                      0.61078 |                      0.061412 |                           0.061412 |              0.79639 |                   0.79639 |       99.60361 |   1.253213 |        -0.405619 | 8.652959 | 16.053708 |         1.031074 |                  5.210498 |                 7.400749 |                      7.400749 |             1.373029 | 100.0 |        1.373029 |                1.373029 | 0.7416 | -1.893074 |  -0.98137 |              0.346666 |                  0.346666 |                   0.346666 |                   1.16859 |                -0.065197 |    -0.821924 |         0.411862 |         -0.821924 |              28.78455 |                 100.0 |               112.102145 |               99.510262 |  8.836 |     0.0 |  1.033243 | 0.06 | 0.338985 | -1.215856 |  0.183349 |          0.0 |     0.747518 | 23.713776 | 22.562863 |      0.0 |      0.0 | -3.243236 | -5.063324 | -8.718422 | 0.956772 | 0.274624 | 33.730062 | 11.067544 |  0.0 |  0.0 | 0.0 |  0.0 | 11.067544 | 2.849301 | 0.032978 |  0.0 |       0.0 |       0.0 |       0.0 |  0.93825 |  1.155047 | 93.769793 | 11.067544 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849301 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032978 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |             78.7 |       5.7 |               101.2 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |            4.56907 |          5.7 |                 0.619748 |                      0.619748 |                      0.50499 |                      0.045128 |                           0.045128 |             0.487741 |                  0.487741 |     100.712259 |    1.28736 |        -0.406588 | 8.660262 | 16.046405 |          1.03126 |                  5.178293 |                 7.386143 |                      7.386143 |             1.539952 | 100.0 |        1.539952 |                1.539952 | 0.7416 |       0.0 |  0.607505 |              0.409117 |                  0.409117 |                   0.409117 |                  1.733999 |                 3.008114 |    -1.324882 |        -2.598997 |         -1.324882 |             37.369415 |                 100.0 |                81.653408 |               98.490732 |  0.128 |  0.0576 |  1.429595 | 0.06 | 0.519628 |  0.082368 |  0.000045 |    -2.578858 |     0.734331 |       0.0 |       0.0 | 0.000786 | 0.000786 |  0.244207 | -2.235392 |  0.607812 | 0.951754 | 0.271751 |  0.045632 |  0.015062 |  0.0 |  0.0 | 0.0 |  0.0 |  0.015062 | 3.836775 | 0.044407 |  0.0 |       0.0 |       0.0 |       0.0 | 1.109403 |  1.289142 | 93.280735 |  0.015062 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836775 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044407 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |             90.5 |       2.2 |               102.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 0.471336 |                      0.471336 |                     0.397748 |                      0.035431 |                           0.035431 |             0.426559 |                  0.426559 |     101.573441 |   1.315658 |        -0.407436 | 8.666669 | 16.039997 |         1.031437 |                  5.150203 |                 7.373328 |                      7.373328 |              0.77253 | 100.0 |         0.77253 |                 0.77253 | 0.7416 |       0.0 |  1.354389 |              0.192007 |                  0.192007 |                   0.192007 |                  1.254261 |                 2.684327 |    -1.062254 |         -2.49232 |         -1.062254 |             96.820757 |                 100.0 |                82.295785 |                98.52021 |  2.666 |   2.666 |  0.193737 |  0.0 |      0.0 | -0.079193 | -0.019502 |    -5.736347 |     0.751457 |       0.0 |       0.0 | 0.043585 |      0.0 | -0.211243 | -0.931632 |  1.349445 | 0.974687 | 0.285005 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 2.050493 | 0.023733 | 0.06 |  2.745193 |  2.745193 | -0.014557 | 0.892385 |  0.676353 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050493 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023733 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |             88.1 |       3.1 |               102.5 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |           2.484933 |          3.1 |                 0.457351 |                      0.457351 |                     0.368122 |                      0.034499 |                           0.034499 |             0.402926 |                  0.402926 |     102.097074 |   1.324197 |        -0.408164 | 8.672175 | 16.034492 |         1.031605 |                  5.126218 |                 7.362317 |                      7.362317 |             1.699321 | 100.0 |        1.699321 |                1.699321 | 0.7416 |       0.0 |   1.58696 |              0.435319 |                  0.435319 |                   0.435319 |                  2.107297 |                 3.808798 |    -1.671978 |         -3.37348 |         -1.671978 |             68.711505 |                 100.0 |                82.295785 |               98.520648 |    0.0 |     0.0 |  0.280006 |  0.0 |      0.0 | -0.118568 |       0.0 |    -6.741743 |     0.743828 |       0.0 |       0.0 | 0.052586 |      0.0 | -0.316273 |  -1.46724 |  1.589966 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 1.094166 | 0.012664 | 0.06 |  2.863761 |  2.863761 | -0.017564 |  0.98328 | -0.169006 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094166 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012664 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |             92.8 |       2.8 |               102.3 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 0.380889 |                      0.380889 |                     0.288325 |                       0.02934 |                            0.02934 |             0.353465 |                  0.353465 |     101.946535 |    1.33374 |         -0.40877 | 8.676772 | 16.029895 |         1.031763 |                  5.106331 |                 7.353123 |                      7.353123 |             2.318249 | 100.0 |        2.318249 |                2.318249 | 0.7416 | -0.563355 |  2.184142 |              0.609046 |                  0.609046 |                   0.609046 |                  3.397964 |                 5.076808 |    -2.788918 |        -4.467762 |         -2.788918 |             76.073452 |                 100.0 |                82.295785 |               98.521014 |    0.0 |     0.0 | -0.031797 |  0.0 |      0.0 | -0.207622 |       0.0 |    -9.857997 |     0.737282 |       0.0 |       0.0 | 0.085596 |      0.0 | -0.553821 | -1.718774 |  2.195167 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.583859 | 0.006758 | 0.06 |  3.071382 |  3.071382 | -0.028589 | 1.068532 | -1.611548 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583859 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.006758 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |             98.1 |       3.7 |               100.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           2.965888 |          3.7 |                 0.471336 |                      0.471336 |                     0.393632 |                      0.035431 |                           0.035431 |             0.462381 |                  0.462381 |      99.537619 |   1.289645 |        -0.409256 | 8.680456 | 16.026211 |         1.031912 |                  5.090537 |                 7.345755 |                      7.345755 |             0.763581 | 100.0 |        0.763581 |                0.763581 | 0.7416 | -5.371828 |  0.099152 |              0.153408 |                  0.153408 |                   0.153408 |                  1.211777 |                   2.3744 |     -1.05837 |        -2.220993 |          -1.05837 |             57.569099 |                 100.0 |                82.295785 |               98.521307 |  6.298 |   6.298 |   0.03395 |  0.0 |      0.0 | -0.264364 |  -0.04607 |    -5.871957 |     0.799094 |       0.0 |       0.0 | 0.326157 |      0.0 | -0.705177 | -1.382387 |  0.133429 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.311554 | 0.003606 | 0.06 |  9.633746 |  9.633746 | -0.108937 | 0.471314 |   -0.9691 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311554 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003606 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |             96.6 |       0.9 |                99.1 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.721432 |          0.9 |                 0.352531 |                      0.352531 |                     0.279984 |                      0.027395 |                           0.027395 |             0.340545 |                  0.340545 |      98.759455 |   1.296889 |         -0.40962 | 8.683222 | 16.023444 |         1.032051 |                  5.078832 |                 7.340222 |                      7.340222 |             0.761825 | 100.0 |        0.761825 |                0.761825 | 0.7416 | -3.230334 |  1.637841 |              0.159896 |                  0.159896 |                   0.159896 |                  1.287913 |                 2.905551 |    -1.128017 |        -2.745655 |         -1.128017 |            236.672963 |                 100.0 |                82.295785 |               98.521527 |    0.0 |     0.0 | -0.001988 |  0.0 |      0.0 | -0.108899 |       0.0 |   -10.226692 |     0.790114 |       0.0 |       0.0 | 0.385193 |      0.0 | -0.290482 | -0.797614 |  1.657559 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.166248 | 0.001924 | 0.06 |  9.742645 |  9.742645 | -0.128655 |  0.53573 | -1.865341 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166248 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001924 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |             97.7 |       1.3 |                99.7 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 0.395811 |                      0.395811 |                     0.303297 |                      0.030356 |                           0.030356 |             0.386708 |                  0.386708 |      99.313292 |   1.297204 |        -0.409863 | 8.685068 | 16.021598 |         1.032182 |                  5.071211 |                  7.33653 |                       7.33653 |             0.760682 | 100.0 |        0.760682 |                0.760682 | 0.7416 | -6.217804 |  0.693349 |              0.165025 |                  0.165025 |                   0.165025 |                  1.257034 |                 2.432501 |     -1.09201 |        -2.267476 |          -1.09201 |            163.850513 |                 100.0 |                82.295785 |               98.521673 |  0.094 |   0.094 | -0.003569 |  0.0 |      0.0 | -0.174381 | -0.001139 |    -9.219359 |     0.783057 |       0.0 |       0.0 | 0.465048 |      0.0 | -0.465153 | -1.083443 |  0.718881 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.088712 | 0.001027 | 0.06 | 10.011026 | 10.011026 | -0.155326 | 0.590452 |  -1.81709 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.088712 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001027 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |             95.8 |       1.9 |                99.7 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                  0.33643 |                       0.33643 |                     0.262456 |                      0.026283 |                           0.026283 |               0.3223 |                    0.3223 |        99.3777 |   1.307793 |        -0.409985 | 8.685992 | 16.020675 |         1.032302 |                  5.067673 |                 7.334683 |                      7.334683 |             0.760151 | 100.0 |        0.760151 |                0.760151 | 0.7416 | -6.056967 |  1.162854 |              0.169935 |                  0.169935 |                   0.169935 |                  1.301361 |                 2.956416 |    -1.131426 |        -2.786481 |         -1.131426 |            112.108246 |                 100.0 |                82.295785 |               98.521747 |    0.0 |     0.0 |  0.031907 |  0.0 |      0.0 | -0.151363 |       0.0 |   -11.034367 |     0.776446 |       0.0 |       0.0 | 0.519952 |      0.0 | -0.403752 | -1.201537 |  1.181192 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.047338 | 0.000548 | 0.06 | 10.162388 | 10.162388 | -0.173664 | 0.645375 | -2.238344 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047338 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000548 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |             97.5 |       2.4 |                99.7 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 0.398857 |                      0.398857 |                     0.309875 |                      0.030563 |                           0.030563 |             0.388886 |                  0.388886 |      99.311114 |   1.296708 |        -0.409985 | 8.685992 | 16.020675 |         1.032413 |                  5.068219 |                 7.334683 |                      7.334683 |             1.336154 | 100.0 |        1.336154 |                1.336154 | 0.7416 | -7.461147 |  0.350194 |              0.304781 |                  0.304781 |                   0.304781 |                  1.393316 |                 2.690071 |    -1.088535 |         -2.38529 |         -1.088535 |             88.752361 |                 100.0 |                82.295785 |               98.521747 |  0.188 |   0.188 |  0.019864 |  0.0 |      0.0 | -0.228727 |  -0.00224 |    -8.947567 |     0.771897 |       0.0 |       0.0 | 0.526991 |      0.0 | -0.610119 | -1.424866 |  0.350305 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.02526 | 0.000292 | 0.06 | 10.579116 | 10.579116 | -0.176015 | 0.685419 | -1.846938 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02526 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000292 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |             95.0 |       2.3 |               100.4 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 0.401924 |                      0.401924 |                     0.303538 |                      0.030771 |                           0.030771 |             0.381828 |                  0.381828 |     100.018172 |   1.305372 |        -0.409863 | 8.685068 | 16.021598 |         1.032515 |                  5.072848 |                  7.33653 |                       7.33653 |             1.544792 | 100.0 |        1.544792 |                1.544792 | 0.7416 |  -6.15646 |  0.723866 |               0.35908 |                   0.35908 |                    0.35908 |                  1.820398 |                 3.207565 |    -1.461319 |        -2.848486 |         -1.461319 |             92.611159 |                 100.0 |                82.295785 |               98.521673 |  0.188 |   0.188 |   0.03385 |  0.0 |      0.0 | -0.220917 |   -0.0022 |    -9.209318 |     0.767555 |       0.0 |       0.0 | 0.508871 |      0.0 | -0.589284 | -1.543588 |  0.715613 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.013479 | 0.000156 | 0.06 | 10.988032 | 10.988032 | -0.169963 |  0.72551 | -1.829204 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000156 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |             93.8 |       4.6 |               100.7 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |            3.68732 |          4.6 |                 0.427219 |                      0.427219 |                     0.352891 |                      0.032479 |                           0.032479 |             0.400731 |                  0.400731 |     100.299269 |   1.305282 |         -0.40962 | 8.683222 | 16.023444 |         1.032607 |                  5.081565 |                 7.340222 |                      7.340222 |             0.762235 | 100.0 |        0.762235 |                0.762235 | 0.7416 | -6.097346 |  0.284988 |              0.167996 |                  0.167996 |                   0.167996 |                  1.259735 |                 2.716442 |    -1.091739 |        -2.548446 |         -1.091739 |              46.30558 |                 100.0 |                82.295785 |               98.521527 |  2.256 |   2.256 |  0.237721 |  0.0 |      0.0 | -0.215444 | -0.022632 |     -7.28687 |     0.779601 |       0.0 |       0.0 | 0.550873 |      0.0 | -0.574686 | -1.697376 |  0.276384 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.007193 | 0.000083 | 0.06 | 13.459476 | 13.459476 | -0.183992 | 0.618694 | -1.372592 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007193 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000083 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |             94.9 |       8.6 |               100.4 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           6.893685 |          8.6 |                 0.563475 |                      0.563475 |                     0.498371 |                      0.041489 |                           0.041489 |             0.534738 |                  0.534738 |      99.865262 |    1.28304 |        -0.409256 | 8.680456 | 16.026211 |         1.032689 |                  5.094372 |                 7.345755 |                      7.345755 |             0.764156 | 100.0 |        0.764156 |                0.764156 | 0.7416 | -4.575305 | -0.411111 |              0.141857 |                  0.141857 |                   0.141857 |                  1.179808 |                 2.282858 |    -1.037951 |        -2.141001 |         -1.037951 |             24.768101 |                 100.0 |                82.295785 |               98.521307 |   6.58 |  5.0995 |  0.461334 |  0.0 |      0.0 | -0.270137 | -0.018542 |    -2.755662 |     0.814361 |       0.0 |       0.0 | 0.509438 | 0.509438 | -0.720577 | -1.863916 | -0.443493 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.003838 | 0.000044 | 0.06 | 19.325461 | 20.309613 |       0.0 | 0.373379 |  -0.21988 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000044 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |             85.4 |       7.6 |               100.8 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           6.092093 |          7.6 |                 0.493024 |                      0.493024 |                     0.428703 |                      0.036869 |                           0.036869 |             0.421042 |                  0.421042 |     100.378958 |   1.297296 |         -0.40877 | 8.676772 | 16.029895 |         1.032762 |                  5.111276 |                 7.353123 |                      7.353123 |             0.766691 | 100.0 |        0.766691 |                0.766691 | 0.7416 | -0.732934 |  0.888297 |              0.161369 |                  0.161369 |                   0.161369 |                  1.273508 |                 2.903665 |    -1.112139 |        -2.742297 |         -1.112139 |             28.027061 |                 100.0 |                82.295785 |               98.521014 |  0.188 |   0.188 |  1.132218 |  0.0 |      0.0 |  0.051303 | -0.001139 |     -4.75437 |     0.789526 |       0.0 |       0.0 | 0.328908 | 0.328908 |  0.136848 | -1.882132 |  0.997013 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.002048 | 0.000024 | 0.06 | 19.792706 | 20.446311 |       0.0 | 0.540148 | -0.366577 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000024 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |             87.2 |       3.8 |               101.8 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                  0.30617 |                       0.30617 |                     0.233412 |                      0.024175 |                           0.024175 |             0.266981 |                  0.266981 |     101.533019 |   1.341718 |        -0.408164 | 8.672175 | 16.034492 |         1.032825 |                  5.132282 |                 7.362317 |                      7.362317 |             2.398433 | 100.0 |        2.398433 |                2.398433 | 0.7416 | -1.221924 |  2.508159 |               0.55496 |                   0.55496 |                    0.55496 |                  3.730222 |                 5.691454 |    -3.175262 |        -5.136494 |         -3.175262 |             56.054123 |                 100.0 |                82.295785 |               98.520648 |    0.0 |     0.0 |  0.145715 |  0.0 |      0.0 | -0.131539 |       0.0 |   -12.484807 |     0.768616 |       0.0 |       0.0 | 0.800223 | 0.653605 | -0.350874 | -2.010188 |  2.775433 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.001093 | 0.000013 | 0.06 |  20.57785 |  20.57785 |  -0.04897 |  0.71554 | -2.133136 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000013 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |             92.7 |       1.5 |               101.9 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 0.318472 |                      0.318472 |                     0.197901 |                      0.025034 |                           0.025034 |             0.295223 |                  0.295223 |     101.604777 |   1.340358 |        -0.407436 | 8.666669 | 16.039997 |         1.032879 |                  5.157401 |                 7.373328 |                      7.373328 |             2.793204 | 100.0 |        2.793204 |                2.793204 | 0.7416 | -7.110453 |  1.528737 |              0.655326 |                  0.655326 |                   0.655326 |                  4.423241 |                 5.150105 |    -3.767915 |        -4.494778 |         -3.767915 |            142.003778 |                 100.0 |                82.295785 |                98.52021 |  1.128 |   1.128 | -0.167474 |  0.0 |      0.0 | -0.217696 | -0.020275 |   -14.494219 |     0.765385 |       0.0 |       0.0 | 1.366309 |      0.0 | -0.580693 | -1.998246 |  1.915839 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000583 | 0.000007 | 0.06 | 21.923545 | 21.923545 | -0.456347 |  0.74626 | -2.920273 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000007 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |             97.8 |       1.3 |               101.7 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                  0.41125 |                       0.41125 |                     0.296322 |                      0.031402 |                           0.031402 |             0.402202 |                  0.402202 |     101.297798 |   1.320718 |        -0.406588 | 8.660262 | 16.046405 |         1.032922 |                  5.186639 |                 7.386143 |                      7.386143 |             0.777996 | 100.0 |        0.777996 |                0.777996 | 0.7416 | -9.734244 | -0.048282 |              0.181176 |                  0.181176 |                   0.181176 |                  1.247575 |                 2.051458 |    -1.066399 |        -1.870283 |         -1.066399 |            163.850513 |                 100.0 |                82.295785 |               98.519701 |  0.846 |   0.846 | -0.001963 |  0.0 |      0.0 | -0.221357 | -0.009371 |    -9.513339 |     0.767125 |       0.0 |       0.0 | 1.373666 |      0.0 | -0.590459 |  -1.33502 | -0.055196 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000311 | 0.000004 | 0.06 | 22.990902 | 22.990902 | -0.458804 | 0.729578 | -2.130391 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000004 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |             95.4 |       3.0 |               102.2 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 0.341723 |                      0.341723 |                     0.235849 |                      0.026649 |                           0.026649 |             0.326003 |                  0.326003 |     101.873997 |   1.339598 |        -0.405619 | 8.652959 | 16.053708 |         1.032956 |                   5.22001 |                 7.400749 |                      7.400749 |             2.504073 | 100.0 |        2.504073 |                2.504073 | 0.7416 | -7.101304 |  1.239908 |              0.595257 |                  0.595257 |                   0.595257 |                    3.6896 |                 4.987128 |    -3.094344 |        -4.391872 |         -3.094344 |             71.001889 |                 100.0 |                82.295785 |                98.51912 |    0.0 |     0.0 | -0.120498 |  0.0 |      0.0 | -0.300538 |       0.0 |   -12.357122 |     0.762285 |       0.0 |       0.0 | 1.365727 |      0.0 |  -0.80167 | -2.352945 |  1.237257 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000166 | 0.000002 | 0.06 |  23.29144 |  23.29144 | -0.456153 | 0.776804 | -2.628699 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |             94.9 |       2.7 |               102.4 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 0.303761 |                      0.303761 |                     0.232742 |                      0.024006 |                           0.024006 |              0.28827 |                   0.28827 |      102.11173 |   1.350039 |         -0.40453 | 8.644768 | 16.061898 |         1.032981 |                  5.257524 |                  7.41713 |                       7.41713 |             0.788629 | 100.0 |        0.788629 |                0.788629 | 0.7416 | -8.762331 |  0.857693 |              0.191133 |                  0.191133 |                   0.191133 |                  1.325064 |                 3.128101 |    -1.133931 |        -2.936968 |         -1.133931 |             78.890988 |                 100.0 |                82.295785 |               98.518469 |    0.0 |     0.0 |  0.073401 |  0.0 |      0.0 | -0.172927 |       0.0 |   -12.520123 |     0.757639 |       0.0 |       0.0 |  1.51821 |      0.0 | -0.461273 | -1.567073 |  0.908622 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000089 | 0.000001 | 0.06 | 23.464367 | 23.464367 | -0.507082 | 0.824618 | -2.744792 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |             96.0 |       1.4 |               102.1 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           1.122228 |          1.4 |                 0.349801 |                      0.349801 |                     0.253668 |                      0.027207 |                           0.027207 |             0.335809 |                  0.335809 |     101.764191 |   1.336727 |        -0.403321 | 8.635699 | 16.070967 |         1.032995 |                  5.299194 |                 7.435268 |                      7.435268 |             0.794879 | 100.0 |        0.794879 |                0.794879 | 0.7416 | -9.149307 |  0.546932 |              0.194815 |                  0.194815 |                   0.194815 |                  1.292505 |                 2.540901 |     -1.09769 |        -2.346087 |          -1.09769 |            152.146905 |                 100.0 |                82.295785 |               98.517748 |  0.188 |   0.188 |  0.019443 |  0.0 |      0.0 | -0.177732 | -0.002908 |   -11.457537 |     0.754913 |       0.0 |       0.0 | 1.510287 |      0.0 | -0.474092 | -1.330616 |  0.541379 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000047 | 0.000001 | 0.06 |   23.8301 |   23.8301 | -0.504436 | 0.853891 | -2.550124 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |             96.5 |       4.2 |               101.6 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |           3.366683 |          4.2 |                 0.690794 |                      0.690794 |                     0.494668 |                      0.049663 |                           0.049663 |             0.666616 |                  0.666616 |     100.933384 |   1.284547 |        -0.401992 | 8.625761 | 16.080906 |            1.033 |                  5.345035 |                 7.455145 |                      7.455145 |             2.770177 | 100.0 |        2.770177 |                2.770177 | 0.7416 | -8.500415 |  -1.23571 |              0.689458 |                  0.689458 |                   0.689458 |                  3.475519 |                 4.119365 |    -2.786061 |        -3.429907 |         -2.786061 |             50.715635 |                 100.0 |                82.295785 |               98.516958 |  0.846 | 0.06345 | -0.125078 |  0.0 |      0.0 | -0.724068 |  0.005795 |    -2.855473 |     0.751114 |       0.0 |       0.0 | 0.974534 |  0.78255 | -1.931414 |  -2.90735 | -1.408857 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000025 |      0.0 | 0.06 | 25.400167 | 25.400167 | -0.064123 | 0.896287 | -0.572814 | 93.280735 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

>>> ktschnee(inf)

hourly simulation
_________________

Now we change the simulation time step size from one day to one hour.
Consequently, we need to change the values of all time-dependent control
parameters by the factor 1/24 (or, in single cases, 24).  Also, we need
to |FixedParameter.restore| all time-dependent fixed parameter values:

>>> pub.timegrids = "1997-08-03", "1997-08-06", "1h"
>>> for parameter in control:
...     if parameter.TIME == True:
...         parameter.value = parameter.value/24
...     if parameter.TIME == False:
...         parameter.value = parameter.value*24
>>> for parameter in fixed:
...     parameter.restore()
>>> parameterstep("1h")
>>> test = IntegrationTest(land)

.. _lland_v3_acker_summer_hourly:

acre (summer)
-------------

The following input values agree with the ones of the respective sub-period
of the daily integration test :ref:`lland_v3_acker_summer_daily`:

>>> inputs.nied.series = 0.0
>>> inputs.teml.series = (
...     16.9, 16.6, 16.4, 16.3, 16.0, 15.9, 16.0, 16.6, 17.4, 19.0, 20.3,
...     21.4, 21.3, 21.8, 22.9, 22.7, 22.5, 21.9, 21.4, 20.7, 19.4, 17.8,
...     17.0, 16.4, 15.9, 15.4, 15.2, 14.9, 14.3, 14.1, 14.3, 15.4, 16.8,
...     18.2, 19.5, 20.3, 21.0, 21.8, 21.9, 21.8, 22.2, 21.8, 20.9, 19.7,
...     17.9, 15.8, 14.8, 14.1, 13.4, 12.8, 12.3, 12.0, 11.7, 11.6, 12.6,
...     14.8, 17.1, 19.4, 20.6, 21.4, 22.5, 23.1, 23.4, 23.8, 24.0, 24.0,
...     23.7, 22.8, 21.3, 19.3, 18.3, 17.4)
>>> inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.5, 0.7, 0.8, 0.5,
...     0.4, 0.5, 0.5, 0.3, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.1, 0.9, 1.0, 1.0, 0.9, 0.8, 0.9, 0.8, 0.9, 0.9, 0.9, 1.0,
...     1.0, 1.0, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6, 0.9,
...     1.0, 0.9, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 0.5, 0.0,
...     0.0, 0.0)
>>> inputs.windspeed.series = (
...     0.8, 0.8, 0.8, 0.8, 0.8, 0.6, 0.9, 0.9, 0.9, 1.3, 1.5, 1.2, 1.3, 1.5,
...     1.9, 1.9, 2.3, 2.4, 2.5, 2.5, 2.2, 1.7, 1.7, 2.3, 2.3, 2.2, 2.3, 2.0,
...     2.3, 2.5, 2.4, 2.5, 2.5, 2.9, 3.1, 3.3, 3.3, 3.2, 2.5, 2.9, 3.6, 4.2,
...     4.2, 3.6, 3.2, 2.2, 2.2, 2.1, 1.8, 1.8, 1.6, 1.2, 1.9, 1.3, 1.6, 1.9,
...     3.2, 4.0, 4.6, 4.1, 4.1, 4.4, 4.6, 4.5, 4.7, 4.5, 4.1, 3.8, 2.5, 1.9,
...     2.2, 2.7)
>>> inputs.relativehumidity.series = (
...     95.1, 94.9, 95.9, 96.7, 97.2, 97.5, 97.7, 97.4, 96.8, 86.1, 76.8,
...     71.8, 67.5, 66.1, 63.4, 62.4, 61.1, 62.1, 67.0, 74.5, 81.2, 86.9,
...     90.1, 90.9, 88.0, 89.2, 88.8, 89.7, 93.0, 93.6, 94.6, 93.3, 86.9,
...     78.9, 75.8, 73.4, 68.3, 61.3, 60.6, 58.8, 57.2, 57.2, 59.9, 66.1,
...     71.7, 80.6, 85.9, 89.3, 91.4, 93.1, 94.6, 95.6, 96.2, 95.7, 95.1,
...     96.3, 89.9, 79.2, 73.4, 67.3, 62.2, 59.6, 55.4, 52.1, 49.5, 48.1,
...     49.4, 57.0, 65.9, 73.0, 76.7, 81.8)
>>> inputs.atmosphericpressure.series = (
...     101.5, 101.5, 101.5, 101.5, 101.5, 101.5, 101.5, 101.5, 101.6, 101.6,
...     101.6, 101.6, 101.6, 101.6, 101.6, 101.6, 101.6, 101.6, 101.6, 101.6,
...     101.6, 101.6, 101.7, 101.7, 101.7, 101.6, 101.6, 101.6, 101.6, 101.7,
...     101.7, 101.7, 101.7, 101.7, 101.7, 101.8, 101.8, 101.8, 101.8, 101.7,
...     101.7, 101.7, 101.7, 101.7, 101.7, 101.8, 101.8, 101.8, 101.8, 101.8,
...     101.8, 101.8, 101.8, 101.8, 101.8, 101.9, 101.9, 101.9, 101.9, 101.9,
...     101.9, 101.9, 101.9, 101.9, 101.8, 101.8, 101.8, 101.8, 101.8, 101.8,
...     101.8, 101.8)
>>> inlet.sequences.sim.series = 0.0


In this and the following "summer" examples, we use the initial conditions
of August 3rd, calculated by the corresponding daily integration tests.
However, this is possible for the state sequences, but not for the (now
higher resolved) log sequences.  Hence, we need to define them manually:

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2,
...        0.1, 0.2, 0.1, 0.2, 0.2, 0.3, 0.0, 0.0,
...        0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.2, 0.5,
...       0.8, 1.1, 1.2, 1.4, 1.9, 1.6, 0.9, 0.8,
...       0.6, 0.4, 0.2, 0.1, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedteml,
...      [13.2, 13.2, 13.1, 12.6, 12.7, 13.0, 13.5, 14.8,
...       16.2, 17.7, 18.8, 19.4, 20.4, 21.0, 21.5, 21.2,
...       20.4, 20.7, 20.2, 19.7, 19.0, 18.0, 17.5, 17.1]),
...     (logs.loggedrelativehumidity,
...      [95.1, 94.5, 94.8, 96.4, 96.6, 97.1, 97.1, 96.7,
...       92.2, 88.5, 81.1, 76.5, 75.1, 70.8, 68.9, 69.2,
...       75.0, 74.0, 77.4, 81.4, 85.3, 90.1, 92.3, 93.8]),
...     (logs.loggedwindspeed2m,
...      [0.8, 1.0, 1.2, 1.3, 0.9, 1.1, 1.3, 1.3,
...       1.9, 2.2, 1.8, 2.3, 2.4, 2.5, 2.4, 2.5,
...       2.1, 2.2, 1.7, 1.7, 1.3, 1.3, 0.7, 0.8]))

This integration test deals with a dry situation.  Hence, the soil water
content (|BoWa|) shows a pronounced decline.  This decline is sharpest around
noon, when evapotranspiration out of the soil (|EvB|) reaches its maximum:

.. integration-test::

    >>> test("lland_v3_acker_summer_hourly",
    ...      axis1=(fluxes.evb, fluxes.qah),
    ...      axis2=states.bowa,
    ...      use_conditions=conditions_acker_summer)
    |                date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |        sp | globalradiation | adjustedglobalradiation |         g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |      ebdn |      bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |              0.0 |             95.1 |       0.8 |               101.5 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |  0.0 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                 1.928423 |                      1.975309 |                          0.0 |                      0.122362 |                           0.124959 |              1.83393 |                  1.696461 |       99.66607 |    1.21073 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.208683 | -0.036532 |                   0.0 |                       0.0 |                   7.982856 |                  0.901196 |                      0.0 |     -0.03755 |              0.0 |           7.08166 |            176.542034 |                 100.0 |               100.629116 |               87.719298 |  0.0 |  0.0 | -0.002965 | -0.002965 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.256857 | 0.071349 | 0.002965 |  0.0 |  0.0 |   0.0 |  nan | 31.308721 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.256857 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.071349 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |              0.0 |             94.9 |       0.8 |               101.5 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |  0.0 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                 1.892018 |                      1.970628 |                          0.0 |                      0.120339 |                             0.1247 |             1.795525 |                  1.694576 |      99.704475 |   1.212158 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.295735 | -0.028172 |                   0.0 |                       0.0 |                   7.982856 |                  0.901844 |                      0.0 |    -0.037577 |              0.0 |          7.081012 |            176.542034 |                 100.0 |               100.629116 |               87.719298 |  0.0 |  0.0 | -0.002685 | -0.002685 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.250222 | 0.069506 |  0.00565 |  0.0 |  0.0 |   0.0 |  nan | 31.326476 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.250222 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.069506 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |              0.0 |             95.9 |       0.8 |               101.5 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |  0.0 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 1.868084 |                       1.96233 |                          0.0 |                      0.119006 |                           0.124241 |             1.791493 |                  1.692183 |      99.708507 |   1.213014 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 |  15.35492 | -0.022574 |                   0.0 |                       0.0 |                   7.982856 |                  0.902355 |                      0.0 |    -0.037598 |              0.0 |            7.0805 |            176.542034 |                 100.0 |               100.629116 |               87.719298 |  0.0 |  0.0 | -0.004107 | -0.004107 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.243758 | 0.067711 | 0.009757 |  0.0 |  0.0 |   0.0 |  nan | 31.338633 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.243758 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.067711 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |              0.0 |             96.7 |       0.8 |               101.5 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |  0.0 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                 1.856216 |                      1.948396 |                          0.0 |                      0.118345 |                           0.123469 |             1.794961 |                  1.689422 |      99.705039 |   1.213417 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.395444 | -0.019538 |                   0.0 |                       0.0 |                   7.982856 |                  0.902364 |                      0.0 |    -0.037598 |              0.0 |          7.080492 |            176.542034 |                 100.0 |               100.629116 |               87.719298 |  0.0 |  0.0 | -0.005232 | -0.005232 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.237462 | 0.065962 | 0.014989 |  0.0 |  0.0 |   0.0 |  nan | 31.347755 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.237462 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.065962 |
    | 1997-08-03 04:00:00 |  0.0 | 16.0 |              0.0 |             97.2 |       0.8 |               101.5 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 |  0.0 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                 1.821009 |                      1.929443 |                          0.0 |                      0.116379 |                           0.122418 |              1.77002 |                   1.68569 |       99.72998 |    1.21479 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                 0.429734 |                     15.566134 |            10.503757 |  0.086656 |        0.007799 |                0.006997 | -0.010417 | 15.425849 | -0.012402 |              0.005318 |                  0.005318 |                   7.982856 |                  0.902341 |                      0.0 |     -0.03228 |              0.0 |          7.080514 |            176.542034 |                 100.0 |               100.629116 |               92.771119 |  0.0 |  0.0 | -0.004819 | -0.004819 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.231328 | 0.064258 | 0.019808 |  0.0 |  0.0 |   0.0 |  nan |  31.34974 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.231328 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.064258 |
    | 1997-08-03 05:00:00 |  0.0 | 15.9 |              0.0 |             97.5 |       0.6 |               101.5 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 |  0.0 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                 1.809403 |                      1.907618 |                          0.0 |                       0.11573 |                           0.121206 |             1.764168 |                  1.682599 |      99.735832 |   1.215237 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  0.968392 |        0.087159 |                0.078957 | -0.010417 | 15.432466 | -0.010099 |              0.060007 |                  0.060007 |                   7.982856 |                  0.901482 |                      0.0 |     0.022445 |              0.0 |          7.081373 |            235.389378 |                 100.0 |               100.629116 |              100.447767 |  0.0 |  0.0 |  0.005073 |  0.005073 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.225353 | 0.062598 | 0.014735 |  0.0 |  0.0 |   0.0 |  nan | 31.349422 | 79.623186 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.225353 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.062598 |
    | 1997-08-03 06:00:00 |  0.0 | 16.0 |              0.0 |             97.7 |       0.9 |               101.5 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 |  0.0 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                 1.821009 |                      1.884011 |                          0.0 |                      0.116379 |                           0.119893 |             1.779125 |                   1.68038 |      99.720875 |   1.214748 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |    2.4897 |        0.224084 |                0.206122 | -0.010417 | 15.431406 | -0.012282 |              0.156653 |                  0.156653 |                   7.982856 |                   0.89977 |                      0.0 |     0.119163 |              0.0 |          7.083086 |            156.926252 |                 100.0 |               100.629116 |              100.447767 |  0.0 |  0.0 |  0.026292 |  0.014735 | 0.008821 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.219531 | 0.060981 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.351287 | 79.614365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.219531 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.060981 |
    | 1997-08-03 07:00:00 |  0.0 | 16.6 |              0.0 |             97.4 |       0.9 |               101.5 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 |  0.0 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                 1.892018 |                      1.865111 |                          0.0 |                      0.120339 |                           0.118841 |             1.842826 |                  1.680931 |      99.657174 |   1.211943 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  4.679733 |        0.421196 |                0.393598 | -0.010417 | 15.437623 | -0.025107 |              0.299135 |                  0.299135 |                   7.982856 |                  0.896826 |                      0.0 |     0.261767 |              0.0 |          7.086029 |            156.926252 |                 100.0 |               100.641253 |               100.45639 |  0.0 |  0.0 |  0.057456 |       0.0 | 0.044017 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.213861 | 0.059406 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.365978 | 79.570349 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.213861 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.059406 |
    | 1997-08-03 08:00:00 |  0.0 | 17.4 |              0.0 |             96.8 |       0.9 |               101.6 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 |  0.0 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                 1.990459 |                      1.846377 |                          0.0 |                      0.125796 |                           0.117796 |             1.926764 |                  1.685281 |      99.673236 |   1.209425 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  7.219274 |        0.649765 |                0.615417 | -0.010417 | 15.486592 |  -0.04133 |              0.467717 |                  0.467717 |                   7.982856 |                  0.891371 |                      0.0 |     0.430576 |              0.0 |          7.091484 |            156.926252 |                 100.0 |               100.701823 |              100.499412 |  0.0 |  0.0 |  0.096099 |       0.0 | 0.073977 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.208337 | 0.057871 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  31.39689 | 79.496372 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.208337 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057871 |
    | 1997-08-03 09:00:00 |  0.0 | 19.0 |              0.2 |             86.1 |       1.3 |               101.6 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 |  0.0 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                 2.200854 |                      1.834142 |                          0.0 |                      0.137341 |                           0.117113 |             1.894936 |                  1.687258 |      99.705064 |   1.202945 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             10.73504 |  9.442017 |         1.18975 |                1.122345 | -0.010417 | 15.589635 | -0.073664 |              0.852983 |                  0.852983 |                    8.15863 |                  0.922604 |                      0.0 |     0.814541 |              0.0 |          7.236026 |            108.641252 |                 100.0 |               100.803646 |              100.571701 |  0.0 |  0.0 |  0.222763 |       0.0 | 0.161018 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.202955 | 0.056376 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.460138 | 79.335354 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.202955 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.056376 |
    | 1997-08-03 10:00:00 |  0.0 | 20.3 |              0.5 |             76.8 |       1.5 |               101.6 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 |  0.0 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                  2.38585 |                      1.830729 |                          0.0 |                      0.147368 |                           0.116922 |             1.832333 |                  1.688695 |      99.767667 |   1.197896 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.966322 | 10.709164 |         1.92774 |                  1.8057 | -0.010417 | 15.800459 |  -0.09719 |              1.372332 |                  1.372332 |                   8.334404 |                  0.955299 |                      0.0 |     1.332528 |              0.0 |          7.379106 |             94.155751 |                 100.0 |               101.025386 |              100.728981 |  0.0 |  0.0 |  0.389923 |       0.0 | 0.275567 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.197713 |  0.05492 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.546911 | 79.059786 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.197713 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.05492 |
    | 1997-08-03 11:00:00 |  0.0 | 21.4 |              0.7 |             71.8 |       1.2 |               101.6 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 |  0.0 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                 2.552842 |                      1.835607 |                          0.0 |                      0.156328 |                           0.117195 |             1.832941 |                   1.69067 |      99.767059 |    1.19342 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            11.544527 | 10.773868 |        2.327265 |                2.214067 | -0.010417 | 16.089704 | -0.114702 |              1.682691 |                  1.682691 |                   8.773841 |                  1.039981 |                      0.0 |     1.639359 |              0.0 |          7.733859 |            117.694689 |                 100.0 |               101.405231 |              100.997942 |  0.0 |  0.0 |   0.46628 |       0.0 | 0.348858 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.192605 | 0.053502 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.651197 | 78.710929 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.192605 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.053502 |
    | 1997-08-03 12:00:00 |  0.0 | 21.3 |              0.8 |             67.5 |       1.3 |               101.6 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 |  0.0 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 2.537251 |                      1.844905 |                          0.0 |                      0.155495 |                           0.117714 |             1.712644 |                  1.692316 |      99.887356 |   1.194363 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.238374 | 10.364714 |        2.425457 |                 2.25556 | -0.010417 | 16.437323 | -0.105034 |              1.714225 |                  1.714225 |                   9.301164 |                  1.142661 |                      0.0 |     1.666614 |              0.0 |          8.158503 |            108.641252 |                 100.0 |               101.886758 |              101.338062 |  0.0 |  0.0 |  0.491066 |       0.0 | 0.360852 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.18763 |  0.05212 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.745814 | 78.350077 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.18763 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.05212 |
    | 1997-08-03 13:00:00 |  0.0 | 21.8 |              0.5 |             66.1 |       1.5 |               101.6 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 |  0.0 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                 2.616045 |                      1.859671 |                          0.0 |                      0.159698 |                           0.118538 |             1.729206 |                  1.694238 |      99.870794 |   1.192265 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.700938 | 10.872489 |         1.95714 |                 1.78608 | -0.010417 | 16.752713 | -0.109021 |              1.357421 |                  1.357421 |                   9.652713 |                  1.211985 |                      0.0 |     1.306922 |              0.0 |          8.440727 |             94.155751 |                 100.0 |               102.385622 |              101.689443 |  0.0 |  0.0 |  0.424802 |       0.0 |  0.30342 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.182784 | 0.050773 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.844419 | 78.046657 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.182784 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.050773 |
    | 1997-08-03 14:00:00 |  0.0 | 22.9 |              0.4 |             63.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 |  0.0 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                 2.796942 |                       1.88551 |                          0.0 |                      0.169283 |                           0.119977 |             1.773261 |                  1.698059 |      99.826739 |   1.187639 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             12.93222 | 10.461143 |        1.694785 |                1.510276 | -0.010417 | 17.081396 | -0.125682 |              1.147809 |                  1.147809 |                   9.828487 |                   1.24753 |                      0.0 |     1.095829 |              0.0 |          8.580957 |             74.333488 |                 100.0 |               102.805714 |              101.984567 |  0.0 |  0.0 |  0.415458 |       0.0 | 0.282559 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.178062 | 0.049462 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.959684 | 77.764098 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.178062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.049462 |
    | 1997-08-03 15:00:00 |  0.0 | 22.7 |              0.5 |             62.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 |  0.0 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                 2.763263 |                      1.918249 |                          0.0 |                      0.167505 |                           0.121797 |             1.724276 |                  1.703725 |      99.875724 |    1.18866 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.394785 |  8.865382 |        1.595844 |                1.396394 | -0.010417 | 17.465613 | -0.113063 |               1.06126 |                   1.06126 |                  10.180036 |                  1.316893 |                      0.0 |     1.006389 |              0.0 |          8.863143 |             74.333488 |                 100.0 |               103.197445 |              102.259131 |  0.0 |  0.0 |  0.393508 |       0.0 | 0.266848 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.173463 | 0.048184 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.06233 |  77.49725 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.173463 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.048184 |
    | 1997-08-03 16:00:00 |  0.0 | 22.5 |              0.5 |             61.1 |       2.3 |               101.6 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 |  0.0 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                 2.729939 |                      1.957676 |                          0.0 |                      0.165743 |                           0.123983 |             1.667993 |                  1.709704 |      99.932007 |   1.189715 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.741708 |  6.474205 |        1.165412 |                1.002589 | -0.010417 | 17.807767 | -0.101352 |              0.761967 |                  0.761967 |                  10.443698 |                  1.370145 |                      0.0 |     0.704878 |              0.0 |          9.073553 |             61.405925 |                 100.0 |               103.567861 |              102.518193 |  0.0 |  0.0 |  0.342513 |       0.0 |   0.2192 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.168982 | 0.046939 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.153266 | 77.278049 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.168982 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.046939 |
    | 1997-08-03 17:00:00 |  0.0 | 21.9 |              0.3 |             62.1 |       2.4 |               101.6 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 |  0.0 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                 2.632058 |                      2.001493 |                          0.0 |                       0.16055 |                           0.126406 |             1.634508 |                  1.718782 |      99.965492 |   1.192283 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.088631 |  3.977658 |         0.57281 |                0.493699 | -0.010417 | 18.110885 | -0.081845 |              0.375211 |                  0.375211 |                   10.70736 |                  1.421011 |                      0.0 |     0.316002 |              0.0 |          9.286348 |             58.847345 |                 100.0 |                103.87248 |               102.73083 |  0.0 |  0.0 |  0.236583 |       0.0 | 0.148202 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.164617 | 0.045727 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.224694 | 77.129847 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.164617 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.045727 |
    | 1997-08-03 18:00:00 |  0.0 | 21.4 |              0.1 |             67.0 |       2.5 |               101.6 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 |  0.0 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                 2.552842 |                      2.046164 |                          0.0 |                      0.156328 |                           0.128868 |             1.710404 |                  1.731481 |      99.889596 |   1.193968 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  1.972911 |        0.213084 |                0.183891 | -0.010417 | 18.348979 | -0.065902 |              0.139757 |                  0.139757 |                  10.795247 |                  1.433696 |                      0.0 |     0.080019 |              0.0 |           9.36155 |             56.493451 |                 100.0 |               104.078611 |              102.874509 |  0.0 |  0.0 |  0.156446 |       0.0 | 0.096042 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.160365 | 0.044546 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.280179 | 77.033805 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.160365 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044546 |
    | 1997-08-03 19:00:00 |  0.0 | 20.7 |              0.0 |             74.5 |       2.5 |               101.6 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 |  0.0 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                 2.445437 |                      2.089517 |                          0.0 |                      0.150575 |                           0.131251 |              1.82185 |                  1.748925 |       99.77815 |   1.196313 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  0.634046 |        0.057067 |                0.049078 | -0.010417 | 18.533931 | -0.046787 |              0.037299 |                  0.037299 |                  10.795247 |                  1.423756 |                      0.0 |    -0.022024 |              0.0 |          9.371491 |             56.493451 |                 100.0 |                104.21227 |              102.967584 |  0.0 |  0.0 |   0.09619 |       0.0 |   0.0585 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.156222 | 0.043395 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.31655 | 76.975305 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.156222 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043395 |
    | 1997-08-03 20:00:00 |  0.0 | 19.4 |              0.0 |             81.2 |       2.2 |               101.6 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 |  0.0 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                 2.256393 |                      2.126995 |                          0.0 |                      0.140363 |                           0.133306 |             1.832191 |                  1.766824 |      99.767809 |   1.201582 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                   0.1364 |                     15.566134 |            14.204272 |  0.008648 |        0.000778 |                0.000669 | -0.010417 | 18.655165 | -0.016088 |              0.000509 |                  0.000509 |                  10.795247 |                  1.412078 |                      0.0 |    -0.058328 |              0.0 |          9.383169 |             64.197103 |                 100.0 |               104.293713 |               89.533534 |  0.0 |  0.0 |   0.04938 |       0.0 | 0.032417 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.152187 | 0.042274 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.322221 | 76.942888 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.152187 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042274 |
    | 1997-08-03 21:00:00 |  0.0 | 17.8 |              0.0 |             86.9 |       1.7 |               101.6 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 |  0.0 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                 2.041337 |                      2.153241 |                          0.0 |                      0.128603 |                           0.134742 |             1.773922 |                  1.781538 |      99.826078 |   1.208454 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.674071 |   0.01888 |                   0.0 |                       0.0 |                  10.795247 |                  1.401564 |                      0.0 |    -0.058399 |              0.0 |          9.393683 |             83.078604 |                 100.0 |               104.338854 |               87.719298 |  0.0 |  0.0 |  0.019258 |       0.0 | 0.013442 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.148256 | 0.041182 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.292925 | 76.929445 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.148256 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.041182 |
    | 1997-08-03 22:00:00 |  0.0 | 17.0 |              0.0 |             90.1 |       1.7 |               101.7 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 |  0.0 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                 1.940693 |                      2.174668 |                          0.0 |                      0.123042 |                           0.135912 |             1.748564 |                  1.795279 |      99.951436 |   1.213101 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.576416 |  0.034051 |                   0.0 |                       0.0 |                  10.795247 |                  1.391158 |                      0.0 |    -0.057965 |              0.0 |          9.404089 |             83.078604 |                 100.0 |               104.357574 |               87.719298 |  0.0 |  0.0 |  0.010298 |       0.0 | 0.007136 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.144426 | 0.040118 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.248458 |  76.92231 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.144426 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.040118 |
    | 1997-08-03 23:00:00 |  0.0 | 16.4 |              0.0 |             90.9 |       2.3 |               101.7 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 |  0.0 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 1.868084 |                      2.192855 |                          0.0 |                      0.119006 |                           0.136905 |             1.698088 |                  1.806456 |     100.001912 |   1.215845 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.428192 |  0.043809 |                   0.0 |                       0.0 |                  10.795247 |                   1.38284 |                      0.0 |    -0.057618 |              0.0 |          9.412407 |             61.405925 |                 100.0 |               104.367512 |               87.719298 |  0.0 |  0.0 |  0.014661 |       0.0 | 0.009245 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.140696 | 0.039082 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.194232 | 76.913065 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039082 |
    | 1997-08-04 00:00:00 |  0.0 | 15.9 |              0.0 |             88.0 |       2.3 |               101.7 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 |  0.0 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                 1.809403 |                      2.187158 |                          0.0 |                       0.11573 |                           0.136594 |             1.592275 |                  1.795292 |     100.107725 |    1.21843 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.24744 |  0.050705 |                   0.0 |                       0.0 |                  10.739718 |                   1.39654 |                      0.0 |    -0.058189 |              0.0 |          9.343179 |             61.405925 |                 100.0 |               104.380388 |               87.719298 |  0.0 |  0.0 |  0.023989 |       0.0 | 0.015041 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.137061 | 0.038073 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.133111 | 76.898024 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.137061 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.038073 |
    | 1997-08-04 01:00:00 |  0.0 | 15.4 |              0.0 |             89.2 |       2.2 |               101.6 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 |  0.0 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                 1.752342 |                      2.180337 |                          0.0 |                      0.112529 |                           0.136222 |             1.563089 |                  1.784515 |     100.036911 |   1.219467 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.043702 |  0.057104 |                   0.0 |                       0.0 |                  10.739718 |                  1.406633 |                      0.0 |     -0.05861 |              0.0 |          9.333085 |             64.197103 |                 100.0 |               104.401338 |               87.719298 |  0.0 |  0.0 |  0.018182 |       0.0 |   0.0115 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.133521 | 0.037089 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.06559 | 76.886524 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.133521 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037089 |
    | 1997-08-04 02:00:00 |  0.0 | 15.2 |              0.0 |             88.8 |       2.3 |               101.6 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 |  0.0 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                 1.729962 |                      2.173535 |                          0.0 |                      0.111271 |                            0.13585 |             1.536206 |                  1.772518 |     100.063794 |   1.220436 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.818633 |  0.056562 |                   0.0 |                       0.0 |                  10.739718 |                  1.418069 |                      0.0 |    -0.059086 |              0.0 |           9.32165 |             61.405925 |                 100.0 |               104.417357 |               87.719298 |  0.0 |  0.0 |   0.02055 |       0.0 |  0.01278 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.130072 | 0.036131 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.998611 | 76.873744 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.130072 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.036131 |
    | 1997-08-04 03:00:00 |  0.0 | 14.9 |              0.0 |             89.7 |       2.0 |               101.6 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 |  0.0 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                 1.696861 |                      2.165623 |                          0.0 |                      0.109405 |                           0.135418 |             1.522084 |                  1.759749 |     100.077916 |   1.221771 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.595369 |   0.05822 |                   0.0 |                       0.0 |                  10.739718 |                   1.43017 |                      0.0 |     -0.05959 |              0.0 |          9.309548 |             70.616813 |                 100.0 |               104.435159 |               87.719298 |  0.0 |  0.0 |   0.01363 |       0.0 | 0.008835 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.126712 | 0.035198 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.929974 | 76.864909 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.126712 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.035198 |
    | 1997-08-04 04:00:00 |  0.0 | 14.3 |              0.0 |             93.0 |       2.3 |               101.6 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 |  0.0 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                 1.632319 |                       2.15605 |                          0.0 |                      0.105752 |                           0.134895 |             1.518057 |                  1.748197 |     100.081943 |    1.22434 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.398044 |                     15.502754 |            14.131208 |   0.07485 |        0.006692 |                0.005726 | -0.010417 |  17.36658 |  0.066238 |              0.004352 |                  0.004352 |                  10.739718 |                  1.440702 |                      0.0 |    -0.055678 |              0.0 |          9.299016 |             61.405925 |                 100.0 |               104.447467 |               93.267161 |  0.0 |  0.0 |  0.007255 |       0.0 | 0.004359 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.123439 | 0.034289 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.853319 |  76.86055 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.123439 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034289 |
    | 1997-08-04 05:00:00 |  0.0 | 14.1 |              0.0 |             93.6 |       2.5 |               101.7 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 |  0.0 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                 1.611289 |                      2.145953 |                          0.0 |                      0.104558 |                           0.134343 |             1.508166 |                  1.736523 |     100.191834 |   1.226451 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.131208 |  0.938714 |        0.083928 |                0.071823 | -0.010417 | 17.111064 |  0.065039 |              0.054586 |                  0.054586 |                  10.739718 |                  1.451305 |                      0.0 |    -0.005885 |              0.0 |          9.288414 |             56.493451 |                 100.0 |               104.453539 |              103.135415 |  0.0 |  0.0 |  0.018162 |       0.0 | 0.010113 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.120251 | 0.033403 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.777864 | 76.850437 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.120251 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.033403 |
    | 1997-08-04 06:00:00 |  0.0 | 14.3 |              0.1 |             94.6 |       2.4 |               101.7 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 |  0.0 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                 1.632319 |                      2.136456 |                          0.0 |                      0.105752 |                           0.133824 |             1.544174 |                  1.726078 |     100.155826 |   1.225432 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.246552 |  2.455474 |        0.263445 |                0.226749 | -0.010417 | 16.859546 |  0.055286 |              0.172329 |                  0.172329 |                   10.82738 |                  1.478175 |                      0.0 |     0.110738 |              0.0 |          9.349205 |             58.847345 |                 100.0 |               104.467628 |              103.145209 |  0.0 |  0.0 |   0.04093 |       0.0 | 0.023217 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.117144 |  0.03254 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.712161 | 76.827221 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.117144 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.03254 |
    | 1997-08-04 07:00:00 |  0.0 | 15.4 |              0.9 |             93.3 |       2.5 |               101.7 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 |  0.0 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                 1.752342 |                      2.129774 |                          0.0 |                      0.112529 |                           0.133458 |             1.634935 |                  1.717041 |     100.065065 |   1.220347 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            15.284647 |  4.650125 |        1.164115 |                1.028795 | -0.010417 | 16.640536 |  0.026796 |              0.781884 |                  0.781884 |                  11.616331 |                   1.64464 |                      0.0 |     0.713358 |              0.0 |          9.971692 |             56.493451 |                 100.0 |               104.499975 |              103.167691 |  0.0 |  0.0 |   0.18972 |       0.0 | 0.107491 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.114118 |   0.0317 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.674949 | 76.719729 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.114118 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0317 |
    | 1997-08-04 08:00:00 |  0.0 | 16.8 |              1.0 |             86.9 |       2.5 |               101.7 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 |  0.0 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                  1.91622 |                       2.12644 |                          0.0 |                      0.121684 |                           0.133275 |             1.665196 |                  1.705582 |     100.034804 |   1.214317 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            16.438085 |  7.209034 |        1.933622 |                1.710811 | -0.010417 | 16.516495 | -0.006124 |              1.300216 |                  1.300216 |                  12.492945 |                   1.83472 |                      0.0 |      1.22377 |              0.0 |         10.658225 |             56.493451 |                 100.0 |               104.649787 |               103.27176 |  0.0 |  0.0 |  0.343734 |       0.0 | 0.198364 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.111171 | 0.030881 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.670656 | 76.521365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.111171 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030881 |
    | 1997-08-04 09:00:00 |  0.0 | 18.2 |              1.0 |             78.9 |       2.9 |               101.7 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 |  0.0 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                 2.093348 |                      2.122001 |                          0.0 |                      0.131461 |                           0.133032 |             1.651651 |                  1.695656 |     100.048349 |   1.208543 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.360836 |  9.458797 |        2.537058 |                2.210424 | -0.010417 | 16.502185 | -0.036673 |              1.679922 |                  1.679922 |                  13.194235 |                  1.989515 |                      0.0 |     1.597026 |              0.0 |          11.20472 |             48.701251 |                 100.0 |               104.926453 |              103.463719 |  0.0 |  0.0 |  0.494183 |       0.0 | 0.274461 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.108299 | 0.030083 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.696912 | 76.246905 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.108299 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030083 |
    | 1997-08-04 10:00:00 |  0.0 | 19.5 |              0.9 |             75.8 |       3.1 |               101.7 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 |  0.0 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                 2.270468 |                      2.117571 |                          0.0 |                      0.141127 |                            0.13279 |             1.721014 |                  1.691233 |      99.978986 |   1.202862 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.822211 | 10.747438 |         2.69052 |                2.317703 | -0.010417 | 16.589706 | -0.062862 |              1.761455 |                  1.761455 |                   13.54488 |                   2.06621 |                      0.0 |     1.675363 |              0.0 |          11.47867 |             45.559235 |                 100.0 |               105.309691 |              103.729123 |  0.0 |  0.0 |  0.550449 |       0.0 | 0.303114 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.105502 | 0.029306 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.749357 | 75.943791 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.105502 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029306 |
    | 1997-08-04 11:00:00 |  0.0 | 20.3 |              0.8 |             73.4 |       3.3 |               101.8 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 |  0.0 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                  2.38585 |                      2.111492 |                          0.0 |                      0.147368 |                           0.132457 |             1.751214 |                  1.687786 |     100.048786 |   1.200635 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.937555 | 10.819272 |        2.515039 |                2.160947 | -0.010417 | 16.764525 | -0.076366 |               1.64232 |                   1.64232 |                  13.632542 |                  2.087634 |                      0.0 |     1.555335 |              0.0 |         11.544908 |             42.798069 |                 100.0 |               105.733535 |              104.021981 |  0.0 |  0.0 |  0.548384 |       0.0 | 0.297669 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.102776 | 0.028549 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.815307 | 75.646122 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.102776 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028549 |
    | 1997-08-04 12:00:00 |  0.0 | 21.0 |              0.9 |             68.3 |       3.3 |               101.8 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 |  0.0 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                 2.490975 |                      2.109837 |                          0.0 |                      0.153018 |                           0.132366 |             1.701336 |                  1.687166 |     100.098664 |   1.198001 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.052899 | 10.408076 |        2.605564 |                2.233851 | -0.010417 | 16.984357 | -0.086738 |              1.697727 |                  1.697727 |                  13.720203 |                  2.106001 |                      0.0 |     1.609977 |              0.0 |         11.614202 |             42.798069 |                 100.0 |               106.150385 |              104.309322 |  0.0 |  0.0 |  0.597559 |       0.0 | 0.327471 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.100122 | 0.027812 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.891628 | 75.318651 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.100122 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027812 |
    | 1997-08-04 13:00:00 |  0.0 | 21.8 |              0.8 |             61.3 |       3.2 |               101.8 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 |  0.0 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                 2.616045 |                      2.109837 |                          0.0 |                      0.159698 |                           0.132366 |             1.603636 |                  1.682947 |     100.196364 |   1.195188 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             18.39893 | 10.917404 |         2.53785 |                2.157989 | -0.010417 | 17.238761 | -0.098523 |              1.640072 |                  1.640072 |                  13.983187 |                  2.166369 |                      0.0 |     1.549806 |              0.0 |         11.816818 |             44.135508 |                 100.0 |               106.609685 |              104.625144 |  0.0 |  0.0 |  0.621801 |       0.0 | 0.348686 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.097535 | 0.027093 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.979734 | 74.969965 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.097535 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027093 |
    | 1997-08-04 14:00:00 |  0.0 | 21.9 |              0.9 |             60.6 |       2.5 |               101.8 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 |  0.0 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                 2.632058 |                      2.104328 |                          0.0 |                       0.16055 |                           0.132064 |             1.595027 |                  1.676097 |     100.204973 |   1.194821 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.975649 | 10.494383 |         2.62717 |                2.208784 | -0.010417 | 17.532448 | -0.094339 |              1.678676 |                  1.678676 |                  14.421494 |                  2.265756 |                      0.0 |      1.58427 |              0.0 |         12.155738 |             56.493451 |                 100.0 |               107.099579 |              104.961105 |  0.0 |  0.0 |  0.589183 |       0.0 | 0.361258 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.095016 | 0.026393 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.063657 | 74.608707 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.095016 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.026393 |
    | 1997-08-04 15:00:00 |  0.0 | 21.8 |              0.9 |             58.8 |       2.9 |               101.7 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 |  0.0 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                 2.616045 |                      2.099381 |                          0.0 |                      0.159698 |                           0.131792 |             1.538235 |                  1.669008 |     100.161765 |   1.194298 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            19.437025 |  8.874215 |        2.221576 |                1.861585 | -0.010417 | 17.812189 | -0.086137 |              1.414805 |                  1.414805 |                  14.772139 |                  2.348437 |                      0.0 |     1.316953 |              0.0 |         12.423701 |             48.701251 |                 100.0 |               107.608064 |              105.308836 |  0.0 |  0.0 |  0.551424 |       0.0 | 0.320065 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.092562 | 0.025712 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.139377 | 74.288642 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.092562 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025712 |
    | 1997-08-04 16:00:00 |  0.0 | 22.2 |              0.9 |             57.2 |       3.6 |               101.7 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 |  0.0 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                 2.680609 |                      2.097734 |                          0.0 |                      0.163129 |                           0.131702 |             1.533308 |                   1.66429 |     100.166692 |   1.192703 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |              19.8984 |   6.45697 |        1.616442 |                1.360212 | -0.010417 |  18.06459 | -0.089325 |              1.033761 |                  1.033761 |                  15.122784 |                  2.428674 |                      0.0 |     0.932566 |              0.0 |          12.69411 |             39.231563 |                 100.0 |               108.059369 |              105.616633 |  0.0 |  0.0 |  0.505761 |       0.0 | 0.270996 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.090171 | 0.025047 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.218285 | 74.017646 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.090171 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025047 |
    | 1997-08-04 17:00:00 |  0.0 | 21.8 |              1.0 |             57.2 |       4.2 |               101.7 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 |  0.0 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                 2.616045 |                      2.097185 |                          0.0 |                      0.159698 |                           0.131672 |             1.496378 |                  1.659573 |     100.203622 |   1.194485 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            20.705807 |  3.945159 |        1.058179 |                0.907937 | -0.010417 | 18.327617 | -0.075003 |              0.690032 |                  0.690032 |                  15.736413 |                  2.564427 |                      0.0 |     0.583181 |              0.0 |         13.171986 |             33.627054 |                 100.0 |               108.442082 |               105.87704 |  0.0 |  0.0 |  0.444603 |       0.0 | 0.220974 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.087841 |   0.0244 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.282872 | 73.796672 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.087841 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0244 |
    | 1997-08-04 18:00:00 |  0.0 | 20.9 |              1.0 |             59.9 |       4.2 |               101.7 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 |  0.0 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 2.475714 |                      2.094444 |                          0.0 |                        0.1522 |                           0.131522 |             1.482953 |                  1.651207 |     100.217047 |   1.198201 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            21.743901 |  1.939642 |        0.520255 |                0.462876 | -0.010417 | 18.542906 | -0.050913 |              0.351786 |                  0.351786 |                  16.525365 |                  2.744116 |                      0.0 |     0.237448 |              0.0 |         13.781249 |             33.627054 |                 100.0 |               108.754562 |              106.089245 |  0.0 |  0.0 |   0.32478 |       0.0 | 0.158718 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.085572 |  0.02377 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.323368 | 73.637953 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.085572 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.02377 |
    | 1997-08-04 19:00:00 |  0.0 | 19.7 |              1.0 |             66.1 |       3.6 |               101.7 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 |  0.0 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                 2.298847 |                       2.08897 |                          0.0 |                      0.142666 |                           0.131221 |             1.519538 |                   1.63958 |     100.180462 |   1.202947 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             22.89734 |   0.60531 |        0.162358 |                0.151461 | -0.010417 | 18.677895 | -0.022077 |              0.115111 |                  0.115111 |                  17.401978 |                  2.950104 |                      0.0 |     -0.00781 |              0.0 |         14.451875 |             39.231563 |                 100.0 |               108.979235 |              106.241591 |  0.0 |  0.0 |  0.183091 |       0.0 | 0.093928 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.083362 | 0.023156 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.335029 | 73.544026 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083362 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023156 |
    | 1997-08-04 20:00:00 |  0.0 | 17.9 |              0.3 |             71.7 |       3.2 |               101.7 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 |  0.0 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                 2.054233 |                      2.080783 |                          0.0 |                      0.129312 |                           0.130772 |             1.472885 |                  1.624918 |     100.227115 |   1.210597 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.104711 |                     15.502754 |            23.243371 |  0.005136 |        0.003091 |                0.002926 | -0.010417 | 18.716764 |  0.017642 |              0.002224 |                  0.002224 |                  17.664962 |                  3.034376 |                      0.0 |    -0.124208 |              0.0 |         14.630586 |             44.135508 |                 100.0 |               109.112286 |               89.357093 |  0.0 |  0.0 |  0.100631 |       0.0 | 0.057096 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.081209 | 0.022558 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  32.30697 |  73.48693 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.081209 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022558 |
    | 1997-08-04 21:00:00 |  0.0 | 15.8 |              0.0 |             80.6 |       2.2 |               101.8 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 |  0.0 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                 1.797863 |                       2.06991 |                          0.0 |                      0.115083 |                           0.130174 |             1.449077 |                  1.610994 |     100.350923 |    1.22071 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.623235 |  0.060982 |                   0.0 |                       0.0 |                  17.664962 |                  3.060678 |                      0.0 |    -0.127528 |              0.0 |         14.604285 |             64.197103 |                 100.0 |               109.193197 |               87.719298 |  0.0 |  0.0 |  0.029981 |       0.0 |  0.01905 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.079111 | 0.021975 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.235572 |  73.46788 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.079111 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021975 |
    | 1997-08-04 22:00:00 |  0.0 | 14.8 |              0.0 |             85.9 |       2.2 |               101.8 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 |  0.0 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                 1.685952 |                      2.058007 |                          0.0 |                      0.108789 |                            0.12952 |             1.448232 |                  1.598128 |     100.351768 |   1.224953 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.38524 |  0.077441 |                   0.0 |                       0.0 |                  17.664962 |                  3.084324 |                      0.0 |    -0.128514 |              0.0 |         14.580638 |             64.197103 |                 100.0 |               109.220198 |               87.719298 |  0.0 |  0.0 |  0.012064 |       0.0 |  0.00758 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.077068 | 0.021408 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.147714 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.077068 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021408 |
    | 1997-08-04 23:00:00 |  0.0 | 14.1 |              0.0 |             89.3 |       2.1 |               101.8 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 |  0.0 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                 1.611289 |                      2.045628 |                          0.0 |                      0.104558 |                           0.128839 |             1.438881 |                  1.587151 |     100.361119 |   1.227981 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.09238 |  0.086235 |                   0.0 |                       0.0 |                  17.664962 |                  3.103639 |                      0.0 |    -0.129318 |              0.0 |         14.561323 |             67.254108 |                 100.0 |               109.230942 |               87.719298 |  0.0 |  0.0 | -0.000111 | -0.000111 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.075077 | 0.020855 | 0.000111 |  0.0 |  0.0 |   0.0 |  nan | 32.051062 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020855 |
    | 1997-08-05 00:00:00 |  0.0 | 13.4 |              0.0 |             91.4 |       1.8 |               101.8 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 |  0.0 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                 1.539538 |                      2.032245 |                          0.0 |                      0.100468 |                           0.128102 |             1.407137 |                  1.579647 |     100.392863 |   1.231127 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.770207 |  0.094396 |                   0.0 |                       0.0 |                   17.58988 |                  3.124611 |                      0.0 |    -0.130192 |              0.0 |         14.465269 |             78.463126 |                 100.0 |               109.230942 |               87.719298 |  0.0 |  0.0 | -0.009107 | -0.009107 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.073138 | 0.020316 | 0.009218 |  0.0 |  0.0 |   0.0 |  nan | 31.946249 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.073138 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020316 |
    | 1997-08-05 01:00:00 |  0.0 | 12.8 |              0.0 |             93.1 |       1.8 |               101.8 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 |  0.0 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                 1.480281 |                      2.018409 |                          0.0 |                      0.097071 |                           0.127339 |             1.378142 |                  1.572172 |     100.421858 |   1.233843 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 |  17.42083 |   0.09981 |                   0.0 |                       0.0 |                   17.58988 |                  3.135507 |                      0.0 |    -0.130646 |              0.0 |         14.454373 |             78.463126 |                 100.0 |               109.230942 |               87.719298 |  0.0 |  0.0 | -0.013348 | -0.013348 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.071248 | 0.019791 | 0.022566 |  0.0 |  0.0 |   0.0 |  nan | 31.836022 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.071248 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019791 |
    | 1997-08-05 02:00:00 |  0.0 | 12.3 |              0.0 |             94.6 |       1.6 |               101.8 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 |  0.0 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                 1.432438 |                      2.003073 |                          0.0 |                      0.094315 |                           0.126493 |             1.355086 |                  1.565068 |     100.444914 |   1.236111 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.053408 |  0.102674 |                   0.0 |                       0.0 |                   17.58988 |                  3.144898 |                      0.0 |    -0.131037 |              0.0 |         14.444982 |             88.271017 |                 100.0 |               109.230942 |               87.719298 |  0.0 |  0.0 | -0.017799 | -0.017799 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.069408 |  0.01928 | 0.040365 |  0.0 |  0.0 |   0.0 |  nan | 31.722932 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.069408 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01928 |
    | 1997-08-05 03:00:00 |  0.0 | 12.0 |              0.0 |             95.6 |       1.2 |               101.8 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 |  0.0 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 1.404387 |                       1.98784 |                          0.0 |                      0.092693 |                           0.125652 |             1.342594 |                  1.558052 |     100.457406 |   1.237469 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 |  16.67644 |  0.101011 |                   0.0 |                       0.0 |                   17.58988 |                   3.15411 |                      0.0 |    -0.131421 |              0.0 |          14.43577 |            117.694689 |                 100.0 |               109.230942 |               87.719298 |  0.0 |  0.0 | -0.020794 | -0.020794 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.067615 | 0.018782 | 0.061159 |  0.0 |  0.0 |   0.0 |  nan | 31.611504 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.067615 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018782 |
    | 1997-08-05 04:00:00 |  0.0 | 11.7 |              0.0 |             96.2 |       1.9 |               101.8 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 |  0.0 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                 1.376819 |                      1.974268 |                          0.0 |                      0.091096 |                           0.124901 |               1.3245 |                  1.550047 |       100.4755 |   1.238856 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.366022 |                     15.438712 |            23.144579 |  0.063733 |         0.00566 |                0.005336 | -0.010417 | 16.305014 |  0.099468 |              0.004056 |                  0.004056 |                   17.58988 |                  3.166293 |                      0.0 |    -0.127873 |              0.0 |         14.423587 |             74.333488 |                 100.0 |               109.230942 |               93.746935 |  0.0 |  0.0 | -0.019617 | -0.019617 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.065868 | 0.018297 | 0.080776 |  0.0 |  0.0 |   0.0 |  nan | 31.501619 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.065868 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018297 |
    | 1997-08-05 05:00:00 |  0.0 | 11.6 |              0.0 |             95.7 |       1.3 |               101.8 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 |  0.0 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 1.367736 |                      1.961295 |                          0.0 |                      0.090568 |                           0.124183 |             1.308923 |                  1.541578 |     100.491077 |   1.239363 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.144579 |  0.908598 |        0.080684 |                0.076088 | -0.010417 | 15.938731 |  0.093717 |              0.057827 |                  0.057827 |                   17.58988 |                  3.179808 |                      0.0 |    -0.074665 |              0.0 |         14.410072 |            108.641252 |                 100.0 |               109.230942 |               106.41204 |  0.0 |  0.0 |   -0.0093 |   -0.0093 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.064167 | 0.017824 | 0.090076 |  0.0 |  0.0 |   0.0 |  nan | 31.397486 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.064167 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.017824 |
    | 1997-08-05 06:00:00 |  0.0 | 12.6 |              0.6 |             95.1 |       1.6 |               101.8 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 |  0.0 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                 1.460978 |                      1.952516 |                          0.0 |                       0.09596 |                           0.123697 |              1.38939 |                  1.535084 |      100.41061 |   1.234655 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  2.420659 |        0.472904 |                 0.45318 | -0.010417 |  15.59162 |  0.064619 |              0.344417 |                  0.344417 |                  18.027018 |                  3.288825 |                      0.0 |     0.207383 |              0.0 |         14.738193 |             88.271017 |                 100.0 |               109.230942 |               106.41204 |  0.0 |  0.0 |   0.05232 |   0.05232 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.06251 | 0.017364 | 0.037756 |  0.0 |  0.0 |   0.0 |  nan |  31.32245 |   73.4603 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.06251 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.017364 |
    | 1997-08-05 07:00:00 |  0.0 | 14.8 |              0.9 |             96.3 |       1.9 |               101.9 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 |  0.0 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                 1.685952 |                      1.949426 |                          0.0 |                      0.108789 |                           0.123526 |             1.623571 |                  1.535091 |     100.276429 |   1.225361 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  4.619722 |        1.148656 |                1.101438 | -0.010417 | 15.341501 |  0.011696 |              0.837093 |                  0.837093 |                  18.027018 |                  3.287363 |                      0.0 |     0.700119 |              0.0 |         14.739656 |             74.333488 |                 100.0 |               109.230942 |               106.41204 |  0.0 |  0.0 |  0.166735 |  0.037756 | 0.079061 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.060895 | 0.016915 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.300337 | 73.381239 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.060895 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016915 |
    | 1997-08-05 08:00:00 |  0.0 | 17.1 |              1.0 |             89.9 |       3.2 |               101.9 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 |  0.0 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                 1.953031 |                       1.95097 |                          0.0 |                      0.123726 |                           0.123612 |             1.755775 |                  1.538746 |     100.144225 |   1.215051 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  7.198198 |        1.917614 |                1.839976 | -0.010417 | 15.267791 | -0.039576 |              1.398382 |                  1.398382 |                  18.027018 |                  3.279458 |                      0.0 |     1.261738 |              0.0 |         14.747561 |             44.135508 |                 100.0 |               109.343038 |              106.487872 |  0.0 |  0.0 |  0.358814 |       0.0 | 0.185653 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.059322 | 0.016478 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.329496 | 73.195586 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.059322 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016478 |
    | 1997-08-05 09:00:00 |  0.0 | 19.4 |              0.9 |             79.2 |       4.0 |               101.9 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 |  0.0 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                 2.256393 |                      1.957159 |                          0.0 |                      0.140363 |                           0.123954 |             1.787063 |                  1.543872 |     100.112937 |   1.205358 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.604725 |  9.475573 |        2.356024 |                2.266269 | -0.010417 | 15.364988 | -0.087156 |              1.722365 |                  1.722365 |                  17.939591 |                  3.250753 |                      0.0 |     1.586917 |              0.0 |         14.688838 |             35.308407 |                 100.0 |               109.606454 |              106.665883 |  0.0 |  0.0 |  0.541756 |       0.0 | 0.263073 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.05779 | 0.016053 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.406236 | 72.932513 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05779 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016053 |
    | 1997-08-05 10:00:00 |  0.0 | 20.6 |              1.0 |             73.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 |  0.0 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                  2.43042 |                      1.962848 |                          0.0 |                      0.149767 |                           0.124269 |             1.783928 |                  1.546397 |     100.116072 |   1.200448 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 | 10.786309 |        2.873495 |                 2.75694 | -0.010417 | 15.620786 | -0.107551 |              2.095274 |                  2.095274 |                  18.027018 |                  3.266966 |                      0.0 |     1.959151 |              0.0 |         14.760052 |             30.702962 |                 100.0 |               109.980185 |              106.917992 |  0.0 |  0.0 |  0.716789 |       0.0 | 0.332173 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.056297 | 0.015638 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  31.50337 |  72.60034 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.056297 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015638 |
    | 1997-08-05 11:00:00 |  0.0 | 21.4 |              1.0 |             67.3 |       4.1 |               101.9 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 |  0.0 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                 2.552842 |                      1.968551 |                          0.0 |                      0.156328 |                           0.124585 |             1.718063 |                  1.545886 |     100.181937 |   1.197482 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.949834 | 10.865527 |        2.894599 |                2.761725 | -0.010417 | 15.944568 | -0.117837 |              2.098911 |                  2.098911 |                  18.201874 |                  3.309837 |                      0.0 |     1.961001 |              0.0 |         14.892037 |             34.447226 |                 100.0 |               110.452867 |                107.2361 |  0.0 |  0.0 |  0.746035 |       0.0 | 0.369918 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.054843 | 0.015234 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.610791 | 72.230422 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.054843 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015234 |
    | 1997-08-05 12:00:00 |  0.0 | 22.5 |              1.0 |             62.2 |       4.1 |               101.9 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 |  0.0 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                 2.729939 |                      1.976351 |                          0.0 |                      0.165743 |                           0.125016 |             1.698022 |                  1.546988 |     100.201978 |   1.193116 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |             24.06487 |  10.45226 |        2.784503 |                 2.65055 | -0.010417 | 16.302636 | -0.133863 |              2.014418 |                  2.014418 |                  18.289301 |                  3.330376 |                      0.0 |     1.875652 |              0.0 |         14.958925 |             34.447226 |                 100.0 |               110.980311 |              107.590074 |  0.0 |  0.0 |  0.771401 |       0.0 | 0.389258 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.053426 | 0.014841 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.734237 | 71.841164 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.053426 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014841 |
    | 1997-08-05 13:00:00 |  0.0 | 23.1 |              1.0 |             59.6 |       4.4 |               101.9 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 |  0.0 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                 2.830978 |                      1.983133 |                          0.0 |                      0.171077 |                           0.125391 |             1.687263 |                  1.550892 |     100.212737 |   1.190747 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.294943 |  10.96314 |        2.920603 |                2.764822 | -0.010417 | 16.714124 | -0.137935 |              2.101265 |                  2.101265 |                  18.464157 |                  3.363175 |                      0.0 |     1.961133 |              0.0 |         15.100981 |             32.098552 |                 100.0 |               111.536545 |              107.962247 |  0.0 |  0.0 |  0.840981 |       0.0 | 0.414609 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.052046 | 0.014457 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.861756 | 71.426555 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.052046 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014457 |
    | 1997-08-05 14:00:00 |  0.0 | 23.4 |              1.0 |             55.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 |  0.0 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                 2.882708 |                      1.990983 |                          0.0 |                      0.173797 |                           0.125825 |              1.59702 |                  1.552718 |      100.30298 |   1.189943 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.409979 | 10.528069 |        2.804699 |                2.649346 | -0.010417 | 17.139185 | -0.135234 |              2.013503 |                  2.013503 |                  18.551584 |                   3.38194 |                      0.0 |     1.872589 |              0.0 |         15.169644 |             30.702962 |                 100.0 |               112.130402 |              108.358325 |  0.0 |  0.0 |  0.865172 |       0.0 | 0.419297 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.050702 | 0.014084 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.986572 | 71.007258 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.050702 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014084 |
    | 1997-08-05 15:00:00 |  0.0 | 23.8 |              1.0 |             52.1 |       4.5 |               101.9 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 |  0.0 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                 2.952961 |                      2.001493 |                          0.0 |                      0.177481 |                           0.126406 |             1.538493 |                  1.555327 |     100.361507 |     1.1886 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.525016 |  8.882849 |        2.366409 |                 2.23335 | -0.010417 | 17.555242 | -0.134887 |              1.697346 |                  1.697346 |                  18.639012 |                  3.400029 |                      0.0 |     1.555678 |              0.0 |         15.238983 |              31.38525 |                 100.0 |               112.732468 |              108.758548 |  0.0 |  0.0 |  0.798412 |       0.0 | 0.393038 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.049392 |  0.01372 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.111043 |  70.61422 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.049392 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01372 |
    | 1997-08-05 16:00:00 |  0.0 | 24.0 |              1.0 |             49.5 |       4.7 |               101.8 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 |  0.0 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                 2.988644 |                      2.010993 |                          0.0 |                      0.179348 |                            0.12693 |             1.479379 |                  1.556257 |     100.320621 |   1.186889 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  6.439028 |         1.71537 |                1.620343 | -0.010417 | 17.970142 | -0.130245 |              1.231461 |                  1.231461 |                   18.72644 |                   3.42173 |                      0.0 |     1.088889 |              0.0 |         15.304709 |             30.049708 |                 100.0 |               113.298222 |              109.133419 |  0.0 |  0.0 |  0.702711 |       0.0 | 0.339609 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.048116 | 0.013366 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.230871 | 70.274611 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.048116 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.013366 |
    | 1997-08-05 17:00:00 |  0.0 | 24.0 |              1.0 |             48.1 |       4.5 |               101.8 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 |  0.0 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                 2.988644 |                      2.022657 |                          0.0 |                      0.179348 |                           0.127573 |             1.437538 |                  1.557615 |     100.362462 |   1.187075 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  3.911896 |        1.042137 |                0.985011 | -0.010417 |  18.36957 | -0.121617 |              0.748608 |                  0.748608 |                   18.72644 |                  3.423912 |                      0.0 |     0.605945 |              0.0 |         15.302528 |              31.38525 |                 100.0 |               113.788174 |              109.457117 |  0.0 |  0.0 |   0.55931 |       0.0 | 0.275426 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.046873 |  0.01302 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.342072 | 69.999185 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.046873 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01302 |
    | 1997-08-05 18:00:00 |  0.0 | 23.7 |              1.0 |             49.4 |       4.1 |               101.8 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 |  0.0 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                  2.93526 |                      2.037589 |                          0.0 |                      0.176554 |                           0.128396 |             1.450018 |                  1.560199 |     100.349982 |   1.188219 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  1.905864 |        0.507726 |                0.480125 | -0.010417 | 18.740238 | -0.107131 |              0.364895 |                  0.364895 |                   18.72644 |                   3.42458 |                      0.0 |     0.222204 |              0.0 |          15.30186 |             34.447226 |                 100.0 |               114.186294 |                109.7195 |  0.0 |  0.0 |  0.406398 |       0.0 | 0.207367 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.045662 | 0.012684 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.438786 | 69.791818 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.045662 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012684 |
    | 1997-08-05 19:00:00 |  0.0 | 22.8 |              1.0 |             57.0 |       3.8 |               101.8 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 |  0.0 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                 2.780058 |                      2.054233 |                          0.0 |                      0.168392 |                           0.129312 |             1.584633 |                  1.565154 |     100.215367 |   1.191234 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  0.576079 |        0.153469 |                0.145175 | -0.010417 | 19.062619 | -0.080727 |              0.110333 |                  0.110333 |                   18.72644 |                  3.420162 |                      0.0 |    -0.032173 |              0.0 |         15.306277 |             37.166744 |                 100.0 |               114.486496 |              109.916969 |  0.0 |  0.0 |  0.257469 |       0.0 | 0.133625 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.044483 | 0.012356 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.509096 | 69.658193 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044483 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012356 |
    | 1997-08-05 20:00:00 |  0.0 | 21.3 |              0.5 |             65.9 |       2.5 |               101.8 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 |  0.0 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 2.537251 |                      2.072623 |                          0.0 |                      0.155495 |                           0.130323 |             1.672048 |                  1.574157 |     100.127952 |   1.196911 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.072689 |                     15.438712 |            24.870125 |  0.002495 |        0.003269 |                0.003122 | -0.010417 | 19.296988 | -0.043265 |              0.002372 |                  0.002372 |                  18.901295 |                  3.445356 |                      0.0 |    -0.141184 |              0.0 |         15.455939 |             56.493451 |                 100.0 |               114.680153 |                89.03222 |  0.0 |  0.0 |  0.102212 |       0.0 | 0.066116 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.043334 | 0.012037 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.541945 | 69.592077 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.043334 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012037 |
    | 1997-08-05 21:00:00 |  0.0 | 19.3 |              0.0 |             73.0 |       1.9 |               101.8 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 |  0.0 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                 2.242395 |                      2.091705 |                          0.0 |                      0.139602 |                           0.131371 |             1.636948 |                  1.582026 |     100.163052 |   1.205255 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.406483 |    0.0023 |                   0.0 |                       0.0 |                  18.901295 |                  3.434698 |                      0.0 |    -0.143112 |              0.0 |         15.466597 |             74.333488 |                 100.0 |               114.776034 |               87.719298 |  0.0 |  0.0 |  0.046259 |       0.0 | 0.031824 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.042215 | 0.011726 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.529228 | 69.560253 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.042215 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011726 |
    | 1997-08-05 22:00:00 |  0.0 | 18.3 |              0.0 |             76.7 |       2.2 |               101.8 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 |  0.0 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                  2.10653 |                       2.11094 |                          0.0 |                      0.132185 |                           0.132426 |             1.615709 |                  1.588483 |     100.184291 |   1.209486 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.364094 |  0.022984 |                   0.0 |                       0.0 |                  18.901295 |                  3.427556 |                      0.0 |    -0.142815 |              0.0 |         15.473739 |             64.197103 |                 100.0 |               114.822199 |               87.719298 |  0.0 |  0.0 |  0.042514 |       0.0 | 0.027773 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.041124 | 0.011423 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.495827 | 69.532479 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041124 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011423 |
    | 1997-08-05 23:00:00 |  0.0 | 17.4 |              0.0 |             81.8 |       2.7 |               101.8 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 |  0.0 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                 1.990459 |                      2.129218 |                          0.0 |                      0.125796 |                           0.133428 |             1.628195 |                  1.595583 |     100.171805 |   1.213176 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.252757 |   0.04002 |                   0.0 |                       0.0 |                  18.901295 |                  3.418333 |                      0.0 |    -0.142431 |              0.0 |         15.482962 |             52.308751 |                 100.0 |               114.862497 |               87.719298 |  0.0 |  0.0 |  0.035583 |       0.0 | 0.021518 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.040062 | 0.011128 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.445391 | 69.510962 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011128 |

.. _lland_v3_acker_heavy_rain_hourly:

acre (heavy rain)
-----------------

When comparing the soil moisture content (|BoWa|) at August 6th, 00:00, we
notice only small differences between the daily
(:ref:`lland_v3_acker_summer_daily`, 70.2 mm) and the hourly
(:ref:`lland_v3_acker_summer_daily`, 69.5) simulation results for a dry
situation.  However, when we compare the differences between our daily
(:ref:`lland_v3_acker_heavy_rain_daily`, 152.8 mm) and our hourly (next
table) test for a wet situation, we see much more pronounced differences,
which are mainly due to the prioritisation of throughfall (|NBes|) over
interception evaporation (|EvI|):

.. integration-test::

    >>> inputs.nied.series = 20.0/24.0
    >>> test("lland_v3_acker_heavy_rain_hourly",
    ...      axis1=(inputs.nied, fluxes.qah),
    ...      axis2=states.bowa,
    ...      use_conditions=conditions_acker_heavy_rain)
    |                date |     nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration |     nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |        sp | globalradiation | adjustedglobalradiation |         g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |     nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |     wada |      qdb |     qib1 | qib2 |      qbb | qkap |     qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |      ebdn |       bowa |    qdgz1 | qdgz2 |    qigz1 | qigz2 |     qbgz |    qdga1 | qdga2 |    qiga1 | qiga2 |     qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.833333 | 16.9 |              0.0 |             95.1 |       0.8 |               101.5 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 | 0.783333 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                 1.928423 |                      1.975309 |                          0.0 |                      0.122362 |                           0.124959 |              1.83393 |                  1.696461 |       99.66607 |    1.21073 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.208683 | -0.036532 |                   0.0 |                       0.0 |                   7.982856 |                  0.901196 |                      0.0 |     -0.03755 |              0.0 |           7.08166 |            176.542034 |                 100.0 |                70.327061 |               87.719298 | 0.083333 |  0.0 | -0.002965 | -0.002965 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.007403 |      0.0 |  0.0 |      0.0 |  0.0 | 0.007403 |  0.80139 | 0.222608 | 0.702965 |  0.0 |  0.0 |   0.0 |  nan | 31.308721 |   102.5625 | 0.007403 |   0.0 |      0.0 |   0.0 |      0.0 |  0.80139 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.222608 |
    | 1997-08-03 01:00:00 | 0.833333 | 16.6 |              0.0 |             94.9 |       0.8 |               101.5 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 | 0.783333 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                 1.892018 |                      1.970628 |                          0.0 |                      0.120339 |                             0.1247 |             1.795525 |                  1.694576 |      99.704475 |   1.212158 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.295735 | -0.028172 |                   0.0 |                       0.0 |                   7.982856 |                  0.901844 |                      0.0 |    -0.037577 |              0.0 |          7.081012 |            176.542034 |                 100.0 |                70.229321 |               87.719298 | 0.786298 |  0.0 | -0.002685 | -0.002685 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786298 | 0.070172 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070172 | 0.781695 | 0.217137 | 0.702685 |  0.0 |  0.0 |   0.0 |  nan | 31.326476 | 103.278626 | 0.070172 |   0.0 |      0.0 |   0.0 |      0.0 | 0.781695 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.217137 |
    | 1997-08-03 02:00:00 | 0.833333 | 16.4 |              0.0 |             95.9 |       0.8 |               101.5 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 | 0.783333 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 1.868084 |                       1.96233 |                          0.0 |                      0.119006 |                           0.124241 |             1.791493 |                  1.692183 |      99.708507 |   1.213014 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 |  15.35492 | -0.022574 |                   0.0 |                       0.0 |                   7.982856 |                  0.902355 |                      0.0 |    -0.037598 |              0.0 |            7.0805 |            176.542034 |                 100.0 |                  69.3082 |               87.719298 | 0.786019 |  0.0 | -0.004107 | -0.004107 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786019 | 0.070722 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070722 | 0.763323 | 0.212034 | 0.704107 |  0.0 |  0.0 |   0.0 |  nan | 31.338633 | 103.993923 | 0.070722 |   0.0 |      0.0 |   0.0 |      0.0 | 0.763323 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.212034 |
    | 1997-08-03 03:00:00 | 0.833333 | 16.3 |              0.0 |             96.7 |       0.8 |               101.5 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 | 0.783333 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                 1.856216 |                      1.948396 |                          0.0 |                      0.118345 |                           0.123469 |             1.794961 |                  1.689422 |      99.705039 |   1.213417 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.395444 | -0.019538 |                   0.0 |                       0.0 |                   7.982856 |                  0.902364 |                      0.0 |    -0.037598 |              0.0 |          7.080492 |            176.542034 |                 100.0 |                68.389393 |               87.719298 | 0.787441 |  0.0 | -0.005232 | -0.005232 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.787441 | 0.071427 |      0.0 |  0.0 |      0.0 |  0.0 | 0.071427 | 0.745441 | 0.207067 | 0.705232 |  0.0 |  0.0 |   0.0 |  nan | 31.347755 | 104.709937 | 0.071427 |   0.0 |      0.0 |   0.0 |      0.0 | 0.745441 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.207067 |
    | 1997-08-03 04:00:00 | 0.833333 | 16.0 |              0.0 |             97.2 |       0.8 |               101.5 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 | 0.783333 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                 1.821009 |                      1.929443 |                          0.0 |                      0.116379 |                           0.122418 |              1.77002 |                   1.68569 |       99.72998 |    1.21479 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                 0.429734 |                     15.566134 |            10.503757 |  0.086656 |        0.007799 |                0.006997 | -0.010417 | 15.425849 | -0.012402 |              0.005318 |                  0.005318 |                   7.982856 |                  0.902341 |                      0.0 |     -0.03228 |              0.0 |          7.080514 |            176.542034 |                 100.0 |                67.470886 |                81.46212 | 0.788565 |  0.0 | -0.004819 | -0.004819 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788565 | 0.072109 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072109 |  0.72804 | 0.202233 | 0.704819 |  0.0 |  0.0 |   0.0 |  nan |  31.34974 | 105.426394 | 0.072109 |   0.0 |      0.0 |   0.0 |      0.0 |  0.72804 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.202233 |
    | 1997-08-03 05:00:00 | 0.833333 | 15.9 |              0.0 |             97.5 |       0.6 |               101.5 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 | 0.783333 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                 1.809403 |                      1.907618 |                          0.0 |                       0.11573 |                           0.121206 |             1.764168 |                  1.682599 |      99.735832 |   1.215237 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  0.968392 |        0.087159 |                0.078957 | -0.010417 | 15.432466 | -0.010099 |              0.060007 |                  0.060007 |                   7.982856 |                  0.901482 |                      0.0 |     0.022445 |              0.0 |          7.081373 |            235.389378 |                 100.0 |                66.553006 |               73.619304 | 0.788152 |  0.0 |  0.005073 |  0.005073 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788152 | 0.072652 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072652 | 0.711104 | 0.197529 | 0.694927 |  0.0 |  0.0 |   0.0 |  nan | 31.349422 | 106.141894 | 0.072652 |   0.0 |      0.0 |   0.0 |      0.0 | 0.711104 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.197529 |
    | 1997-08-03 06:00:00 | 0.833333 | 16.0 |              0.0 |             97.7 |       0.9 |               101.5 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 | 0.783333 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                 1.821009 |                      1.884011 |                          0.0 |                      0.116379 |                           0.119893 |             1.779125 |                   1.68038 |      99.720875 |   1.214748 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |    2.4897 |        0.224084 |                0.206122 | -0.010417 | 15.431406 | -0.012282 |              0.156653 |                  0.156653 |                   7.982856 |                   0.89977 |                      0.0 |     0.119163 |              0.0 |          7.083086 |            156.926252 |                 100.0 |                65.637518 |               72.818235 |  0.77826 |  0.0 |  0.026292 |  0.026292 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.77826 | 0.072311 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072311 | 0.694607 | 0.192946 | 0.673708 |  0.0 |  0.0 |   0.0 |  nan | 31.351287 | 106.847843 | 0.072311 |   0.0 |      0.0 |   0.0 |      0.0 | 0.694607 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.192946 |
    | 1997-08-03 07:00:00 | 0.833333 | 16.6 |              0.0 |             97.4 |       0.9 |               101.5 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 | 0.783333 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                 1.892018 |                      1.865111 |                          0.0 |                      0.120339 |                           0.118841 |             1.842826 |                  1.680931 |      99.657174 |   1.211943 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  4.679733 |        0.421196 |                0.393598 | -0.010417 | 15.437623 | -0.025107 |              0.299135 |                  0.299135 |                   7.982856 |                  0.896826 |                      0.0 |     0.261767 |              0.0 |          7.086029 |            156.926252 |                 100.0 |                64.735367 |               72.024255 | 0.757042 |  0.0 |  0.057456 |  0.057456 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.757042 | 0.070885 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070885 | 0.678514 | 0.188476 | 0.642544 |  0.0 |  0.0 |   0.0 |  nan | 31.365978 |    107.534 | 0.070885 |   0.0 |      0.0 |   0.0 |      0.0 | 0.678514 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.188476 |
    | 1997-08-03 08:00:00 | 0.833333 | 17.4 |              0.0 |             96.8 |       0.9 |               101.6 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 | 0.783333 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                 1.990459 |                      1.846377 |                          0.0 |                      0.125796 |                           0.117796 |             1.926764 |                  1.685281 |      99.673236 |   1.209425 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  7.219274 |        0.649765 |                0.615417 | -0.010417 | 15.486592 |  -0.04133 |              0.467717 |                  0.467717 |                   7.982856 |                  0.891371 |                      0.0 |     0.430576 |              0.0 |          7.091484 |            156.926252 |                 100.0 |                63.859549 |               71.249063 | 0.725878 |  0.0 |  0.096099 |  0.096099 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.725878 | 0.068473 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068473 | 0.662788 | 0.184108 | 0.603901 |  0.0 |  0.0 |   0.0 |  nan |  31.39689 | 108.191405 | 0.068473 |   0.0 |      0.0 |   0.0 |      0.0 | 0.662788 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.184108 |
    | 1997-08-03 09:00:00 | 0.833333 | 19.0 |              0.2 |             86.1 |       1.3 |               101.6 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 | 0.783333 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                 2.200854 |                      1.834142 |                          0.0 |                      0.137341 |                           0.117113 |             1.894936 |                  1.687258 |      99.705064 |   1.202945 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             10.73504 |  9.442017 |         1.18975 |                1.122345 | -0.010417 | 15.589635 | -0.073664 |              0.852983 |                  0.852983 |                    8.15863 |                  0.922604 |                      0.0 |     0.814541 |              0.0 |          7.236026 |            108.641252 |                 100.0 |                63.021373 |               70.503105 | 0.687235 |  0.0 |  0.222763 |  0.222763 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687235 | 0.065285 |      0.0 |  0.0 |      0.0 |  0.0 | 0.065285 | 0.647395 | 0.179832 | 0.477237 |  0.0 |  0.0 |   0.0 |  nan | 31.460138 | 108.813354 | 0.065285 |   0.0 |      0.0 |   0.0 |      0.0 | 0.647395 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.179832 |
    | 1997-08-03 10:00:00 | 0.833333 | 20.3 |              0.5 |             76.8 |       1.5 |               101.6 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 | 0.783333 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                  2.38585 |                      1.830729 |                          0.0 |                      0.147368 |                           0.116922 |             1.832333 |                  1.688695 |      99.767667 |   1.197896 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.966322 | 10.709164 |         1.92774 |                  1.8057 | -0.010417 | 15.800459 |  -0.09719 |              1.372332 |                  1.372332 |                   8.334404 |                  0.955299 |                      0.0 |     1.332528 |              0.0 |          7.379106 |             94.155751 |                 100.0 |                62.229232 |               69.794418 |  0.56057 |  0.0 |  0.389923 |  0.389923 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.56057 | 0.053582 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053582 | 0.632206 | 0.175613 | 0.310077 |  0.0 |  0.0 |   0.0 |  nan | 31.546911 | 109.320342 | 0.053582 |   0.0 |      0.0 |   0.0 |      0.0 | 0.632206 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.175613 |
    | 1997-08-03 11:00:00 | 0.833333 | 21.4 |              0.7 |             71.8 |       1.2 |               101.6 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 | 0.783333 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                 2.552842 |                      1.835607 |                          0.0 |                      0.156328 |                           0.117195 |             1.832941 |                   1.69067 |      99.767059 |    1.19342 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            11.544527 | 10.773868 |        2.327265 |                2.214067 | -0.010417 | 16.089704 | -0.114702 |              1.682691 |                  1.682691 |                   8.773841 |                  1.039981 |                      0.0 |     1.639359 |              0.0 |          7.733859 |            117.694689 |                 100.0 |                61.584098 |               69.214576 |  0.39341 |  0.0 |   0.46628 |   0.46628 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.39341 | 0.037782 |      0.0 |  0.0 |      0.0 |  0.0 | 0.037782 | 0.617055 | 0.171404 |  0.23372 |  0.0 |  0.0 |   0.0 |  nan | 31.651197 | 109.675971 | 0.037782 |   0.0 |      0.0 |   0.0 |      0.0 | 0.617055 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.171404 |
    | 1997-08-03 12:00:00 | 0.833333 | 21.3 |              0.8 |             67.5 |       1.3 |               101.6 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 | 0.783333 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 2.537251 |                      1.844905 |                          0.0 |                      0.155495 |                           0.117714 |             1.712644 |                  1.692316 |      99.887356 |   1.194363 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.238374 | 10.364714 |        2.425457 |                 2.25556 | -0.010417 | 16.437323 | -0.105034 |              1.714225 |                  1.714225 |                   9.301164 |                  1.142661 |                      0.0 |     1.666614 |              0.0 |          8.158503 |            108.641252 |                 100.0 |                61.131877 |               68.806682 | 0.317053 |  0.0 |  0.491066 |  0.491066 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.317053 | 0.030555 |      0.0 |  0.0 |      0.0 |  0.0 | 0.030555 | 0.601998 | 0.167222 | 0.208934 |  0.0 |  0.0 |   0.0 |  nan | 31.745814 | 109.962469 | 0.030555 |   0.0 |      0.0 |   0.0 |      0.0 | 0.601998 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.167222 |
    | 1997-08-03 13:00:00 | 0.833333 | 21.8 |              0.5 |             66.1 |       1.5 |               101.6 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 | 0.783333 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                 2.616045 |                      1.859671 |                          0.0 |                      0.159698 |                           0.118538 |             1.729206 |                  1.694238 |      99.870794 |   1.192265 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.700938 | 10.872489 |         1.95714 |                 1.78608 | -0.010417 | 16.752713 | -0.109021 |              1.357421 |                  1.357421 |                   9.652713 |                  1.211985 |                      0.0 |     1.306922 |              0.0 |          8.440727 |             94.155751 |                 100.0 |                60.767747 |               68.477376 | 0.292267 |  0.0 |  0.424802 |  0.424802 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.292267 | 0.028251 |      0.0 |  0.0 |      0.0 |  0.0 | 0.028251 | 0.587207 | 0.163113 | 0.275198 |  0.0 |  0.0 |   0.0 |  nan | 31.844419 | 110.226486 | 0.028251 |   0.0 |      0.0 |   0.0 |      0.0 | 0.587207 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.163113 |
    | 1997-08-03 14:00:00 | 0.833333 | 22.9 |              0.4 |             63.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 | 0.783333 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                 2.796942 |                       1.88551 |                          0.0 |                      0.169283 |                           0.119977 |             1.773261 |                  1.698059 |      99.826739 |   1.187639 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             12.93222 | 10.461143 |        1.694785 |                1.510276 | -0.010417 | 17.081396 | -0.125682 |              1.147809 |                  1.147809 |                   9.828487 |                   1.24753 |                      0.0 |     1.095829 |              0.0 |          8.580957 |             74.333488 |                 100.0 |                60.432333 |               68.173354 | 0.358532 |  0.0 |  0.415458 |  0.415458 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.358532 | 0.034766 |      0.0 |  0.0 |      0.0 |  0.0 | 0.034766 | 0.572854 | 0.159126 | 0.284542 |  0.0 |  0.0 |   0.0 |  nan | 31.959684 | 110.550251 | 0.034766 |   0.0 |      0.0 |   0.0 |      0.0 | 0.572854 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.159126 |
    | 1997-08-03 15:00:00 | 0.833333 | 22.7 |              0.5 |             62.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 | 0.783333 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                 2.763263 |                      1.918249 |                          0.0 |                      0.167505 |                           0.121797 |             1.724276 |                  1.703725 |      99.875724 |    1.18866 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.394785 |  8.865382 |        1.595844 |                1.396394 | -0.010417 | 17.465613 | -0.113063 |               1.06126 |                   1.06126 |                  10.180036 |                  1.316893 |                      0.0 |     1.006389 |              0.0 |          8.863143 |             74.333488 |                 100.0 |                  60.0212 |               67.799797 | 0.367875 |  0.0 |  0.393508 |  0.393508 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.367875 | 0.035799 |      0.0 |  0.0 |      0.0 |  0.0 | 0.035799 | 0.558968 | 0.155269 | 0.306492 |  0.0 |  0.0 |   0.0 |  nan |  32.06233 | 110.882327 | 0.035799 |   0.0 |      0.0 |   0.0 |      0.0 | 0.558968 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.155269 |
    | 1997-08-03 16:00:00 | 0.833333 | 22.5 |              0.5 |             61.1 |       2.3 |               101.6 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 | 0.783333 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                 2.729939 |                      1.957676 |                          0.0 |                      0.165743 |                           0.123983 |             1.667993 |                  1.709704 |      99.932007 |   1.189715 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.741708 |  6.474205 |        1.165412 |                1.002589 | -0.010417 | 17.807767 | -0.101352 |              0.761967 |                  0.761967 |                  10.443698 |                  1.370145 |                      0.0 |     0.704878 |              0.0 |          9.073553 |             61.405925 |                 100.0 |                59.599725 |               67.415809 | 0.389826 |  0.0 |  0.342513 |  0.342513 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.389826 | 0.038075 |      0.0 |  0.0 |      0.0 |  0.0 | 0.038075 | 0.545483 | 0.151523 | 0.357487 |  0.0 |  0.0 |   0.0 |  nan | 32.153266 | 111.234078 | 0.038075 |   0.0 |      0.0 |   0.0 |      0.0 | 0.545483 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.151523 |
    | 1997-08-03 17:00:00 | 0.833333 | 21.9 |              0.3 |             62.1 |       2.4 |               101.6 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 | 0.783333 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                 2.632058 |                      2.001493 |                          0.0 |                       0.16055 |                           0.126406 |             1.634508 |                  1.718782 |      99.965492 |   1.192283 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.088631 |  3.977658 |         0.57281 |                0.493699 | -0.010417 | 18.110885 | -0.081845 |              0.375211 |                  0.375211 |                   10.70736 |                  1.421011 |                      0.0 |     0.316002 |              0.0 |          9.286348 |             58.847345 |                 100.0 |                59.153511 |               67.008135 | 0.440821 |  0.0 |  0.236583 |  0.236583 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.440821 |  0.04323 |      0.0 |  0.0 |      0.0 |  0.0 |  0.04323 | 0.532443 | 0.147901 | 0.463417 |  0.0 |  0.0 |   0.0 |  nan | 32.224694 | 111.631669 |  0.04323 |   0.0 |      0.0 |   0.0 |      0.0 | 0.532443 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.147901 |
    | 1997-08-03 18:00:00 | 0.833333 | 21.4 |              0.1 |             67.0 |       2.5 |               101.6 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 | 0.783333 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                 2.552842 |                      2.046164 |                          0.0 |                      0.156328 |                           0.128868 |             1.710404 |                  1.731481 |      99.889596 |   1.193968 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  1.972911 |        0.213084 |                0.183891 | -0.010417 | 18.348979 | -0.065902 |              0.139757 |                  0.139757 |                  10.795247 |                  1.433696 |                      0.0 |     0.080019 |              0.0 |           9.36155 |             56.493451 |                 100.0 |                58.649429 |               66.546166 |  0.54675 |  0.0 |  0.156446 |  0.156446 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.54675 | 0.053875 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053875 | 0.519945 | 0.144429 | 0.543554 |  0.0 |  0.0 |   0.0 |  nan | 32.280179 | 112.124544 | 0.053875 |   0.0 |      0.0 |   0.0 |      0.0 | 0.519945 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.144429 |
    | 1997-08-03 19:00:00 | 0.833333 | 20.7 |              0.0 |             74.5 |       2.5 |               101.6 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 | 0.783333 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                 2.445437 |                      2.089517 |                          0.0 |                      0.150575 |                           0.131251 |              1.82185 |                  1.748925 |       99.77815 |   1.196313 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  0.634046 |        0.057067 |                0.049078 | -0.010417 | 18.533931 | -0.046787 |              0.037299 |                  0.037299 |                  10.795247 |                  1.423756 |                      0.0 |    -0.022024 |              0.0 |          9.371491 |             56.493451 |                 100.0 |                58.024955 |               65.971756 | 0.626887 |  0.0 |   0.09619 |   0.09619 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.626887 | 0.062121 |      0.0 |  0.0 |      0.0 |  0.0 | 0.062121 | 0.508013 | 0.141115 |  0.60381 |  0.0 |  0.0 |   0.0 |  nan |  32.31655 |  112.68931 | 0.062121 |   0.0 |      0.0 |   0.0 |      0.0 | 0.508013 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.141115 |
    | 1997-08-03 20:00:00 | 0.833333 | 19.4 |              0.0 |             81.2 |       2.2 |               101.6 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 | 0.783333 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                 2.256393 |                      2.126995 |                          0.0 |                      0.140363 |                           0.133306 |             1.832191 |                  1.766824 |      99.767809 |   1.201582 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                   0.1364 |                     15.566134 |            14.204272 |  0.008648 |        0.000778 |                0.000669 | -0.010417 | 18.655165 | -0.016088 |              0.000509 |                  0.000509 |                  10.795247 |                  1.412078 |                      0.0 |    -0.058328 |              0.0 |          9.383169 |             64.197103 |                 100.0 |                 57.30995 |               83.797685 | 0.687143 |  0.0 |   0.04938 |   0.04938 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687143 | 0.068523 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068523 | 0.496578 | 0.137938 |  0.65062 |  0.0 |  0.0 |   0.0 |  nan | 32.322221 | 113.307931 | 0.068523 |   0.0 |      0.0 |   0.0 |      0.0 | 0.496578 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.137938 |
    | 1997-08-03 21:00:00 | 0.833333 | 17.8 |              0.0 |             86.9 |       1.7 |               101.6 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 | 0.783333 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                 2.041337 |                      2.153241 |                          0.0 |                      0.128603 |                           0.134742 |             1.773922 |                  1.781538 |      99.826078 |   1.208454 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.674071 |   0.01888 |                   0.0 |                       0.0 |                  10.795247 |                  1.401564 |                      0.0 |    -0.058399 |              0.0 |          9.393683 |             83.078604 |                 100.0 |                56.527431 |               87.719298 | 0.733954 |  0.0 |  0.019258 |  0.019258 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.733954 | 0.073689 |      0.0 |  0.0 |      0.0 |  0.0 | 0.073689 | 0.485588 | 0.134886 | 0.680742 |  0.0 |  0.0 |   0.0 |  nan | 32.292925 | 113.968195 | 0.073689 |   0.0 |      0.0 |   0.0 |      0.0 | 0.485588 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.134886 |
    | 1997-08-03 22:00:00 | 0.833333 | 17.0 |              0.0 |             90.1 |       1.7 |               101.7 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 | 0.783333 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                 1.940693 |                      2.174668 |                          0.0 |                      0.123042 |                           0.135912 |             1.748564 |                  1.795279 |      99.951436 |   1.213101 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.576416 |  0.034051 |                   0.0 |                       0.0 |                  10.795247 |                  1.391158 |                      0.0 |    -0.057965 |              0.0 |          9.404089 |             83.078604 |                 100.0 |                 55.69299 |               87.719298 | 0.764075 |  0.0 |  0.010298 |  0.010298 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.764075 | 0.077261 |      0.0 |  0.0 |      0.0 |  0.0 | 0.077261 | 0.474995 | 0.131943 | 0.689702 |  0.0 |  0.0 |   0.0 |  nan | 32.248458 |  114.65501 | 0.077261 |   0.0 |      0.0 |   0.0 |      0.0 | 0.474995 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.131943 |
    | 1997-08-03 23:00:00 | 0.833333 | 16.4 |              0.0 |             90.9 |       2.3 |               101.7 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 | 0.783333 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 1.868084 |                      2.192855 |                          0.0 |                      0.119006 |                           0.136905 |             1.698088 |                  1.806456 |     100.001912 |   1.215845 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.428192 |  0.043809 |                   0.0 |                       0.0 |                  10.795247 |                   1.38284 |                      0.0 |    -0.057618 |              0.0 |          9.412407 |             61.405925 |                 100.0 |                 54.82581 |               87.719298 | 0.773036 |  0.0 |  0.014661 |  0.014661 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.773036 | 0.078736 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078736 |  0.46474 | 0.129094 | 0.685339 |  0.0 |  0.0 |   0.0 |  nan | 32.194232 | 115.349309 | 0.078736 |   0.0 |      0.0 |   0.0 |      0.0 |  0.46474 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.129094 |
    | 1997-08-04 00:00:00 | 0.833333 | 15.9 |              0.0 |             88.0 |       2.3 |               101.7 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 | 0.783333 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                 1.809403 |                      2.187158 |                          0.0 |                       0.11573 |                           0.136594 |             1.592275 |                  1.795292 |     100.107725 |    1.21843 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.24744 |  0.050705 |                   0.0 |                       0.0 |                  10.739718 |                   1.39654 |                      0.0 |    -0.058189 |              0.0 |          9.343179 |             61.405925 |                 100.0 |                53.950005 |               87.719298 | 0.768673 |  0.0 |  0.023989 |  0.023989 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.768673 | 0.078861 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078861 | 0.454771 | 0.126325 | 0.676011 |  0.0 |  0.0 |   0.0 |  nan | 32.133111 | 116.039121 | 0.078861 |   0.0 |      0.0 |   0.0 |      0.0 | 0.454771 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.126325 |
    | 1997-08-04 01:00:00 | 0.833333 | 15.4 |              0.0 |             89.2 |       2.2 |               101.6 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 | 0.783333 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                 1.752342 |                      2.180337 |                          0.0 |                      0.112529 |                           0.136222 |             1.563089 |                  1.784515 |     100.036911 |   1.219467 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.043702 |  0.057104 |                   0.0 |                       0.0 |                  10.739718 |                  1.406633 |                      0.0 |     -0.05861 |              0.0 |          9.333085 |             64.197103 |                 100.0 |                53.080668 |               87.719298 | 0.759344 |  0.0 |  0.018182 |  0.018182 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.759344 | 0.078462 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078462 | 0.445055 | 0.123627 | 0.681818 |  0.0 |  0.0 |   0.0 |  nan |  32.06559 | 116.720003 | 0.078462 |   0.0 |      0.0 |   0.0 |      0.0 | 0.445055 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.123627 |
    | 1997-08-04 02:00:00 | 0.833333 | 15.2 |              0.0 |             88.8 |       2.3 |               101.6 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 | 0.783333 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                 1.729962 |                      2.173535 |                          0.0 |                      0.111271 |                            0.13585 |             1.536206 |                  1.772518 |     100.063794 |   1.220436 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.818633 |  0.056562 |                   0.0 |                       0.0 |                  10.739718 |                  1.418069 |                      0.0 |    -0.059086 |              0.0 |           9.32165 |             61.405925 |                 100.0 |                52.223358 |               87.719298 | 0.765151 |  0.0 |   0.02055 |   0.02055 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765151 | 0.079625 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079625 | 0.435601 |    0.121 |  0.67945 |  0.0 |  0.0 |   0.0 |  nan | 31.998611 | 117.405529 | 0.079625 |   0.0 |      0.0 |   0.0 |      0.0 | 0.435601 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 |    0.121 |
    | 1997-08-04 03:00:00 | 0.833333 | 14.9 |              0.0 |             89.7 |       2.0 |               101.6 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 | 0.783333 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                 1.696861 |                      2.165623 |                          0.0 |                      0.109405 |                           0.135418 |             1.522084 |                  1.759749 |     100.077916 |   1.221771 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.595369 |   0.05822 |                   0.0 |                       0.0 |                  10.739718 |                   1.43017 |                      0.0 |     -0.05959 |              0.0 |          9.309548 |             70.616813 |                 100.0 |                 51.36096 |               87.719298 | 0.762784 |  0.0 |   0.01363 |   0.01363 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.762784 | 0.079941 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079941 |  0.42641 | 0.118447 |  0.68637 |  0.0 |  0.0 |   0.0 |  nan | 31.929974 | 118.088372 | 0.079941 |   0.0 |      0.0 |   0.0 |      0.0 |  0.42641 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.118447 |
    | 1997-08-04 04:00:00 | 0.833333 | 14.3 |              0.0 |             93.0 |       2.3 |               101.6 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 | 0.783333 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                 1.632319 |                       2.15605 |                          0.0 |                      0.105752 |                           0.134895 |             1.518057 |                  1.748197 |     100.081943 |    1.22434 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.398044 |                     15.502754 |            14.131208 |   0.07485 |        0.006692 |                0.005726 | -0.010417 |  17.36658 |  0.066238 |              0.004352 |                  0.004352 |                  10.739718 |                  1.440702 |                      0.0 |    -0.055678 |              0.0 |          9.299016 |             61.405925 |                 100.0 |                50.502684 |               73.397763 | 0.769703 |  0.0 |  0.007255 |  0.007255 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.769703 | 0.081237 |      0.0 |  0.0 |      0.0 |  0.0 | 0.081237 | 0.417477 | 0.115966 | 0.692745 |  0.0 |  0.0 |   0.0 |  nan | 31.853319 | 118.776838 | 0.081237 |   0.0 |      0.0 |   0.0 |      0.0 | 0.417477 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.115966 |
    | 1997-08-04 05:00:00 | 0.833333 | 14.1 |              0.0 |             93.6 |       2.5 |               101.7 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 | 0.783333 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                 1.611289 |                      2.145953 |                          0.0 |                      0.104558 |                           0.134343 |             1.508166 |                  1.736523 |     100.191834 |   1.226451 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.131208 |  0.938714 |        0.083928 |                0.071823 | -0.010417 | 17.111064 |  0.065039 |              0.054586 |                  0.054586 |                  10.739718 |                  1.451305 |                      0.0 |    -0.005885 |              0.0 |          9.288414 |             56.493451 |                 100.0 |                49.638078 |               58.024051 | 0.776079 |  0.0 |  0.018162 |  0.018162 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.776079 | 0.082492 |      0.0 |  0.0 |      0.0 |  0.0 | 0.082492 | 0.408808 | 0.113558 | 0.681838 |  0.0 |  0.0 |   0.0 |  nan | 31.777864 | 119.470425 | 0.082492 |   0.0 |      0.0 |   0.0 |      0.0 | 0.408808 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.113558 |
    | 1997-08-04 06:00:00 | 0.833333 | 14.3 |              0.1 |             94.6 |       2.4 |               101.7 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 | 0.783333 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                 1.632319 |                      2.136456 |                          0.0 |                      0.105752 |                           0.133824 |             1.544174 |                  1.726078 |     100.155826 |   1.225432 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.246552 |  2.455474 |        0.263445 |                0.226749 | -0.010417 | 16.859546 |  0.055286 |              0.172329 |                  0.172329 |                   10.82738 |                  1.478175 |                      0.0 |     0.110738 |              0.0 |          9.349205 |             58.847345 |                 100.0 |                 48.85611 |               57.260179 | 0.765171 |  0.0 |   0.04093 |   0.04093 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765171 | 0.081905 | 0.003897 |  0.0 | 0.000003 |  0.0 | 0.081905 | 0.400397 | 0.111221 |  0.65907 |  0.0 |  0.0 |   0.0 |  nan | 31.712161 |  120.14979 | 0.081905 |   0.0 | 0.003897 |   0.0 | 0.000003 | 0.400372 |   0.0 | 0.000025 |   0.0 |      0.0 |   0.0 | 0.111221 |
    | 1997-08-04 07:00:00 | 0.833333 | 15.4 |              0.9 |             93.3 |       2.5 |               101.7 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 | 0.783333 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                 1.752342 |                      2.129774 |                          0.0 |                      0.112529 |                           0.133458 |             1.634935 |                  1.717041 |     100.065065 |   1.220347 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            15.284647 |  4.650125 |        1.164115 |                1.028795 | -0.010417 | 16.640536 |  0.026796 |              0.781884 |                  0.781884 |                  11.616331 |                   1.64464 |                      0.0 |     0.713358 |              0.0 |          9.971692 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 | 0.742403 |  0.0 |   0.18972 |   0.18972 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.742403 |  0.08001 | 0.003919 |  0.0 | 0.000031 |  0.0 |  0.08001 | 0.392197 | 0.108943 |  0.51028 |  0.0 |  0.0 |   0.0 |  nan | 31.674949 | 120.808233 |  0.08001 |   0.0 | 0.003919 |   0.0 | 0.000031 | 0.392121 |   0.0 | 0.000076 |   0.0 |      0.0 |   0.0 | 0.108943 |
    | 1997-08-04 08:00:00 | 0.833333 | 16.8 |              1.0 |             86.9 |       2.5 |               101.7 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 | 0.783333 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                  1.91622 |                       2.12644 |                          0.0 |                      0.121684 |                           0.133275 |             1.665196 |                  1.705582 |     100.034804 |   1.214317 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            16.438085 |  7.209034 |        1.933622 |                1.710811 | -0.010417 | 16.516495 | -0.006124 |              1.300216 |                  1.300216 |                  12.492945 |                   1.83472 |                      0.0 |      1.22377 |              0.0 |         10.658225 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 | 0.593614 |  0.0 |  0.343734 |  0.343734 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.593614 | 0.064359 | 0.003941 |  0.0 | 0.000059 |  0.0 | 0.064359 | 0.383982 | 0.106662 | 0.356266 |  0.0 |  0.0 |   0.0 |  nan | 31.670656 | 121.333488 | 0.064359 |   0.0 | 0.003941 |   0.0 | 0.000059 | 0.383856 |   0.0 | 0.000126 |   0.0 |      0.0 |   0.0 | 0.106662 |
    | 1997-08-04 09:00:00 | 0.833333 | 18.2 |              1.0 |             78.9 |       2.9 |               101.7 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 | 0.783333 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                 2.093348 |                      2.122001 |                          0.0 |                      0.131461 |                           0.133032 |             1.651651 |                  1.695656 |     100.048349 |   1.208543 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.360836 |  9.458797 |        2.537058 |                2.210424 | -0.010417 | 16.502185 | -0.036673 |              1.679922 |                  1.679922 |                  13.194235 |                  1.989515 |                      0.0 |     1.597026 |              0.0 |          11.20472 |             48.701251 |                 100.0 |                 48.85611 |               57.260179 | 0.439599 |  0.0 |  0.494183 |  0.494183 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.439599 | 0.047881 | 0.003958 |  0.0 | 0.000081 |  0.0 | 0.047881 | 0.375565 | 0.104324 | 0.205817 |  0.0 |  0.0 |   0.0 |  nan | 31.696912 | 121.721168 | 0.047881 |   0.0 | 0.003958 |   0.0 | 0.000081 | 0.375389 |   0.0 | 0.000176 |   0.0 |      0.0 |   0.0 | 0.104324 |
    | 1997-08-04 10:00:00 | 0.833333 | 19.5 |              0.9 |             75.8 |       3.1 |               101.7 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 | 0.783333 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                 2.270468 |                      2.117571 |                          0.0 |                      0.141127 |                            0.13279 |             1.721014 |                  1.691233 |      99.978986 |   1.202862 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.822211 | 10.747438 |         2.69052 |                2.317703 | -0.010417 | 16.589706 | -0.062862 |              1.761455 |                  1.761455 |                   13.54488 |                   2.06621 |                      0.0 |     1.675363 |              0.0 |          11.47867 |             45.559235 |                 100.0 |                 48.85611 |               57.260179 | 0.289151 |  0.0 |  0.550449 |  0.550449 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.289151 | 0.031596 | 0.003971 |  0.0 | 0.000097 |  0.0 | 0.031596 | 0.366943 | 0.101929 | 0.149551 |  0.0 |  0.0 |   0.0 |  nan | 31.749357 | 121.974655 | 0.031596 |   0.0 | 0.003971 |   0.0 | 0.000097 | 0.366718 |   0.0 | 0.000225 |   0.0 |      0.0 |   0.0 | 0.101929 |
    | 1997-08-04 11:00:00 | 0.833333 | 20.3 |              0.8 |             73.4 |       3.3 |               101.8 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 | 0.783333 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                  2.38585 |                      2.111492 |                          0.0 |                      0.147368 |                           0.132457 |             1.751214 |                  1.687786 |     100.048786 |   1.200635 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.937555 | 10.819272 |        2.515039 |                2.160947 | -0.010417 | 16.764525 | -0.076366 |               1.64232 |                   1.64232 |                  13.632542 |                  2.087634 |                      0.0 |     1.555335 |              0.0 |         11.544908 |             42.798069 |                 100.0 |                 48.85611 |               57.260179 | 0.232884 |  0.0 |  0.548384 |  0.548384 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.232884 | 0.025506 | 0.003979 |  0.0 | 0.000107 |  0.0 | 0.025506 | 0.358256 | 0.099516 | 0.151616 |  0.0 |  0.0 |   0.0 |  nan | 31.815307 | 122.177947 | 0.025506 |   0.0 | 0.003979 |   0.0 | 0.000107 | 0.357982 |   0.0 | 0.000274 |   0.0 |      0.0 |   0.0 | 0.099516 |
    | 1997-08-04 12:00:00 | 0.833333 | 21.0 |              0.9 |             68.3 |       3.3 |               101.8 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 | 0.783333 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                 2.490975 |                      2.109837 |                          0.0 |                      0.153018 |                           0.132366 |             1.701336 |                  1.687166 |     100.098664 |   1.198001 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.052899 | 10.408076 |        2.605564 |                2.233851 | -0.010417 | 16.984357 | -0.086738 |              1.697727 |                  1.697727 |                  13.720203 |                  2.106001 |                      0.0 |     1.609977 |              0.0 |         11.614202 |             42.798069 |                 100.0 |                 48.85611 |               57.260179 |  0.23495 |  0.0 |  0.597559 |  0.597559 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.23495 | 0.025785 | 0.003986 |  0.0 | 0.000116 |  0.0 | 0.025785 |  0.34972 | 0.097144 | 0.102441 |  0.0 |  0.0 |   0.0 |  nan | 31.891628 | 122.383011 | 0.025785 |   0.0 | 0.003986 |   0.0 | 0.000116 | 0.349398 |   0.0 | 0.000322 |   0.0 |      0.0 |   0.0 | 0.097144 |
    | 1997-08-04 13:00:00 | 0.833333 | 21.8 |              0.8 |             61.3 |       3.2 |               101.8 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 | 0.783333 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                 2.616045 |                      2.109837 |                          0.0 |                      0.159698 |                           0.132366 |             1.603636 |                  1.682947 |     100.196364 |   1.195188 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             18.39893 | 10.917404 |         2.53785 |                2.157989 | -0.010417 | 17.238761 | -0.098523 |              1.640072 |                  1.640072 |                  13.983187 |                  2.166369 |                      0.0 |     1.549806 |              0.0 |         11.816818 |             44.135508 |                 100.0 |                 48.85611 |               57.260179 | 0.185775 |  0.0 |  0.621801 |  0.621801 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.185775 | 0.020426 | 0.003992 |  0.0 | 0.000124 |  0.0 | 0.020426 | 0.341339 | 0.094816 | 0.078199 |  0.0 |  0.0 |   0.0 |  nan | 31.979734 | 122.544243 | 0.020426 |   0.0 | 0.003992 |   0.0 | 0.000124 | 0.340969 |   0.0 |  0.00037 |   0.0 |      0.0 |   0.0 | 0.094816 |
    | 1997-08-04 14:00:00 | 0.833333 | 21.9 |              0.9 |             60.6 |       2.5 |               101.8 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 | 0.783333 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                 2.632058 |                      2.104328 |                          0.0 |                       0.16055 |                           0.132064 |             1.595027 |                  1.676097 |     100.204973 |   1.194821 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.975649 | 10.494383 |         2.62717 |                2.208784 | -0.010417 | 17.532448 | -0.094339 |              1.678676 |                  1.678676 |                  14.421494 |                  2.265756 |                      0.0 |      1.58427 |              0.0 |         12.155738 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 | 0.161533 |  0.0 |  0.589183 |  0.589183 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.161533 | 0.017787 | 0.003998 |  0.0 | 0.000131 |  0.0 | 0.017787 | 0.333072 |  0.09252 | 0.110817 |  0.0 |  0.0 |   0.0 |  nan | 32.063657 |  122.68386 | 0.017787 |   0.0 | 0.003998 |   0.0 | 0.000131 | 0.332655 |   0.0 | 0.000417 |   0.0 |      0.0 |   0.0 |  0.09252 |
    | 1997-08-04 15:00:00 | 0.833333 | 21.8 |              0.9 |             58.8 |       2.9 |               101.7 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 | 0.783333 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                 2.616045 |                      2.099381 |                          0.0 |                      0.159698 |                           0.131792 |             1.538235 |                  1.669008 |     100.161765 |   1.194298 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            19.437025 |  8.874215 |        2.221576 |                1.861585 | -0.010417 | 17.812189 | -0.086137 |              1.414805 |                  1.414805 |                  14.772139 |                  2.348437 |                      0.0 |     1.316953 |              0.0 |         12.423701 |             48.701251 |                 100.0 |                 48.85611 |               57.260179 |  0.19415 |  0.0 |  0.551424 |  0.551424 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.19415 | 0.021412 | 0.004002 |  0.0 | 0.000137 |  0.0 | 0.021412 | 0.325032 | 0.090287 | 0.148576 |  0.0 |  0.0 |   0.0 |  nan | 32.139377 |  122.85246 | 0.021412 |   0.0 | 0.004002 |   0.0 | 0.000137 | 0.324569 |   0.0 | 0.000463 |   0.0 |      0.0 |   0.0 | 0.090287 |
    | 1997-08-04 16:00:00 | 0.833333 | 22.2 |              0.9 |             57.2 |       3.6 |               101.7 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 | 0.783333 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                 2.680609 |                      2.097734 |                          0.0 |                      0.163129 |                           0.131702 |             1.533308 |                   1.66429 |     100.166692 |   1.192703 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |              19.8984 |   6.45697 |        1.616442 |                1.360212 | -0.010417 |  18.06459 | -0.089325 |              1.033761 |                  1.033761 |                  15.122784 |                  2.428674 |                      0.0 |     0.932566 |              0.0 |          12.69411 |             39.231563 |                 100.0 |                 48.85611 |               57.260179 | 0.231909 |  0.0 |  0.505761 |  0.505761 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.231909 | 0.025623 | 0.004008 |  0.0 | 0.000144 |  0.0 | 0.025623 | 0.317302 | 0.088139 | 0.194239 |  0.0 |  0.0 |   0.0 |  nan | 32.218285 | 123.054595 | 0.025623 |   0.0 | 0.004008 |   0.0 | 0.000144 | 0.316793 |   0.0 | 0.000509 |   0.0 |      0.0 |   0.0 | 0.088139 |
    | 1997-08-04 17:00:00 | 0.833333 | 21.8 |              1.0 |             57.2 |       4.2 |               101.7 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 | 0.783333 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                 2.616045 |                      2.097185 |                          0.0 |                      0.159698 |                           0.131672 |             1.496378 |                  1.659573 |     100.203622 |   1.194485 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            20.705807 |  3.945159 |        1.058179 |                0.907937 | -0.010417 | 18.327617 | -0.075003 |              0.690032 |                  0.690032 |                  15.736413 |                  2.564427 |                      0.0 |     0.583181 |              0.0 |         13.171986 |             33.627054 |                 100.0 |                 48.85611 |               57.260179 | 0.277572 |  0.0 |  0.444603 |  0.444603 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.277572 | 0.030737 | 0.004014 |  0.0 | 0.000152 |  0.0 | 0.030737 | 0.309893 | 0.086081 | 0.255397 |  0.0 |  0.0 |   0.0 |  nan | 32.282872 | 123.297264 | 0.030737 |   0.0 | 0.004014 |   0.0 | 0.000152 | 0.309338 |   0.0 | 0.000555 |   0.0 |      0.0 |   0.0 | 0.086081 |
    | 1997-08-04 18:00:00 | 0.833333 | 20.9 |              1.0 |             59.9 |       4.2 |               101.7 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 | 0.783333 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 2.475714 |                      2.094444 |                          0.0 |                        0.1522 |                           0.131522 |             1.482953 |                  1.651207 |     100.217047 |   1.198201 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            21.743901 |  1.939642 |        0.520255 |                0.462876 | -0.010417 | 18.542906 | -0.050913 |              0.351786 |                  0.351786 |                  16.525365 |                  2.744116 |                      0.0 |     0.237448 |              0.0 |         13.781249 |             33.627054 |                 100.0 |                 48.85611 |               57.260179 |  0.33873 |  0.0 |   0.32478 |   0.32478 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.33873 |  0.03761 | 0.004022 |  0.0 | 0.000162 |  0.0 |  0.03761 |  0.30283 |  0.08412 |  0.37522 |  0.0 |  0.0 |   0.0 |  nan | 32.323368 |   123.5942 |  0.03761 |   0.0 | 0.004022 |   0.0 | 0.000162 |  0.30223 |   0.0 |   0.0006 |   0.0 |      0.0 |   0.0 |  0.08412 |
    | 1997-08-04 19:00:00 | 0.833333 | 19.7 |              1.0 |             66.1 |       3.6 |               101.7 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 | 0.783333 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                 2.298847 |                       2.08897 |                          0.0 |                      0.142666 |                           0.131221 |             1.519538 |                   1.63958 |     100.180462 |   1.202947 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             22.89734 |   0.60531 |        0.162358 |                0.151461 | -0.010417 | 18.677895 | -0.022077 |              0.115111 |                  0.115111 |                  17.401978 |                  2.950104 |                      0.0 |     -0.00781 |              0.0 |         14.451875 |             39.231563 |                 100.0 |                 48.85611 |               57.260179 | 0.458554 |  0.0 |  0.183091 |  0.183091 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.458554 | 0.051092 | 0.004032 |  0.0 | 0.000175 |  0.0 | 0.051092 | 0.296214 | 0.082282 | 0.516909 |  0.0 |  0.0 |   0.0 |  nan | 32.335029 | 123.997455 | 0.051092 |   0.0 | 0.004032 |   0.0 | 0.000175 |  0.29557 |   0.0 | 0.000644 |   0.0 |      0.0 |   0.0 | 0.082282 |
    | 1997-08-04 20:00:00 | 0.833333 | 17.9 |              0.3 |             71.7 |       3.2 |               101.7 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 | 0.783333 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                 2.054233 |                      2.080783 |                          0.0 |                      0.129312 |                           0.130772 |             1.472885 |                  1.624918 |     100.227115 |   1.210597 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.104711 |                     15.502754 |            23.243371 |  0.005136 |        0.003091 |                0.002926 | -0.010417 | 18.716764 |  0.017642 |              0.002224 |                  0.002224 |                  17.664962 |                  3.034376 |                      0.0 |    -0.124208 |              0.0 |         14.630586 |             44.135508 |                 100.0 |                 48.85611 |               83.091125 | 0.600242 |  0.0 |  0.100631 |  0.100631 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.600242 | 0.067188 | 0.004045 |  0.0 | 0.000192 |  0.0 | 0.067188 | 0.290152 | 0.080598 | 0.599369 |  0.0 |  0.0 |   0.0 |  nan |  32.30697 | 124.526272 | 0.067188 |   0.0 | 0.004045 |   0.0 | 0.000192 | 0.289464 |   0.0 | 0.000689 |   0.0 |      0.0 |   0.0 | 0.080598 |
    | 1997-08-04 21:00:00 | 0.833333 | 15.8 |              0.0 |             80.6 |       2.2 |               101.8 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 | 0.783333 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                 1.797863 |                       2.06991 |                          0.0 |                      0.115083 |                           0.130174 |             1.449077 |                  1.610994 |     100.350923 |    1.22071 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 | 18.623235 |  0.060982 |                   0.0 |                       0.0 |                  17.664962 |                  3.060678 |                      0.0 |    -0.127528 |              0.0 |         14.604285 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 | 0.682702 |  0.0 |  0.029981 |  0.029981 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.682702 | 0.076847 | 0.004062 |  0.0 | 0.000214 |  0.0 | 0.076847 |  0.28458 |  0.07905 | 0.670019 |  0.0 |  0.0 |   0.0 |  nan | 32.235572 | 125.127851 | 0.076847 |   0.0 | 0.004062 |   0.0 | 0.000214 | 0.283847 |   0.0 | 0.000732 |   0.0 |      0.0 |   0.0 |  0.07905 |
    | 1997-08-04 22:00:00 | 0.833333 | 14.8 |              0.0 |             85.9 |       2.2 |               101.8 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 | 0.783333 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                 1.685952 |                      2.058007 |                          0.0 |                      0.108789 |                            0.12952 |             1.448232 |                  1.598128 |     100.351768 |   1.224953 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.38524 |  0.077441 |                   0.0 |                       0.0 |                  17.664962 |                  3.084324 |                      0.0 |    -0.128514 |              0.0 |         14.580638 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 | 0.753353 |  0.0 |  0.012064 |  0.012064 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.753353 | 0.085331 | 0.004082 |  0.0 | 0.000239 |  0.0 | 0.085331 | 0.279386 | 0.077607 | 0.687936 |  0.0 |  0.0 |   0.0 |  nan | 32.147714 | 125.791553 | 0.085331 |   0.0 | 0.004082 |   0.0 | 0.000239 |  0.27861 |   0.0 | 0.000776 |   0.0 |      0.0 |   0.0 | 0.077607 |
    | 1997-08-04 23:00:00 | 0.833333 | 14.1 |              0.0 |             89.3 |       2.1 |               101.8 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 | 0.783333 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                 1.611289 |                      2.045628 |                          0.0 |                      0.104558 |                           0.128839 |             1.438881 |                  1.587151 |     100.361119 |   1.227981 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | -0.010417 |  18.09238 |  0.086235 |                   0.0 |                       0.0 |                  17.664962 |                  3.103639 |                      0.0 |    -0.129318 |              0.0 |         14.561323 |             67.254108 |                 100.0 |                 48.85611 |               87.719298 | 0.771269 |  0.0 | -0.000111 | -0.000111 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.771269 | 0.087938 | 0.004103 |  0.0 | 0.000266 |  0.0 | 0.087938 | 0.274471 | 0.076242 | 0.700111 |  0.0 |  0.0 |   0.0 |  nan | 32.051062 | 126.470515 | 0.087938 |   0.0 | 0.004103 |   0.0 | 0.000266 | 0.273652 |   0.0 | 0.000819 |   0.0 |      0.0 |   0.0 | 0.076242 |
    | 1997-08-05 00:00:00 | 0.833333 | 13.4 |              0.0 |             91.4 |       1.8 |               101.8 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 | 0.783333 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                 1.539538 |                      2.032245 |                          0.0 |                      0.100468 |                           0.128102 |             1.407137 |                  1.579647 |     100.392863 |   1.231127 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.770207 |  0.094396 |                   0.0 |                       0.0 |                   17.58988 |                  3.124611 |                      0.0 |    -0.130192 |              0.0 |         14.465269 |             78.463126 |                 100.0 |                 48.85611 |               87.719298 | 0.783444 |  0.0 | -0.009107 | -0.009107 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.783444 | 0.089926 | 0.004126 |  0.0 | 0.000295 |  0.0 | 0.089926 | 0.269742 | 0.074928 | 0.709107 |  0.0 |  0.0 |   0.0 |  nan | 31.946249 | 127.159612 | 0.089926 |   0.0 | 0.004126 |   0.0 | 0.000295 |  0.26888 |   0.0 | 0.000862 |   0.0 |      0.0 |   0.0 | 0.074928 |
    | 1997-08-05 01:00:00 | 0.833333 | 12.8 |              0.0 |             93.1 |       1.8 |               101.8 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 | 0.783333 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                 1.480281 |                      2.018409 |                          0.0 |                      0.097071 |                           0.127339 |             1.378142 |                  1.572172 |     100.421858 |   1.233843 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 |  17.42083 |   0.09981 |                   0.0 |                       0.0 |                   17.58988 |                  3.135507 |                      0.0 |    -0.130646 |              0.0 |         14.454373 |             78.463126 |                 100.0 |                 48.85611 |               87.719298 | 0.792441 |  0.0 | -0.013348 | -0.013348 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792441 | 0.091575 | 0.004148 |  0.0 | 0.000323 |  0.0 | 0.091575 | 0.265184 | 0.073662 | 0.713348 |  0.0 |  0.0 |   0.0 |  nan | 31.836022 | 127.856006 | 0.091575 |   0.0 | 0.004148 |   0.0 | 0.000323 | 0.264279 |   0.0 | 0.000904 |   0.0 |      0.0 |   0.0 | 0.073662 |
    | 1997-08-05 02:00:00 | 0.833333 | 12.3 |              0.0 |             94.6 |       1.6 |               101.8 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 | 0.783333 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                 1.432438 |                      2.003073 |                          0.0 |                      0.094315 |                           0.126493 |             1.355086 |                  1.565068 |     100.444914 |   1.236111 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 | 17.053408 |  0.102674 |                   0.0 |                       0.0 |                   17.58988 |                  3.144898 |                      0.0 |    -0.131037 |              0.0 |         14.444982 |             88.271017 |                 100.0 |                 48.85611 |               87.719298 | 0.796681 |  0.0 | -0.017799 | -0.017799 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.796681 | 0.092692 | 0.004171 |  0.0 | 0.000352 |  0.0 | 0.092692 |  0.26078 | 0.072439 | 0.717799 |  0.0 |  0.0 |   0.0 |  nan | 31.722932 | 128.555473 | 0.092692 |   0.0 | 0.004171 |   0.0 | 0.000352 | 0.259832 |   0.0 | 0.000947 |   0.0 |      0.0 |   0.0 | 0.072439 |
    | 1997-08-05 03:00:00 | 0.833333 | 12.0 |              0.0 |             95.6 |       1.2 |               101.8 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 | 0.783333 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 1.404387 |                       1.98784 |                          0.0 |                      0.092693 |                           0.125652 |             1.342594 |                  1.558052 |     100.457406 |   1.237469 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | -0.010417 |  16.67644 |  0.101011 |                   0.0 |                       0.0 |                   17.58988 |                   3.15411 |                      0.0 |    -0.131421 |              0.0 |          14.43577 |            117.694689 |                 100.0 |                 48.85611 |               87.719298 | 0.801132 |  0.0 | -0.020794 | -0.020794 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.801132 | 0.093844 | 0.004194 |  0.0 | 0.000381 |  0.0 | 0.093844 | 0.256519 | 0.071255 | 0.720794 |  0.0 |  0.0 |   0.0 |  nan | 31.611504 | 129.258186 | 0.093844 |   0.0 | 0.004194 |   0.0 | 0.000381 |  0.25553 |   0.0 | 0.000989 |   0.0 | 0.000001 |   0.0 | 0.071255 |
    | 1997-08-05 04:00:00 | 0.833333 | 11.7 |              0.0 |             96.2 |       1.9 |               101.8 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 | 0.783333 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                 1.376819 |                      1.974268 |                          0.0 |                      0.091096 |                           0.124901 |               1.3245 |                  1.550047 |       100.4755 |   1.238856 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.366022 |                     15.438712 |            23.144579 |  0.063733 |         0.00566 |                0.005336 | -0.010417 | 16.305014 |  0.099468 |              0.004056 |                  0.004056 |                   17.58988 |                  3.166293 |                      0.0 |    -0.127873 |              0.0 |         14.423587 |             74.333488 |                 100.0 |                 48.85611 |               73.423526 | 0.804127 |  0.0 | -0.019617 | -0.019617 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.804127 | 0.094836 | 0.004217 |  0.0 | 0.000411 |  0.0 | 0.094836 | 0.252397 |  0.07011 | 0.719617 |  0.0 |  0.0 |   0.0 |  nan | 31.501619 |  129.96285 | 0.094836 |   0.0 | 0.004217 |   0.0 | 0.000411 | 0.251366 |   0.0 |  0.00103 |   0.0 | 0.000001 |   0.0 |  0.07011 |
    | 1997-08-05 05:00:00 | 0.833333 | 11.6 |              0.0 |             95.7 |       1.3 |               101.8 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 | 0.783333 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 1.367736 |                      1.961295 |                          0.0 |                      0.090568 |                           0.124183 |             1.308923 |                  1.541578 |     100.491077 |   1.239363 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.144579 |  0.908598 |        0.080684 |                0.076088 | -0.010417 | 15.938731 |  0.093717 |              0.057827 |                  0.057827 |                   17.58988 |                  3.179808 |                      0.0 |    -0.074665 |              0.0 |         14.410072 |            108.641252 |                 100.0 |                 48.85611 |               57.260179 |  0.80295 |  0.0 |   -0.0093 |   -0.0093 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.80295 | 0.095339 |  0.00424 |  0.0 |  0.00044 |  0.0 | 0.095339 | 0.248402 | 0.069001 |   0.7093 |  0.0 |  0.0 |   0.0 |  nan | 31.397486 | 130.665781 | 0.095339 |   0.0 |  0.00424 |   0.0 |  0.00044 |  0.24733 |   0.0 | 0.001072 |   0.0 | 0.000001 |   0.0 | 0.069001 |
    | 1997-08-05 06:00:00 | 0.833333 | 12.6 |              0.6 |             95.1 |       1.6 |               101.8 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 | 0.783333 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                 1.460978 |                      1.952516 |                          0.0 |                       0.09596 |                           0.123697 |              1.38939 |                  1.535084 |      100.41061 |   1.234655 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  2.420659 |        0.472904 |                 0.45318 | -0.010417 |  15.59162 |  0.064619 |              0.344417 |                  0.344417 |                  18.027018 |                  3.288825 |                      0.0 |     0.207383 |              0.0 |         14.738193 |             88.271017 |                 100.0 |                 48.85611 |               57.260179 | 0.792634 |  0.0 |   0.05232 |   0.05232 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792634 | 0.094745 | 0.004262 |  0.0 | 0.000469 |  0.0 | 0.094745 |  0.24451 | 0.067919 |  0.64768 |  0.0 |  0.0 |   0.0 |  nan |  31.32245 | 131.358937 | 0.094745 |   0.0 | 0.004262 |   0.0 | 0.000469 | 0.243396 |   0.0 | 0.001113 |   0.0 | 0.000001 |   0.0 | 0.067919 |
    | 1997-08-05 07:00:00 | 0.833333 | 14.8 |              0.9 |             96.3 |       1.9 |               101.9 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 | 0.783333 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                 1.685952 |                      1.949426 |                          0.0 |                      0.108789 |                           0.123526 |             1.623571 |                  1.535091 |     100.276429 |   1.225361 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  4.619722 |        1.148656 |                1.101438 | -0.010417 | 15.341501 |  0.011696 |              0.837093 |                  0.837093 |                  18.027018 |                  3.287363 |                      0.0 |     0.700119 |              0.0 |         14.739656 |             74.333488 |                 100.0 |                 48.85611 |               57.260179 | 0.731013 |  0.0 |  0.166735 |  0.166735 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.731013 | 0.087936 | 0.004285 |  0.0 | 0.000498 |  0.0 | 0.087936 | 0.240623 |  0.06684 | 0.533265 |  0.0 |  0.0 |   0.0 |  nan | 31.300337 | 131.997231 | 0.087936 |   0.0 | 0.004285 |   0.0 | 0.000498 | 0.239468 |   0.0 | 0.001154 |   0.0 | 0.000001 |   0.0 |  0.06684 |
    | 1997-08-05 08:00:00 | 0.833333 | 17.1 |              1.0 |             89.9 |       3.2 |               101.9 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 | 0.783333 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                 1.953031 |                       1.95097 |                          0.0 |                      0.123726 |                           0.123612 |             1.755775 |                  1.538746 |     100.144225 |   1.215051 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  7.198198 |        1.917614 |                1.839976 | -0.010417 | 15.267791 | -0.039576 |              1.398382 |                  1.398382 |                  18.027018 |                  3.279458 |                      0.0 |     1.261738 |              0.0 |         14.747561 |             44.135508 |                 100.0 |                 48.85611 |               57.260179 | 0.616599 |  0.0 |  0.358814 |  0.358814 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.616599 | 0.074588 | 0.004306 |  0.0 | 0.000525 |  0.0 | 0.074588 | 0.236577 | 0.065716 | 0.341186 |  0.0 |  0.0 |   0.0 |  nan | 31.329496 | 132.534411 | 0.074588 |   0.0 | 0.004306 |   0.0 | 0.000525 |  0.23538 |   0.0 | 0.001195 |   0.0 | 0.000001 |   0.0 | 0.065716 |
    | 1997-08-05 09:00:00 | 0.833333 | 19.4 |              0.9 |             79.2 |       4.0 |               101.9 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 | 0.783333 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                 2.256393 |                      1.957159 |                          0.0 |                      0.140363 |                           0.123954 |             1.787063 |                  1.543872 |     100.112937 |   1.205358 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.604725 |  9.475573 |        2.356024 |                2.266269 | -0.010417 | 15.364988 | -0.087156 |              1.722365 |                  1.722365 |                  17.939591 |                  3.250753 |                      0.0 |     1.586917 |              0.0 |         14.688838 |             35.308407 |                 100.0 |                 48.85611 |               57.260179 | 0.424519 |  0.0 |  0.541756 |  0.541756 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.424519 | 0.051574 | 0.004323 |  0.0 | 0.000547 |  0.0 | 0.051574 | 0.232165 |  0.06449 | 0.158244 |  0.0 |  0.0 |   0.0 |  nan | 31.406236 | 132.902486 | 0.051574 |   0.0 | 0.004323 |   0.0 | 0.000547 | 0.230928 |   0.0 | 0.001236 |   0.0 | 0.000001 |   0.0 |  0.06449 |
    | 1997-08-05 10:00:00 | 0.833333 | 20.6 |              1.0 |             73.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 | 0.783333 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                  2.43042 |                      1.962848 |                          0.0 |                      0.149767 |                           0.124269 |             1.783928 |                  1.546397 |     100.116072 |   1.200448 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 | 10.786309 |        2.873495 |                 2.75694 | -0.010417 | 15.620786 | -0.107551 |              2.095274 |                  2.095274 |                  18.027018 |                  3.266966 |                      0.0 |     1.959151 |              0.0 |         14.760052 |             30.702962 |                 100.0 |                 48.85611 |               57.260179 | 0.241578 |  0.0 |  0.716789 |       0.7 | 0.010363 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.241578 | 0.029429 | 0.004335 |  0.0 | 0.000563 |  0.0 | 0.029429 | 0.227285 | 0.063135 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  31.50337 | 133.099374 | 0.029429 |   0.0 | 0.004335 |   0.0 | 0.000563 | 0.226008 |   0.0 | 0.001276 |   0.0 | 0.000001 |   0.0 | 0.063135 |
    | 1997-08-05 11:00:00 | 0.833333 | 21.4 |              1.0 |             67.3 |       4.1 |               101.9 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 | 0.783333 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                 2.552842 |                      1.968551 |                          0.0 |                      0.156328 |                           0.124585 |             1.718063 |                  1.545886 |     100.181937 |   1.197482 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.949834 | 10.865527 |        2.894599 |                2.761725 | -0.010417 | 15.944568 | -0.117837 |              2.098911 |                  2.098911 |                  18.201874 |                  3.309837 |                      0.0 |     1.961001 |              0.0 |         14.892037 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.746035 |       0.7 | 0.029836 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.221998 | 0.061666 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.610791 | 133.137795 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.220681 |   0.0 | 0.001316 |   0.0 | 0.000001 |   0.0 | 0.061666 |
    | 1997-08-05 12:00:00 | 0.833333 | 22.5 |              1.0 |             62.2 |       4.1 |               101.9 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 | 0.783333 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                 2.729939 |                      1.976351 |                          0.0 |                      0.165743 |                           0.125016 |             1.698022 |                  1.546988 |     100.201978 |   1.193116 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |             24.06487 |  10.45226 |        2.784503 |                 2.65055 | -0.010417 | 16.302636 | -0.133863 |              2.014418 |                  2.014418 |                  18.289301 |                  3.330376 |                      0.0 |     1.875652 |              0.0 |         14.958925 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.771401 |       0.7 | 0.046898 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010167 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010167 | 0.216599 | 0.060166 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.734237 | 133.159147 | 0.010167 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.215243 |   0.0 | 0.001355 |   0.0 | 0.000001 |   0.0 | 0.060166 |
    | 1997-08-05 13:00:00 | 0.833333 | 23.1 |              1.0 |             59.6 |       4.4 |               101.9 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 | 0.783333 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                 2.830978 |                      1.983133 |                          0.0 |                      0.171077 |                           0.125391 |             1.687263 |                  1.550892 |     100.212737 |   1.190747 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.294943 |  10.96314 |        2.920603 |                2.764822 | -0.010417 | 16.714124 | -0.137935 |              2.101265 |                  2.101265 |                  18.464157 |                  3.363175 |                      0.0 |     1.961133 |              0.0 |         15.100981 |             32.098552 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.840981 |       0.7 | 0.091225 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010169 | 0.004344 |  0.0 | 0.000573 |  0.0 | 0.010169 | 0.211341 | 0.058706 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.861756 | 133.136169 | 0.010169 |   0.0 | 0.004344 |   0.0 | 0.000573 | 0.209946 |   0.0 | 0.001394 |   0.0 | 0.000001 |   0.0 | 0.058706 |
    | 1997-08-05 14:00:00 | 0.833333 | 23.4 |              1.0 |             55.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 | 0.783333 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                 2.882708 |                      1.990983 |                          0.0 |                      0.173797 |                           0.125825 |              1.59702 |                  1.552718 |      100.30298 |   1.189943 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.409979 | 10.528069 |        2.804699 |                2.649346 | -0.010417 | 17.139185 | -0.135234 |              2.013503 |                  2.013503 |                  18.551584 |                   3.38194 |                      0.0 |     1.872589 |              0.0 |         15.169644 |             30.702962 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.865172 |       0.7 | 0.105748 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010167 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010167 | 0.206219 | 0.057283 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 31.986572 | 133.098671 | 0.010167 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.204785 |   0.0 | 0.001432 |   0.0 | 0.000001 |   0.0 | 0.057283 |
    | 1997-08-05 15:00:00 | 0.833333 | 23.8 |              1.0 |             52.1 |       4.5 |               101.9 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 | 0.783333 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                 2.952961 |                      2.001493 |                          0.0 |                      0.177481 |                           0.126406 |             1.538493 |                  1.555327 |     100.361507 |     1.1886 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.525016 |  8.882849 |        2.366409 |                 2.23335 | -0.010417 | 17.555242 | -0.134887 |              1.697346 |                  1.697346 |                  18.639012 |                  3.400029 |                      0.0 |     1.555678 |              0.0 |         15.238983 |              31.38525 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.798412 |       0.7 | 0.063779 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 |  0.20123 | 0.055897 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.111043 | 133.103149 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.199758 |   0.0 |  0.00147 |   0.0 | 0.000001 |   0.0 | 0.055897 |
    | 1997-08-05 16:00:00 | 0.833333 | 24.0 |              1.0 |             49.5 |       4.7 |               101.8 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 | 0.783333 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                 2.988644 |                      2.010993 |                          0.0 |                      0.179348 |                            0.12693 |             1.479379 |                  1.556257 |     100.320621 |   1.186889 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  6.439028 |         1.71537 |                1.620343 | -0.010417 | 17.970142 | -0.130245 |              1.231461 |                  1.231461 |                   18.72644 |                   3.42173 |                      0.0 |     1.088889 |              0.0 |         15.304709 |             30.049708 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.702711 |       0.7 | 0.001736 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 |  0.19637 | 0.054547 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 32.230871 | 133.169669 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.194861 |   0.0 | 0.001508 |   0.0 | 0.000002 |   0.0 | 0.054547 |
    | 1997-08-05 17:00:00 | 0.833333 | 24.0 |              1.0 |             48.1 |       4.5 |               101.8 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 | 0.783333 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                 2.988644 |                      2.022657 |                          0.0 |                      0.179348 |                           0.127573 |             1.437538 |                  1.557615 |     100.362462 |   1.187075 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  3.911896 |        1.042137 |                0.985011 | -0.010417 |  18.36957 | -0.121617 |              0.748608 |                  0.748608 |                   18.72644 |                  3.423912 |                      0.0 |     0.605945 |              0.0 |         15.302528 |              31.38525 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |   0.55931 |   0.55931 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 |  0.01017 | 0.004344 |  0.0 | 0.000574 |  0.0 |  0.01017 | 0.191636 | 0.053232 |  0.14069 |  0.0 |  0.0 |   0.0 |  nan | 32.342072 | 133.237914 |  0.01017 |   0.0 | 0.004344 |   0.0 | 0.000574 |  0.19009 |   0.0 | 0.001544 |   0.0 | 0.000002 |   0.0 | 0.053232 |
    | 1997-08-05 18:00:00 | 0.833333 | 23.7 |              1.0 |             49.4 |       4.1 |               101.8 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 | 0.783333 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                  2.93526 |                      2.037589 |                          0.0 |                      0.176554 |                           0.128396 |             1.450018 |                  1.560199 |     100.349982 |   1.188219 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  1.905864 |        0.507726 |                0.480125 | -0.010417 | 18.740238 | -0.107131 |              0.364895 |                  0.364895 |                   18.72644 |                   3.42458 |                      0.0 |     0.222204 |              0.0 |          15.30186 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.224023 |  0.0 |  0.406398 |  0.406398 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.224023 | 0.027375 | 0.004346 |  0.0 | 0.000577 |  0.0 | 0.027375 | 0.187248 | 0.052013 | 0.293602 |  0.0 |  0.0 |   0.0 |  nan | 32.438786 |  133.42964 | 0.027375 |   0.0 | 0.004346 |   0.0 | 0.000577 | 0.185665 |   0.0 | 0.001581 |   0.0 | 0.000002 |   0.0 | 0.052013 |
    | 1997-08-05 19:00:00 | 0.833333 | 22.8 |              1.0 |             57.0 |       3.8 |               101.8 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 | 0.783333 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                 2.780058 |                      2.054233 |                          0.0 |                      0.168392 |                           0.129312 |             1.584633 |                  1.565154 |     100.215367 |   1.191234 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  0.576079 |        0.153469 |                0.145175 | -0.010417 | 19.062619 | -0.080727 |              0.110333 |                  0.110333 |                   18.72644 |                  3.420162 |                      0.0 |    -0.032173 |              0.0 |         15.306277 |             37.166744 |                 100.0 |                 48.85611 |               57.260179 | 0.376935 |  0.0 |  0.257469 |  0.257469 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.376935 | 0.046172 | 0.004353 |  0.0 | 0.000585 |  0.0 | 0.046172 | 0.183439 | 0.050955 | 0.442531 |  0.0 |  0.0 |   0.0 |  nan | 32.509096 | 133.755465 | 0.046172 |   0.0 | 0.004353 |   0.0 | 0.000585 | 0.181821 |   0.0 | 0.001617 |   0.0 | 0.000002 |   0.0 | 0.050955 |
    | 1997-08-05 20:00:00 | 0.833333 | 21.3 |              0.5 |             65.9 |       2.5 |               101.8 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 | 0.783333 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 2.537251 |                      2.072623 |                          0.0 |                      0.155495 |                           0.130323 |             1.672048 |                  1.574157 |     100.127952 |   1.196911 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.072689 |                     15.438712 |            24.870125 |  0.002495 |        0.003269 |                0.003122 | -0.010417 | 19.296988 | -0.043265 |              0.002372 |                  0.002372 |                  18.901295 |                  3.445356 |                      0.0 |    -0.141184 |              0.0 |         15.455939 |             56.493451 |                 100.0 |                 48.85611 |               84.453771 | 0.525865 |  0.0 |  0.102212 |  0.102212 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.525865 | 0.064653 | 0.004363 |  0.0 | 0.000598 |  0.0 | 0.064653 | 0.180211 | 0.050059 | 0.597788 |  0.0 |  0.0 |   0.0 |  nan | 32.541945 | 134.211715 | 0.064653 |   0.0 | 0.004363 |   0.0 | 0.000598 | 0.178556 |   0.0 | 0.001652 |   0.0 | 0.000002 |   0.0 | 0.050059 |
    | 1997-08-05 21:00:00 | 0.833333 | 19.3 |              0.0 |             73.0 |       1.9 |               101.8 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 | 0.783333 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                 2.242395 |                      2.091705 |                          0.0 |                      0.139602 |                           0.131371 |             1.636948 |                  1.582026 |     100.163052 |   1.205255 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.406483 |    0.0023 |                   0.0 |                       0.0 |                  18.901295 |                  3.434698 |                      0.0 |    -0.143112 |              0.0 |         15.466597 |             74.333488 |                 100.0 |                 48.85611 |               87.719298 | 0.681121 |  0.0 |  0.046259 |  0.046259 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.681121 | 0.084155 | 0.004378 |  0.0 | 0.000617 |  0.0 | 0.084155 | 0.177557 | 0.049321 | 0.653741 |  0.0 |  0.0 |   0.0 |  nan | 32.529228 | 134.803686 | 0.084155 |   0.0 | 0.004378 |   0.0 | 0.000617 | 0.175867 |   0.0 | 0.001688 |   0.0 | 0.000002 |   0.0 | 0.049321 |
    | 1997-08-05 22:00:00 | 0.833333 | 18.3 |              0.0 |             76.7 |       2.2 |               101.8 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 | 0.783333 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                  2.10653 |                       2.11094 |                          0.0 |                      0.132185 |                           0.132426 |             1.615709 |                  1.588483 |     100.184291 |   1.209486 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.364094 |  0.022984 |                   0.0 |                       0.0 |                  18.901295 |                  3.427556 |                      0.0 |    -0.142815 |              0.0 |         15.473739 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 | 0.737074 |  0.0 |  0.042514 |  0.042514 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.737074 | 0.091595 | 0.004397 |  0.0 | 0.000642 |  0.0 | 0.091595 |  0.17532 |   0.0487 | 0.657486 |  0.0 |  0.0 |   0.0 |  nan | 32.495827 | 135.444126 | 0.091595 |   0.0 | 0.004397 |   0.0 | 0.000642 | 0.173595 |   0.0 | 0.001723 |   0.0 | 0.000002 |   0.0 |   0.0487 |
    | 1997-08-05 23:00:00 | 0.833333 | 17.4 |              0.0 |             81.8 |       2.7 |               101.8 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 | 0.783333 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                 1.990459 |                      2.129218 |                          0.0 |                      0.125796 |                           0.133428 |             1.628195 |                  1.595583 |     100.171805 |   1.213176 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | -0.010417 | 19.252757 |   0.04002 |                   0.0 |                       0.0 |                  18.901295 |                  3.418333 |                      0.0 |    -0.142431 |              0.0 |         15.482962 |             52.308751 |                 100.0 |                 48.85611 |               87.719298 | 0.740819 |  0.0 |  0.035583 |  0.035583 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.740819 | 0.092614 | 0.004418 |  0.0 | 0.000669 |  0.0 | 0.092614 |  0.17325 | 0.048125 | 0.664417 |  0.0 |  0.0 |   0.0 |  nan | 32.445391 | 136.087245 | 0.092614 |   0.0 | 0.004418 |   0.0 | 0.000669 |  0.17149 |   0.0 | 0.001758 |   0.0 | 0.000002 |   0.0 | 0.048125 |

.. _lland_v3_water_hourly:

water
-----

For our hourly integration test on water type |WASSER|, we deviate from the
input data of the daily simulation (:ref:`lland_v3_wasser_daily`) in setting
precipitation to the constant value of 0.05 mm/h.  Neither the actual (|EvI|)
or the potential evaporation (|EvPo|) does show a diurnal pattern due to
method |Return_Penman_V1| relying on daily aggregated input data (e.g.
|DailyNetRadiation|) only (the same holds for water types |FLUSS| and |SEE|):

.. integration-test::

    >>> lnk(WASSER)
    >>> inputs.nied.series = 0.05
    >>> test("lland_v3_water_hourly",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah),
    ...      use_conditions=conditions_wasser)
    |                date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration |  nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |        sp | globalradiation | adjustedglobalradiation |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.05 | 16.9 |              0.0 |             95.1 |       0.8 |               101.5 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 | 0.047 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                 1.928423 |                      1.975309 |                          0.0 |                      0.122362 |                           0.124959 |              1.83393 |                  1.696461 |       99.66607 |    1.21073 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   3.151127 |                  0.901196 |                      0.0 |     -0.03755 |              0.0 |          2.249931 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.036205 | 0.036205 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.010795 | 0.002999 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002999 |
    | 1997-08-03 01:00:00 | 0.05 | 16.6 |              0.0 |             94.9 |       0.8 |               101.5 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 | 0.047 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                 1.892018 |                      1.970628 |                          0.0 |                      0.120339 |                             0.1247 |             1.795525 |                  1.694576 |      99.704475 |   1.212158 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   3.151127 |                  0.901844 |                      0.0 |    -0.037577 |              0.0 |          2.249283 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.036073 | 0.036073 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.010927 | 0.003035 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003035 |
    | 1997-08-03 02:00:00 | 0.05 | 16.4 |              0.0 |             95.9 |       0.8 |               101.5 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 | 0.047 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 1.868084 |                       1.96233 |                          0.0 |                      0.119006 |                           0.124241 |             1.791493 |                  1.692183 |      99.708507 |   1.213014 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   3.151127 |                  0.902355 |                      0.0 |    -0.037598 |              0.0 |          2.248772 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.035724 | 0.035724 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011276 | 0.003132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003132 |
    | 1997-08-03 03:00:00 | 0.05 | 16.3 |              0.0 |             96.7 |       0.8 |               101.5 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 | 0.047 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                 1.856216 |                      1.948396 |                          0.0 |                      0.118345 |                           0.123469 |             1.794961 |                  1.689422 |      99.705039 |   1.213417 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   3.151127 |                  0.902364 |                      0.0 |    -0.037598 |              0.0 |          2.248763 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.035169 | 0.035169 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011831 | 0.003286 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003286 |
    | 1997-08-03 04:00:00 | 0.05 | 16.0 |              0.0 |             97.2 |       0.8 |               101.5 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 | 0.047 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                 1.821009 |                      1.929443 |                          0.0 |                      0.116379 |                           0.122418 |              1.77002 |                   1.68569 |       99.72998 |    1.21479 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                 0.429734 |                     15.566134 |            10.503757 |  0.086656 |        0.007799 |                0.006997 | 0.0 | nan | 0.0 |              0.002099 |                  0.002099 |                   3.151127 |                  0.902341 |                      0.0 |    -0.035498 |              0.0 |          2.248786 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.034397 | 0.034397 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.012603 | 0.003501 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003501 |
    | 1997-08-03 05:00:00 | 0.05 | 15.9 |              0.0 |             97.5 |       0.6 |               101.5 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 | 0.047 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                 1.809403 |                      1.907618 |                          0.0 |                       0.11573 |                           0.121206 |             1.764168 |                  1.682599 |      99.735832 |   1.215237 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  0.968392 |        0.087159 |                0.078957 | 0.0 | nan | 0.0 |              0.023687 |                  0.023687 |                   3.151127 |                  0.901482 |                      0.0 |    -0.013875 |              0.0 |          2.249645 |            355.009444 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.033483 | 0.033483 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.013517 | 0.003755 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003755 |
    | 1997-08-03 06:00:00 | 0.05 | 16.0 |              0.0 |             97.7 |       0.9 |               101.5 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 | 0.047 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                 1.821009 |                      1.884011 |                          0.0 |                      0.116379 |                           0.119893 |             1.779125 |                   1.68038 |      99.720875 |   1.214748 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |    2.4897 |        0.224084 |                0.206122 | 0.0 | nan | 0.0 |              0.061837 |                  0.061837 |                   3.151127 |                   0.89977 |                      0.0 |     0.024346 |              0.0 |          2.251357 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.032456 | 0.032456 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.014544 |  0.00404 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00404 |
    | 1997-08-03 07:00:00 | 0.05 | 16.6 |              0.0 |             97.4 |       0.9 |               101.5 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 | 0.047 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                 1.892018 |                      1.865111 |                          0.0 |                      0.120339 |                           0.118841 |             1.842826 |                  1.680931 |      99.657174 |   1.211943 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  4.679733 |        0.421196 |                0.393598 | 0.0 | nan | 0.0 |              0.118079 |                  0.118079 |                   3.151127 |                  0.896826 |                      0.0 |     0.080712 |              0.0 |          2.254301 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.031557 | 0.031557 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.015443 |  0.00429 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00429 |
    | 1997-08-03 08:00:00 | 0.05 | 17.4 |              0.0 |             96.8 |       0.9 |               101.6 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 | 0.047 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                 1.990459 |                      1.846377 |                          0.0 |                      0.125796 |                           0.117796 |             1.926764 |                  1.685281 |      99.673236 |   1.209425 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.503757 |  7.219274 |        0.649765 |                0.615417 | 0.0 | nan | 0.0 |              0.184625 |                  0.184625 |                   3.151127 |                  0.891371 |                      0.0 |     0.147485 |              0.0 |          2.259756 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.030534 | 0.030534 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.016466 | 0.004574 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004574 |
    | 1997-08-03 09:00:00 | 0.05 | 19.0 |              0.2 |             86.1 |       1.3 |               101.6 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 | 0.047 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                 2.200854 |                      1.834142 |                          0.0 |                      0.137341 |                           0.117113 |             1.894936 |                  1.687258 |      99.705064 |   1.202945 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             10.73504 |  9.442017 |         1.18975 |                1.122345 | 0.0 | nan | 0.0 |              0.336704 |                  0.336704 |                   3.220512 |                  0.922604 |                      0.0 |     0.298262 |              0.0 |          2.297908 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.030275 | 0.030275 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.016725 | 0.004646 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004646 |
    | 1997-08-03 10:00:00 | 0.05 | 20.3 |              0.5 |             76.8 |       1.5 |               101.6 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 | 0.047 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                  2.38585 |                      1.830729 |                          0.0 |                      0.147368 |                           0.116922 |             1.832333 |                  1.688695 |      99.767667 |   1.197896 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            10.966322 | 10.709164 |         1.92774 |                  1.8057 | 0.0 | nan | 0.0 |               0.54171 |                   0.54171 |                   3.289897 |                  0.955299 |                      0.0 |     0.501906 |              0.0 |          2.334598 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.030379 | 0.030379 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.016621 | 0.004617 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004617 |
    | 1997-08-03 11:00:00 | 0.05 | 21.4 |              0.7 |             71.8 |       1.2 |               101.6 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 | 0.047 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                 2.552842 |                      1.835607 |                          0.0 |                      0.156328 |                           0.117195 |             1.832941 |                   1.69067 |      99.767059 |    1.19342 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            11.544527 | 10.773868 |        2.327265 |                2.214067 | 0.0 | nan | 0.0 |               0.66422 |                   0.66422 |                   3.463358 |                  1.039981 |                      0.0 |     0.620888 |              0.0 |          2.423377 |            177.504722 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.031338 | 0.031338 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.015662 | 0.004351 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004351 |
    | 1997-08-03 12:00:00 | 0.05 | 21.3 |              0.8 |             67.5 |       1.3 |               101.6 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 | 0.047 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 2.537251 |                      1.844905 |                          0.0 |                      0.155495 |                           0.117714 |             1.712644 |                  1.692316 |      99.887356 |   1.194363 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.238374 | 10.364714 |        2.425457 |                 2.25556 | 0.0 | nan | 0.0 |              0.676668 |                  0.676668 |                   3.671512 |                  1.142661 |                      0.0 |     0.629057 |              0.0 |          2.528851 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.032665 | 0.032665 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.014335 | 0.003982 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003982 |
    | 1997-08-03 13:00:00 | 0.05 | 21.8 |              0.5 |             66.1 |       1.5 |               101.6 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 | 0.047 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                 2.616045 |                      1.859671 |                          0.0 |                      0.159698 |                           0.118538 |             1.729206 |                  1.694238 |      99.870794 |   1.192265 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            12.700938 | 10.872489 |         1.95714 |                 1.78608 | 0.0 | nan | 0.0 |              0.535824 |                  0.535824 |                   3.810281 |                  1.211985 |                      0.0 |     0.485325 |              0.0 |          2.598296 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.033843 | 0.033843 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.013157 | 0.003655 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003655 |
    | 1997-08-03 14:00:00 | 0.05 | 22.9 |              0.4 |             63.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 | 0.047 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                 2.796942 |                       1.88551 |                          0.0 |                      0.169283 |                           0.119977 |             1.773261 |                  1.698059 |      99.826739 |   1.187639 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |             12.93222 | 10.461143 |        1.694785 |                1.510276 | 0.0 | nan | 0.0 |              0.453083 |                  0.453083 |                   3.879666 |                   1.24753 |                      0.0 |     0.401102 |              0.0 |          2.632136 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.034948 | 0.034948 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.012052 | 0.003348 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003348 |
    | 1997-08-03 15:00:00 | 0.05 | 22.7 |              0.5 |             62.4 |       1.9 |               101.6 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 | 0.047 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                 2.763263 |                      1.918249 |                          0.0 |                      0.167505 |                           0.121797 |             1.724276 |                  1.703725 |      99.875724 |    1.18866 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.394785 |  8.865382 |        1.595844 |                1.396394 | 0.0 | nan | 0.0 |              0.418918 |                  0.418918 |                   4.018435 |                  1.316893 |                      0.0 |     0.364048 |              0.0 |          2.701542 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.036637 | 0.036637 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.010363 | 0.002879 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002879 |
    | 1997-08-03 16:00:00 | 0.05 | 22.5 |              0.5 |             61.1 |       2.3 |               101.6 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 | 0.047 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                 2.729939 |                      1.957676 |                          0.0 |                      0.165743 |                           0.123983 |             1.667993 |                  1.709704 |      99.932007 |   1.189715 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            13.741708 |  6.474205 |        1.165412 |                1.002589 | 0.0 | nan | 0.0 |              0.300777 |                  0.300777 |                   4.122512 |                  1.370145 |                      0.0 |     0.243687 |              0.0 |          2.752367 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.038445 | 0.038445 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.008555 | 0.002376 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002376 |
    | 1997-08-03 17:00:00 | 0.05 | 21.9 |              0.3 |             62.1 |       2.4 |               101.6 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 | 0.047 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                 2.632058 |                      2.001493 |                          0.0 |                       0.16055 |                           0.126406 |             1.634508 |                  1.718782 |      99.965492 |   1.192283 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.088631 |  3.977658 |         0.57281 |                0.493699 | 0.0 | nan | 0.0 |               0.14811 |                   0.14811 |                   4.226589 |                  1.421011 |                      0.0 |     0.088901 |              0.0 |          2.805578 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.040342 | 0.040342 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.006658 |  0.00185 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00185 |
    | 1997-08-03 18:00:00 | 0.05 | 21.4 |              0.1 |             67.0 |       2.5 |               101.6 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 | 0.047 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                 2.552842 |                      2.046164 |                          0.0 |                      0.156328 |                           0.128868 |             1.710404 |                  1.731481 |      99.889596 |   1.193968 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  1.972911 |        0.213084 |                0.183891 | 0.0 | nan | 0.0 |              0.055167 |                  0.055167 |                   4.261282 |                  1.433696 |                      0.0 |     -0.00457 |              0.0 |          2.827585 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.041848 | 0.041848 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.005152 | 0.001431 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001431 |
    | 1997-08-03 19:00:00 | 0.05 | 20.7 |              0.0 |             74.5 |       2.5 |               101.6 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 | 0.047 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                 2.445437 |                      2.089517 |                          0.0 |                      0.150575 |                           0.131251 |              1.82185 |                  1.748925 |       99.77815 |   1.196313 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      1.0 |                     15.566134 |            14.204272 |  0.634046 |        0.057067 |                0.049078 | 0.0 | nan | 0.0 |              0.014723 |                  0.014723 |                   4.261282 |                  1.423756 |                      0.0 |      -0.0446 |              0.0 |          2.837526 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.043061 | 0.043061 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.003939 | 0.001094 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001094 |
    | 1997-08-03 20:00:00 | 0.05 | 19.4 |              0.0 |             81.2 |       2.2 |               101.6 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 | 0.047 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                 2.256393 |                      2.126995 |                          0.0 |                      0.140363 |                           0.133306 |             1.832191 |                  1.766824 |      99.767809 |   1.201582 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                   0.1364 |                     15.566134 |            14.204272 |  0.008648 |        0.000778 |                0.000669 | 0.0 | nan | 0.0 |              0.000201 |                  0.000201 |                   4.261282 |                  1.412078 |                      0.0 |    -0.058636 |              0.0 |          2.849203 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.043962 | 0.043962 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.003038 | 0.000844 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000844 |
    | 1997-08-03 21:00:00 | 0.05 | 17.8 |              0.0 |             86.9 |       1.7 |               101.6 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 | 0.047 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                 2.041337 |                      2.153241 |                          0.0 |                      0.128603 |                           0.134742 |             1.773922 |                  1.781538 |      99.826078 |   1.208454 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.261282 |                  1.401564 |                      0.0 |    -0.058399 |              0.0 |          2.859717 |            125.297451 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.044514 | 0.044514 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.002486 |  0.00069 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00069 |
    | 1997-08-03 22:00:00 | 0.05 | 17.0 |              0.0 |             90.1 |       1.7 |               101.7 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 | 0.047 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                 1.940693 |                      2.174668 |                          0.0 |                      0.123042 |                           0.135912 |             1.748564 |                  1.795279 |      99.951436 |   1.213101 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.261282 |                  1.391158 |                      0.0 |    -0.057965 |              0.0 |          2.870124 |            125.297451 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.044975 | 0.044975 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.002025 | 0.000563 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000563 |
    | 1997-08-03 23:00:00 | 0.05 | 16.4 |              0.0 |             90.9 |       2.3 |               101.7 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 | 0.047 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 1.868084 |                      2.192855 |                          0.0 |                      0.119006 |                           0.136905 |             1.698088 |                  1.806456 |     100.001912 |   1.215845 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            14.204272 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.261282 |                   1.38284 |                      0.0 |    -0.057618 |              0.0 |          2.878441 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.045525 | 0.045525 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.001475 |  0.00041 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00041 |
    | 1997-08-04 00:00:00 | 0.05 | 15.9 |              0.0 |             88.0 |       2.3 |               101.7 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 | 0.047 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                 1.809403 |                      2.187158 |                          0.0 |                       0.11573 |                           0.136594 |             1.592275 |                  1.795292 |     100.107725 |    1.21843 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.239363 |                   1.39654 |                      0.0 |    -0.058189 |              0.0 |          2.842823 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.04554 |  0.04554 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.00146 | 0.000406 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000406 |
    | 1997-08-04 01:00:00 | 0.05 | 15.4 |              0.0 |             89.2 |       2.2 |               101.6 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 | 0.047 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                 1.752342 |                      2.180337 |                          0.0 |                      0.112529 |                           0.136222 |             1.563089 |                  1.784515 |     100.036911 |   1.219467 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.239363 |                  1.406633 |                      0.0 |     -0.05861 |              0.0 |           2.83273 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.045787 | 0.045787 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.001213 | 0.000337 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000337 |
    | 1997-08-04 02:00:00 | 0.05 | 15.2 |              0.0 |             88.8 |       2.3 |               101.6 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 | 0.047 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                 1.729962 |                      2.173535 |                          0.0 |                      0.111271 |                            0.13585 |             1.536206 |                  1.772518 |     100.063794 |   1.220436 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.239363 |                  1.418069 |                      0.0 |    -0.059086 |              0.0 |          2.821294 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.046085 | 0.046085 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.000915 | 0.000254 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000254 |
    | 1997-08-04 03:00:00 | 0.05 | 14.9 |              0.0 |             89.7 |       2.0 |               101.6 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 | 0.047 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                 1.696861 |                      2.165623 |                          0.0 |                      0.109405 |                           0.135418 |             1.522084 |                  1.759749 |     100.077916 |   1.221771 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            14.131208 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   4.239363 |                   1.43017 |                      0.0 |     -0.05959 |              0.0 |          2.809192 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.046319 | 0.046319 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.000681 | 0.000189 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000189 |
    | 1997-08-04 04:00:00 | 0.05 | 14.3 |              0.0 |             93.0 |       2.3 |               101.6 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 | 0.047 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                 1.632319 |                       2.15605 |                          0.0 |                      0.105752 |                           0.134895 |             1.518057 |                  1.748197 |     100.081943 |    1.22434 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.398044 |                     15.502754 |            14.131208 |   0.07485 |        0.006692 |                0.005726 | 0.0 | nan | 0.0 |              0.001718 |                  0.001718 |                   4.239363 |                  1.440702 |                      0.0 |    -0.058312 |              0.0 |           2.79866 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.046528 | 0.046528 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.000472 | 0.000131 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000131 |
    | 1997-08-04 05:00:00 | 0.05 | 14.1 |              0.0 |             93.6 |       2.5 |               101.7 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 | 0.047 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                 1.611289 |                      2.145953 |                          0.0 |                      0.104558 |                           0.134343 |             1.508166 |                  1.736523 |     100.191834 |   1.226451 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.131208 |  0.938714 |        0.083928 |                0.071823 | 0.0 | nan | 0.0 |              0.021547 |                  0.021547 |                   4.239363 |                  1.451305 |                      0.0 |    -0.038924 |              0.0 |          2.788058 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.046795 | 0.046795 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.000205 | 0.000057 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000057 |
    | 1997-08-04 06:00:00 | 0.05 | 14.3 |              0.1 |             94.6 |       2.4 |               101.7 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 | 0.047 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                 1.632319 |                      2.136456 |                          0.0 |                      0.105752 |                           0.133824 |             1.544174 |                  1.726078 |     100.155826 |   1.225432 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            14.246552 |  2.455474 |        0.263445 |                0.226749 | 0.0 | nan | 0.0 |              0.068025 |                  0.068025 |                   4.273966 |                  1.478175 |                      0.0 |     0.006434 |              0.0 |          2.795791 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.047182 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 07:00:00 | 0.05 | 15.4 |              0.9 |             93.3 |       2.5 |               101.7 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 | 0.047 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                 1.752342 |                      2.129774 |                          0.0 |                      0.112529 |                           0.133458 |             1.634935 |                  1.717041 |     100.065065 |   1.220347 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            15.284647 |  4.650125 |        1.164115 |                1.028795 | 0.0 | nan | 0.0 |              0.308638 |                  0.308638 |                   4.585394 |                   1.64464 |                      0.0 |     0.240112 |              0.0 |          2.940754 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |   0.0492 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 08:00:00 | 0.05 | 16.8 |              1.0 |             86.9 |       2.5 |               101.7 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 | 0.047 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                  1.91622 |                       2.12644 |                          0.0 |                      0.121684 |                           0.133275 |             1.665196 |                  1.705582 |     100.034804 |   1.214317 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            16.438085 |  7.209034 |        1.933622 |                1.710811 | 0.0 | nan | 0.0 |              0.513243 |                  0.513243 |                   4.931426 |                   1.83472 |                      0.0 |     0.436797 |              0.0 |          3.096706 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.051571 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 09:00:00 | 0.05 | 18.2 |              1.0 |             78.9 |       2.9 |               101.7 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 | 0.047 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                 2.093348 |                      2.122001 |                          0.0 |                      0.131461 |                           0.133032 |             1.651651 |                  1.695656 |     100.048349 |   1.208543 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.360836 |  9.458797 |        2.537058 |                2.210424 | 0.0 | nan | 0.0 |              0.663127 |                  0.663127 |                   5.208251 |                  1.989515 |                      0.0 |     0.580231 |              0.0 |          3.218735 |              73.45023 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.053462 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 10:00:00 | 0.05 | 19.5 |              0.9 |             75.8 |       3.1 |               101.7 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 | 0.047 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                 2.270468 |                      2.117571 |                          0.0 |                      0.141127 |                            0.13279 |             1.721014 |                  1.691233 |      99.978986 |   1.202862 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.822211 | 10.747438 |         2.69052 |                2.317703 | 0.0 | nan | 0.0 |              0.695311 |                  0.695311 |                   5.346663 |                   2.06621 |                      0.0 |     0.609219 |              0.0 |          3.280453 |             68.711505 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054455 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 11:00:00 | 0.05 | 20.3 |              0.8 |             73.4 |       3.3 |               101.8 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 | 0.047 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                  2.38585 |                      2.111492 |                          0.0 |                      0.147368 |                           0.132457 |             1.751214 |                  1.687786 |     100.048786 |   1.200635 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            17.937555 | 10.819272 |        2.515039 |                2.160947 | 0.0 | nan | 0.0 |              0.648284 |                  0.648284 |                   5.381266 |                  2.087634 |                      0.0 |     0.561299 |              0.0 |          3.293633 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054881 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 12:00:00 | 0.05 | 21.0 |              0.9 |             68.3 |       3.3 |               101.8 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 | 0.047 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                 2.490975 |                      2.109837 |                          0.0 |                      0.153018 |                           0.132366 |             1.701336 |                  1.687166 |     100.098664 |   1.198001 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.052899 | 10.408076 |        2.605564 |                2.233851 | 0.0 | nan | 0.0 |              0.670155 |                  0.670155 |                    5.41587 |                  2.106001 |                      0.0 |     0.582405 |              0.0 |          3.309868 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.055387 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 13:00:00 | 0.05 | 21.8 |              0.8 |             61.3 |       3.2 |               101.8 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 | 0.047 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                 2.616045 |                      2.109837 |                          0.0 |                      0.159698 |                           0.132366 |             1.603636 |                  1.682947 |     100.196364 |   1.195188 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             18.39893 | 10.917404 |         2.53785 |                2.157989 | 0.0 | nan | 0.0 |              0.647397 |                  0.647397 |                   5.519679 |                  2.166369 |                      0.0 |     0.557131 |              0.0 |           3.35331 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.056372 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 14:00:00 | 0.05 | 21.9 |              0.9 |             60.6 |       2.5 |               101.8 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 | 0.047 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                 2.632058 |                      2.104328 |                          0.0 |                       0.16055 |                           0.132064 |             1.595027 |                  1.676097 |     100.204973 |   1.194821 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            18.975649 | 10.494383 |         2.62717 |                2.208784 | 0.0 | nan | 0.0 |              0.662635 |                  0.662635 |                   5.692695 |                  2.265756 |                      0.0 |     0.568229 |              0.0 |          3.426939 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.057374 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 15:00:00 | 0.05 | 21.8 |              0.9 |             58.8 |       2.9 |               101.7 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 | 0.047 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                 2.616045 |                      2.099381 |                          0.0 |                      0.159698 |                           0.131792 |             1.538235 |                  1.669008 |     100.161765 |   1.194298 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            19.437025 |  8.874215 |        2.221576 |                1.861585 | 0.0 | nan | 0.0 |              0.558476 |                  0.558476 |                   5.831107 |                  2.348437 |                      0.0 |     0.460624 |              0.0 |           3.48267 |              73.45023 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.058283 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 16:00:00 | 0.05 | 22.2 |              0.9 |             57.2 |       3.6 |               101.7 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 | 0.047 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                 2.680609 |                      2.097734 |                          0.0 |                      0.163129 |                           0.131702 |             1.533308 |                   1.66429 |     100.166692 |   1.192703 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |              19.8984 |   6.45697 |        1.616442 |                1.360212 | 0.0 | nan | 0.0 |              0.408064 |                  0.408064 |                    5.96952 |                  2.428674 |                      0.0 |     0.306869 |              0.0 |          3.540846 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.05932 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 17:00:00 | 0.05 | 21.8 |              1.0 |             57.2 |       4.2 |               101.7 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 | 0.047 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                 2.616045 |                      2.097185 |                          0.0 |                      0.159698 |                           0.131672 |             1.496378 |                  1.659573 |     100.203622 |   1.194485 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            20.705807 |  3.945159 |        1.058179 |                0.907937 | 0.0 | nan | 0.0 |              0.272381 |                  0.272381 |                   6.211742 |                  2.564427 |                      0.0 |      0.16553 |              0.0 |          3.647315 |             50.715635 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.061051 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 18:00:00 | 0.05 | 20.9 |              1.0 |             59.9 |       4.2 |               101.7 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 | 0.047 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 2.475714 |                      2.094444 |                          0.0 |                        0.1522 |                           0.131522 |             1.482953 |                  1.651207 |     100.217047 |   1.198201 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |            21.743901 |  1.939642 |        0.520255 |                0.462876 | 0.0 | nan | 0.0 |              0.138863 |                  0.138863 |                    6.52317 |                  2.744116 |                      0.0 |     0.024525 |              0.0 |          3.779055 |             50.715635 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.063121 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 19:00:00 | 0.05 | 19.7 |              1.0 |             66.1 |       3.6 |               101.7 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 | 0.047 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                 2.298847 |                       2.08897 |                          0.0 |                      0.142666 |                           0.131221 |             1.519538 |                   1.63958 |     100.180462 |   1.202947 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      1.0 |                     15.502754 |             22.89734 |   0.60531 |        0.162358 |                0.151461 | 0.0 | nan | 0.0 |              0.045438 |                  0.045438 |                   6.869202 |                  2.950104 |                      0.0 |    -0.077483 |              0.0 |          3.919098 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |   0.0652 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 20:00:00 | 0.05 | 17.9 |              0.3 |             71.7 |       3.2 |               101.7 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 | 0.047 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                 2.054233 |                      2.080783 |                          0.0 |                      0.129312 |                           0.130772 |             1.472885 |                  1.624918 |     100.227115 |   1.210597 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                 0.104711 |                     15.502754 |            23.243371 |  0.005136 |        0.003091 |                0.002926 | 0.0 | nan | 0.0 |              0.000878 |                  0.000878 |                   6.973011 |                  3.034376 |                      0.0 |    -0.125554 |              0.0 |          3.938635 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.06592 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 21:00:00 | 0.05 | 15.8 |              0.0 |             80.6 |       2.2 |               101.8 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 | 0.047 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                 1.797863 |                       2.06991 |                          0.0 |                      0.115083 |                           0.130174 |             1.449077 |                  1.610994 |     100.350923 |    1.22071 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.973011 |                  3.060678 |                      0.0 |    -0.127528 |              0.0 |          3.912334 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.065866 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 22:00:00 | 0.05 | 14.8 |              0.0 |             85.9 |       2.2 |               101.8 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 | 0.047 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                 1.685952 |                      2.058007 |                          0.0 |                      0.108789 |                            0.12952 |             1.448232 |                  1.598128 |     100.351768 |   1.224953 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.973011 |                  3.084324 |                      0.0 |    -0.128514 |              0.0 |          3.888687 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.065746 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 23:00:00 | 0.05 | 14.1 |              0.0 |             89.3 |       2.1 |               101.8 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 | 0.047 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                 1.611289 |                      2.045628 |                          0.0 |                      0.104558 |                           0.128839 |             1.438881 |                  1.587151 |     100.361119 |   1.227981 |         0.293016 | 4.601956 | 20.104711 |         0.972464 |                 35.762943 |                      0.0 |                     15.502754 |            23.243371 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.973011 |                  3.103639 |                      0.0 |    -0.129318 |              0.0 |          3.869372 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.065422 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 00:00:00 | 0.05 | 13.4 |              0.0 |             91.4 |       1.8 |               101.8 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 | 0.047 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                 1.539538 |                      2.032245 |                          0.0 |                      0.100468 |                           0.128102 |             1.407137 |                  1.579647 |     100.392863 |   1.231127 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.943374 |                  3.124611 |                      0.0 |    -0.130192 |              0.0 |          3.818763 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.064471 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 01:00:00 | 0.05 | 12.8 |              0.0 |             93.1 |       1.8 |               101.8 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 | 0.047 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                 1.480281 |                      2.018409 |                          0.0 |                      0.097071 |                           0.127339 |             1.378142 |                  1.572172 |     100.421858 |   1.233843 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.943374 |                  3.135507 |                      0.0 |    -0.130646 |              0.0 |          3.807867 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.063963 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 02:00:00 | 0.05 | 12.3 |              0.0 |             94.6 |       1.6 |               101.8 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 | 0.047 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                 1.432438 |                      2.003073 |                          0.0 |                      0.094315 |                           0.126493 |             1.355086 |                  1.565068 |     100.444914 |   1.236111 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.943374 |                  3.144898 |                      0.0 |    -0.131037 |              0.0 |          3.798476 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.063323 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 03:00:00 | 0.05 | 12.0 |              0.0 |             95.6 |       1.2 |               101.8 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 | 0.047 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 1.404387 |                       1.98784 |                          0.0 |                      0.092693 |                           0.125652 |             1.342594 |                  1.558052 |     100.457406 |   1.237469 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            23.144579 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   6.943374 |                   3.15411 |                      0.0 |    -0.131421 |              0.0 |          3.789264 |            177.504722 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062667 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 04:00:00 | 0.05 | 11.7 |              0.0 |             96.2 |       1.9 |               101.8 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 | 0.047 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                 1.376819 |                      1.974268 |                          0.0 |                      0.091096 |                           0.124901 |               1.3245 |                  1.550047 |       100.4755 |   1.238856 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.366022 |                     15.438712 |            23.144579 |  0.063733 |         0.00566 |                0.005336 | 0.0 | nan | 0.0 |              0.001601 |                  0.001601 |                   6.943374 |                  3.166293 |                      0.0 |    -0.130328 |              0.0 |          3.777081 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062183 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 05:00:00 | 0.05 | 11.6 |              0.0 |             95.7 |       1.3 |               101.8 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 | 0.047 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 1.367736 |                      1.961295 |                          0.0 |                      0.090568 |                           0.124183 |             1.308923 |                  1.541578 |     100.491077 |   1.239363 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.144579 |  0.908598 |        0.080684 |                0.076088 | 0.0 | nan | 0.0 |              0.022826 |                  0.022826 |                   6.943374 |                  3.179808 |                      0.0 |    -0.109666 |              0.0 |          3.763566 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.061583 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 06:00:00 | 0.05 | 12.6 |              0.6 |             95.1 |       1.6 |               101.8 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 | 0.047 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                 1.460978 |                      1.952516 |                          0.0 |                       0.09596 |                           0.123697 |              1.38939 |                  1.535084 |      100.41061 |   1.234655 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  2.420659 |        0.472904 |                 0.45318 | 0.0 | nan | 0.0 |              0.135954 |                  0.135954 |                   7.115928 |                  3.288825 |                      0.0 |     -0.00108 |              0.0 |          3.827103 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062021 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 07:00:00 | 0.05 | 14.8 |              0.9 |             96.3 |       1.9 |               101.9 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 | 0.047 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                 1.685952 |                      1.949426 |                          0.0 |                      0.108789 |                           0.123526 |             1.623571 |                  1.535091 |     100.276429 |   1.225361 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  4.619722 |        1.148656 |                1.101438 | 0.0 | nan | 0.0 |              0.330431 |                  0.330431 |                   7.115928 |                  3.287363 |                      0.0 |     0.193458 |              0.0 |          3.828566 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.061775 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 08:00:00 | 0.05 | 17.1 |              1.0 |             89.9 |       3.2 |               101.9 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 | 0.047 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                 1.953031 |                       1.95097 |                          0.0 |                      0.123726 |                           0.123612 |             1.755775 |                  1.538746 |     100.144225 |   1.215051 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 |  7.198198 |        1.917614 |                1.839976 | 0.0 | nan | 0.0 |              0.551993 |                  0.551993 |                   7.115928 |                  3.279458 |                      0.0 |     0.415349 |              0.0 |          3.836471 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.061896 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 09:00:00 | 0.05 | 19.4 |              0.9 |             79.2 |       4.0 |               101.9 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 | 0.047 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                 2.256393 |                      1.957159 |                          0.0 |                      0.140363 |                           0.123954 |             1.787063 |                  1.543872 |     100.112937 |   1.205358 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.604725 |  9.475573 |        2.356024 |                2.266269 | 0.0 | nan | 0.0 |              0.679881 |                  0.679881 |                   7.081417 |                  3.250753 |                      0.0 |     0.544433 |              0.0 |          3.830665 |             53.251417 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062093 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 10:00:00 | 0.05 | 20.6 |              1.0 |             73.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 | 0.047 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                  2.43042 |                      1.962848 |                          0.0 |                      0.149767 |                           0.124269 |             1.783928 |                  1.546397 |     100.116072 |   1.200448 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.719761 | 10.786309 |        2.873495 |                 2.75694 | 0.0 | nan | 0.0 |              0.827082 |                  0.827082 |                   7.115928 |                  3.266966 |                      0.0 |     0.690958 |              0.0 |          3.848962 |              46.30558 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062733 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 11:00:00 | 0.05 | 21.4 |              1.0 |             67.3 |       4.1 |               101.9 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 | 0.047 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                 2.552842 |                      1.968551 |                          0.0 |                      0.156328 |                           0.124585 |             1.718063 |                  1.545886 |     100.181937 |   1.197482 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            23.949834 | 10.865527 |        2.894599 |                2.761725 | 0.0 | nan | 0.0 |              0.828518 |                  0.828518 |                    7.18495 |                  3.309837 |                      0.0 |     0.690608 |              0.0 |          3.875113 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.063478 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 12:00:00 | 0.05 | 22.5 |              1.0 |             62.2 |       4.1 |               101.9 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 | 0.047 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                 2.729939 |                      1.976351 |                          0.0 |                      0.165743 |                           0.125016 |             1.698022 |                  1.546988 |     100.201978 |   1.193116 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |             24.06487 |  10.45226 |        2.784503 |                 2.65055 | 0.0 | nan | 0.0 |              0.795165 |                  0.795165 |                   7.219461 |                  3.330376 |                      0.0 |     0.656399 |              0.0 |          3.889085 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.064117 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 13:00:00 | 0.05 | 23.1 |              1.0 |             59.6 |       4.4 |               101.9 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 | 0.047 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                 2.830978 |                      1.983133 |                          0.0 |                      0.171077 |                           0.125391 |             1.687263 |                  1.550892 |     100.212737 |   1.190747 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.294943 |  10.96314 |        2.920603 |                2.764822 | 0.0 | nan | 0.0 |              0.829447 |                  0.829447 |                   7.288483 |                  3.363175 |                      0.0 |     0.689314 |              0.0 |          3.925307 |             48.410379 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.064896 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 14:00:00 | 0.05 | 23.4 |              1.0 |             55.4 |       4.6 |               101.9 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 | 0.047 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                 2.882708 |                      1.990983 |                          0.0 |                      0.173797 |                           0.125825 |              1.59702 |                  1.552718 |      100.30298 |   1.189943 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.409979 | 10.528069 |        2.804699 |                2.649346 | 0.0 | nan | 0.0 |              0.794804 |                  0.794804 |                   7.322994 |                   3.38194 |                      0.0 |      0.65389 |              0.0 |          3.941053 |              46.30558 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.065782 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 15:00:00 | 0.05 | 23.8 |              1.0 |             52.1 |       4.5 |               101.9 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 | 0.047 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                 2.952961 |                      2.001493 |                          0.0 |                      0.177481 |                           0.126406 |             1.538493 |                  1.555327 |     100.361507 |     1.1886 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.525016 |  8.882849 |        2.366409 |                 2.23335 | 0.0 | nan | 0.0 |              0.670005 |                  0.670005 |                   7.357505 |                  3.400029 |                      0.0 |     0.528337 |              0.0 |          3.957476 |             47.334593 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.066681 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 16:00:00 | 0.05 | 24.0 |              1.0 |             49.5 |       4.7 |               101.8 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 | 0.047 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                 2.988644 |                      2.010993 |                          0.0 |                      0.179348 |                            0.12693 |             1.479379 |                  1.556257 |     100.320621 |   1.186889 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  6.439028 |         1.71537 |                1.620343 | 0.0 | nan | 0.0 |              0.486103 |                  0.486103 |                   7.392016 |                   3.42173 |                      0.0 |     0.343531 |              0.0 |          3.970285 |             45.320355 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.06748 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 17:00:00 | 0.05 | 24.0 |              1.0 |             48.1 |       4.5 |               101.8 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 | 0.047 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                 2.988644 |                      2.022657 |                          0.0 |                      0.179348 |                           0.127573 |             1.437538 |                  1.557615 |     100.362462 |   1.187075 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  3.911896 |        1.042137 |                0.985011 | 0.0 | nan | 0.0 |              0.295503 |                  0.295503 |                   7.392016 |                  3.423912 |                      0.0 |      0.15284 |              0.0 |          3.968104 |             47.334593 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.068039 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 18:00:00 | 0.05 | 23.7 |              1.0 |             49.4 |       4.1 |               101.8 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 | 0.047 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                  2.93526 |                      2.037589 |                          0.0 |                      0.176554 |                           0.128396 |             1.450018 |                  1.560199 |     100.349982 |   1.188219 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  1.905864 |        0.507726 |                0.480125 | 0.0 | nan | 0.0 |              0.144038 |                  0.144038 |                   7.392016 |                   3.42458 |                      0.0 |     0.001347 |              0.0 |          3.967436 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.068633 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 19:00:00 | 0.05 | 22.8 |              1.0 |             57.0 |       3.8 |               101.8 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 | 0.047 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                 2.780058 |                      2.054233 |                          0.0 |                      0.168392 |                           0.129312 |             1.584633 |                  1.565154 |     100.215367 |   1.191234 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      1.0 |                     15.438712 |            24.640052 |  0.576079 |        0.153469 |                0.145175 | 0.0 | nan | 0.0 |              0.043553 |                  0.043553 |                   7.392016 |                  3.420162 |                      0.0 |    -0.098954 |              0.0 |          3.971854 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.069308 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 20:00:00 | 0.05 | 21.3 |              0.5 |             65.9 |       2.5 |               101.8 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 | 0.047 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 2.537251 |                      2.072623 |                          0.0 |                      0.155495 |                           0.130323 |             1.672048 |                  1.574157 |     100.127952 |   1.196911 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                 0.072689 |                     15.438712 |            24.870125 |  0.002495 |        0.003269 |                0.003122 | 0.0 | nan | 0.0 |              0.000936 |                  0.000936 |                   7.461037 |                  3.445356 |                      0.0 |     -0.14262 |              0.0 |          4.015681 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.070109 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 21:00:00 | 0.05 | 19.3 |              0.0 |             73.0 |       1.9 |               101.8 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 | 0.047 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                 2.242395 |                      2.091705 |                          0.0 |                      0.139602 |                           0.131371 |             1.636948 |                  1.582026 |     100.163052 |   1.205255 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   7.461037 |                  3.434698 |                      0.0 |    -0.143112 |              0.0 |          4.026339 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.070708 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 22:00:00 | 0.05 | 18.3 |              0.0 |             76.7 |       2.2 |               101.8 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 | 0.047 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                  2.10653 |                       2.11094 |                          0.0 |                      0.132185 |                           0.132426 |             1.615709 |                  1.588483 |     100.184291 |   1.209486 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   7.461037 |                  3.427556 |                      0.0 |    -0.142815 |              0.0 |          4.033482 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.071405 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 23:00:00 | 0.05 | 17.4 |              0.0 |             81.8 |       2.7 |               101.8 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 | 0.047 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                 1.990459 |                      2.129218 |                          0.0 |                      0.125796 |                           0.133428 |             1.628195 |                  1.595583 |     100.171805 |   1.213176 |         0.288036 | 4.633978 | 20.072689 |          0.97278 |                 35.520271 |                      0.0 |                     15.438712 |            24.870125 |       0.0 |             0.0 |                     0.0 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   7.461037 |                  3.418333 |                      0.0 |    -0.142431 |              0.0 |          4.042705 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.07218 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

    >>> inputs.nied.series = 0.0

.. _lland_v3_nadelw_summer_hourly:

conifers (summer)
-----------------

Now we repeat the daily integration test :ref:`lland_v3_nadelw_summer_daily`
on an hourly basis. In contrast to the calculation for water areas (see
:ref:`lland_v3_water_hourly`), both the interception evaporation (|EvI|)
and the soil evapotranspiration (|EvB|) rely on hourly input data (see method
|Return_PenmanMonteith_V1|) and thus show a diurnal pattern.  |lland_v3|
always adds condensed water to the interception storage (|Inzp|).  Hence,
|EvI| is sometimes negative during the nighttime, but |Inzp| dries out as
soon as the sun starts to rise:

.. integration-test::

    >>> lnk(NADELW)
    >>> test("lland_v3_nadelw_summer_hourly",
    ...      axis1=(fluxes.evi, fluxes.evb),
    ...      axis2=states.inzp,
    ...      use_conditions=conditions_nadelw_summer)
    |                date | nied | teml | sunshineduration | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair | solardeclination |      tsa |       tsu | earthsundistance | extraterrestrialradiation | possiblesunshineduration | dailypossiblesunshineduration | dailyglobalradiation |        sp | globalradiation | adjustedglobalradiation |         g |        tz |        wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |      ebdn |      bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |              0.0 |             95.1 |       0.8 |               101.5 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |  0.0 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.283992 |          0.8 |                 1.928423 |                      1.975309 |                          0.0 |                      0.122362 |                           0.124959 |              1.83393 |                  1.696461 |       99.66607 |    1.21073 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.208683 | -0.036532 |                   0.0 |                       0.0 |                   9.243306 |                  0.901196 |                      0.0 |     -0.03755 |              0.0 |          8.342111 |                 117.5 |                 100.0 |               170.875609 |               69.444444 |  0.0 |  0.0 | -0.001025 | -0.001025 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.221929 | 0.061647 | 0.001025 |  0.0 |  0.0 |   0.0 |  nan | 31.308721 | 78.724681 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.221929 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.061647 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |              0.0 |             94.9 |       0.8 |               101.5 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |  0.0 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.283992 |          0.8 |                 1.892018 |                      1.970628 |                          0.0 |                      0.120339 |                             0.1247 |             1.795525 |                  1.694576 |      99.704475 |   1.212158 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.295735 | -0.028172 |                   0.0 |                       0.0 |                   9.243306 |                  0.901844 |                      0.0 |    -0.037577 |              0.0 |          8.341463 |                 117.5 |                 100.0 |               173.430933 |               69.444444 |  0.0 |  0.0 | -0.000655 | -0.000655 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.216196 | 0.060055 |  0.00168 |  0.0 |  0.0 |   0.0 |  nan | 31.326476 | 78.724681 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.216196 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.060055 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |              0.0 |             95.9 |       0.8 |               101.5 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |  0.0 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.283992 |          0.8 |                 1.868084 |                       1.96233 |                          0.0 |                      0.119006 |                           0.124241 |             1.791493 |                  1.692183 |      99.708507 |   1.213014 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 |  15.35492 | -0.022574 |                   0.0 |                       0.0 |                   9.243306 |                  0.902355 |                      0.0 |    -0.037598 |              0.0 |          8.340951 |                 117.5 |                 100.0 |               173.240523 |               69.444444 |  0.0 |  0.0 | -0.002635 | -0.002635 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.210612 | 0.058503 | 0.004315 |  0.0 |  0.0 |   0.0 |  nan | 31.338633 | 78.724681 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.210612 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.058503 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |              0.0 |             96.7 |       0.8 |               101.5 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |  0.0 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.283992 |          0.8 |                 1.856216 |                      1.948396 |                          0.0 |                      0.118345 |                           0.123469 |             1.794961 |                  1.689422 |      99.705039 |   1.213417 |         0.297909 | 4.570266 |   20.1364 |         0.972155 |                 36.001693 |                      0.0 |                     15.566134 |            10.503757 |       0.0 |             0.0 |                     0.0 | -0.010417 | 15.395444 | -0.019538 |                   0.0 |                       0.0 |                   9.243306 |                  0.902364 |                      0.0 |    -0.037598 |              0.0 |          8.340943 |                 117.5 |                 100.0 |               172.677019 |               69.444444 |  0.0 |  0.0 | -0.004198 | -0.004198 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.205172 | 0.056992 | 0.008513 |  0.0 |  0.0 |   0.0 |  nan | 31.347755 | 78.724681 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.205172 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.056992 |
    | 1997-08-03 04:00:00 |  0.