import type { PipelineBase, StackAsset, StackDeployment } from "aws-cdk-lib/pipelines";
import type { Construct } from "constructs";
import type { WorkflowProps } from "github-actions-cdk";
import { Workflow } from "github-actions-cdk";
import type { AwsCredentialsProvider } from "./aws-credentials";
import type { IJobPhase, StackOptions } from "./jobs";
import { StageJob, Synth } from "./steps";
export interface PipelineWorkflowProps extends WorkflowProps {
    /** The pipeline being defined, including stages and jobs. */
    readonly pipeline: PipelineBase;
    /** Options for configuring individual stacks in the pipeline. */
    readonly stackOptions: Record<string, StackOptions>;
    /** Optional phase to execute before build jobs. */
    readonly preBuild?: IJobPhase;
    /** Optional phase to execute after build jobs. */
    readonly postBuild?: IJobPhase;
    /** AWS credentials provider for authenticating with AWS services. */
    readonly awsCredentials: AwsCredentialsProvider;
    /** Output directory for CloudFormation templates. */
    readonly cdkoutDir: string;
}
/**
 * Represents a workflow that manages the pipeline jobs for synthesizing, publishing, and deploying resources.
 *
 * This class extends the Workflow class from the GitHub Actions CDK to provide
 * a structured way of orchestrating pipeline jobs based on the AWS CDK pipeline graph.
 */
export declare class PipelineWorkflow extends Workflow {
    readonly awsCredentials: AwsCredentialsProvider;
    readonly cdkoutDir: string;
    private readonly stackOptions;
    /**
     * Constructs a new instance of PipelineWorkflow.
     *
     * @param scope - The scope in which this construct is defined.
     * @param id - Unique identifier for this workflow.
     * @param props - Properties for configuring the pipeline workflow.
     */
    constructor(scope: Construct, id: string, props: PipelineWorkflowProps);
    /**
     * Creates a job for synthesizing the application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of job IDs that this job depends on.
     * @param synth - The Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    protected jobForSynth(id: string, needs: string[], synth: Synth, preBuild?: IJobPhase, postBuild?: IJobPhase): void;
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of job IDs that this job depends on.
     * @param assets - The stack assets to publish.
     */
    protected jobForPublish(id: string, needs: string[], assets: StackAsset[]): void;
    /**
     * Creates a job for deploying a stack.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of job IDs that this job depends on.
     * @param stack - The stack deployment information.
     */
    protected jobForDeploy(id: string, needs: string[], stack: StackDeployment): void;
    /**
     * Creates a job for running a stage job.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of job IDs that this job depends on.
     * @param job - The stage job configuration.
     */
    protected jobForStage(id: string, needs: string[], job: StageJob): void;
    /**
     * Renders the dependencies for a given node in the graph.
     *
     * @param node - The graph node whose dependencies need to be rendered.
     * @returns An array of unique IDs representing the node's dependencies.
     */
    private renderDependencies;
}
