import type { StackAsset, StackDeployment } from "aws-cdk-lib/pipelines";
import type { Construct } from "constructs";
import { Job, type JobProps } from "github-actions-cdk";
import type { AwsCredentialsProvider } from "./aws-credentials";
import type { StageOptions } from "./wave";
/**
 * Properties for configuring a pipeline job.
 */
export interface PipelineJobProps extends JobProps {
    /**
     * The provider for AWS credentials to be used within this job.
     */
    readonly awsCredentials: AwsCredentialsProvider;
    /**
     * The directory where the CDK output files are located.
     */
    readonly cdkoutDir: string;
}
/**
 * Represents a job in the pipeline that requires AWS credentials and CDK output.
 */
export declare class PipelineJob extends Job {
    readonly awsCredentials: AwsCredentialsProvider;
    readonly cdkoutDir: string;
    /**
     * Constructs a new instance of `PipelineJob`.
     *
     * @param scope - The scope in which this job is defined.
     * @param id - The unique identifier for this job.
     * @param props - The properties for the pipeline job.
     */
    constructor(scope: Construct, id: string, props: PipelineJobProps);
}
/**
 * Interface for defining a phase of job steps in the pipeline.
 */
export interface IJobPhase {
    /**
     * Defines the steps to be executed for this job phase.
     *
     * @param job - The pipeline job in which to add the steps.
     */
    steps(job: PipelineJob): void;
}
/**
 * Properties for a synthetic pipeline job, including build phases and commands.
 */
export interface SynthPipelineJobProps extends PipelineJobProps {
    /**
     * Optional pre-build phase steps.
     */
    readonly preBuild?: IJobPhase;
    /**
     * Optional post-build phase steps.
     */
    readonly postBuild?: IJobPhase;
    /**
     * Commands to run for installation before the build.
     */
    readonly installCommands?: string[];
    /**
     * Commands to run for the build.
     */
    readonly commands: string[];
}
/**
 * A job that synthesizes the CloudFormation template using CDK.
 */
export declare class SynthPipelineJob extends PipelineJob {
    constructor(scope: Construct, id: string, props: SynthPipelineJobProps);
}
/**
 * Properties for a publish pipeline job.
 */
export interface PublishPipelineJobProps extends PipelineJobProps {
    /**
     * The stack assets to be published.
     */
    readonly assets: StackAsset[];
    /**
     * Optional version of the CDK CLI to use for publishing.
     */
    readonly cdkCliVersion?: string;
}
/**
 * A job that publishes stack assets to AWS.
 */
export declare class PublishPipelineJob extends PipelineJob {
    constructor(scope: Construct, id: string, props: PublishPipelineJobProps);
}
/**
 * Options for the deployment of a stack.
 */
export interface StackOptions {
    /**
     * The GitHub environment for the stack deployment.
     */
    readonly environment: StageOptions["gitHubEnvironment"];
    /**
     * The capabilities for the stack deployment.
     */
    readonly capabilities: StageOptions["stackCapabilities"];
}
/**
 * Properties for a deployment pipeline job.
 */
export interface DeployPipelineJobProps extends PipelineJobProps {
    /**
     * The stack to be deployed.
     */
    readonly stack: StackDeployment;
    /**
     * Optional stack-specific options.
     */
    readonly stackOptions?: StackOptions;
}
/**
 * A job that deploys a CloudFormation stack.
 */
export declare class DeployPipelineJob extends PipelineJob {
    constructor(scope: Construct, id: string, props: DeployPipelineJobProps);
}
/**
 * Properties for a stage pipeline job.
 */
export interface StagePipelineJobProps extends PipelineJobProps {
    /**
     * The phase that defines the steps to execute in this job.
     */
    readonly phase: IJobPhase;
}
/**
 * A job that executes a specific phase of steps in the pipeline.
 */
export declare class StagePipelineJob extends PipelineJob {
    constructor(scope: Construct, id: string, props: StagePipelineJobProps);
}
