import { type IOpenIdConnectProvider, Role } from "aws-cdk-lib/aws-iam";
import { Construct } from "constructs";
/**
 * Properties for configuring the GitHub Actions OpenID Connect provider.
 */
export interface GitHubActionsOpenIdConnectProviderProps {
    /**
     * Optional thumbprints to verify GitHub's certificates.
     *
     * Ensure to update these when GitHub rotates their certificates.
     *
     * @default - Uses predefined, up-to-date thumbprints.
     */
    readonly thumbprints?: string[];
}
/**
 * Represents an OpenID Connect (OIDC) provider for GitHub Actions.
 *
 * This provider allows GitHub Actions to assume roles in AWS by connecting through OpenID Connect.
 */
export declare class GitHubActionsOpenIdConnectProvider extends Construct {
    /**
     * Imports an existing GitHub Actions OpenID Connect provider by ARN.
     *
     * @param scope - The construct scope to define the provider within.
     * @returns The imported OIDC provider interface.
     */
    static fromGitHubActionsOpenIdConnectProvider(scope: Construct): IOpenIdConnectProvider;
    /**
     * Constructs a new instance of `GitHubActionsOpenIdConnectProvider`.
     *
     * @param scope - The construct scope to define the provider within.
     * @param id - The unique identifier for this provider.
     * @param props - Optional properties for the OpenID Connect provider.
     */
    constructor(scope: Construct, id: string, props?: GitHubActionsOpenIdConnectProviderProps);
}
/**
 * Properties for creating a GitHub Actions IAM role.
 */
export interface GitHubActionsRoleProps {
    /**
     * The name for the GitHub Actions IAM role.
     *
     * @default - "GitHubActionsRole"
     */
    readonly roleName?: string;
    /**
     * The OpenID Connect provider that GitHub Actions will use to assume this role.
     */
    readonly provider: IOpenIdConnectProvider;
    /**
     * A list of GitHub repositories that are permitted to assume this role.
     *
     * Each repository should be formatted as `owner/repo`.
     */
    readonly repos?: string[];
    /**
     * Additional custom subject claims to allow for the role.
     *
     * Each claim should conform to the format used in GitHub OIDC conditions.
     */
    readonly subjectClaims?: string[];
}
/**
 * Creates an IAM Role for GitHub Actions workflows using an OpenID Connect provider.
 *
 * The role includes policies allowing the assumption of bootstrap roles and access to ECR authorization.
 */
export declare class GitHubActionsRole extends Role {
    /**
     * Constructs a new instance of `GitHubActionsRole`.
     *
     * @param scope - The construct scope to define the role within.
     * @param id - The unique identifier for this role.
     * @param props - The properties for configuring the GitHub Actions role.
     */
    constructor(scope: Construct, id: string, props: GitHubActionsRoleProps);
}
