"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionsPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const constructs_1 = require("constructs");
const adapter_1 = require("./adapter");
const wave_1 = require("./wave");
const workflow_1 = require("./workflow");
/**
 * Constructs a GitHub Actions pipeline for deploying AWS resources.
 *
 * This construct provides methods to define the workflow, add stages, and manage waves of jobs.
 */
class GitHubActionsPipeline extends constructs_1.Construct {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsPipeline", version: "0.0.19" };
    innerPipeline;
    /**
     * Constructs a new instance of `GitHubActionsPipeline`.
     *
     * @param scope - The parent construct.
     * @param id - Unique identifier for this construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.innerPipeline = new InnerPipeline(this, id, props);
    }
    /**
     * Returns the name of the workflow.
     */
    get workflowName() {
        return this.innerPipeline.workflowName;
    }
    /**
     * Returns the output directory for the workflow files.
     */
    get workflowOutdir() {
        return this.innerPipeline.workflowOutdir;
    }
    /**
     * Returns the filename for the workflow.
     */
    get workflowFilename() {
        return this.innerPipeline.workflowFilename;
    }
    /**
     * Adds a stage to the pipeline with GitHub-specific options.
     *
     * @param stage - The stage to add to the pipeline.
     * @param options - Optional configuration for the stage.
     * @returns The deployment information for the added stage.
     */
    addStage(stage, options = {}) {
        return this.innerPipeline.addStageWithGitHubOptions(stage, options);
    }
    /**
     * Adds a wave of jobs to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Configuration options for the wave.
     * @returns The created GitHub wave.
     */
    addWave(id, options = {}) {
        return this.innerPipeline.addGitHubWave(id, options);
    }
}
exports.GitHubActionsPipeline = GitHubActionsPipeline;
/**
 * Inner class that extends the PipelineBase and implements IWaveStageAdder.
 * This manages the core functionalities of the GitHub Actions pipeline.
 */
class InnerPipeline extends pipelines_1.PipelineBase {
    workflowName;
    workflowOutdir;
    workflowFilename;
    preBuild;
    postBuild;
    awsCredentials;
    stackOptions = {};
    adapter;
    /**
     * Constructs a new instance of `InnerPipeline`.
     *
     * @param scope - The parent construct.
     * @param id - Unique identifier for this construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.workflowName = props.workflowName ?? "Deploy";
        this.workflowOutdir = props.workflowOutdir ?? ".github/workflows";
        this.workflowFilename = props.workflowFilename ?? "deploy";
        this.preBuild = props.preBuild;
        this.postBuild = props.postBuild;
        this.awsCredentials = props.awsCredentials;
        this.adapter = new adapter_1.AwsCdkAdapter(this, {
            outdir: this.workflowOutdir,
        });
    }
    /**
     * Adds a stage deployment from a wave.
     *
     * @param stageDeployment - The stage deployment to add.
     * @param options - Optional configuration for the stage.
     */
    addStageFromWave(stageDeployment, options) {
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
    }
    /**
     * Adds a stage with GitHub-specific options to the pipeline.
     *
     * @param stage - The stage to add.
     * @param options - Optional configuration for the stage.
     * @returns The deployment information for the added stage.
     */
    addStageWithGitHubOptions(stage, options = {}) {
        const stageDeployment = this.addStage(stage, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, "capabilities", options?.stackCapabilities);
        return stageDeployment;
    }
    /**
     * Adds a GitHub wave to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Configuration options for the wave.
     * @returns The created GitHub wave.
     */
    addGitHubWave(id, options = {}) {
        const wave = new wave_1.GitHubWave(id, this, {
            pre: options.preJobs,
            post: options.postJobs,
        });
        this.waves.push(wave._innerWave);
        return wave;
    }
    /**
     * Builds the pipeline workflow.
     *
     * This method is called during the CDK synthesis phase to generate the necessary workflow files.
     */
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error("The GitHub Workflow must be defined in the scope of an App");
        }
        const names = app.node
            .findAll()
            .filter((node) => aws_cdk_lib_1.Stack.isStack(node))
            .map((stack) => stack.stackName);
        new workflow_1.PipelineWorkflow(this.adapter, this.workflowFilename, {
            name: this.workflowName,
            commentAtTop: this.renderYamlComment(names),
            pipeline: this,
            stackOptions: this.stackOptions,
            preBuild: this.preBuild,
            postBuild: this.postBuild,
            cdkoutDir: app.outdir,
            awsCredentials: this.awsCredentials,
        });
    }
    /**
     * Renders a YAML comment for the workflow file.
     *
     * @param stackNames - The names of the stacks deployed by this pipeline.
     * @returns A string containing the rendered YAML comment.
     */
    renderYamlComment(stackNames) {
        const header = "Generated by github-actions-cdk, DO NOT EDIT DIRECTLY!\n\n";
        const stackListHeader = "Deployed stacks from this pipeline:\n";
        const stackList = stackNames.map((stack) => `- ${stack}`).join("\n");
        return header + stackListHeader + stackList;
    }
    /**
     * Adds properties to the stack options for each stack in the deployment.
     *
     * @param stacks - The list of stack deployments.
     * @param key - The property key to set.
     * @param value - The value to assign to the property.
     */
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackOptions[stack.stackArtifactId] = {
                ...this.stackOptions[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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