"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const github_actions_cdk_1 = require("github-actions-cdk");
const jobs_1 = require("./jobs");
const steps_1 = require("./steps");
/**
 * Represents a workflow that manages the pipeline jobs for synthesizing, publishing, and deploying resources.
 *
 * This class extends the Workflow class from the GitHub Actions CDK to provide
 * a structured way of orchestrating pipeline jobs based on the AWS CDK pipeline graph.
 */
class PipelineWorkflow extends github_actions_cdk_1.Workflow {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.PipelineWorkflow", version: "0.0.19" };
    awsCredentials;
    cdkoutDir;
    stackOptions;
    /**
     * Constructs a new instance of PipelineWorkflow.
     *
     * @param scope - The scope in which this construct is defined.
     * @param id - Unique identifier for this workflow.
     * @param props - Properties for configuring the pipeline workflow.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.awsCredentials = props.awsCredentials;
        this.cdkoutDir = props.cdkoutDir;
        this.stackOptions = props.stackOptions;
        const structure = new helpers_internal_1.PipelineGraph(props.pipeline, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false, // The changeset is created and executed in a single job.
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!(0, helpers_internal_1.isGraph)(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    switch (node.data?.type) {
                        case "step":
                            if (node.data?.isBuildStep) {
                                if (node.data?.step instanceof steps_1.Synth) {
                                    this.jobForSynth(node.uniqueId, this.renderDependencies(node), node.data.step, props.preBuild, props.postBuild);
                                }
                                else {
                                    throw new Error("Only SynthStep is supported as a build step");
                                }
                            }
                            else if (node.data?.step instanceof steps_1.StageJob) {
                                this.jobForStage(node.uniqueId, this.renderDependencies(node), node.data.step);
                            }
                            break;
                        case "publish-assets":
                            this.jobForPublish(node.uniqueId, this.renderDependencies(node), node.data.assets);
                            break;
                        case "execute":
                            this.jobForDeploy(node.uniqueId, this.renderDependencies(node), node.data.stack);
                            break;
                        default:
                            throw new Error(`Unknown node type: ${node.data?.type}`);
                    }
                }
            }
        }
    }
    /**
     * Creates a job for synthesizing the application.
     *
     * @param id - Unique identifier for the synth job.
     * @param needs - List of job IDs that this job depends on.
     * @param synth - The Synth step configuration.
     * @param preBuild - Optional jobs to run before the synth job.
     * @param postBuild - Optional jobs to run after the synth job.
     */
    jobForSynth(id, needs, synth, preBuild, postBuild) {
        new jobs_1.SynthPipelineJob(this, id, {
            name: "Synthesize",
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            env: synth.env,
            preBuild,
            postBuild,
            installCommands: synth.installCommands,
            commands: synth.commands,
            awsCredentials: this.awsCredentials,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for publishing stack assets.
     *
     * @param id - Unique identifier for the publish job.
     * @param needs - List of job IDs that this job depends on.
     * @param assets - The stack assets to publish.
     */
    jobForPublish(id, needs, assets) {
        new jobs_1.PublishPipelineJob(this, id, {
            name: `Publish Assets ${id}`,
            needs,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            assets,
            awsCredentials: this.awsCredentials,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for deploying a stack.
     *
     * @param id - Unique identifier for the deploy job.
     * @param needs - List of job IDs that this job depends on.
     * @param stack - The stack deployment information.
     */
    jobForDeploy(id, needs, stack) {
        const stackOptions = this.stackOptions[stack.stackArtifactId];
        new jobs_1.DeployPipelineJob(this, id, {
            name: `Deploy ${stack.stackArtifactId}`,
            needs,
            environment: stackOptions?.environment,
            permissions: {
                contents: github_actions_cdk_1.PermissionLevel.READ,
                idToken: this.awsCredentials.permissionLevel(),
            },
            stack,
            stackOptions,
            awsCredentials: this.awsCredentials,
            cdkoutDir: this.cdkoutDir,
        });
    }
    /**
     * Creates a job for running a stage job.
     *
     * @param id - Unique identifier for the stage job.
     * @param needs - List of job IDs that this job depends on.
     * @param job - The stage job configuration.
     */
    jobForStage(id, needs, job) {
        new jobs_1.StagePipelineJob(this, id, {
            name: job.id,
            needs,
            phase: job.props,
            awsCredentials: this.awsCredentials,
            cdkoutDir: this.cdkoutDir,
            ...job.props,
        });
    }
    /**
     * Renders the dependencies for a given node in the graph.
     *
     * @param node - The graph node whose dependencies need to be rendered.
     * @returns An array of unique IDs representing the node's dependencies.
     */
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map((x) => x.uniqueId);
    }
}
exports.PipelineWorkflow = PipelineWorkflow;
/**
 * Flattens an iterable of arrays into a single iterable.
 *
 * @param xs - The input iterable containing arrays to flatten.
 * @returns An iterable of the flattened elements.
 */
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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