import * as fs from 'fs';
import * as path from 'path';
import { createHash } from 'crypto';
import { createFilter } from '@rollup/pluginutils';

const HELPERS_ID = '\0wasmHelpers.js';
const nodeFilePath = `
var fs = require("fs")
var path = require("path")

return new Promise((resolve, reject) => {
  fs.readFile(path.resolve(__dirname, filepath), (error, buffer) => {
    if (error != null) {
      reject(error)
    } else {
      resolve(_instantiateOrCompile(buffer, imports, false))
    }
  });
});
`;
const nodeDecode = `
buf = Buffer.from(src, 'base64')
`;
const browserFilePath = `
return _instantiateOrCompile(fetch(filepath), imports, true);
`;
const browserDecode = `
var raw = globalThis.atob(src)
var rawLength = raw.length
buf = new Uint8Array(new ArrayBuffer(rawLength))
for(var i = 0; i < rawLength; i++) {
   buf[i] = raw.charCodeAt(i)
}
`;
const autoModule = `
var buf = null
var isNode = typeof process !== 'undefined' && process.versions != null && process.versions.node != null

if (filepath && isNode) {
  ${nodeFilePath}
} else if (filepath) {
  ${browserFilePath}
}

if (isNode) {
  ${nodeDecode}
} else {
  ${browserDecode}
}
`;
const nodeModule = `
var buf = null
if (filepath) {
  ${nodeFilePath}
}

${nodeDecode}
`;
const browserModule = `
var buf = null
if (filepath) {
  ${browserFilePath}
}

${browserDecode}
`;
const autoInlineModule = `
var buf = null
var isNode = typeof process !== 'undefined' && process.versions != null && process.versions.node != null
if (isNode) {
  ${nodeDecode}
} else {
  ${browserDecode}
}
`;
const envModule = (env) => {
    switch (env) {
        case 'auto':
            return autoModule;
        case 'auto-inline':
            return autoInlineModule;
        case 'browser':
            return browserModule;
        case 'node':
            return nodeModule;
        default:
            return null;
    }
};
const getHelpersModule = (env) => `
function _loadWasmModule (sync, filepath, src, imports) {
  function _instantiateOrCompile(source, imports, stream) {
    var instantiateFunc = stream ? WebAssembly.instantiateStreaming : WebAssembly.instantiate;
    var compileFunc = stream ? WebAssembly.compileStreaming : WebAssembly.compile;

    if (imports) {
      return instantiateFunc(source, imports)
    } else {
      return compileFunc(source)
    }
  }

  ${envModule(env)}

  if(sync) {
    var mod = new WebAssembly.Module(buf)
    return imports ? new WebAssembly.Instance(mod, imports) : mod
  } else {
    return _instantiateOrCompile(buf, imports, false)
  }
}
export { _loadWasmModule };
`;

function wasm(options = {}) {
    const { sync = [], maxFileSize = 14 * 1024, publicPath = '', targetEnv = 'auto', fileName = '[hash][extname]' } = options;
    const syncFiles = sync.map((x) => path.resolve(x));
    const copies = Object.create(null);
    const filter = createFilter(options.include, options.exclude);
    return {
        name: 'wasm',
        resolveId(id) {
            if (id === HELPERS_ID) {
                return id;
            }
            return null;
        },
        load(id) {
            if (id === HELPERS_ID) {
                return getHelpersModule(targetEnv);
            }
            if (!filter(id)) {
                return null;
            }
            if (!/\.wasm$/.test(id)) {
                return null;
            }
            this.addWatchFile(id);
            return Promise.all([fs.promises.stat(id), fs.promises.readFile(id)]).then(([stats, buffer]) => {
                if (targetEnv === 'auto-inline') {
                    return buffer.toString('binary');
                }
                if ((maxFileSize && stats.size > maxFileSize) || maxFileSize === 0) {
                    const hash = createHash('sha1').update(buffer).digest('hex').substr(0, 16);
                    const ext = path.extname(id);
                    const name = path.basename(id, ext);
                    const outputFileName = fileName
                        .replace(/\[hash\]/g, hash)
                        .replace(/\[extname\]/g, ext)
                        .replace(/\[name\]/g, name);
                    const publicFilepath = `${publicPath}${outputFileName}`;
                    // only copy if the file is not marked `sync`, `sync` files are always inlined
                    if (syncFiles.indexOf(id) === -1) {
                        copies[id] = {
                            filename: outputFileName,
                            publicFilepath,
                            buffer
                        };
                    }
                }
                return buffer.toString('binary');
            });
        },
        transform(code, id) {
            if (!filter(id)) {
                return null;
            }
            if (code && /\.wasm$/.test(id)) {
                const isSync = syncFiles.indexOf(id) !== -1;
                const publicFilepath = copies[id] ? `'${copies[id].publicFilepath}'` : null;
                let src;
                if (publicFilepath === null) {
                    src = Buffer.from(code, 'binary').toString('base64');
                    src = `'${src}'`;
                }
                else {
                    if (isSync) {
                        this.error('non-inlined files can not be `sync`.');
                    }
                    src = null;
                }
                return {
                    map: {
                        mappings: ''
                    },
                    code: `import { _loadWasmModule } from ${JSON.stringify(HELPERS_ID)};
export default function(imports){return _loadWasmModule(${+isSync}, ${publicFilepath}, ${src}, imports)}`
                };
            }
            return null;
        },
        generateBundle: async function write() {
            await Promise.all(Object.keys(copies).map(async (name) => {
                const copy = copies[name];
                this.emitFile({
                    type: 'asset',
                    source: copy.buffer,
                    name: 'Rollup WASM Asset',
                    fileName: copy.filename
                });
            }));
        }
    };
}

export { wasm as default, wasm };
//# sourceMappingURL=index.js.map
