import { execSync } from 'node:child_process';
import { d as defineCommand, c as consola, b as colors } from '../shared/nuxi.9d864d83.mjs';
import 'node:util';
import 'node:path';
import 'node:process';
import 'node:tty';
import { r as readPackageJSON } from '../shared/nuxi.d499a257.mjs';
import { g as getPackageManager, p as packageManagerLocks } from '../shared/nuxi.5045634f.mjs';
import { r as rmRecursive, t as touchFile } from '../shared/nuxi.c6c42c8d.mjs';
import { c as cleanupNuxtDirs, n as nuxtVersionToGitIdentifier } from '../shared/nuxi.f51a69f9.mjs';
import { s as sharedArgs, l as legacyRootDirArgs, r as resolve, a as relative } from '../shared/nuxi.3f4caf85.mjs';
import 'node:url';
import 'node:fs';
import '../shared/nuxi.4456790d.mjs';
import 'node:module';
import '../shared/nuxi.ae708cae.mjs';
import 'node:assert';
import 'node:v8';

async function getNuxtVersion(path) {
  try {
    const pkg = await readPackageJSON("nuxt", { url: path });
    if (!pkg.version) {
      consola.warn("Cannot find any installed nuxt versions in ", path);
    }
    return pkg.version || null;
  } catch {
    return null;
  }
}
const upgrade = defineCommand({
  meta: {
    name: "upgrade",
    description: "Upgrade nuxt"
  },
  args: {
    ...sharedArgs,
    ...legacyRootDirArgs,
    force: {
      type: "boolean",
      alias: "f",
      description: "Force upgrade to recreate lockfile and node_modules"
    }
  },
  async run(ctx) {
    const cwd = resolve(ctx.args.cwd || ctx.args.rootDir || ".");
    const packageManager = getPackageManager(cwd);
    if (!packageManager) {
      console.error("Cannot detect Package Manager in", cwd);
      process.exit(1);
    }
    const packageManagerVersion = execSync(`${packageManager} --version`).toString("utf8").trim();
    consola.info("Package Manager:", packageManager, packageManagerVersion);
    const currentVersion = await getNuxtVersion(cwd) || "[unknown]";
    consola.info("Current nuxt version:", currentVersion);
    const pmLockFile = resolve(cwd, packageManagerLocks[packageManager]);
    const forceRemovals = ["node_modules", relative(process.cwd(), pmLockFile)].map((p) => colors.cyan(p)).join(" and ");
    if (ctx.args.force === void 0) {
      ctx.args.force = await consola.prompt(
        `Would you like to recreate ${forceRemovals} to fix problems with hoisted dependency versions and ensure you have the most up-to-date dependencies?`,
        {
          type: "confirm",
          default: true
        }
      );
    }
    if (ctx.args.force) {
      consola.info(
        `Recreating ${forceRemovals}. If you encounter any issues, revert the changes and try with \`--no-force\``
      );
      await rmRecursive([pmLockFile, resolve(cwd, "node_modules")]);
      await touchFile(pmLockFile);
    }
    consola.info("Installing latest Nuxt 3 release...");
    execSync(
      `${packageManager} ${packageManager === "yarn" ? "add" : "install"} -D nuxt`,
      { stdio: "inherit", cwd }
    );
    await cleanupNuxtDirs(cwd);
    const upgradedVersion = await getNuxtVersion(cwd) || "[unknown]";
    consola.info("Upgraded nuxt version:", upgradedVersion);
    if (upgradedVersion === currentVersion) {
      consola.success("You're already using the latest version of nuxt.");
    } else {
      consola.success(
        "Successfully upgraded nuxt from",
        currentVersion,
        "to",
        upgradedVersion
      );
      const commitA = nuxtVersionToGitIdentifier(currentVersion);
      const commitB = nuxtVersionToGitIdentifier(upgradedVersion);
      if (commitA && commitB) {
        consola.info(
          "Changelog:",
          `https://github.com/nuxt/nuxt/compare/${commitA}...${commitB}`
        );
      }
    }
  }
});

export { upgrade as default };
