"use strict";
// Copyright 2020 Google Inc. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.AsyncEmbeddedCompiler = void 0;
const child_process_1 = require("child_process");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const compiler_path_1 = require("./compiler-path");
/**
 * An asynchronous wrapper for the embedded Sass compiler that exposes its stdio
 * streams as Observables.
 */
class AsyncEmbeddedCompiler {
    constructor() {
        /** The underlying process that's being wrapped. */
        this.process = (0, child_process_1.spawn)(compiler_path_1.compilerPath, { windowsHide: true });
        /** The child process's exit event. */
        this.exit$ = new Promise(resolve => {
            this.process.on('exit', code => resolve(code));
        });
        /** The buffers emitted by the child process's stdout. */
        this.stdout$ = new rxjs_1.Observable(observer => {
            this.process.stdout.on('data', buffer => observer.next(buffer));
        }).pipe((0, operators_1.takeUntil)(this.exit$));
        /** The buffers emitted by the child process's stderr. */
        this.stderr$ = new rxjs_1.Observable(observer => {
            this.process.stderr.on('data', buffer => observer.next(buffer));
        }).pipe((0, operators_1.takeUntil)(this.exit$));
    }
    /** Writes `buffer` to the child process's stdin. */
    writeStdin(buffer) {
        this.process.stdin.write(buffer);
    }
    /** Kills the child process, cleaning up all associated Observables. */
    close() {
        this.process.stdin.end();
    }
}
exports.AsyncEmbeddedCompiler = AsyncEmbeddedCompiler;
//# sourceMappingURL=async-compiler.js.map