"use strict";
// Copyright 2020 Google Inc. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dispatcher = void 0;
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const embedded_sass_pb_1 = require("./vendor/embedded-protocol/embedded_sass_pb");
const request_tracker_1 = require("./request-tracker");
const utils_1 = require("./utils");
/**
 * Dispatches requests, responses, and events.
 *
 * Accepts callbacks for processing different types of outbound requests. When
 * an outbound request arrives, this runs the appropriate callback to process
 * it, and then sends the result inbound. A single callback must be provided for
 * each outbound request type. The callback does not need to set the response
 * ID; the dispatcher handles it.
 *
 * Consumers can send an inbound request. This returns a promise that will
 * either resolve with the corresponding outbound response, or error if any
 * Protocol Errors were encountered. The consumer does not need to set the
 * request ID; the dispatcher handles it.
 *
 * Outbound events are exposed as Observables.
 *
 * Errors are not otherwise exposed to the top-level. Instead, they are surfaced
 * as an Observable that consumers may choose to subscribe to. Subscribers must
 * perform proper error handling.
 */
class Dispatcher {
    constructor(outboundMessages$, writeInboundMessage, outboundRequestHandlers) {
        this.outboundMessages$ = outboundMessages$;
        this.writeInboundMessage = writeInboundMessage;
        this.outboundRequestHandlers = outboundRequestHandlers;
        // Tracks the IDs of all inbound requests. An outbound response with matching
        // ID and type will remove the ID.
        this.pendingInboundRequests = new request_tracker_1.RequestTracker();
        // Tracks the IDs of all outbound requests. An inbound response with matching
        // ID and type will remove the ID.
        this.pendingOutboundRequests = new request_tracker_1.RequestTracker();
        // All outbound messages. If we detect any errors while dispatching messages,
        // this completes.
        this.messages$ = new rxjs_1.Subject();
        // If the dispatcher encounters an error, this errors out. It is publicly
        // exposed as a readonly Observable.
        this.errorInternal$ = new rxjs_1.Subject();
        /**
         * If the dispatcher encounters an error, this errors out. Upon error, the
         * dispatcher rejects all promises awaiting an outbound response, and silently
         * closes all subscriptions to outbound events.
         */
        this.error$ = this.errorInternal$.pipe();
        /**
         * Outbound log events. If an error occurs, the dispatcher closes this
         * silently.
         */
        this.logEvents$ = this.messages$.pipe((0, operators_1.filter)(message => message.type === embedded_sass_pb_1.OutboundMessage.MessageCase.LOG_EVENT), (0, operators_1.map)(message => message.payload));
        this.outboundMessages$
            .pipe((0, operators_1.mergeMap)(message => {
            const result = this.handleOutboundMessage(message);
            return result instanceof Promise
                ? result.then(() => message)
                : [message];
        }))
            .subscribe({
            next: message => this.messages$.next(message),
            error: error => this.throwAndClose(error),
            complete: () => {
                this.messages$.complete();
                this.errorInternal$.complete();
            },
        });
    }
    /**
     * Sends a CompileRequest inbound. Passes the corresponding outbound
     * CompileResponse or an error to `callback`.
     *
     * This uses an old-style callback argument so that it can work either
     * synchronously or asynchronously. If the underlying stdout stream emits
     * events synchronously, `callback` will be called synchronously.
     */
    sendCompileRequest(request, callback) {
        this.handleInboundRequest(request, embedded_sass_pb_1.InboundMessage.MessageCase.COMPILE_REQUEST, embedded_sass_pb_1.OutboundMessage.MessageCase.COMPILE_RESPONSE, callback);
    }
    // Rejects with `error` all promises awaiting an outbound response, and
    // silently closes all subscriptions awaiting outbound events.
    throwAndClose(error) {
        this.messages$.complete();
        this.errorInternal$.error(error);
    }
    // Keeps track of all outbound messages. If the outbound `message` contains a
    // request or response, registers it with pendingOutboundRequests. If it
    // contains a request, runs the appropriate callback to generate an inbound
    // response, and then sends it inbound.
    handleOutboundMessage(message) {
        switch (message.type) {
            case embedded_sass_pb_1.OutboundMessage.MessageCase.LOG_EVENT:
                return undefined;
            case embedded_sass_pb_1.OutboundMessage.MessageCase.COMPILE_RESPONSE:
                this.pendingInboundRequests.resolve(message.payload.getId(), message.type);
                return undefined;
            case embedded_sass_pb_1.OutboundMessage.MessageCase.IMPORT_REQUEST: {
                const request = message.payload;
                const id = request.getId();
                const type = embedded_sass_pb_1.InboundMessage.MessageCase.IMPORT_RESPONSE;
                this.pendingOutboundRequests.add(id, type);
                return (0, utils_1.thenOr)(this.outboundRequestHandlers.handleImportRequest(request), response => {
                    this.sendInboundMessage(id, response, type);
                });
            }
            case embedded_sass_pb_1.OutboundMessage.MessageCase.FILE_IMPORT_REQUEST: {
                const request = message.payload;
                const id = request.getId();
                const type = embedded_sass_pb_1.InboundMessage.MessageCase.FILE_IMPORT_RESPONSE;
                this.pendingOutboundRequests.add(id, type);
                return (0, utils_1.thenOr)(this.outboundRequestHandlers.handleFileImportRequest(request), response => {
                    this.sendInboundMessage(id, response, type);
                });
            }
            case embedded_sass_pb_1.OutboundMessage.MessageCase.CANONICALIZE_REQUEST: {
                const request = message.payload;
                const id = request.getId();
                const type = embedded_sass_pb_1.InboundMessage.MessageCase.CANONICALIZE_RESPONSE;
                this.pendingOutboundRequests.add(id, type);
                return (0, utils_1.thenOr)(this.outboundRequestHandlers.handleCanonicalizeRequest(request), response => {
                    this.sendInboundMessage(id, response, type);
                });
            }
            case embedded_sass_pb_1.OutboundMessage.MessageCase.FUNCTION_CALL_REQUEST: {
                const request = message.payload;
                const id = request.getId();
                const type = embedded_sass_pb_1.InboundMessage.MessageCase.FUNCTION_CALL_RESPONSE;
                this.pendingOutboundRequests.add(id, type);
                return (0, utils_1.thenOr)(this.outboundRequestHandlers.handleFunctionCallRequest(request), response => {
                    this.sendInboundMessage(id, response, type);
                });
            }
            default:
                throw Error(`Unknown message type ${message.type}`);
        }
    }
    // Sends a `request` of type `requestType` inbound. Returns a promise that
    // will either resolve with the corresponding outbound response of type
    // `responseType`, or error if any Protocol Errors were encountered.
    handleInboundRequest(request, requestType, responseType, callback) {
        if (this.messages$.isStopped) {
            callback(new Error('Tried writing to closed dispatcher'), undefined);
            return;
        }
        this.messages$
            .pipe((0, operators_1.filter)(message => message.type === responseType), (0, operators_1.map)(message => message.payload), (0, operators_1.filter)(response => response.getId() === request.getId()))
            .subscribe({ next: response => callback(null, response) });
        this.error$.subscribe({ error: error => callback(error, undefined) });
        try {
            this.sendInboundMessage(this.pendingInboundRequests.nextId, request, requestType);
        }
        catch (error) {
            this.throwAndClose(error);
        }
    }
    // Sends a message inbound. Keeps track of all pending inbound requests.
    sendInboundMessage(id, payload, type) {
        payload.setId(id);
        if (type === embedded_sass_pb_1.InboundMessage.MessageCase.COMPILE_REQUEST) {
            this.pendingInboundRequests.add(id, embedded_sass_pb_1.OutboundMessage.MessageCase.COMPILE_RESPONSE);
        }
        else if (type === embedded_sass_pb_1.InboundMessage.MessageCase.IMPORT_RESPONSE ||
            type === embedded_sass_pb_1.InboundMessage.MessageCase.FILE_IMPORT_RESPONSE ||
            type === embedded_sass_pb_1.InboundMessage.MessageCase.CANONICALIZE_RESPONSE ||
            type === embedded_sass_pb_1.InboundMessage.MessageCase.FUNCTION_CALL_RESPONSE) {
            this.pendingOutboundRequests.resolve(id, type);
        }
        else {
            throw Error(`Unknown message type ${type}`);
        }
        this.writeInboundMessage({
            payload,
            type,
        });
    }
}
exports.Dispatcher = Dispatcher;
//# sourceMappingURL=dispatcher.js.map