"use strict";
// Copyright 2021 Google Inc. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.FunctionRegistry = void 0;
const util_1 = require("util");
const utils = require("./utils");
const embedded_sass_pb_1 = require("./vendor/embedded-protocol/embedded_sass_pb");
const utils_1 = require("./utils");
const protofier_1 = require("./protofier");
const value_1 = require("./value");
/**
 * The next ID to use for a function. The embedded protocol requires that
 * function IDs be globally unique.
 */
let nextFunctionID = 0;
/**
 * Tracks functions that are defined on the host so that the compiler can
 * execute them.
 */
class FunctionRegistry {
    constructor(functionsBySignature) {
        this.functionsByName = new Map();
        this.functionsById = new Map();
        this.idsByFunction = new Map();
        for (const [signature, fn] of Object.entries(functionsBySignature !== null && functionsBySignature !== void 0 ? functionsBySignature : {})) {
            const openParen = signature.indexOf('(');
            if (openParen === -1) {
                throw new Error(`options.functions: "${signature}" is missing "("`);
            }
            this.functionsByName.set(signature.substring(0, openParen), fn);
        }
    }
    /** Registers `fn` as a function that can be called using the returned ID. */
    register(fn) {
        return utils.putIfAbsent(this.idsByFunction, fn, () => {
            const id = nextFunctionID;
            nextFunctionID += 1;
            this.functionsById.set(id, fn);
            return id;
        });
    }
    /**
     * Returns the function to which `request` refers and returns its response.
     */
    call(request) {
        const protofier = new protofier_1.Protofier(this);
        const fn = this.get(request);
        return (0, utils_1.catchOr)(() => {
            return (0, utils_1.thenOr)(fn(request
                .getArgumentsList()
                .map(value => protofier.deprotofy(value))), result => {
                if (!(result instanceof value_1.Value)) {
                    const name = request.getName().length === 0
                        ? 'anonymous function'
                        : `"${request.getName()}"`;
                    throw (`options.functions: ${name} returned non-Value: ` +
                        (0, util_1.inspect)(result));
                }
                const response = new embedded_sass_pb_1.InboundMessage.FunctionCallResponse();
                response.setSuccess(protofier.protofy(result));
                response.setAccessedArgumentListsList(protofier.accessedArgumentLists);
                return response;
            });
        }, error => {
            const response = new embedded_sass_pb_1.InboundMessage.FunctionCallResponse();
            response.setError(`${error}`);
            return response;
        });
    }
    /** Returns the function to which `request` refers. */
    get(request) {
        if (request.getIdentifierCase() ===
            embedded_sass_pb_1.OutboundMessage.FunctionCallRequest.IdentifierCase.NAME) {
            const fn = this.functionsByName.get(request.getName());
            if (fn)
                return fn;
            throw new Error('Invalid OutboundMessage.FunctionCallRequest: there is no function ' +
                `named "${request.getName()}"`);
        }
        else {
            const fn = this.functionsById.get(request.getFunctionId());
            if (fn)
                return fn;
            throw new Error('Invalid OutboundMessage.FunctionCallRequest: there is no function ' +
                `with ID "${request.getFunctionId()}"`);
        }
    }
}
exports.FunctionRegistry = FunctionRegistry;
//# sourceMappingURL=function-registry.js.map