"use strict";
// Copyright 2021 Google LLC. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImporterRegistry = void 0;
const p = require("path");
const url_1 = require("url");
const util_1 = require("util");
const utils = require("./utils");
const embedded_sass_pb_1 = require("./vendor/embedded-protocol/embedded_sass_pb");
const utils_1 = require("./utils");
/**
 * A registry of importers defined in the host that can be invoked by the
 * compiler.
 */
class ImporterRegistry {
    constructor(options) {
        var _a, _b;
        /** A map from importer IDs to their corresponding importers. */
        this.importersById = new Map();
        /** A map from file importer IDs to their corresponding importers. */
        this.fileImportersById = new Map();
        /** The next ID to use for an importer. */
        this.id = 0;
        this.importers = ((_a = options === null || options === void 0 ? void 0 : options.importers) !== null && _a !== void 0 ? _a : [])
            .map(importer => this.register(importer))
            .concat(((_b = options === null || options === void 0 ? void 0 : options.loadPaths) !== null && _b !== void 0 ? _b : []).map(path => {
            const proto = new embedded_sass_pb_1.InboundMessage.CompileRequest.Importer();
            proto.setPath(p.resolve(path));
            return proto;
        }));
    }
    /** Converts an importer to a proto without adding it to `this.importers`. */
    register(importer) {
        const proto = new embedded_sass_pb_1.InboundMessage.CompileRequest.Importer();
        if ('canonicalize' in importer) {
            if ('findFileUrl' in importer) {
                throw new Error('Importer may not contain both canonicalize() and findFileUrl(): ' +
                    (0, util_1.inspect)(importer));
            }
            proto.setImporterId(this.id);
            this.importersById.set(this.id, importer);
        }
        else {
            proto.setFileImporterId(this.id);
            this.fileImportersById.set(this.id, importer);
        }
        this.id += 1;
        return proto;
    }
    /** Handles a canonicalization request. */
    canonicalize(request) {
        const importer = this.importersById.get(request.getImporterId());
        if (!importer) {
            throw utils.compilerError('Unknown CanonicalizeRequest.importer_id');
        }
        return (0, utils_1.catchOr)(() => {
            return (0, utils_1.thenOr)(importer.canonicalize(request.getUrl(), {
                fromImport: request.getFromImport(),
            }), url => {
                const proto = new embedded_sass_pb_1.InboundMessage.CanonicalizeResponse();
                if (url !== null)
                    proto.setUrl(url.toString());
                return proto;
            });
        }, error => {
            const proto = new embedded_sass_pb_1.InboundMessage.CanonicalizeResponse();
            proto.setError(`${error}`);
            return proto;
        });
    }
    /** Handles an import request. */
    import(request) {
        const importer = this.importersById.get(request.getImporterId());
        if (!importer) {
            throw utils.compilerError('Unknown ImportRequest.importer_id');
        }
        return (0, utils_1.catchOr)(() => {
            return (0, utils_1.thenOr)(importer.load(new url_1.URL(request.getUrl())), result => {
                const proto = new embedded_sass_pb_1.InboundMessage.ImportResponse();
                if (result) {
                    if (typeof result.contents !== 'string') {
                        throw Error(`Invalid argument (contents): must be a string but was: ${result.contents.constructor.name}`);
                    }
                    if (result.sourceMapUrl && !result.sourceMapUrl.protocol) {
                        throw Error('Invalid argument (sourceMapUrl): must be absolute but was: ' +
                            result.sourceMapUrl);
                    }
                    const success = new embedded_sass_pb_1.InboundMessage.ImportResponse.ImportSuccess();
                    success.setContents(result.contents);
                    success.setSyntax(utils.protofySyntax(result.syntax));
                    if (result.sourceMapUrl) {
                        success.setSourceMapUrl(result.sourceMapUrl.toString());
                    }
                    proto.setSuccess(success);
                }
                return proto;
            });
        }, error => {
            const proto = new embedded_sass_pb_1.InboundMessage.ImportResponse();
            proto.setError(`${error}`);
            return proto;
        });
    }
    /** Handles a file import request. */
    fileImport(request) {
        const importer = this.fileImportersById.get(request.getImporterId());
        if (!importer) {
            throw utils.compilerError('Unknown FileImportRequest.importer_id');
        }
        return (0, utils_1.catchOr)(() => {
            return (0, utils_1.thenOr)(importer.findFileUrl(request.getUrl(), {
                fromImport: request.getFromImport(),
            }), url => {
                const proto = new embedded_sass_pb_1.InboundMessage.FileImportResponse();
                if (url) {
                    if (url.protocol !== 'file:') {
                        throw (`FileImporter ${(0, util_1.inspect)(importer)} returned non-file: URL ` +
                            +`"${url}" for URL "${request.getUrl()}".`);
                    }
                    proto.setFileUrl(url.toString());
                }
                return proto;
            });
        }, error => {
            const proto = new embedded_sass_pb_1.InboundMessage.FileImportResponse();
            proto.setError(`${error}`);
            return proto;
        });
    }
}
exports.ImporterRegistry = ImporterRegistry;
//# sourceMappingURL=importer-registry.js.map