"use strict";
// Copyright 2020 Google Inc. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageTransformer = void 0;
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const utils_1 = require("./utils");
const embedded_sass_pb_1 = require("./vendor/embedded-protocol/embedded_sass_pb");
/**
 * Encodes InboundTypedMessages into protocol buffers and decodes protocol
 * buffers into OutboundTypedMessages. Any Embedded Protocol violations that can
 * be detected at the message level are encapsulated here and reported as
 * errors.
 *
 * This transformer communicates via In/OutboundTypedMessages instead of raw
 * In/OutboundMessages in order to expose more type information to consumers.
 * This makes the stream of messages from the transformer easier to interact
 * with.
 */
class MessageTransformer {
    constructor(outboundProtobufs$, writeInboundProtobuf) {
        this.outboundProtobufs$ = outboundProtobufs$;
        this.writeInboundProtobuf = writeInboundProtobuf;
        // The decoded messages are written to this Subject. It is publicly exposed
        // as a readonly Observable.
        this.outboundMessagesInternal$ = new rxjs_1.Subject();
        /**
         * The OutboundTypedMessages, decoded from protocol buffers. If any errors are
         * detected while encoding/decoding, this Observable will error out.
         */
        this.outboundMessages$ = this.outboundMessagesInternal$.pipe();
        this.outboundProtobufs$
            .pipe((0, operators_1.map)(decode))
            .subscribe(this.outboundMessagesInternal$);
    }
    /**
     * Converts the inbound `message` to a protocol buffer.
     */
    writeInboundMessage(message) {
        try {
            this.writeInboundProtobuf(encode(message));
        }
        catch (error) {
            this.outboundMessagesInternal$.error(error);
        }
    }
}
exports.MessageTransformer = MessageTransformer;
// Decodes a protobuf `buffer` into an OutboundTypedMessage, ensuring that all
// mandatory message fields are populated. Throws if `buffer` cannot be decoded
// into a valid message, or if the message itself contains a Protocol Error.
function decode(buffer) {
    var _a, _b, _c;
    let message;
    try {
        message = embedded_sass_pb_1.OutboundMessage.deserializeBinary(buffer);
    }
    catch (error) {
        throw (0, utils_1.compilerError)('Invalid buffer');
    }
    let payload;
    const type = message.getMessageCase();
    switch (type) {
        case embedded_sass_pb_1.OutboundMessage.MessageCase.LOG_EVENT:
            payload = message.getLogEvent();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.COMPILE_RESPONSE:
            if (((_a = message.getCompileResponse()) === null || _a === void 0 ? void 0 : _a.getResultCase()) ===
                embedded_sass_pb_1.OutboundMessage.CompileResponse.ResultCase.RESULT_NOT_SET) {
                throw (0, utils_1.compilerError)('OutboundMessage.CompileResponse.result is not set');
            }
            payload = message.getCompileResponse();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.IMPORT_REQUEST:
            payload = message.getImportRequest();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.FILE_IMPORT_REQUEST:
            payload = message.getFileImportRequest();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.CANONICALIZE_REQUEST:
            payload = message.getCanonicalizeRequest();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.FUNCTION_CALL_REQUEST:
            if (((_b = message.getFunctionCallRequest()) === null || _b === void 0 ? void 0 : _b.getIdentifierCase()) ===
                embedded_sass_pb_1.OutboundMessage.FunctionCallRequest.IdentifierCase.IDENTIFIER_NOT_SET) {
                throw (0, utils_1.compilerError)('OutboundMessage.FunctionCallRequest.identifier is not set');
            }
            payload = message.getFunctionCallRequest();
            break;
        case embedded_sass_pb_1.OutboundMessage.MessageCase.ERROR:
            throw (0, utils_1.hostError)(`${(_c = message.getError()) === null || _c === void 0 ? void 0 : _c.getMessage()}`);
        case embedded_sass_pb_1.OutboundMessage.MessageCase.MESSAGE_NOT_SET:
            throw (0, utils_1.compilerError)('OutboundMessage.message is not set');
        default:
            throw (0, utils_1.compilerError)(`Unknown message type ${message.toString()}`);
    }
    if (!payload)
        throw (0, utils_1.compilerError)('OutboundMessage missing payload');
    return {
        payload,
        type,
    };
}
// Encodes an InboundTypedMessage into a protocol buffer.
function encode(message) {
    const inboundMessage = new embedded_sass_pb_1.InboundMessage();
    switch (message.type) {
        case embedded_sass_pb_1.InboundMessage.MessageCase.COMPILE_REQUEST:
            inboundMessage.setCompileRequest(message.payload);
            break;
        case embedded_sass_pb_1.InboundMessage.MessageCase.IMPORT_RESPONSE:
            inboundMessage.setImportResponse(message.payload);
            break;
        case embedded_sass_pb_1.InboundMessage.MessageCase.FILE_IMPORT_RESPONSE:
            inboundMessage.setFileImportResponse(message.payload);
            break;
        case embedded_sass_pb_1.InboundMessage.MessageCase.CANONICALIZE_RESPONSE:
            inboundMessage.setCanonicalizeResponse(message.payload);
            break;
        case embedded_sass_pb_1.InboundMessage.MessageCase.FUNCTION_CALL_RESPONSE:
            inboundMessage.setFunctionCallResponse(message.payload);
            break;
    }
    return Buffer.from(inboundMessage.serializeBinary());
}
//# sourceMappingURL=message-transformer.js.map