"use strict";
// Copyright 2021 Google Inc. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protofier = void 0;
const immutable_1 = require("immutable");
const proto = require("./vendor/embedded-protocol/embedded_sass_pb");
const utils = require("./utils");
const argument_list_1 = require("./value/argument-list");
const color_1 = require("./value/color");
const function_1 = require("./value/function");
const list_1 = require("./value/list");
const map_1 = require("./value/map");
const number_1 = require("./value/number");
const string_1 = require("./value/string");
const null_1 = require("./value/null");
const boolean_1 = require("./value/boolean");
/**
 * A class that converts [Value] objects into protobufs.
 *
 * A given [Protofier] instance is valid only within the scope of a single
 * custom function call.
 */
class Protofier {
    constructor(
    /**
     * The registry of custom functions that can be invoked by the compiler.
     * This is used to register first-class functions so that the compiler may
     * invoke them.
     */
    functions) {
        this.functions = functions;
        /** All the argument lists returned by `deprotofy()`. */
        this.argumentLists = [];
    }
    /**
     * Returns IDs of all argument lists passed to `deprotofy()` whose keywords
     * have been accessed.
     */
    get accessedArgumentLists() {
        return this.argumentLists
            .filter(list => list.keywordsAccessed)
            .map(list => list.id);
    }
    /** Converts `value` to its protocol buffer representation. */
    protofy(value) {
        const result = new proto.Value();
        if (value instanceof string_1.SassString) {
            const string = new proto.Value.String();
            string.setText(value.text);
            string.setQuoted(value.hasQuotes);
            result.setString(string);
        }
        else if (value instanceof number_1.SassNumber) {
            result.setNumber(this.protofyNumber(value));
        }
        else if (value instanceof color_1.SassColor) {
            if (value.hasCalculatedHsl) {
                const color = new proto.Value.HslColor();
                color.setHue(value.hue);
                color.setSaturation(value.saturation);
                color.setLightness(value.lightness);
                color.setAlpha(value.alpha);
                result.setHslColor(color);
            }
            else {
                const color = new proto.Value.RgbColor();
                color.setRed(value.red);
                color.setGreen(value.green);
                color.setBlue(value.blue);
                color.setAlpha(value.alpha);
                result.setRgbColor(color);
            }
        }
        else if (value instanceof list_1.SassList) {
            const list = new proto.Value.List();
            list.setSeparator(this.protofySeparator(value.separator));
            list.setHasBrackets(value.hasBrackets);
            for (const element of value.asList) {
                list.addContents(this.protofy(element));
            }
            result.setList(list);
        }
        else if (value instanceof argument_list_1.SassArgumentList) {
            const list = new proto.Value.ArgumentList();
            list.setId(value.id);
            list.setSeparator(this.protofySeparator(value.separator));
            for (const element of value.asList) {
                list.addContents(this.protofy(element));
            }
            const keywords = list.getKeywordsMap();
            for (const [key, mapValue] of value.keywordsInternal) {
                keywords.set(key, this.protofy(mapValue));
            }
            result.setArgumentList(list);
        }
        else if (value instanceof map_1.SassMap) {
            const map = new proto.Value.Map();
            for (const [key, mapValue] of value.contents) {
                const entry = new proto.Value.Map.Entry();
                entry.setKey(this.protofy(key));
                entry.setValue(this.protofy(mapValue));
                map.addEntries(entry);
            }
            result.setMap(map);
        }
        else if (value instanceof function_1.SassFunction) {
            if (value.id !== undefined) {
                const fn = new proto.Value.CompilerFunction();
                fn.setId(value.id);
                result.setCompilerFunction(fn);
            }
            else {
                const fn = new proto.Value.HostFunction();
                fn.setId(this.functions.register(value.callback));
                fn.setSignature(value.signature);
                result.setHostFunction(fn);
            }
        }
        else if (value === boolean_1.sassTrue) {
            result.setSingleton(proto.SingletonValue.TRUE);
        }
        else if (value === boolean_1.sassFalse) {
            result.setSingleton(proto.SingletonValue.FALSE);
        }
        else if (value === null_1.sassNull) {
            result.setSingleton(proto.SingletonValue.NULL);
        }
        else {
            throw utils.compilerError(`Unknown Value ${value}`);
        }
        return result;
    }
    /** Converts `number` to its protocol buffer representation. */
    protofyNumber(number) {
        const value = new proto.Value.Number();
        value.setValue(number.value);
        for (const unit of number.numeratorUnits) {
            value.addNumerators(unit);
        }
        for (const unit of number.denominatorUnits) {
            value.addDenominators(unit);
        }
        return value;
    }
    /** Converts `separator` to its protocol buffer representation. */
    protofySeparator(separator) {
        switch (separator) {
            case ',':
                return proto.ListSeparator.COMMA;
            case ' ':
                return proto.ListSeparator.SPACE;
            case '/':
                return proto.ListSeparator.SLASH;
            case null:
                return proto.ListSeparator.UNDECIDED;
            default:
                throw utils.compilerError(`Unknown ListSeparator ${separator}`);
        }
    }
    /** Converts `value` to its JS representation. */
    deprotofy(value) {
        switch (value.getValueCase()) {
            case proto.Value.ValueCase.STRING: {
                const string = value.getString();
                return string.getText().length === 0
                    ? string_1.SassString.empty({ quotes: string.getQuoted() })
                    : new string_1.SassString(string.getText(), { quotes: string.getQuoted() });
            }
            case proto.Value.ValueCase.NUMBER:
                return this.deprotofyNumber(value.getNumber());
            case proto.Value.ValueCase.RGB_COLOR: {
                const color = value.getRgbColor();
                return new color_1.SassColor({
                    red: color.getRed(),
                    green: color.getGreen(),
                    blue: color.getBlue(),
                    alpha: color.getAlpha(),
                });
            }
            case proto.Value.ValueCase.HSL_COLOR: {
                const color = value.getHslColor();
                return new color_1.SassColor({
                    hue: color.getHue(),
                    saturation: color.getSaturation(),
                    lightness: color.getLightness(),
                    alpha: color.getAlpha(),
                });
            }
            case proto.Value.ValueCase.LIST: {
                const list = value.getList();
                const separator = this.deprotofySeparator(list.getSeparator());
                const contents = list.getContentsList();
                if (separator === null && contents.length > 1) {
                    throw utils.compilerError(`Value.List ${list} can't have an undecided separator because it ` +
                        `has ${contents.length} elements`);
                }
                return new list_1.SassList(contents.map(element => this.deprotofy(element)), { separator, brackets: list.getHasBrackets() });
            }
            case proto.Value.ValueCase.ARGUMENT_LIST: {
                const list = value.getArgumentList();
                const separator = this.deprotofySeparator(list.getSeparator());
                const contents = list.getContentsList();
                if (separator === null && contents.length > 1) {
                    throw utils.compilerError(`Value.List ${list} can't have an undecided separator because it ` +
                        `has ${contents.length} elements`);
                }
                const result = new argument_list_1.SassArgumentList(contents.map(element => this.deprotofy(element)), (0, immutable_1.OrderedMap)([...list.getKeywordsMap().entries()].map(([key, value]) => [
                    key,
                    this.deprotofy(value),
                ])), separator, list.getId());
                this.argumentLists.push(result);
                return result;
            }
            case proto.Value.ValueCase.MAP:
                return new map_1.SassMap((0, immutable_1.OrderedMap)(value
                    .getMap()
                    .getEntriesList()
                    .map(entry => {
                    const key = entry.getKey();
                    if (!key)
                        throw utils.mandatoryError('Value.Map.Entry.key');
                    const value = entry.getValue();
                    if (!value)
                        throw utils.mandatoryError('Value.Map.Entry.value');
                    return [this.deprotofy(key), this.deprotofy(value)];
                })));
            case proto.Value.ValueCase.COMPILER_FUNCTION:
                return new function_1.SassFunction(value.getCompilerFunction().getId());
            case proto.Value.ValueCase.HOST_FUNCTION:
                throw utils.compilerError('The compiler may not send Value.host_function.');
            case proto.Value.ValueCase.SINGLETON:
                switch (value.getSingleton()) {
                    case proto.SingletonValue.TRUE:
                        return boolean_1.sassTrue;
                    case proto.SingletonValue.FALSE:
                        return boolean_1.sassFalse;
                    case proto.SingletonValue.NULL:
                        return null_1.sassNull;
                    default:
                        throw utils.compilerError(`Unknown Value.singleton ${value.getSingleton()}`);
                }
            default:
                throw utils.mandatoryError('Value.value');
        }
    }
    /** Converts `number` to its JS representation. */
    deprotofyNumber(number) {
        return new number_1.SassNumber(number.getValue(), {
            numeratorUnits: number.getNumeratorsList(),
            denominatorUnits: number.getDenominatorsList(),
        });
    }
    /** Converts `separator` to its JS representation. */
    deprotofySeparator(separator) {
        switch (separator) {
            case proto.ListSeparator.COMMA:
                return ',';
            case proto.ListSeparator.SPACE:
                return ' ';
            case proto.ListSeparator.SLASH:
                return '/';
            case proto.ListSeparator.UNDECIDED:
                return null;
            default:
                throw utils.compilerError(`Unknown separator ${separator}`);
        }
    }
}
exports.Protofier = Protofier;
//# sourceMappingURL=protofier.js.map