"use strict";
// Copyright 2021 Google LLC. Use of this source code is governed by an
// MIT-style license that can be found in the LICENSE file or at
// https://opensource.org/licenses/MIT.
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncEmbeddedCompiler = void 0;
const rxjs_1 = require("rxjs");
const sync_process_1 = require("./sync-process");
const compiler_path_1 = require("./compiler-path");
/**
 * A synchronous wrapper for the embedded Sass compiler that exposes its stdio
 * streams as Observables.
 */
class SyncEmbeddedCompiler {
    constructor() {
        /** The underlying process that's being wrapped. */
        this.process = new sync_process_1.SyncProcess(compiler_path_1.compilerPath, { windowsHide: true });
        /** The buffers emitted by the child process's stdout. */
        this.stdout$ = new rxjs_1.Subject();
        /** The buffers emitted by the child process's stderr. */
        this.stderr$ = new rxjs_1.Subject();
    }
    /** Writes `buffer` to the child process's stdin. */
    writeStdin(buffer) {
        this.process.stdin.write(buffer);
    }
    yield() {
        const event = this.process.yield();
        switch (event.type) {
            case 'stdout':
                this.stdout$.next(event.data);
                return true;
            case 'stderr':
                this.stderr$.next(event.data);
                return true;
            case 'exit':
                return false;
        }
    }
    /** Blocks until the underlying process exits. */
    yieldUntilExit() {
        while (this.yield()) {
            // Any events will be handled by `this.yield()`.
        }
    }
    /** Kills the child process, cleaning up all associated Observables. */
    close() {
        this.process.stdin.end();
    }
}
exports.SyncEmbeddedCompiler = SyncEmbeddedCompiler;
//# sourceMappingURL=sync-compiler.js.map