"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const bundling_1 = require("../lib/bundling");
jest.mock('../lib/bundling', () => {
    return {
        Bundling: {
            parcel: jest.fn().mockReturnValue({
                bind: () => {
                    return { inlineCode: 'code' };
                },
                bindToResource: () => { return; },
            }),
        },
    };
});
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
    jest.clearAllMocks();
});
test('NodejsFunction with .ts handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler1.ts'),
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('NodejsFunction with .js handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler2.js'),
    }));
});
test('NodejsFunction with container env vars', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1', {
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    });
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    }));
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py',
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts',
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8,
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
//# sourceMappingURL=data:application/json;base64,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