"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('managed policy', () => {
    let app;
    let stack;
    beforeEach(() => {
        app = new cdk.App();
        stack = new cdk.Stack(app, 'MyStack', { env: { account: '1234', region: 'us-east-1' } });
    });
    test('simple AWS managed policy', () => {
        const mp = lib_1.ManagedPolicy.fromAwsManagedPolicyName("service-role/SomePolicy");
        expect(stack.resolve(mp.managedPolicyArn)).toEqual({
            "Fn::Join": ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':iam::aws:policy/service-role/SomePolicy'
                ]]
        });
    });
    test('simple customer managed policy', () => {
        const mp = lib_1.ManagedPolicy.fromManagedPolicyName(stack, 'MyCustomerManagedPolicy', "SomeCustomerPolicy");
        expect(stack.resolve(mp.managedPolicyArn)).toEqual({
            "Fn::Join": ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':iam::1234:policy/SomeCustomerPolicy'
                ]]
        });
    });
    test('managed policy with statements', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', { managedPolicyName: 'MyManagedPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        ManagedPolicyName: 'MyManagedPolicyName',
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17'
                        },
                        Path: '/',
                        Description: ''
                    }
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            { Ref: 'MyManagedPolicy9F3720AE' }
                        ]
                    }
                }
            }
        });
    });
    test('policy name can be omitted, in which case the logical id will be used', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17'
                        },
                        Path: '/',
                        Description: ''
                    }
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            { Ref: 'MyManagedPolicy9F3720AE' }
                        ]
                    }
                }
            }
        });
    });
    test('via props, managed policy can be attached to users, groups and roles and permissions, description and path can be added', () => {
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.ManagedPolicy(stack, 'MyTestManagedPolicy', {
            managedPolicyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            description: 'My Policy Description',
            path: 'tahiti/is/a/magical/place',
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        expect(stack).toMatchTemplate({
            Resources: {
                User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                },
                MyTestManagedPolicy6535D9F5: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        Description: 'My Policy Description',
                        Path: 'tahiti/is/a/magical/place',
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }]
                    }
                }
            }
        });
    });
    test('idempotent if a principal (user/group/role) is attached twice', () => {
        const p = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        const group = new lib_1.Group(stack, 'MyGroup');
        p.attachToGroup(group);
        p.attachToGroup(group);
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        p.attachToRole(role);
        p.attachToRole(role);
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17'
                        },
                        Description: '',
                        Path: '/',
                        Users: [{ Ref: 'MyUserDC45028B' }],
                        Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        Roles: [{ Ref: 'MyRoleF48FFE04' }]
                    }
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                }
            }
        });
    });
    test('users, groups, roles and permissions can be added using methods', () => {
        const p = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy', {
            managedPolicyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: {
                            Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17'
                        },
                        ManagedPolicyName: 'Foo',
                        Description: '',
                        Path: '/',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }]
                    }
                },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                }
            }
        });
    });
    test('policy can be attached to users, groups or role via methods on the principal', () => {
        const policy = new lib_1.ManagedPolicy(stack, 'MyManagedPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        expect(stack).toMatchTemplate({
            Resources: {
                MyManagedPolicy9F3720AE: {
                    Type: 'AWS::IAM::ManagedPolicy',
                    Properties: {
                        PolicyDocument: {
                            Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17'
                        },
                        Description: '',
                        Path: '/'
                    }
                },
                MyUserDC45028B: { Type: 'AWS::IAM::User', Properties: { ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }] } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group', Properties: { ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }] } },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [{ Ref: 'MyManagedPolicy9F3720AE' }],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                }
            }
        });
    });
    test('policy from AWS managed policy lookup can be attached to users, groups or role via methods on the principal', () => {
        const policy = lib_1.ManagedPolicy.fromAwsManagedPolicyName('AnAWSManagedPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyUserDC45028B: {
                    Type: 'AWS::IAM::User',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::aws:policy/AnAWSManagedPolicy"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::aws:policy/AnAWSManagedPolicy"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::aws:policy/AnAWSManagedPolicy"
                                    ]
                                ]
                            }
                        ],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                }
            }
        });
    });
    test('policy from customer managed policy lookup can be attached to users, groups or role via methods', () => {
        const policy = lib_1.ManagedPolicy.fromManagedPolicyName(stack, 'MyManagedPolicy', 'ACustomerManagedPolicyName');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.addManagedPolicy(policy);
        group.addManagedPolicy(policy);
        role.addManagedPolicy(policy);
        expect(stack).toMatchTemplate({
            Resources: {
                MyUserDC45028B: {
                    Type: 'AWS::IAM::User',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::1234:policy/ACustomerManagedPolicyName"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                MyGroupCBA54B1B: {
                    Type: 'AWS::IAM::Group',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::1234:policy/ACustomerManagedPolicyName"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                MyRoleF48FFE04: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        ManagedPolicyArns: [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        { Ref: "AWS::Partition" },
                                        ":iam::1234:policy/ACustomerManagedPolicyName"
                                    ]
                                ]
                            }
                        ],
                        AssumeRolePolicyDocument: {
                            Statement: [{
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' }
                                }],
                            Version: '2012-10-17'
                        }
                    }
                }
            }
        });
    });
    test('fails if policy document is empty', () => {
        new lib_1.ManagedPolicy(stack, 'MyPolicy');
        expect(() => app.synth())
            .toThrow(/Managed Policy is empty. You must add statements to the policy/);
    });
    test('managed policy name is correctly calculated', () => {
        const mp = new lib_1.ManagedPolicy(stack, 'Policy');
        mp.addStatements(new lib_1.PolicyStatement({
            actions: ['a:abc'],
        }));
        expect(stack.resolve(mp.managedPolicyName)).toEqual({
            "Fn::Select": [1,
                { "Fn::Split": ["/",
                        { "Fn::Select": [5,
                                { "Fn::Split": [":",
                                        { Ref: "Policy23B91518", }] },], },], },]
        });
    });
    test('cross-stack hard-name contains the right resource type', () => {
        const mp = new lib_1.ManagedPolicy(stack, 'Policy', {
            managedPolicyName: cdk.PhysicalName.GENERATE_IF_NEEDED
        });
        mp.addStatements(new lib_1.PolicyStatement({
            actions: ['a:abc'],
        }));
        const stack2 = new cdk.Stack(app, 'Stack2', { env: { account: '5678', region: 'us-east-1' } });
        new cdk.CfnOutput(stack2, 'Output', {
            value: mp.managedPolicyArn
        });
        expect(stack2).toMatchTemplate({
            Outputs: {
                Output: {
                    Value: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":iam::1234:policy/mystackmystackpolicy17395e221b1b6deaf875"
                            ]
                        ]
                    }
                }
            }
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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