"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
describe('IAM policy', () => {
    test('fails when policy is empty', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        expect(() => app.synth()).toThrow(/Policy is empty/);
    });
    test('policy with statements', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        expect(stack).toMatchTemplate({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
    });
    test('policy name can be omitted, in which case the logical id will be used', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        expect(stack).toMatchTemplate({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
    });
    test('policy can be attached users, groups and roles and added permissions via props', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        expect(stack).toMatchTemplate({ Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
    });
    test('idempotent if a principal (user/group/role) is attached twice', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        expect(stack).toMatchTemplate({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
    });
    test('users, groups, roles and permissions can be added using methods', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        expect(stack).toMatchTemplate({ Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
    });
    test('policy can be attached to users, groups or role via methods on the principal', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        expect(stack).toMatchTemplate({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
    });
    test('fails if policy name is not unique within a user/group/role', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        expect(() => p2.attachToUser(user)).toThrow(/A policy named "Foo" is already attached/);
        expect(() => p2.attachToGroup(group)).toThrow(/A policy named "Foo" is already attached/);
        expect(() => p2.attachToRole(role)).toThrow(/A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
    });
    test('fails if policy is not attached to a principal', () => {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        expect(() => app.synth()).toThrow(/Policy must be attached to at least one principal: user, group or role/);
    });
    test("generated policy name is the same as the logical id if it's shorter than 128 characters", () => {
        const stack = new core_1.Stack();
        createPolicyWithLogicalId(stack, 'Foo');
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            "PolicyName": "Foo",
        });
    });
    test('generated policy name only uses the last 128 characters of the logical id', () => {
        const stack = new core_1.Stack();
        const logicalId128 = 'a' + dup(128 - 2) + 'a';
        const logicalIdOver128 = 'PREFIX' + logicalId128;
        createPolicyWithLogicalId(stack, logicalIdOver128);
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            "PolicyName": logicalId128,
        });
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    });
});
function createPolicyWithLogicalId(stack, logicalId) {
    const policy = new lib_1.Policy(stack, logicalId);
    const cfnPolicy = policy.node.defaultChild;
    cfnPolicy.overrideLogicalId(logicalId); // force a particular logical ID
    // add statements & principal to satisfy validation
    policy.addStatements(new lib_1.PolicyStatement({
        actions: ['*'],
        resources: ['*'],
    }));
    policy.attachToRole(new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.AnyPrincipal() }));
}
//# sourceMappingURL=data:application/json;base64,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