"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath, props.logLevel);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('tsc');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        const pathJoin = osPathJoin(options.osPlatform);
        let relativeEntryPath = pathJoin(options.inputDir, this.relativeEntryPath);
        let tscCommand = '';
        if (this.props.preCompilation) {
            const tsconfig = (_a = this.props.tsconfig) !== null && _a !== void 0 ? _a : util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
            if (!tsconfig) {
                throw new Error('Cannot find a `tsconfig.json` but `preCompilation` is set to `true`, please specify it via `tsconfig`');
            }
            const compilerOptions = util_1.getTsconfigCompilerOptions(tsconfig);
            tscCommand = `${options.tscRunner} "${relativeEntryPath}" ${compilerOptions}`;
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
        }
        const loaders = Object.entries((_b = this.props.loader) !== null && _b !== void 0 ? _b : {});
        const defines = Object.entries((_c = this.props.define) !== null && _c !== void 0 ? _c : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : this.props.sourceMap;
        const sourceMapMode = (_e = this.props.sourceMapMode) !== null && _e !== void 0 ? _e : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_f = this.props.sourcesContent) !== null && _f !== void 0 ? _f : true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${relativeEntryPath}"`,
            `--target=${(_g = this.props.target) !== null && _g !== void 0 ? _g : toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_h = this.relativeDepsLockFilePath) !== null && _h !== void 0 ? _h : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_k = (_j = this.props.commandHooks) === null || _j === void 0 ? void 0 : _j.beforeBundling(options.inputDir, options.outputDir)) !== null && _k !== void 0 ? _k : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_m = (this.props.nodeModules && ((_l = this.props.commandHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall(options.inputDir, options.outputDir)))) !== null && _m !== void 0 ? _m : [],
            depsCommand,
            ...(_p = (_o = this.props.commandHooks) === null || _o === void 0 ? void 0 : _o.afterBundling(options.inputDir, options.outputDir)) !== null && _p !== void 0 ? _p : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > "${filePath}"`;
        }
        return `echo '${stringifiedData}' > "${filePath}"`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUE2RTtBQUM3RSxxQ0FBcUM7QUFDckMsaUVBQTZEO0FBQzdELHVEQUFtRDtBQUNuRCxtQ0FBdUU7QUFDdkUsaUNBQXVGO0FBRXZGLE1BQU0scUJBQXFCLEdBQUcsR0FBRyxDQUFDO0FBc0NsQzs7R0FFRztBQUNILE1BQWEsUUFBUTtJQXFDbkIsWUFBNkIsS0FBb0I7O1FBQXBCLFVBQUssR0FBTCxLQUFLLENBQWU7UUFDL0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxnQ0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTFGLFFBQVEsQ0FBQyxtQkFBbUIsU0FBRyxRQUFRLENBQUMsbUJBQW1CLG1DQUFJLDBDQUFtQixDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNyRyxRQUFRLENBQUMsZUFBZSxTQUFHLFFBQVEsQ0FBQyxlQUFlLG1DQUFJLDBDQUFtQixDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV6RixJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3BGLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBRXRHLElBQUksSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixLQUFLLENBQUMsZ0JBQWdCLDZCQUE2QixJQUFJLENBQUMsV0FBVyxLQUFLLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxDQUFDLENBQUM7U0FDeko7UUFFRCxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDbEIsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1NBQzNGO1FBRUQsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLG9CQUFZLENBQUMsR0FBRztlQUM5QixDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssb0JBQU8sQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxvQkFBTyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3ZGLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxVQUFVLENBQUMsQ0FBQztTQUMxRztRQUVELElBQUksQ0FBQyxTQUFTLEdBQUc7WUFDZixTQUFHLEtBQUssQ0FBQyxlQUFlLG1DQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRTtTQUMzQixDQUFDO1FBRUYsa0JBQWtCO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDO1FBQ3BGLElBQUksQ0FBQyxLQUFLLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxPQUFDLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxFQUMzRztZQUNFLFNBQVMsRUFBRTtnQkFDVCxTQUFHLEtBQUssQ0FBQyxTQUFTLG1DQUFJLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLO2dCQUN4QyxlQUFlLFFBQUUsS0FBSyxDQUFDLGNBQWMsbUNBQUkscUJBQXFCO2FBQy9EO1lBQ0QsUUFBUSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsY0FBYztTQUM1QyxDQUFDLENBQ0YsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsbUNBQW1DO1FBRTlFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQztZQUNqRCxRQUFRLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxrQkFBa0I7WUFDN0MsU0FBUyxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsbUJBQW1CO1lBQy9DLGFBQWEsRUFBRSxTQUFTO1lBQ3hCLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLFVBQVUsRUFBRSxPQUFPO1NBQ3BCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyw2RUFBNkU7UUFDN0Usa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxHQUFHLENBQUM7UUFFNUIsaUJBQWlCO1FBQ2pCLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUUsRUFBRSwrQkFBK0I7WUFDL0QsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztTQUM5QztLQUNGO0lBbEdEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFzQjtRQUN6QyxPQUFPLGlCQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUU7WUFDekMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTO1lBQzVCLGFBQWEsRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ3RGLFFBQVEsRUFBRSxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUM7U0FDaEMsQ0FBQyxDQUFDO0tBQ0o7SUFFTSxNQUFNLENBQUMsNkJBQTZCO1FBQ3pDLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxTQUFTLENBQUM7S0FDdEM7SUFFTSxNQUFNLENBQUMseUJBQXlCO1FBQ3JDLElBQUksQ0FBQyxlQUFlLEdBQUcsU0FBUyxDQUFDO0tBQ2xDO0lBbUZPLHFCQUFxQixDQUFDLE9BQStCOztRQUMzRCxNQUFNLFFBQVEsR0FBRyxVQUFVLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hELElBQUksaUJBQWlCLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDM0UsSUFBSSxVQUFVLEdBQUcsRUFBRSxDQUFDO1FBRXBCLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDN0IsTUFBTSxRQUFRLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLG1DQUFJLGFBQU0sQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDaEcsSUFBSSxDQUFDLFFBQVEsRUFBRTtnQkFDYixNQUFNLElBQUksS0FBSyxDQUFDLHVHQUF1RyxDQUFDLENBQUM7YUFDMUg7WUFDRCxNQUFNLGVBQWUsR0FBRyxpQ0FBMEIsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM3RCxVQUFVLEdBQUcsR0FBRyxPQUFPLENBQUMsU0FBUyxLQUFLLGlCQUFpQixLQUFLLGVBQWUsRUFBRSxDQUFDO1lBQzlFLGlCQUFpQixHQUFHLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7U0FDckU7UUFFRCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUN4RCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUV4RCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxLQUFLLEtBQUssSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxDQUFDLENBQUM7U0FDekU7UUFFRCxNQUFNLGdCQUFnQixTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxtQ0FBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUMxRSxNQUFNLGFBQWEsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUkscUJBQWEsQ0FBQyxPQUFPLENBQUM7UUFDeEUsTUFBTSxjQUFjLEdBQUcsYUFBYSxLQUFLLHFCQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUNyRyxNQUFNLGNBQWMsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsbUNBQUksSUFBSSxDQUFDO1FBRXpELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxLQUFLLG9CQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztRQUNsRixNQUFNLGNBQWMsR0FBYTtZQUMvQixPQUFPLENBQUMsYUFBYTtZQUNyQixVQUFVLEVBQUUsSUFBSSxpQkFBaUIsR0FBRztZQUNwQyxZQUFZLE1BQUEsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLG1DQUFJLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQy9ELGlCQUFpQjtZQUNqQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzdELGNBQWMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDckQsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUN4QyxHQUFHLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUMzRCxHQUFHLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLHFCQUFxQixjQUFjLEVBQUUsQ0FBQztZQUNoRSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsY0FBYyxRQUFRLEVBQUUsQ0FBQztZQUMzRCxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsWUFBWSxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7WUFDMUQsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLFlBQVksR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUM1RSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3BFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDL0MsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxRQUFRLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDM0csR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLGlCQUFpQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzlGLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2hGLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2hGLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDaEUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtTQUNyRixDQUFDO1FBRUYsSUFBSSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDMUIsMkVBQTJFO1lBQzNFLDRCQUE0QjtZQUM1QixNQUFNLE9BQU8sR0FBRyxhQUFNLENBQUMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ3ZFLElBQUksQ0FBQyxPQUFPLEVBQUU7Z0JBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyw4RkFBOEYsQ0FBQyxDQUFDO2FBQ2pIO1lBRUQsMkRBQTJEO1lBQzNELE1BQU0sWUFBWSxHQUFHLDBCQUFtQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQzFFLE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUVwRCxNQUFNLFlBQVksR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLFFBQVEsUUFBRSxJQUFJLENBQUMsd0JBQXdCLG1DQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7WUFFL0csb0VBQW9FO1lBQ3BFLFdBQVcsR0FBRyxLQUFLLENBQUM7Z0JBQ2xCLFNBQVMsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsY0FBYyxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQztnQkFDbEYsU0FBUyxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDdkYsU0FBUyxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO2dCQUM1QyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO2FBQzdDLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxLQUFLLENBQUM7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxjQUFjLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsU0FBUyxvQ0FBSyxFQUFFO1lBQ3JGLFVBQVU7WUFDVixjQUFjLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztZQUN4QixTQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLFdBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLDBDQUFFLGFBQWEsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxTQUFTLEVBQUMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ2hILFdBQVc7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxhQUFhLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsU0FBUyxvQ0FBSyxFQUFFO1NBQ3JGLENBQUMsQ0FBQztLQUNKO0lBRU8sd0JBQXdCOztRQUM5QixNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDakMsTUFBTSxrQkFBa0IsR0FBRyxDQUFDLFNBQWlCLEVBQUUsT0FBNEIsRUFBRSxHQUF5QixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUM7WUFDcEksUUFBUSxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzFCLFNBQVM7WUFDVCxhQUFhLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDekYsU0FBUyxFQUFFLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7WUFDbEYsVUFBVTtTQUNYLENBQUMsQ0FBQztRQUNILE1BQU0sV0FBVyxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxtQ0FBSSxFQUFFLENBQUM7UUFDakQsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQztRQUU3QixPQUFPO1lBQ0wsU0FBUyxDQUFDLFNBQWlCO2dCQUN6QixJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixFQUFFO29CQUNqQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO29CQUNwRixPQUFPLEtBQUssQ0FBQztpQkFDZDtnQkFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsR0FBRyxxQkFBcUIsR0FBRyxDQUFDLEVBQUU7b0JBQ2pGLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLHFCQUFxQixjQUFjLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO2lCQUN4SDtnQkFFRCxNQUFNLFlBQVksR0FBRyxrQkFBa0IsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLG1CQUFtQixFQUFFLFFBQVEsQ0FBQyxlQUFlLENBQUMsQ0FBQztnQkFFM0csV0FBSSxDQUNGLFVBQVUsS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUN2QztvQkFDRSxVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUk7b0JBQ3BDLFlBQVk7aUJBQ2IsRUFDRDtvQkFDRSxHQUFHLEVBQUUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxXQUFXLEVBQUU7b0JBQ3ZDLEtBQUssRUFBRTt3QkFDTCxRQUFRO3dCQUNSLE9BQU8sQ0FBQyxNQUFNO3dCQUNkLFNBQVM7cUJBQ1Y7b0JBQ0QsR0FBRztvQkFDSCx3QkFBd0IsRUFBRSxVQUFVLEtBQUssT0FBTztpQkFDakQsQ0FBQyxDQUFDO2dCQUVMLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztTQUNGLENBQUM7S0FDSDtDQUNGO0FBeE9ELDRCQXdPQztBQVVEOztHQUVHO0FBQ0gsTUFBTSxTQUFTO0lBQ2IsWUFBNkIsVUFBMkI7UUFBM0IsZUFBVSxHQUFWLFVBQVUsQ0FBaUI7S0FBSTtJQUVyRCxTQUFTLENBQUMsUUFBZ0IsRUFBRSxJQUFTO1FBQzFDLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLE9BQU8sRUFBRTtZQUMvQixPQUFPLFNBQVMsZUFBZSxRQUFRLFFBQVEsR0FBRyxDQUFDO1NBQ3BEO1FBRUQsT0FBTyxTQUFTLGVBQWUsUUFBUSxRQUFRLEdBQUcsQ0FBQztLQUNwRDtJQUVNLElBQUksQ0FBQyxHQUFXLEVBQUUsSUFBWTtRQUNuQyxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssT0FBTyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxHQUFHLE1BQU0sSUFBSSxHQUFHLENBQUM7U0FDbEM7UUFFRCxPQUFPLE9BQU8sR0FBRyxNQUFNLElBQUksR0FBRyxDQUFDO0tBQ2hDO0lBRU0sZUFBZSxDQUFDLEdBQVc7UUFDaEMsT0FBTyxPQUFPLEdBQUcsR0FBRyxDQUFDO0tBQ3RCO0NBQ0Y7QUFFRDs7R0FFRztBQUNILFNBQVMsS0FBSyxDQUFDLFFBQWtCO0lBQy9CLE9BQU8sUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7QUFDaEQsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxVQUFVLENBQUMsUUFBeUI7SUFDM0MsT0FBTyxVQUFTLEdBQUcsS0FBZTtRQUNoQyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLENBQUM7UUFDbkMsZ0RBQWdEO1FBQ2hELElBQUksRUFBRSxDQUFDLFFBQVEsRUFBRSxLQUFLLE9BQU8sSUFBSSxRQUFRLEtBQUssT0FBTyxFQUFFO1lBQ3JELE9BQU8sTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDbkM7UUFDRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDLENBQUM7QUFDSixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFFBQVEsQ0FBQyxPQUFnQjtJQUNoQyxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUVoRCxJQUFJLENBQUMsS0FBSyxFQUFFO1FBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0tBQ3pEO0lBRUQsT0FBTyxPQUFPLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO0FBQzNCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBvcyBmcm9tICdvcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQXJjaGl0ZWN0dXJlLCBBc3NldENvZGUsIENvZGUsIFJ1bnRpbWUgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFBhY2thZ2VJbnN0YWxsYXRpb24gfSBmcm9tICcuL3BhY2thZ2UtaW5zdGFsbGF0aW9uJztcbmltcG9ydCB7IFBhY2thZ2VNYW5hZ2VyIH0gZnJvbSAnLi9wYWNrYWdlLW1hbmFnZXInO1xuaW1wb3J0IHsgQnVuZGxpbmdPcHRpb25zLCBPdXRwdXRGb3JtYXQsIFNvdXJjZU1hcE1vZGUgfSBmcm9tICcuL3R5cGVzJztcbmltcG9ydCB7IGV4ZWMsIGV4dHJhY3REZXBlbmRlbmNpZXMsIGZpbmRVcCwgZ2V0VHNjb25maWdDb21waWxlck9wdGlvbnMgfSBmcm9tICcuL3V0aWwnO1xuXG5jb25zdCBFU0JVSUxEX01BSk9SX1ZFUlNJT04gPSAnMCc7XG5cbi8qKlxuICogQnVuZGxpbmcgcHJvcGVydGllc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1bmRsaW5nUHJvcHMgZXh0ZW5kcyBCdW5kbGluZ09wdGlvbnMge1xuICAvKipcbiAgICogUGF0aCB0byBsb2NrIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNMb2NrRmlsZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogRW50cnkgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJ1bnRpbWUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcblxuICAvKipcbiAgICogVGhlIHN5c3RlbSBhcmNoaXRlY3R1cmUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmU7XG5cbiAgLyoqXG4gICAqIFBhdGggdG8gcHJvamVjdCByb290XG4gICAqL1xuICByZWFkb25seSBwcm9qZWN0Um9vdDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSdW4gY29tcGlsYXRpb24gdXNpbmcgYHRzY2AgYmVmb3JlIGJ1bmRsaW5nXG4gICAqL1xuICByZWFkb25seSBwcmVDb21waWxhdGlvbj86IGJvb2xlYW5cblxufVxuXG4vKipcbiAqIEJ1bmRsaW5nIHdpdGggZXNidWlsZFxuICovXG5leHBvcnQgY2xhc3MgQnVuZGxpbmcgaW1wbGVtZW50cyBjZGsuQnVuZGxpbmdPcHRpb25zIHtcbiAgLyoqXG4gICAqIGVzYnVpbGQgYnVuZGxlZCBMYW1iZGEgYXNzZXQgY29kZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBidW5kbGUob3B0aW9uczogQnVuZGxpbmdQcm9wcyk6IEFzc2V0Q29kZSB7XG4gICAgcmV0dXJuIENvZGUuZnJvbUFzc2V0KG9wdGlvbnMucHJvamVjdFJvb3QsIHtcbiAgICAgIGFzc2V0SGFzaDogb3B0aW9ucy5hc3NldEhhc2gsXG4gICAgICBhc3NldEhhc2hUeXBlOiBvcHRpb25zLmFzc2V0SGFzaCA/IGNkay5Bc3NldEhhc2hUeXBlLkNVU1RPTSA6IGNkay5Bc3NldEhhc2hUeXBlLk9VVFBVVCxcbiAgICAgIGJ1bmRsaW5nOiBuZXcgQnVuZGxpbmcob3B0aW9ucyksXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGNsZWFyRXNidWlsZEluc3RhbGxhdGlvbkNhY2hlKCk6IHZvaWQge1xuICAgIHRoaXMuZXNidWlsZEluc3RhbGxhdGlvbiA9IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgY2xlYXJUc2NJbnN0YWxsYXRpb25DYWNoZSgpOiB2b2lkIHtcbiAgICB0aGlzLnRzY0luc3RhbGxhdGlvbiA9IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIGVzYnVpbGRJbnN0YWxsYXRpb24/OiBQYWNrYWdlSW5zdGFsbGF0aW9uO1xuICBwcml2YXRlIHN0YXRpYyB0c2NJbnN0YWxsYXRpb24/OiBQYWNrYWdlSW5zdGFsbGF0aW9uO1xuXG4gIC8vIENvcmUgYnVuZGxpbmcgb3B0aW9uc1xuICBwdWJsaWMgcmVhZG9ubHkgaW1hZ2U6IGNkay5Eb2NrZXJJbWFnZTtcbiAgcHVibGljIHJlYWRvbmx5IGNvbW1hbmQ6IHN0cmluZ1tdO1xuICBwdWJsaWMgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuICBwdWJsaWMgcmVhZG9ubHkgd29ya2luZ0RpcmVjdG9yeTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgbG9jYWw/OiBjZGsuSUxvY2FsQnVuZGxpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBwcm9qZWN0Um9vdDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlRW50cnlQYXRoOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVsYXRpdmVUc2NvbmZpZ1BhdGg/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgZXh0ZXJuYWxzOiBzdHJpbmdbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBwYWNrYWdlTWFuYWdlcjogUGFja2FnZU1hbmFnZXI7XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogQnVuZGxpbmdQcm9wcykge1xuICAgIHRoaXMucGFja2FnZU1hbmFnZXIgPSBQYWNrYWdlTWFuYWdlci5mcm9tTG9ja0ZpbGUocHJvcHMuZGVwc0xvY2tGaWxlUGF0aCwgcHJvcHMubG9nTGV2ZWwpO1xuXG4gICAgQnVuZGxpbmcuZXNidWlsZEluc3RhbGxhdGlvbiA9IEJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ2VzYnVpbGQnKTtcbiAgICBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPSBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ3RzYycpO1xuXG4gICAgdGhpcy5wcm9qZWN0Um9vdCA9IHByb3BzLnByb2plY3RSb290O1xuICAgIHRoaXMucmVsYXRpdmVFbnRyeVBhdGggPSBwYXRoLnJlbGF0aXZlKHRoaXMucHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy5lbnRyeSkpO1xuICAgIHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoID0gcGF0aC5yZWxhdGl2ZSh0aGlzLnByb2plY3RSb290LCBwYXRoLnJlc29sdmUocHJvcHMuZGVwc0xvY2tGaWxlUGF0aCkpO1xuXG4gICAgaWYgKHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoLmluY2x1ZGVzKCcuLicpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkIGRlcHNMb2NrRmlsZVBhdGg6ICR7cHJvcHMuZGVwc0xvY2tGaWxlUGF0aH0gdG8gYmUgdW5kZXIgcHJvamVjdFJvb3Q6ICR7dGhpcy5wcm9qZWN0Um9vdH0gKCR7dGhpcy5yZWxhdGl2ZURlcHNMb2NrRmlsZVBhdGh9KWApO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy50c2NvbmZpZykge1xuICAgICAgdGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA9IHBhdGgucmVsYXRpdmUodGhpcy5wcm9qZWN0Um9vdCwgcGF0aC5yZXNvbHZlKHByb3BzLnRzY29uZmlnKSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnByZUNvbXBpbGF0aW9uICYmICEvXFwudHN4PyQvLnRlc3QocHJvcHMuZW50cnkpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3ByZUNvbXBpbGF0aW9uIGNhbiBvbmx5IGJlIHVzZWQgd2l0aCB0eXBlc2NyaXB0IGZpbGVzJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmZvcm1hdCA9PT0gT3V0cHV0Rm9ybWF0LkVTTVxuICAgICAgICAmJiAocHJvcHMucnVudGltZSA9PT0gUnVudGltZS5OT0RFSlNfMTBfWCB8fCBwcm9wcy5ydW50aW1lID09PSBSdW50aW1lLk5PREVKU18xMl9YKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBFQ01BU2NyaXB0IG1vZHVsZSBvdXRwdXQgZm9ybWF0IGlzIG5vdCBzdXBwb3J0ZWQgYnkgdGhlICR7cHJvcHMucnVudGltZS5uYW1lfSBydW50aW1lYCk7XG4gICAgfVxuXG4gICAgdGhpcy5leHRlcm5hbHMgPSBbXG4gICAgICAuLi5wcm9wcy5leHRlcm5hbE1vZHVsZXMgPz8gWydhd3Mtc2RrJ10sIC8vIE1hcmsgYXdzLXNkayBhcyBleHRlcm5hbCBieSBkZWZhdWx0IChhdmFpbGFibGUgaW4gdGhlIHJ1bnRpbWUpXG4gICAgICAuLi5wcm9wcy5ub2RlTW9kdWxlcyA/PyBbXSwgLy8gTWFyayB0aGUgbW9kdWxlcyB0aGF0IHdlIGFyZSBnb2luZyB0byBpbnN0YWxsIGFzIGV4dGVybmFscyBhbHNvXG4gICAgXTtcblxuICAgIC8vIERvY2tlciBidW5kbGluZ1xuICAgIGNvbnN0IHNob3VsZEJ1aWxkSW1hZ2UgPSBwcm9wcy5mb3JjZURvY2tlckJ1bmRsaW5nIHx8ICFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uO1xuICAgIHRoaXMuaW1hZ2UgPSBzaG91bGRCdWlsZEltYWdlID8gcHJvcHMuZG9ja2VySW1hZ2UgPz8gY2RrLkRvY2tlckltYWdlLmZyb21CdWlsZChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGliJyksXG4gICAgICB7XG4gICAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICAgIC4uLnByb3BzLmJ1aWxkQXJncyA/PyB7fSxcbiAgICAgICAgICBJTUFHRTogcHJvcHMucnVudGltZS5idW5kbGluZ0ltYWdlLmltYWdlLFxuICAgICAgICAgIEVTQlVJTERfVkVSU0lPTjogcHJvcHMuZXNidWlsZFZlcnNpb24gPz8gRVNCVUlMRF9NQUpPUl9WRVJTSU9OLFxuICAgICAgICB9LFxuICAgICAgICBwbGF0Zm9ybTogcHJvcHMuYXJjaGl0ZWN0dXJlLmRvY2tlclBsYXRmb3JtLFxuICAgICAgfSlcbiAgICAgIDogY2RrLkRvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnZHVtbXknKTsgLy8gRG8gbm90IGJ1aWxkIGlmIHdlIGRvbid0IG5lZWQgdG9cblxuICAgIGNvbnN0IGJ1bmRsaW5nQ29tbWFuZCA9IHRoaXMuY3JlYXRlQnVuZGxpbmdDb21tYW5kKHtcbiAgICAgIGlucHV0RGlyOiBjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUixcbiAgICAgIG91dHB1dERpcjogY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19PVVRQVVRfRElSLFxuICAgICAgZXNidWlsZFJ1bm5lcjogJ2VzYnVpbGQnLCAvLyBlc2J1aWxkIGlzIGluc3RhbGxlZCBnbG9iYWxseSBpbiB0aGUgZG9ja2VyIGltYWdlXG4gICAgICB0c2NSdW5uZXI6ICd0c2MnLCAvLyB0c2MgaXMgaW5zdGFsbGVkIGdsb2JhbGx5IGluIHRoZSBkb2NrZXIgaW1hZ2VcbiAgICAgIG9zUGxhdGZvcm06ICdsaW51eCcsIC8vIGxpbnV4IGRvY2tlciBpbWFnZVxuICAgIH0pO1xuICAgIHRoaXMuY29tbWFuZCA9IFsnYmFzaCcsICctYycsIGJ1bmRsaW5nQ29tbWFuZF07XG4gICAgdGhpcy5lbnZpcm9ubWVudCA9IHByb3BzLmVudmlyb25tZW50O1xuICAgIC8vIEJ1bmRsaW5nIHNldHMgdGhlIHdvcmtpbmcgZGlyZWN0b3J5IHRvIGNkay5Bc3NldFN0YWdpbmcuQlVORExJTkdfSU5QVVRfRElSXG4gICAgLy8gYW5kIHdlIHdhbnQgdG8gZm9yY2UgbnB4IHRvIHVzZSB0aGUgZ2xvYmFsbHkgaW5zdGFsbGVkIGVzYnVpbGQuXG4gICAgdGhpcy53b3JraW5nRGlyZWN0b3J5ID0gJy8nO1xuXG4gICAgLy8gTG9jYWwgYnVuZGxpbmdcbiAgICBpZiAoIXByb3BzLmZvcmNlRG9ja2VyQnVuZGxpbmcpIHsgLy8gb25seSBpZiBEb2NrZXIgaXMgbm90IGZvcmNlZFxuICAgICAgdGhpcy5sb2NhbCA9IHRoaXMuZ2V0TG9jYWxCdW5kbGluZ1Byb3ZpZGVyKCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVCdW5kbGluZ0NvbW1hbmQob3B0aW9uczogQnVuZGxpbmdDb21tYW5kT3B0aW9ucyk6IHN0cmluZyB7XG4gICAgY29uc3QgcGF0aEpvaW4gPSBvc1BhdGhKb2luKG9wdGlvbnMub3NQbGF0Zm9ybSk7XG4gICAgbGV0IHJlbGF0aXZlRW50cnlQYXRoID0gcGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgdGhpcy5yZWxhdGl2ZUVudHJ5UGF0aCk7XG4gICAgbGV0IHRzY0NvbW1hbmQgPSAnJztcblxuICAgIGlmICh0aGlzLnByb3BzLnByZUNvbXBpbGF0aW9uKSB7XG4gICAgICBjb25zdCB0c2NvbmZpZyA9IHRoaXMucHJvcHMudHNjb25maWcgPz8gZmluZFVwKCd0c2NvbmZpZy5qc29uJywgcGF0aC5kaXJuYW1lKHRoaXMucHJvcHMuZW50cnkpKTtcbiAgICAgIGlmICghdHNjb25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBhIGB0c2NvbmZpZy5qc29uYCBidXQgYHByZUNvbXBpbGF0aW9uYCBpcyBzZXQgdG8gYHRydWVgLCBwbGVhc2Ugc3BlY2lmeSBpdCB2aWEgYHRzY29uZmlnYCcpO1xuICAgICAgfVxuICAgICAgY29uc3QgY29tcGlsZXJPcHRpb25zID0gZ2V0VHNjb25maWdDb21waWxlck9wdGlvbnModHNjb25maWcpO1xuICAgICAgdHNjQ29tbWFuZCA9IGAke29wdGlvbnMudHNjUnVubmVyfSBcIiR7cmVsYXRpdmVFbnRyeVBhdGh9XCIgJHtjb21waWxlck9wdGlvbnN9YDtcbiAgICAgIHJlbGF0aXZlRW50cnlQYXRoID0gcmVsYXRpdmVFbnRyeVBhdGgucmVwbGFjZSgvXFwudHMoeD8pJC8sICcuanMkMScpO1xuICAgIH1cblxuICAgIGNvbnN0IGxvYWRlcnMgPSBPYmplY3QuZW50cmllcyh0aGlzLnByb3BzLmxvYWRlciA/PyB7fSk7XG4gICAgY29uc3QgZGVmaW5lcyA9IE9iamVjdC5lbnRyaWVzKHRoaXMucHJvcHMuZGVmaW5lID8/IHt9KTtcblxuICAgIGlmICh0aGlzLnByb3BzLnNvdXJjZU1hcCA9PT0gZmFsc2UgJiYgdGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3NvdXJjZU1hcE1vZGUgY2Fubm90IGJlIHVzZWQgd2hlbiBzb3VyY2VNYXAgaXMgZmFsc2UnKTtcbiAgICB9XG5cbiAgICBjb25zdCBzb3VyY2VNYXBFbmFibGVkID0gdGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlID8/IHRoaXMucHJvcHMuc291cmNlTWFwO1xuICAgIGNvbnN0IHNvdXJjZU1hcE1vZGUgPSB0aGlzLnByb3BzLnNvdXJjZU1hcE1vZGUgPz8gU291cmNlTWFwTW9kZS5ERUZBVUxUO1xuICAgIGNvbnN0IHNvdXJjZU1hcFZhbHVlID0gc291cmNlTWFwTW9kZSA9PT0gU291cmNlTWFwTW9kZS5ERUZBVUxUID8gJycgOiBgPSR7dGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlfWA7XG4gICAgY29uc3Qgc291cmNlc0NvbnRlbnQgPSB0aGlzLnByb3BzLnNvdXJjZXNDb250ZW50ID8/IHRydWU7XG5cbiAgICBjb25zdCBvdXRGaWxlID0gdGhpcy5wcm9wcy5mb3JtYXQgPT09IE91dHB1dEZvcm1hdC5FU00gPyAnaW5kZXgubWpzJyA6ICdpbmRleC5qcyc7XG4gICAgY29uc3QgZXNidWlsZENvbW1hbmQ6IHN0cmluZ1tdID0gW1xuICAgICAgb3B0aW9ucy5lc2J1aWxkUnVubmVyLFxuICAgICAgJy0tYnVuZGxlJywgYFwiJHtyZWxhdGl2ZUVudHJ5UGF0aH1cImAsXG4gICAgICBgLS10YXJnZXQ9JHt0aGlzLnByb3BzLnRhcmdldCA/PyB0b1RhcmdldCh0aGlzLnByb3BzLnJ1bnRpbWUpfWAsXG4gICAgICAnLS1wbGF0Zm9ybT1ub2RlJyxcbiAgICAgIC4uLnRoaXMucHJvcHMuZm9ybWF0ID8gW2AtLWZvcm1hdD0ke3RoaXMucHJvcHMuZm9ybWF0fWBdIDogW10sXG4gICAgICBgLS1vdXRmaWxlPVwiJHtwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgb3V0RmlsZSl9XCJgLFxuICAgICAgLi4udGhpcy5wcm9wcy5taW5pZnkgPyBbJy0tbWluaWZ5J10gOiBbXSxcbiAgICAgIC4uLnNvdXJjZU1hcEVuYWJsZWQgPyBbYC0tc291cmNlbWFwJHtzb3VyY2VNYXBWYWx1ZX1gXSA6IFtdLFxuICAgICAgLi4uc291cmNlc0NvbnRlbnQgPyBbXSA6IFtgLS1zb3VyY2VzLWNvbnRlbnQ9JHtzb3VyY2VzQ29udGVudH1gXSxcbiAgICAgIC4uLnRoaXMuZXh0ZXJuYWxzLm1hcChleHRlcm5hbCA9PiBgLS1leHRlcm5hbDoke2V4dGVybmFsfWApLFxuICAgICAgLi4ubG9hZGVycy5tYXAoKFtleHQsIG5hbWVdKSA9PiBgLS1sb2FkZXI6JHtleHR9PSR7bmFtZX1gKSxcbiAgICAgIC4uLmRlZmluZXMubWFwKChba2V5LCB2YWx1ZV0pID0+IGAtLWRlZmluZToke2tleX09JHtKU09OLnN0cmluZ2lmeSh2YWx1ZSl9YCksXG4gICAgICAuLi50aGlzLnByb3BzLmxvZ0xldmVsID8gW2AtLWxvZy1sZXZlbD0ke3RoaXMucHJvcHMubG9nTGV2ZWx9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMua2VlcE5hbWVzID8gWyctLWtlZXAtbmFtZXMnXSA6IFtdLFxuICAgICAgLi4udGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA/IFtgLS10c2NvbmZpZz0ke3BhdGhKb2luKG9wdGlvbnMuaW5wdXREaXIsIHRoaXMucmVsYXRpdmVUc2NvbmZpZ1BhdGgpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLm1ldGFmaWxlID8gW2AtLW1ldGFmaWxlPSR7cGF0aEpvaW4ob3B0aW9ucy5vdXRwdXREaXIsICdpbmRleC5tZXRhLmpzb24nKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5iYW5uZXIgPyBbYC0tYmFubmVyOmpzPSR7SlNPTi5zdHJpbmdpZnkodGhpcy5wcm9wcy5iYW5uZXIpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmZvb3RlciA/IFtgLS1mb290ZXI6anM9JHtKU09OLnN0cmluZ2lmeSh0aGlzLnByb3BzLmZvb3Rlcil9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMuY2hhcnNldCA/IFtgLS1jaGFyc2V0PSR7dGhpcy5wcm9wcy5jaGFyc2V0fWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLm1haW5GaWVsZHMgPyBbYC0tbWFpbi1maWVsZHM9JHt0aGlzLnByb3BzLm1haW5GaWVsZHMuam9pbignLCcpfWBdIDogW10sXG4gICAgXTtcblxuICAgIGxldCBkZXBzQ29tbWFuZCA9ICcnO1xuICAgIGlmICh0aGlzLnByb3BzLm5vZGVNb2R1bGVzKSB7XG4gICAgICAvLyBGaW5kICdwYWNrYWdlLmpzb24nIGNsb3Nlc3QgdG8gZW50cnkgZm9sZGVyLCB3ZSBhcmUgZ29pbmcgdG8gZXh0cmFjdCB0aGVcbiAgICAgIC8vIG1vZHVsZXMgdmVyc2lvbnMgZnJvbSBpdC5cbiAgICAgIGNvbnN0IHBrZ1BhdGggPSBmaW5kVXAoJ3BhY2thZ2UuanNvbicsIHBhdGguZGlybmFtZSh0aGlzLnByb3BzLmVudHJ5KSk7XG4gICAgICBpZiAoIXBrZ1BhdGgpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBhIGBwYWNrYWdlLmpzb25gIGluIHRoaXMgcHJvamVjdC4gVXNpbmcgYG5vZGVNb2R1bGVzYCByZXF1aXJlcyBhIGBwYWNrYWdlLmpzb25gLicpO1xuICAgICAgfVxuXG4gICAgICAvLyBEZXRlcm1pbmUgZGVwZW5kZW5jaWVzIHZlcnNpb25zLCBsb2NrIGZpbGUgYW5kIGluc3RhbGxlclxuICAgICAgY29uc3QgZGVwZW5kZW5jaWVzID0gZXh0cmFjdERlcGVuZGVuY2llcyhwa2dQYXRoLCB0aGlzLnByb3BzLm5vZGVNb2R1bGVzKTtcbiAgICAgIGNvbnN0IG9zQ29tbWFuZCA9IG5ldyBPc0NvbW1hbmQob3B0aW9ucy5vc1BsYXRmb3JtKTtcblxuICAgICAgY29uc3QgbG9ja0ZpbGVQYXRoID0gcGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgdGhpcy5yZWxhdGl2ZURlcHNMb2NrRmlsZVBhdGggPz8gdGhpcy5wYWNrYWdlTWFuYWdlci5sb2NrRmlsZSk7XG5cbiAgICAgIC8vIENyZWF0ZSBkdW1teSBwYWNrYWdlLmpzb24sIGNvcHkgbG9jayBmaWxlIGlmIGFueSBhbmQgdGhlbiBpbnN0YWxsXG4gICAgICBkZXBzQ29tbWFuZCA9IGNoYWluKFtcbiAgICAgICAgb3NDb21tYW5kLndyaXRlSnNvbihwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgJ3BhY2thZ2UuanNvbicpLCB7IGRlcGVuZGVuY2llcyB9KSxcbiAgICAgICAgb3NDb21tYW5kLmNvcHkobG9ja0ZpbGVQYXRoLCBwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgdGhpcy5wYWNrYWdlTWFuYWdlci5sb2NrRmlsZSkpLFxuICAgICAgICBvc0NvbW1hbmQuY2hhbmdlRGlyZWN0b3J5KG9wdGlvbnMub3V0cHV0RGlyKSxcbiAgICAgICAgdGhpcy5wYWNrYWdlTWFuYWdlci5pbnN0YWxsQ29tbWFuZC5qb2luKCcgJyksXG4gICAgICBdKTtcbiAgICB9XG5cbiAgICByZXR1cm4gY2hhaW4oW1xuICAgICAgLi4udGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmJlZm9yZUJ1bmRsaW5nKG9wdGlvbnMuaW5wdXREaXIsIG9wdGlvbnMub3V0cHV0RGlyKSA/PyBbXSxcbiAgICAgIHRzY0NvbW1hbmQsXG4gICAgICBlc2J1aWxkQ29tbWFuZC5qb2luKCcgJyksXG4gICAgICAuLi4odGhpcy5wcm9wcy5ub2RlTW9kdWxlcyAmJiB0aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYmVmb3JlSW5zdGFsbChvcHRpb25zLmlucHV0RGlyLCBvcHRpb25zLm91dHB1dERpcikpID8/IFtdLFxuICAgICAgZGVwc0NvbW1hbmQsXG4gICAgICAuLi50aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYWZ0ZXJCdW5kbGluZyhvcHRpb25zLmlucHV0RGlyLCBvcHRpb25zLm91dHB1dERpcikgPz8gW10sXG4gICAgXSk7XG4gIH1cblxuICBwcml2YXRlIGdldExvY2FsQnVuZGxpbmdQcm92aWRlcigpOiBjZGsuSUxvY2FsQnVuZGxpbmcge1xuICAgIGNvbnN0IG9zUGxhdGZvcm0gPSBvcy5wbGF0Zm9ybSgpO1xuICAgIGNvbnN0IGNyZWF0ZUxvY2FsQ29tbWFuZCA9IChvdXRwdXREaXI6IHN0cmluZywgZXNidWlsZDogUGFja2FnZUluc3RhbGxhdGlvbiwgdHNjPzogUGFja2FnZUluc3RhbGxhdGlvbikgPT4gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQoe1xuICAgICAgaW5wdXREaXI6IHRoaXMucHJvamVjdFJvb3QsXG4gICAgICBvdXRwdXREaXIsXG4gICAgICBlc2J1aWxkUnVubmVyOiBlc2J1aWxkLmlzTG9jYWwgPyB0aGlzLnBhY2thZ2VNYW5hZ2VyLnJ1bkJpbkNvbW1hbmQoJ2VzYnVpbGQnKSA6ICdlc2J1aWxkJyxcbiAgICAgIHRzY1J1bm5lcjogdHNjICYmICh0c2MuaXNMb2NhbCA/IHRoaXMucGFja2FnZU1hbmFnZXIucnVuQmluQ29tbWFuZCgndHNjJykgOiAndHNjJyksXG4gICAgICBvc1BsYXRmb3JtLFxuICAgIH0pO1xuICAgIGNvbnN0IGVudmlyb25tZW50ID0gdGhpcy5wcm9wcy5lbnZpcm9ubWVudCA/PyB7fTtcbiAgICBjb25zdCBjd2QgPSB0aGlzLnByb2plY3RSb290O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHRyeUJ1bmRsZShvdXRwdXREaXI6IHN0cmluZykge1xuICAgICAgICBpZiAoIUJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24pIHtcbiAgICAgICAgICBwcm9jZXNzLnN0ZGVyci53cml0ZSgnZXNidWlsZCBjYW5ub3QgcnVuIGxvY2FsbHkuIFN3aXRjaGluZyB0byBEb2NrZXIgYnVuZGxpbmcuXFxuJyk7XG4gICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uLnZlcnNpb24uc3RhcnRzV2l0aChgJHtFU0JVSUxEX01BSk9SX1ZFUlNJT059LmApKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBFeHBlY3RlZCBlc2J1aWxkIHZlcnNpb24gJHtFU0JVSUxEX01BSk9SX1ZFUlNJT059LnggYnV0IGdvdCAke0J1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24udmVyc2lvbn1gKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IGxvY2FsQ29tbWFuZCA9IGNyZWF0ZUxvY2FsQ29tbWFuZChvdXRwdXREaXIsIEJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24sIEJ1bmRsaW5nLnRzY0luc3RhbGxhdGlvbik7XG5cbiAgICAgICAgZXhlYyhcbiAgICAgICAgICBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ2NtZCcgOiAnYmFzaCcsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgb3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJyA/ICcvYycgOiAnLWMnLFxuICAgICAgICAgICAgbG9jYWxDb21tYW5kLFxuICAgICAgICAgIF0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgZW52OiB7IC4uLnByb2Nlc3MuZW52LCAuLi5lbnZpcm9ubWVudCB9LFxuICAgICAgICAgICAgc3RkaW86IFsgLy8gc2hvdyBvdXRwdXRcbiAgICAgICAgICAgICAgJ2lnbm9yZScsIC8vIGlnbm9yZSBzdGRpb1xuICAgICAgICAgICAgICBwcm9jZXNzLnN0ZGVyciwgLy8gcmVkaXJlY3Qgc3Rkb3V0IHRvIHN0ZGVyclxuICAgICAgICAgICAgICAnaW5oZXJpdCcsIC8vIGluaGVyaXQgc3RkZXJyXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgY3dkLFxuICAgICAgICAgICAgd2luZG93c1ZlcmJhdGltQXJndW1lbnRzOiBvc1BsYXRmb3JtID09PSAnd2luMzInLFxuICAgICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbmludGVyZmFjZSBCdW5kbGluZ0NvbW1hbmRPcHRpb25zIHtcbiAgcmVhZG9ubHkgaW5wdXREaXI6IHN0cmluZztcbiAgcmVhZG9ubHkgb3V0cHV0RGlyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGVzYnVpbGRSdW5uZXI6IHN0cmluZztcbiAgcmVhZG9ubHkgdHNjUnVubmVyPzogc3RyaW5nO1xuICByZWFkb25seSBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm07XG59XG5cbi8qKlxuICogT1MgYWdub3N0aWMgY29tbWFuZFxuICovXG5jbGFzcyBPc0NvbW1hbmQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IG9zUGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge31cblxuICBwdWJsaWMgd3JpdGVKc29uKGZpbGVQYXRoOiBzdHJpbmcsIGRhdGE6IGFueSk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RyaW5naWZpZWREYXRhID0gSlNPTi5zdHJpbmdpZnkoZGF0YSk7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBlY2hvIF4ke3N0cmluZ2lmaWVkRGF0YX1eID4gXCIke2ZpbGVQYXRofVwiYDtcbiAgICB9XG5cbiAgICByZXR1cm4gYGVjaG8gJyR7c3RyaW5naWZpZWREYXRhfScgPiBcIiR7ZmlsZVBhdGh9XCJgO1xuICB9XG5cbiAgcHVibGljIGNvcHkoc3JjOiBzdHJpbmcsIGRlc3Q6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBjb3B5IFwiJHtzcmN9XCIgXCIke2Rlc3R9XCJgO1xuICAgIH1cblxuICAgIHJldHVybiBgY3AgXCIke3NyY31cIiBcIiR7ZGVzdH1cImA7XG4gIH1cblxuICBwdWJsaWMgY2hhbmdlRGlyZWN0b3J5KGRpcjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gYGNkIFwiJHtkaXJ9XCJgO1xuICB9XG59XG5cbi8qKlxuICogQ2hhaW4gY29tbWFuZHNcbiAqL1xuZnVuY3Rpb24gY2hhaW4oY29tbWFuZHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgcmV0dXJuIGNvbW1hbmRzLmZpbHRlcihjID0+ICEhYykuam9pbignICYmICcpO1xufVxuXG4vKipcbiAqIFBsYXRmb3JtIHNwZWNpZmljIHBhdGggam9pblxuICovXG5mdW5jdGlvbiBvc1BhdGhKb2luKHBsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0pIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKC4uLnBhdGhzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gICAgY29uc3Qgam9pbmVkID0gcGF0aC5qb2luKC4uLnBhdGhzKTtcbiAgICAvLyBJZiB3ZSBhcmUgb24gd2luMzIgYnV0IG5lZWQgcG9zaXggc3R5bGUgcGF0aHNcbiAgICBpZiAob3MucGxhdGZvcm0oKSA9PT0gJ3dpbjMyJyAmJiBwbGF0Zm9ybSAhPT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGpvaW5lZC5yZXBsYWNlKC9cXFxcL2csICcvJyk7XG4gICAgfVxuICAgIHJldHVybiBqb2luZWQ7XG4gIH07XG59XG5cbi8qKlxuICogQ29udmVydHMgYSBydW50aW1lIHRvIGFuIGVzYnVpbGQgbm9kZSB0YXJnZXRcbiAqL1xuZnVuY3Rpb24gdG9UYXJnZXQocnVudGltZTogUnVudGltZSk6IHN0cmluZyB7XG4gIGNvbnN0IG1hdGNoID0gcnVudGltZS5uYW1lLm1hdGNoKC9ub2RlanMoXFxkKykvKTtcblxuICBpZiAoIW1hdGNoKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZXh0cmFjdCB2ZXJzaW9uIGZyb20gcnVudGltZS4nKTtcbiAgfVxuXG4gIHJldHVybiBgbm9kZSR7bWF0Y2hbMV19YDtcbn1cbiJdfQ==