"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk_1.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')
            ]
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*']
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.Standard, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs8.10',
                Timeout: 300,
            }
        });
        this.functionArn = resource.getAtt('Arn').toString();
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    const s3 = new (require('aws-sdk').S3)();
    const https = require("https");
    const url = require("url");
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse("FAILED", err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse("SUCCESS");
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || "See the details in CloudWatch Log Stream: " + context.logStreamName,
            PhysicalResourceId: context.logStreamName,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "content-type": "",
                "content-length": responseBody.length
            }
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on("error", (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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