"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
class IdentitySourceBase extends core_1.Resource {
}
class IdentitySource extends IdentitySourceBase {
    /**
     * Create an Identity Source from its ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceArn The Identity Source ARN.
     */
    static fromIdentitySourceArn(scope, id, identitySourceArn) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceArn,
        });
    }
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope, id, attrs) {
        class Import extends IdentitySourceBase {
            constructor(identitySourceArn, identitySourceId) {
                super(scope, id);
                this.identitySourceArn = identitySourceArn;
                this.identitySourceId = identitySourceId;
            }
        }
        let identitySourceArn;
        let identitySourceId;
        const stack = core_1.Stack.of(scope);
        if (!attrs.identitySourceId) {
            if (!attrs.identitySourceArn) {
                throw new Error('One of identitySourceId or identitySourceArn is required!');
            }
            identitySourceArn = attrs.identitySourceArn;
            const maybeId = stack.splitArn(attrs.identitySourceArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for IdentitySource must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            identitySourceId = maybeId;
        }
        else {
            if (attrs.identitySourceArn) {
                throw new Error('Only one of identitySourceArn or identitySourceId can be provided');
            }
            identitySourceId = attrs.identitySourceId;
            identitySourceArn = stack.formatArn({
                resource: 'identity-source',
                resourceName: attrs.identitySourceId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(identitySourceArn, identitySourceId);
    }
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope, id, identitySourceId) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceId,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.clientIds =
            props.configuration.cognitoUserPoolConfiguration.clientIds ?? [];
        this.userPoolArn =
            props.configuration.cognitoUserPoolConfiguration.userPool.userPoolArn;
        const cognitoGroupConfiguration = props.configuration.cognitoUserPoolConfiguration.groupConfiguration?.groupEntityType
            ? {
                groupEntityType: props.configuration.cognitoUserPoolConfiguration.groupConfiguration.groupEntityType,
            }
            : undefined;
        this.identitySource = new aws_verifiedpermissions_1.CfnIdentitySource(this, id, {
            configuration: {
                cognitoUserPoolConfiguration: {
                    clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                    userPoolArn: this.userPoolArn,
                    groupConfiguration: cognitoGroupConfiguration,
                },
            },
            policyStoreId: props.policyStore.policyStoreId,
            principalEntityType: props.principalEntityType,
        });
        this.discoveryUrl = this.identitySource.attrDetailsDiscoveryUrl;
        this.identitySourceId = this.identitySource.attrIdentitySourceId;
        this.identitySourceArn = this.stack.formatArn({
            resource: 'identity-source',
            resourceName: this.identitySourceId,
            service: 'verifiedpermissions',
        });
        this.openIdIssuer = this.identitySource.attrDetailsOpenIdIssuer;
        this.policyStore = props.policyStore;
        this.cognitoGroupEntityType = cognitoGroupConfiguration?.groupEntityType;
    }
    /**
     * Add a User Pool Client
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient) {
        this.clientIds.push(userPoolClient.userPoolClientId);
    }
}
exports.IdentitySource = IdentitySource;
_a = JSII_RTTI_SYMBOL_1;
IdentitySource[_a] = { fqn: "@cdklabs/cdk-verified-permissions.IdentitySource", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,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