"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
    }) {
        super(scope, id);
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: props.validationSettings,
            description: props.description,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,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