"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return;
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        return iam.Grant.addToPrincipalAndResource({
            grantee,
            actions,
            resource: this,
            resourcePolicyPrincipal: principal,
            // if the key is used in a cross-environment matter,
            // we can't access the Key ARN (they don't have physical names),
            // so fall back to using '*'. ToDo we need to make this better... somehow
            resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        });
    }
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return undefined;
        }
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantee);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(core_1.Construct.isConstruct(grantee))) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.policy) {
            this.policy = props.policy;
        }
        else {
            this.policy = new iam.PolicyDocument();
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).parseArn(keyArn).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            "kms:Create*",
            "kms:Describe*",
            "kms:Enable*",
            "kms:List*",
            "kms:Put*",
            "kms:Update*",
            "kms:Revoke*",
            "kms:Disable*",
            "kms:Get*",
            "kms:Delete*",
            "kms:ScheduleKeyDeletion",
            "kms:CancelKeyDeletion",
            "kms:GenerateDataKey",
            "kms:TagResource",
            "kms:UntagResource"
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()]
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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