"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkInterface = exports.AddressingV6 = exports.AddressingV4 = exports.InterfaceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
/**
 * The type of Network Interface.
 */
var InterfaceType;
(function (InterfaceType) {
    /**
     * A standard ENI.
     */
    InterfaceType["INTERFACE"] = "interface";
    /**
     * An Elastic Fabric Adapter ENI.
     */
    InterfaceType["EFA"] = "efa";
    /**
     * An ENI for use with ECS awsvpc trunking.
     */
    InterfaceType["TRUNK"] = "trunk";
})(InterfaceType || (exports.InterfaceType = InterfaceType = {}));
/**
 * Used to assign IPv4 addresses to a Network Interface.
 */
class AddressingV4 {
    /**
     * Specify one or more IPv4 delegated prefixes to assign.
     *
     * IPv4 prefixes must be within a CIDR of /28.
     *
     * @param prefixes - The IPv4 delegated prefixes
     */
    static prefixes(prefixes) {
        return new AddressingV4({
            ipv4Prefixes: prefixes.map((p) => ({ ipv4Prefix: p })),
        });
    }
    /**
     * Specify a number of IPv4 delegated prefixes to automatically assign.
     *
     * @param count - The number of automatic IPv4 delegated prefixes
     */
    static prefixCount(count) {
        return new AddressingV4({ ipv4PrefixCount: count });
    }
    /**
     * Specify a private IPv4 address.
     *
     * @param ip - The actual IP address
     */
    static privateAddress(ip) {
        return new AddressingV4({ privateIpAddress: ip });
    }
    /**
     * Specify a primary IPv4 address and one or more secondary IPv4 addresses.
     *
     * @param primary - The primary address
     * @param secondary - Any secondary addresses
     */
    static privateAddresses(primary, ...secondary) {
        const privateIpAddresses = [{ privateIpAddress: primary, primary: true }];
        if (secondary.length) {
            privateIpAddresses.push(...secondary.map((a) => ({ privateIpAddress: a, primary: false })));
        }
        return new AddressingV4({ privateIpAddresses });
    }
    /**
     * Specify a primary IPv4 address and a number of secondary addresses.
     *
     * @param primary - The primary address
     * @param count - The number of secondary addresses
     */
    static privateAddressAndSecondaryCount(primary, count) {
        return new AddressingV4({
            privateIpAddresses: [{ privateIpAddress: primary, primary: true }],
            secondaryPrivateIpAddressCount: count,
        });
    }
    /**
     * @ignore
     */
    constructor(props) {
        this.props = props;
    }
}
exports.AddressingV4 = AddressingV4;
_a = JSII_RTTI_SYMBOL_1;
AddressingV4[_a] = { fqn: "shady-island.networking.AddressingV4", version: "0.1.23" };
/**
 * Used to assign IPv6 addresses to a Network Interface.
 */
class AddressingV6 {
    /**
     * Specify one or more IPv6 delegated prefixes to assign.
     *
     * IPv6 prefixes must be within a CIDR of /80.
     *
     * @param prefixes - The IPv6 delegated prefixes
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static prefixes(prefixes, enablePrimary) {
        return new AddressingV6({
            ipv6Prefixes: prefixes.map((p) => ({ ipv6Prefix: p })),
            ...(enablePrimary !== undefined
                ? { enablePrimaryIpv6: enablePrimary }
                : {}),
        });
    }
    /**
     * Specify a number of IPv6 delegated prefixes to automatically assign.
     *
     * @param count - The number of automatic IPv6 delegated prefixes
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static prefixCount(count, enablePrimary) {
        return new AddressingV6({
            ipv6PrefixCount: count,
            ...(enablePrimary !== undefined
                ? { enablePrimaryIpv6: enablePrimary }
                : {}),
        });
    }
    /**
     * Specify one or more IPv6 addresses to assign.
     *
     * @param ips - The IPv6 addresses
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static addresses(ips, enablePrimary) {
        return new AddressingV6({
            ipv6Addresses: ips.map((a) => ({ ipv6Address: a })),
            ...(enablePrimary !== undefined
                ? { enablePrimaryIpv6: enablePrimary }
                : {}),
        });
    }
    /**
     * Specify a number of IPv6 addresses to automatically assign.
     *
     * @param count - The number of automatic IPv6 addresses
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static addressCount(count, enablePrimary) {
        return new AddressingV6({
            ipv6AddressCount: count,
            ...(enablePrimary !== undefined
                ? { enablePrimaryIpv6: enablePrimary }
                : {}),
        });
    }
    /**
     * @ignore
     */
    constructor(props) {
        this.props = props;
    }
}
exports.AddressingV6 = AddressingV6;
_b = JSII_RTTI_SYMBOL_1;
AddressingV6[_b] = { fqn: "shady-island.networking.AddressingV6", version: "0.1.23" };
/**
 * A Network Interface.
 */
class NetworkInterface extends aws_cdk_lib_1.Resource {
    /**
     * Import an existing Network Interface from the given attributes.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param attribs - The Network Interface attributes.
     * @returns The imported Network Interface
     */
    static fromNetworkInterfaceAttributes(scope, id, attribs) {
        class ImportedNetworkInterface extends constructs_1.Construct {
            constructor() {
                super(...arguments);
                this.networkInterfaceId = attribs.networkInterfaceId;
                this.connections = new aws_ec2_1.Connections({
                    securityGroups: attribs.securityGroups,
                });
            }
        }
        return new ImportedNetworkInterface(scope, id);
    }
    /**
     * Creates a new Example.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { vpc, subnet, description, securityGroups = [] } = props;
        this.subnet = subnet;
        if (securityGroups.length === 0) {
            securityGroups.push(new aws_ec2_1.SecurityGroup(this, "SecurityGroup", {
                vpc,
                description: `Created for ${this.node.path}`,
            }));
        }
        this.connections = new aws_ec2_1.Connections({ securityGroups });
        const securityGroupsToken = aws_cdk_lib_1.Lazy.list({
            produce: () => {
                return this.connections.securityGroups.map((sg) => sg.securityGroupId);
            },
        });
        const eni = new aws_ec2_1.CfnNetworkInterface(this, "Resource", {
            description,
            subnetId: subnet.subnetId,
            groupSet: securityGroupsToken,
            sourceDestCheck: props.enableSourceDestCheck !== false,
            interfaceType: props.interfaceType,
            ...(props.ipv4?.props ?? {}),
            ...(props.ipv6?.props ?? {}),
        });
        this.networkInterfaceId = eni.attrId;
        this.privateIpv4Address = eni.attrPrimaryPrivateIpAddress;
        this.ipv6Address = eni.attrPrimaryIpv6Address;
        if (props.elasticIp) {
            new aws_ec2_1.CfnEIPAssociation(this, "Association", {
                allocationId: props.elasticIp.allocationId,
                networkInterfaceId: eni.ref,
                privateIpAddress: eni.attrPrimaryPrivateIpAddress,
            });
        }
        if (props.removalPolicy) {
            this.applyRemovalPolicy(props.removalPolicy);
        }
    }
}
exports.NetworkInterface = NetworkInterface;
_c = JSII_RTTI_SYMBOL_1;
NetworkInterface[_c] = { fqn: "shady-island.networking.NetworkInterface", version: "0.1.23" };
//# sourceMappingURL=data:application/json;base64,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