import { RemovalPolicy, Resource } from "aws-cdk-lib";
import { Connections, IConnectable, ISecurityGroup, ISubnet, IVpc } from "aws-cdk-lib/aws-ec2";
import { Construct, IConstruct } from "constructs";
import { IElasticIp } from "./elastic-ip";
/**
 * The type of Network Interface.
 */
export declare enum InterfaceType {
    /**
     * A standard ENI.
     */
    INTERFACE = "interface",
    /**
     * An Elastic Fabric Adapter ENI.
     */
    EFA = "efa",
    /**
     * An ENI for use with ECS awsvpc trunking.
     */
    TRUNK = "trunk"
}
/**
 * Used to assign IPv4 addresses to a Network Interface.
 */
export declare class AddressingV4 {
    /**
     * Specify one or more IPv4 delegated prefixes to assign.
     *
     * IPv4 prefixes must be within a CIDR of /28.
     *
     * @param prefixes - The IPv4 delegated prefixes
     */
    static prefixes(prefixes: string[]): AddressingV4;
    /**
     * Specify a number of IPv4 delegated prefixes to automatically assign.
     *
     * @param count - The number of automatic IPv4 delegated prefixes
     */
    static prefixCount(count: number): AddressingV4;
    /**
     * Specify a private IPv4 address.
     *
     * @param ip - The actual IP address
     */
    static privateAddress(ip: string): AddressingV4;
    /**
     * Specify a primary IPv4 address and one or more secondary IPv4 addresses.
     *
     * @param primary - The primary address
     * @param secondary - Any secondary addresses
     */
    static privateAddresses(primary: string, ...secondary: string[]): AddressingV4;
    /**
     * Specify a primary IPv4 address and a number of secondary addresses.
     *
     * @param primary - The primary address
     * @param count - The number of secondary addresses
     */
    static privateAddressAndSecondaryCount(primary: string, count: number): AddressingV4;
    readonly props: Record<string, any>;
    /**
     * @ignore
     */
    private constructor();
}
/**
 * Used to assign IPv6 addresses to a Network Interface.
 */
export declare class AddressingV6 {
    /**
     * Specify one or more IPv6 delegated prefixes to assign.
     *
     * IPv6 prefixes must be within a CIDR of /80.
     *
     * @param prefixes - The IPv6 delegated prefixes
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static prefixes(prefixes: string[], enablePrimary?: boolean): AddressingV6;
    /**
     * Specify a number of IPv6 delegated prefixes to automatically assign.
     *
     * @param count - The number of automatic IPv6 delegated prefixes
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static prefixCount(count: number, enablePrimary?: boolean): AddressingV6;
    /**
     * Specify one or more IPv6 addresses to assign.
     *
     * @param ips - The IPv6 addresses
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static addresses(ips: string[], enablePrimary?: boolean): AddressingV6;
    /**
     * Specify a number of IPv6 addresses to automatically assign.
     *
     * @param count - The number of automatic IPv6 addresses
     * @param enablePrimary - Whether to enable a primary IPv6 GUA (default: no)
     */
    static addressCount(count: number, enablePrimary?: boolean): AddressingV6;
    readonly props: Record<string, any>;
    /**
     * @ignore
     */
    private constructor();
}
/**
 * An Elastic Network Interface.
 */
export interface INetworkInterface extends IConstruct, IConnectable {
    /**
     * The ID of this Network Interface.
     */
    readonly networkInterfaceId: string;
}
/**
 * Constructor properties for NetworkInterface.
 */
export interface NetworkInterfaceProps {
    /**
     * The VPC where this Network Interface will be created.
     */
    readonly vpc: IVpc;
    /**
     * The subnet where this Network Interface will be created.
     */
    readonly subnet: ISubnet;
    /**
     * A description for this Network Interface.
     */
    readonly description?: string;
    /**
     * How to assign IPv4 addresses.
     *
     * The default behavior depends on the VPC. If it's a dual stack VPC, EC2 will
     * allocate a single private IP address from the VPC IPv4 CIDR range. If it's
     * IPv6-only, EC2 won't allocate an IPv4 address.
     *
     * @default - Dependent on VPC settings
     */
    readonly ipv4?: AddressingV4;
    /**
     * How to assign IPv6 addresses.
     *
     * The default behavior depends on the VPC. If there are no IPv6 CIDRs defined
     * for the VPC, EC2 won't allocate an IPv6 address. If it's a dual stack or an
     * IPv6-only VPC, EC2 will allocate an IPv6 address if the subnet auto-assigns
     * one.
     *
     * @default - Dependent on VPC and subnet settings.
     */
    readonly ipv6?: AddressingV6;
    /**
     * The type of interface (i.e. interface, efa, trunk).
     *
     * @default - InterfaceType.INTERFACE
     */
    readonly interfaceType?: InterfaceType;
    /**
     * Enable the source/destination check.
     *
     * @default - true
     */
    readonly enableSourceDestCheck?: boolean;
    /**
     * The security groups to assign to the Network Interface.
     *
     * @default - A new one is created
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * An Elastic IP Address to associate with this Network Interface.
     *
     * Provding an Elastic IP
     */
    readonly elasticIp?: IElasticIp;
    /**
     * The removal policy for this resource.
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Attributes to import an existing Network Interface.
 */
export interface NetworkInterfaceAttributes {
    /**
     * The ID of this Network Interface.
     */
    readonly networkInterfaceId: string;
    /**
     * The security groups assigned to the Network Interface.
     */
    readonly securityGroups: ISecurityGroup[];
}
/**
 * A Network Interface.
 */
export declare class NetworkInterface extends Resource implements INetworkInterface {
    /**
     * Import an existing Network Interface from the given attributes.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param attribs - The Network Interface attributes.
     * @returns The imported Network Interface
     */
    static fromNetworkInterfaceAttributes(scope: Construct, id: string, attribs: NetworkInterfaceAttributes): INetworkInterface;
    readonly subnet: ISubnet;
    readonly networkInterfaceId: string;
    readonly connections: Connections;
    readonly privateIpv4Address: string;
    readonly ipv6Address: string;
    /**
     * Creates a new Example.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param props - Initialization properties for this construct.
     */
    constructor(scope: Construct, id: string, props: NetworkInterfaceProps);
}
