import { SecretValue } from "aws-cdk-lib";
import { ListenerCondition } from "aws-cdk-lib/aws-elasticloadbalancingv2";
import { ISecret } from "aws-cdk-lib/aws-secretsmanager";
import { IConstruct, Construct } from "constructs";
/**
 * Properties for the SecretHttpHeader constructor.
 */
export interface SecretHttpHeaderProps {
    /**
     * The name of the secret HTTP header.
     *
     * @default - X-Secret-Passphrase
     */
    readonly headerName?: string;
}
/**
 * Interface for SecretHttpHeader.
 */
export interface ISecretHttpHeader extends IConstruct {
    /**
     * The name of the secret header.
     */
    readonly headerName: string;
    /**
     * The value of the secret header.
     */
    readonly headerValue: SecretValue;
    /**
     * Get a ListenerCondition that represents this secret header.
     *
     * @returns The appropriate ListenerCondition.
     */
    createListenerCondition(): ListenerCondition;
    /**
     * Gets the custom headers for a CloudFront origin configuration.
     *
     * @returns An object with the header name and header value.
     */
    createOriginCustomHeaders(): Record<string, string>;
}
/**
 * Common behavior for secret HTTP header constructs.
 */
declare abstract class BaseSecretHttpHeader extends Construct {
    abstract headerName: string;
    abstract headerValue: SecretValue;
    createListenerCondition(): ListenerCondition;
    createOriginCustomHeaders(): Record<string, string>;
}
/**
 * Configure a secret header an ALB can require for every request.
 */
export declare class SecretHttpHeader extends BaseSecretHttpHeader implements ISecretHttpHeader {
    /**
     * Gets the default header name.
     *
     * @returns the default header name
     */
    static get defaultHeaderName(): string;
    /**
     * Create a SecretHttpHeader from an existing Secrets Manager secret.
     *
     * The secret must be in JSON format and have two fields: `name` and `value`.
     *
     * @param scope - The parent scope
     * @param id - The ID for the new construct
     * @param secret - The existing Secrets Manager secret
     * @returns The new ISecretHttpHeader
     */
    static fromSecret(scope: Construct, id: string, secret: ISecret): ISecretHttpHeader;
    /**
     * The Secrets Manager secret that contains the name and value of the header.
     */
    readonly secret: ISecret;
    private readonly _name;
    private readonly _value;
    /**
     * Creates a new SecretHttpHeader.
     *
     * @param scope - The parent scope.
     * @param id - The construct identifier.
     * @param props - The configuration properties.
     */
    constructor(scope: Construct, id: string, props?: SecretHttpHeaderProps);
    get headerName(): string;
    get headerValue(): SecretValue;
}
export {};
