import * as React from 'react';
import { FileBrowserModel } from '@jupyterlab/filebrowser';
import { IRenderMimeRegistry } from '@jupyterlab/rendermime';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { GitExtension } from '../model';
import { Git, ILogMessage } from '../tokens';
/**
 * Interface describing component properties.
 */
export interface IGitPanelProps {
    /**
     * Git extension data model.
     */
    model: GitExtension;
    /**
     * MIME type registry.
     */
    renderMime: IRenderMimeRegistry;
    /**
     * Git extension settings.
     */
    settings: ISettingRegistry.ISettings;
    /**
     * File browser model.
     */
    filebrowser: FileBrowserModel;
}
/**
 * Interface describing component state.
 */
export interface IGitPanelState {
    /**
     * Boolean indicating whether the user is currently in a Git repository.
     */
    inGitRepository: boolean;
    /**
     * List of branches.
     */
    branches: Git.IBranch[];
    /**
     * Current branch.
     */
    currentBranch: string;
    /**
     * List of changed files.
     */
    files: Git.IStatusFile[];
    /**
     * List of prior commits.
     */
    pastCommits: Git.ISingleCommitInfo[];
    /**
     * Panel tab identifier.
     */
    tab: number;
    /**
     * Boolean indicating whether UI interaction should be suspended (e.g., due to pending command).
     */
    suspend: boolean;
    /**
     * Boolean indicating whether to show an alert.
     */
    alert: boolean;
    /**
     * Log message.
     */
    log: ILogMessage;
}
/**
 * React component for rendering a panel for performing Git operations.
 */
export declare class GitPanel extends React.Component<IGitPanelProps, IGitPanelState> {
    /**
     * Returns a React component for rendering a panel for performing Git operations.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props: IGitPanelProps);
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount(): void;
    refreshBranch: () => Promise<void>;
    refreshHistory: () => Promise<void>;
    refreshStatus: () => Promise<void>;
    /**
     * Refresh widget, update all content
     */
    refresh: () => Promise<void>;
    /**
     * Commits all marked files.
     *
     * @param message - commit message
     * @returns a promise which commits the files
     */
    commitMarkedFiles: (message: string) => Promise<void>;
    /**
     * Commits all staged files.
     *
     * @param message - commit message
     * @returns a promise which commits the files
     */
    commitStagedFiles: (message: string) => Promise<void>;
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render(): React.ReactElement;
    /**
     * Renders a toolbar.
     *
     * @returns React element
     */
    private _renderToolbar;
    /**
     * Renders the main panel.
     *
     * @returns React element
     */
    private _renderMain;
    /**
     * Renders panel tabs.
     *
     * @returns React element
     */
    private _renderTabs;
    /**
     * Renders a panel for viewing and committing file changes.
     *
     * @returns React element
     */
    private _renderChanges;
    /**
     * Renders a panel for viewing commit history.
     *
     * @returns React element
     */
    private _renderHistory;
    /**
     * Renders a component to provide UI feedback.
     *
     * @returns React element
     */
    private _renderFeedback;
    /**
     * Renders a panel for prompting a user to find a Git repository.
     *
     * @returns React element
     */
    private _renderWarning;
    /**
     * Sets the suspension state.
     *
     * @param bool - boolean indicating whether to suspend UI interaction
     */
    private _suspend;
    /**
     * Sets the current component log message.
     *
     * @param msg - log message
     */
    private _log;
    /**
     * Callback invoked upon changing the active panel tab.
     *
     * @param event - event object
     * @param tab - tab number
     */
    private _onTabChange;
    /**
     * Callback invoked upon refreshing a repository.
     *
     * @returns promise which refreshes a repository
     */
    private _onRefresh;
    /**
     * Callback invoked upon clicking on the feedback modal.
     *
     * @param event - event object
     */
    private _onFeedbackModalClick;
    /**
     * Callback invoked upon closing a feedback alert.
     *
     * @param event - event object
     */
    private _onFeedbackAlertClose;
    /**
     * Determines whether a user has a known Git identity.
     *
     * @param path - repository path
     * @returns a promise which returns a success status
     */
    private _hasIdentity;
    private _hasStagedFile;
    private _hasUnStagedFile;
    /**
     * List of marked files.
     */
    private get _markedFiles();
    /**
     * List of sorted modified files.
     */
    private get _sortedFiles();
    private _previousRepoPath;
}
