import * as React from 'react';
import TextareaAutosize from 'react-textarea-autosize';
import { classes } from 'typestyle';
import { showErrorMessage } from '@jupyterlab/apputils';
import Dialog from '@material-ui/core/Dialog';
import DialogActions from '@material-ui/core/DialogActions';
import ClearIcon from '@material-ui/icons/Clear';
import { sleep } from '../utils';
import { actionsWrapperClass, commitFormClass, commitSummaryClass, commitDescriptionClass, buttonClass, cancelButtonClass, closeButtonClass, contentWrapperClass, resetRevertDialogClass, submitButtonClass, titleClass, titleWrapperClass } from '../style/ResetRevertDialog';
import { SuspendModal } from './SuspendModal';
import { Alert } from './Alert';
/**
 * React component for rendering a dialog for reseting or reverting a single commit.
 */
export class ResetRevertDialog extends React.Component {
    /**
     * Returns a React component for reseting or reverting a single commit.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon updating a commit message summary.
         *
         * @param event - event object
         */
        this._onSummaryChange = (event) => {
            this.setState({
                summary: event.target.value
            });
        };
        /**
         * Callback invoked upon updating a commit message description.
         *
         * @param event - event object
         */
        this._onDescriptionChange = (event) => {
            this.setState({
                description: event.target.value
            });
        };
        /**
         * Callback invoked upon clicking on a dialog.
         *
         * @param event - event object
         */
        this._onClick = (event) => {
            event.stopPropagation();
        };
        /**
         * Callback invoked upon closing the dialog.
         *
         * @param event - event object
         */
        this._onClose = (event) => {
            event.stopPropagation();
            this.props.onClose();
            this._reset();
        };
        /**
         * Callback invoked upon clicking a button to reset or revert a commit.
         *
         * @param event - event object
         */
        this._onSubmit = async () => {
            const shortCommit = this.props.commit.commit.slice(0, 7);
            let err;
            this.setState({
                disabled: true
            });
            if (this.props.action === 'reset') {
                this._log({
                    severity: 'info',
                    message: 'Discarding changes...'
                });
                this._suspend(true);
                try {
                    await Promise.all([
                        sleep(1000),
                        this.props.model.resetToCommit(this.props.commit.commit)
                    ]);
                }
                catch (error) {
                    err = error;
                }
                this._suspend(false);
                if (err) {
                    this._log({
                        severity: 'error',
                        message: 'Failed to discard changes.'
                    });
                    showErrorMessage('Error Discarding Changes', `Failed to discard changes after ${shortCommit}: ${err}`);
                }
                else {
                    this._log({
                        severity: 'success',
                        message: 'Successfully discarded changes.'
                    });
                }
            }
            else {
                this._log({
                    severity: 'info',
                    message: 'Reverting changes...'
                });
                this._suspend(true);
                try {
                    await Promise.all([
                        sleep(1000),
                        this.props.model.revertCommit(this._commitMessage(), this.props.commit.commit)
                    ]);
                }
                catch (error) {
                    err = error;
                }
                this._suspend(false);
                if (err) {
                    this._log({
                        severity: 'error',
                        message: 'Failed to revert changes.'
                    });
                    showErrorMessage('Error Reverting Changes', `Failed to revert ${shortCommit}: ${err}`);
                }
                else {
                    this._log({
                        severity: 'success',
                        message: 'Successfully reverted changes.'
                    });
                }
            }
            this._reset();
            this.props.onClose();
        };
        /**
         * Callback invoked upon clicking on the feedback modal.
         *
         * @param event - event object
         */
        this._onFeedbackModalClick = () => {
            this._suspend(false);
        };
        /**
         * Callback invoked upon closing a feedback alert.
         *
         * @param event - event object
         */
        this._onFeedbackAlertClose = () => {
            this.setState({
                alert: false
            });
        };
        this.state = {
            summary: '',
            description: '',
            disabled: false,
            suspend: false,
            alert: false,
            log: {
                severity: 'info',
                message: ''
            }
        };
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement(React.Fragment, null,
            this._renderDialog(),
            this._renderFeedback()));
    }
    /**
     * Renders a dialog.
     *
     * @returns React element
     */
    _renderDialog() {
        const shortCommit = this.props.commit.commit.slice(0, 7);
        return (React.createElement(Dialog, { classes: {
                paper: resetRevertDialogClass
            }, open: this.props.open, onClick: this._onClick, onClose: this._onClose },
            React.createElement("div", { className: titleWrapperClass },
                React.createElement("p", { className: titleClass }, this.props.action === 'revert'
                    ? 'Revert Changes'
                    : 'Reset Changes'),
                React.createElement("button", { className: closeButtonClass },
                    React.createElement(ClearIcon, { titleAccess: "Close this dialog", fontSize: "small", onClick: this._onClose }))),
            React.createElement("div", { className: contentWrapperClass },
                React.createElement("p", null, this.props.action === 'revert'
                    ? "These changes will be reverted. Only commit if you're sure you're okay losing these changes."
                    : `All changes after commit ${shortCommit} will be gone forever (hard reset). Are you sure?`),
                this.props.action === 'revert' ? (React.createElement("div", { className: commitFormClass },
                    React.createElement("input", { className: commitSummaryClass, type: "text", placeholder: this._defaultSummary(), title: "Enter a commit message summary (a single line, preferably less than 50 characters)", value: this.state.summary, onChange: this._onSummaryChange }),
                    React.createElement(TextareaAutosize, { className: commitDescriptionClass, minRows: 5, placeholder: this._defaultDescription(), title: "Enter a commit message description", value: this.state.description, onChange: this._onDescriptionChange }))) : null),
            React.createElement(DialogActions, { className: actionsWrapperClass },
                React.createElement("input", { disabled: this.state.disabled, className: classes(buttonClass, cancelButtonClass), type: "button", title: "Cancel changes", value: "Cancel", onClick: this._onClose }),
                React.createElement("input", { disabled: this.state.disabled, className: classes(buttonClass, submitButtonClass), type: "button", title: "Submit changes", value: "Submit", onClick: this._onSubmit }))));
    }
    /**
     * Renders a component to provide UI feedback.
     *
     * @returns React element
     */
    _renderFeedback() {
        return (React.createElement(React.Fragment, null,
            React.createElement(SuspendModal, { open: this.props.suspend && this.state.suspend, onClick: this._onFeedbackModalClick }),
            React.createElement(Alert, { open: this.state.alert, message: this.state.log.message, severity: this.state.log.severity, onClose: this._onFeedbackAlertClose })));
    }
    /**
     * Sets the suspension state.
     *
     * @param bool - boolean indicating whether to suspend UI interaction
     */
    _suspend(bool) {
        if (this.props.suspend) {
            this.setState({
                suspend: bool
            });
        }
    }
    /**
     * Sets the current component log message.
     *
     * @param msg - log message
     */
    _log(msg) {
        this.setState({
            alert: true,
            log: msg
        });
    }
    /**
     * Returns a default commit summary for reverting a commit.
     *
     * @returns default commit summary
     */
    _defaultSummary() {
        const summary = this.props.commit.commit_msg.split('\n')[0];
        return `Revert "${summary}"`;
    }
    /**
     * Returns a default commit description for reverting a commit.
     *
     * @returns default commit description
     */
    _defaultDescription() {
        return `This reverts commit ${this.props.commit.commit}`;
    }
    /**
     * Returns a commit message for reverting a commit.
     *
     * @returns commit message
     */
    _commitMessage() {
        const summary = this.state.summary || this._defaultSummary();
        const desc = this.state.description || this._defaultDescription();
        return summary + '\n\n' + desc + '\n';
    }
    /**
     * Resets component state.
     */
    _reset() {
        this.setState({
            summary: '',
            description: '',
            disabled: false
        });
    }
}
//# sourceMappingURL=ResetRevertDialog.js.map